/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <frontend.h>
#include <glib.h>
#include <locale.h>

#include "common.h"
#include "value.h"
#include "readable.h"
#include "logging.h"

/*
 * Since limits.h sometimes doesn't define long long min/max values at times, do it here.
 */
#ifndef INT64_MAX
#define INT64_MAX	9223372036854775807LL
#endif
#ifndef INT64_MIN
#define INT64_MIN	(-INT64_MAX - 1LL)
#endif
#ifndef UINT64_MAX
#define UINT64_MAX	18446744073709551615ULL
#endif

#define MAX_UNITS	5
#define KB_TO_SECTORS_MULTIPLIER ((u_int64_t)2)
#define MB_TO_SECTORS_MULTIPLIER ((u_int64_t)(KB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define GB_TO_SECTORS_MULTIPLIER ((u_int64_t)(MB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define TB_TO_SECTORS_MULTIPLIER ((u_int64_t)(GB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))
#define PB_TO_SECTORS_MULTIPLIER ((u_int64_t)(TB_TO_SECTORS_MULTIPLIER * (u_int64_t)1024))

static const gdouble conversion_factor [MAX_UNITS] = {	KB_TO_SECTORS_MULTIPLIER,
							MB_TO_SECTORS_MULTIPLIER,
							GB_TO_SECTORS_MULTIPLIER,
							TB_TO_SECTORS_MULTIPLIER,
							PB_TO_SECTORS_MULTIPLIER };

static const value_unit_t units [MAX_UNITS] = {	EVMS_Unit_Kilobytes,
						EVMS_Unit_Megabytes,
						EVMS_Unit_Gigabytes,
						EVMS_Unit_Terabytes,
						EVMS_Unit_Petabytes };

/**
 *	value_less_than_integer - allows comparing a value_t to an integer
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@number: the integer value to check against
 *
 *	This routine takes a value_t of type value_type_t and
 *	and integer value and determines if the value is less
 *	than the supplied integer.
 */
gboolean value_less_than_integer(value_t value, value_type_t type, gint number)
{
	gboolean result = FALSE;

	switch (type) {
	case EVMS_Type_String:
		if (value.s)
			result = strtol(value.s, NULL, 0) < number;
		break;
	case EVMS_Type_Char:
		result = value.c < number;
		break;
	case EVMS_Type_Unsigned_Char:
		result = value.uc < number;
		break;
	case EVMS_Type_Real32:
		result = value.r32 < number;
		break;
	case EVMS_Type_Real64:
		result = value.r64 < number;
		break;
	case EVMS_Type_Int8:
		result = value.i8 < number;
		break;
	case EVMS_Type_Unsigned_Int8:
		result = value.ui8 < number;
		break;
	case EVMS_Type_Int16:
		result = value.i16 < number;
		break;
	case EVMS_Type_Unsigned_Int16:
		result = value.ui16 < number;
		break;
	case EVMS_Type_Int:
		result = value.i < number;
		break;
	case EVMS_Type_Int32:
		result = value.i32 < number;
		break;
	case EVMS_Type_Unsigned_Int:
		result = value.ui < number;
		break;
	case EVMS_Type_Unsigned_Int32:
		result = value.ui32 < number;
		break;
	case EVMS_Type_Int64:
		result = value.i64 < number;
		break;
	case EVMS_Type_Unsigned_Int64:
		result = value.ui64 < number;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return result;
}

/**
 *	convert_value_to_string - produce a string that represents a value_t
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@format: the suggested format (currently only hex supported)
 *
 *	This routine takes a value_t of type value_type_t and
 *	produces a string that represents the value. The string
 *	should be freed with g_free() when no longer needed.
 */
gchar *convert_value_to_string(value_t value, value_type_t type, value_format_t format)
{
	gchar *string = NULL;

	switch (type) {
	case EVMS_Type_Boolean:
		string = value.b ? g_strdup(_("Yes")) : g_strdup(_("No"));
		break;
	case EVMS_Type_String:
		string = value.s ? g_strdup(value.s) : g_strdup("");
		break;
	case EVMS_Type_Char:
	case EVMS_Type_Unsigned_Char:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#hhx", value.uc);
		else
			string = g_strdup_printf("%c", value.uc);
		break;
	case EVMS_Type_Real32:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%.2a", value.r32);
		else
			string = g_strdup_printf("%.2f", value.r32);
		break;
	case EVMS_Type_Real64:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%.2a", value.r64);
		else
			string = g_strdup_printf("%.2f", value.r64);
		break;
	case EVMS_Type_Int8:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#hhx", value.i8);
		else
			string = g_strdup_printf("%hhd", value.i8);
		break;
	case EVMS_Type_Unsigned_Int8:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#hhx", value.ui8);
		else
			string = g_strdup_printf("%hhu", value.ui8);
		break;
	case EVMS_Type_Int16:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#hx", value.i16);
		else
			string = g_strdup_printf("%hd", value.i16);
		break;
	case EVMS_Type_Unsigned_Int16:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#hx", value.ui16);
		else
			string = g_strdup_printf("%hu", value.ui16);
		break;
	case EVMS_Type_Int:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#x", value.i);
		else
			string = g_strdup_printf("%d", value.i);
		break;
	case EVMS_Type_Int32:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#x", value.i32);
		else
			string = g_strdup_printf("%d", value.i32);
		break;
	case EVMS_Type_Unsigned_Int:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#x", value.ui);
		else
			string = g_strdup_printf("%u", value.ui);
		break;
	case EVMS_Type_Unsigned_Int32:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#x", value.ui32);
		else
			string = g_strdup_printf("%u", value.ui32);
		break;
	case EVMS_Type_Int64:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#"PRIx64, value.i64);
		else
			string = g_strdup_printf("%"PRId64, value.i64);
		break;
	case EVMS_Type_Unsigned_Int64:
		if (format == EVMS_Format_Hex)
			string = g_strdup_printf("%#"PRIx64, value.ui64);
		else
			string = g_strdup_printf("%"PRIu64, value.ui64);
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return string;
}

/**
 *	convert_string_to_value - takes a string and value_type_t to produce a value_t
 *	@string: the value in text form
 *	@type: the id of the data type
 *	@maxlen: maximum length of buffer if a string value
 *	@value: the address of where to store the value
 *
 *	This routine takes a string and a value_type_t identifier
 *	and produces the corresponding basic data type.
 */
gint convert_string_to_value(gchar *string, value_type_t type, gint maxlen, value_t *value)
{
	gint rc = 0;

	switch (type) {
	case EVMS_Type_Boolean:
		if (g_strcasecmp(_("True"), string) == 0 ||
			g_strcasecmp(_("Yes"), string) == 0)
			value->b = TRUE;
		else
			value->b = FALSE;
		break;
	case EVMS_Type_String:
		value->s = g_malloc0(maxlen + 1);
		if (string)
			strcpy(value->s, string);
		break;
	case EVMS_Type_Char:
	case EVMS_Type_Unsigned_Char:
		value->uc = *string;
		break;
	case EVMS_Type_Real32:
		value->r32 = strtod(string, NULL);
		rc = errno;
		break;
	case EVMS_Type_Real64:
		value->r64 = strtod(string, NULL);
		rc = errno;
		break;
	case EVMS_Type_Int8:
		value->i32 = strtol(string, NULL, 0);
		if (errno == 0 && (value->i32 < SCHAR_MIN || value->i32 > SCHAR_MAX))
			rc = ERANGE;
		else
			rc = errno;
		break;
	case EVMS_Type_Unsigned_Int8:
		value->ui32 = strtoul(string, NULL, 0);
		if (errno == 0 && value->ui32 > UCHAR_MAX)
			rc = ERANGE;
		else
			rc = errno;
		break;
	case EVMS_Type_Int16:
		value->i32 = strtol(string, NULL, 0);
		if (errno == 0 && (value->i32 < SHRT_MIN || value->i32 > SHRT_MAX))
			rc = ERANGE;
		else
			rc = errno;
		break;
	case EVMS_Type_Unsigned_Int16:
		value->ui32 = strtoul(string, NULL, 0);
		if (errno == 0 && value->ui32 > USHRT_MAX)
			rc = ERANGE;
		else
			rc = errno;
		break;
	case EVMS_Type_Int:
		value->i = strtol(string, NULL, 0);
		rc = errno;
		break;
	case EVMS_Type_Int32:
		value->i32 = strtol(string, NULL, 0);
		rc = errno;
		break;
	case EVMS_Type_Unsigned_Int:
		value->ui = strtoul(string, NULL, 0);
		rc = errno;
		break;
	case EVMS_Type_Unsigned_Int32:
		value->ui32 = strtoul(string, NULL, 0);
		rc = errno;
		break;
	case EVMS_Type_Int64:
		value->i64 = strtoll(string, NULL, 0);
		rc = errno;
		break;
	case EVMS_Type_Unsigned_Int64:
		value->ui64 = strtoull(string, NULL, 0);
		rc = errno;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
		rc = EINVAL;
	}
	return rc;
}

/**
 *	allocate_value_list - create a value list
 *	@count: number of entries in the list
 *	@type: the id of the data type
 *	@size: max string size
 *
 *	This routine creates a value list. If the list is a list of
 *	string types then it also allocates string buffers of the
 *	given size + 1.
 */
value_list_t *allocate_value_list(guint count, value_type_t type, guint size)
{
	value_list_t *vlist;

	vlist = g_malloc0(sizeof(value_list_t) + (count * sizeof(value_t)));

	if (type == EVMS_Type_String) {
		guint i;

		size++;
		for (i = 0; i < count; i++) {
			vlist->value[i].s = g_malloc(size);
			*(vlist->value[i].s) = '\0';
		}
	}
	return vlist;
}

/**
 *	duplicate_value - make a copy of a value_t
 *	@source_value: the union of the basic data types
 *	@type: the id of the data type
 *	@is_list: if the value is actually a value_list_t *
 *	@list_count: if a list, the actual allocated item count
 *	@maxlen: the maximum length of the string buffer
 *	@target_value: address of value_t to store duplicated contents
 *
 *	This routine takes a value and copies the contents
 *	to another value_t. If the value type indicates it
 *	is a string then the string is allocated of the max
 *	length and the string is duplicated. If the value
 *	type describes a value list then the list is duplicated.
 */
void duplicate_value(value_t source_value, value_type_t type, gboolean is_list,
			guint list_count, gint maxlen, value_t *target_value)
{
	/*
	 * Add one to maxlen as it is used for strings so always
	 * ensure one more byte for the terminator byte.
	 */
	maxlen += 1;

	if (is_list) {
		if (type == EVMS_Type_String) {
			guint i;

			target_value->list = allocate_value_list(list_count, type, maxlen);

			for (i = 0; i < source_value.list->count; i++) {
				strcpy(target_value->list->value[i].s, source_value.list->value[i].s);
			}
			target_value->list->count = source_value.list->count;
		} else {
			target_value->list = g_memdup(source_value.list,
				(sizeof(value_list_t) + (list_count * sizeof(value_t))));
		}
	} else if (type == EVMS_Type_String) {
		target_value->s = g_malloc(maxlen);
		strcpy(target_value->s, source_value.s);
	} else {
		memcpy(target_value, &source_value, sizeof(value_t));
	}
}

/**
 *	clear_value - erase/free the contents of a value_t
 *	@value: the address of the union of basic data types
 *	@type: the id of the data type
 *	@is_list: if the value is actually a value_list_t *
 *	@list_count: if this is a list, this is the allocated item count
 *
 *	This routine takes the address of a value and clears
 *	the value contents. If the value is a list of values
 *	then the list is freed. If the value contains a string
 *	then the string is freed.
 */
void clear_value(value_t *value, value_type_t type, gboolean is_list, guint list_count)
{
	if (is_list && value->list != NULL) {
		if (type == EVMS_Type_String) {
			guint i;

			for (i = 0; i < list_count; i++) {
				g_free(value->list->value[i].s);
			}
		}
		g_free(value->list);
	} else if (type == EVMS_Type_String) {
		g_free(value->s);
	}
	memset(value, 0, sizeof(value_t));
}

/**
 *	convert_value_to_float - convert a value_t to a floating point number
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *
 *	This routine takes a value (hopefully numeric) and converts
 *	it to a float.
 */
gfloat convert_value_to_float(value_t value, value_type_t type)
{
	gfloat float_value = 0.0;

	switch (type) {
	case EVMS_Type_Real32:
		float_value = value.r32;
		break;
	case EVMS_Type_Real64:
		float_value = value.r64;
		break;
	case EVMS_Type_Char:
		float_value = value.c * 1.0;
		break;
	case EVMS_Type_Unsigned_Char:
		float_value = value.uc * 1.0;
		break;
	case EVMS_Type_Int8:
		float_value = value.i8 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int8:
		float_value = value.ui8 * 1.0;
		break;
	case EVMS_Type_Int16:
		float_value = value.i16 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int16:
		float_value = value.ui16 * 1.0;
		break;
	case EVMS_Type_Int:
		float_value = value.i * 1.0;
		break;
	case EVMS_Type_Int32:
		float_value = value.i32 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int:
		float_value = value.ui * 1.0;
		break;
	case EVMS_Type_Unsigned_Int32:
		float_value = value.ui32 * 1.0;
		break;
	case EVMS_Type_Int64:
		float_value = value.i64 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int64:
		float_value = value.ui64 * 1.0;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return float_value;
}

/**
 *	convert_float_to_value - convert a float to a value_t
 *	@float_value: the float to convert to a value_t type
 *	@type: the id of the data type
 *	@value: addresss to store converted value type
 *
 *	This routine takes a float and converts it to a value_t.
 */
void convert_float_to_value(gfloat float_value, value_type_t type, value_t *value)
{
	switch (type) {
	case EVMS_Type_Real32:
		value->r32 = float_value;
		break;
	case EVMS_Type_Real64:
		value->r64 = float_value;
		break;
	case EVMS_Type_Char:
	case EVMS_Type_Unsigned_Char:
	case EVMS_Type_Int8:
	case EVMS_Type_Unsigned_Int8:
		value->ui8 = float_value;
		break;
	case EVMS_Type_Int16:
	case EVMS_Type_Unsigned_Int16:
		value->ui16 = float_value;
		break;
	case EVMS_Type_Int:
	case EVMS_Type_Unsigned_Int:
		value->ui = float_value;
		break;
	case EVMS_Type_Int32:
	case EVMS_Type_Unsigned_Int32:
		value->ui32 = float_value;
		break;
	case EVMS_Type_Int64:
	case EVMS_Type_Unsigned_Int64:
		value->ui64 = float_value;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
}

/**
 *	convert_value_to_double - convert a value_t to a double
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *
 *	This routine takes a value (hopefully numeric) and converts
 *	it to a double.
 */
gdouble convert_value_to_double(value_t value, value_type_t type)
{
	gdouble double_value = 0.0;

	switch (type) {
	case EVMS_Type_Real32:
		double_value = value.r32;
		break;
	case EVMS_Type_Real64:
		double_value = value.r64;
		break;
	case EVMS_Type_Char:
		double_value = value.c * 1.0;
		break;
	case EVMS_Type_Unsigned_Char:
		double_value = value.uc * 1.0;
		break;
	case EVMS_Type_Int8:
		double_value = value.i8 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int8:
		double_value = value.ui8 * 1.0;
		break;
	case EVMS_Type_Int16:
		double_value = value.i16 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int16:
		double_value = value.ui16 * 1.0;
		break;
	case EVMS_Type_Int:
		double_value = value.i * 1.0;
		break;
	case EVMS_Type_Int32:
		double_value = value.i32 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int:
		double_value = value.ui * 1.0;
		break;
	case EVMS_Type_Unsigned_Int32:
		double_value = value.ui32 * 1.0;
		break;
	case EVMS_Type_Int64:
		double_value = value.i64 * 1.0;
		break;
	case EVMS_Type_Unsigned_Int64:
		double_value = value.ui64 * 1.0;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return double_value;
}

/**
 *	convert_double_to_value - convert a double to a value_t
 *	@double_value: the double to convert to a value_t
 *	@type: the id of the data type
 *	@value: addresss to store converted value type
 *
 *	This routine takes a double and converts it to a value_t type.
 */
void convert_double_to_value(gdouble double_value, value_type_t type, value_t *value)
{
	switch (type) {
	case EVMS_Type_Real32:
		value->r32 = double_value;
		break;
	case EVMS_Type_Real64:
		value->r64 = double_value;
		break;
	case EVMS_Type_Char:
	case EVMS_Type_Unsigned_Char:
	case EVMS_Type_Int8:
	case EVMS_Type_Unsigned_Int8:
		value->ui8 = double_value;
		break;
	case EVMS_Type_Int16:
	case EVMS_Type_Unsigned_Int16:
		value->ui16 = double_value;
		break;
	case EVMS_Type_Int:
	case EVMS_Type_Unsigned_Int:
		value->ui = double_value;
		break;
	case EVMS_Type_Int32:
	case EVMS_Type_Unsigned_Int32:
		value->ui32 = double_value;
		break;
	case EVMS_Type_Int64:
	case EVMS_Type_Unsigned_Int64:
		value->ui64 = double_value;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
}

/**
 *	convert_value_to_ui64 - convert a value_t to a 64-bit unsigned
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *
 *	This routine takes a value (hopefully numeric) and converts
 *	it to a 64-bit unsigned value.
 */
u_int64_t convert_value_to_ui64(value_t value, value_type_t type)
{
	u_int64_t ui64_value = 0;

	switch (type) {
	case EVMS_Type_Real32:
		ui64_value = value.r32;
		break;
	case EVMS_Type_Real64:
		ui64_value = value.r64;
		break;
	case EVMS_Type_Char:
		ui64_value = value.c;
		break;
	case EVMS_Type_Unsigned_Char:
		ui64_value = value.uc;
		break;
	case EVMS_Type_Int8:
		ui64_value = value.i8;
		break;
	case EVMS_Type_Unsigned_Int8:
		ui64_value = value.ui8;
		break;
	case EVMS_Type_Int16:
		ui64_value = value.i16;
		break;
	case EVMS_Type_Unsigned_Int16:
		ui64_value = value.ui16;
		break;
	case EVMS_Type_Int:
		ui64_value = value.i;
		break;
	case EVMS_Type_Int32:
		ui64_value = value.i32;
		break;
	case EVMS_Type_Unsigned_Int:
		ui64_value = value.ui;
		break;
	case EVMS_Type_Unsigned_Int32:
		ui64_value = value.ui32;
		break;
	case EVMS_Type_Int64:
		ui64_value = value.i64;
		break;
	case EVMS_Type_Unsigned_Int64:
		ui64_value = value.ui64;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return ui64_value;
}

/**
 *	convert_sector_value_to_string - convert a sector value to a string
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@new_unit: address to write the unit type converted to
 *
 *	This routine takes a value (hopefully numeric) which is a value
 *	expressed in sectors and converts it to a string expressed in
 *	bytes, kilobytes, megabytes, gigabytes, or terabytes.
 */
gchar *convert_sector_value_to_string(value_t value, value_type_t type, value_unit_t *new_unit)
{
	u_int64_t size_in_sectors;

	size_in_sectors = convert_value_to_ui64(value, type);
	return make_sectors_readable_size_string(size_in_sectors, new_unit);
}

/**
 *	convert_sector_value_to_new_unit_string - returns sector size as human readable string
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@new_unit: address to write the unit type converted to
 *
 *	This routine takes a value (hopefully numeric) which is a value
 *	expressed in sectors and converts it to a string expressed in
 *	bytes, kilobytes, megabytes, gigabytes, or terabytes. This routine
 *	differs from convert_sector_value_to_string() in that it does not
 *	determine the unit to use but is given the unit to convert to.
 */
gchar *convert_sector_value_to_new_unit_string(value_t value, value_type_t type, value_unit_t new_unit)
{
	value_t converted_value;

	converted_value.r32 = convert_sector_value_to_float_with_unit(value, type, new_unit);
	return convert_value_to_string(converted_value, EVMS_Type_Real32, EVMS_Format_Normal);
}

/**
 *	get_unit_from_mnemonic - return value_unit_t for a unit string
 *	@mnemonic: the string containing word representing a unit measurement
 *
 *	This routine takes a string which is hopefully a word
 *	or mnemonic such as "bytes", "KB", "MB", "GB", or "TB"
 *	and if recognized returns the unit type this corresponds
 *	to.
 */
value_unit_t get_unit_from_mnemonic(gchar *mnemonic)
{
	value_unit_t unit;

	unit = EVMS_Unit_None;
	g_strstrip(mnemonic);

	if (*mnemonic == 'M' || *mnemonic == 'm')
		unit = EVMS_Unit_Megabytes;
	else if (*mnemonic == 'G' || *mnemonic == 'g')
		unit = EVMS_Unit_Gigabytes;
	else if (*mnemonic == 'B' || *mnemonic == 'b')
		unit = EVMS_Unit_Bytes;
	else if (*mnemonic == 'K' || *mnemonic == 'k')
		unit = EVMS_Unit_Kilobytes;
	else if (*mnemonic == 'T' || *mnemonic == 't')
		unit = EVMS_Unit_Terabytes;
	else if (*mnemonic == 'P' || *mnemonic == 'p')
		unit = EVMS_Unit_Petabytes;
	else if (*mnemonic == 'S' || *mnemonic == 's')
		unit = EVMS_Unit_Sectors;
	else
		log_warning("%s: mnemonic or word not recognized: %s.\n", __FUNCTION__, mnemonic);

	return unit;
}

/**
 *	convert_sector_value_to_float_with_unit - convert sector size to float
 *	@value: the value expressed in sectors
 *	@type: the value type, e.g. EVMS_Type_Int
 *	@conversion_unit: the unit of measure to convert to, e.g. KB, TB, etc.
 *
 *	This routine takes a value expressed in sector units and
 *	converts it to a real number that expresses the value in
 *	the conversion unit, i.e. KB, GB, TB, etc.
 */
gfloat convert_sector_value_to_float_with_unit(value_t value, value_type_t type, value_unit_t conversion_unit)
{
	gfloat new_size;

	new_size = convert_value_to_float(value, type);

	switch (conversion_unit) {
	case EVMS_Unit_Bytes:
		new_size *= EVMS_VSECTOR_SIZE;
		break;
	case EVMS_Unit_Kilobytes:
		new_size /= KB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Megabytes:
		new_size /= MB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Gigabytes:
		new_size /= GB_TO_SECTORS_MULTIPLIER;
		break;
        case EVMS_Unit_Terabytes:
		new_size /= TB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Petabytes:
		new_size /= PB_TO_SECTORS_MULTIPLIER;
		break;
	default:
		log_debug("%s: We are not going to convert this unit of measurement: %d\n",
			__FUNCTION__, conversion_unit);
		break;
	}
	return new_size;
}

/**
 *	select_readable_unit_for_sector_value - determine best unit for a sector value
 *	@value: the input size value
 *	@type: id of the size type, i.e. EVMS_Type_Real32 or EVMS_Type_Unsigned_Int64
 *
 *	This routine takes a value in sector units and determines the
 *	highest unit of measure (KB,MB,GB) that we can use to express
 *	the value in a more meaningful way.
 */
value_unit_t select_readable_unit_for_sector_value(value_t value, value_type_t type)
{
	gint i;
	u_int64_t value_as_ui64, divisor;
	gdouble result;
	value_unit_t unit = EVMS_Unit_Sectors;

	value_as_ui64 = convert_value_to_ui64(value, type);

	for (i = 0; i < MAX_UNITS; i++) {
		divisor = conversion_factor[i];
		result = value_as_ui64 / divisor;

		if (result > 0.0)
			unit = units[i];
		else
			break;
	}
	return unit;
}

/**
 *	convert_size_value_to_sector_value - convert a size value to a sector value
 *	@size_value: the input size value
 *	@size_type: id of the size type, i.e. EVMS_Type_Real32 or EVMS_Type_Unsigned_Int64
 *	@size_unit: the size value unit (hopefully bytes, KB, MB, GB, TB, or PB)
 *	@sec_value: address to write the size_value as a value in sector units
 *	@sec_type: id of the sector value type, i.e. EVMS_Type_Int64 or EVMS_Type_Unsigned_Int
 *
 *      This routine takes a value representing a size in some user understandable
 *      unit of measurement such as KB, TB or GB and typically either as a real
 *      or integer number and it gets converted to a value expressed in terms of
 *      EVMS_Unit_Sectors unit of measurement.
 */
void convert_size_value_to_sector_value(value_t size_value, value_type_t size_type,
					value_unit_t size_unit,
					value_t *sec_value, value_type_t sec_type)
{
	gdouble size_as_double, size_in_sectors;

	/*
	 * For the sake of simplicity. Do calculations with reals then
	 * convert to a integer value if necessary.
	 */
	size_as_double = convert_value_to_double(size_value, size_type);

	switch (size_unit) {
	case EVMS_Unit_Bytes:
		size_in_sectors = size_as_double / EVMS_VSECTOR_SIZE;
		break;
	case EVMS_Unit_Kilobytes:
		size_in_sectors = size_as_double * KB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Megabytes:
		size_in_sectors = size_as_double * MB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Gigabytes:
		size_in_sectors = size_as_double * GB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Terabytes:
		size_in_sectors = size_as_double * TB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Petabytes:
		size_in_sectors = size_as_double * PB_TO_SECTORS_MULTIPLIER;
		break;
	default:
		log_debug("%s: We are not going to convert this unit of measurement: %d\n",
			__FUNCTION__, size_unit);
	case EVMS_Unit_Sectors:
		size_in_sectors = size_as_double;
		break;
	}
	convert_double_to_value(size_in_sectors, sec_type, sec_value);
}

/**
 *	convert_float_to_sector_value - convert a float size to a sector size
 *	@size: the sector unit size as a real number
 *	@type: the type of the sector value, e.g. EVMS_Type_Int64
 *	@unit: the unit of the size, e.g. KB, TB, etc.
 *	@value: address to write sector value to
 *
 *	This routine takes a float that expresses a size unit
 *	(KB,MB,TB,etc.) and converts it to a value_t of value_type_t
 *	expressed in a EVMS_Unit_Sectors unit of measurement.
 */
void convert_float_to_sector_value(gfloat size, value_type_t type, value_unit_t unit, value_t *value)
{
	gfloat size_in_sectors;

	switch (unit) {
	case EVMS_Unit_Bytes:
		size_in_sectors = size / EVMS_VSECTOR_SIZE;
		break;
	case EVMS_Unit_Kilobytes:
		size_in_sectors = size * KB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Megabytes:
		size_in_sectors = size * MB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Gigabytes:
		size_in_sectors = size * GB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Terabytes:
		size_in_sectors = size * TB_TO_SECTORS_MULTIPLIER;
		break;
	case EVMS_Unit_Petabytes:
		size_in_sectors = size * PB_TO_SECTORS_MULTIPLIER;
		break;
	default:
		log_debug("%s: We are not going to convert this unit of measurement: %d\n",
			__FUNCTION__, unit);
	case EVMS_Unit_Sectors:
		size_in_sectors = size;
		break;
	}
	convert_float_to_value(size_in_sectors, type, value);
}

/**
 *	convert_size_string_to_sector_value - take a numeric size string and return sector value_t
 *	@string: the string containing a size, i.e. "10 MB" or "1.4GB"
 *	@unit: unit of value (superseded if unit mnemonic found)
 *	@type_out: value_type_t to convert the value to on return
 *	@value_out: address to write sector value to
 *
 *	This routine takes a string which is hopefully a numeric
 *	followed by the word "bytes", "KB", "MB", "GB", or "TB"
 *	and strips the word off to convert the string to a numeric.
 *	The numeric is then converted to a sector multiple based
 *	on the trailing mnemonic unit value.
 */
void convert_size_string_to_sector_value(gchar *string, value_unit_t unit,
					value_type_t type_out, value_t *value_out)
{
	value_out->i64 = 0;

	if (string != NULL) {
		guint length = 0;
		gchar *number;
		value_t value_in;
		value_type_t type_in;
		struct lconv * locale_conv = localeconv();
		char decimal_point = '.';

		if (locale_conv != NULL) {
			if (locale_conv->decimal_point != NULL) {
				if (*locale_conv->decimal_point != '\0') {
					decimal_point = *locale_conv->decimal_point;
				}
			}
		}

		/*
		 * Copy digits, blank space and decimal point to temp string
		 * until we hit an alpha character.
		 */
		number = g_malloc(strlen(string) + 1);
		type_in = EVMS_Type_Unsigned_Int64;
		value_in.i64 = 0;

		while (*string && (isdigit(*string) || *string == decimal_point || isspace(*string))) {
			if (*string == decimal_point)
				type_in = EVMS_Type_Real64;

			number[length] = *string;
			length++;
			string++;
		}
		number[length] = '\0';
		convert_string_to_value(number, type_in, 256, &value_in);

		/*
		 * If there is something remaining in the string check to see
		 * if it is a recognizable size measurement and if so, set the
		 * the unit to this so we can convert it to a sector unit value
		 * of the expected output value type.
		 */
		if (*string != '\0')
			unit = get_unit_from_mnemonic(string);

		convert_size_value_to_sector_value(value_in, type_in, unit, value_out, type_out);
		g_free(number);
	}
}

/**
 *	convert_value_to_string_with_unit - create string with unit word for a value
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@unit: the value unit measurement
 *	@format: the format for the the display value
 *
 *	This routine takes a value and converts it to a string with
 *	the short form of the unit value concatenated to the string.
 */
gchar *convert_value_to_string_with_unit(value_t value, value_type_t type,
					value_unit_t unit, value_format_t format)
{
	gchar *string;
	gchar *unit_string;

	unit_string = make_unit_readable_string(unit, TRUE);

	if (unit_string != NULL) {
		gchar *value_string;

		value_string = convert_value_to_string(value, type, format);
		string = g_strjoin(" ", value_string, unit_string, NULL);

		g_free(unit_string);
		g_free(value_string);
	} else {
		string = convert_value_to_string(value, type, format);
	}
	return string;
}

/**
 *	make_value_readable_string - return a value as a string with a unit
 *	@value: the union of the basic data types
 *	@type: the id of the data type
 *	@unit: the value unit measurement
 *	@format: the format for the the display value
 *	@ok_to_convert: TRUE if we are allowed to convert to something readable
 *
 *	This routine takes a value and converts it to a string with
 *	the short form of the unit value concatenated to the string.
 *	If ok_to_convert is TRUE, it converts it first to a more
 *	readable form, e.g. sectors or bytes to MB or GB.
 */
gchar *make_value_readable_string(value_t value, value_type_t type, value_unit_t unit,
					value_format_t format, gboolean ok_to_convert)
{
	gchar *string;

	switch (unit) {
	case EVMS_Unit_Sectors:
		if (ok_to_convert)
			string = make_sectors_readable_string(convert_value_to_ui64(value, type));
		else
			string = convert_value_to_string_with_unit(value, type, unit, format);
		break;
	case EVMS_Unit_Kilobytes:
		if (ok_to_convert)
			string = make_kbytes_readable_string(convert_value_to_ui64(value, type));
		else
			string = convert_value_to_string_with_unit(value, type, unit, format);
		break;
	case EVMS_Unit_Bytes: {
		u_int64_t bytes = convert_value_to_ui64(value, type);

		/*
		 * While admittedly not very accurate we will convert
		 * the bytes to a sector unit multiple to display in
		 * KB, MB, etc. if we are allowed to convert and the
		 * number of bytes is greater than 64K.
		 */
		if (ok_to_convert && bytes > 65535)
			string = make_sectors_readable_string(bytes / EVMS_VSECTOR_SIZE);
		else
			string = convert_value_to_string_with_unit(value, type, unit, format);
		}
		break;
	default:
		string = convert_value_to_string_with_unit(value, type, unit, format);
		break;
	}
	return string;
}

/**
 *	validate_numeric_string_characters - check a string as a numeric
 *	@string: the value in text form
 *
 *	This routine takes a string and checks it for a proper
 *	numeric. Acceptable characters include 0-9, A-F, a-f,
 *	X and x. A + or - is acceptable as the first non-whitespace
 *	character. Oh yeah, all whitespace is ignored.
 */
gint validate_numeric_string_characters(gchar *string)
{
	gint i, len, rc = 0;
	gchar *stripped;

	stripped = g_strstrip(string);
    	len = strlen(stripped);

	for (i = 0; i < len; i++) {
		if (isdigit(stripped[i]))
			continue;
		else if (i > 1 && isxdigit(stripped[i]) &&
				(stripped[1] == 'x' || stripped[1] == 'X') && stripped[0] == '0')
			continue;
		else if (isspace(stripped[i]))
			continue;
		else if (i == 0 && (stripped[0] == '+' || stripped[0] == '-'))
			continue;
		else if (i == 1 && (stripped[1] == 'x' || stripped[1] == 'X') && stripped[0] == '0')
			continue;
		else {
			rc = EINVAL;
			break;
		}
	}
	return rc;
}

/**
 *	validate_string_as_numeric_value - validate a string as numeric
 *	@string: the value in text form
 *	@type: the id of the data type
 *
 *	This routine takes a string and a value_type_t identifier
 *	and validates that the string as numeric is within bounds
 *	for the basic numeric type it represents.
 */
gchar *validate_string_as_numeric_value(gchar *string, value_type_t type)
{
	gchar *msg = NULL;
	gchar *number_buffer1;
	gchar *number_buffer2;

	if (type >= EVMS_Type_Int && type <= EVMS_Type_Unsigned_Int64) {
		gint rc = 0;
		value_t value;

		/*
		 * Do a preliminary scan of the string for non-valid characters.
		 * We accept 0-9, A-F, a-f, X or x. + or - is valid for the first
		 * character only. We ignore whitespace as we do the scan/check.
		 */
		rc = validate_numeric_string_characters(string);

		/*
		 * Use convert_to_string_to_value() to do the bounds check on the numeric
		 * for us. If we get a range error, produce a acceptable values message
		 * for the given type.
		 */

		if (rc == 0)
			rc = convert_string_to_value(string, type, 0, &value);

		if (rc != 0) {
			switch (type) {
			case EVMS_Type_Int8:
				msg = g_strdup_printf(_("Valid values are %hhd to %hhd."), SCHAR_MIN, SCHAR_MAX);
				break;
			case EVMS_Type_Unsigned_Int8:
				msg = g_strdup_printf(_("Valid values are %hhu to %hhu."), 0, UCHAR_MAX);
				break;
			case EVMS_Type_Int16:
				msg = g_strdup_printf(_("Valid values are %hd to %hd."), SHRT_MIN, SHRT_MAX);
				break;
			case EVMS_Type_Unsigned_Int16:
				msg = g_strdup_printf(_("Valid values are %hu to %hu."), 0, USHRT_MAX);
				break;
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
				msg = g_strdup_printf(_("Valid values are %d to %d."), INT_MIN, INT_MAX);
				break;
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				msg = g_strdup_printf(_("Valid values are %u to %u."), 0, UINT_MAX);
				break;
			case EVMS_Type_Int64:
				number_buffer1 = g_strdup_printf("%"PRId64, INT64_MIN);
				number_buffer2 = g_strdup_printf("%"PRId64, INT64_MAX);
				msg = g_strdup_printf(_("Valid values are %s to %s."), number_buffer1, number_buffer2);
				g_free(number_buffer1);
				g_free(number_buffer2);
				break;
			case EVMS_Type_Unsigned_Int64:
				number_buffer1 = g_strdup_printf("%"PRIu64, UINT64_MAX);
				msg = g_strdup_printf(_("Valid values are %u to %s."), 0, number_buffer1);
				g_free(number_buffer1);
				break;
			default:
				log_debug("%s: Gave no message for a numeric type of %d.\n", __FUNCTION__, type);
			}
		}
	}
	return msg;
}

/**
 *	values_are_equal - compare two values for equality
 *	@value1: the first value
 *	@value2: the second value
 *	@type: the id of the data type of the values
 *
 *	This routine compares two values of the same type for equality.
 */
gboolean values_are_equal(value_t value1, value_t value2, value_type_t type)
{
	gboolean result = FALSE;

	switch (type) {
	case EVMS_Type_String:
		if (value1.s == value2.s)
			result = TRUE;
		else if (value1.s && value2.s)
			result = g_strcasecmp(value1.s, value2.s) == 0;
		break;
	case EVMS_Type_Char:
	case EVMS_Type_Unsigned_Char:
		result = value1.uc == value2.uc;
		break;
	case EVMS_Type_Real32:
		result = value1.r32 == value2.r32;
		break;
	case EVMS_Type_Real64:
		result = value1.r64 == value2.r64;
		break;
	case EVMS_Type_Int8:
	case EVMS_Type_Unsigned_Int8:
		result = value1.ui8 == value2.ui8;
		break;
	case EVMS_Type_Int16:
	case EVMS_Type_Unsigned_Int16:
		result = value1.ui16 == value2.ui16;
		break;
	case EVMS_Type_Int:
	case EVMS_Type_Unsigned_Int:
		result = value1.ui == value2.ui;
		break;
	case EVMS_Type_Int32:
	case EVMS_Type_Unsigned_Int32:
		result = value1.ui32 == value2.ui32;
		break;
	case EVMS_Type_Int64:
	case EVMS_Type_Unsigned_Int64:
		result = value1.ui64 == value2.ui64;
		break;
	default:
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return result;
}

/**
 *	value_in_value_list - determine if value is in value list
 *	@value: the value
 *	@vlist: the value list to search through
 *	@type: the id of the data type of the values
 *
 *	This routine determines if a value like the one given exists
 *	in the given value list.
 */
gboolean value_in_value_list(value_t value, value_list_t *vlist, value_type_t type)
{
	guint i;
	gboolean is_value_in_list = FALSE;

	for (i = 0; i < vlist->count; i++) {
		if (values_are_equal(value, vlist->value[i], type)) {
			is_value_in_list = TRUE;
			break;
		}
	}
	return is_value_in_list;
}

/**
 *	clamp_value - fit value within a min and max range
 *	@value: the input value to check
 *	@range: the pointer to the range structure containing min & max
 *	@type: the id of the data type of the values
 *
 *	This routine ensures that a value falls between a min
 *	and max range. If below the min, it is set equal to
 *	the min. If above the max, it is set equal to the max.
 */
value_t clamp_value(value_t value, value_range_t *range, value_type_t type)
{
	value_t ret_value;

	switch (type) {
	case EVMS_Type_Char:
		ret_value.c = CLAMP(value.c, range->min.c, range->max.c);
		break;
	case EVMS_Type_Unsigned_Char:
		ret_value.uc = CLAMP(value.uc, range->min.uc, range->max.uc);
		break;
	case EVMS_Type_Real32:
		ret_value.r32 = CLAMP(value.r32, range->min.r32, range->max.r32);
		break;
	case EVMS_Type_Real64:
		ret_value.r64 = CLAMP(value.r64, range->min.r64, range->max.r64);
		break;
	case EVMS_Type_Int8:
		ret_value.i8 = CLAMP(value.i8, range->min.i8, range->max.i8);
		break;
	case EVMS_Type_Unsigned_Int8:
		ret_value.ui8 = CLAMP(value.ui8, range->min.ui8, range->max.ui8);
		break;
	case EVMS_Type_Int16:
		ret_value.i16 = CLAMP(value.i16, range->min.i16, range->max.i16);
		break;
	case EVMS_Type_Unsigned_Int16:
		ret_value.ui16 = CLAMP(value.ui16, range->min.ui16, range->max.ui16);
		break;
	case EVMS_Type_Int:
		ret_value.i = CLAMP(value.i, range->min.i, range->max.i);
		break;
	case EVMS_Type_Unsigned_Int:
		ret_value.ui = CLAMP(value.ui, range->min.ui, range->max.ui);
		break;
	case EVMS_Type_Int32:
		ret_value.i32 = CLAMP(value.i32, range->min.i32, range->max.i32);
		break;
	case EVMS_Type_Unsigned_Int32:
		ret_value.ui32 = CLAMP(value.ui32, range->min.ui32, range->max.ui32);
		break;
	case EVMS_Type_Int64:
		ret_value.i64 = CLAMP(value.i64, range->min.i64, range->max.i64);
		break;
	case EVMS_Type_Unsigned_Int64:
		ret_value.ui64 = CLAMP(value.ui64, range->min.ui64, range->max.ui64);
		break;
	default:
		ret_value = value;
		log_warning("%s: Value type of %d was not handled.\n", __FUNCTION__, type);
	}
	return ret_value;
}

/**
 *	snap_value_to_constraint_increment - attempt to snap the given value to the closest range step
 *	@value: the input value to check
 *	@range: the pointer to the range structure containing min, max & increment
 *	@type: the id of the data type of the values
 *
 *	This routine checks the value to see if it is an increment multiple in the given
 *	range. If not, it attempts to snap it to the closest increment multiple.
 */
value_t snap_value_to_constraint_increment(value_t value, value_range_t *range, value_type_t type)
{
	value_t snap;
	gfloat inc, tmp, val, org_val, lower;
	
	val = convert_value_to_float(value, type);
	org_val = val;

	inc = convert_value_to_float(range->increment, type);
	lower = convert_value_to_float(range->min, type);
	tmp = (val - lower) / inc;

	if (tmp - floorf(tmp) < ceilf(tmp) - tmp)
		val = lower + floorf(tmp) * inc;
	else
		val = lower + ceilf(tmp) * inc;

	if (fabsf(val - org_val) > 1e-5)
		convert_float_to_value(val, type, &snap);
	else
		snap = value;
	
	return snap;
}
