/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <string.h>
#include <time.h>
#include <frontend.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "task.h"
#include "options.h"
#include "enum.h"
#include "readable.h"
#include "views.h"
#include "callbacks.h"
#include "logging.h"

/**
 *	set_selected_object - set the selected object for a task
 *	@task: the task handle
 *	@handle: the object handle
 *
 *	This routine attempts to set the selected object for a task.
 */
int set_selected_object(task_handle_t task, object_handle_t handle)
{
	gint rc;
	gchar mem[sizeof(handle_array_t) + sizeof(object_handle_t)];
	handle_array_t *ha = (handle_array_t *) mem;
	declined_handle_array_t *declined_list;

	ha->count = 1;
	ha->handle[0] = handle;

	rc = evms_set_selected_objects(task, ha, &declined_list, NULL);
	if (rc == 0 && declined_list) {
		if (declined_list->count > 0)
			rc = EINVAL;
		evms_free(declined_list);
	}
	return rc;
}

/**
 *	is_task_acceptable_object - checks to see if object acceptable to plugin for certain task
 *	@plugin: the plugin to check against
 *	@handle: the handle of the object to see if acceptable
 *	@action: the task action
 *
 *	This routine checks to see if the given object is an
 *      acceptable object for the given task of the given plugin.
 */
gboolean is_task_acceptable_object(plugin_handle_t plugin, object_handle_t object, task_action_t action)
{
	gboolean result = FALSE;
	task_handle_t task;

	if ((evms_create_task(plugin, action, &task)) == 0) {
		result = set_selected_object(task, object) == 0;
		evms_destroy_task(task);
	}
	return result;
}

/**
 *	acceptable_objects_available - checks to see if any acceptable objects exist for a task
 *	@task: the task handle
 *
 *	This routine checks to see if any acceptable objects are available for a given task.
 */
gboolean acceptable_objects_available(task_handle_t task)
{
	int rc;
	gboolean result = FALSE;
	handle_array_t *acceptable_list;

	rc = evms_get_acceptable_objects(task, &acceptable_list);
	if (rc != 0)  {
		log_error("%s: evms_get_acceptable_objects() returned error code %d.\n",
				__FUNCTION__, rc);
	} else {
		if (acceptable_list->count > 0)
			result = TRUE;
		evms_free(acceptable_list);
	}
	return result;
}

/**
 *	can_plugin_do_action - check to see if plugin can perform task action
 *	@plugin: the plugin to check for the task action
 *	@action: the task action code
 *
 *	This routine attempts to see if the plugin could attempt the
 *	given task action. We basically make a determination by
 *	creating a task, checking the acceptable object count and the
 *	minimum selection count.
 */
gboolean can_plugin_do_action(plugin_handle_t plugin, task_action_t action)
{
	task_handle_t task;
	gboolean result = FALSE;

	if ((evms_create_task(plugin, action, &task)) == 0) {
		handle_array_t *acceptable_list;

		if ((evms_get_acceptable_objects(task, &acceptable_list) == 0) &&
				acceptable_list->count > 0) {
			guint min;
			guint max;

			if ((evms_get_selected_object_limits(task, &min, &max) == 0) &&
					acceptable_list->count >= min) {
				result = TRUE;
			}
		}
		evms_free(acceptable_list);
		evms_destroy_task(task);
	}
	return result;
}

/**
 *	get_task_action_string - return the string describing a task command
 *	@action: the task action
 *
 *	This routine returns a string corresponding to the task action given.
 */
gchar *get_task_action_string(task_action_t action)
{
	gchar *action_text = NULL;

	switch (action) {
	case EVMS_Task_Create:
	case EVMS_Task_Create_Container:
		action_text = g_strdup(_("Create"));
		break;
	case EVMS_Task_Assign_Plugin:
		action_text = g_strdup(_("Add"));
		break;
	case EVMS_Task_Expand:
		action_text = g_strdup(_("Expand"));
		break;
	case EVMS_Task_Shrink:
		action_text = g_strdup(_("Shrink"));
		break;
	case EVMS_Task_Slide:
		action_text = g_strdup(_("Slide"));
		break;
	case EVMS_Task_Move:
		action_text = g_strdup(_("Move"));
		break;
	case EVMS_Task_fsck:
		action_text = g_strdup(_("Check Filesystem"));
		break;
	case EVMS_Task_mkfs:
		action_text = g_strdup(_("Make Filesystem"));
		break;
	default:
		log_debug("%s: Unknown task action %d received.\n", __FUNCTION__, action);
		break;
	}
	return action_text;
}

/**
 *	get_action_string_for_task - return the string describing a task command
 *	@task: the task context handle
 *
 *	This routine returns a string corresponding to the task
 *	operation at hand, i.e. "Create", "Expand", etc.
 */
gchar *get_action_string_for_task(task_handle_t task)
{
	task_action_t action;
	gchar *action_text = NULL;

	if (evms_get_task_action(task, &action) == 0)
		action_text = get_task_action_string(action);

	return action_text;
}

/**
 *	make_handle_array_from_selection_list - make an EVMS handle array from clist selection list
 *	@selections - linked list with handles corresponding to rows selected
 *
 *	This routine takes a selection list from a clist and extracts the data
 *	(object handles associated with the list row) and generates a handle
 *	array filled with valid (non-zero) handles.
 */
handle_array_t *make_handle_array_from_selection_list(GList *selections)
{
	gint count, i;
	handle_array_t *harray;

	count = g_list_length(selections);
	harray = g_malloc0(sizeof(handle_array_t) + (sizeof(object_handle_t) * count));

	for (i = 0; i < count; i++) {
		object_handle_t handle;
		struct clist_item *item;

		item = g_list_nth_data(selections, i);
		handle = GPOINTER_TO_UINT(item->user_data);

		if (handle != 0) {
			harray->handle[harray->count] = handle;
			harray->count++;
		}
	}

	return harray;
}

/**
 *	are_handle_arrays_equal - compares two handles arrays to see if they match
 *	array1: the first handle array
 *	array2: the second handle array
 *
 *	This routine takes two handle arrays and checks to see if the counts
 *	match. If they do, we memcmp both to see if they contain the exact
 *	same handles. We return TRUE if the arrays are a match.
 */
gboolean are_handle_arrays_equal(handle_array_t *array1, handle_array_t *array2)
{
	gboolean arrays_equal = FALSE;

	if (array1 != NULL && array2 != NULL && array1->count == array2->count) {
		gint array_size;

		array_size = sizeof(handle_array_t) +
			(array1->count * sizeof(object_handle_t));
		arrays_equal = memcmp(array1, array2, array_size) == 0;
	}
	return arrays_equal;
}

/**
 *	object_is_only_available - check if only one acceptable object and matches given handle
 *	@task: the task handle
 *	@handle: the object handle to check
 *
 *	This routine checks to see if the acceptable objects list contains one
 *	object and whether the handle matches that of the given handle.
 */
inline gboolean object_is_only_available(task_handle_t task, engine_handle_t handle)
{
	gboolean result = FALSE;
	handle_array_t *acceptable_list;

	if (evms_get_acceptable_objects(task, &acceptable_list) == 0 &&
        	acceptable_list->count == 1  && acceptable_list->handle[0] == handle) {
			result = TRUE;
	}
	return result;
}

/**
 *	determine_task_requirements - determine dialogs needed for a task
 *	@task: the task handle
 *	@handle: the handle to thing to initiate task with
 *	@option_count: the number of options the task has
 *	@objects_min: the minimum number of acceptable objects needed
 *	@objects_max: the maximum number of acceptable objects that can be selected
 *
 *	This routine determines what requirements are needed by the task in order
 *	to properly invoked. We check option count and the min and max for the
 *	acceptable objects as well. This information is used in order to present
 *	the proper user panels to gather the selections and configuration for the
 *	task.
 *
 *	While we are here, we also set the selected object if we find that there
 *	is only one and the plugin can accept at least one to continue.
 */
task_requirements_t determine_task_requirements(task_handle_t task, engine_handle_t handle,
						int option_count, int objects_min,
						int objects_max)
{
	task_requirements_t requirements;

	if (objects_max != 0 && option_count == 0) {
		if (objects_min <= 1 && object_is_only_available(task, handle) == TRUE) {
			/*
			 * It looks like we have only one acceptable object
			 * and it's the one for the handle given. Make it the
			 * selected object and simply provide a confirmation
			 * dialog for the operation.
			 */
			if (set_selected_object(task, handle) == 0) {
				requirements = TASK_REQUIRES_NO_ADDITIONAL_ARGS;
			} else {
				requirements = TASK_REQUIRES_OBJECTS_ONLY;
			}
		} else {
			requirements = TASK_REQUIRES_OBJECTS_ONLY;
		}
	} else if (objects_max == 0 && option_count > 0) {
		requirements = TASK_REQUIRES_OPTIONS_ONLY;
	} else if (objects_max != 0 && option_count > 0) {
		if (objects_min <= 1 && object_is_only_available(task, handle) == TRUE) {
			/*
			 * It looks like we have only one acceptable object
			 * and the plugin can accept either any minimum or a
			 * minimum of one. Go ahead and set our only selection
			 * and skip ahead to option configuration.
			 */
			if (set_selected_object(task, handle) == 0) {
				requirements = TASK_REQUIRES_OPTIONS_ONLY;
			} else {
				requirements = TASK_REQUIRES_OBJECTS_AND_OPTIONS;
			}
		} else {
			requirements = TASK_REQUIRES_OBJECTS_AND_OPTIONS;
		}
	} else {
		requirements = TASK_REQUIRES_NO_ADDITIONAL_ARGS;
	}
	return requirements;
}

/**
 *	create_task - create a task and determine requirements (acceptable objects/options)
 *	@handle: the handle of the reference object or plugin to initiate this task
 *	@action: the task function number
 *	@requirements: the address to store the returned task requirements
 *	@task: the address to store the returned task handle
 *	@objects_min: address to store the minimum objects required
 *	@objects_max: address to store the maximum objects expected
 *
 *	This routine creates a task and determines, given a handle and task action,
 *	whether the task will need to present an acceptable objects selection list
 *	and/or an options configuration panel or none at all.
 */
int create_task(engine_handle_t handle, task_action_t action,
		task_requirements_t *requirements, task_handle_t *task,
		u_int32_t *objects_min, u_int32_t *objects_max)
{
	int rc, option_count;

	*requirements = TASK_REQUIREMENTS_UNKNOWN;

	rc = evms_create_task(handle, action, task);
	if (rc == 0) {
		rc = evms_get_selected_object_limits(*task, objects_min, objects_max);

		if (rc == 0) {
			rc = evms_get_option_count(*task, &option_count);

			if (rc == 0) {
				*requirements = determine_task_requirements(*task,
									handle,
									option_count,
									*objects_min,
									*objects_max);
			} else {
				evms_destroy_task(*task);
				log_error("%s: evms_get_option_count() returned error code %d.\n",
						__FUNCTION__, rc);
			}
		} else {
			evms_destroy_task(*task);
			log_error("%s: evms_get_selected_object_limits() returned error code %d.\n",
					__FUNCTION__, rc);
		}
	} else {
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	}

	log_debug("%s: The task requirement code is %d\n", __FUNCTION__, *requirements);

	return rc;
}

/**
 *	format_resulting_item - return column strings for a row in the resulting objects clist
 *	@handle: the thing handle
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the resulting objects clist containing the size and name of an object.
 */
int format_resulting_item(object_handle_t handle, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		switch (object->type) {
		case DISK:
		case REGION:
		case SEGMENT:
		case EVMS_OBJECT:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
			break;
		case CONTAINER:
			g_ptr_array_add(text, g_strdup(object->info.container.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.container.size));
			break;
		case VOLUME:
			g_ptr_array_add(text, g_strdup(object->info.volume.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.volume.vol_size));
			break;
		default:
			rc = EPERM;
			log_warning("%s: Type %u was not processed.\n", __FUNCTION__, object->type);
			break;
		}
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_resulting_objects_clist - populate the the resulting objects' window clist
 *	@clist: the resulting objects window clist
 *	@resulting_objects: the address of an array containing handles for objects produced by task
 *
 *	This routine populates the resulting objects window clist with the objects that were
 *	produced by the invocation of a task.
 */
void populate_resulting_objects_clist(struct clist *clist, handle_array_t *resulting_objects)
{
	int i;

	for (i = 0; i < resulting_objects->count; i++) {
		GPtrArray *text;

		text = g_ptr_array_new();
		if (format_resulting_item(resulting_objects->handle[i], text) == 0)
			append_item(clist, text, NULL, NULL);
		g_ptr_array_free(text, TRUE);
	}
}

/**
 *	create_resulting_objects_dialog - create a dialog listing the objects that resulted from a task
 *	@dialog: the task dialog that initiated the creation of this one
 *	@verb: the command verb
 *	@resulting_objects: handle array of resulting objects
 *
 *	This routine creates a list dialog populated with the objects produced by a task.
 */
void create_resulting_objects_dialog(struct dialog_window *dialog, char *verb, handle_array_t *resulting_objects)
{
	char *prompt;
	struct selwin *list_dialog;

	prompt = g_strdup_printf(_("%s completed successfully producing the objects listed above"), verb);

	list_dialog = create_list_dialog(_("New Objects"), NULL, prompt, 2, NULL);

	set_clist_column_count(list_dialog->clist, 2);
	set_clist_column_info(list_dialog->clist, 0, calc_clist_column_width(list_dialog->clist, 0.50),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(list_dialog->clist, 1, calc_clist_column_width(list_dialog->clist, 0.12),
				get_clist_column_end(list_dialog->clist, 0),
				CLIST_TEXT_JUSTIFY_RIGHT);

	print_clist_column_title(list_dialog->clist, 0, _("Name"));
	print_clist_column_title(list_dialog->clist, 1, _("Size"));

	populate_resulting_objects_clist(list_dialog->clist, resulting_objects);

	append_dialog_to_list((struct dialog_window *)list_dialog, dialog->list);
	dialog->list->current = get_next_dialog(dialog);

	g_free(prompt);
}

/**
 *	get_task_from_dialog_data - retrieve the task handle from the hash table associated with dialog
 *	@dialog: the task dialog containing the task handle stored in a hash table
 *
 *	This routine retrieves the task handle stored in the hash table associated with the
 *	task dialog.
 */
inline task_handle_t get_task_from_dialog_data(struct dialog_window *dialog)
{
	task_handle_t task;
	GHashTable *hash_table;

	hash_table = dialog->user_data;
	task = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "task"));

	return task;
}

/**
 *	task_action_button_activated - callback invoke when task action button activated
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the task action button in a task dialog
 *	window is activated.
 */
int task_action_button_activated(struct menu_item *item)
{
	int rc;
	char *verb;
	task_handle_t task;
	GHashTable *hash_table;
	handle_array_t *resulting_objects;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	task = get_task_from_dialog_data(dialog);
	verb = g_hash_table_lookup(hash_table, "verb");

	enable_message_queuing();
	rc = evms_invoke_task(task, &resulting_objects);
	disable_message_queuing();
	display_queued_messages();

	if (rc == 0) {
		if (resulting_objects != NULL && resulting_objects->count > 0)
			create_resulting_objects_dialog(dialog, verb, resulting_objects);
		else
			dialog->status = DLG_STATUS_CLOSING;
		evms_free(resulting_objects);
	}

	update_status(verb, rc);
	if (rc == 0)
		refresh_views();
	return 0;
}

/**
 *	create_task_confirmation_dialog - create a confirmation dialog for a task with no options or objects
 *	@list: the dialog list for the sequence of dialogs
 *	@task: the task handle
 *	@title: the title for the operation
 *	@verb: the verb to use to complete the operation
 *	@help: detailed help for the action
 *
 *	This routine creates a dialog window for a task that has no selectable objects or options.
 *	In other words, invoking the task is all that is necessary.
 **/
struct dialog_window *create_task_confirmation_dialog(struct dialog_list *list, task_handle_t task,
							gchar *title, gchar *verb, gchar *help)
{
	char *prompt;
	struct dialog_window *dialog;

	prompt = g_strdup_printf(_("%s or cancel the operation?"), verb);

	dialog = create_dialog_window(title, help,
				(dialog_event_handler)NULL,
				(dialog_show_func)NULL,
				(dialog_delete_func)NULL,
				(dialog_delete_cb)NULL,
				verb,
				(menuitem_activate_cb)task_action_button_activated,
				NULL, NULL, NULL);

	print_centered(dialog->win, getmaxy(dialog->win) / 2, prompt);
	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);

	g_free(prompt);

	return dialog;
}

/**
 *	create_task_options_dialog - create a dialog to allow setting task configuration options
 *	@task: the task handle
 *	@title: the title for the operation
 *	@verb: the verb to use to complete the operation
 *	@help: detailed help for the action
 *	@hash_table: the hash table to be associated with the dialog
 *
 *	This routine creates a dialog window for setting the active configuration options of
 *	a task.
 **/
struct dialog_window *create_task_options_dialog(task_handle_t task, gchar *title, gchar *verb, gchar *help,
						GHashTable *hash_table)
{
	char *titulo;
	char *prompt;
	struct selwin *selwin;
	struct dialog_window *dialog;

	titulo = g_strdup_printf(_("%s - Configuration Options"), title);
	prompt = g_strdup_printf(_("Use spacebar to select an option to change. Select %s to finish."), verb);

	selwin = create_selection_window(titulo, help, prompt, verb,
					(menuitem_activate_cb)task_action_button_activated,
					NULL, NULL, NULL);

	set_clist_column_count(selwin->clist, 2);
	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)edit_option);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)NULL);

	dialog = (struct dialog_window *)selwin;
	dialog->user_data = hash_table;
	populate_options_clist(task, dialog, selwin->clist);

	g_free(prompt);
	g_free(titulo);

	return dialog;
}

/**
 *	on_delete_options_dialog - callback invoked when options dialog is to be deleted
 *	@dialog: the task dialog
 *
 *	This routine is invoked when the configure options dialog is to be deleted.
 *	This callback is setup only when the dialog was not the only one for a task.
 *	Unlike on_delete_task_dialog() it does not destroy the task. This is handled
 *	by the dialog that initially created the task.
 */
void on_delete_options_dialog(struct dialog_window *dialog)
{
	GHashTable *hash_table;
	handle_array_t *prev_selections;

	hash_table = dialog->user_data;
	prev_selections = g_hash_table_lookup(hash_table, "prev_selections");

	if (prev_selections != NULL)
		evms_free(prev_selections);

	g_hash_table_destroy(hash_table);
}

/**
 *	configure_options_button_activated - callback invoked when moving to options dialog
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the "Next" button on the acceptable objects dialog is
 *	activated and we want to create or show the configure options dialog.
 */
int configure_options_button_activated(struct menu_item *item)
{
	int rc;
	task_handle_t task;
	GHashTable *objs_hash_table;
	struct dialog_window *objs_dialog;
	handle_array_t *curr_selections, *prev_selections;

	objs_dialog = item->user_data;
	objs_hash_table = objs_dialog->user_data;
	task = get_task_from_dialog_data(objs_dialog);
	prev_selections = g_hash_table_lookup(objs_hash_table, "prev_selections");

	rc = evms_get_selected_objects(task, &curr_selections);
	if (rc == 0) {
		/*
		 * If our previous selection differ from the current selections then
		 * we need to ensure that the options dialogs gets (re)created in order
		 * to be accurate.
		 */
		if (are_handle_arrays_equal(curr_selections, prev_selections) == FALSE) {
			char *title, *verb;
			GList *next_dialog;
			GHashTable *opts_hash_table;
			struct dialog_window *opts_dialog;

			/*
			 * If we had any dialogs already created after this one,
			 * delete them before creating the new options dialog.
			 */
			next_dialog = get_next_dialog(objs_dialog);
			if (next_dialog != NULL)
				delete_dialog_list(next_dialog);

			verb = g_hash_table_lookup(objs_hash_table, "verb");
			title = g_hash_table_lookup(objs_hash_table, "title");

			opts_hash_table = g_hash_table_new(g_str_hash, g_str_equal);
			opts_dialog = create_task_options_dialog(task, title, verb, NULL, opts_hash_table);

			g_hash_table_insert(opts_hash_table, "task", GUINT_TO_POINTER(task));
			g_hash_table_insert(opts_hash_table, "verb", verb);
			opts_dialog->user_data = opts_hash_table;

			append_dialog_to_list(opts_dialog, objs_dialog->list);

			if (prev_selections != NULL)
				evms_free(prev_selections);

			g_hash_table_insert(objs_hash_table, "prev_selections", curr_selections);

			set_dialog_delete_cb(opts_dialog, (dialog_delete_cb)on_delete_options_dialog);
		} else {
			evms_free(curr_selections);
		}
		objs_dialog->list->current = get_next_dialog(objs_dialog);
	} else {
		show_message_dialog(_("Error"), _("Unable to determine what our selections are."));
		log_error("%s: evms_get_selected_objects() returned error code %d.\n", __FUNCTION__, rc);
	}

	return 0;
}

/**
 *	format_declination_item - return column strings for a row in the declination window clist
 *	@declined: address of struct containing the object declined and reason code
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the declination window clist containing the name of a declined object and
 *	the the declination reason.
 */
int format_declination_item(declined_handle_t *declined, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(declined->handle, &object);
	if (rc == 0) {
		switch (object->type) {
		case DISK:
		case REGION:
		case SEGMENT:
		case EVMS_OBJECT:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			break;
		case CONTAINER:
			g_ptr_array_add(text, g_strdup(object->info.container.name));
			break;
		case VOLUME:
			g_ptr_array_add(text, g_strdup(object->info.volume.name));
			break;
		case PLUGIN:
			g_ptr_array_add(text, g_strdup(object->info.plugin.long_name));
			break;
		default:
			log_warning("%s: Type %u was not processed.\n", __FUNCTION__, object->type);
			break;
		}
		g_ptr_array_add(text, g_strdup(evms_strerror(ABS(declined->reason))));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_declination_clist - populate the the declination window clist with names and reasons
 *	@clist: the declination window clist
 *	@declined_objects: the address of an array containing handles and declination reason codes
 *
 *	This routine populates the declination window clist with the objects that have been declined
 *	by a plugin from a evms_set_selected_objects() call.
 */
void populate_declination_clist(struct clist *clist, declined_handle_array_t *declined_objects)
{
	int i;

	for (i = 0; i < declined_objects->count; i++) {
		GPtrArray *text;

		text = g_ptr_array_new();
		if (format_declination_item(&(declined_objects->declined[i]), text) == 0)
			append_item(clist, text, NULL, NULL);
		g_ptr_array_free(text, TRUE);
	}
}

/**
 *	display_declination_window - display a dialog window with a list of objects declined by a plugin
 *	@declined_objects: the address of an array containing handles and declination reason codes
 *
 *	This routine displays a dialog with the list of objects declined by a plugin from
 *	a evms_set_selected_objects() call. We also list the reason next to the name of
 *	each object.
 */
void display_declination_window(declined_handle_array_t *declined_objects)
{
	struct selwin *list_dialog;
	struct dialog_window *dialog;

	list_dialog = create_list_dialog(_("Declined Objects and Reasons"),
				NULL,
				_("The objects listed above were rejected for the stated reasons."),
				2, declined_objects);
	dialog = (struct dialog_window *)list_dialog;

	set_clist_column_count(list_dialog->clist, 2);
	set_clist_column_info(list_dialog->clist, 0, calc_clist_column_width(list_dialog->clist, 0.35),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(list_dialog->clist, 1, calc_clist_column_width(list_dialog->clist, 0.65),
				get_clist_column_end(list_dialog->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(list_dialog->clist, 0, _("Storage Object"));
	print_clist_column_title(list_dialog->clist, 1, _("Declination Reason"));

	populate_declination_clist(list_dialog->clist, declined_objects);
	process_modal_dialog(dialog);
}

/**
 *	mark_selected_rows - mark items in acceptable objects clist that are considered selected
 *	@clist: the acceptable objects clist
 *	@selected_objects: handle array of objects the engine thinks are selected
 *
 *	This routine uses the given handle array to determine which items in the acceptable
 *	objects clist should be marked as selected.
 */
void mark_selected_rows(struct clist *clist, handle_array_t *selected_objects)
{
	gint i;

	/*
	 * Find the row corresponding to each object handle in the
	 * selected list and mark those rows as selected in the
	 * order provided by the list.
	 */
	for (i = 0; i < selected_objects->count; i++) {
		GList *item;

		item = find_item_from_handle(clist, selected_objects->handle[i]);
		if (item != NULL)
			select_item(clist, item->data);
	}
}

/**
 *	reload_acceptable_objects - reload acceptable objects in the clist and mark selected ones
 *	@clist: the clist containing the acceptable objects
 *
 *	This routine is typically scheduled for invocation after processing of other clist events
 *	in order to reload the clist with the current set of acceptable objects and marking
 *	objects that the engine task believes are currently selected.
 */
int reload_acceptable_objects(struct clist *clist)
{
	gint rc;
	task_handle_t task;
	handle_array_t *acceptable_list;
	struct dialog_window *dialog = clist->user_data;

	task = get_task_from_dialog_data(dialog);

	rc = evms_get_acceptable_objects(task, &acceptable_list);
	if (rc == 0) {
		guint min, max;
		object_handle_t focus_handle = 0;
		handle_array_t *selected_list;
		clist_select_item_cb select_cb;
		clist_unselect_item_cb unselect_cb;

		/*
		 * Temporarily replace the select and unselect clist item
		 * callbacks with the default callbacks. We restore the
		 * originals later on.
		 */
		select_cb = get_clist_select_item_cb(clist);
		unselect_cb = get_clist_unselect_item_cb(clist);
		set_clist_select_item_cb(clist, (clist_select_item_cb)row_selected);
		set_clist_unselect_item_cb(clist,(clist_unselect_item_cb)row_unselected);

		/*
		 * Remember focus item's handle to attempt to set focus on it after reload
		 */
		if (clist->focus_item != NULL) {
			struct clist_item *focus_item;
			focus_item = clist->focus_item->data;
			focus_handle = GPOINTER_TO_UINT(focus_item->user_data);
		}

		clear_clist(clist);

		rc = evms_get_selected_object_limits(task, &min, &max);
		if (rc == 0)
			set_selection_mode(clist, (max != 1 ? CLIST_MODE_MULTI : CLIST_MODE_SINGLE));

		clist_populate(clist, enum_acceptable_objects, clist_allow_all,
			format_standard_item, NULL, GUINT_TO_POINTER(task));

		rc = evms_get_selected_objects(task, &selected_list);
		if (rc == 0) {
			int count;
			GList *focus_item;

			mark_selected_rows(clist, selected_list);
			/*
			 * Determine if Next button needs to be made
			 * insensitive if now below our selection minimum.
			 */
			count = g_list_length(clist->selections);
			if (count < min || count == 0)
				make_next_button_insensitive(dialog);

			/*
			 * Try to restore the focus item using the handle that had focus
			 * before we cleared the listed and reloaded it.
			 */
			focus_item = find_item_from_handle(clist, focus_handle);
			if (focus_item != NULL)
				set_clist_focus_item(clist, focus_item);

			evms_free(selected_list);
		} else {
			log_error("%s: evms_get_selected_objects () returned error code %d.\n",
					__FUNCTION__, rc);
		}

		set_clist_select_item_cb(clist, select_cb);
		set_clist_unselect_item_cb(clist, unselect_cb);
		evms_free(acceptable_list);

		draw_clist(clist);
	} else {
		log_error("%s: evms_get_acceptable_objects() returned error code %d.\n", __FUNCTION__, rc);
	}
	return 0;
}

/**
 *	update_selected_objects - call engine to update selected object for a task
 *	@clist: the list of acceptable objects to choose from
 *	@task: the task handle
 *	@count: address of count of rows selected to update if necessary
 *
 *	This routine calls the engine to update the selected objects for a task.
 *	It handles any side-effects from the operation such as dealing with declined
 *	objects or having to schedule a reload of the acceptable objects in the clist.
 */
int update_selected_objects(struct clist *clist, task_handle_t task, int *count)
{
	int rc;
	task_effect_t effect = 0;
	handle_array_t *handle_array;
	declined_handle_array_t *declined_list = NULL;

	handle_array = make_handle_array_from_selection_list(clist->selections);

	rc = evms_set_selected_objects(task, handle_array, &declined_list, &effect);
	if (rc == 0) {
		/*
		 * Check to see if we ended up with a declined list or/and
		 * we need to reload our acceptable list and sync with the
		 * plugin on the selections.
		 */
		if ((declined_list && declined_list->count > 0) ||
			(effect & EVMS_Effect_Reload_Objects)) {
			if (declined_list) {
				display_declination_window(declined_list);
				log_warning("%s: Some of the object selections were declined.\n",
						__FUNCTION__);
				evms_free(declined_list);
			}
			sched_clist_idle_func(clist, (clist_idle_func)reload_acceptable_objects);
		}
	} else {
		show_message_dialog(_("Error"), _("Unable to tell plugin what our selections were."));
		log_error("%s: evms_set_selected_objects() returned error code %d.\n",
				__FUNCTION__, rc);
	}
	g_free(handle_array);
	return rc;
}

/**
 *	on_select_acceptable_object - callback invoked when selecting an acceptable object
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked for the row_selected event in the acceptable objects
 *	dialog clist. If the min for selected objects is met, we enable the "Next" button
 *	and call update_selected_objects() to give our current selections to the engine.
 *	Otherwise we simply "mark" the object as selected. If there is any
 *	the new selection to the engine through the
 */
int on_select_acceptable_object(struct clist *clist, struct clist_item *item)
{
	int rc;
	u_int32_t min, max;
	task_handle_t task;
	struct dialog_window *dialog = clist->user_data;

	task = get_task_from_dialog_data(dialog);

	rc = evms_get_selected_object_limits(task, &min, &max);
	if (rc == 0) {
		gint length = g_list_length(clist->selections);
		log_debug("%s: len = %i, min = %i, max = %i\n", __FUNCTION__, length, min, max);

		if (length <= max || max == -1) {
			rc = update_selected_objects(clist, task, &length);
			if (rc == 0) {
				mark_selection(item);
				if (length >= min)
					make_next_button_sensitive(dialog);
			}
		} else if (length > max) {
			/*
			 * Return some sort of error code so that selection is undone
			 * as this selection exceeds the max. The user must unselect
			 * something first.
			 */
			rc = E2BIG;
		} else {
			/*
			 * Haven't reached the min needed to call the engine so just
			 * mark the object as selected for now...
			 */
			mark_selection(item);
		}
	} else {
		log_error("%s: evms_get_selected_object_limits() return error code %u\n",
				__FUNCTION__, rc);
	}
	return rc;
}

/**
 *	on_unselect_acceptable_object - callback invoked when unselecting an acceptable object
 *	@clist: the clist for the row item unselected
 *	@item: the clist item that was unselected
 *
 *	This routine is invoked for the row_unselected event in the acceptable objects
 *	dialog clist.
 */
int on_unselect_acceptable_object(struct clist *clist, struct clist_item *item)
{
	int rc;
	u_int32_t min, max;
	task_handle_t task;
	struct dialog_window *dialog = clist->user_data;

	task = get_task_from_dialog_data(dialog);

	rc = evms_get_selected_object_limits(task, &min, &max);
	if (rc == 0) {
		gint length = g_list_length(clist->selections);
		log_debug("%s: len = %i, min = %i, max = %i\n", __FUNCTION__, length, min, max);

		rc = update_selected_objects(clist, task, &length);
		if (rc == 0) {
			unmark_selection(item);

			/*
			 * Disallow the user from continuing to the next dialog
			 * if we fall below the minimum required selected objects.
			 */
			if (length < min || length == 0) {
				make_next_button_insensitive(dialog);
			}
		}
	} else {
		log_error("%s: evms_get_selected_object_limits() return error code %u\n",
				__FUNCTION__, rc);
	}
	return rc;
}

/**
 *	create_acceptable_objects_selection_dialog - create the initial selection of acceptable objects
 *	@task: the task handle
 *	@title: the window title
 *	@verb: the string to use for the task completion button
 *	@help: the help text to use if this is the last dialog
 *	@requirements: the task requirements
 *	@hash_table: the hash table to associate with dialog window
 *
 *	This routines creates a standard selection window containing a list of acceptable objects
 *	for a task.
 */
struct dialog_window *create_acceptable_objects_selection_dialog(task_handle_t task, char *title, char *verb,
								task_requirements_t requirements, char *help,
								GHashTable *hash_table)
{
	u_int32_t min = 0, max = 0;
	struct selwin *selwin;
	char *window_title, *prompt;
	struct dialog_window *dialog;
	menuitem_activate_cb activate_cb;

	if (requirements == TASK_REQUIRES_OBJECTS_ONLY) {
		prompt = g_strdup_printf(_("Use the spacebar key to make selections and choose %s to finish"),
					verb);
		activate_cb = (menuitem_activate_cb)task_action_button_activated;
	} else {
		help = NULL;
		verb = NULL;
		prompt = NULL;
		activate_cb = (menuitem_activate_cb)configure_options_button_activated;
	}

	if (title != NULL)
		window_title = g_strdup_printf(_("%s - Acceptable Objects"), title);
	else
		window_title = g_strdup(_("Acceptable Objects"));

	selwin = create_selection_window(window_title,
					help, prompt,
					verb,
					activate_cb,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	dialog->user_data = hash_table;

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Name"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)on_select_acceptable_object);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)on_unselect_acceptable_object);

	if (evms_get_selected_object_limits(task, &min, &max) == 0)
		set_selection_mode(selwin->clist, (max != 1 ? CLIST_MODE_MULTI : CLIST_MODE_SINGLE));

	clist_populate(selwin->clist, enum_acceptable_objects, clist_allow_all,
			format_standard_item, NULL, GUINT_TO_POINTER(task));

	if (g_list_length(selwin->clist->choices) == 1 && 1 >= min)
		select_item(selwin->clist, selwin->clist->choices->data);

	g_free(prompt);
	g_free(window_title);

	return dialog;
}

/**
 *	on_delete_task_dialog - callback invoked when task dialog is to be deleted
 *	@dialog: the task dialog
 *
 *	This routine is invoked when the plugin that created a task is to be
 *	deleted. It takes care of destroying the task and any other dynamically
 *	allocated resources associated with the dialog.
 */
void on_delete_task_dialog(struct dialog_window *dialog)
{
	int rc;
	GHashTable *hash_table;

	hash_table = dialog->user_data;

	g_free(g_hash_table_lookup(hash_table, "verb"));
	g_free(g_hash_table_lookup(hash_table, "title"));

	rc = evms_destroy_task(get_task_from_dialog_data(dialog));
	if (rc != 0)
		log_error("%s: evms_destroy_task() returned error code %d.\n", __FUNCTION__, rc);

	g_hash_table_destroy(hash_table);
}

/**
 *	create_task_dialog - start a task and present the appropriate dialogs windows
 *	@list: the dialog list we are to become part of
 *	@handle: the handle for a plugin or reference object for the task
 *	@action: the id of the function to perform
 *	@title: the title string for the operation
 *	@verb: the string to use as the text for the button to complete the operation
 *	@help: the address of the string with detailed help on the operation
 *
 *	This routine calls the appropriate function to create a task, determine the
 *	requirements for the task and create the appropriate initial dialog window to
 *	involve the user in performing the task. We append ourselves to the dialog
 *	list if one is supplied. If we are the first dialog in a series or the first
 *	dialog then we make the "Previous" button not visible.
 */
struct dialog_window *create_task_dialog(struct dialog_list *list, engine_handle_t handle,
			task_action_t action, gchar *title, gchar *verb, gchar *help)
{
	int rc;
	u_int32_t min, max;
	task_handle_t task;
	GHashTable *hash_table;
	struct dialog_window *dialog = NULL;
	task_requirements_t requirements;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);

	rc = create_task(handle, action, &requirements, &task, &min, &max);

	if (rc == 0) {
		g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));

		switch (requirements) {
		case TASK_REQUIRES_OPTIONS_ONLY:
			dialog = create_task_options_dialog(task, title, verb, help, hash_table);
			break;
		case TASK_REQUIRES_OBJECTS_ONLY:
		case TASK_REQUIRES_OBJECTS_AND_OPTIONS:
			dialog = create_acceptable_objects_selection_dialog(task, title, verb,
									requirements, help, hash_table);
			break;
		case TASK_REQUIRES_NO_ADDITIONAL_ARGS:
			dialog = create_task_confirmation_dialog(list, task, title, verb, help);
			break;
		default:
			log_error("%s: Unable to determine task requirements.\n", __FUNCTION__);
			break;
		}
	}
	if (dialog != NULL) {
		if (list == NULL) {
			list = g_new0(struct dialog_list, 1);
			init_dialog_list(list);
			set_menu_item_visibility(dialog->prev_button, FALSE);
		}
		append_dialog_to_list(dialog, list);
		dialog->user_data = hash_table;

		g_hash_table_insert(hash_table, "verb", g_strdup(verb));
		g_hash_table_insert(hash_table, "title", g_strdup(title));

		set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_task_dialog);
	} else {
		g_hash_table_destroy(hash_table);
	}
	return dialog;
}

/**
 *	on_init_task_menuitem_activated - create a task and create the appropriate initial dialog
 *	@item: the menu item activated
 *
 *	This routine is invoked when a menu item (Next button) is activated in a
 *	selection dialog window. We expect that item->user_data points to the
 *	current dialog. In that dialog, dialog->user_data is expected to be
 *	hash table containing values accessible through predefined string keys.
 */
int on_init_task_menuitem_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct dialog_window *dialog;
	engine_handle_t saved_handle, current_handle;

	dialog = item->user_data;
	hash_table = dialog->user_data;
	saved_handle = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "handle"));
	current_handle = get_selected_handle(((struct selwin *)dialog)->clist);

	if (current_handle != saved_handle) {
		char *help;
		char *verb;
		char *title;
		GList *next_dialog;
		task_action_t action;
		struct dialog_window *task_dialog;

		/*
		 * Since the selected item that initiated the task changed, check
		 * to see if we had previously built some task dialogs. If so, we
		 * need to clear those out of the dialog list to replace it with
		 * the new one.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		help = g_hash_table_lookup(hash_table, "help");
		verb = g_hash_table_lookup(hash_table, "verb");
		title = g_hash_table_lookup(hash_table, "title");
		action = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "action"));

		task_dialog = create_task_dialog(dialog->list, current_handle, action, title,
					verb ? verb : get_task_action_string(action), help);
		if (task_dialog != NULL) {
			g_hash_table_insert(hash_table, "handle", GUINT_TO_POINTER(current_handle));
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		/*
		 * No selection change so simply start processing events for next window.
		 */
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}
