/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: dm.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "segs.h"
#include "dm.h"



/*
 *  Function:  get_DM_info
 *
 *  Called when a DM activate fails.  The job here is to see if the
 *  segment is active in the kernel.  If so ... compare the target
 *  mapping against the engine segment object.  If the segments starting
 *  LBA and SIZE match ... then we dont need to activate the segment
 *  because it is already active and has a correct mapping ... return 0.
 */
static int get_DM_info( DISKSEG *seg )
{
	dm_target_t *targets=NULL;
	dm_device_t *dev = NULL;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("seg= %s\n", seg->name);

	rc = EngFncs->dm_update_status(seg);
	if (!rc) {
		if (seg->flags & SOFLAG_ACTIVE) {
			LOG_DEBUG("segment IS active in the kernel\n");
			rc = EngFncs->dm_get_targets(seg, &targets);
			if ( !rc && targets != NULL) {
				dev = (dm_device_t *) targets->data.linear;
				if ( seg->start == dev->start &&
				     seg->size  == targets->length) {
					LOG_DEBUG("kernel object matches ... marking segment active\n");
					rc = 0;
				}
				else {
					LOG_ERROR("error, got a DM object using our segment name but the metadata differs. dont know what to do!\n");
					rc = ENODEV; // dont know how to handle this.
				}
			}
			else {
				rc = ENODEV;
			}
			if (targets) EngFncs->dm_deallocate_targets(targets);
		}
		else {
			LOG_DEBUG("segment is NOT active in the kernel\n");
			rc=ENODEV;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: SEG_can_activate
 */
int SEG_can_activate( DISKSEG *seg )
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: SEG_activate
 *
 *  Called in order to create a mapping of a disk partition
 *  in the kernel.
 *
 *  Note: if there is a valid mirror target in the private
 *        data area of this segment ... then the segment is
 *        being moved and we need to provide a mirror target
 *        mapping between the segment and the target segment.
 */
int SEG_activate( DISKSEG *seg )
{
	int rc=EINVAL;
	LOGICALDISK *ld;
	dm_target_t target;
	dm_device_t linear;
	SEG_PRIVATE_DATA *pdata;


	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {
		LOG_DEBUG("seg = %s\n", seg->name);

		rc = 0;

		pdata = (SEG_PRIVATE_DATA *)seg->private_data;

		// we are either going to rename an existing dm
		// object or else create a new dm mapping ...
		if ( pdata->flags & SEG_NEEDS_DM_RENAME) {
			LOG_DEBUG("private dm rename flag is on for this object ... renaming DM object %s to %s\n",
				  seg->dev_name, seg->name);
			rc = EngFncs->dm_rename(seg, seg->dev_name, seg->name);
			if (!rc) {
				strncpy(seg->dev_name, seg->name, EVMS_NAME_SIZE);
				pdata->flags &= ~SEG_NEEDS_DM_RENAME;
				seg->flags   &= ~SOFLAG_NEEDS_ACTIVATE;
			}
		}
		else {
			LOG_DEBUG("activating this segment\n");
			target.start = 0;
			target.length = seg->size;
			target.params = NULL;
			target.next = NULL;
			target.type = DM_TARGET_LINEAR;
			target.data.linear = &linear;
			linear.major = ld->dev_major;
			linear.minor = ld->dev_minor;
			linear.start = seg->start;

			rc = EngFncs->dm_activate(seg, &target);
			if (rc) {
				rc = get_DM_info(seg);	// FAILED ... test if already active
			}

			if (!rc) {
				strncpy(seg->dev_name, seg->name, EVMS_NAME_SIZE);
				seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			}
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Function: SEG_can_deactivate
 */
int SEG_can_deactivate( DISKSEG *seg )
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: SEG_deactivate
 *
 *  Called in order to delete an existing mapping of a disk
 *  partition in the kernel.
 */
int SEG_deactivate( DISKSEG *seg )
{
	int rc;
	char saved_name[EVMS_NAME_SIZE+1];

	LOG_ENTRY();

	strncpy(saved_name, seg->name, EVMS_NAME_SIZE);
	strncpy(seg->name, seg->dev_name, EVMS_NAME_SIZE);
	rc = EngFncs->dm_deactivate(seg);
	strncpy(seg->name, saved_name, EVMS_NAME_SIZE);

	if (!rc) {
		seg->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

