/*
 *
 * vim:ts=8:sw=3:sts=3:expandtab:cino=>5n-2f0^-2{2(0W1st0
 */
#ifndef ELEMENTARY_H
#define ELEMENTARY_H

/**
 * @file Elementary.h
 * @brief Elementary's API
 *
 * Elementary API.
 */

#define ELM_UNIX
#undef ELM_WIN32
#undef ELM_WINCE
#define ELM_EDBUS
#define ELM_EFREET
#define ELM_ETHUMB
#undef ELM_DEBUG
#define ELM_ALLOCA_H
#define ELM_LIBINTL_H

/* Standard headers for standard system calls etc. */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/param.h>
#include <dlfcn.h>
#include <math.h>
#include <fnmatch.h>
#include <limits.h>
#include <ctype.h>
#include <time.h>
#include <dirent.h>
#include <pwd.h>
#include <errno.h>

#ifdef ELM_UNIX
# include <locale.h>
# ifdef ELM_LIBINTL_H
#  include <libintl.h>
# endif
# include <signal.h>
# include <grp.h>
# include <glob.h>
#endif

#ifdef ELM_ALLOCA_H
# include <alloca.h>
#endif

#if defined (ELM_WIN32) || defined (ELM_WINCE)
# include <malloc.h>
# ifndef alloca
#  define alloca _alloca
# endif
#endif


/* EFL headers */
#include <Eina.h>
#include <Eet.h>
#include <Evas.h>
#include <Ecore.h>
#include <Ecore_Evas.h>
#include <Ecore_File.h>
#include <Ecore_IMF.h>
#include <Edje.h>

#ifdef ELM_EDBUS
# include <E_DBus.h>
# include <E_Hal.h>
#endif

#ifdef ELM_EFREET
# include <Efreet.h>
# include <Efreet_Mime.h>
# include <Efreet_Trash.h>
#endif

#ifdef ELM_ETHUMB
# include <Ethumb_Client.h>
#endif

#ifdef EAPI
# undef EAPI
#endif

#ifdef _WIN32
# ifdef ELEMENTARY_BUILD
#  ifdef DLL_EXPORT
#   define EAPI __declspec(dllexport)
#  else
#   define EAPI
#  endif /* ! DLL_EXPORT */
# else
#  define EAPI __declspec(dllimport)
# endif /* ! EFL_EVAS_BUILD */
#else
# ifdef __GNUC__
#  if __GNUC__ >= 4
#   define EAPI __attribute__ ((visibility("default")))
#  else
#   define EAPI
#  endif
# else
#  define EAPI
# endif
#endif /* ! _WIN32 */


/* allow usage from c++ */
#ifdef __cplusplus
extern "C" {
#endif

#define ELM_VERSION_MAJOR 0
#define ELM_VERSION_MINOR 7

   typedef struct _Elm_Version
     {
        int major;
        int minor;
        int micro;
        int revision;
     } Elm_Version;

   EAPI extern Elm_Version *elm_version;

/* handy macros */
#define ELM_RECTS_INTERSECT(x, y, w, h, xx, yy, ww, hh) (((x) < ((xx) + (ww))) && ((y) < ((yy) + (hh))) && (((x) + (w)) > (xx)) && (((y) + (h)) > (yy)))
#define ELM_PI 3.14159265358979323846

  /**
   * Defines couple of standard Evas_Object layers to be used
   * with evas_object_layer_set().
   *
   * @note whenever extending with new values, try to keep some padding
   *       to siblings so there is room for further extensions.
   */
  typedef enum _Elm_Object_Layer
    {
      ELM_OBJECT_LAYER_BACKGROUND = EVAS_LAYER_MIN + 64, /**< where to place backgrounds */
      ELM_OBJECT_LAYER_DEFAULT = 0, /**< Evas_Object default layer (and thus for Elementary) */
      ELM_OBJECT_LAYER_FOCUS = EVAS_LAYER_MAX - 128, /**< where focus object visualization is */
      ELM_OBJECT_LAYER_TOOLTIP = EVAS_LAYER_MAX - 64, /**< where to show tooltips */
      ELM_OBJECT_LAYER_CURSOR = EVAS_LAYER_MAX - 32, /**< where to show cursors */
      ELM_OBJECT_LAYER_LAST /**< last layer known by Elementary */
    } Elm_Object_Layer;

/**************************************************************************/
   EAPI extern int ELM_ECORE_EVENT_ETHUMB_CONNECT;

  /* Objects */
  /**
   * Emitted when policy value changed.
   */
  EAPI extern int ELM_EVENT_POLICY_CHANGED;
  typedef struct _Elm_Event_Policy_Changed
    {
       unsigned int policy;
       int new_value;
       int old_value;
    } Elm_Event_Policy_Changed;

  /**
   * Policy identifiers.
   *
   * @see elm_policy_set()
   */
  typedef enum _Elm_Policy
    {
      ELM_POLICY_QUIT, /**< when should quit application automatically.
			* @see Elm_Policy_Quit.
			*/
      ELM_POLICY_LAST
    } Elm_Policy;

  typedef enum _Elm_Policy_Quit
    {
      ELM_POLICY_QUIT_NONE = 0, /**< never quit application automatically */
      ELM_POLICY_QUIT_LAST_WINDOW_CLOSED /**< quit when last window is closed */
    } Elm_Policy_Quit;

   typedef enum _Elm_Focus_Direction
     {
        ELM_FOCUS_PREVIOUS,
        ELM_FOCUS_NEXT
     } Elm_Focus_Direction;

   typedef enum _Elm_Text_Format
     {
	ELM_TEXT_FORMAT_PLAIN_UTF8,
	ELM_TEXT_FORMAT_MARKUP_UTF8
     } Elm_Text_Format;

  /**
   * Called back when a widget's tooltip is activated and needs content.
   * @param data user-data given to elm_object_tooltip_content_cb_set()
   * @param obj owner widget.
   */
   typedef Evas_Object *(*Elm_Tooltip_Content_Cb) (void *data, Evas_Object *obj);

  /**
   * Called back when a widget's item tooltip is activated and needs content.
   * @param data user-data given to elm_object_tooltip_content_cb_set()
   * @param obj owner widget.
   * @param item context dependent item. As an example, if tooltip was
   *        set on Elm_List_Item, then it is of this type.
   */
   typedef Evas_Object *(*Elm_Tooltip_Item_Content_Cb) (void *data, Evas_Object *obj, void *item);

   typedef Eina_Bool (*Elm_Event_Cb) (void *data, Evas_Object *obj, Evas_Object *src, Evas_Callback_Type type, void *event_info);

#ifndef ELM_LIB_QUICKLAUNCH
#define ELM_MAIN() int main(int argc, char **argv) {elm_init(argc, argv); return elm_main(argc, argv);}
#else
#define ELM_MAIN() int main(int argc, char **argv) {return elm_quicklaunch_fallback(argc, argv);}
#endif

/**************************************************************************/
   /* General calls */
   EAPI int          elm_init(int argc, char **argv);
   EAPI int          elm_shutdown(void);
   EAPI void         elm_run(void);
   EAPI void         elm_exit(void);

   EAPI void         elm_quicklaunch_mode_set(Eina_Bool ql_on);
   EAPI Eina_Bool    elm_quicklaunch_mode_get(void);
   EAPI int          elm_quicklaunch_init(int argc, char **argv);
   EAPI int          elm_quicklaunch_sub_init(int argc, char **argv);
   EAPI int          elm_quicklaunch_sub_shutdown(void);
   EAPI int          elm_quicklaunch_shutdown(void);
   EAPI void         elm_quicklaunch_seed(void);
   EAPI Eina_Bool    elm_quicklaunch_prepare(int argc, char **argv);
   EAPI Eina_Bool    elm_quicklaunch_fork(int argc, char **argv, char *cwd, void (postfork_func) (void *data), void *postfork_data);
   EAPI void         elm_quicklaunch_cleanup(void);
   EAPI int          elm_quicklaunch_fallback(int argc, char **argv);
   EAPI char        *elm_quicklaunch_exe_path_get(const char *exe);

   EAPI Eina_Bool    elm_need_efreet(void);
   EAPI Eina_Bool    elm_need_e_dbus(void);
   EAPI Eina_Bool    elm_need_ethumb(void);

   EAPI Eina_Bool    elm_policy_set(unsigned int policy, int value);
   EAPI int          elm_policy_get(unsigned int policy);

   EAPI void         elm_all_flush(void);
   EAPI int          elm_cache_flush_interval_get(void);
   EAPI void         elm_cache_flush_interval_set(int size);
   EAPI void         elm_cache_flush_interval_all_set(int size);
   EAPI int          elm_font_cache_get(void);
   EAPI void         elm_font_cache_set(int size);
   EAPI void         elm_font_cache_all_set(int size);
   EAPI int          elm_image_cache_get(void);
   EAPI void         elm_image_cache_set(int size);
   EAPI void         elm_image_cache_all_set(int size);
   EAPI int          elm_edje_file_cache_get(void);
   EAPI void         elm_edje_file_cache_set(int size);
   EAPI void         elm_edje_file_cache_all_set(int size);
   EAPI int          elm_edje_collection_cache_get(void);
   EAPI void         elm_edje_collection_cache_set(int size);
   EAPI void         elm_edje_collection_cache_all_set(int size);

   EAPI void         elm_object_scale_set(Evas_Object *obj, double scale) EINA_ARG_NONNULL(1);
   EAPI double       elm_object_scale_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_object_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_disabled_set(Evas_Object *obj, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_object_disabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI Eina_Bool    elm_object_widget_check(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_object_parent_widget_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_object_top_widget_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_object_widget_type_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI double       elm_scale_get(void);
   EAPI void         elm_scale_set(double scale);
   EAPI void         elm_scale_all_set(double scale);

   EAPI Eina_Bool    elm_config_save(void);
   EAPI void         elm_config_reload(void);

   EAPI const char  *elm_profile_current_get(void);
   EAPI const char  *elm_profile_dir_get(const char *profile, Eina_Bool is_user);
   EAPI void         elm_profile_dir_free(const char *p_dir);
   EAPI Eina_List   *elm_profile_list_get(void);
   EAPI void         elm_profile_list_free(Eina_List *l);
   EAPI void         elm_profile_set(const char *profile);
   EAPI void         elm_profile_all_set(const char *profile);

   EAPI const char  *elm_engine_current_get(void);
   EAPI void         elm_engine_set(const char *engine);

  typedef struct _Elm_Text_Class
    {
       const char *name;
       const char *desc;
    } Elm_Text_Class;

  typedef struct _Elm_Font_Overlay
    {
       const char    *text_class;
       const char    *font;
       Evas_Font_Size size;
    } Elm_Font_Overlay;

  typedef struct _Elm_Font_Properties
    {
       const char *name;
       Eina_List  *styles;
    } Elm_Font_Properties;

   EAPI const Eina_List     *elm_text_classes_list_get(void);
   EAPI void                 elm_text_classes_list_free(const Eina_List *list);

   EAPI const Eina_List     *elm_font_overlay_list_get(void);
   EAPI void                 elm_font_overlay_set(const char *text_class, const char *font, Evas_Font_Size size);
   EAPI void                 elm_font_overlay_unset(const char *text_class);
   EAPI void                 elm_font_overlay_apply();
   EAPI void                 elm_font_overlay_all_apply();

   EAPI Elm_Font_Properties *elm_font_properties_get(const char *font) EINA_ARG_NONNULL(1);
   EAPI void                 elm_font_properties_free(Elm_Font_Properties *efp) EINA_ARG_NONNULL(1);
   EAPI const char          *elm_font_fontconfig_name_get(const char *name, const char *style) EINA_ARG_NONNULL(1);
   EAPI void                 elm_font_fontconfig_name_free(const char *name) EINA_ARG_NONNULL(1);
   EAPI Eina_Hash           *elm_font_available_hash_add(Eina_List *list);
   EAPI void                 elm_font_available_hash_del(Eina_Hash *hash);

   EAPI Evas_Coord   elm_finger_size_get(void);
   EAPI void         elm_finger_size_set(Evas_Coord size);
   EAPI void         elm_finger_size_all_set(Evas_Coord size);

   EAPI Eina_Bool    elm_focus_highlight_enabled_get(void);
   EAPI void         elm_focus_highlight_enabled_set(Eina_Bool enable);
   EAPI Eina_Bool    elm_focus_highlight_animate_get(void);
   EAPI void         elm_focus_highlight_animate_set(Eina_Bool animate);

   EAPI Eina_Bool    elm_object_focus_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_focus(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_unfocus(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_focus_allow_set(Evas_Object *obj, Eina_Bool enable) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_object_focus_allow_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI void         elm_object_focus_custom_chain_set(Evas_Object *obj, Eina_List *objs) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_focus_custom_chain_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_object_focus_custom_chain_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_focus_custom_chain_append(Evas_Object *obj, Evas_Object *child, Evas_Object *relative_child) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_object_focus_custom_chain_prepend(Evas_Object *obj, Evas_Object *child, Evas_Object *relative_child) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_object_focus_cycle(Evas_Object *obj, Elm_Focus_Direction dir) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_focus_direction_go(Evas_Object *obj, int x, int y) EINA_ARG_NONNULL(1);

   EAPI void         elm_object_scroll_hold_push(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_scroll_hold_pop(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_scroll_freeze_push(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_scroll_freeze_pop(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_scroll_lock_x_set(Evas_Object *obj, Eina_Bool lock) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_scroll_lock_y_set(Evas_Object *obj, Eina_Bool lock) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_object_scroll_lock_x_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_object_scroll_lock_y_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI void         elm_object_signal_emit(Evas_Object *obj, const char *emission, const char *source) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_signal_callback_add(Evas_Object *obj, const char *emission, const char *source, void (*func) (void *data, Evas_Object *o, const char *emission, const char *source), void *data) EINA_ARG_NONNULL(1, 4);
   EAPI void        *elm_object_signal_callback_del(Evas_Object *obj, const char *emission, const char *source, void (*func) (void *data, Evas_Object *o, const char *emission, const char *source)) EINA_ARG_NONNULL(1, 4);

   EAPI void         elm_object_event_callback_add(Evas_Object *obj, Elm_Event_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void        *elm_object_event_callback_del(Evas_Object *obj, Elm_Event_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);

   EAPI void         elm_coords_finger_size_adjust(int times_w, Evas_Coord *w, int times_h, Evas_Coord *h);

   EAPI double       elm_longpress_timeout_get(void);
   EAPI void         elm_longpress_timeout_set(double longpress_timeout);

   /* debug
    * don't use it unless you are sure
    */
   EAPI void         elm_object_tree_dump(const Evas_Object *top);
   EAPI void         elm_object_tree_dot_dump(const Evas_Object *top, const char *file);


   /* theme */
   typedef struct _Elm_Theme Elm_Theme;

   EAPI Elm_Theme   *elm_theme_new(void);
   EAPI void         elm_theme_free(Elm_Theme *th);
   EAPI void         elm_theme_copy(Elm_Theme *th, Elm_Theme *thdst);
   EAPI void         elm_theme_ref_set(Elm_Theme *th, Elm_Theme *thref);
   EAPI Elm_Theme   *elm_theme_ref_get(Elm_Theme *th);
   EAPI Elm_Theme   *elm_theme_default_get(void);
   EAPI void         elm_theme_overlay_add(Elm_Theme *th, const char *item);
   EAPI void         elm_theme_overlay_del(Elm_Theme *th, const char *item);
   EAPI void         elm_theme_extension_add(Elm_Theme *th, const char *item);
   EAPI void         elm_theme_extension_del(Elm_Theme *th, const char *item);
   EAPI void         elm_theme_set(Elm_Theme *th, const char *theme);
   EAPI const char  *elm_theme_get(Elm_Theme *th);
   EAPI const Eina_List *elm_theme_list_get(const Elm_Theme *th);
   EAPI char        *elm_theme_list_item_path_get(const char *f, Eina_Bool *in_search_path);
   EAPI void         elm_theme_flush(Elm_Theme *th);
   EAPI void         elm_theme_full_flush(void);

   EAPI void         elm_theme_all_set(const char *theme);

   EAPI Eina_List   *elm_theme_name_available_list_new(void);
   EAPI void         elm_theme_name_available_list_free(Eina_List *list);

   EAPI void         elm_object_theme_set(Evas_Object *obj, Elm_Theme *th) EINA_ARG_NONNULL(1);
   EAPI Elm_Theme   *elm_object_theme_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* win */
   typedef enum _Elm_Win_Type
     {
        ELM_WIN_BASIC,
        ELM_WIN_DIALOG_BASIC,
        ELM_WIN_DESKTOP,
        ELM_WIN_DOCK,
        ELM_WIN_TOOLBAR,
        ELM_WIN_MENU,
        ELM_WIN_UTILITY,
        ELM_WIN_SPLASH
     } Elm_Win_Type;

   typedef enum _Elm_Win_Keyboard_Mode
     {
	ELM_WIN_KEYBOARD_UNKNOWN,
        ELM_WIN_KEYBOARD_OFF,
        ELM_WIN_KEYBOARD_ON,
        ELM_WIN_KEYBOARD_ALPHA,
        ELM_WIN_KEYBOARD_NUMERIC,
        ELM_WIN_KEYBOARD_PIN,
        ELM_WIN_KEYBOARD_PHONE_NUMBER,
        ELM_WIN_KEYBOARD_HEX,
        ELM_WIN_KEYBOARD_TERMINAL,
        ELM_WIN_KEYBOARD_PASSWORD,
        ELM_WIN_KEYBOARD_IP,
        ELM_WIN_KEYBOARD_HOST,
        ELM_WIN_KEYBOARD_FILE,
        ELM_WIN_KEYBOARD_URL,
        ELM_WIN_KEYBOARD_KEYPAD,
        ELM_WIN_KEYBOARD_J2ME
     } Elm_Win_Keyboard_Mode;

   EAPI Evas_Object *elm_win_add(Evas_Object *parent, const char *name, Elm_Win_Type type);
   EAPI void         elm_win_resize_object_add(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_resize_object_del(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_title_set(Evas_Object *obj, const char *title) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_win_title_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_autodel_set(Evas_Object *obj, Eina_Bool autodel) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_autodel_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_activate(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_lower(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_raise(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_borderless_set(Evas_Object *obj, Eina_Bool borderless) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_borderless_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_shaped_set(Evas_Object *obj, Eina_Bool shaped) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_shaped_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_alpha_set(Evas_Object *obj, Eina_Bool alpha) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_transparent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_transparent_set(Evas_Object *obj, Eina_Bool transparent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_alpha_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_override_set(Evas_Object *obj, Eina_Bool override) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_override_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_fullscreen_set(Evas_Object *obj, Eina_Bool fullscreen) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_fullscreen_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_maximized_set(Evas_Object *obj, Eina_Bool maximized) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_maximized_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_iconified_set(Evas_Object *obj, Eina_Bool iconified) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_iconified_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_layer_set(Evas_Object *obj, int layer) EINA_ARG_NONNULL(1);
   EAPI int          elm_win_layer_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_rotation_set(Evas_Object *obj, int rotation) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_rotation_with_resize_set(Evas_Object *obj, int rotation) EINA_ARG_NONNULL(1);
   EAPI int          elm_win_rotation_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_sticky_set(Evas_Object *obj, Eina_Bool sticky) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_sticky_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_conformant_set(Evas_Object *obj, Eina_Bool conformant) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_conformant_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_quickpanel_set(Evas_Object *obj, Eina_Bool quickpanel) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_quickpanel_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_quickpanel_priority_major_set(Evas_Object *obj, int priority) EINA_ARG_NONNULL(1);
   EAPI int          elm_win_quickpanel_priority_major_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_quickpanel_priority_minor_set(Evas_Object *obj, int priority) EINA_ARG_NONNULL(1);
   EAPI int          elm_win_quickpanel_priority_minor_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_quickpanel_zone_set(Evas_Object *obj, int zone) EINA_ARG_NONNULL(1);
   EAPI int          elm_win_quickpanel_zone_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_focus_highlight_enabled_set(Evas_Object *obj, Eina_Bool enabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_focus_highlight_enabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_focus_highlight_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_win_focus_highlight_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /*...
    * ecore_x_icccm_hints_set -> accepts_focus (add to ecore_evas)
    * ecore_x_icccm_hints_set -> window_group (add to ecore_evas)
    * ecore_x_icccm_size_pos_hints_set -> request_pos (add to ecore_evas)
    * ecore_x_icccm_client_leader_set -> l (add to ecore_evas)
    * ecore_x_icccm_window_role_set -> role (add to ecore_evas)
    * ecore_x_icccm_transient_for_set -> forwin (add to ecore_evas)
    * ecore_x_netwm_window_type_set -> type (add to ecore_evas)
    *
    * (add to ecore_x) set netwm argb icon! (add to ecore_evas)
    * (blank mouse, private mouse obj, defaultmouse)
    *
    */
   EAPI void         elm_win_keyboard_mode_set(Evas_Object *obj, Elm_Win_Keyboard_Mode mode) EINA_ARG_NONNULL(1);
   EAPI Elm_Win_Keyboard_Mode elm_win_keyboard_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_keyboard_win_set(Evas_Object *obj, Eina_Bool is_keyboard) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_win_keyboard_win_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI void         elm_win_screen_position_get(const Evas_Object *obj, int *x, int *y) EINA_ARG_NONNULL(1);

   EAPI Evas_Object *elm_win_inwin_add(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_inwin_activate(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_win_inwin_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_win_inwin_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_win_inwin_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * minimal
    * minimal_vertical
    */
   /* X specific calls - won't work on non-x engines (return 0) */
   EAPI Ecore_X_Window elm_win_xwindow_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "delete,request" - the user requested to delete the window
    * "focus,in" - window got focus
    * "focus,out" - window lost focus
    * "moved" - window that holds the canvas was moved
    */
  
  /* bg */
  typedef enum _Elm_Bg_Option
    {
      ELM_BG_OPTION_CENTER,
      ELM_BG_OPTION_SCALE,
      ELM_BG_OPTION_STRETCH,
      ELM_BG_OPTION_TILE
    } Elm_Bg_Option;
   EAPI Evas_Object  *elm_bg_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_file_set(Evas_Object *obj, const char *file, const char *group) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_file_get(const Evas_Object *obj, const char **file, const char **group) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_option_set(Evas_Object *obj, Elm_Bg_Option option) EINA_ARG_NONNULL(1);
   EAPI Elm_Bg_Option elm_bg_option_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_color_set(Evas_Object *obj, int r, int g, int b) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_color_get(const Evas_Object *obj, int *r, int *g, int *b) EINA_ARG_NONNULL(1);
   EAPI void          elm_bg_overlay_set(Evas_Object *obj, Evas_Object *overlay) EINA_ARG_NONNULL(1);
   EAPI Evas_Object  *elm_bg_overlay_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object  *elm_bg_overlay_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    */

   /* icon */
   typedef enum _Elm_Icon_Lookup_Order
     {
        ELM_ICON_LOOKUP_FDO_THEME, /**< icon look up order: freedesktop, theme */
        ELM_ICON_LOOKUP_THEME_FDO, /**< icon look up order: theme, freedesktop */
        ELM_ICON_LOOKUP_FDO,       /**< icon look up order: freedesktop */
        ELM_ICON_LOOKUP_THEME      /**< icon look up order: theme */
     } Elm_Icon_Lookup_Order;

   EAPI Evas_Object          *elm_icon_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_icon_file_set(Evas_Object *obj, const char *file, const char *group) EINA_ARG_NONNULL(1, 2);
   EAPI void                  elm_icon_file_get(const Evas_Object *obj, const char **file, const char **group) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_icon_standard_set(Evas_Object *obj, const char *name) EINA_ARG_NONNULL(1);
   EAPI const char           *elm_icon_standard_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_smooth_set(Evas_Object *obj, Eina_Bool smooth) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_icon_smooth_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_no_scale_set(Evas_Object *obj, Eina_Bool no_scale) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_icon_no_scale_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_scale_set(Evas_Object *obj, Eina_Bool scale_up, Eina_Bool scale_down) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_scale_get(const Evas_Object *obj, Eina_Bool *scale_up, Eina_Bool *scale_down) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_fill_outside_set(Evas_Object *obj, Eina_Bool fill_outside) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_icon_fill_outside_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_prescale_set(Evas_Object *obj, int size) EINA_ARG_NONNULL(1);
   EAPI int                   elm_icon_prescale_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_icon_order_lookup_set(Evas_Object *obj, Elm_Icon_Lookup_Order order) EINA_ARG_NONNULL(1);
   EAPI Elm_Icon_Lookup_Order elm_icon_order_lookup_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the icon
    */

   /* image */
   typedef enum _Elm_Image_Orient
     {
	ELM_IMAGE_ORIENT_NONE,
	ELM_IMAGE_ROTATE_90_CW,
	ELM_IMAGE_ROTATE_180_CW,
	ELM_IMAGE_ROTATE_90_CCW,
	ELM_IMAGE_FLIP_HORIZONTAL,
	ELM_IMAGE_FLIP_VERTICAL,
	ELM_IMAGE_FLIP_TRANSPOSE,
	ELM_IMAGE_FLIP_TRANSVERSE
     } Elm_Image_Orient;
   EAPI Evas_Object *elm_image_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_image_file_set(Evas_Object *obj, const char *file, const char *group) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_image_file_get(const Evas_Object *obj, const char **file, const char **group) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_smooth_set(Evas_Object *obj, Eina_Bool smooth) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_image_smooth_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_object_size_get(const Evas_Object *obj, int *w, int *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_no_scale_set(Evas_Object *obj, Eina_Bool no_scale) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_image_no_scale_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_scale_set(Evas_Object *obj, Eina_Bool scale_up, Eina_Bool scale_down) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_scale_get(const Evas_Object *obj, Eina_Bool *scale_up, Eina_Bool *scale_down) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_fill_outside_set(Evas_Object *obj, Eina_Bool fill_outside) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_image_fill_outside_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_prescale_set(Evas_Object *obj, int size) EINA_ARG_NONNULL(1);
   EAPI int          elm_image_prescale_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_orient_set(Evas_Object *obj, Elm_Image_Orient orient) EINA_ARG_NONNULL(1);
   EAPI Elm_Image_Orient elm_image_orient_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_image_editable_set(Evas_Object *obj, Eina_Bool set) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_image_editable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the image
    */

   /* box */
   typedef struct _Elm_Box_Transition Elm_Box_Transition;

   EAPI Evas_Object *elm_box_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_box_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_homogenous_set(Evas_Object *obj, Eina_Bool homogenous) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_box_homogenous_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_pack_start(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_pack_end(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_pack_before(Evas_Object *obj, Evas_Object *subobj, Evas_Object *before) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_pack_after(Evas_Object *obj, Evas_Object *subobj, Evas_Object *after) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_unpack(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_unpack_all(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_box_children_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_padding_set(Evas_Object *obj, Evas_Coord horizontal, Evas_Coord vertical) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_padding_get(const Evas_Object *obj, Evas_Coord *horizontal, Evas_Coord *vertical) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_align_set(Evas_Object *obj, double horizontal, double vertical) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_align_get(const Evas_Object *obj, double *horizontal, double *vertical) EINA_ARG_NONNULL(1);

   EAPI void         elm_box_layout_set(Evas_Object *obj, Evas_Object_Box_Layout cb, const void *data, void (*free_data)(void *data)) EINA_ARG_NONNULL(1);
   EAPI void         elm_box_layout_transition(Evas_Object *obj, Evas_Object_Box_Data *priv, void *data);
   EAPI Elm_Box_Transition *elm_box_transition_new(const double duration, Evas_Object_Box_Layout start_layout, void *start_layout_data, void(*start_layout_free_data)(void *data), Evas_Object_Box_Layout end_layout, void *end_layout_data, void(*end_layout_free_data)(void *data), void(*transition_end_cb)(void *data), void *transition_end_data) EINA_ARG_NONNULL(2, 5);
   EAPI void         elm_box_transition_free(void *data);
   /* smart callbacks called:
    */

   /* button */
   EAPI Evas_Object *elm_button_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_button_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_button_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_button_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_button_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_button_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_button_autorepeat_set(Evas_Object *obj, Eina_Bool on) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_button_autorepeat_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_button_autorepeat_initial_timeout_set(Evas_Object *obj, double t) EINA_ARG_NONNULL(1);
   EAPI double       elm_button_autorepeat_initial_timeout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_button_autorepeat_gap_timeout_set(Evas_Object *obj, double t) EINA_ARG_NONNULL(1);
   EAPI double       elm_button_autorepeat_gap_timeout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * anchor
    * hoversel_vertical
    * hoversel_vertical_entry
    */
   /* smart callbacks called:
    * "clicked" - the user clicked the button
    * "repeated" - the user pressed the button without releasing it
    * "unpressed" - when the button is unpressed (released)
    */

   /* fileselector */
   EAPI Evas_Object *elm_fileselector_button_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_button_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_fileselector_button_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_fileselector_button_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_window_title_set(Evas_Object *obj, const char *title) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_button_window_title_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_window_size_set(Evas_Object *obj, Evas_Coord width, Evas_Coord height) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_window_size_get(const Evas_Object *obj, Evas_Coord *width, Evas_Coord *height) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_path_set(Evas_Object *obj, const char *path) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_button_path_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_expandable_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_button_expandable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_folder_only_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_button_folder_only_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_is_save_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_button_is_save_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_button_inwin_mode_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_button_inwin_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * anchor
    * hoversel_vertical
    * hoversel_vertical_entry
    */
   /* smart callbacks called:
    * "file,chosen" - the user has selected a path, whose string pointer comes
                      as event info
    */

   EAPI Evas_Object *elm_fileselector_entry_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_button_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_entry_button_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_button_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_fileselector_entry_button_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_fileselector_entry_button_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_window_title_set(Evas_Object *obj, const char *title) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_entry_window_title_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_window_size_set(Evas_Object *obj, Evas_Coord width, Evas_Coord height) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_window_size_get(const Evas_Object *obj, Evas_Coord *width, Evas_Coord *height) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_path_set(Evas_Object *obj, const char *path) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_entry_path_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_expandable_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_entry_expandable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_folder_only_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_entry_folder_only_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_is_save_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_entry_is_save_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_inwin_mode_set(Evas_Object *obj, Eina_Bool value) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_entry_inwin_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_entry_selected_set(Evas_Object *obj, const char *path) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_entry_selected_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* scroller policy */
   typedef enum _Elm_Scroller_Policy
     {
	ELM_SCROLLER_POLICY_AUTO = 0,
	ELM_SCROLLER_POLICY_ON,
	ELM_SCROLLER_POLICY_OFF,
	ELM_SCROLLER_POLICY_LAST
     }
   Elm_Scroller_Policy;

   EAPI Evas_Object *elm_scroller_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_content_set(Evas_Object *obj, Evas_Object *child) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_scroller_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_scroller_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_custom_widget_base_theme_set(Evas_Object *obj, const char *widget, const char *base) EINA_ARG_NONNULL(1, 2, 3);
   EAPI void         elm_scroller_content_min_limit(Evas_Object *obj, Eina_Bool w, Eina_Bool h) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_region_show(Evas_Object *obj, Evas_Coord x, Evas_Coord y, Evas_Coord w, Evas_Coord h) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_policy_set(Evas_Object *obj, Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_policy_get(const Evas_Object *obj, Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_region_get(const Evas_Object *obj, Evas_Coord *x, Evas_Coord *y, Evas_Coord *w, Evas_Coord *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_child_size_get(const Evas_Object *obj, Evas_Coord *w, Evas_Coord *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_page_relative_set(Evas_Object *obj, double h_pagerel, double v_pagerel) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_page_size_set(Evas_Object *obj, Evas_Coord h_pagesize, Evas_Coord v_pagesize) EINA_ARG_NONNULL(1);
   EAPI void         elm_scroller_region_bring_in(Evas_Object *obj, Evas_Coord x, Evas_Coord y, Evas_Coord w, Evas_Coord h) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "edge,left"
    * "edge,right"
    * "edge,top"
    * "edge,bottom"
    * "scroll"
    * "scroll,anim,start"
    * "scroll,anim,stop"
    * "scroll,drag,start"
    * "scroll,drag,stop"
    */

   /* label */
   EAPI Evas_Object *elm_label_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_label_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_line_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_label_line_wrap_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_wrap_width_set(Evas_Object *obj, Evas_Coord w) EINA_ARG_NONNULL(1);
   EAPI Evas_Coord   elm_label_wrap_width_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_fontsize_set(Evas_Object *obj, int fontsize) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_text_color_set(Evas_Object *obj, unsigned int r, unsigned int g, unsigned int b, unsigned int a) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_text_align_set(Evas_Object *obj, const char *alignmode) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_background_color_set(Evas_Object *obj, unsigned int r, unsigned int g, unsigned int b, unsigned int a) EINA_ARG_NONNULL(1);
   EAPI void         elm_label_ellipsis_set(Evas_Object *obj, Eina_Bool ellipsis) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * marker
    */
   /* smart callbacks called:
    */

   /* toggle */
   EAPI Evas_Object *elm_toggle_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_toggle_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_toggle_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_toggle_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_states_labels_set(Evas_Object *obj, const char *onlabel, const char *offlabel) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_states_labels_get(const Evas_Object *obj, const char **onlabel, const char **offlabel) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_state_set(Evas_Object *obj, Eina_Bool state) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_toggle_state_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_toggle_state_pointer_set(Evas_Object *obj, Eina_Bool *statep) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "changed" - the user toggled the state
    */

   /* frame */
   EAPI Evas_Object *elm_frame_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_frame_label_set(Evas_Object *obj, const char *label); EINA_ARG_NONNULL(1)
   EAPI const char  *elm_frame_label_get(const Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI void         elm_frame_content_set(Evas_Object *obj, Evas_Object *content); EINA_ARG_NONNULL(1)
   EAPI Evas_Object *elm_frame_content_get(const Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI Evas_Object *elm_frame_content_unset(Evas_Object *obj); EINA_ARG_NONNULL(1)
   /* available styles:
    * default
    * pad_small
    * pad_medium
    * pad_large
    * pad_huge
    * outdent_top
    * outdent_bottom
    */
   /* smart callbacks called:
    */

   /* table */
   EAPI Evas_Object *elm_table_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_table_homogenous_set(Evas_Object *obj, Eina_Bool homogenous) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_table_homogeneous_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_table_padding_set(Evas_Object *obj, Evas_Coord horizontal, Evas_Coord vertical) EINA_ARG_NONNULL(1);
   EAPI void         elm_table_padding_get(const Evas_Object *obj, Evas_Coord *horizontal, Evas_Coord *vertical) EINA_ARG_NONNULL(1);
   EAPI void         elm_table_pack(Evas_Object *obj, Evas_Object *subobj, int x, int y, int w, int h) EINA_ARG_NONNULL(1);
   EAPI void         elm_table_unpack(Evas_Object *obj, Evas_Object *subobj) EINA_ARG_NONNULL(1);
   EAPI void         elm_table_clear(Evas_Object *obj, Eina_Bool clear) EINA_ARG_NONNULL(1);

   /* gengrid */
   typedef struct _Elm_Gengrid_Item_Class Elm_Gengrid_Item_Class;
   typedef struct _Elm_Gengrid_Item_Class_Func Elm_Gengrid_Item_Class_Func;
   typedef struct _Elm_Gengrid_Item Elm_Gengrid_Item; /**< Item of Elm_Gengrid. Sub-type of Elm_Widget_Item */
   typedef char        *(*GridItemLabelGetFunc) (void *data, Evas_Object *obj, const char *part);
   typedef Evas_Object *(*GridItemIconGetFunc)  (void *data, Evas_Object *obj, const char *part);
   typedef Eina_Bool    (*GridItemStateGetFunc) (void *data, Evas_Object *obj, const char *part);
   typedef void         (*GridItemDelFunc)      (void *data, Evas_Object *obj);

   struct _Elm_Gengrid_Item_Class
     {
	const char *item_style;
	struct _Elm_Gengrid_Item_Class_Func
          {
	     GridItemLabelGetFunc label_get;
	     GridItemIconGetFunc icon_get;
	     GridItemStateGetFunc state_get;
	     GridItemDelFunc del;
	  } func;
     };

   EAPI Evas_Object *elm_gengrid_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_size_set(Evas_Object *obj, Evas_Coord w, Evas_Coord h) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_size_get(const Evas_Object *obj, Evas_Coord *w, Evas_Coord *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_align_set(Evas_Object *obj, double align_x, double align_y) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_align_get(const Evas_Object *obj, double *align_x, double *align_y) EINA_ARG_NONNULL(1);

   EAPI void	     elm_gengrid_always_select_mode_set(Evas_Object *obj, Eina_Bool always_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_always_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_gengrid_no_select_mode_set(Evas_Object *obj, Eina_Bool no_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_no_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_gengrid_multi_select_set(Evas_Object *obj, Eina_Bool multi) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_multi_select_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_gengrid_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void	     elm_gengrid_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_page_relative_set(Evas_Object *obj, double h_pagerel, double v_pagerel) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_page_size_set(Evas_Object *obj, Evas_Coord h_pagesize, Evas_Coord v_pagesize) EINA_ARG_NONNULL(1);
   EAPI void	     elm_gengrid_horizontal_set(Evas_Object *obj, Eina_Bool setting) EINA_ARG_NONNULL(1);

   EAPI Elm_Gengrid_Item *elm_gengrid_first_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_last_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI Elm_Gengrid_Item *elm_gengrid_item_next_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_item_prev_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI Evas_Object      *elm_gengrid_item_gengrid_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_del(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_update(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void        *elm_gengrid_item_data_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_data_set(Elm_Gengrid_Item *item, const void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_pos_get(const Elm_Gengrid_Item *item, unsigned int *x, unsigned int *y) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_selected_set(Elm_Gengrid_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_item_selected_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI const Evas_Object *elm_gengrid_item_object_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_show(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_bring_in(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_disabled_set(Elm_Gengrid_Item *item, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_item_disabled_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);

   EAPI void         elm_gengrid_item_tooltip_text_set(Elm_Gengrid_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_tooltip_content_cb_set(Elm_Gengrid_Item *item, Elm_Tooltip_Item_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_tooltip_unset(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_tooltip_style_set(Elm_Gengrid_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_gengrid_item_tooltip_style_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_cursor_set(Elm_Gengrid_Item *item, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_gengrid_item_cursor_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_cursor_unset(Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_cursor_style_set(Elm_Gengrid_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_gengrid_item_cursor_style_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_gengrid_item_cursor_engine_only_set(Elm_Gengrid_Item *item, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_gengrid_item_cursor_engine_only_get(const Elm_Gengrid_Item *item) EINA_ARG_NONNULL(1);

   EAPI void	     elm_gengrid_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_gengrid_selected_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI Elm_Gengrid_Item *elm_gengrid_item_append(Evas_Object *obj, const Elm_Gengrid_Item_Class *gic,  const void *data, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_item_prepend(Evas_Object *obj, const Elm_Gengrid_Item_Class *gic, const void *data, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_item_insert_before(Evas_Object *obj, const Elm_Gengrid_Item_Class *gic, const void *data, Elm_Gengrid_Item *relative, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   EAPI Elm_Gengrid_Item *elm_gengrid_item_insert_after(Evas_Object *obj, const Elm_Gengrid_Item_Class *gic, const void *data, Elm_Gengrid_Item *relative, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    *
    * selected - User has selected a item.
    * unselected - User has unselected a item.
    * clicked - User has double-clicked a item.
    * realized - An evas object for a item was built.
    * unrealized - An evas object for a item was deleted.
    * changed - An item has been added, removed, resized or moved,
    * or gengrid has been resized or horizontal property has been changed.
    * scroll - the content has been scrolled (moved).
    * "scroll,drag,start" - dragging the contents around has started.
    * "scroll,drat,stop" - dragging the contents around has stopped.
    * drag - Gengrid is being dragged.
    * "drag,start,up" - Gengrid has been dragged (not scrolled) up.
    * "drag,start,down" - Gengrid has been dragged (not scrolled) down.
    * "drag,start,left" - Gengrid has been dragged (not scrolled) left.
    * "drag,start,rigth" - Gengrid has been dragged (nto scrolled) right.
    * "drag,stop" - Gengrid is not being dragged.
    */

   /* clock */
   typedef enum _Elm_Clock_Digedit
     {
        ELM_CLOCK_NONE         = 0,
        ELM_CLOCK_HOUR_DECIMAL = 1 << 0,
        ELM_CLOCK_HOUR_UNIT    = 1 << 1,
        ELM_CLOCK_MIN_DECIMAL  = 1 << 2,
        ELM_CLOCK_MIN_UNIT     = 1 << 3,
        ELM_CLOCK_SEC_DECIMAL  = 1 << 4,
        ELM_CLOCK_SEC_UNIT     = 1 << 5,
        ELM_CLOCK_ALL	       = (1 << 6) - 1
     } Elm_Clock_Digedit;

   EAPI Evas_Object *elm_clock_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_clock_time_set(Evas_Object *obj, int hrs, int min, int sec) EINA_ARG_NONNULL(1);
   EAPI void         elm_clock_time_get(const Evas_Object *obj, int *hrs, int *min, int *sec) EINA_ARG_NONNULL(1);
   EAPI void         elm_clock_edit_set(Evas_Object *obj, Eina_Bool edit) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_clock_edit_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_clock_digit_edit_set(Evas_Object *obj, Elm_Clock_Digedit digedit) EINA_ARG_NONNULL(1);
   EAPI Elm_Clock_Digedit elm_clock_digit_edit_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_clock_show_am_pm_set(Evas_Object *obj, Eina_Bool am_pm) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_clock_show_am_pm_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_clock_show_seconds_set(Evas_Object *obj, Eina_Bool seconds) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_clock_show_seconds_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_clock_interval_set(Evas_Object *obj, double interval) EINA_ARG_NONNULL(1);
   EAPI double	     elm_clock_interval_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "changed" - the user changed the time
    */

   /* layout */
   EAPI Evas_Object *elm_layout_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_layout_file_set(Evas_Object *obj, const char *file, const char *group) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_layout_theme_set(Evas_Object *obj, const char *clas, const char *group, const char *style) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_content_set(Evas_Object *obj, const char *swallow, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI const Evas_Object *elm_layout_content_get(const Evas_Object *obj, const char *swallow) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_layout_content_unset(Evas_Object *obj, const char *swallow) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_text_set(Evas_Object *obj, const char *part, const char *text) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_layout_text_get(const Evas_Object *obj, const char *part) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_box_append(Evas_Object *obj, const char *part, Evas_Object *child) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_box_prepend(Evas_Object *obj, const char *part, Evas_Object *child) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_box_insert_before(Evas_Object *obj, const char *part, Evas_Object *child, const Evas_Object *reference) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_box_insert_at(Evas_Object *obj, const char *part, Evas_Object *child, unsigned int pos) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_layout_box_remove(Evas_Object *obj, const char *part, Evas_Object *child) EINA_ARG_NONNULL(1, 2, 3);
   EAPI void         elm_layout_box_remove_all(Evas_Object *obj, const char *part, Eina_Bool clear) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_layout_table_pack(Evas_Object *obj, const char *part, Evas_Object *child_obj, unsigned short col, unsigned short row, unsigned short colspan, unsigned short rowspan) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_layout_table_unpack(Evas_Object *obj, const char *part, Evas_Object *child_obj) EINA_ARG_NONNULL(1, 2, 3);
   EAPI void         elm_layout_table_clear(Evas_Object *obj, const char *part, Eina_Bool clear) EINA_ARG_NONNULL(1, 2);
   EAPI Evas_Object *elm_layout_edje_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_layout_sizing_eval(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool     elm_layout_part_cursor_set(Evas_Object *obj, const char *part_name, const char *cursor) EINA_ARG_NONNULL(1, 2);
   EAPI const char   *elm_layout_part_cursor_get(const Evas_Object *obj, const char *part_name) EINA_ARG_NONNULL(1, 2);
   EAPI void          elm_layout_part_cursor_unset(Evas_Object *obj, const char *part_name) EINA_ARG_NONNULL(1, 2);
   EAPI Eina_Bool     elm_layout_part_cursor_style_set(Evas_Object *obj, const char *part_name, const char *style) EINA_ARG_NONNULL(1, 2);
   EAPI const char   *elm_layout_part_cursor_style_get(const Evas_Object *obj, const char *part_name) EINA_ARG_NONNULL(1, 2);
   EAPI Eina_Bool     elm_layout_part_cursor_engine_only_set(Evas_Object *obj, const char *part_name, Eina_Bool engine_only) EINA_ARG_NONNULL(1, 2);
   EAPI Eina_Bool     elm_layout_part_cursor_engine_only_get(const Evas_Object *obj, const char *part_name) EINA_ARG_NONNULL(1, 2);
/**
 * @def elm_layout_icon_set
 * Convienience macro to set the icon object in a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_icon_set(_ly, _obj) \
  do {                                                             \
       const char *sig;                                            \
       elm_layout_content_set((_ly), "elm.swallow.icon", (_obj));  \
       if ((_obj))                                                 \
          sig = "elm,state,icon,visible";                          \
       else                                                        \
          sig = "elm,state,icon,hidden";                           \
       elm_object_signal_emit((_ly), sig, "elm");                  \
  } while(0)

/**
 * @def elm_layout_icon_get
 * Convienience macro to get the icon object from a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_icon_get(_ly) \
  elm_layout_content_get((_ly), "elm.swallow.icon")

/**
 * @def elm_layout_end_set
 * Convienience macro to set the end object in a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_end_set(_ly, _obj) \
  do {                                                             \
       const char *sig;                                            \
       elm_layout_content_set((_ly), "elm.swallow.end", (_obj));  \
       if ((_obj))                                                 \
          sig = "elm,state,end,visible";                          \
       else                                                        \
          sig = "elm,state,end,hidden";                           \
       elm_object_signal_emit((_ly), sig, "elm");                  \
  } while(0)

/**
 * @def elm_layout_end_get
 * Convienience macro to get the end object in a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_end_get(_ly) \
  elm_layout_content_get((_ly), "elm.swallow.end")

/**
 * @def elm_layout_label_set
 * Convienience macro to set the label in a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_label_set(_ly, _txt) \
  elm_layout_text_set((_ly), "elm.text", (_txt))

/**
 * @def elm_layout_label_get
 * Convienience macro to get the label in a layout that follows the
 * Elementary naming convention for its parts.
 *
 * @ingroup Layout
 */
#define elm_layout_label_get(_ly) \
  elm_layout_text_get((_ly), "elm.text")

   /* smart callbacks called:
    */

   /* notify */
   typedef enum _Elm_Notify_Orient
     {
	 ELM_NOTIFY_ORIENT_TOP,
	 ELM_NOTIFY_ORIENT_CENTER,
	 ELM_NOTIFY_ORIENT_BOTTOM,
	 ELM_NOTIFY_ORIENT_LEFT,
	 ELM_NOTIFY_ORIENT_RIGHT,
	 ELM_NOTIFY_ORIENT_TOP_LEFT,
	 ELM_NOTIFY_ORIENT_TOP_RIGHT,
	 ELM_NOTIFY_ORIENT_BOTTOM_LEFT,
	 ELM_NOTIFY_ORIENT_BOTTOM_RIGHT,
	 ELM_NOTIFY_ORIENT_LAST
     } Elm_Notify_Orient;
   EAPI Evas_Object *elm_notify_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_notify_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_notify_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_notify_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_notify_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_notify_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_notify_orient_set(Evas_Object *obj, Elm_Notify_Orient orient) EINA_ARG_NONNULL(1);
   EAPI Elm_Notify_Orient	    elm_notify_orient_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_notify_timeout_set(Evas_Object *obj, double timeout) EINA_ARG_NONNULL(1);
   EAPI double       elm_notify_timeout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_notify_repeat_events_set(Evas_Object *obj, Eina_Bool repeat) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_notify_repeat_events_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "timeout" - when timeout happens on notify and it's hidden
    * "block,clicked" - when it's hidden by a click outside of the notify's view
    */

   /* hover */
   typedef enum _Elm_Hover_Axis
     {
	ELM_HOVER_AXIS_NONE,
	  ELM_HOVER_AXIS_HORIZONTAL,
	  ELM_HOVER_AXIS_VERTICAL,
	  ELM_HOVER_AXIS_BOTH
     } Elm_Hover_Axis;
   EAPI Evas_Object *elm_hover_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_hover_target_set(Evas_Object *obj, Evas_Object *target) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hover_target_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hover_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hover_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hover_content_set(Evas_Object *obj, const char *swallow, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hover_content_get(const Evas_Object *obj, const char *swallow) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hover_content_unset(Evas_Object *obj, const char *swallow) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_hover_best_content_location_get(const Evas_Object *obj, Elm_Hover_Axis pref_axis) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * popout
    * menu
    * hoversel_vertical
    */
   /* smart callbacks called:
    * "clicked" - the user clicked the empty space in the hover to dismiss
    * "smart,changed" - a content object placed under the "smart"
    *                   policy was replaced to a new slot direction.
    */

   /* entry */
   typedef struct _Elm_Entry_Anchor_Info Elm_Entry_Anchor_Info;
   struct _Elm_Entry_Anchor_Info
     {
	const char *name;
	int button;
	Evas_Coord x, y, w, h;
     };
   typedef enum _Elm_Icon_Type
     {
	ELM_ICON_NONE,
	  ELM_ICON_FILE,
	  ELM_ICON_STANDARD
     } Elm_Icon_Type;
   typedef struct _Elm_Hoversel_Item Elm_Hoversel_Item; /**< Item of Elm_Hoversel. Sub-type of Elm_Widget_Item */
  
   EAPI Evas_Object *elm_entry_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_single_line_set(Evas_Object *obj, Eina_Bool single_line) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_single_line_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_password_set(Evas_Object *obj, Eina_Bool password) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_password_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_entry_set(Evas_Object *obj, const char *entry) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_entry_entry_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_entry_selection_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_entry_insert(Evas_Object *obj, const char *entry) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_line_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_line_char_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_editable_set(Evas_Object *obj, Eina_Bool editable) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_editable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_select_none(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_select_all(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_next(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_prev(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_up(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_down(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_begin_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_end_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_line_begin_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_line_end_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_selection_begin(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_cursor_selection_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_is_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_cursor_is_visible_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_entry_cursor_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_selection_cut(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_selection_copy(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_selection_paste(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_context_menu_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_context_menu_item_add(Evas_Object *obj, const char *label, const char *icon_file, Elm_Icon_Type icon_type, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_context_menu_disabled_set(Evas_Object *obj, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_context_menu_disabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_item_provider_append(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_entry_item_provider_prepend(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_entry_item_provider_remove(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_entry_text_filter_append(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_entry_text_filter_prepend(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_entry_text_filter_remove(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI char        *elm_entry_markup_to_utf8(const char *s);
   EAPI char        *elm_entry_utf8_to_markup(const char *s);
   EAPI void         elm_entry_file_set(Evas_Object *obj, const char *file, Elm_Text_Format format) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_file_get(const Evas_Object *obj, const char **file, Elm_Text_Format *format) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_file_save(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_entry_autosave_set(Evas_Object *obj, Eina_Bool autosave) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_entry_autosave_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* pre-made filters for entries */
   typedef struct _Elm_Entry_Filter_Limit_Size Elm_Entry_Filter_Limit_Size;
   struct _Elm_Entry_Filter_Limit_Size
     {
        int max_char_count;
        int max_byte_count;
     };
   EAPI void         elm_entry_filter_limit_size(void *data, Evas_Object *entry, char **text) EINA_ARG_NONNULL(1, 2, 3);
   typedef struct _Elm_Entry_Filter_Accept_Set Elm_Entry_Filter_Accept_Set;
   struct _Elm_Entry_Filter_Accept_Set
     {
        const char *accepted;
        const char *rejected;
     };
   EAPI void         elm_entry_filter_accept_set(void *data, Evas_Object *entry, char **text) EINA_ARG_NONNULL(1, 3);
   /* smart callbacks called:
    * "changed" - the text content changed
    * "selection,start" - the user started selecting text
    * "selection,changed" - the user modified the selection size/location
    * "selection,cleared" - the user cleared the selection
    * "selection,paste" - the user requested a paste of text
    * "selection,copy" - the user copied the text
    * "selection,cut" - the user cut the text
    * "cursor,changed" - the cursor changed position
    * "anchor,clicked" - achor called was clicked | event_info = Elm_Entry_Anchor_Info
    * "activated" - when the enter key is pressed (useful for single line)
    * "press" - when finger/mouse is pressed down
    * "clicked" - when finger/mouse is pressed and released (without a drag etc.)
    * "clicked,double" - when finger/mouse is double-pressed
    * "longpressed" - the entry has been longpressed
    * "focused" - the entry has received keyboard focus
    * "unfocused" - keyboard focus is gone
    */

   /* composite widgets - these basically put together basic widgets above
    * in convenient packages that do more than basic stuff */

   /* anchorview */
   typedef struct _Elm_Entry_Anchorview_Info Elm_Entry_Anchorview_Info;
   struct _Elm_Entry_Anchorview_Info
     {
	const char *name;
	int button;
	Evas_Object *hover;
	struct
          {
             Evas_Coord x, y, w, h;
	  } anchor, hover_parent;
	Eina_Bool hover_left : 1;
	Eina_Bool hover_right : 1;
	Eina_Bool hover_top : 1;
	Eina_Bool hover_bottom : 1;
     };
   EAPI Evas_Object *elm_anchorview_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_text_set(Evas_Object *obj, const char *text) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_anchorview_text_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI void         elm_anchorview_hover_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_anchorview_hover_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_hover_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_anchorview_hover_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_hover_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorview_item_provider_append(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorview, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_anchorview_item_provider_prepend(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorview, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_anchorview_item_provider_remove(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorview, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   /* smart callbacks called:
    * "anchor,clicked" - achor called was clicked | event_info = Elm_Entry_Anchorview_Info
    */

   /* anchorblock */
   typedef struct _Elm_Entry_Anchorblock_Info Elm_Entry_Anchorblock_Info;
   struct _Elm_Entry_Anchorblock_Info
     {
	const char *name;
	int button;
	Evas_Object *hover;
	struct
          {
	     Evas_Coord x, y, w, h;
	  } anchor, hover_parent;
	Eina_Bool hover_left : 1;
	Eina_Bool hover_right : 1;
	Eina_Bool hover_top : 1;
	Eina_Bool hover_bottom : 1;
     };
   EAPI Evas_Object *elm_anchorblock_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorblock_text_set(Evas_Object *obj, const char *text) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_anchorblock_text_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorblock_hover_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_anchorblock_hover_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorblock_hover_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_anchorblock_hover_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorblock_hover_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_anchorblock_item_provider_append(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorblock, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_anchorblock_item_provider_prepend(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorblock, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_anchorblock_item_provider_remove(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *anchorblock, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   /* smart callbacks called:
    * "anchor,clicked" - achor called was clicked | event_info = Elm_Entry_Anchorblock_Info
    */

   /* bubble */
   EAPI Evas_Object *elm_bubble_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_bubble_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_bubble_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_bubble_info_set(Evas_Object *obj, const char *info) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_bubble_info_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   EAPI void         elm_bubble_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_bubble_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_bubble_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_bubble_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_bubble_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_bubble_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_bubble_corner_set(Evas_Object *obj, const char *corner) EINA_ARG_NONNULL(1, 2);
   EAPI const char*  elm_bubble_corner_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the bubble
    */

   /* photo */
   EAPI Evas_Object *elm_photo_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_photo_file_set(Evas_Object *obj, const char *file) EINA_ARG_NONNULL(1);
   EAPI void         elm_photo_size_set(Evas_Object *obj, int size) EINA_ARG_NONNULL(1);
   EAPI void         elm_photo_fill_inside_set(Evas_Object *obj, Eina_Bool fill) EINA_ARG_NONNULL(1);
   EAPI void         elm_photo_editable_set(Evas_Object *obj, Eina_Bool set) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the icon
    */

   /* thumb */
   typedef enum _Elm_Thumb_Animation_Setting
     {
       ELM_THUMB_ANIMATION_START = 0, /* Play animation once */
       ELM_THUMB_ANIMATION_LOOP,      /* Keep playing animation until stop is requested */
       ELM_THUMB_ANIMATION_STOP,
       ELM_THUMB_ANIMATION_LAST
     } Elm_Thumb_Animation_Setting;

   EAPI Evas_Object *elm_thumb_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_thumb_reload(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	     elm_thumb_file_set(Evas_Object *obj, const char *file, const char *key) EINA_ARG_NONNULL(1);
   EAPI void	     elm_thumb_file_get(const Evas_Object *obj, const char **file, const char **key) EINA_ARG_NONNULL(1);
   EAPI void	     elm_thumb_path_get(const Evas_Object *obj, const char **file, const char **key) EINA_ARG_NONNULL(1);
   EAPI void	     elm_thumb_animate_set(Evas_Object *obj, Elm_Thumb_Animation_Setting s) EINA_ARG_NONNULL(1);
   EAPI Elm_Thumb_Animation_Setting elm_thumb_animate_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI struct _Ethumb_Client *elm_thumb_ethumb_client_get(void);
   EAPI Eina_Bool    elm_thumb_ethumb_client_connected(void);
   EAPI Eina_Bool    elm_thumb_editable_set(Evas_Object *obj, Eina_Bool edit) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_thumb_editable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* available styles:
    * default
    * noframe
    *
    * smart callbacks called:
    * "clicked" - the user clicked the thumb
    * "clicked,double" - the user double clicked the thumb
    * "press" - the user pressed the thumb
    * "generate,start" - the thumbnail generation started
    * "generate,stop" - the thumbnail generation stopped
    * "generate,error" - the thumbnail generation failed
    * "load,error" - the thumbnail image loading failed
    */
  
   /* hoversel */
   EAPI Evas_Object *elm_hoversel_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_hoversel_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_hover_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hoversel_hover_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_hoversel_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hoversel_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_hoversel_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_hover_begin(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_hover_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_hoversel_expanded_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List * elm_hoversel_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Hoversel_Item *elm_hoversel_item_add(Evas_Object *obj, const char *label, const char *icon_file, Elm_Icon_Type icon_type, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_item_del(Elm_Hoversel_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_item_del_cb_set(Elm_Hoversel_Item *it, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI void        *elm_hoversel_item_data_get(const Elm_Hoversel_Item *it) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_hoversel_item_label_get(const Elm_Hoversel_Item *it) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_item_icon_set(Elm_Hoversel_Item *it, const char *icon_file, const char *icon_group, Elm_Icon_Type icon_type) EINA_ARG_NONNULL(1);
   EAPI void         elm_hoversel_item_icon_get(const Elm_Hoversel_Item *it, const char **icon_file, const char **icon_group, Elm_Icon_Type *icon_type) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the hoversel button and popped up the sel
    * "selected" - an item in the hoversel list is selected
    * "dismissed" - the hover is dismissed
    */

   /* toolbar */
   typedef enum _Elm_Toolbar_Shrink_Mode
     {
        ELM_TOOLBAR_SHRINK_NONE,   /**< set toolbar minimun size to fit all the items */
        ELM_TOOLBAR_SHRINK_HIDE,   /**< hide excess items */
        ELM_TOOLBAR_SHRINK_SCROLL, /**< allow accessing excess items through a scroller */
        ELM_TOOLBAR_SHRINK_MENU    /**< inserts a button to pop up a menu with excess items */
     } Elm_Toolbar_Shrink_Mode;

   typedef struct _Elm_Toolbar_Item Elm_Toolbar_Item; /**< Item of Elm_Toolbar. Sub-type of Elm_Widget_Item */
   typedef struct _Elm_Toolbar_Item_State Elm_Toolbar_Item_State; /** State of a Elm_Toolbar_Item */
  
   EAPI Evas_Object      *elm_toolbar_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_icon_size_set(Evas_Object *obj, int icon_size) EINA_ARG_NONNULL(1);
   EAPI int               elm_toolbar_icon_size_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_icon_order_lookup_set(Evas_Object *obj, Elm_Icon_Lookup_Order order) EINA_ARG_NONNULL(1);
   EAPI Elm_Icon_Lookup_Order elm_toolbar_icon_order_lookup_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_no_select_mode_set(Evas_Object *obj, Eina_Bool no_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_no_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_append(Evas_Object *obj, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_prepend(Evas_Object *obj, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_insert_before(Evas_Object *obj, Elm_Toolbar_Item *before, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_insert_after(Evas_Object *obj, Elm_Toolbar_Item *after, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_first_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_last_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_next_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_prev_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI Evas_Object      *elm_toolbar_item_toolbar_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_priority_set(Elm_Toolbar_Item *item, int priority) EINA_ARG_NONNULL(1);
   EAPI int               elm_toolbar_item_priority_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_toolbar_item_icon_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_toolbar_item_label_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_label_set(Elm_Toolbar_Item *item, const char *label) EINA_ARG_NONNULL(1);
   EAPI void             *elm_toolbar_item_data_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_data_set(Elm_Toolbar_Item *item, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_item_find_by_label(const Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_selected_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_selected_set(Elm_Toolbar_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item *elm_toolbar_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_icon_set(Elm_Toolbar_Item *item, const char *icon) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_del(Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_del_cb_set(Elm_Toolbar_Item *item, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_disabled_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_disabled_set(Elm_Toolbar_Item *item, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_separator_set(Elm_Toolbar_Item *item, Eina_Bool separator) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_separator_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_mode_shrink_set(Evas_Object *obj, Elm_Toolbar_Shrink_Mode shrink_mode) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Shrink_Mode elm_toolbar_mode_shrink_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_homogenous_set(Evas_Object *obj, Eina_Bool homogenous) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_homogenous_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_menu_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object      *elm_toolbar_menu_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_align_set(Evas_Object *obj, double align) EINA_ARG_NONNULL(1);
   EAPI double            elm_toolbar_align_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_menu_set(Elm_Toolbar_Item *item, Eina_Bool menu) EINA_ARG_NONNULL(1);
   EAPI Evas_Object      *elm_toolbar_item_menu_get(Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item_State *elm_toolbar_item_state_add(Elm_Toolbar_Item *item, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_state_del(Elm_Toolbar_Item *item, Elm_Toolbar_Item_State *state) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_state_set(Elm_Toolbar_Item *it, Elm_Toolbar_Item_State *state) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_state_unset(Elm_Toolbar_Item *it) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item_State *elm_toolbar_item_state_get(const Elm_Toolbar_Item *it) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item_State *elm_toolbar_item_state_next(Elm_Toolbar_Item *it) EINA_ARG_NONNULL(1);
   EAPI Elm_Toolbar_Item_State *elm_toolbar_item_state_prev(Elm_Toolbar_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_tooltip_text_set(Elm_Toolbar_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_tooltip_content_cb_set(Elm_Toolbar_Item *item, Elm_Tooltip_Item_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_tooltip_unset(Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_tooltip_style_set(Elm_Toolbar_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_toolbar_item_tooltip_style_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_cursor_set(Elm_Toolbar_Item *item, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_toolbar_item_cursor_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_cursor_unset(Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_cursor_style_set(Elm_Toolbar_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_toolbar_item_cursor_style_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_toolbar_item_cursor_engine_only_set(Elm_Toolbar_Item *item, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_toolbar_item_cursor_engine_only_get(const Elm_Toolbar_Item *item) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - when the user clicks on a toolbar item and becomes selected
    */
   /* available styles:
    * default
    * transparent (no background or shadow, just show the provided content)
    */
  
   /* tooltip */
   EAPI double       elm_tooltip_delay_get(void);
   EAPI Eina_Bool    elm_tooltip_delay_set(double delay);
   EAPI void         elm_object_tooltip_show(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_tooltip_hide(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_tooltip_text_set(Evas_Object *obj, const char *text) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_object_tooltip_content_cb_set(Evas_Object *obj, Elm_Tooltip_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_tooltip_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_tooltip_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_object_tooltip_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_cursor_set(Evas_Object *obj, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_object_cursor_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_cursor_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_cursor_style_set(Evas_Object *obj, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_object_cursor_style_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_object_cursor_engine_only_set(Evas_Object *obj, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_object_cursor_engine_only_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI int          elm_cursor_engine_only_get(void);
   EAPI Eina_Bool    elm_cursor_engine_only_set(int engine_only);

   /* menu */
   typedef struct _Elm_Menu_Item Elm_Menu_Item; /**< Item of Elm_Menu. Sub-type of Elm_Widget_Item */
   EAPI Evas_Object       *elm_menu_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Object       *elm_menu_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_move(Evas_Object *obj, Evas_Coord x, Evas_Coord y) EINA_ARG_NONNULL(1);
   EAPI Evas_Object       *elm_menu_object_get(const Elm_Menu_Item *it) EINA_ARG_NONNULL(1);
   EAPI Elm_Menu_Item     *elm_menu_item_add(Evas_Object *obj, Elm_Menu_Item *parent, const char *icon, const char *label, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_label_set(Elm_Menu_Item *item, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char        *elm_menu_item_label_get(const Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_icon_set(Elm_Menu_Item *item, const char *icon) EINA_ARG_NONNULL(1, 2);
   EAPI const char        *elm_menu_item_icon_get(const Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI const Evas_Object *elm_menu_item_object_icon_get(const Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_disabled_set(Elm_Menu_Item *item, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool          elm_menu_item_disabled_get(const Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Menu_Item     *elm_menu_item_separator_add(Evas_Object *obj, Elm_Menu_Item *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool          elm_menu_item_is_separator(Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_del(Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_del_cb_set(Elm_Menu_Item *it, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI void              *elm_menu_item_data_get(const Elm_Menu_Item *it) EINA_ARG_NONNULL(1);
   EAPI void               elm_menu_item_data_set(Elm_Menu_Item *item, const void *data) EINA_ARG_NONNULL(1);
   EAPI const Eina_List   *elm_menu_item_subitems_get(const Elm_Menu_Item *item) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - the user clicked the empty space in the menu to dismiss. event_info is NULL.
    */

   /* list */
   typedef enum _Elm_List_Mode
     {
	ELM_LIST_COMPRESS = 0,
	ELM_LIST_SCROLL,
	ELM_LIST_LIMIT,
        ELM_LIST_EXPAND,
	ELM_LIST_LAST
     } Elm_List_Mode;
   typedef struct _Elm_List_Item Elm_List_Item; /**< Item of Elm_List. Sub-type of Elm_Widget_Item */
   EAPI Evas_Object     *elm_list_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_item_append(Evas_Object *obj, const char *label, Evas_Object *icon, Evas_Object *end, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_item_prepend(Evas_Object *obj, const char *label, Evas_Object *icon, Evas_Object *end, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_item_insert_before(Evas_Object *obj, Elm_List_Item *before, const char *label, Evas_Object *icon, Evas_Object *end, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
   EAPI Elm_List_Item   *elm_list_item_insert_after(Evas_Object *obj, Elm_List_Item *after, const char *label, Evas_Object *icon, Evas_Object *end, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
   EAPI Elm_List_Item   *elm_list_item_sorted_insert(Evas_Object *obj, const char *label, Evas_Object *icon, Evas_Object *end, Evas_Smart_Cb func, const void *data, Eina_Compare_Cb cmp_func) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_go(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_multi_select_set(Evas_Object *obj, Eina_Bool multi) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_multi_select_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_mode_set(Evas_Object *obj, Elm_List_Mode mode) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Mode    elm_list_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_always_select_mode_set(Evas_Object *obj, Eina_Bool always_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool	 elm_list_always_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_list_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_list_selected_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_separator_set(Elm_List_Item *it, Eina_Bool setting) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_item_separator_get(const Elm_List_Item *it) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_selected_set(Elm_List_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_item_selected_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_show(Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_bring_in(Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_del(Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_del_cb_set(Elm_List_Item *item, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI void            *elm_list_item_data_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI Evas_Object     *elm_list_item_icon_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_icon_set(Elm_List_Item *item, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object     *elm_list_item_end_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_end_set(Elm_List_Item *item, Evas_Object *end) EINA_ARG_NONNULL(1);
   EAPI Evas_Object     *elm_list_item_base_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI const char      *elm_list_item_label_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_label_set(Elm_List_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_item_prev(const Elm_List_Item *it) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Item   *elm_list_item_next(const Elm_List_Item *it) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_tooltip_text_set(Elm_List_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_tooltip_content_cb_set(Elm_List_Item *item, Elm_Tooltip_Item_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_tooltip_unset(Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_tooltip_style_set(Elm_List_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char      *elm_list_item_tooltip_style_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_cursor_set(Elm_List_Item *item, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char      *elm_list_item_cursor_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_cursor_unset(Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_cursor_style_set(Elm_List_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char      *elm_list_item_cursor_style_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_cursor_engine_only_set(Elm_List_Item *item, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_item_cursor_engine_only_get(const Elm_List_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_item_disabled_set(Elm_List_Item *it, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool        elm_list_item_disabled_get(const Elm_List_Item *it) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_scroller_policy_set(Evas_Object *obj, Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v) EINA_ARG_NONNULL(1);
   EAPI void             elm_list_scroller_policy_get(const Evas_Object *obj, Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - when the user double-clicked an item
    * "selected" - when the user selected an item
    * "unselected" - when the user selected an item
    * "longpressed" - an item in the hoversel list is long-pressed
    * "scroll,edge,top" - the list is scrolled until the top edge
    * "scroll,edge,bottom" - the list is scrolled until the bottom edge
    * "scroll,edge,left" - the list is scrolled until the left edge
    * "scroll,edge,right" - the list is scrolled until the right edge
    */

   /* slider */
   EAPI Evas_Object       *elm_slider_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char        *elm_slider_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object       *elm_slider_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_slider_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_end_set(Evas_Object *obj, Evas_Object *end) EINA_ARG_NONNULL(1);
   EAPI Evas_Object       *elm_slider_end_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_slider_end_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_span_size_set(Evas_Object *obj, Evas_Coord size) EINA_ARG_NONNULL(1);
   EAPI Evas_Coord         elm_slider_span_size_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_unit_format_set(Evas_Object *obj, const char *format) EINA_ARG_NONNULL(1);
   EAPI const char        *elm_slider_unit_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_indicator_format_set(Evas_Object *obj, const char *indicator) EINA_ARG_NONNULL(1);
   EAPI const char        *elm_slider_indicator_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_indicator_format_function_set(Evas_Object *obj, const char *(*func)(double val)) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool          elm_slider_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_min_max_set(Evas_Object *obj, double min, double max) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_min_max_get(const Evas_Object *obj, double *min, double *max) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_value_set(Evas_Object *obj, double val) EINA_ARG_NONNULL(1);
   EAPI double             elm_slider_value_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_inverted_set(Evas_Object *obj, Eina_Bool inverted) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool          elm_slider_inverted_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void               elm_slider_indicator_show_set(Evas_Object *obj, Eina_Bool show) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool          elm_slider_indicator_show_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* actionslider */
   typedef enum _Elm_Actionslider_Pos
     {
        ELM_ACTIONSLIDER_NONE = 0,
        ELM_ACTIONSLIDER_LEFT = 1 << 0,
        ELM_ACTIONSLIDER_CENTER = 1 << 1,
        ELM_ACTIONSLIDER_RIGHT = 1 << 2,
        ELM_ACTIONSLIDER_ALL = (1 << 3) -1
     } Elm_Actionslider_Pos;

   EAPI Evas_Object          *elm_actionslider_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void                  elm_actionslider_labels_set(Evas_Object *obj, const char *left_label, const char *center_label, const char *right_label) EINA_ARG_NONNULL(1);
   EAPI void                  elm_actionslider_labels_get(const Evas_Object *obj, const char **left_label, const char **center_label, const char **right_label) EINA_ARG_NONNULL(1);
   EAPI const char           *elm_actionslider_selected_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_actionslider_indicator_pos_set(Evas_Object *obj, Elm_Actionslider_Pos pos) EINA_ARG_NONNULL(1);
   EAPI Elm_Actionslider_Pos  elm_actionslider_indicator_pos_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_actionslider_magnet_pos_set(Evas_Object *obj, Elm_Actionslider_Pos pos) EINA_ARG_NONNULL(1);
   EAPI Elm_Actionslider_Pos  elm_actionslider_magnet_pos_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_actionslider_enabled_pos_set(Evas_Object *obj, Elm_Actionslider_Pos pos) EINA_ARG_NONNULL(1);
   EAPI Elm_Actionslider_Pos  elm_actionslider_enabled_pos_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "selected" - when user selects a position (the label is passed as
    * event info)".
    * "pos_changed" - when a button reaches to the special position like
    * "left", "right" and "center".
    */

   /* genlist */
   typedef enum _Elm_Genlist_Item_Flags
     {
	ELM_GENLIST_ITEM_NONE = 0,
	ELM_GENLIST_ITEM_SUBITEMS = (1 << 0)
     } Elm_Genlist_Item_Flags;
   typedef struct _Elm_Genlist_Item_Class Elm_Genlist_Item_Class;
   typedef struct _Elm_Genlist_Item       Elm_Genlist_Item; /**< Item of Elm_Genlist. Sub-type of Elm_Widget_Item */
   typedef struct _Elm_Genlist_Item_Class_Func Elm_Genlist_Item_Class_Func;
   typedef char        *(*GenlistItemLabelGetFunc) (void *data, Evas_Object *obj, const char *part);
   typedef Evas_Object *(*GenlistItemIconGetFunc)  (void *data, Evas_Object *obj, const char *part);
   typedef Eina_Bool    (*GenlistItemStateGetFunc) (void *data, Evas_Object *obj, const char *part);
   typedef void         (*GenlistItemDelFunc)      (void *data, Evas_Object *obj);

   struct _Elm_Genlist_Item_Class
     {
	const char *item_style;
	struct {
          GenlistItemLabelGetFunc label_get;
          GenlistItemIconGetFunc icon_get;
          GenlistItemStateGetFunc state_get;
          GenlistItemDelFunc del;
        } func;
     };

   EAPI Evas_Object      *elm_genlist_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_multi_select_set(Evas_Object *obj, Eina_Bool multi) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_multi_select_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_horizontal_mode_set(Evas_Object *obj, Elm_List_Mode mode) EINA_ARG_NONNULL(1);
   EAPI Elm_List_Mode     elm_genlist_horizontal_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_always_select_mode_set(Evas_Object *obj, Eina_Bool always_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_always_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_no_select_mode_set(Evas_Object *obj, Eina_Bool no_select) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_no_select_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_compress_mode_set(Evas_Object *obj, Eina_Bool compress) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_compress_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_height_for_width_mode_set(Evas_Object *obj, Eina_Bool height_for_width) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_height_for_width_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_homogeneous_set(Evas_Object *obj, Eina_Bool homogeneous) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_homogeneous_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_block_count_set(Evas_Object *obj, int n) EINA_ARG_NONNULL(1);
   EAPI int               elm_genlist_block_count_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_longpress_timeout_set(Evas_Object *obj, double timeout) EINA_ARG_NONNULL(1);
   EAPI double            elm_genlist_longpress_timeout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* operations to add items */
   EAPI Elm_Genlist_Item *elm_genlist_item_append(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Genlist_Item *parent, Elm_Genlist_Item_Flags flags, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_item_prepend(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Genlist_Item *parent, Elm_Genlist_Item_Flags flags, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_item_insert_before(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Genlist_Item *before, Elm_Genlist_Item_Flags flags, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1, 4);
   EAPI Elm_Genlist_Item *elm_genlist_item_insert_after(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Genlist_Item *after, Elm_Genlist_Item_Flags flags, Evas_Smart_Cb func, const void *func_data) EINA_ARG_NONNULL(1, 4);
   /* operations to retrieve existing items */
   EAPI Elm_Genlist_Item *elm_genlist_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List  *elm_genlist_selected_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_List        *elm_genlist_realized_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_at_xy_item_get(const Evas_Object *obj, Evas_Coord x, Evas_Coord y, int *posret) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_first_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_last_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_scroller_policy_set(Evas_Object *obj, Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_scroller_policy_get(const Evas_Object *obj, Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v) EINA_ARG_NONNULL(1);
   /* available item styles:
    * default
    * default_style - The text part is a textblock
    * double_label
    * icon_top_text_bottom
    */
   /* Genlist Item operation */
   EAPI Elm_Genlist_Item *elm_genlist_item_next_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_item_prev_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI Evas_Object      *elm_genlist_item_genlist_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Genlist_Item *elm_genlist_item_parent_get(const Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_subitems_clear(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_selected_set(Elm_Genlist_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_item_selected_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_expanded_set(Elm_Genlist_Item *item, Eina_Bool expanded) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_item_expanded_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI int               elm_genlist_item_expanded_depth_get(const Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_disabled_set(Elm_Genlist_Item *item, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_item_disabled_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_display_only_set(Elm_Genlist_Item *it, Eina_Bool display_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_item_display_only_get(const Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_show(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_bring_in(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_top_show(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_top_bring_in(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_middle_show(Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_middle_bring_in(Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_del(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void             *elm_genlist_item_data_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_data_set(Elm_Genlist_Item *it, const void *data) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_icons_orphan(Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI const Evas_Object *elm_genlist_item_object_get(const Elm_Genlist_Item *it) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_update(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_item_class_update(Elm_Genlist_Item *it, const Elm_Genlist_Item_Class *itc) EINA_ARG_NONNULL(1, 2);
   EAPI void              elm_genlist_item_tooltip_text_set(Elm_Genlist_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_tooltip_content_cb_set(Elm_Genlist_Item *item, Elm_Tooltip_Item_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_tooltip_unset(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_tooltip_style_set(Elm_Genlist_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_genlist_item_tooltip_style_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_cursor_set(Elm_Genlist_Item *item, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_genlist_item_cursor_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_cursor_unset(Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_cursor_style_set(Elm_Genlist_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char       *elm_genlist_item_cursor_style_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);
   EAPI void              elm_genlist_item_cursor_engine_only_set(Elm_Genlist_Item *item, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool         elm_genlist_item_cursor_engine_only_get(const Elm_Genlist_Item *item) EINA_ARG_NONNULL(1);

   /* check */
   EAPI Evas_Object *elm_check_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_check_label_set(Evas_Object *obj, const char *label); EINA_ARG_NONNULL(1)
   EAPI const char  *elm_check_label_get(const Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI void         elm_check_icon_set(Evas_Object *obj, Evas_Object *icon); EINA_ARG_NONNULL(1)
   EAPI Evas_Object *elm_check_icon_get(const Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI Evas_Object *elm_check_icon_unset(Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI void         elm_check_state_set(Evas_Object *obj, Eina_Bool state); EINA_ARG_NONNULL(1)
   EAPI Eina_Bool    elm_check_state_get(const Evas_Object *obj); EINA_ARG_NONNULL(1)
   EAPI void         elm_check_state_pointer_set(Evas_Object *obj, Eina_Bool *statep); EINA_ARG_NONNULL(1)

   /* radio */
   EAPI Evas_Object *elm_radio_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_radio_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_radio_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_radio_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_group_add(Evas_Object *obj, Evas_Object *group) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_state_value_set(Evas_Object *obj, int value) EINA_ARG_NONNULL(1);
   EAPI int          elm_radio_state_value_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_value_set(Evas_Object *obj, int value) EINA_ARG_NONNULL(1);
   EAPI int          elm_radio_value_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_radio_value_pointer_set(Evas_Object *obj, int *valuep) EINA_ARG_NONNULL(1);

   /* pager */
   EAPI Evas_Object *elm_pager_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_pager_content_push(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI void         elm_pager_content_pop(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_pager_content_promote(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_pager_content_bottom_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_pager_content_top_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* available item styles:
    * default
    * fade
    * fade_translucide
    * fade_invisible
    */

   typedef struct _Elm_Slideshow_Item_Class Elm_Slideshow_Item_Class;
   typedef struct _Elm_Slideshow_Item_Class_Func Elm_Slideshow_Item_Class_Func;
   typedef struct _Elm_Slideshow_Item       Elm_Slideshow_Item; /**< Item of Elm_Slideshow. Sub-type of Elm_Widget_Item */
   typedef Evas_Object *(*SlideshowItemGetFunc) (void *data, Evas_Object *obj);
   typedef void         (*SlideshowItemDelFunc) (void *data, Evas_Object *obj);

   struct _Elm_Slideshow_Item_Class
     {
	struct _Elm_Slideshow_Item_Class_Func {
	     SlideshowItemGetFunc get;
	     SlideshowItemDelFunc del;
	} func;
     };

   EAPI Evas_Object        *elm_slideshow_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Elm_Slideshow_Item *elm_slideshow_item_add(Evas_Object *obj, const Elm_Slideshow_Item_Class *itc, const void *data) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_show(Elm_Slideshow_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_next(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_previous(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List    *elm_slideshow_transitions_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_transition_set(Evas_Object *obj, const char *transition) EINA_ARG_NONNULL(1);
   EAPI const char         *elm_slideshow_transition_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_timeout_set(Evas_Object *obj ,int timeout) EINA_ARG_NONNULL(1);
   EAPI int                 elm_slideshow_timeout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_loop_set(Evas_Object *obj, Eina_Bool loop) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool           elm_slideshow_loop_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List    *elm_slideshow_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_item_del(Elm_Slideshow_Item *item) EINA_ARG_NONNULL(1);
   EAPI void               *elm_slideshow_item_data_get(const Elm_Slideshow_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Slideshow_Item *elm_slideshow_item_current_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object*        elm_slideshow_item_object_get(const Elm_Slideshow_Item* item) EINA_ARG_NONNULL(1);
   EAPI const char         *elm_slideshow_layout_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_slideshow_layout_set(Evas_Object *obj, const char *layout) EINA_ARG_NONNULL(1);
   EAPI const Eina_List    *elm_slideshow_layouts_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* file selector */
   EAPI Evas_Object *elm_fileselector_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_is_save_set(Evas_Object *obj, Eina_Bool is_save) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_is_save_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_folder_only_set(Evas_Object *obj, Eina_Bool only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_folder_only_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_buttons_ok_cancel_set(Evas_Object *obj, Eina_Bool buttons) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_buttons_ok_cancel_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_expandable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_expandable_set(Evas_Object *obj, Eina_Bool expand) EINA_ARG_NONNULL(1);
   EAPI void         elm_fileselector_path_set(Evas_Object *obj, const char *path) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_path_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_fileselector_selected_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_fileselector_selected_set(Evas_Object *obj, const char *path) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "selected" - the user click on a file
    * "directory,open" - the list is populate with a new content. event_info is a directory.
    * "done" - the user click on the ok or cancel buttons
    */

   /* progressbar */
   EAPI Evas_Object *elm_progressbar_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_pulse_set(Evas_Object *obj, Eina_Bool pulse) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_progressbar_pulse_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_pulse(Evas_Object *obj, Eina_Bool state) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_value_set(Evas_Object *obj, double val) EINA_ARG_NONNULL(1);
   EAPI double       elm_progressbar_value_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_label_set(Evas_Object *obj, const char *label) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_progressbar_label_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_progressbar_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_progressbar_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_span_size_set(Evas_Object *obj, Evas_Coord size) EINA_ARG_NONNULL(1);
   EAPI Evas_Coord   elm_progressbar_span_size_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_unit_format_set(Evas_Object *obj, const char *format) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_progressbar_unit_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_progressbar_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_progressbar_inverted_set(Evas_Object *obj, Eina_Bool inverted) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_progressbar_inverted_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    */
   /* available item styles:
    * default
    * wheel (simple style, no text, no progression, only pulse is available)
    */

   /* separator */
   EAPI Evas_Object *elm_separator_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_separator_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_separator_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    */

   /* spinner */
   EAPI Evas_Object *elm_spinner_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_label_format_set(Evas_Object *obj, const char *fmt) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_spinner_label_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_min_max_set(Evas_Object *obj, double min, double max) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_min_max_get(const Evas_Object *obj, double *min, double *max) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_step_set(Evas_Object *obj, double step) EINA_ARG_NONNULL(1);
   EAPI double       elm_spinner_step_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_value_set(Evas_Object *obj, double val) EINA_ARG_NONNULL(1);
   EAPI double       elm_spinner_value_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_spinner_wrap_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_editable_set(Evas_Object *obj, Eina_Bool editable) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_spinner_editable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_special_value_add(Evas_Object *obj, double value, const char *label) EINA_ARG_NONNULL(1);
   EAPI void         elm_spinner_interval_set(Evas_Object *obj, double interval) EINA_ARG_NONNULL(1);
   EAPI double       elm_spinner_interval_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "changed" - when the spinner value changes
    * "delay,changed" - when the spinner value changed, but a small time after a change (use this if you only want to respond to a change once the spinner is held still for a short while).
    */
   /* available item styles:
    * default
    * vertical (two up/down buttons at the right side and text left aligned)
    */

   /* index */
   typedef struct _Elm_Index_Item Elm_Index_Item; /**< Item of Elm_Index. Sub-type of Elm_Widget_Item */
  
   EAPI Evas_Object *elm_index_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_active_set(Evas_Object *obj, Eina_Bool active) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_level_set(Evas_Object *obj, int level) EINA_ARG_NONNULL(1);
   EAPI int          elm_index_item_level_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void        *elm_index_item_selected_get(const Evas_Object *obj, int level) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_append(Evas_Object *obj, const char *letter, const void *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_prepend(Evas_Object *obj, const char *letter, const void *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_append_relative(Evas_Object *obj, const char *letter, const void *item, const void *relative) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_prepend_relative(Evas_Object *obj, const char *letter, const void *item, const void *relative) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_sorted_insert(Evas_Object *obj, const char *letter, const void *item, Eina_Compare_Cb cmp_func, Eina_Compare_Cb cmp_data_func) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_del(Evas_Object *obj, const void *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Index_Item *elm_index_item_find(Evas_Object *obj, const void *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_go(Evas_Object *obj, int level) EINA_ARG_NONNULL(1);
   EAPI void        *elm_index_item_data_get(const Elm_Index_Item *item) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_data_set(Elm_Index_Item *it, const void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_index_item_del_cb_set(Elm_Index_Item *it, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_index_item_letter_get(const Elm_Index_Item *item) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "changed" - when the selected index item changes
    * "delay,changed" - when the selected index item changes, but after some small idle period
    * "selected" - when the user releases a finger and selects an item
    * "level,up" - when the user moves a finger from the first level to the second level
    * "level,down" - when the user moves a finger from the second level to the first level
    */

   /* photocam */
   typedef enum _Elm_Photocam_Zoom_Mode
     {
	ELM_PHOTOCAM_ZOOM_MODE_MANUAL = 0,
	ELM_PHOTOCAM_ZOOM_MODE_AUTO_FIT,
	ELM_PHOTOCAM_ZOOM_MODE_AUTO_FILL,
	ELM_PHOTOCAM_ZOOM_MODE_LAST
     } Elm_Photocam_Zoom_Mode;
  
   EAPI Evas_Object *elm_photocam_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Evas_Load_Error elm_photocam_file_set(Evas_Object *obj, const char *file) EINA_ARG_NONNULL(1);
   EAPI const char * elm_photocam_file_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_zoom_set(Evas_Object *obj, double zoom) EINA_ARG_NONNULL(1);
   EAPI double       elm_photocam_zoom_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_zoom_mode_set(Evas_Object *obj, Elm_Photocam_Zoom_Mode mode) EINA_ARG_NONNULL(1);
   EAPI Elm_Photocam_Zoom_Mode elm_photocam_zoom_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_image_size_get(const Evas_Object *obj, int *w, int *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_region_get(const Evas_Object *obj, int *x, int *y, int *w, int *h) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_image_region_show(Evas_Object *obj, int x, int y, int w, int h) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_image_region_bring_in(Evas_Object *obj, int x, int y, int w, int h) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_paused_set(Evas_Object *obj, Eina_Bool paused) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_photocam_paused_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_photocam_internal_image_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_bounce_set(Evas_Object *obj,  Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_photocam_bounce_get(const Evas_Object *obj,  Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "clicked" - when image clicked
    * "press" - when mouse/finger held down initially on image
    * "longpressed" - when mouse/finger held for long time on image
    * "clicked,double" - when mouse/finger double-clicked
    * "load" - when photo load begins
    * "loaded" - when photo load done
    * "load,details" - when detailed image load begins
    * "loaded,details" - when detailed image load done
    * "zoom,start" - when zooming started
    * "zoom,stop" - when zooming stopped
    * "zoom,change" - when auto zoom mode changed zoom level
    * "scroll - the content has been scrolled (moved)
    * "scroll,anim,start" - scrolling animation has started
    * "scroll,anim,stop" - scrolling animation has stopped
    * "scroll,drag,start" - dragging the contents around has started
    * "scroll,drag,stop" - dragging the contents around has stopped
    */

   /* map */
   typedef enum _Elm_Map_Zoom_Mode
     {
	ELM_MAP_ZOOM_MODE_MANUAL,
	ELM_MAP_ZOOM_MODE_AUTO_FIT,
	ELM_MAP_ZOOM_MODE_AUTO_FILL,
	ELM_MAP_ZOOM_MODE_LAST
     } Elm_Map_Zoom_Mode;

   typedef enum _Elm_Map_Sources
     {
	ELM_MAP_SOURCE_MAPNIK,
	ELM_MAP_SOURCE_OSMARENDER,
	ELM_MAP_SOURCE_CYCLEMAP,
	ELM_MAP_SOURCE_MAPLINT,
	ELM_MAP_SOURCE_CUSTOM_1,
	ELM_MAP_SOURCE_CUSTOM_2,
	ELM_MAP_SOURCE_CUSTOM_3,
	ELM_MAP_SOURCE_CUSTOM_4,
	ELM_MAP_SOURCE_CUSTOM_5,
	ELM_MAP_SOURCE_CUSTOM_6,
	ELM_MAP_SOURCE_CUSTOM_7,
	ELM_MAP_SOURCE_LAST
     } Elm_Map_Sources;

   typedef struct _Elm_Map_Marker       Elm_Map_Marker;
   typedef struct _Elm_Map_Marker_Class Elm_Map_Marker_Class;
   typedef struct _Elm_Map_Group_Class  Elm_Map_Group_Class;
   typedef Evas_Object *(*ElmMapMarkerGetFunc)      (Evas_Object *obj, Elm_Map_Marker *marker, void *data);
   typedef void         (*ElmMapMarkerDelFunc)      (Evas_Object *obj, Elm_Map_Marker *marker, void *data, Evas_Object *o);
   typedef Evas_Object *(*ElmMapMarkerIconGetFunc)  (Evas_Object *obj, Elm_Map_Marker *marker, void *data);
   typedef Evas_Object *(*ElmMapGroupIconGetFunc)   (Evas_Object *obj, void *data);
   typedef char        *(*ElmMapSourceURLFunc)      (int x, int y, int zoom);
  
   EAPI Evas_Object *elm_map_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_zoom_set(Evas_Object *obj, int zoom) EINA_ARG_NONNULL(1);
   EAPI double       elm_map_zoom_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_zoom_mode_set(Evas_Object *obj, Elm_Map_Zoom_Mode mode) EINA_ARG_NONNULL(1);
   EAPI Elm_Map_Zoom_Mode elm_map_zoom_mode_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_geo_region_get(const Evas_Object *obj, double *lon, double *lat) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_geo_region_bring_in(Evas_Object *obj, double lon, double lat) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_geo_region_show(Evas_Object *obj, double lon, double lat) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_paused_set(Evas_Object *obj, Eina_Bool paused) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_map_paused_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_paused_markers_set(Evas_Object *obj, Eina_Bool paused) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_map_paused_markers_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void	       elm_map_utils_convert_coord_into_geo(int x, int y, int size, double *lon, double *lat);
   EAPI void         elm_map_utils_convert_geo_into_coord(double lon, double lat, int size, int *x, int *y);
   EAPI Elm_Map_Marker *elm_map_marker_add(Evas_Object *obj, double lon, double lat, Elm_Map_Marker_Class *clas, Elm_Map_Group_Class *clas_group, void *data) EINA_ARG_NONNULL(1, 4, 5);
   EAPI void         elm_map_max_marker_per_group_set(Evas_Object *obj, int max) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_remove(Elm_Map_Marker *marker) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_bring_in(Elm_Map_Marker *marker) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_show(Elm_Map_Marker *marker) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_markers_list_show(Eina_List *markers) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_map_marker_object_get(const Elm_Map_Marker *marker) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_update(Elm_Map_Marker *marker) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_bubbles_close(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Map_Group_Class * elm_map_group_class_new(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_style_set(Elm_Map_Group_Class *clas, const char *style) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_icon_cb_set(Elm_Map_Group_Class *clas, ElmMapGroupIconGetFunc icon_get) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_data_set(Elm_Map_Group_Class *clas, void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_zoom_displayed_set(Elm_Map_Group_Class *clas, int zoom) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_zoom_grouped_set(Elm_Map_Group_Class *clas, int zoom) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_group_class_hide_set(Evas_Object *obj, Elm_Map_Group_Class *clas, Eina_Bool hide) EINA_ARG_NONNULL(1, 2);
   EAPI Elm_Map_Marker_Class * elm_map_marker_class_new(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_class_style_set(Elm_Map_Marker_Class *clas, const char *style) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_class_icon_cb_set(Elm_Map_Marker_Class *clas, ElmMapMarkerIconGetFunc icon_get) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_class_get_cb_set(Elm_Map_Marker_Class *clas, ElmMapMarkerGetFunc get) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_marker_class_del_cb_set(Elm_Map_Marker_Class *clas, ElmMapMarkerDelFunc del) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_source_set(Evas_Object *obj, Elm_Map_Sources source) EINA_ARG_NONNULL(1);
   EAPI Elm_Map_Sources elm_map_source_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_map_source_custom_api_set(Elm_Map_Sources source, const char *label, int zoom_min, int zoom_max, ElmMapSourceURLFunc url_cb) EINA_ARG_NONNULL(2, 5);
   EAPI int          elm_map_source_zoom_min_get(Elm_Map_Sources source);
   EAPI int          elm_map_source_zoom_max_get(Elm_Map_Sources source);
   EAPI const char  *elm_map_source_name_get(Elm_Map_Sources source);
   /* smart callbacks called:
    * "clicked" - when image clicked
    * "press" - when mouse/finger held down initially on image
    * "longpressed" - when mouse/finger held for long time on image
    * "clicked,double" - when mouse/finger double-clicked
    * "load,details" - when detailed image load begins
    * "loaded,details" - when detailed image load done
    * "zoom,start" - when zooming started
    * "zoom,stop" - when zooming stopped
    * "zoom,change" - when auto zoom mode changed zoom level
    * "scroll - the content has been scrolled (moved)
    * "scroll,anim,start" - scrolling animation has started
    * "scroll,anim,stop" - scrolling animation has stopped
    * "scroll,drag,start" - dragging the contents around has started
    * "scroll,drag,stop" - dragging the contents around has stopped
    */

   /* panel */
   typedef enum _Elm_Panel_Orient
     {
        ELM_PANEL_ORIENT_TOP,
        ELM_PANEL_ORIENT_BOTTOM,
        ELM_PANEL_ORIENT_LEFT,
        ELM_PANEL_ORIENT_RIGHT,
     } Elm_Panel_Orient;
  
   EAPI Evas_Object          *elm_panel_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panel_orient_set(Evas_Object *obj, Elm_Panel_Orient orient) EINA_ARG_NONNULL(1);
   EAPI Elm_Panel_Orient      elm_panel_orient_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panel_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panel_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panel_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panel_hidden_set(Evas_Object *obj, Eina_Bool hidden) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_panel_hidden_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panel_toggle(Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* panes */
   /**
    * TODO
    *
    * Update the minimun height of the bar in the theme. No minimun should be set in the vertical theme
    * Add events (move, start ...)
    */
   EAPI Evas_Object          *elm_panes_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panes_content_left_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panes_content_right_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panes_content_left_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panes_content_right_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panes_content_left_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_panes_content_right_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI double                elm_panes_content_left_size_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panes_content_left_size_set(Evas_Object *obj, double size) EINA_ARG_NONNULL(1);
   EAPI void                  elm_panes_horizontal_set(Evas_Object *obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_panes_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* flip */
   typedef enum _Elm_Flip_Mode
     {
        ELM_FLIP_ROTATE_Y_CENTER_AXIS,
        ELM_FLIP_ROTATE_X_CENTER_AXIS,
        ELM_FLIP_ROTATE_XZ_CENTER_AXIS,
        ELM_FLIP_ROTATE_YZ_CENTER_AXIS,
        ELM_FLIP_CUBE_LEFT,
        ELM_FLIP_CUBE_RIGHT,
        ELM_FLIP_CUBE_UP,
        ELM_FLIP_CUBE_DOWN
     } Elm_Flip_Mode;

   EAPI Evas_Object *elm_flip_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_flip_content_front_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI void         elm_flip_content_back_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_flip_content_front_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_flip_content_back_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_flip_content_front_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_flip_content_back_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_flip_front_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_flip_perspective_set(Evas_Object *obj, Evas_Coord foc, Evas_Coord x, Evas_Coord y) EINA_ARG_NONNULL(1);
   EAPI void         elm_flip_go(Evas_Object *obj, Elm_Flip_Mode mode) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "animate,done" - when a flip animation is finished
    */
  
   /* scrolledentry */
   EAPI Evas_Object *elm_scrolled_entry_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_single_line_set(Evas_Object *obj, Eina_Bool single_line) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_single_line_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_password_set(Evas_Object *obj, Eina_Bool password) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_password_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_entry_set(Evas_Object *obj, const char *entry) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_scrolled_entry_entry_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_scrolled_entry_selection_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_entry_insert(Evas_Object *obj, const char *entry) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_line_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_line_char_wrap_set(Evas_Object *obj, Eina_Bool wrap) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_editable_set(Evas_Object *obj, Eina_Bool editable) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_editable_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_select_none(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_select_all(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_next(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_prev(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_up(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_down(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_begin_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_end_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_line_begin_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_line_end_set(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_selection_begin(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_cursor_selection_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_is_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_cursor_is_visible_format_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const char  *elm_scrolled_entry_cursor_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_selection_cut(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_selection_copy(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_selection_paste(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_context_menu_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_context_menu_item_add(Evas_Object *obj, const char *label, const char *icon_file, Elm_Icon_Type icon_type, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_context_menu_disabled_set(Evas_Object *obj, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_context_menu_disabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_scrollbar_policy_set(Evas_Object *obj, Elm_Scroller_Policy h, Elm_Scroller_Policy v) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_icon_set(Evas_Object *obj, Evas_Object *icon) EINA_ARG_NONNULL(1, 2);
   EAPI Evas_Object *elm_scrolled_entry_icon_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_scrolled_entry_icon_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_icon_visible_set(Evas_Object *obj, Eina_Bool setting) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_end_set(Evas_Object *obj, Evas_Object *end) EINA_ARG_NONNULL(1, 2);
   EAPI Evas_Object *elm_scrolled_entry_end_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_scrolled_entry_end_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_end_visible_set(Evas_Object *obj, Eina_Bool setting) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_item_provider_append(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_item_provider_prepend(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_item_provider_remove(Evas_Object *obj, Evas_Object *(*func) (void *data, Evas_Object *entry, const char *item), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_text_filter_append(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_text_filter_prepend(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_text_filter_remove(Evas_Object *obj, void (*func) (void *data, Evas_Object *entry, char **text), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_scrolled_entry_file_set(Evas_Object *obj, const char *file, Elm_Text_Format format) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_file_get(const Evas_Object *obj, const char **file, Elm_Text_Format *format) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_file_save(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_scrolled_entry_autosave_set(Evas_Object *obj, Eina_Bool autosave) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_scrolled_entry_autosave_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* conformant */
   EAPI Evas_Object *elm_conformant_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_conformant_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_conformant_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_conformant_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* mapbuf */
   EAPI Evas_Object *elm_mapbuf_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void         elm_mapbuf_content_set(Evas_Object *obj, Evas_Object *content) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_mapbuf_content_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object *elm_mapbuf_content_unset(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_mapbuf_enabled_set(Evas_Object *obj, Eina_Bool enabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_mapbuf_enabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_mapbuf_smooth_set(Evas_Object *obj, Eina_Bool smooth) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_mapbuf_smooth_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_mapbuf_alpha_set(Evas_Object *obj, Eina_Bool alpha) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_mapbuf_alpha_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);

   /* flipselector */
   typedef struct _Elm_Flipselector_Item Elm_Flipselector_Item;

   EAPI Evas_Object               *elm_flipselector_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_flip_next(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_flip_prev(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_item_append(Evas_Object *obj, const char *label, void (*func)(void *data, Evas_Object *obj, void *event_info), void *data) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_item_prepend(Evas_Object *obj, const char *label, void (*func)(void *data, Evas_Object *obj, void *event_info), void *data) EINA_ARG_NONNULL(1);
   EAPI const Eina_List           *elm_flipselector_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_first_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_last_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_item_selected_set(Elm_Flipselector_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool                  elm_flipselector_item_selected_get(const Elm_Flipselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_item_del(Elm_Flipselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI const char                *elm_flipselector_item_label_get(const Elm_Flipselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_item_label_set(Elm_Flipselector_Item *item, const char *label) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_item_prev_get(Elm_Flipselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Flipselector_Item       *elm_flipselector_item_next_get(Elm_Flipselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                       elm_flipselector_interval_set(Evas_Object *obj, double interval) EINA_ARG_NONNULL(1);
   EAPI double                     elm_flipselector_interval_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "selected" - when flipselector selected item is changed
    * "overflowed" - when flipselector item is changed to first item
    *                from last item
    * "underflowed" - when flipselector item is changed to last item
    *                 from first item.
    */
   /* available styles:
    * default
    */

   /* animator */
   typedef enum 
     {
        ELM_ANIMATOR_CURVE_LINEAR,
        ELM_ANIMATOR_CURVE_IN_OUT,
        ELM_ANIMATOR_CURVE_IN,
        ELM_ANIMATOR_CURVE_OUT
     } Elm_Animator_Curve_Style;
   typedef struct _Elm_Animator Elm_Animator;
  /**
   * Called back per loop of an elementary animators cycle
   * @param data user-data given to elm_animator_operation_callback_set()
   * @param animator the animator being run
   * @param double the position in the animation
   */
   typedef void (*Elm_Animator_Operation_Cb) (void *data, Elm_Animator *animator, double frame);
  /**
   * Called back when an elementary animator finishes
   * @param data user-data given to elm_animator_completion_callback_set()
   */
   typedef void (*Elm_Animator_Completion_Cb) (void *data);

   EAPI Elm_Animator* elm_animator_add(Evas_Object *parent);
   EAPI void          elm_animator_del(Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_duration_set(Elm_Animator *animator, double duration) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_operation_callback_set(Elm_Animator *animator, Elm_Animator_Operation_Cb func, void *data) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_completion_callback_set(Elm_Animator *animator, Elm_Animator_Completion_Cb func, void *data) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_stop(Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_repeat_set(Elm_Animator *animator, unsigned int repeat_cnt) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_animate(Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_curve_style_set(Elm_Animator *animator, Elm_Animator_Curve_Style cs) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_auto_reverse_set(Elm_Animator *animator, Eina_Bool reverse) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool     elm_animator_auto_reverse_get(const Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool     elm_animator_operating_get(const Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI unsigned int  elm_animator_repeat_get(const Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_pause(Elm_Animator *animator) EINA_ARG_NONNULL(1);
   EAPI void          elm_animator_resume(Elm_Animator *animator) EINA_ARG_NONNULL(1);

   /* calendar */
   typedef enum 
     {
        ELM_CALENDAR_UNIQUE, 
        ELM_CALENDAR_DAILY,
        ELM_CALENDAR_WEEKLY,
        ELM_CALENDAR_MONTHLY, 
        ELM_CALENDAR_ANNUALLY
     } Elm_Calendar_Mark_Repeat;
   typedef struct _Elm_Calendar_Mark Elm_Calendar_Mark;

   EAPI Evas_Object *elm_calendar_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI const char **elm_calendar_weekdays_names_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_weekdays_names_set(Evas_Object *obj, const char *weekdays[]) EINA_ARG_NONNULL(1, 2);
   EAPI double       elm_calendar_interval_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_interval_set(Evas_Object *obj, double interval) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_min_max_year_get(const Evas_Object *obj, int *min, int *max) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_min_max_year_set(Evas_Object *obj, int min, int max) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_calendar_day_selection_enabled_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_day_selection_enabled_set(Evas_Object *obj, Eina_Bool enabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool    elm_calendar_selected_time_get(const Evas_Object *obj, struct tm *selected_time) EINA_ARG_NONNULL(1, 2);
   EAPI void         elm_calendar_selected_time_set(Evas_Object *obj, struct tm *selected_time) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_format_function_set(Evas_Object *obj, char * (*format_function) (struct tm *stime)) EINA_ARG_NONNULL(1);
   EAPI Elm_Calendar_Mark *elm_calendar_mark_add(Evas_Object *obj, const char *mark_type, struct tm *mark_time, Elm_Calendar_Mark_Repeat repeat) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_mark_del(Elm_Calendar_Mark *mark) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_marks_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List *elm_calendar_marks_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_marks_draw(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_text_saturday_color_set(Evas_Object *obj, int pos) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_text_sunday_color_set(Evas_Object *obj, int pos) EINA_ARG_NONNULL(1);
   EAPI void         elm_calendar_text_weekday_color_set(Evas_Object *obj, int pos) EINA_ARG_NONNULL(1);
   
   /* smart callbacks called:
    * changed - emitted when the user select a day or change the displayed
    * month.
    */

   /* diskselector */
   typedef struct _Elm_Diskselector_Item Elm_Diskselector_Item;

   EAPI Evas_Object          *elm_diskselector_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_diskselector_round_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_round_set(Evas_Object *obj, Eina_Bool round) EINA_ARG_NONNULL(1);
   EAPI int                   elm_diskselector_side_label_lenght_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_side_label_lenght_set(Evas_Object *obj, int len) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_scroller_policy_get(const Evas_Object *obj, Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_scroller_policy_set(Evas_Object *obj, Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_clear(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI const Eina_List      *elm_diskselector_items_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_item_append(Evas_Object *obj, const char *label, Evas_Object *icon, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void	              elm_diskselector_item_del(Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_del_cb_set(Elm_Diskselector_Item *item, Evas_Smart_Cb func) EINA_ARG_NONNULL(1);
   EAPI void                 *elm_diskselector_item_data_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI Evas_Object          *elm_diskselector_item_icon_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_icon_set(Elm_Diskselector_Item *item, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI const char           *elm_diskselector_item_label_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_label_set(Elm_Diskselector_Item *item, const char *label) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_selected_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_selected_set(Elm_Diskselector_Item *item, Eina_Bool selected) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_diskselector_item_selected_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_first_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_last_item_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_item_prev_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI Elm_Diskselector_Item  *elm_diskselector_item_next_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_tooltip_text_set(Elm_Diskselector_Item *item, const char *text) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_tooltip_content_cb_set(Elm_Diskselector_Item *item, Elm_Tooltip_Item_Content_Cb func, const void *data, Evas_Smart_Cb del_cb) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_tooltip_unset(Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_tooltip_style_set(Elm_Diskselector_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char*           elm_diskselector_item_tooltip_style_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_cursor_set(Elm_Diskselector_Item *item, const char *cursor) EINA_ARG_NONNULL(1);
   EAPI const char*           elm_diskselector_item_cursor_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_cursor_unset(Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_cursor_style_set(Elm_Diskselector_Item *item, const char *style) EINA_ARG_NONNULL(1);
   EAPI const char*           elm_diskselector_item_cursor_style_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                  elm_diskselector_item_cursor_engine_only_set(Elm_Diskselector_Item *item, Eina_Bool engine_only) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool             elm_diskselector_item_cursor_engine_only_get(const Elm_Diskselector_Item *item) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "selected" - when item is selected (scroller stops)
    */

   /* colorselector */
   EAPI Evas_Object *elm_colorselector_add(Evas_Object *parent) EINA_ARG_NONNULL(1);
   EAPI void elm_colorselector_color_set(Evas_Object *obj, int r, int g , int b, int a) EINA_ARG_NONNULL(1);
   EAPI void elm_colorselector_color_get(const Evas_Object *obj, int *r, int *g , int *b, int *a) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "changed" - when the color value changes
    */

   /* ctxpopup */
   typedef struct _Elm_Ctxpopup_Item Elm_Ctxpopup_Item;
  
   EAPI Evas_Object        *elm_ctxpopup_add(Evas_Object* parent) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_hover_end(Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Evas_Object        *elm_ctxpopup_item_icon_get(const Elm_Ctxpopup_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_item_icon_set(Elm_Ctxpopup_Item *item, Evas_Object *icon) EINA_ARG_NONNULL(1);
   EAPI const char         *elm_ctxpopup_item_label_get(const Elm_Ctxpopup_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_item_label_set(Elm_Ctxpopup_Item *item, const char *label) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_hover_parent_set(Evas_Object *obj, Evas_Object *parent) EINA_ARG_NONNULL(1, 2);
   EAPI Evas_Object        *elm_ctxpopup_hover_parent_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_clear(Evas_Object* obj) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_horizontal_set(Evas_Object* obj, Eina_Bool horizontal) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool           elm_ctxpopup_horizontal_get(const Evas_Object *obj) EINA_ARG_NONNULL(1);
   EAPI Elm_Ctxpopup_Item  *elm_ctxpopup_item_append(Evas_Object *obj, const char *label, Evas_Object *icon, Evas_Smart_Cb func, const void *data) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_item_del(Elm_Ctxpopup_Item *item) EINA_ARG_NONNULL(1);
   EAPI void                elm_ctxpopup_item_disabled_set(Elm_Ctxpopup_Item *item, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool           elm_ctxpopup_item_disabled_get(const Elm_Ctxpopup_Item *item) EINA_ARG_NONNULL(1);
   /* smart callbacks called:
    * "dismissed" - the ctxpopup was dismissed
    */

   /* transit */
   typedef enum
     {
        ELM_TRANSIT_TWEEN_MODE_LINEAR,
        ELM_TRANSIT_TWEEN_MODE_SINUSOIDAL,
        ELM_TRANSIT_TWEEN_MODE_DECELERATE,
        ELM_TRANSIT_TWEEN_MODE_ACCELERATE
     } Elm_Transit_Tween_Mode;
   typedef enum
     {
        ELM_TRANSIT_EFFECT_FLIP_AXIS_X,
        ELM_TRANSIT_EFFECT_FLIP_AXIS_Y
     } Elm_Fx_Flip_Axis;
   typedef enum
     {
	ELM_TRANSIT_EFFECT_WIPE_DIR_LEFT,
	ELM_TRANSIT_EFFECT_WIPE_DIR_RIGHT,
	ELM_TRANSIT_EFFECT_WIPE_DIR_UP,
	ELM_TRANSIT_EFFECT_WIPE_DIR_DOWN
     } Elm_Fx_Wipe_Dir;
   typedef enum
     {
        ELM_TRANSIT_EFFECT_WIPE_TYPE_HIDE,
        ELM_TRANSIT_EFFECT_WIPE_TYPE_SHOW
     } Elm_Fx_Wipe_Type;
  
   typedef struct _Elm_Transit Elm_Transit;

   EAPI Elm_Transit           *elm_transit_add(double duration);
   EAPI void                   elm_transit_del(Elm_Transit *transit) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_effect_add(Elm_Transit *transit, void (*cb)(void *data, Elm_Transit *transit, double progress), void *data, void (*data_free_cb)(void *data, Elm_Transit *transit)) EINA_ARG_NONNULL(1, 2);
   EAPI void                   elm_transit_effect_del(Elm_Transit *transit, void (*cb)(void *data, Elm_Transit *transit, double progress), void *data) EINA_ARG_NONNULL(1, 2);
   EAPI void                   elm_transit_object_add(Elm_Transit *transit, Evas_Object *obj) EINA_ARG_NONNULL(1, 2);
   EAPI void                   elm_transit_object_remove(Elm_Transit *transit, Evas_Object *obj) EINA_ARG_NONNULL(1, 2);
   EAPI const Eina_List       *elm_transit_objects_get(const Elm_Transit *transit) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_event_block_set(Elm_Transit *transit, Eina_Bool disabled) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool              elm_transit_event_block_get(const Elm_Transit *transit) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_del_cb_set(Elm_Transit *transit, void (*cb)(void *data, Elm_Transit* transit), void *data) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_auto_reverse_set(Elm_Transit *transit, Eina_Bool reverse) EINA_ARG_NONNULL(1);
   EAPI Eina_Bool              elm_transit_auto_reverse_get(Elm_Transit *transit) EINA_ARG_NONNULL(1) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_repeat_times_set(Elm_Transit *transit, int repeat) EINA_ARG_NONNULL(1);
   EAPI int                    elm_transit_repeat_times_get(Elm_Transit *transit) EINA_ARG_NONNULL(1);
   EAPI void                   elm_transit_tween_mode_set(Elm_Transit *transit, Elm_Transit_Tween_Mode tween_mode) EINA_ARG_NONNULL(1);
   EAPI Elm_Transit_Tween_Mode elm_transit_tween_mode_get(Elm_Transit *transit) EINA_ARG_NONNULL(1) EINA_ARG_NONNULL(1);

   EAPI void           *elm_transit_effect_resizing_context_new(Evas_Coord from_w, Evas_Coord from_h, Evas_Coord to_w, Evas_Coord to_h);
   EAPI void            elm_transit_effect_resizing_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_resizing_context_free(void *data, Elm_Transit *transit);

   EAPI void           *elm_transit_effect_translation_context_new(Evas_Coord from_dx, Evas_Coord from_dy, Evas_Coord to_dx, Evas_Coord to_dy);
   EAPI void            elm_transit_effect_translation_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_translation_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1);

   EAPI void           *elm_transit_effect_zoom_context_new(float from_rate, float to_rate);
   EAPI void            elm_transit_effect_zoom_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_zoom_context_free(void *data, Elm_Transit *transit);

   EAPI void           *elm_transit_effect_flip_context_new(Elm_Fx_Flip_Axis axis, Eina_Bool cw);
   EAPI void            elm_transit_effect_flip_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_flip_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1, 2);

   EAPI void           *elm_transit_effect_resizable_flip_context_new(Elm_Fx_Flip_Axis axis, Eina_Bool cw);
   EAPI void            elm_transit_effect_resizable_flip_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1);
   EAPI void            elm_transit_effect_resizable_flip_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1);

   EAPI void           *elm_transit_effect_wipe_context_new(Elm_Fx_Wipe_Type type, Elm_Fx_Wipe_Dir dir);
   EAPI void            elm_transit_effect_wipe_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_wipe_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1, 2);

   EAPI void           *elm_transit_effect_color_context_new(unsigned int from_r, unsigned int from_g, unsigned int from_b, unsigned int from_a, unsigned int to_r, unsigned int to_g, unsigned int to_b, unsigned int to_a);
   EAPI void            elm_transit_effect_color_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_color_context_free(void *data, Elm_Transit *transit);

   EAPI void           *elm_transit_effect_fade_context_new(void);
   EAPI void            elm_transit_effect_fade_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1);
   EAPI void            elm_transit_effect_fade_context_free(void *data, Elm_Transit *tranasit) EINA_ARG_NONNULL(1);

   EAPI void           *elm_transit_effect_blend_context_new(void);
   EAPI void            elm_transit_effect_blend_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_blend_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1);

   EAPI void           *elm_transit_effect_rotation_context_new(float from_degree, float to_degree, Eina_Bool cw);
   EAPI void            elm_transit_effect_rotation_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_rotation_context_free(void *data, Elm_Transit *transit);

   EAPI void           *elm_transit_effect_image_animation_context_new(Eina_List *images);
   EAPI void            elm_transit_effect_image_animation_op(void *data, Elm_Transit *transit, double progress) EINA_ARG_NONNULL(1, 2);
   EAPI void            elm_transit_effect_image_animation_context_free(void *data, Elm_Transit *transit) EINA_ARG_NONNULL(1);

  typedef struct _Elm_Store                      Elm_Store;
  typedef struct _Elm_Store_Filesystem           Elm_Store_Filesystem;
  typedef struct _Elm_Store_Item                 Elm_Store_Item;
  typedef struct _Elm_Store_Item_Filesystem      Elm_Store_Item_Filesystem;
  typedef struct _Elm_Store_Item_Info            Elm_Store_Item_Info;
  typedef struct _Elm_Store_Item_Info_Filesystem Elm_Store_Item_Info_Filesystem;
  typedef struct _Elm_Store_Item_Mapping         Elm_Store_Item_Mapping;
  typedef struct _Elm_Store_Item_Mapping_Empty   Elm_Store_Item_Mapping_Empty;
  typedef struct _Elm_Store_Item_Mapping_Icon    Elm_Store_Item_Mapping_Icon;
  typedef struct _Elm_Store_Item_Mapping_Photo   Elm_Store_Item_Mapping_Photo;
  typedef struct _Elm_Store_Item_Mapping_Custom  Elm_Store_Item_Mapping_Custom;

  typedef Eina_Bool (*Elm_Store_Item_List_Cb) (void *data, Elm_Store_Item_Info *info);
  typedef void      (*Elm_Store_Item_Fetch_Cb) (void *data, Elm_Store_Item *sti);
  typedef void      (*Elm_Store_Item_Unfetch_Cb) (void *data, Elm_Store_Item *sti);
  typedef void     *(*Elm_Store_Item_Mapping_Cb) (void *data, Elm_Store_Item *sti, const char *part);
  
  typedef enum
    {
      ELM_STORE_ITEM_MAPPING_NONE = 0,
      
      ELM_STORE_ITEM_MAPPING_LABEL, // const char * -> label
      ELM_STORE_ITEM_MAPPING_STATE, // Eina_Bool -> state
      ELM_STORE_ITEM_MAPPING_ICON, // char * -> icon path
      ELM_STORE_ITEM_MAPPING_PHOTO, // char * -> photo path
      ELM_STORE_ITEM_MAPPING_CUSTOM, // item->custom(it->data, it, part) -> void * (-> any)
      // can add more here as needed by common apps
      ELM_STORE_ITEM_MAPPING_LAST
    } Elm_Store_Item_Mapping_Type;
  
  struct _Elm_Store_Item_Mapping_Icon
    {
      // FIXME: allow edje file icons
      int                   w, h;
      Elm_Icon_Lookup_Order lookup_order;
      Eina_Bool             standard_name : 1;
      Eina_Bool             no_scale : 1;
      Eina_Bool             smooth : 1;
      Eina_Bool             scale_up : 1;
      Eina_Bool             scale_down : 1;
    };
  
  struct _Elm_Store_Item_Mapping_Empty
    {
      Eina_Bool             dummy;
    };
  
  struct _Elm_Store_Item_Mapping_Photo
    {
      int                   size;
    };
  
  struct _Elm_Store_Item_Mapping_Custom
    {
      Elm_Store_Item_Mapping_Cb func;
    };
  
  struct _Elm_Store_Item_Mapping
    {
      Elm_Store_Item_Mapping_Type     type;
      const char                     *part;
      int                             offset;
      union {
        Elm_Store_Item_Mapping_Empty  empty;
        Elm_Store_Item_Mapping_Icon   icon;
        Elm_Store_Item_Mapping_Photo  photo;
        Elm_Store_Item_Mapping_Custom custom;
        // add more types here
      } details;
    };
  
  struct _Elm_Store_Item_Info
    {
      Elm_Genlist_Item_Class       *item_class;
      const Elm_Store_Item_Mapping *mapping;
      void                         *data;
      char                         *sort_id;
    };

  struct _Elm_Store_Item_Info_Filesystem
    {
      Elm_Store_Item_Info  base;
      char                *path;
    };
  
#define ELM_STORE_ITEM_MAPPING_END { ELM_STORE_ITEM_MAPPING_NONE, NULL, 0, { .empty = { EINA_TRUE } } }
#define ELM_STORE_ITEM_MAPPING_OFFSET(st, it) offsetof(st, it)
  
  EAPI void                    elm_store_free(Elm_Store *st);
  
  EAPI Elm_Store              *elm_store_filesystem_new(void);
  EAPI void                    elm_store_filesystem_directory_set(Elm_Store *st, const char *dir) EINA_ARG_NONNULL(1);
  EAPI const char             *elm_store_filesystem_directory_get(const Elm_Store *st) EINA_ARG_NONNULL(1);
  EAPI const char             *elm_store_item_filesystem_path_get(const Elm_Store_Item *sti) EINA_ARG_NONNULL(1);
  
  EAPI void                    elm_store_target_genlist_set(Elm_Store *st, Evas_Object *obj) EINA_ARG_NONNULL(1);
  
  EAPI void                    elm_store_cache_set(Elm_Store *st, int max) EINA_ARG_NONNULL(1);
  EAPI int                     elm_store_cache_get(const Elm_Store *st) EINA_ARG_NONNULL(1);
  EAPI void                    elm_store_list_func_set(Elm_Store *st, Elm_Store_Item_List_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
  EAPI void                    elm_store_fetch_func_set(Elm_Store *st, Elm_Store_Item_Fetch_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
  EAPI void                    elm_store_fetch_thread_set(Elm_Store *st, Eina_Bool use_thread) EINA_ARG_NONNULL(1);
  EAPI Eina_Bool               elm_store_fetch_thread_get(const Elm_Store *st) EINA_ARG_NONNULL(1);
  
  EAPI void                    elm_store_unfetch_func_set(Elm_Store *st, Elm_Store_Item_Unfetch_Cb func, const void *data) EINA_ARG_NONNULL(1, 2);
  EAPI void                    elm_store_sorted_set(Elm_Store *st, Eina_Bool sorted) EINA_ARG_NONNULL(1);
  EAPI Eina_Bool               elm_store_sorted_get(const Elm_Store *st) EINA_ARG_NONNULL(1);
  EAPI void                    elm_store_item_data_set(Elm_Store_Item *sti, void *data) EINA_ARG_NONNULL(1);
  EAPI void                   *elm_store_item_data_get(Elm_Store_Item *sti) EINA_ARG_NONNULL(1);
  EAPI const Elm_Store        *elm_store_item_store_get(const Elm_Store_Item *sti) EINA_ARG_NONNULL(1);
  EAPI const Elm_Genlist_Item *elm_store_item_genlist_item_get(const Elm_Store_Item *sti) EINA_ARG_NONNULL(1);
  
#ifdef __cplusplus
}
#endif

#endif
