/*
 * Eee Applet
 * Copyright (C) 2008 Olivier Rolland <billl@users.sourceforge.net>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "eee-monitor-procfs.h"
#include "eee-procfs.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define EEE_MONITOR_INTERVAL 1000

#define EEE_MONITOR_PROCFS_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE((obj), EEE_TYPE_MONITOR_PROCFS, EeeMonitorProcfsPriv))

struct _EeeMonitorProcfsPriv
{
  gint src;

  gboolean eee_online;
  gboolean asus_online;

  gint     fsb_speed;
  gboolean fan_manual;
  gint     fan_speed;
  gint     fan_rpm;
  gint     temperature;
  gint     brightness;
  gboolean camera;
  gboolean cardr;
  gboolean wlan;
};

/* Signals */
enum
{
  ONLINE,
  N_SIGNALS
};

static void     eee_monitor_procfs_finalize (GObject    *object);
static gboolean eee_monitor_procfs_run      (EeeMonitor *monitor);

static guint signals[N_SIGNALS];

G_DEFINE_TYPE (EeeMonitorProcfs, eee_monitor_procfs, EEE_TYPE_MONITOR)

static void
eee_monitor_procfs_init (EeeMonitorProcfs *monitor)
{
  monitor->priv = EEE_MONITOR_PROCFS_GET_PRIVATE (monitor);
}

static void
eee_monitor_procfs_class_init (EeeMonitorProcfsClass *klass)
{
  GObjectClass    *object_class;
  EeeMonitorClass *monitor_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = eee_monitor_procfs_finalize;

  monitor_class = EEE_MONITOR_CLASS (klass);
  monitor_class->run = eee_monitor_procfs_run;

  /* Signals */
  signals[ONLINE] =
    g_signal_new ("online",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        G_STRUCT_OFFSET (EeeMonitorProcfsClass, online),
        NULL, NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE, 0);

  g_type_class_add_private (klass, sizeof (EeeMonitorProcfsPriv));
}

static void
eee_monitor_procfs_finalize (GObject *object)
{
  EeeMonitorProcfs *monitor;

  monitor = EEE_MONITOR_PROCFS (object);

  if (monitor->priv->src)
  {
    g_source_remove (monitor->priv->src);
    monitor->priv->src = 0;
  }

  G_OBJECT_CLASS (eee_monitor_procfs_parent_class)->finalize (object);
}

static gboolean
eee_monitor_procfs_parse (EeeMonitorProcfs *monitor, const gchar *file, gint *val)
{
  GError *error = NULL;
  gchar *buffer = NULL;

  if (!g_file_get_contents (file, &buffer, NULL, &error))
  {
    g_warning (error->message);
    g_error_free (error);

    return FALSE;
  }

  *val = atoi (buffer);

  g_free (buffer);

  return TRUE;
}

static gboolean
eee_monitor_procfs_watch (EeeMonitor *monitor)
{
  gboolean eee_online, asus_online, changed = FALSE;

  EeeMonitorProcfs *procfs;
  gint val;

  procfs = EEE_MONITOR_PROCFS (monitor);

  eee_online = eee_monitor_procfs_is_online (EEE_PROCFS_EEE);
  asus_online = eee_monitor_procfs_is_online (EEE_PROCFS_ASUS);

  if (eee_online != procfs->priv->eee_online || asus_online != procfs->priv->asus_online)
  {
    procfs->priv->eee_online = eee_online;
    procfs->priv->asus_online = asus_online;
    g_signal_emit (procfs, signals[ONLINE], 0);
  }

  if (eee_online)
  {
    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_FSB, &val))
      return FALSE;
    if (val != procfs->priv->fsb_speed)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("fsb: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->fsb_speed = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_FAN_MANUAL, &val))
      return FALSE;
    if (val != procfs->priv->fan_manual)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("fan manual: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->fan_manual = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_FAN_SPEED, &val))
      return FALSE;
    if (val != procfs->priv->fan_speed)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("fan speed: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->fan_speed = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_FAN_RPM, &val))
      return FALSE;
/*
    if (val != procfs->priv->fan_rpm)
      changed = TRUE;
*/
    procfs->priv->fan_rpm = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_TEMPERATURE, &val))
      return FALSE;
    if (val != procfs->priv->temperature)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("temperature: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->temperature = val;
  }

  if (asus_online)
  {
    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_BRIGHTNESS, &val))
      return FALSE;
    if (val != procfs->priv->brightness)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("brightness: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->brightness = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_CAMERA, &val))
      return FALSE;
    if (val != procfs->priv->camera)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("camera: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->camera = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_CARDR, &val))
      return FALSE;
    if (val != procfs->priv->cardr)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("cardr: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->cardr = val;

    if (!eee_monitor_procfs_parse (EEE_MONITOR_PROCFS (monitor), EEE_PROC_WLAN, &val))
      return FALSE;
    if (val != procfs->priv->wlan)
    {
#ifdef G_ENABLE_DEBUG
      g_debug ("wlan: %d", val);
#endif
      changed = TRUE;
    }
    procfs->priv->wlan = val;
  }

  if (changed)
    g_signal_emit_by_name (monitor, "changed");

  return TRUE;
}

static gboolean
eee_monitor_procfs_run (EeeMonitor *monitor)
{
  EeeMonitorProcfs *procfs;

  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), FALSE);

  procfs = EEE_MONITOR_PROCFS (monitor);

  if (procfs->priv->src > 0)
    return TRUE;

  procfs->priv->src =
    g_timeout_add (EEE_MONITOR_INTERVAL,
        (GSourceFunc) eee_monitor_procfs_watch,
        (gpointer) monitor);

  return procfs->priv->src > 0;
}

EeeMonitor *
eee_monitor_procfs_new (void)
{
  return g_object_new (EEE_TYPE_MONITOR_PROCFS, NULL);
}

gboolean
eee_monitor_procfs_is_online (EeeProcfsType type)
{
  gchar *dir;

  if (type == EEE_PROCFS_ASUS)
    dir = "/proc/acpi/asus";
  else
    dir = "/proc/eee";

  return g_file_test (dir, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR);
}

gint
eee_monitor_procfs_get_fsb_speed (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), -1);

  return monitor->priv->fsb_speed;
}

gboolean
eee_monitor_procfs_get_fan_manual (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), FALSE);

  return monitor->priv->fan_manual;
}

gint
eee_monitor_procfs_get_fan_speed (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), -1);

  return monitor->priv->fan_speed;
}

gint
eee_monitor_procfs_get_fan_rpm (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), -1);

  return monitor->priv->fan_rpm;
}

gint
eee_monitor_procfs_get_temperature (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), -1);

  return monitor->priv->temperature;
}

gint
eee_monitor_procfs_get_brightness (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), -1);

  return monitor->priv->brightness;
}

gboolean
eee_monitor_procfs_get_camera (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), FALSE);

  return monitor->priv->camera;
}

gboolean
eee_monitor_procfs_get_card_reader (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), FALSE);

  return monitor->priv->cardr;
}

gboolean
eee_monitor_procfs_get_wlan (EeeMonitorProcfs *monitor)
{
  g_return_val_if_fail (EEE_IS_MONITOR_PROCFS (monitor), FALSE);

  return monitor->priv->wlan;
}

