"""This module defines a Form class that wraps FFC forms and UFC forms
into a cpp.Form (dolfin::Form)."""

__author__ = "Johan Hake (hake@simula.no)"
__date__ = "2008-12-04 -- 2009-12-11"
__copyright__ = "Copyright (C) 2008 Johan Hake"
__license__  = "GNU LGPL Version 2.1"

# Modified by Anders Logg, 2008

__all__ = ["Form"]

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Import JIT compiler
from dolfin.compilemodules.jit import jit

# Note that we need to store _compiled_form and _compiled_coefficients
# to prevent Python from garbage-collecting these while still in use.
# FIXME: Figure out how to solve this with shared_ptr

class Form(cpp.Form):

    def __init__(self, form,
                 function_spaces=None,
                 coefficients=None,
                 form_compiler_parameters=None,
                 common_cell=None):
        "Create JIT-compiled form from any given form (compiled or not)."

        # Compile form if necessary
        if not hasattr(form, "create_cell_integral"):
            (self._compiled_form, module, self.form_data) = jit(form, form_compiler_parameters, common_cell)

        else:
            self._compiled_form = form
            self.form_data = None

        # Extract test spaces and coefficients
        self.function_spaces = _extract_function_spaces(self.form_data,
                                                        self._compiled_form,
                                                        function_spaces)

        (self.coefficients, self._compiled_coefficients) \
                             = _extract_coefficients(self.form_data,
                                                     coefficients)

        # Initialize base class
        cpp.Form.__init__(self, self._compiled_form, self.function_spaces, self.coefficients)

def _extract_function_spaces(form_data, compiled_form, given_function_spaces):
    "Extract list of test spaces."

    function_space_error = "Error while extracting test and/or trial spaces. "

    function_spaces = []

    if given_function_spaces is None:
        if not hasattr(form_data,"original_arguments"):
            raise TypeError, function_space_error + \
                  "Missing data about basis functions in form data."
        for func in form_data.original_arguments:
            if not isinstance(func.function_space(), cpp.FunctionSpace):
                raise TypeError, function_space_error
            function_spaces.append(func.function_space())
    else:
        if not isinstance(given_function_spaces, (list, cpp.FunctionSpace)):
            raise TypeError, function_space_error
        if isinstance(given_function_spaces, list):
            if len(given_function_spaces) != compiled_form.rank():
                raise ValueError, function_space_error + \
                      " Wrong number of test spaces (should be %d)." % compiled_form.rank()
            for V in given_function_spaces:
                function_spaces.append(V)
        else:
            for i in xrange(compiled_form.rank()):
                function_spaces.append(given_function_spaces)

    return function_spaces

def _extract_coefficients(form_data, given_coefficients):
    "Extract list of coefficients."

    coefficient_error = "Error while extracting coefficients. "

    coefficients = []
    _compiled_coefficients = []

    # Return if nothing to extract
    if form_data is None and given_coefficients is None:
        return (coefficients,  _compiled_coefficients)

    if given_coefficients is None:
        if not hasattr(form_data, "original_coefficients"):
            raise TypeError, coefficient_error + \
                  "Missing data about coefficients in form data."
        for c in form_data.original_coefficients:
            if not isinstance(c, cpp.GenericFunction):
                raise TypeError, coefficient_error + \
                      "Either provide a dict of cpp.GenericFunctions, or use Function to define your form."
            coefficients.append(c)
    else:
        # FIXME: I have disabled compiled_functions based on strings for now
        #       We could ofcourse add it back, but they need a FunctionSpace to
        #       be initialized.
        ## Compile all strings as dolfin::Function
        #string_expressions = []
        #for c in coefficients:
        #    # Note: To allow tuples of floats or ints below, this logic becomes more involved...
        #    if isinstance(c, (tuple, str)):
        #        string_expressions.append(c)
        #if string_expressions:
        #    compiled_functions = compile_functions(string_expressions, mesh)
        #    compiled_functions.reverse()
        #
        # Build list of coefficients
        error_info = "Provide a 'list' with cpp.GenericFunctions"
        if not isinstance(given_coefficients, list):
            raise TypeError, coefficient_error + error_info
        for c in given_coefficients:
            # FIXME: I have turned of these for now. Should probably add something for
            #       at least constant functions
            # Note: We could generalize this to support more objects
            # like sympy expressions, tuples for constant vectors, etc...
            #if isinstance(c, (float, int)):
            #    c = cpp.Function(mesh, float(c))
            #elif isinstance(c, (tuple, str)):
            #    c = compiled_functions.pop()
            if not isinstance(c, cpp.GenericFunction):
                raise TypeError, coefficient_error
            coefficients.append(c)
            _compiled_coefficients.append(c)

    return (coefficients, _compiled_coefficients)
