__author__ = "Anders Logg (logg@simula.no)"
__date__ = "2009-02-12 -- 2011-01-24"
__copyright__ = "Copyright (C) 2009 Anders Logg"
__license__  = "GNU LGPL Version 2.1"

__all__ = ["compute_vertex_map", "compute_edge_map", "init_global_edge_indices"]

from dolfin.cpp import Mesh, Vertex, error, edges, info

def compute_vertex_map(mesh0, mesh1):
    """Compute map from vertices of mesh0 to vertices of mesh1. It is
    assumed that both meshes have a MeshFunction over the vertices
    named "global vertex indices" which contain a mapping from the
    local vertices to a common global vertex numbering."""

    # Check arguments
    if not isinstance(mesh0, Mesh):
        raise TypeError, "expected 'Mesh' as argument"
    if not isinstance(mesh1, Mesh):
        raise TypeError, "expected 'Mesh' as argument"

    # Get global vertex numbers
    vertices0 = mesh0.data().mesh_function("global vertex indices")
    vertices1 = mesh1.data().mesh_function("global vertex indices")

    # Check mappings
    if vertices0 is None or vertices1 is None:
        error("Unable to computed vertex map; global vertex indices are missing.")

    # Compute global-to-local mapping for mesh1
    global_to_local_mesh1 = {}
    for i in range(vertices1.size()):
        global_to_local_mesh1[vertices1[i]] = i

    # Compute local-to-local mapping
    vertex_map = {}
    for i in range(vertices0.size()):
        global_vertex = vertices0[i]
        if global_vertex in global_to_local_mesh1:
            vertex_map[i] = global_to_local_mesh1[global_vertex]

    return vertex_map

def compute_edge_map(mesh0, mesh1):
    """Compute map from edges of mesh0 to vertices of mesh1. It is
    assumed that both meshes have a MeshFunction over the vertices
    named "global vertex indices" which contain a mapping from the
    local vertices to a common global vertex numbering."""

    # Check arguments
    if not isinstance(mesh0, Mesh):
        raise TypeError, "expected 'Mesh' as argument"
    if not isinstance(mesh1, Mesh):
        raise TypeError, "expected 'Mesh' as argument"

    # Get global vertex numbers
    vertices0 = mesh0.data().mesh_function("global vertex indices")
    vertices1 = mesh1.data().mesh_function("global vertex indices")

    # Check mappings
    if vertices0 is None or vertices1 is None:
        error("Unable to computed vertex map; global vertex indices are missing.")

    # Initialize edges
    mesh0.init(1)
    mesh1.init(1)

    # Build global to local map from vertex pair to local edge for mesh0
    global_to_local_mesh0 = {}
    for edge in edges(mesh0):
        v = [vertices0[int(i)] for i in edge.entities(0)]
        v.sort()
        global_to_local_mesh0[tuple(v)] = edge.index()

    # Build global to local map from vertex pair to local edge for mesh1
    global_to_local_mesh1 = {}
    for edge in edges(mesh1):
        v = [vertices1[int(i)] for i in edge.entities(0)]
        v.sort()
        global_to_local_mesh1[tuple(v)] = edge.index()

    # Get common edges
    common_edges = set(global_to_local_mesh0.iterkeys()).intersection(set(global_to_local_mesh1.iterkeys()))

    # Compute map
    edge_map = {}
    for edge in common_edges:
        edge_map[global_to_local_mesh0[edge]] = global_to_local_mesh1[edge]

    return edge_map

def init_global_edge_indices(submesh, mesh):
    "Initialize data 'global_edge_indices' for submesh"

    # Check arguments
    if not isinstance(submesh, Mesh):
        raise TypeError, "expected 'Mesh' as argument"
    if not isinstance(mesh, Mesh):
        raise TypeError, "expected 'Mesh' as argument"

    # Check if edge map has already been computed
    if not submesh.data().mesh_function("global_edge_indices") is None:
        info("Edge map 'global_edge_indices' already computed, not computing again.")

    # Get global vertex numbers
    global_vertex_indices = submesh.data().mesh_function("global vertex indices")
    if global_vertex_indices is None:
        error("Unable to compute edge map; data 'global vertex indices' is missing.")

    # Make sure that we have edges for both meshes
    submesh.init(1)
    mesh.init(1)

    # Make sure we have vertex-edge connectivity for global mesh
    mesh.init(0, 1)

    # Create the edge map
    global_edge_indices = submesh.data().create_mesh_function("global edge indices")
    global_edge_indices.init(1)

    # Iterate over the edges and figure out their global number
    for local_edge in edges(submesh):

        # Get global indices for edge vertices
        v0, v1 = local_edge.entities(0)
        V0 = Vertex(mesh, global_vertex_indices[int(v0)])
        V1 = Vertex(mesh, global_vertex_indices[int(v1)])

        # Get outgoing edges from the two global vertices
        edges0 = set(V0.entities(1))
        edges1 = set(V1.entities(1))

        # Check intersection
        common_edges = edges0.intersection(edges1)
        if not len(common_edges) == 1:
            error("Unable to compute edge map; global vertices do not share exactly one common edge.")
        global_edge_index = list(common_edges)[0]

        # Set value
        global_edge_indices[local_edge.index()] = global_edge_index
