"This module provides a collection of small but useful utility functions."

__author__ = "Anders Logg (logg@simula.no)"
__date__ = "2008-10-22 -- 2009-12-11"
__copyright__ = "Copyright (C) 2009 Anders Logg"
__license__  = "GNU LGPL Version 2.1"

__all__ = ["AutoSubDomain", "DirichletBC", "PeriodicBC", "homogenize"]

import types

import dolfin.cpp as cpp
from dolfin.function.constant import Constant
from dolfin.compilemodules.subdomains import compile_subdomains

class AutoSubDomain(cpp.SubDomain):
    "Wrapper class for creating a SubDomain from an inside() function."

    def __init__(self, inside_function):
        "Create SubDomain subclass for given inside() function"

        # Check that we get a function
        if not isinstance(inside_function, types.FunctionType):
            cpp.error("Expecting a function, not %s.", str(type(inside_function)))
        self.inside_function = inside_function

        # Check the number of arguments
        if not inside_function.func_code.co_argcount in (1, 2):
            cpp.error("Expecting a function of the form inside(x) or inside(x, on_boundary).")
        self.num_args = inside_function.func_code.co_argcount

        cpp.SubDomain.__init__(self)

    def inside(self, x, on_boundary):
        "Return true for points inside the subdomain"

        if self.num_args == 1:
            return self.inside_function(x)
        else:
            return self.inside_function(x, on_boundary)

class DirichletBC(cpp.DirichletBC):

    def __init__(self, *args):
        "Create Dirichlet boundary condition."

        # Special case for value specified as float, tuple or similar
        if len(args) >= 2 and not isinstance(args[1], cpp.GenericFunction):
            constant = Constant(args[1]) # let Constant handle all problems
            args = args[:1] + (constant,) + args[2:]

        # Special case for sub domain specified as a function
        if len(args) >= 3 and isinstance(args[2], types.FunctionType):
            sub_domain = AutoSubDomain(args[2])
            args = args[:2] + (sub_domain,) + args[3:]

        # Special case for sub domain specified as a string
        if len(args) >= 3 and isinstance(args[2], str):
            sub_domain = compile_subdomains(args[2])
            args = args[:2] + (sub_domain,) + args[3:]

        # Store domain arguments
        self.domain_args = args[2:]

        # Initialize base class
        cpp.DirichletBC.__init__(self, *args)

    # Set doc string
    __init__.__doc__ = cpp.DirichletBC.__init__.__doc__

# No fancy handling of PeriodicBC since it requires two different callbacks
PeriodicBC = cpp.PeriodicBC

def homogenize(bc):
    """Return a homogeneous version of the given boundary condition.
    If the given boundary condition is a list of boundary conditions,
    then a list of homogeneous boundary conditions is returned. Only
    Dirichlet boundary conditions are handled. Other types of boundary
    conditions (like periodic) are ignored."""

    # Handle case when boundary condition is a list
    if isinstance(bc, (list, tuple)):
        bcs = bc
        return [homogenize(bc) for bc in bcs]

    # Only consider Dirichlet boundary conditions
    if not isinstance(bc, cpp.DirichletBC):
        return bc

    # Create zero function
    V = bc.function_space()
    if V.element().value_rank() == 0:
        zero = Constant(0)
    elif V.element().value_rank() == 1:
        zero = Constant([0]*V.element().value_dimension(0))
    else:
        cpp.error("Unhandled value rank %d for homogenization of boundary conditions.",
                  V.element().value_rank())

    # Create homogeneous boundary condition
    if len(bc.domain_args) == 1:
        new_bc = cpp.DirichletBC(V, zero, bc.domain_args[0])
    elif len(bc.domain_args) == 2:
        new_bc = cpp.DirichletBC(V, zero, bc.domain_args[0], bc.domain_args[1])
    else:
        cpp.error("Unable to create homogeneous Dirichlet boundary condition. Unknown type of boundary specification.")
    new_bc.domain_args = bc.domain_args

    return new_bc
