/* guilist.c -- GUI interface for manipulating song/artist/disc lists
 *
 * Copyright (c) 1998-2002  Mike Oliphant <oliphant@gtk.org>
 *
 *   http://www.nostatic.org/grip
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <mysql.h>
#include <stdio.h>
#include <string.h>
#include "idlist.h"
#include "listgui.h"
#include "song.h"

static gchar *song_titles[]={"Title","Artist","Disc","Length","Track"};
static gchar *song_fmt[]={NULL,NULL,NULL,NULL,"%02s"};
static gchar *artist_titles[]={"Artist"};
static gchar *disc_titles[]={"Disc","Artist"};

static void RowMove(GtkCList *clist,gint pos1,gint pos2,gpointer user_data);
static void SyncIDList(IDList *idlist,GtkCList *clist);
static char *SkipThe(char *s);
static int NameCompare(GtkCList *clist,gconstpointer ptr1,gconstpointer ptr2);
static void SelectRow(GtkWidget *widget,gint row,gint column,
		      GdkEventButton *event,gpointer data);
static void ClickColumn(GtkCList *clist,gint column,gpointer data);

static void RowMove(GtkCList *clist,gint pos1,gint pos2,gpointer user_data)
{
  int tmp;
  IDList *idlist;

  idlist=(IDList *)user_data;

  if(idlist) {
      tmp=idlist->ids[pos1];

    if(pos1<pos2) {
      memmove(idlist->ids+pos1,idlist->ids+pos1+1,(pos2-pos1)*sizeof(int));
    }
    else {
      memmove(idlist->ids+pos2+1,idlist->ids+pos2,(pos1-pos2)*sizeof(int));
    }

    idlist->ids[pos2]=tmp;
  }
}

static void SyncIDList(IDList *idlist,GtkCList *clist)
{
  int pos;

  for(pos=0;pos<idlist->num_ids;pos++) {
    idlist->ids[pos]=(int)gtk_clist_get_row_data(clist,pos);
  }
}

static char *SkipThe(char *s)
{
  if(!strncasecmp(s,"the ",4)) return s+4;

  return s;
}

static int NameCompare(GtkCList *clist,gconstpointer ptr1,gconstpointer ptr2)

{
  GtkCListRow *r1,*r2;
  char *s1,*s2;

  r1=(GtkCListRow *)ptr1;
  r2=(GtkCListRow *)ptr2;

  s1=GTK_CELL_TEXT(r1->cell[clist->sort_column])->text;
  s2=GTK_CELL_TEXT(r2->cell[clist->sort_column])->text;

  return strcasecmp(SkipThe(s1),SkipThe(s2));
}

static void SelectRow(GtkWidget *widget,gint row,gint column,
		      GdkEventButton *event,gpointer data)
{
  IDList *idlist;

  idlist=(IDList *)data;
  
  if(idlist) {
    idlist->current_id=row;
  }
}

static void ClickColumn(GtkCList *clist,gint column,gpointer data)
{
  IDList *idlist;
  GtkSortType sort_type=GTK_SORT_ASCENDING;

  idlist=(IDList *)data;
  
  if(idlist) {
    gchar *s1,*s2;
    
    gtk_clist_get_text(clist,0,column,&s1);
    gtk_clist_get_text(clist,idlist->num_ids-1,column,&s2);

    if(strcmp(s1,s2)<0) sort_type=GTK_SORT_DESCENDING;
  }
  
  gtk_clist_set_sort_type(clist,sort_type);
  gtk_clist_set_sort_column(clist,column);
  gtk_clist_sort(clist);

  if(idlist) {
    SyncIDList(idlist,clist);
  }
}

/* Make a song CList widget from an id list. The address of the clist widget
   is passed in, and the object to embed (in this case a scrolled window)
   is returned */
GtkWidget *IDListToCList(MYSQL *mysql,IDList *idlist,
			 GtkWidget **in_clist,gchar **titles,gchar **fmts,
			 int num_titles,
			 gchar *id_query)
{
  GtkWidget *clist;
  GtkWidget *scroll;
  int pos;
  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char query[256];
  IDList *old_idlist;

  /* Only create the clist if it isn't already there */
  if(!*in_clist || !GTK_WIDGET_REALIZED(GTK_WIDGET(*in_clist))) {
    clist=gtk_clist_new_with_titles(num_titles,titles);
 
    gtk_clist_set_reorderable(GTK_CLIST(clist),TRUE);
    gtk_clist_set_compare_func(GTK_CLIST(clist),NameCompare);
    
    scroll=gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(scroll),clist);
  }
  else {
    /* Clear the clist and get rid of exising signal handlers */
    clist=*in_clist;
    gtk_clist_clear(GTK_CLIST(clist));

    old_idlist=(IDList *)gtk_object_get_user_data(GTK_OBJECT(clist));

    if(old_idlist) {
      gtk_signal_disconnect_by_func(GTK_OBJECT(clist),
				    GTK_SIGNAL_FUNC(RowMove),
				    old_idlist);
      gtk_signal_disconnect_by_func(GTK_OBJECT(clist),
				    GTK_SIGNAL_FUNC(ClickColumn),
				    old_idlist);
      gtk_signal_disconnect_by_func(GTK_OBJECT(clist),
				    GTK_SIGNAL_FUNC(SelectRow),
				    old_idlist);
    }
  }

  if(idlist) {
    gtk_clist_freeze(GTK_CLIST(clist));

    for(pos=0;pos<idlist->num_ids;pos++) {
      g_snprintf(query,256,id_query,
		 idlist->ids[pos]);

      if(mysql_query(mysql,query)||
	 !(res = mysql_store_result(mysql))||
	 !(row=mysql_fetch_row(res))) {
	printf("Query error on [%s] for pos %d\n",query,pos);
      }
      else {
	gtk_clist_append(GTK_CLIST(clist),row);
	gtk_clist_set_row_data(GTK_CLIST(clist),pos,
			       (gpointer)idlist->ids[pos]);

	mysql_free_result(res);
      }
    }

    gtk_signal_connect(GTK_OBJECT(clist),"row_move",
		       GTK_SIGNAL_FUNC(RowMove),(gpointer)idlist);
    gtk_signal_connect(GTK_OBJECT(clist),"click_column",
		       GTK_SIGNAL_FUNC(ClickColumn),(gpointer)idlist);
    gtk_signal_connect(GTK_OBJECT(clist),"select_row",
		       GTK_SIGNAL_FUNC(SelectRow),(gpointer)idlist);

    gtk_clist_thaw(GTK_CLIST(clist));
  }

  gtk_object_set_user_data(GTK_OBJECT(clist),(gpointer)idlist);    

  gtk_widget_show(clist);

  *in_clist=clist;

  return scroll;
}

/* Make a song CList widget from an id list */
GtkWidget *IDListToSongCList(MYSQL *mysql,IDList *idlist,
			     GtkWidget **in_clist)
{
  GtkWidget *listobj;

  listobj=IDListToCList(mysql,idlist,in_clist,song_titles,song_fmt,5,
			"SELECT song.title,artist,disc.title,play_time,"
			"LPAD(track_num+1,2,'0') FROM artist,disc,song "
			"WHERE song.id=%d "
			"AND artist.id=song.artistid AND disc.id=song.discid");

  gtk_clist_set_column_width(GTK_CLIST(*in_clist),0,200);
  gtk_clist_set_column_width(GTK_CLIST(*in_clist),1,100);
  gtk_clist_set_column_width(GTK_CLIST(*in_clist),2,150);
  gtk_clist_set_column_width(GTK_CLIST(*in_clist),3,40);
  gtk_clist_set_column_width(GTK_CLIST(*in_clist),4,30);
  gtk_clist_set_column_justification(GTK_CLIST(*in_clist),4,
				     GTK_JUSTIFY_RIGHT);

  return listobj;
}

/* Make an artist CList widget from an id list */
GtkWidget *IDListToArtistCList(MYSQL *mysql,IDList *idlist,
			       GtkWidget **in_clist)
{
  GtkWidget *listobj;

  listobj=IDListToCList(mysql,idlist,in_clist,artist_titles,NULL,1,
			"SELECT artist FROM artist WHERE id=%d");

  gtk_clist_set_column_width(GTK_CLIST(*in_clist),0,200);

  return listobj;
}

/* Make an disc CList widget from an id list */
GtkWidget *IDListToDiscCList(MYSQL *mysql,IDList *idlist,
			     GtkWidget **in_clist)
{
  GtkWidget *listobj;

  listobj=IDListToCList(mysql,idlist,in_clist,disc_titles,NULL,2,
			"SELECT disc.title,artist.artist FROM disc,artist "
			"WHERE disc.id=%d AND disc.artistid=artist.id");

  gtk_clist_set_column_width(GTK_CLIST(*in_clist),0,200);
  gtk_clist_select_row(GTK_CLIST(*in_clist),idlist->current_id,0);

  return listobj;
}
