/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dom/cria-block-priv.h>

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <goffice/utils/go-color.h>

#define CDEBUG_TYPE cria_block_get_type
#include <cdebug/cdebug.h>

#include <dom/cria-format.h>
#include <dom/cria-enums.h>
#include <utils/cria-units.h>

#include "application.h"

enum {
	PROP_0,
	PROP_ALIGNMENT_HORIZONTAL,
	PROP_ALIGNMENT_VERTICAL,
	PROP_COLOR,
	PROP_FONT_FAMILY,
	PROP_FONT_SIZE,
	PROP_MODEL,
	PROP_POS_LEFT,
	PROP_POS_RIGHT,
	PROP_POS_TOP,
	PROP_POS_BOTTOM
};

enum {
	FORMAT_CHANGED,
	N_SIGNALS
};

G_DEFINE_TYPE(CriaBlock, cria_block, CRIA_TYPE_SLIDE_ELEMENT);

static	void	cria_block_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_block_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_block_signals[N_SIGNALS] = { 0 };

static void
cb_finalize(GObject* object) {
	CriaBlock* self = CRIA_BLOCK(object);

	cdebug("finalize()", "");
	cria_lepton_unref(self->format);
	self->format = 0;

	g_object_unref(self->model);
	self->model = NULL;

	G_OBJECT_CLASS(cria_block_parent_class)->finalize(object);
}

static void
cria_block_class_init (CriaBlockClass	* cria_block_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_block_class);

	/* setting up property system */
	g_object_class->set_property = cria_block_set_property;
	g_object_class->get_property = cria_block_get_property;
	g_object_class->finalize = cb_finalize;

	g_object_class_install_property(g_object_class,
					PROP_ALIGNMENT_HORIZONTAL,
					g_param_spec_enum("alignment",
							  "Horizontal Alignment",
							  "The horizontal alignment of embedded elements",
							  CRIA_TYPE_ALIGNMENT,
							  CRIA_ALIGNMENT_UNSET,
							  G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_COLOR,
					g_param_spec_pointer("color",
							     "Color",
							     "The text's foreground color",
							     G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_BOTTOM,
					g_param_spec_uint64("bottom",
							    "Bottom",
							    "The width of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_LEFT,
					g_param_spec_uint64("left",
							    "Left",
							    "The Position of the topleft corner of the layout box",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_RIGHT,
					g_param_spec_uint64("right",
							    "Right",
							    "The height of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_MODEL,
					g_param_spec_object("model",
							    "text model",
							    "The text contained in this block",
							    CRIA_TYPE_TEXT_MODEL,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_TOP,
					g_param_spec_uint64("top",
							    "Top",
							    "The vertical position of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_ALIGNMENT_VERTICAL,
					g_param_spec_enum("valignment",
							  "Vertical Alignment",
							  "The vertical alignment of embedded elements",
							  CRIA_TYPE_VALIGNMENT,
							  CRIA_VALIGNMENT_UNSET,
							  G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	/* setting up the block class */
	cria_block_class->format_changed = NULL;
	
	cria_block_signals[FORMAT_CHANGED] = g_signal_new("format-changed",
					CRIA_TYPE_BLOCK,
					G_SIGNAL_RUN_LAST,
					G_STRUCT_OFFSET(CriaBlockClass, format_changed),
					NULL, NULL,
					g_cclosure_marshal_VOID__FLAGS,
					G_TYPE_NONE,
					1,
					CRIA_TYPE_FORMAT_DOMAIN);
}

/**
 * cria_block_get_alignment:
 * @self: the block to get the alignment from
 *
 * Get the horizontal alignment of content in this block
 *
 * Returns the horizontal alignment of content
 */
CriaAlignment
cria_block_get_alignment(CriaBlock const* self) {
	g_return_val_if_fail(CRIA_IS_BLOCK(self), CRIA_ALIGNMENT_UNSET);

	return self->alignment;
}

/**
 * cria_block_get_color:
 * @self: a #CriaBlock
 * 
 * Get the foreground color of a text block.
 *
 * Returns the color of this text block; if it's not set it returns a default
 * value.
 */
GOColor const*
cria_block_get_color(CriaBlock const* self) {
	g_return_val_if_fail(CRIA_IS_BLOCK(self), NULL);

	return self->color;
}

/**
 * cria_block_get_format:
 * @block: a #CriaBlock
 *
 * Get the format information for a block.
 *
 * Returns the format information for the block (as a #CriaLepton).
 */
CriaLepton
cria_block_get_format(CriaBlock const* self) {
	return self->format;
}

/**
 * cria_block_get_model:
 * @self: a #CriaBlock
 *
 * Get the text that should be rendered into a block.
 *
 * Returns the text that should be rendered into a block
 */
CriaTextModel*
cria_block_get_model(CriaBlock const* self) {
	g_return_val_if_fail(CRIA_IS_BLOCK(self), NULL);
	
	return self->model;
}

/**
 * cria_block_get_position:
 * @self: a #CriaBlock
 *
 * Get the position of a #CriaBlock.
 *
 * Returns the block's position. May return NULL if unset.
 */
GORect const*
cria_block_get_position(CriaBlock const* self) {
	g_return_val_if_fail(CRIA_IS_BLOCK(self), NULL);

	return self->position;
}

static void
cria_block_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaBlock* self;

	self = CRIA_BLOCK(object);

	switch(prop_id) {
	case PROP_ALIGNMENT_HORIZONTAL:
		g_value_set_enum(value, cria_block_get_alignment(self));
		break;
	case PROP_ALIGNMENT_VERTICAL:
		g_value_set_enum(value, cria_block_get_valignment(self));
		break;
	case PROP_COLOR:
#warning: "FIXME: this is ugly"
		g_value_set_pointer(value, (gpointer)cria_block_get_color(self));
		break;
	case PROP_POS_LEFT:
		g_value_set_uint64(value, self->position->left);
		break;
	case PROP_POS_RIGHT:
		g_value_set_uint64(value, self->position->right);
		break;
	case PROP_POS_TOP:
		g_value_set_uint64(value, self->position->top);
		break;
	case PROP_POS_BOTTOM:
		g_value_set_uint64(value, self->position->bottom);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_block_get_valignment:
 * @self: the block to get the alignment from
 *
 * Get the vertical alignment of content in this block
 *
 * Returns the vertical alignment of content
 */
CriaVAlignment
cria_block_get_valignment(CriaBlock const* self) {
	g_return_val_if_fail(CRIA_IS_BLOCK(self), CRIA_VALIGNMENT_UNSET);

	return self->valignment;
}

static void
cria_block_init(CriaBlock* self) {
	self->model = g_object_new(CRIA_TYPE_TEXT_MODEL, NULL);
}

CriaBlock*
cria_block_new(const gchar* name) {
	return g_object_new(CRIA_TYPE_BLOCK, "name", name, NULL);
}

void
cria_block_set_alignment(CriaBlock* self, CriaAlignment align) {
	g_return_if_fail(CRIA_IS_BLOCK(self));

	if(align == self->alignment) {
		return;
	}

	self->alignment = align;

	g_object_notify(G_OBJECT(self), "alignment");
}

/**
 * cria_block_set_color:
 * @self: a #CriaBlock
 * @color: a #GOColor
 *
 * Set the foreground color of a text block.
 */
void
cria_block_set_color(CriaBlock* self, GOColor const* color) {
	g_return_if_fail(CRIA_IS_BLOCK(self));

	if(!self->color && color) {
		self->color = g_new0(GOColor, 1);
	} else if(self->color && !color) {
		g_free(self->color);
		self->color = NULL;
	}

	if(color) {
		*self->color = *color;
	}

	g_object_notify(G_OBJECT(self), "color");
}

void
cria_block_set_font_family(CriaBlock* self, const gchar* font) {
	gchar     * new_format;
	CriaLepton  other;
	
	g_return_if_fail(CRIA_IS_BLOCK(self));

	new_format = cria_format_set(self->format, "font-family", font);
	other = cria_lepton_ref(new_format);
	g_free(new_format);
	
	if(other != self->format) {
		CriaLepton three = self->format;
		self->format = other;
		other = three;

		g_signal_emit(self, cria_block_signals[FORMAT_CHANGED], 0, (CriaFormatFont | CriaFormatSize));
	}
	
	cria_lepton_unref(other);
}

/**
 * cria_block_set_font_size:
 * @self: a #CriaBlock
 * @size: a string specifying a font size
 *
 * Sets the font size.
 */
void
cria_block_set_font_size(CriaBlock* self, const gchar* size) {
	gchar     * new_format;
	CriaLepton  other;
	
	g_return_if_fail(CRIA_IS_BLOCK(self));

	new_format = cria_format_set(self->format, "font-size", size);
	other = cria_lepton_ref(new_format);
	g_free(new_format);

	if(other != self->format) {
		CriaLepton three = self->format;
		self->format = other;
		other = three;

		g_signal_emit(self, cria_block_signals[FORMAT_CHANGED], 0, (CriaFormatFont | CriaFormatSize));
	}

	cria_lepton_unref(other);
}

/**
 * cria_block_set_font_size_int:
 * @self: a #CriaBlock
 * @size: the font size (in MCS)
 *
 * Sets the font size in master coordinates.
 */
void
cria_block_set_font_size_int(CriaBlock* self, gint size) {
	gchar* new_size;
	g_return_if_fail(CRIA_IS_BLOCK(self));

	new_size = g_strdup_printf("%dpx", size / MASTER_COORDINATES_PER_POINT);
	cria_block_set_font_size(self, new_size);
	g_free(new_size);
}

void
cria_block_set_font_weight(CriaBlock* self, const gchar* weight) {
	static const gchar* const valid_values[] = {
		"100", "200", "300",
		"400", "500", "600",
		"700", "800", "900",
		"bold", "bolder",
		"inherit",
		"lighter", "normal"
	}; /* according to http://www.w3.org/TR/REC-CSS2/fonts.html#font-styling */
	guint      i;
	
	g_return_if_fail(CRIA_IS_BLOCK(self));
	g_return_if_fail(weight);

	for(i = 0; i < sizeof(valid_values) && strcmp(valid_values[i], weight) < 1; i++) {
		if(!strcmp(weight, valid_values[i])) {
			gchar* new_format = cria_format_set(self->format, "font-weight", valid_values[i]);
			CriaLepton new = cria_lepton_ref(new_format);

			if(new != self->format) {
				CriaLepton three = self->format;
				self->format = new;
				new = three;

				g_signal_emit(self, cria_block_signals[FORMAT_CHANGED], 0, (CriaFormatFont | CriaFormatSize));
			}
			
			cria_lepton_unref(new);
			g_free(new_format);

			return;
		}
	}

	g_critical("Unknown font-weight value: %s", weight);
}

void
cria_block_set_font_style(CriaBlock* self, const gchar* style) {
	static const gchar* const valid_values[] = {
		"inherit", "italic",
		"normal", "oblique"
	}; /* according to http://www.w3.org/TR/REC-CSS2/fonts.html#font-styling */
	guint i;

	g_return_if_fail(CRIA_IS_BLOCK(self));
	g_return_if_fail(style);

	for(i = 0; i < sizeof(valid_values) && strcmp(valid_values[i], style) < 1; i++) {
		if(!strcmp(style, valid_values[i])) {
			gchar* new_format = cria_format_set(self->format, "font-style", valid_values[i]);
			CriaLepton new = cria_lepton_ref(new_format);

			if(new != self->format) {
				CriaLepton three = self->format;
				self->format = new;
				new = three;

				g_signal_emit(self, cria_block_signals[FORMAT_CHANGED], 0, (CriaFormatFont | CriaFormatSize));
			}

			cria_lepton_unref(new);
			g_free(new_format);
			return;
		}
	}

	g_critical("Unknown font-style value: %s", style);
}

static void
cria_block_set_pos_bottom(CriaBlock* self, GODistance bottom) {
	g_return_if_fail(CRIA_IS_BLOCK(self));

	cdebugo(self, "setPosBottom()", "%lli", bottom);
	
	if(self->position == NULL) {
		self->position = g_new0(GORect,1);
	}

	if(bottom != self->position->bottom) {
		self->position->bottom = bottom;

		g_object_notify(G_OBJECT(self), "bottom");
	}
}

static void
cria_block_set_pos_left(CriaBlock* self, GODistance left) {
	g_return_if_fail(CRIA_IS_BLOCK(self));
	
	cdebugo(self, "setPosLeft()", "%lli", left);
	
	if(self->position == NULL) {
		self->position = g_new0(GORect,1);
	}

	if(left != self->position->left) {
		self->position->left = left;

		g_object_notify(G_OBJECT(self), "left");
	}
}

static void
cria_block_set_pos_right(CriaBlock* self, GODistance right) {
	g_return_if_fail(CRIA_IS_BLOCK(self));
	
	cdebugo(self, "setPosRight()", "%lli", right);
	
	if(self->position == NULL) {
		self->position = g_new0(GORect,1);
	}

	if(right != self->position->right) {
		self->position->right = right;

		g_object_notify(G_OBJECT(self), "right");
	}
}

static void
cria_block_set_pos_top(CriaBlock* self, GODistance top) {
	g_return_if_fail(CRIA_IS_BLOCK(self));
	
	cdebugo(self, "setPosTop()", "%lli", top);
	
	if(self->position == NULL) {
		self->position = g_new0(GORect,1);
	}

	if(top != self->position->top) {
		self->position->top = top;

		g_object_notify(G_OBJECT(self), "top");
	}
}

/**
 * cria_block_set_position:
 * @self: a #CriaBlock
 * @x1: a horizontal position
 * @y1: a vertical position
 * @x2: a horizontal position
 * @y2: a vertical position
 *
 * Set the position of a block.
 */
void
cria_block_set_position(CriaBlock* self, gdouble x1, gdouble y1, gdouble x2, gdouble y2) {
	cria_block_set_pos_left(self, MIN(x1, x2));
	cria_block_set_pos_right(self, MAX(x1, x2));
	
	cria_block_set_pos_top(self, MIN(y1, y2));
	cria_block_set_pos_bottom(self, MAX(y1, y2));
}

static void
cria_block_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaBlock* self;
	
	self = CRIA_BLOCK(object);
	
	switch(prop_id) {
	case PROP_ALIGNMENT_HORIZONTAL:
		cria_block_set_alignment(self, g_value_get_enum(value));
		break;
	case PROP_ALIGNMENT_VERTICAL:
		cria_block_set_valignment(self, g_value_get_enum(value));
		break;
	case PROP_COLOR:
		cria_block_set_color(self, g_value_get_pointer(value));
		break;
	case PROP_POS_LEFT:
		cria_block_set_pos_left(self, g_value_get_uint64(value));
		break;
	case PROP_POS_RIGHT:
		cria_block_set_pos_right(self, g_value_get_uint64(value));
		break;
	case PROP_POS_TOP:
		cria_block_set_pos_top(self, g_value_get_uint64(value));
		break;
	case PROP_POS_BOTTOM:
		cria_block_set_pos_bottom(self, g_value_get_uint64(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

void
cria_block_set_model(CriaBlock* self, CriaTextModel* model) {
	g_return_if_fail(CRIA_IS_BLOCK(self));
	g_return_if_fail(CRIA_IS_TEXT_MODEL(model));

	if(self->model == model) {
		return;
	}

	g_object_unref(self->model);
	self->model = g_object_ref(model);

	g_object_notify(G_OBJECT(self), "model");
}

void
cria_block_set_valignment(CriaBlock* self, CriaVAlignment valign) {
	g_return_if_fail(CRIA_IS_BLOCK(self));
	
	self->valignment = valign;

	g_object_notify(G_OBJECT(self), "valignment");
}

