#ifndef CAMPANEL_HPP
#define CAMPANEL_HPP

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <qdialog.h>
#include <qimage.h>
#include <qobject.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qsize.h>
#include <qwidget.h>

class CCamPanel: public QWidget
{
   Q_OBJECT
private:
   // Image and visible size
   QSize ImgSize, VisSize;
   QString Name, Description;
   QImage ImgNull; // Default empty image
   bool Drawable;	///< Drawable: if false, does not show up in a list, and can never be drawn
   int Usage;		///< Usage counter; 0 means unused
   int PanelType;	///< Type; see \ref PanelTypes

   void CallIntg(int n, uchar *dst, uchar *src);

protected:   
   /// Image width; set by SetImageSize();
   int image_w;
   /// Image height; set by SetImageSize();
   int image_h;
   /// Half image width; set by SetImageSize();
   int half_w;
   /// Half image height; set by SetImageSize();
   int half_h;

   /// Ready-to-use QImage object for RGB image
   QImage ImgRGB;
   /// Ready-to-use QImage object for Y data.
   QImage ImgY;
   /// Ready-to-use QImage object for U data.
   QImage ImgU;
   /// Ready-to-use QImage object for V data.
   QImage ImgV;
   /// Ready-to-use QPixmap for graphs
   QPixmap PixGraph;

   QPopupMenu *pMenu;
   bool MenuVisible;

   CCamPanel(const QString &new_name, const QString &new_description, int panel_type, bool draw = TRUE, QWidget *parent = 0, const char *name = 0);
   void SetImageSize(const QSize &new_size);
   void SetVisibleSize(const QSize &new_size);
   void CreateImages();
   
   void ConnectUsage(CCamPanel *parent);
   void ConnectResizes(CCamPanel *parent);
   
public:
   /** The following enumeration helps in determining how a panel
       should be handled, and how many images it houses.
     */
   enum PanelTypes {
     None,      ///< Undrawable
     RGB,	///< Single RGB image
     YUV420,    ///< YUV images in 4:2:0 format, 1 large and 2 smaller
     Graph,     ///< Not an image
   };
  
   virtual ~CCamPanel();
   
   QString GetName() const;
   QString GetDescription() const;
   int GetPanelType() const;
   QSize GetImageSize() const;
   QSize GetVisibleSize() const;
   bool IsDrawable();
   int IsUsed();
   
//   virtual void Draw(QPainter *p);
   virtual const QImage &GetImage(int n = 0);
   const QPixmap &GetPixmap() const;
   
   virtual QDialog *GetPropertiesDialog() const;

   // Operators
   const CCamPanel &copy(const CCamPanel &src);
   const CCamPanel &operator += (const CCamPanel &add);
   CCamPanel &operator -= (const CCamPanel &minus);

   // overloaded from QWidget
   virtual QSize sizeHint() const;
   virtual QSizePolicy sizePolicy() const;
   virtual void paintEvent(QPaintEvent *);

public slots:
   virtual void SetSize(const QSize &new_size);

   void IncrementUse();
   void DecrementUse();
   
   virtual void RightClick(const QPoint &pos);

signals:
   /// Emitted when the principle image-size has been changed.
   void ChangedImageSize(const QSize &new_size);
   /// Emitted when the visible size has been changed.
   void ChangedVisibleSize(const QSize &new_size);
   /// The visibility has changed.
   void ChangedVisibility(bool new_visibility);
   /// Only emitted when the panel is turned on.
   void ChangedToVisible();
   /// Only emitted when the panel is turned off.
   void ChangedToHidden();
   /// Emitted whenever the usage counter changes.
   void ChangedUsage(int new_usage);
   /// Emitted when the usage counter increases to 1 (first use)
   void ChangedToUsed();
   /// Emitted when the usage counter drops to 0 (not used any more)
   void ChangedToUnused();
   /// Emitted after the panel's data has been updated.
   void Updated();
};

#endif
