/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "CommandLineParser.h"

// for exit status
#include <cstdlib>
// for std::find
#include <algorithm>

#include <iostream>

//- CamiTK
#include <Core.h>

using namespace camitk;

// ------------- constructor -----------------
CommandLineParser::CommandLineParser(int argc, char *argv[]) {
  // fiilename without path (using / for unix and \\ for windows
  std::string nameArg(argv[0]);
  size_t found = nameArg.find_last_of("/\\");
  if (found!=std::string::npos)
    name = nameArg.substr(found+1);
  else
    name = nameArg;
  for (unsigned int i=1; i<argc; i++) {
    arguments.push_back(argv[i]);
  }
}

// ------------- setDescription -----------------
void CommandLineParser::setDescription(std::string description) {
  this->description = description;
}

// ------------- option -----------------
bool CommandLineParser::option(std::string option) {
  bool optionFound = false;
  // for the first letter
  std::string shortOption("-");
  shortOption += option.at(0);  
  optionFound = (std::find(arguments.begin(), arguments.end(), shortOption) != arguments.end());
  if (!optionFound) {
    // check for complete name
    std::string longOption("--");
    longOption += option;
    optionFound = (std::find(arguments.begin(), arguments.end(), longOption) != arguments.end());
  }
  return optionFound;
}

// ------------- parse -----------------
void CommandLineParser::help() {
    std::cout << name << std::endl;
    std::cout << std::endl;
    std::cout << "Usage: " << name << " [OPTIONS]... [FILES]..." << std::endl;
    std::cout << std::endl;
    
    std::cout << "Build using " << Core::version << std::endl;
    std::cout << std::endl;
    
    std::cout << description << std::endl;
    std::cout << std::endl;
    
    std::cout << "Generic options:" << std::endl;
    std::cout << "  -h, --help         Print help and exit" << std::endl;
    std::cout << "  -v, --version      Print version and exit" << std::endl;
    std::cout << std::endl;  
    
    std::cout << "Options:" << std::endl;    
    for (std::map<std::string, std::string>::iterator it=options.begin(); it!=options.end();it++) {
      // TODO compute space in order to align all description string as a second column
      std::cout << "  -" << (*it).first.at(0) << ", --" << (*it).first << "    " << (*it).second << std::endl;
    }

    std::cout << "Arguments:" << std::endl;
    std::cout << "  URL                Document(s) to open" << std::endl;
}

// ------------- version -----------------
void CommandLineParser::version() {
    std::cout << name << " build using " << Core::version << std::endl;
}

// ------------- parse -----------------
void CommandLineParser::parse() {
  if (option("help")) {
    help();
    exit (EXIT_SUCCESS);
  }
  else 
    if (option("version")) {
      version();
      exit(EXIT_SUCCESS);      
    }
    else {
      // check for all options, see if they exists
      bool parsingError = false;
      std::vector<std::string>::iterator it=arguments.begin();
      while (it!=arguments.end() && !parsingError) {
	// there is a parse error here if a given argument is not a possible option
	// if single dash option
	if ((*it).size()==2) {
	  std::map<std::string, std::string>::iterator itMap = options.begin();
	  while(itMap != options.end() && (*itMap).first.at(0) != (*it).at(1))
	    itMap++;
	  parsingError = (itMap == options.end());
	}
	else {
	  // double dash option
	  parsingError = (options.find((*it).substr(2,(*it).size()-2)) == options.end());
	}
	// next one
	it++;
      }
      // check if we were not at remaining 
      if (parsingError) {
	it--;
	parsingError = false;
	while (it!=arguments.end() && !parsingError) {
	  // there is a parse error here if the arguments does not start with "-"
	  parsingError = ((*it).at(0) == '-');
	  // fill up remainingArguments
	  if (!parsingError) {
	    remainingArguments.push_back((*it));
	  }
	  it++;
	}
      }
      if (parsingError) {
	help();
	exit (EXIT_FAILURE);
      }
    }    
}

// ------------- addOption -----------------
void CommandLineParser::addOption(std::string optionName, std::string description) {
  options.insert(std::pair<std::string, std::string>(optionName, description));
}
  
  
// ------------- getRemainingParameters -----------------
std::vector<std::string> & CommandLineParser::getRemainingArguments() {
  return remainingArguments;
}
