/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
// because of namespace problems with Component, put these include first
#include <monitoring/manager/MonitoringManagerFactory.h>
#include <monitoring/simulator/SimulatorFactory.h>
#include <mml/MonitorIn.hxx>
#include <lml/Translation.h>
#include <MultiComponent.h>

#include "GenerateModel.h"
#include <Component.h>
#include <Log.h>
#include <Core.h>
#include <Application.h>

#include <MeshComponent.h>
#include <VtkMeshComponent.h>
#include <MMLComponent.h>

#include <vtkCleanPolyData.h>
#include <vtkPolyData.h>
#include <vtkPointLocator.h>
#include <vtkSelectEnclosedPoints.h>
#include <vtkAppendPolyData.h>
#include <vtkTetra.h>
#include <vtkUnstructuredGrid.h>
#include <vtkCellArray.h>

#include <QMessageBox>
#include <QFileInfo>

#include "tetgen.h" // Defined tetgenio, tetrahedralize().

#include <set>



// -------------------- GenerateModel --------------------
GenerateModel::GenerateModel(ActionExtension* extension) : Action(extension) {
    this->setName("Generate Model");
    this->setDescription("Generates a model from a mesh: mml, msh, sofa scn, pml and corresponding lml are generated");
    this->setComponent("MeshComponent");
    this->setFamily("Mesh Processing");
    this->addTag("Build Biomechanical Model");
}


// --------------- apply -------------------
Action::ApplyStatus GenerateModel::apply() {
    CAMITK_INFO("GenerateModel", "apply", "Tetrahedalizing " << getTargets().last()->getName().toStdString());

    // set waiting cursor
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    camitk::Component *myComponent = getTargets().last();

    //-- clean the polydata
    // remove redundant points
    int tolerance = 1; // TODO ask the user
    vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
    cleaner->AddInput ( myComponent->getPointSet() );
    cleaner->PointMergingOn();
    cleaner->SetTolerance ( double ( tolerance ) *0.01 );  // % of the Bounding Box
    cleaner->ConvertLinesToPointsOn ();
    cleaner->ConvertPolysToLinesOn ();
    cleaner->ConvertStripsToPolysOn ();
    cleaner->Update();

    /* This is probably not necessary:
    // some triangles can have degenerate into lines, some points might be
    // no more part of any triangle => remove all points that are not used in a triangle
    std::set<int> selectedPoints;
    for (unsigned int i=0;i<cleaner->GetOutput()->GetNumberOfCells();i++) {
        if (cleaner->GetOutput()->GetCell(i)->GetNumberOfPoints()==3)
            for (unsigned int j=0;j<3;j++)
                selectedPoints.insert(cleaner->GetOutput()->GetCell(i)->GetPointId(j));
    }
    // now use a selected filter to keep only the selectedPoints and the cells they composed
    ...
    */

    vtkSmartPointer<vtkPolyData> cleanTriangulatedSurface = cleaner->GetOutput();
    if ( !cleanTriangulatedSurface || !cleanTriangulatedSurface->GetPoints() )    {
        QMessageBox::warning ( NULL, "Type Error", "Can't generate physical model from vtkUnstructuredGrid\n(need a vtkPolyData)" );
        return;
    }

    //-- compute mesh dimensions
    double *bounds = cleanTriangulatedSurface->GetPoints()->GetBounds(); // [xmin,xmax, ymin,ymax, zmin,zmax]
    double boundingRadius = 0.0;
    for ( unsigned int i = 0;i < 3;i++ ) {
        boundingRadius += ( bounds[i*2+1] - bounds[i*2] ) * ( bounds[i*2+1] - bounds[i*2] ) / 4.0;
    }
    boundingRadius = sqrt ( boundingRadius );

    //-- computes the subdivision depending on the mesh resolution
    vtkSmartPointer<vtkPointLocator> subdivisionLocator = vtkSmartPointer<vtkPointLocator>::New();
    subdivisionLocator->SetDataSet ( cleanTriangulatedSurface );
    subdivisionLocator->SetNumberOfPointsPerBucket ( 1 );
    subdivisionLocator->BuildLocator();
    int nbDiv[3];
    subdivisionLocator->GetDivisions ( nbDiv );

    //-- 1. generates a regular grid in the bounding box
    qsrand ( QDateTime::currentDateTime().toTime_t() );
    double xDiv = ( bounds[1] - bounds[0] ) / nbDiv[0];
    double yDiv = ( bounds[3] - bounds[2] ) / nbDiv[1];
    double zDiv = ( bounds[5] - bounds[4] ) / nbDiv[2];
    double x, y, z;

    vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();
    for ( int i = 0; i <= nbDiv[0]; i++ ) {
        x = bounds[0] + i * xDiv;
        for ( int j = 0; j <= nbDiv[1]; j++ ) {
            y = bounds[2] + j * yDiv;
            for ( int k = 0; k <= nbDiv[2]; k++ ) {
                z = bounds[4] + k * zDiv;
                // randomize
                x += ( xDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
                y += ( yDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
                z += ( zDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
                points->InsertNextPoint ( x, y, z );
            }
        }
    }

    //-- 2. select internal points only
    vtkSmartPointer<vtkPolyData> gridPoints = vtkSmartPointer<vtkPolyData>::New();
    gridPoints->SetPoints ( points );
    vtkSmartPointer<vtkSelectEnclosedPoints> select = vtkSmartPointer<vtkSelectEnclosedPoints>::New();
    select->SetInput ( gridPoints );
    select->SetSurface ( cleanTriangulatedSurface );
    select->Update();

    vtkSmartPointer<vtkPoints> insidePointsPoints = vtkSmartPointer<vtkPoints>::New();
    for ( int i = 0; i < gridPoints->GetPoints()->GetNumberOfPoints(); i++ ) {
        if ( select->IsInside ( i ) ) {
            insidePointsPoints->InsertNextPoint ( gridPoints->GetPoints()->GetPoint ( i ) );
        }
    }

    vtkSmartPointer<vtkPolyData> insidePoints = vtkSmartPointer<vtkPolyData>::New();
    insidePoints->SetPoints ( insidePointsPoints );

    //-- 3. Generates one structure with all the points
    vtkSmartPointer<vtkAppendPolyData> appender = vtkSmartPointer<vtkAppendPolyData>::New();
    appender->AddInput ( cleanTriangulatedSurface );
    appender->AddInput ( insidePoints );
    appender->Update();
    vtkSmartPointer<vtkPolyData> everything = cleanTriangulatedSurface; //appender->GetOutput();

    //-- 4. use tetgen to generate delaunay tetra
    // 4.1 build the tetgen data structures using just the points
    tetgenio in;
    // All indices start from 1.
    in.firstnumber = 1;

    in.numberofpoints = everything->GetPoints()->GetNumberOfPoints();
    in.pointlist = new REAL[in.numberofpoints * 3];
    int index = 0;
    double xyz[3];
    for ( int i = 0; i < everything->GetPoints()->GetNumberOfPoints(); i++ ) {
        everything->GetPoints()->GetPoint ( i, xyz );
        in.pointlist[index++] = xyz[0];
        in.pointlist[index++] = xyz[1];
        in.pointlist[index++] = xyz[2];
    }

    // generates all facets, use only triangles
    unsigned nrOfTriangles = 0;
    for (int i = 0;i < everything->GetNumberOfCells(); i++) {
        if (everything->GetCell(i)->GetNumberOfPoints() == 3)
            nrOfTriangles++;
    }

    in.numberoffacets = nrOfTriangles;
    in.facetlist = new tetgenio::facet[in.numberoffacets];
    //in.facetmarkerlist = new int[in.numberoffacets];

    // generates tetgen triangle facets
    for (int i = 0, facetId = 0; i < everything->GetNumberOfCells(); i++) {
        vtkCell *c = everything->GetCell(i);
        if (c->GetNumberOfPoints() == 3) {
            //in.facetmarkerlist[facetId] = 1; // set boundary marker (but why?)
            tetgenio::facet *f = &in.facetlist[facetId];
            f->numberofpolygons = 1;
            f->polygonlist = new tetgenio::polygon[f->numberofpolygons];
            f->numberofholes = 0;
            f->holelist = NULL;
            tetgenio::polygon *p = &f->polygonlist[0];
            p->numberofvertices = 3;
            p->vertexlist = new int[p->numberofvertices];
            p->vertexlist[0] = in.firstnumber + c->GetPointId(0);
            p->vertexlist[1] = in.firstnumber + c->GetPointId(1);
            p->vertexlist[2] = in.firstnumber + c->GetPointId(2);
            // next time, process next facet
            facetId++;
        }
    }
    // save input mesh
    //in.save_poly("tet");
    //in.save_faces("tet");
    //in.save_nodes("tet");

    //in.numberoffacets = 0;

    // 4.2 GenerateModel the PLC
    // GenerateModel the PLC. Switches are chosen to read a PLC (p),
    //   do quality mesh generation (q) with a specified quality bound
    //   (1.414), and apply a maximum volume constraint (a0.1).
    // NOTE (warning) with the q option tetgen will add some vertex, the point list has to be updated!
    //tetrahedralize("pq1.414a0.1", &in, &out);
    tetgenio firstPass, secondPass, out;
    tetrahedralize("pq1.414V", &in, &out);
    /*
    tetrahedralize("-rq1.414", &firstPass, &secondPass);
    tetrahedralize("-ra0.5", &secondPass, &out);
    */
    // save Output elements
    //out.save_elements ((char *) QString(QFileInfo(myComponent->getFileName()).completeBaseName() + "_out").toStdString().c_str());
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();
    for (int i = 0;i < out.numberofpoints;i++) {
        thePoints->InsertNextPoint(out.pointlist[i*3], out.pointlist[i*3+1], out.pointlist[i*3+2]);
    }

    // -- 5. export to VTK
    // 5.1 import the cells
    vtkSmartPointer<vtkCellArray> theCells = vtkSmartPointer<vtkCellArray>::New();
    for (int i = 0; i < out.numberoftetrahedra; i++) {
        vtkSmartPointer<vtkCell> c = vtkSmartPointer<vtkTetra>::New();

        /* Vtk/Pml tetrahedron                   Tetgen tetrahedron
        *       3                                          3
        *     /| \                                        /| \
        *    / |  \                                      / |  \
        *   1..|...\ 2                                  0..|...\ 2
        *   \  |   /                      vs            \  |   /
        *    \ |  /                                      \ |  /
        *     \|/                                         \|/
        *      0                                           1
        * => 1 and 0 are inversed
        */
        c->GetPointIds()->SetId(0, out.tetrahedronlist[i * out.numberofcorners + 1] - in.firstnumber);
        c->GetPointIds()->SetId(1, out.tetrahedronlist[i * out.numberofcorners + 0] - in.firstnumber);
        c->GetPointIds()->SetId(2, out.tetrahedronlist[i * out.numberofcorners + 2] - in.firstnumber);
        c->GetPointIds()->SetId(3, out.tetrahedronlist[i * out.numberofcorners + 3] - in.firstnumber);
        theCells->InsertNextCell(c);
    }

    // 5.2 generates new point set (unstructured grid)
    vtkSmartPointer<vtkUnstructuredGrid> newPointSet = vtkSmartPointer<vtkUnstructuredGrid>::New();
    newPointSet->Allocate(theCells->GetNumberOfCells());
    newPointSet->SetCells(VTK_TETRA, theCells);
    newPointSet->SetPoints(thePoints);
    newPointSet->Update();

    //-- 6. Generate vtk
    // VtkMeshUtil::saveUnstructuredGridToFile(newPointSet, QString(QFileInfo(myComponent->getFileName()).completeBaseName() + "_GenerateModeld.vtk").toStdString());

    /*
    //-- 6. generate the PML
    QString pmlFilename = QFileInfo(myComponent->getFileName()).completeBaseName() + "-model.pml";

    ofstream outputFile ( pmlFilename.toUtf8() );
    outputFile << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
    outputFile << "<!-- Created by " << Core::shortVersion << " " << QDate::currentDate().year() << " -->" << endl;
    QDateTime now = QDateTime::currentDateTime();
    outputFile << "<!-- " << now.toString().toStdString() << " " << now.toTime_t() << "-->" << endl;
    outputFile << "<physicalModel name=\"Physical Model\" >" << endl;

    //--- atoms
    outputFile << "<atoms>" << endl;
    outputFile << "<structuralComponent name=\"Particles\">" << endl;

    // surface points
    for ( int i = 0; i < newPointSet->GetPoints()->GetNumberOfPoints(); i++ ) {
        newPointSet->GetPoints()->GetPoint(i, xyz);
        // write atom
        outputFile << "<atom>" << endl;
        outputFile << "<atomProperties index=\"" << i << "\" x=\"" << xyz[0] << "\"  y=\"" << xyz[1] << "\" z=\"" << xyz[2] << "\"  />" << endl;
        outputFile << "</atom>" << endl;
    }

    outputFile << "</structuralComponent>" << endl;
    outputFile << "</atoms>" << endl;

    //--- Regions
    outputFile << "<exclusiveComponents>" << endl;
    outputFile << "<multiComponent name=\"Exclusive Components\">" << endl;
    outputFile << "<structuralComponent name=\"Elements\" mode=\"WIREFRAME_AND_SURFACE\"  >" << endl;
    outputFile << "<color r=\".95\" g=\"0.89\" b=\"0.63\" a=\"1.0\"/>" << endl;
    outputFile << "<nrOfStructures value=\"" << newPointSet->GetCells()->GetNumberOfCells() << "\"/>" << endl;

    for ( unsigned int i = 0; i < newPointSet->GetCells()->GetNumberOfCells() ; i++ ) {
        unsigned int nPoints = newPointSet->GetCell(i)->GetNumberOfPoints();
        // just to make sure this is a tetra
        if (nPoints == 4) {
            outputFile << "<cell>" << endl;
            outputFile << "<cellProperties type=\"TETRAHEDRON\" />" << endl;
            for ( unsigned int j = 0; j < nPoints; j++ ) {
                unsigned int id = newPointSet->GetCell(i)->GetPointId(j);
                outputFile << "<atomRef index=\"" << id << "\" />" << endl;
            }
            outputFile << "</cell>" << endl;
        }
    }

    outputFile << "</structuralComponent>" << endl;
    outputFile << "</multiComponent>" << endl;
    outputFile << "</exclusiveComponents>" << endl;
    outputFile << "<informativeComponents/>" << endl;
    outputFile << "</physicalModel>" << endl;
    outputFile.close();

    // and finally load the model!
    Application::getMainWindow()->open(pmlFilename);
    */

    //-- 6 Generate .msh and .scn
    // see http://www.geuz.org/gmsh/doc/texinfo/gmsh.html#File-formats for the msh format
    // this is a subset working in sofa
    QString baseFilename = QFileInfo(myComponent->getFileName()).absolutePath() + "/" + QFileInfo(myComponent->getFileName()).completeBaseName();

    QString mshFilename = baseFilename + "-model.msh";
    ofstream mshFile ( mshFilename.toUtf8() );
    mshFile << "$NOD" << endl;
    mshFile << newPointSet->GetPoints()->GetNumberOfPoints() << endl;
    for ( int i = 0; i < newPointSet->GetPoints()->GetNumberOfPoints(); i++ ) {
        newPointSet->GetPoints()->GetPoint(i, xyz);
        // write atom
        mshFile << i + 1 << " " << xyz[0] << " " << xyz[1] << " " << xyz[2] << endl;
    }
    mshFile << "$ENDNOD" << endl;
    mshFile << "$ELM" << endl;
    mshFile << newPointSet->GetCells()->GetNumberOfCells() << endl;

    for (int i = 0; i < newPointSet->GetCells()->GetNumberOfCells() ; i++ ) {
        unsigned int nPoints = newPointSet->GetCell(i)->GetNumberOfPoints();
        // just to make sure this is a tetra
        if (nPoints == 4) {
            // beware of the format (different from vtk)
            //      Tetrahedron:                          Tetrahedron10:
            //
            //                         v
            //                       .
            //                     ,/
            //                    /
            //                 2                                     2
            //               ,/|`\                                 ,/|`\
            //             ,/  |  `\                             ,/  |  `\
            //           ,/    '.   `\                         ,6    '.   `5
            //         ,/       |     `\                     ,/       8     `\
            //       ,/         |       `\                 ,/         |       `\
            //      0-----------'.--------1 --> u         0--------4--'.--------1
            //       `\.         |      ,/                 `\.         |      ,/
            //          `\.      |    ,/                      `\.      |    ,9
            //             `\.   '. ,/                           `7.   '. ,/
            //                `\. |/                                `\. |/
            //                   `3                                    `3
            //                      `\.
            //                         ` w
            mshFile << i + 1 << " 4 1 1 4 " << newPointSet->GetCell(i)->GetPointId(0) + 1 << " "
            << newPointSet->GetCell(i)->GetPointId(2) + 1 << " "
            << newPointSet->GetCell(i)->GetPointId(1) + 1 << " "
            << newPointSet->GetCell(i)->GetPointId(3) + 1 << endl;
        }
    }

    mshFile << "$ENDELM" << endl;
    mshFile.close();

    // list the 20% lowest points (from -x to +x)
    double lowestPercentage = 0.2;
    std::set<int> dofIndex;
    for ( int i = 0; i < newPointSet->GetPoints()->GetNumberOfPoints(); i++ ) {
        newPointSet->GetPoints()->GetPoint(i, xyz);
        if (xyz[0] < bounds[0] + (bounds[1] - bounds[0])*lowestPercentage)
            dofIndex.insert(i);
    }

    // largely inspired (i.e. completely copied) from liver.scn demo file
    QString scnFilename = baseFilename + "-model.scn";
    ofstream scnFile ( scnFilename.toUtf8() );
    scnFile << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
    scnFile << "<!-- Created by " << Core::shortVersion << " -->" << endl;
    QDateTime now = QDateTime::currentDateTime();
    scnFile << "<!-- " << now.toString().toStdString() << " " << now.toTime_t() << "-->" << endl;
    scnFile << "<Node name=\"root\" gravity=\"-9.81 0 0\" dt=\"0.02\" showBehaviorModels=\"0\" showCollisionModels=\"0\" showMappings=\"0\" showForceFields=\"0\">" << endl;
    scnFile << "  <DefaultPipeline name=\"CollisionPipeline\"  verbose=\"0\" />" << endl;
    scnFile << "  <BruteForceDetection name=\"N2\" />" << endl;
    scnFile << "  <DefaultContactManager name=\"collision response\"  response=\"default\"/>" << endl;
    scnFile << "  <Node name=\"CamiTKModel\" gravity=\"0 -9.81 0\" depend=\"topo dofs\">" << endl;
    scnFile << "    <EulerImplicitSolver name=\"cg_odesolver\"  printLog=\"0\" />" << endl;
    scnFile << "    <CGLinearSolver template=\"GraphScattered\" name=\"linear solver\"  iterations=\"25\"  tolerance=\"1e-09\" threshold=\"1e-09\" />" << endl;
    scnFile << "    <MeshLoader name=\"meshLoader\"  filename=\"" << mshFilename.toStdString() << "\" />" << endl;
    scnFile << "    <TetrahedronSetTopologyContainer name=\"topo\" />" << endl;
    scnFile << "    <MechanicalObject template=\"Vec3d\" name=\"dofs\"  position=\"0 0 0\"  velocity=\"0 0 0\"  force=\"0 0 0\"  derivX=\"0 0 0\"  free_position=\"0 0 0\"  free_velocity=\"0 0 0\"  restScale=\"1\" />" << endl;
    scnFile << "    <TetrahedronSetGeometryAlgorithms template=\"Vec3d\" name=\"GeomAlgo\" />" << endl;
    scnFile << "    <DiagonalMass template=\"Vec3d\" name=\"computed using mass density\"  massDensity=\"1\" />" << endl;
    scnFile << "    <TetrahedralCorotationalFEMForceField template=\"Vec3d\" name=\"FEM\"  method=\"large\"  poissonRatio=\"0.4\"  youngModulus=\"12500\"  computeGlobalMatrix=\"0\" />" << endl;
    /*
    // Without LML, you would have to do this:
    scnFile << "    <FixedConstraint template=\"Vec3d\" name=\"FixedConstraint\" indices=\"";
    std::set<int>::const_iterator pos;
    //preincrement and predecrement are faster
    //than postincrement and postdecrement...
    for (pos = dofIndex.begin(); pos != dofIndex.end(); ++pos)
        scnFile << *pos << " ";

    scnFile << "\" />" << endl;
    */
    scnFile << "  </Node>" << endl;
    scnFile << "</Node>" << endl;
    scnFile.close();

    //-- 7. Generate LML
    Loads l;
    Translation *t = new Translation();
    std::set<int>::const_iterator pos;
    for (pos = dofIndex.begin(); pos != dofIndex.end(); ++pos)
        t->addTarget(*pos);
    t->setUnit(TranslationUnit::MM);
    Direction d;
    d.setNullX();
    d.setNullY();
    d.setNullZ();
    t->setDirection(d);
    t->addValueEvent(1,0.0);
    l.addLoad(t);
    QString lmlFilename = baseFilename + "-boundary-condition.lml";
    ofstream lmlofs(lmlFilename.toStdString().c_str());
    l.xmlPrint(lmlofs);
    lmlofs.close();
    
    //-- 8 Generate MML
    // generate a default runable mml using this .scn
    // dt and refresh
    QString pmlFilename = QFileInfo(myComponent->getFileName()).completeBaseName() + "-model.pml";
    mml::TimeParameter dt=mml::TimeParameter(0.1,"s");
    mml::TimeParameter refresh=mml::TimeParameter(0.1,"s");
    mml::MonitoringIn mmlIn(dt,refresh,"sofa");
    // add the simulator file
    mmlIn.simulatorFile(QFileInfo(scnFilename).fileName().toStdString().c_str());
    // add stopping criteria
    mml::Position notALotOfMovement(mml::Threshold(0.1), "mm");
    notALotOfMovement.method().scope(mml::Scope::Any);
    mml::Time enoughTimeToStart(mml::MinThreshold(0.5), "s");
    enoughTimeToStart.method().scope(mml::Scope::Any);
    mml::MultipleCriteria multipleCriteria(mml::LogicalOp::And);
    multipleCriteria.criteria().push_back(notALotOfMovement);
    multipleCriteria.criteria().push_back(enoughTimeToStart);
    mml::StoppingCriteria stoppingCriteria;
    stoppingCriteria.multipleCriteria(multipleCriteria);
    mmlIn.stoppingCriteria(stoppingCriteria);
    //TODO 
    // add monitors
    //mml::Monitor displacement(mml::TimeParameter(0.0,"s"),mml::TimeParameter(1000000.0,"s"),mml::MonitorType::PointSetDistance,1,QFileInfo(pmlFilename).fileName().toStdString().c_str());//TODO change!
    //mml::Monitors monitors;
    //monitors.monitor().push_back(displacement);
    //mmlIn.monitors(monitors);
    // add lml
    mmlIn.lmlFile(QFileInfo(lmlFilename).fileName().toStdString().c_str());
    
    xml_schema::namespace_infomap map;
    map[""].name = "";
    map[""].schema = "";

    QString mmlFilename = baseFilename + "-model.mml";
    ofstream mmlofs(mmlFilename.toStdString().c_str());
    mml::monitoringIn(mmlofs, mmlIn, map);
    mmlofs.close();
    
    // 9. Automatically generate proper PML from SCN
    // (this is a temporary hack) read the mml file to create the proper pml
    // this hack should become a call to a static method to transform a sofa scn to a pml
    MonitoringManager *mml = MonitoringManagerFactory::createManager(mmlFilename.toStdString().c_str());

    // 9.1 Modify mml to use pml file instead of sofa
    mmlIn.simulatorFile().reset();
    mmlIn.pmlFile(pmlFilename.toStdString().c_str());
    // reopen for rewrite
    mmlofs.open(mmlFilename.toStdString().c_str(), ios_base::trunc);
    mml::monitoringIn(mmlofs, mmlIn, map);
    mmlofs.close();
    
    // 9.2 Modify the pml to have a visualize a nice cut
    QString completePmlFilename = baseFilename + "-model.pml";
    PhysicalModel *pml = new PhysicalModel(completePmlFilename.toStdString().c_str());

    // add the informative components containing the fixed dof
    StructuralComponent *fixedDof = new StructuralComponent(pml,"Fixed Dof");
    for (pos = dofIndex.begin(); pos != dofIndex.end(); ++pos)
        fixedDof->addStructure(pml->getAtom(*pos));
    pml->getInformativeComponents()->addSubComponent(fixedDof);

    // add the informative components for the nice cut (50%)
    lowestPercentage = 0.65;
    StructuralComponent *niceCut = new StructuralComponent(pml,"Lowest 65% elements");
    StructuralComponent *tetra = dynamic_cast<StructuralComponent*>(pml->getComponentByName("Tetras-CamiTKModel"));
    if (tetra) {
        for (unsigned int i=0; i<tetra->getNumberOfCells(); i++) {
            Cell *c = tetra->getCell(i);
            bool allUnder = true;
            unsigned int j=0;
            while (j<c->getNumberOfStructures() && allUnder) {
                Atom * a = dynamic_cast<Atom *>(c->getStructure(j));
                a->getPosition(xyz);
                allUnder = (xyz[0] < bounds[0] + (bounds[1] - bounds[0])*lowestPercentage);
                j++;
            }
            if (allUnder)
                niceCut->addStructure(c);
        }
        pml->getInformativeComponents()->addSubComponent(niceCut);
    }

    // reopen for rewrite
    ofstream pmlofs(completePmlFilename.toStdString().c_str(), ios_base::trunc);
    pml->xmlPrint(pmlofs);
    pmlofs.close();

    //CAMITK_INFO("GenerateModel", "apply", "parsing " << mmlFilename.toStdString());
    Application::getMainWindow()->open(mmlFilename);

    // restore the normal cursor
    QApplication::restoreOverrideCursor();

    return SUCCESS;
}
