##
# Copyright (c) 2005-2011 Apple Inc. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
##

from twext.web2 import responsecode
from twext.web2.dav import davxml
from twext.web2.dav.util import davXMLFromStream
from twext.web2.http import HTTPError
from twext.web2.iweb import IResponse
from twext.web2.stream import MemoryStream
from twext.web2.test.test_server import SimpleRequest

from twisted.internet.defer import inlineCallbacks

from twistedcaldav import caldavxml
from twistedcaldav.test.util import HomeTestCase, TestCase

class Properties (HomeTestCase):
    """
    CalDAV properties
    """
    def test_free_busy_set_prop(self):
        """
        Test for PROPFIND on Inbox with missing calendar-free-busy-set property.
        """

        inbox_uri  = "/inbox/"

        def propfind_cb(response):
            response = IResponse(response)

            if response.code != responsecode.MULTI_STATUS:
                self.fail("Incorrect response to PROPFIND: %s" % (response.code,))

            def got_xml(doc):
                if not isinstance(doc.root_element, davxml.MultiStatus):
                    self.fail("PROPFIND response XML root element is not multistatus: %r" % (doc.root_element,))

                response = doc.root_element.childOfType(davxml.Response)
                href = response.childOfType(davxml.HRef)
                self.failUnless(str(href) == inbox_uri)

                for propstat in response.childrenOfType(davxml.PropertyStatus):
                    status = propstat.childOfType(davxml.Status)
                    if status.code != responsecode.OK:
                        self.fail("Unable to read requested properties (%s): %r"
                                  % (status, propstat.childOfType(davxml.PropertyContainer).toxml()))

                container = propstat.childOfType(davxml.PropertyContainer)

                #
                # Check CalDAV:calendar-free-busy-set
                #

                free_busy_set = container.childOfType(caldavxml.CalendarFreeBusySet)
                if not free_busy_set:
                    self.fail("Expected CalDAV:calendar-free-busy-set element; but got none.")

                if not free_busy_set.children:
                    self.fail("Expected non-empty CalDAV:calendar-free-busy-set element.")

            return davXMLFromStream(response.stream).addCallback(got_xml)

        query = davxml.PropertyFind(
                    davxml.PropertyContainer(
                        caldavxml.CalendarFreeBusySet(),
                    ),
                )

        request = SimpleRequest(self.site, "PROPFIND", inbox_uri)
        request.stream = MemoryStream(query.toxml())
        return self.send(request, propfind_cb)

class DefaultCalendar (TestCase):

    def setUp(self):
        super(DefaultCalendar, self).setUp()
        self.createStockDirectoryService()
        self.setupCalendars()

    @inlineCallbacks
    def test_pick_default_calendar(self):
        """
        Make calendar
        """
        

        request = SimpleRequest(self.site, "GET", "/calendars/users/wsanchez/")
        inbox = yield request.locateResource("/calendars/users/wsanchez/inbox")

        # default property initially not present
        try:
            inbox.readDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        except HTTPError:
            pass
        else:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not empty")

        yield inbox.pickNewDefaultCalendar(request)

        try:
            default = inbox.readDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        except HTTPError:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not present")
        else:
            self.assertEqual(str(default.children[0]), "/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/calendar")

        request._newStoreTransaction.abort()

    @inlineCallbacks
    def test_pick_default_other(self):
        """
        Make calendar
        """
        

        request = SimpleRequest(self.site, "GET", "/calendars/users/wsanchez/")
        inbox = yield request.locateResource("/calendars/users/wsanchez/inbox")

        # default property not present
        try:
            inbox.readDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        except HTTPError:
            pass
        else:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not empty")

        # Create a new default calendar
        newcalendar = yield request.locateResource("/calendars/users/wsanchez/newcalendar")
        yield newcalendar.createCalendarCollection()
        inbox.writeDeadProperty(caldavxml.ScheduleDefaultCalendarURL(
            davxml.HRef("/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/newcalendar")
        ))
        
        # Delete the normal calendar
        calendar = yield request.locateResource("/calendars/users/wsanchez/calendar")
        yield calendar.storeRemove(request, False, "/calendars/users/wsanchez/calendar")

        inbox.removeDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        
        # default property not present
        try:
            inbox.readDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        except HTTPError:
            pass
        else:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not empty")
        request._newStoreTransaction.commit()

        request = SimpleRequest(self.site, "GET", "/calendars/users/wsanchez/")
        inbox = yield request.locateResource("/calendars/users/wsanchez/inbox")
        yield inbox.pickNewDefaultCalendar(request)

        try:
            default = inbox.readDeadProperty(caldavxml.ScheduleDefaultCalendarURL)
        except HTTPError:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not present")
        else:
            self.assertEqual(str(default.children[0]), "/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/newcalendar")

        request._newStoreTransaction.abort()

    @inlineCallbacks
    def test_fix_shared_default(self):
        """
        Make calendar
        """
        

        request = SimpleRequest(self.site, "GET", "/calendars/users/wsanchez/")
        inbox = yield request.locateResource("/calendars/users/wsanchez/inbox")

        # Create a new default calendar
        newcalendar = yield request.locateResource("/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/newcalendar")
        yield newcalendar.createCalendarCollection()
        inbox.writeDeadProperty(caldavxml.ScheduleDefaultCalendarURL(
            davxml.HRef("/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/newcalendar")
        ))
        try:
            default = yield inbox.readProperty(caldavxml.ScheduleDefaultCalendarURL, request)
        except HTTPError:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not present")
        else:
            self.assertEqual(str(default.children[0]), "/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/newcalendar")
        
        # Force the new calendar to think it is a virtual share
        newcalendar._isVirtualShare = True
        
        try:
            default = yield inbox.readProperty(caldavxml.ScheduleDefaultCalendarURL, request)
        except HTTPError:
            self.fail("caldavxml.ScheduleDefaultCalendarURL is not present")
        else:
            self.assertEqual(str(default.children[0]), "/calendars/__uids__/6423F94A-6B76-4A3A-815B-D52CFD77935D/calendar")

        request._newStoreTransaction.abort()
