/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "kozo-entry-private.h"
#include "kozo-field-private.h"

/* Tweakable parameters */

/* Amount of entry structures per memory block */
#define BLOCK_SIZE 16

static GMutex *mem_lock;
static GMemChunk *mem_chunk;
static GTrashStack *trash_stack = NULL;

KozoEntry *
kozo_entry_new (guint8 *data,
                guint32 length)
{
        KozoEntry *entry;

        g_mutex_lock (mem_lock);

        entry = g_trash_stack_pop (&trash_stack);
        if (!entry)
                entry = g_chunk_new (KozoEntry, mem_chunk);

        g_mutex_unlock (mem_lock);

        entry->data = data;
        entry->length = length;

        return entry;
}

void
kozo_entry_free (KozoEntry *entry)
{
        g_free (entry->data);

        g_mutex_lock (mem_lock);

        g_trash_stack_push (&trash_stack, entry);

        g_mutex_unlock (mem_lock);
}

G_INLINE_FUNC guint32
kozo_entry_get_field_offset (KozoEntry *entry,
                             int        field_id)
{
        return *(guint32 *) (entry->data + (field_id * sizeof (guint32)));
}

KozoField *
kozo_entry_get_field (KozoEntry *entry,
                      int        field_id)
{
        guint32 begin, end;

        begin = kozo_entry_get_field_offset (entry, field_id);
        end = kozo_entry_get_field_offset (entry, field_id + 1);

        return kozo_field_new (field_id, FALSE,
                               entry->data + begin,
                               end - begin);
}

const guint8 *
kozo_entry_get_raw_data (KozoEntry *entry,
                         guint32   *length)
{
        *length = entry->length;
        return entry->data;
}

void
kozo_entry_init (void)
{
        mem_lock = g_mutex_new ();

        mem_chunk = g_mem_chunk_create (KozoEntry,
                                        BLOCK_SIZE,
                                        G_ALLOC_ONLY);
}

void
kozo_entry_shutdown (void)
{
        g_mem_chunk_destroy (mem_chunk);

        g_mutex_free (mem_lock);
}

void
kozo_entry_dump (KozoEntry *entry,
                 int        n_fields)
{
        int i;

        for (i = 0; i < n_fields; i++) {
                KozoField *field;
                const guint8 *data;
                guint32 len, j;

                field = kozo_entry_get_field (entry, i);
                data = kozo_field_get_raw_data (field, &len);

                g_print ("[%d] ", i);
                for (j = 0; j < len; j++) {
                        if (g_ascii_isprint (data[j])) {
                                g_print ("%c", data[j]);
                        } else {
                                g_print ("%x", data[j]);
                        }
                }
                g_print ("\n");
        }
}
