<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    globals-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a manager of global variables which     */
/*              might be POSTed or incoming on URL's. This is used      */
/*              when the PHP 'register_globals' flag is set FALSE, and  */
/*              thereby preventing this happening automatically.        */
/*                                                                      */
/*              This class checks the PHP configuration and acts in the */
/*              appropriate way:                                        */
/*                                                                      */
/*              If register_globals is set to "On"                      */
/*                   - this class does nothing                          */
/*              If register_globals set to "Off"                        */
/*                   - the class globalises vars with names which have  */
/*                     been registered with it.                         */
/*                                                                      */
/*              NOTE: The php.ini flag 'track_vars' must be ON for this */
/*                    to work. This is usually on by default.           */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

/**
* A class to handle globalisation of specific variables. This class
* allows us to turn off the problematical Php 'register_variables'
* option in php.ini which allows _anything_ to be accepted into the global
* scope without a by-your-leave and is therefore a security risk. With
* this scheme specifically named variables can be globalised and all
* others will be ignored. If you think of a Php page/script as a function
* and the globals as its parameters, you will see the light.
*
* Use it at the top of a page as follows. We are assuming that the page
* needs to have vars '$mode', and '$auth_code' POSTed, and that it has a
* COOKIE called 'mycookie', and also uses '$HTTP_HOST' from the
* environment:
*
*   $glob = new globals();
*   $glob->register("mode,auth_code", "post");   // Only from POSTs
*   $glob->register("mycookie",       "cookie"); // Only from cookies
*   $glob->globalise();
*
*  Or, if you prefer to do everything in one hit, this variation is
*  also fine, although less exacting with the sources:
*
*   $glob = new globals("mode,auth_code,mycookie", "post,cookie");
*
*    NB: You can call the globalise() method as many times as you
*        like. New vars will be added as you define them each time
*        you call the method, and any existing ones will simply be
*        re-globalised, with no harm done.
*
*  NOTE: For those using the Phplib framework, all of the Web Server
*        variables like $HTTP_HOST, $PHP_SELF etc. are automatically
*        globalised for you, as are the website cookies. To add your
*        own GET/POST vars you should put something like this at the
*        top of each webpage, immediately AFTER you have included your
*        site-webpage.php or application.php:
*
*        $RESPONSE->globalise("myvar1,myvar2", "get,post");
*
*        This will make sure you have access to vars which have been
*        POSTed or GOT via a URL.
*
*  NOTE: Variable naming can incorporate a Perl regular expression.
*        To use this feature, you must begin the variable name with
*        the character "^".
*
*        Eg. if you have a set of vars named as 'myvarNN' where NN is
*        a numeric counter: $myvar1, $myvar2.. etc. then just call the
*        variable "^myvar[0-9]+" in the list. With regular expressions
*        that means "myvar" followed by one or more digits. To do the
*        same thing, but allowing _any_ chars to come after the stem,
*        call it "^myvar.*". NB: take note of the dot "." which means
*        'any char'; the asterisk "*" means 'any number of', so the whole
*         thing means 'a var which begins with "myvar" followed by zero
*        or more other characters. If you have an array type variable
*        being posted to your page, called something like:
*        '^myvar[0-9]+[]', this is ok. The square brackets will be
*        escaped for you in the Perl pattern, so no worries.
*
*        One last thing: when naming your variable as a Perl regular
*        expression don't end it with the dollar "$" end-of-line char
*        since this is added automatically.
* @package core
*/
class globals {
  // Public
  /** Whether the globals registration is active */
  var $active;
  /** Array containing vars to globalise, and their sources */
  var $varnames;

  // Private
  /** @access private */
  var $vname_env;
  /** @access private */
  var $vname_get;
  /** @access private */
  var $vname_post;
  /** @access private */
  var $vname_cookie;
  /** @access private */
  var $vname_server;
  //.....................................................................
  /**
  * Creates the object which contains the vars to globalise and the
  * methods to globalise them. Takes optional parameters which are vars
  * to globalise.
  * @param string $varnames  Comma-delimited list of vars to globalise
  * @param string $sources   Comma-delimited list of sources for vars
  */
  function globals($varnames="", $sources="env,get,post,cookie,server") {
    // If this is defined, then globals are being
    // registered by Php, otherwise we do it..
    global $HTTP_HOST;
    if (isset($HTTP_HOST) && !empty($HTTP_HOST)) {
      $this->active = false;
    }
    else {
      $this->active = true;
      // Get names of globals we will use. These depend on the
      // version of Php, since we won't rely on the old ones
      // sticking around forever..
      if (version_compare(phpversion(), "4.1.0", "ge")) {
        $this->vname_env = "_ENV";
        $this->vname_get = "_GET";
        $this->vname_post = "_POST";
        $this->vname_cookie = "_COOKIE";
        $this->vname_server = "_SERVER";
      }
      else {
        $this->vname_env = "";
        $this->vname_get = "HTTP_GET_VARS";
        $this->vname_post = "HTTP_POST_VARS";
        $this->vname_cookie = "HTTP_COOKIE_VARS";
        $this->vname_server = "";
      }
    }

    // Now, register given vars, and globalise if we
    // have any defined at this point..
    if ($varnames != "") {
      $this->globalise($varnames, $sources);
    }
  } // globals

  //.....................................................................
  /**
  * Register the given comma-delimited list of varnames, These are the
  * names of global variables which can come from 'outside  sources'.
  * Optionally specify the allowed sources for these, with the default
  * being all the usual ones: GET, POST, or COOKIEs.
  * NB: The ordering of these is important, and determines how vars
  * will be overwritten if defined from multiple sources.
  * @param string $varnames  Comma-delimited list of vars to globalise
  * @param string $sources   Comma-delimited list of sources for vars
  */
  function register($varnames, $sources="env,get,post,cookie,server") {
    if ($this->active) {
      $vars = explode(",", $varnames);
      foreach ($vars as $var) {
        $this->varnames[$var] = strtolower($sources);
      }
    }
  } // register

  //.....................................................................
  /**
  * Globalises all variables from all possible sources. This
  * duplicates the effect of Php register_globals = On in the
  * Php configuration .ini file.
  * NB: This always operates in EGPCS order.
  */
  function globalise_all() {
    if ($this->active) {
      // Environment vars..
      $varnames = $this->array_varnames($this->vname_env);
      $this->register($varnames);

      // GET vars
      $varnames = $this->array_varnames($this->vname_get);
      $this->register($varnames);

      // POSTed vars..
      $varnames = $this->array_varnames($this->vname_post);
      $this->register($varnames);

      // COOKIE vars..
      $varnames = $this->array_varnames($this->vname_cookie);
      $this->register($varnames);

      // SERVER vars..
      $varnames = $this->array_varnames($this->vname_server);
      $this->register($varnames);

      // Now globalise all of these..
      $this->globalise();
    }
  } // globalise_all

  //.....................................................................
  /**
  * Globalise the variables which have already been registered. Also
  * accepts varnames and sources as parameters, and registers these if
  * they are given before globalising, thus saving you having to use
  * the register() method if you only have a simple set of vars with
  * one source.
  * @param string $varnames  Comma-delimited list of vars to globalise
  * @param string $sources   Comma-delimited list of sources for vars
  */
  function globalise($varnames="", $sources="env,get,post,cookie,server") {
    if ($this->active) {
      // Firstly, register any variables given..
      if ($varnames != "") {
        $this->register($varnames, $sources);
      }
      // Perform globalisation. Hopefully there won't be any protest
      // marches against this variety.
      if (isset($this->varnames) && count($this->varnames > 0)) {
        foreach($this->varnames as $var => $srclist) {
          $sources = explode(",", $srclist);
          foreach($sources as $source) {
            switch ($source) {
              case "env":
                $this->create_global($var, $this->vname_env);
                break;
              case "get":
                $this->create_global($var, $this->vname_get);
                break;
              case "post":
                $this->create_global($var, $this->vname_post);
                break;
              case "cookie":
                $this->create_global($var, $this->vname_cookie);
                break;
              case "server":
                $this->create_global($var, $this->vname_server);
                break;
            } // switch
          } // foreach
        } // while
      } // if got varnames
    } // if active

  } // globalise

  //.....................................................................
  /**
  * Gets all the varnames from the given array and returns them as
  * a comma delimited string.
  * This is an internal method.
  * @param string $globarrname Name of global array to use as source
  * @access private
  */
  function array_varnames($globarrname) {
    $s = "";
    global $$globarrname;
    $arr = $$globarrname;
    if (isset($arr) && is_array($arr) && count($arr) > 0) {
      while (list($varname, $val) = each($arr)) {
        if ($s != "") $s .= ",";
        $s .= "$varname";
      }
    }
    return $s;
  } // array_varnames

  //.....................................................................
  /**
  * Find variables in the given source array with a pattern.
  * The pattern may just be a straightforward name, or a
  * regular expression (Perl syntax).
  * Returns a comma-delimited list of variable names which
  * match the pattern in the given array.
  * This is an internal method.
  * @param string $varname  Var name, can be a Perl-compatible regex
  * @param string $srcname  Name of source array holding variables
  * @access private
  */
  function find_varnames($varname, $srcname) {
    $names = array();
    if ($varname != "" && $srcname != "") {
      global $$srcname;
      $arr = $$srcname;
      if (is_array($arr) && count($arr) > 0) {
        if (substr($varname, 0, 1) == "^") {
          // Regular expression varname to match..
          $patt = "/" . $varname . "$/";
          str_replace("[]", "\[\]", $patt);
          foreach ($arr as $varname => $val) {
            if (preg_match($patt, $varname)) {
              $names[$varname] = $val;
            }
          }
        }
        else {
          // Just a bare varname..
          if (isset($arr[$varname])) {
            $names[$varname] = $arr[$varname];
          }
        }
      }
    }
    return $names;
  } // find_varnames

  //.....................................................................
  /**
  * Create a global variable and assign it the given value.
  * If the variable already exists, then it is overwritten.
  * This is an internal method.
  * @param array  $varname  The var name or pattern matching many varnames
  * @param string $srcname  The name of the source array to find vars in
  * @access private
  */
  function create_global($varname, $srcname) {
    if ($varname != "" && $srcname != "") {
      $names = $this->find_varnames($varname, $srcname);
      foreach ($names as $varname => $val) {
        global $$srcname;
        global $$varname;
        if (!isset($$varname)) {
          $srcarray = $$srcname;
          // Only assign it if it exists in the source array..
          if (isset($srcarray[$varname])) {
            $$varname = $val;
            debugbr("GLOBALISED: $varname = '" . $val . "'", DBG_DEBUG);
          }
        }
        else {
          debugbr("ALREADY GLOBAL: $varname", DBG_DEBUG);
        }
      } // foreach
    }
  } // create global

} // globals class

// ----------------------------------------------------------------------
?>