#!/bin/bash
# Install a new Axyl website instance
# This script will also make sure certain basics are installed
# such as the /var/lib/axyl area with the Lucene system.
# Paul Waite
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
set -e

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ] ; then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# This script requires that Axyl is properly installed. If it isn't
# then the following vars will not be defined, which is an error.
if [ "$AXUSER" = "" -o "$AXYL_DATA" = "" ] ; then
  echo "ERROR: Axyl is not configured properly and one of AXUSER, and/or"
  echo "AXYL_DATA are undefined. Check /etc/axyl/axyl.conf exists and"
  echo "contains these settings. Otherwise, do you need to run script"
  echo "./setup-axyl.sh first?"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# PARAMETERS & ANNOUNCE
# This scripts can be invoked in two ways:
# 1) No parameters - 'empty' install type, all values provided by Q&A
# 2) Installation name or '__axyl__demo__' provided as $1

INSTALLTYPE=empty
if [ "$1" != "" ] ; then
  if [ "$1" = "__axyl__demo__" ] ; then
    # Call to install the Axyl demo website..
    if [ -f ${AXYL_HOME}/db/postgres/axyl_demo.db ] ; then
      INSTALLTYPE=demo
    else
      echo "error: install axyl-doc package for demo installation"
      exit 1
    fi
  else
    # Assume to be installation name..
    INST_NAME=$1
  fi
fi

echo ""
echo -n "AXYL "
if [ "$AXYL_VERSION" != "unknown" ] ; then
  echo -n "$AXYL_VERSION "
fi
if [ "$INSTALLTYPE" = "empty" ] ; then
  echo "- NEW WEBSITE CREATION"
else
  echo "- DEMO WEBSITE INSTALLATION"
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# USERS AND GROUPS
# Set up the required users and groups

# The Axyl user..
mkuser $AXUSER system default any default default "Axyl Administrator"

# Webserver user and group..
WWWUSER=$APACHE_USER
WWWGROUP=$APACHE_GROUP
WWWADMIN=wwwadmin
mkuser $WWWUSER system $WWWGROUP any /var/www default "WWW User"

# Web hosting administration group..
mkgroup $WWWADMIN any

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# AXYL INSTALLATION LOCATION
if [ -z $AXYL_HOME ] ; then
  AXYL_HOME=/usr/lib/axyl
  echo ""
  echo "Axyl Location"
  echo "Enter the full path of your main Axyl distribution.."
  echo -n "[$AXYL_HOME] :"
  read ANS
  [ "$ANS" != "" ] && AXYL_HOME=$ANS
else
  echo "Installing from Axyl installation at '$AXYL_HOME'"
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INSTALLATION NAME
if [ -z $INST_NAME ] ; then
  echo ""
  if [ "$INSTALLTYPE" = "demo" ] ; then
    INST_NAME=axyl_demo
    echo "Axyl Demo Website Installation Name"
    echo "Enter a name for this Axyl demo website installation. The name "
    echo "will be used as a default for choice of Postgres database "
    echo "name, as well as various directory names, so make sure it is "
    echo "unique, or the installation may fail. Usually the best indication"
    echo "of uniqueness is just to check whether there is a Postgres database"
    echo "already in existence with the same name."
    echo ""
    echo -n "Demo Axyl website installation name (no spaces or hyphens) :"
  else
    INST_NAME=
    echo "New Website Installation Name"
    echo "Enter a name for this new Axyl website installation. The name "
    echo "will be used as a default for choice of Postgres database "
    echo "name, as well as various directory names, so make sure it is "
    echo "unique, or the installation may fail. Usually the best indication"
    echo "of uniqueness is just to check whether there is a Postgres database"
    echo "already in existence with the same name."
    echo ""
    echo -n "New Axyl website installation name (no spaces or hyphens) :"
  fi
  read ANS
  if [ "$ANS" != "" ] ; then
    INST_NAME=$ANS
  else
    echo "A name is mandatory. Aborting.."
    exit 1
  fi
else
  echo "Installing Axyl website '$INST_NAME'"
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# AXYL VAR DATA AREA
# Tha main usage of this area is by the Luceneserver which comes
# with Axyl, and stores Lucene configs, and indexes here. The
# Lucene indexes can get big, so symlink as necessary to manage
# that aspect of the operation.
if [ -z $AXYL_DATA ] ; then
  AXYL_DATA=/var/lib/axyl
  echo ""
  echo "Axyl Data Area"
  echo "Enter the full path of the Axyl data area.."
  echo -n "[$AXYL_DATA] :"
  read ANS
  [ "$ANS" != "" ] && AXYL_DATA=$ANS
else
  echo "Axyl data is located in '$AXYL_DATA'"
fi

# Create directory and set perms..
mkthisdir $AXYL_DATA
chown -R ${AXUSER}:${AXUSER} ${AXYL_DATA}
chmod -R 0644 ${AXYL_DATA}
find ${AXYL_DATA} -type d -exec chmod 6755 {} \;

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# AXYL LOGGING AREA
if [ -d /var/log ] ; then
  AXYL_LOGS=/var/log/axyl
else
  AXYL_LOGS=${AXYL_DATA}/log
fi

# Create directory and set perms..
mkthisdir $AXYL_LOGS
chown -R ${AXUSER}:${AXUSER} ${AXYL_LOGS}
find ${AXYL_LOGS} -type f -exec chmod 6755 {} \;
find ${AXYL_LOGS} -type d -exec chmod 6755 {} \;

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# DATABASE
# Pick a database type..
DBTYPE=1
echo ""
echo "Database Configuration"
echo "Pick a database type:"
echo "     1 - Postgresql"
echo "     2 - MySQL"
echo "     3 - ODBC"
echo "     4 - Oracle"
echo "     5 - MS SQL Server"
echo "     6 - Other"
echo -n "Database type: [$DBTYPE] :"
read ANS
[ "$ANS" != "" ] && DBTYPE=$ANS

# Choose a database name..
DBNAME=$INST_NAME
echo ""
echo "Enter a name for this new database. Make sure that this"
echo "does not already exist, or the installation will fail."
echo ""
echo -n "Enter Axyl database name [$DBNAME] :"
read ANS
[ "$ANS" != "" ] && DBNAME=$ANS

# Username
# We try to select an appropriate default for this..
DBUSER="`who -m | cut -d' ' -f1`"
[ -z $DBUSER ] && DBUSER=$AXUSER
echo ""
echo "Enter the username which the application will connect to the database"
echo "as. Since this has to be a valid machine account, if it doesn't exist"
echo "then it will be created for you."
echo -n "Database connect username: [$DBUSER] :"
read ANS
[ "$ANS" != "" ] && DBUSER=$ANS

# Database user password..
PASSOK=0
while [ $PASSOK -eq 0 ] ; do
  DBPASSWD="none"
  echo ""
  echo "Enter a password for this user or just hit ENTER to indicate that no"
  echo "password is required."
  echo -n "Database connect password: [$DBPASSWD] :"
  read ANS
  if [ "$ANS" != "" ] ; then
    DBPASSWD=$ANS
    echo -n "Confirm password :"
    read ANS
    if [ "$ANS" = "$DBPASSWD" ] ; then
      PASSOK=1
    else
      echo ""
      echo "Passwords don't match! Try again please.."
    fi
  else
    PASSOK=1
  fi
done

# Host/port..
DBHOST=direct
echo ""
echo "If your database server is on a remote host, enter the hostname here,"
echo "or just hit ENTER if the connection is to be a direct one to a server"
echo "on the local machine"
echo -n "Remote database server hostname: [$DBHOST] :"
read ANS
[ "$ANS" != "" ] && DBHOST=$ANS

DBPORT=5432
if [ "$DBHOST" != "direct" ] ; then
  echo ""
  echo "Enter the port that the remote database server is listening on. For"
  echo "Postgresql, the default port of $DBPORT will probably be right, unless"
  echo "you are running a multi-version cluster."
  echo -n "Remote database server port: [$DBPORT] :"
  read ANS
  [ "$ANS" != "" ] && DBPORT=$ANS
fi

# Axyl Metadata Extension
META=n
if [ "$INSTALLTYPE" = "empty" ] ; then
  echo ""
  echo "Axyl can be created with fully-defined Meta Data Schemas, namely"
  echo "the Dublin Core, AGLS, and NZGLS definitions. This is appropriate if"
  echo "you are going to be creating a website which requires meta data"
  echo "content categorisation to any of these standards (eg. a Government"
  echo "website). If you want this to be installed, say so now."
  echo -n "Install Axyl Metadata Extension (y/n) [${META}] :"
  read ANS
  [ "$ANS" = "y" -o "$ANS" = "Y" ] && META=y
fi

# Axyl Microsites Extension
MICRO=n
if [ "$INSTALLTYPE" = "empty" ] ; then
  echo ""
  echo "Axyl can be created with the ability to create and manage 'microsites'"
  echo "which make use of Axyl themeing. You can create simple content-managed"
  echo "sub-sites within your single Axyl website via a web-interface if you"
  echo "enable this extension. If you want this to be installed, say so now."
  echo -n "Install Axyl Microsite Extension (y/n) [${MICRO}] :"
  read ANS
  [ "$ANS" = "y" -o "$ANS" = "Y" ] && MICRO=y
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# CREATE THE AXYL WEBSITE ROOT
WWW_PATH=/var/www/$INST_NAME
echo ""
echo "Enter the full path of the target website to create"
echo -n "[$WWW_PATH] :"
read ANS
[ "$ANS" != "" ] && WWW_PATH=$ANS

# Create the full destination path..
mkthisdir $WWW_PATH

# Copy across initial content..
dupedir ${AXYL_HOME}/www ${WWW_PATH}

# Axyl library images and javascript. These need
# to be actually in the website hiearchy..
mkthisdir ${WWW_PATH}/lib
if [ ! -L ${WWW_PATH}/lib/img ] ; then
  ln -s ${AXYL_HOME}/lib/img ${WWW_PATH}/lib/img
fi
if [ ! -L ${WWW_PATH}/lib/js ] ; then
  ln -s ${AXYL_HOME}/lib/js ${WWW_PATH}/lib/js
fi

# If non-Postgres installation, then we have to replace some of the
# maintenance scripts with ones which are compatible with other
# databases. The standard scripts use the Axyl maintainer-defs.php
# module which is currently not DB-independent.
if [ $DBTYPE -ne 1 ] ; then
  cp ${AXYL_HOME}/compat/www/*.php ${WWW_PATH}
fi

# If demo, then copy across the demo content. This includes
# the managed-content pages, and the catalog images which are
# referred-to by the demo database content layouts.
if [ "$INSTALLTYPE" = "demo" ] ; then
  find ${AXYL_HOME}/demo/var/catalog -type f -exec cp {} ${WWW_PATH}/var/catalog/ \;
  find ${AXYL_HOME}/demo/var/cm -type f -exec cp {} ${WWW_PATH}/var/cm/ \;
fi

# PERMISSIONS
# Set the ownerships and permissions on all files and dirs..

# Generally writeable only for axyl and WWW administrators..
chown -R ${AXUSER}:${WWWADMIN} ${WWW_PATH}
chmod -R 0664 ${WWW_PATH}
chmod -R g+sw ${WWW_PATH}
find ${WWW_PATH} -type d -exec chmod a+x {} \;

# Except var directory is writeable to webserver too..
chgrp -R ${WWWUSER} ${WWW_PATH}/var

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INSTALL DATABASE
# At this point we have everything we need for this task..
${AXYL_HOME}/install/install-db.sh $INSTALLTYPE $DBTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META $MICRO $AXYL_VERSION $INST_NAME $WWW_PATH

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# APACHE CONFIGURATION
echo ""
echo "Apache Configuration"
echo "Do you want to create a virtual host for apache?"
echo -n "Enter Y or N [Y] :"
read ANS
[ "$ANS" = "" ] && ANS="y"
if [ "$ANS" = "y" -o "$ANS" = "Y" -o "$ANS" = "yes" ] ; then
  CONFIG_APACHE=yes
  CONF=${APACHE_CONFDIR}/httpd.conf
  TMP=/tmp/$$

  echo ""
  echo "Ok, you now have to specify the IP address, or hostname of the"
  echo "server that will be serving your website. Apache allows this to"
  echo "be 'loose', as in '*:80' which indicates any IP address, on"
  echo "port 80 (the default below)."
  echo ""
  
  # Display current NameVirtualHost settings if any..
  A=`perl -n -e "m;^[\s]*NameVirtualHost;i && s;NameVirtualHost ;;i && print;" $CONF`
  if [ "$A" != "" ] ; then
    echo "Your installation has some NameVirtualHost settings, apparently, and"
    echo "these are listed here for a source of answers to the below."
    echo "Your current NameVirtualHosts: $A"
    echo ""
  fi
  
  DISP="Everything on Port 80"
  VHOST="*:80"
  set -f
  echo -n "Enter IP or hostname for this virtual host [${DISP}] :"
  read ANS
  [ "$ANS" != "" ] && VHOST=$ANS
  set +f

  # Server name
  SERVERNAME=
  VSERVERNAME=
  A=`perl -n -e "m;^[\s]*ServerName;i && print;" $CONF`
  if [ "$A" != "" ] ; then
    for bit in $A ; do
      SERVERNAME=$bit
    done
    host="`echo $SERVERNAME | cut -d'.' -f1`"
    if [ "$host" != "" ] ; then
      VSERVERNAME="`echo $SERVERNAME | sed -e \"s/${host}/${DBNAME}/\"`"
    fi
  fi
  if [ "$VSERVERNAME" = "" ] ; then
    VSERVERNAME=$DBNAME
  fi
  echo -n "Enter the ServerName [${VSERVERNAME}] :"
  read ANS
  [ "$ANS" != "" ] && VSERVERNAME=$ANS

  # Admin e-mail address
  ADMIN_EMAIL="webmaster@"
  ADMIN_EMAIL="${ADMIN_EMAIL}${VSERVERNAME}"
  echo -n "Enter the admin e-mail address [${ADMIN_EMAIL}] :"
  read ANS
  [ "$ANS" != "" ] && ADMIN_EMAIL=$ANS

  # Document root
  OLDPWD=`pwd`
  cd ../..
  DOCROOT=$WWW_PATH
  cd $OLDPWD
  echo -n "Enter the document root [${DOCROOT}] :"
  read ANS
  [ "$ANS" != "" ] && DOCROOT=$ANS

  # Memory limit
  MAXMEM=32
  echo -n "Maximum memory (in Mb) that PHP should use [${MAXMEM}] :"
  read ANS
  [ "$ANS" != "" ] && MAXMEM=$ANS

  # Max upload filesize
  MAXUPLOAD=5
  echo -n "Maximum size of an uploaded file (in Mb) [${MAXUPLOAD}] :"
  read ANS
  [ "$ANS" != "" ] && MAXUPLOAD=$ANS
else
  CONFIG_APACHE=no
  VHOST=none
  VSERVERNAME=none
  ADMIN_EMAIL=none
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# LUCENE SERVER DOMAIN
# This is different from the Lucene system install above. This is simply
# concerned with the creation of a domain for the new website on the
# standard Axyl Luceneserver. A single Luceneserver can service numerous
# domains (websites).
# We only do anything if the axyl-lucene package is installed...
if [ -e ${AXYL_HOME}/lucene/servers ] ; then
  if [ ! -f ${AXYL_DATA}/lucene/servers/axyl/etc/application/${INST_NAME}.properties ] ; then
    echo ""
    echo "LuceneServer Domain"
    echo "Even if you have no large-scale document indexing requirement"
    echo "it is recommended you set up a Luceneserver Domain here so that"
    echo "the Axyl content management system can index website content"
    echo "which you can then use to provide user search facilities on your"
    echo "new website."
    echo ""
    echo -n "Set up a Luceneserver Domain for $INST_NAME? Y or N [Y] :"
    read ANS
    [ "$ANS" = "" ] && ANS="y"
    if [ "$ANS" = "y" -o "$ANS" = "Y" -o "$ANS" = "yes" ] ; then
      PROPS=${AXYL_DATA}/lucene/servers/axyl/etc/application/${INST_NAME}.properties
      cp ${AXYL_HOME}/lucene/servers/axyl/etc/application/axyl.properties ${PROPS}
      sed -e "s;_LU_DOMAIN_;$INST_NAME;g" ${PROPS} > $$.tmp && mv $$.tmp ${PROPS}
      echo "Lucene domain for $INST_NAME installed."
      # Make sure Axyl lucene started..
      if [ -x /etc/init.d/axyl-lucene ] ; then
        /etc/init.d/axyl-lucene start
      fi
    fi
  else
    echo "A Lucene domain for $INST_NAME is already installed."
  fi

  # PERMISSIONS
  # Set the ownerships and permissions..
  chown -R ${AXUSER}:${AXUSER} ${AXYL_DATA}/lucene/servers
  chmod -R 0664 ${AXYL_DATA}/lucene/servers
  find ${AXYL_DATA}/lucene -type d -exec chmod 6775 {} \;
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# AXYL APPLICATION CONFIGURATION
echo ""
echo "setting up website application config file (application.xml).."
${AXYL_HOME}/install/config-app-db.pl $INST_NAME $WWW_PATH $DBTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META $MICRO
# Fix up perms..
chown ${AXUSER}:${WWWADMIN} ${WWW_PATH}/application.xml
chmod 0664 ${WWW_PATH}/application.xml

# Customise Apache if required..
echo ""
if [ "$CONFIG_APACHE" = "yes" ] ; then
  echo "Configuring Apache"
  set -f
  ${AXYL_HOME}/install/config-apache.sh $VHOST $VSERVERNAME $ADMIN_EMAIL $DOCROOT $MAXMEM $MAXUPLOAD
  set +f
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Update Axyl websites database
echo ""
echo "updating Axyl website database.."
${AXYL_HOME}/install/register-axyl-website.sh $INST_NAME $DBTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $DOCROOT $VHOST $VSERVERNAME $ADMIN_EMAIL

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# PERMISSIONS
# We have already set these above, but just to ensure consistency we
# pass the 'official' permissions setting script over the website..
cd $WWW_PATH
${AXYL_HOME}/install/set-axyl-website-perms.sh

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# FINISHED
echo ""
echo "Axyl has been installed."
if [ "$DBTYPE" -eq 1 ] ; then
  echo "The Postgres database '$DBNAME' has been created, owned by '$DBUSER'"
  echo -n "and populated with "
  if [ "$INSTALLTYPE" = "demo" ] ; then
    echo "all data required for the demo website."
  else
    echo "only the basic Axyl core data, ready for you"
    echo "to develop your own app. The intention is that you will develop"
    echo "custom schema, and/or extend the existing Axyl schema as you go"
    echo "forward. Future enhancements of Axyl will involve schema changes"
    echo "as well as code, but you can still choose to take advantage of"
    echo "these at any time by using the 'dbdiff.php' utility to be found"
    echo "in the 'scripts' directory. In fact, Axyl upgrades will also"
    echo "advise you how to use the 'axyl-databases-upgrade.php' script to"
    echo "bring all of your Axyl websites up to current spec."
  fi
else
  echo "Please ensure that your database is set up such that user '$DBUSER'"
  echo "can connect to it."
fi

echo ""
if [ "$CONFIG_APACHE" = "yes" ] ; then
  echo "Apache was configured with server name '${VSERVERNAME}'"
  echo "however you will now have to set up your DNS or /etc/hosts file"
  echo "appropriately to resolve this for visiting web clients."
fi

echo ""
echo "The Axyl configuration is in your 'application.xml' in the website"
echo "root directory '${WWW_PATH}'. To maintain this the"
echo "recommended approach is to change it using the GUI maint script"
echo "lib/control-panel.php, by soft-linking it up into the website root"
echo "and then pointing your browser at it. If you do this, then make sure"
echo "'application.xml' is writeable to the webserver, so you can make the"
echo "necessary changes."
echo ""
echo "However, this should be only ever be done on a developer copy of the"
echo "website, and not on the Live version. Once you have it as required,"
echo "you can then copy it over the top of your production copy, which is"
echo "always kept as non-writeable to the web-server."
echo ""
echo "The Axyl Team"

# END