<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    control-panel.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Axyl control panel                                      */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

/** Views of control panel available */

/** Standard (default) view */
define("CP_VIEW_DEFAULT", 0);
/** View of authentication & security settings */
define("CP_VIEW_AUTH",    1);
/** View of database settings */
define("CP_VIEW_DB",      2);
/** View of debugging settings */
define("CP_VIEW_DEBUG",   3);

/** User data is maintained on the local database in Axyl format
 * This is the default. */
define("LOCAL_AUTH", 0);
/** User data is maintained on a remote database */
define("REMOTE_AUTH_REMOTEDB", 1);
/** User data maintained on an LDAP server (not yet implemented) */
define("REMOTE_AUTH_LDAP", 2);
/** Used to indicate items do not have a remote mapping */
define("NOT_MAPPED", "");


$CPTABS = array(
  CP_VIEW_DEFAULT => "Main",
  CP_VIEW_AUTH    => "Authentication & Security",
  CP_VIEW_DB      => "Database",
  CP_VIEW_DEBUG   => "Debugging"
);

// Initialise view if required..
if (!isset($cp_view)) {
  $cp_view = CP_VIEW_DEFAULT;
}

// Local library has Axyl images..
$LIBDIR = "/lib";

// Axyl installation settings..
$AXYL_HOME = "";
$AXYL_CONF = "/etc/axyl/axyl.conf";
if (file_exists($AXYL_CONF)) {
  $result = exec("grep \"AXYL_HOME=\" $AXYL_CONF");
  if ($result != "") {
    $bits = explode("=", $result);
    if (is_dir($bits[1])) {
      $AXYL_HOME = $bits[1];
    }
  }
}

// Name of our master form..
$formname = "cpform";

// These are the candidate Axyl fields for remote
// authentication mapping..      
$REMOTE_AUTH_FIELDNAMES = array(
      "user_id",
      "password",
      "full_name",
      "honorific_prefix",
      "first_name",
      "mid_names",
      "last_name",
      "email",
      "address",
      "phone",
      "fax",
      "mobile"
      );

// ----------------------------------------------------------------------
// Include required modules..
/** Sundry contants & defs */
require("constants.php");
/** Renderable module defs */
require("renderable.php");
/** Form handling */
require("form-defs.php");
/** Utilities */
require("utils.php");
/** Debugger defs */
require("debugger.php");
/** Record maintainer module */
require("recmaint-defs.php");
/** Application setup */
require("application-defs.php");

// ----------------------------------------------------------------------
// FUNCTIONS
/**
* Determine the index of Nth database entry..
* @access private
*/
function getdbindex($Nth) {
  global $app;
  $dbix = -1; $dbpos = 0;
  for ($ix = 0; $ix < count($app->settings); $ix++) {
    $setting = $app->settings[$ix];
    if ($setting->name == "database") {
      if ($dbpos == $Nth) {
        $dbix = $ix;
        break;
      }
      else {
        $dbpos += 1;
      }
    }
  }
  return $dbix;
}
/**
* Determine the index of last database entry..
* @access private
*/
function getlastdbindex() {
  global $app;
  $dbix = -1;
  for ($ix = 0; $ix < count($app->settings); $ix++) {
    $setting = $app->settings[$ix];
    if ($setting->name == "database") {
      $dbix = $ix;
    }
  }
  return $dbix;
}
/**
* Delete the Nth database entry. Database entries are numbered
* from zero (first database entry) upwards..
* @access private
*/
function deletedbentry($Nth) {
  global $app;
  $dbix = getdbindex($Nth);
  if ($dbix != -1) {
    $setting = $app->settings[$dbix];
    if ($setting->name == "database") {
      unset($app->settings[$dbix]);
    }
  }
  return $dbix;
}

// ----------------------------------------------------------------------
// CONVERSION OF OLD APPLICATION.PHP FILE TO NEW XML SCHEMA
$error = false;
$user_msg = "";
$appfile = new inputfile("application.php");
if ($appfile->opened) {
  $appfile->readall();
  $appfile->closefile();
  $appstuff = $appfile->content;
  if (strstr($appstuff, "\$TEMPLATESDIR =")) {
    if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
      copy("application.php", "application.php.bak");
      if (file_exists("application.php.bak")) {

        if (is_writeable("application.php")) {
          copy("$AXYL_HOME/lib/application.php", "application.php");
          copy("$AXYL_HOME/lib/default-application.xml", "application.xml");

          if (file_exists("application.xml")) {

            $app = new application("application.xml");
            //echo "converting..<br>";

            // DEFINITIONS
            if (preg_match("/define\(\"APP_NAME\",[\s]*\"(.*?)\"/", $appstuff, $matches)) {
              $app->definitions["APP_NAME"] = $matches[1];
              //echo "setting APP_NAME to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/define\(\"APP_PREFIX\",[\s]*\"(.*?)\"/", $appstuff, $matches)) {
              $app->definitions["APP_PREFIX"] = $matches[1];
              //echo "setting APP_PREFIX to [" . $matches[1] . "]<br>";
            }

            // GLOBALS
            if (preg_match("/^[$]TEMPLATESDIR[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["TEMPLATESDIR"] = $matches[1];
              //echo "setting TEMPLATESDIR to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]IMAGESDIR[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["IMAGESDIR"] = $matches[1];
              //echo "setting IMAGESDIR to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]WEBMASTER_PERSON[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["WEBMASTER_PERSON"] = $matches[1];
              //echo "setting WEBMASTER_PERSON to [" . $matches[1] . "]<br>";
            }
            if (preg_match("/^[$]WEBMASTER_EMAIL[\s]*\=[\s]*\"(.*?)\"/m", $appstuff, $matches)) {
              $app->globals["WEBMASTER_EMAIL"] = $matches[1];
              //echo "setting WEBMASTER_EMAIL to [" . $matches[1] . "]<br>";
            }

            // SETTINGS
            if (preg_match("/^[$]RESPONSE->set_encoding\(\"(.*?)\"\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "encoding", "encoding");
              //echo "setting char encoding to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "char encoding defaulted<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_blocked_ips\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "badips", "badips");
              //echo "setting blocked ips to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "blocked ips defaulted<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_sessiontype\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "SESS_DATABASE_BACKED"), "database_backed", "database_backed");
              //echo "setting database-backed is " . ($matches[1] == "SESS_DATABASE_BACKED" ? "true" : "false") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_lifetime\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "SESS_FOREVER":          $life = 315360000; break;
                case "SESS_1_YEAR":           $life = 31536000;  break;
                case "SESS_1_MONTH":          $life = 2592000;   break;
                case "SESS_1_WEEK":           $life = 604800;    break;
                case "SESS_1_DAY":            $life = 86400;     break;
                case "SESS_12_HOURS":         $life = 43200;     break;
                case "SESS_8_HOURS":          $life = 28800;     break;
                case "SESS_4_HOURS":          $life = 14400;     break;
                case "SESS_1_HOUR":           $life = 3600;      break;
                case "SESS_20_MINS":          $life = 1200;      break;
                case "SESS_BROWSER_LIFETIME": $life = -1;        break;
                case "SESS_ZERO_LIFETIME":    $life = 0;         break;
                default:  $life = -1;
              }
              $app->setparameter($life, "lifetime", "lifetime");
              //echo "setting cookie life to [" . $matches[1] . "($life)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_cookiename\((.*?)\)/m", $appstuff, $matches)) {
              if ($matches[1] != "APP_PREFIX . \"_session_id\"") {
                $app->setparameter($matches[1], "cookiename", "cookiename");
                //echo "setting cookiename to [" . $matches[1] . "]<br>";
              }
              //else {
              //  echo "setting cookiename to default<br>";
              //}
            }

            if (preg_match("/^[$]RESPONSE->set_keep\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "KEEP_ENABLED"), "keep", "keep");
              //echo "setting keep status " . ($matches[1] == "KEEP_ENABLED" ? "ON" : "OFF") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->globalise_all\(\)/m", $appstuff, $matches)) {
              $app->setparameter(true, "globalise", "globalise");
              //echo "setting globalise all ON<br>";
            }
            else {
              $app->setparameter(false, "globalise", "globalise");
              //echo "setting globalise all OFF<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_compression_type\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "NO_COMPRESSION":       $comp = 0; break;
                case "BUILTIN_COMPRESSION":  $comp = 1; break;
                case "CUSTOM_COMPRESSION":   $comp = 2; break;
                default:  $comp = 0;
              }
              $app->setparameter($comp, "compression_type", "compression_type");
              //echo "setting compression type to [" . $matches[1] . "($comp)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_compression_minsize\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "compression_threshold", "compression_threshold");
              //echo "setting compression threshold to [" . $matches[1] . "]<br>";
            }
            else {
              //echo "compression threshold is defaulted (0)<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_buffering_mode\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter(($matches[1] == "BUFFERED"), "buffered_output", "buffered_output");
              //echo "setting buffered output " . ($matches[1] == "BUFFERED" ? "ON" : "OFF") . "<br>";
            }

            if (preg_match("/^[$]RESPONSE->set_page_expirysecs\((.*?)\)/m", $appstuff, $matches)) {
              $app->setparameter($matches[1], "expiry", "expiry");
              //echo "setting page expiry to [" . $matches[1] . "]<br>";
            }
            //else {
            //  echo "compression page expiry is defaulted (-1)<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_authentication_type\((.*?)\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "NO_AUTHENTICATION":    $auth = 0; break;
                case "HTTP_AUTHENTICATION":  $auth = 1; break;
                case "FORM_AUTHENTICATION":  $auth = 2; break;
                default:  $auth = 2;
              }
              $app->setparameter($auth, "authtype", "authtype");
              //echo "setting authentication type to [" . $matches[1] . "($auth)]<br>";
            }

            if (preg_match("/^[$]RESPONSE->on_authentication_fail\((.*?)(,(.*?))*?\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "AUTHFAIL_DIE_MSG":     $authf = 0; break;
                case "AUTHFAIL_DIE_SILENT":  $authf = 1; break;
                case "AUTHFAIL_REDIRECT":    $authf = 2; break;
                case "AUTHFAIL_GUEST":       $authf = 3; break;
                default:  $authf = 0;
              }
              $app->setparameter($authf, "authfail", "authfailopt");
              //echo "setting auth fail option to [" . $matches[1] . "($authf)]<br>";
            }

            if (isset($matches[3])) {
              $authurl = preg_replace("/['\"]/", "", $matches[3]);
              $app->setparameter($authurl, "authfail", "authfailurl");
              //echo "setting auth fail URL to [$authurl]<br>";
            }
            //else {
            //  echo "no URL<br>";
            //}

            if (preg_match("/^[$]RESPONSE->on_logins_exceeded\((.*?)(,(.*?))*?\)/m", $appstuff, $matches)) {
              switch ($matches[1]) {
                case "SESS_ALLOW":           $logexc = 0; break;
                case "SESS_ALLOW_CULL":      $logexc = 1; break;
                case "SESS_BLOCK_MSG":       $logexc = 2; break;
                case "SESS_BLOCK_SILENT":    $logexc = 3; break;
                case "SESS_BLOCK_REDIRECT":  $logexc = 4; break;
                case "SESS_BLOCK_GUEST":     $logexc = 5; break;
                default:  $logexc = 0;
              }
              $app->setparameter($logexc, "loglimit", "logexceedopt");
              //echo "setting logins exceeded option to [" . $matches[1] . "]<br>";
            }

            if (isset($matches[3])) {
              $logexcurl = preg_replace("/['\"]/", "", $matches[3]);
              $app->setparameter($logexcurl, "loglimit", "logexceedurl");
              //echo "setting logins exceeded URL to [$logexcurl]<br>";
            }
            //else {
            //  echo "no URL<br>";
            //}

            if (preg_match("/^[$]RESPONSE->set_persistent_hosts\((.*?)\)/m", $appstuff, $matches)) {
              if ($matches[1] != "\"\"") {
                $app->setparameter($matches[1], "permhosts", "permhosts");
                //echo "setting persistent hosts list to [" . $matches[1] . "]<br>";
              }
              //else {
              //  echo "null persistent hosts list.<br>";
              //}
            }
            //else {
            //  echo "no persistent hosts.<br>";
            //}

            $patt  = "RESPONSE->add_database\(\n";
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // DB type
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // name
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // user
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // password
            $patt .= "[\s]*?\"(.*?)\",.*\n";       // host
            $patt .= "[\s]*?\"(.*?)\".*\n";        // port
            $patt .= "(([\s])*?(DEFAULT_DATASOURCE))*";  // default flag

            // Purge existing database settings..
            $newsettings = array();
            for ($ix=0; $ix < count($app->settings); $ix++) {
              $setting = $app->settings[$ix];
              if ($setting->name != "database") {
                $newsettings[] = $setting;
              }
            }
            $app->settings = $newsettings;

            preg_match_all("/$patt/", $appstuff, $matches);
            for ($i=0; $i < count($matches[0]); $i++) {
              /*
              echo "database defs:<br>";
              echo " type: " . $matches[1][$i] . "<br>";
              echo " name: " . $matches[2][$i] . "<br>";
              echo " user: " . $matches[3][$i] . "<br>";
              echo " pass: " . $matches[4][$i] . "<br>";
              echo " host: " . $matches[5][$i] . "<br>";
              echo " port: " . $matches[6][$i] . "<br>";
              */
              $dbsetting = new setting("database", "add_database");
              $parameter = new parameter("type", "string");
              $parameter->setvalue($matches[1][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("name", "string");
              $parameter->setvalue($matches[2][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("user", "string");
              $parameter->setvalue($matches[3][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("password", "string");
              $parameter->setvalue($matches[4][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("host", "string");
              $parameter->setvalue($matches[5][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);

              $parameter = new parameter("port", "integer");
              $parameter->setvalue($matches[6][$i]);
              $dbsetting->addparameter($parameter->name, $parameter);
              if (isset($matches[9][$i]) && $matches[9][$i] == "DEFAULT_DATASOURCE") {
                $defaultdb = $dbsetting;
                //echo "default DB<br>";
              }
              else {
                $secdbs[] = $dbsetting;
                //echo "secondary DB<br>";
              }
            } // for

            if (isset($defaultdb)) {
              $app->settings[] = $defaultdb;
            }
            if (isset($secdbs)) {
              foreach ($secdbs as $db) {
                $app->settings[] = $db;
              }
            }

            // Save all conversion changes..
            $app->save();
            $user_msg  = "Your application configuration has been converted to XML format. ";
            $user_msg .= "The old file has been saved as 'application.php.bak'. A new file ";
            $user_msg .= "'application.xml' has been created, which should only ever be ";
            $user_msg .= "changed by using this Control Panel.";
          }
          else {
            $error = true;
            $user_msg = "Conversion aborted due to problems creating XML file."
                            . "<br>Please fix & retry.";
          }
        }
        else {
          $error = true;
          $user_msg = "Conversion aborted. File 'application.php must be "
                          . "writeable by webserver.<br>Please fix & retry.";
        }
      } // // application.php.bak exists
      else {
        $error = true;
        $user_msg = "Conversion aborted due to problems backing up data."
                        . "<br>Please fix & retry.";
      }
    } // lib/default-application.xml exists
  } // old format file
} // appfile opened

// ----------------------------------------------------------------------
// FAILSAFE TO DEFAULT
// Failsafe - if no application XML file, copy default into place..
if (!file_exists("application.xml")) {
  if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
    copy("$AXYL_HOME/lib/default-application.xml", "application.xml");
  }
}

// ----------------------------------------------------------------------
// READ XML APPLICATION SETTINGS
// Read in current application..
if (file_exists("application.xml") && is_writeable("application.xml")) {
  $app = new application();
}
else {
  $error = true;
  $user_msg = "Error: Please make 'application.xml' writeable to the webserver.";
}

// ----------------------------------------------------------------------
// SYNCHRONIZE
// Make sure that the current application XML file has all the required
// defs, globals & settings. For this we have to refer to the Axyl HOME
// file default-application.xml, so find Axyl HOME first of all..
if (!$error) {
  if (is_dir($AXYL_HOME)) {
    $defaultapp = new application("$AXYL_HOME/lib/default-application.xml");
    $synced = $app->synchronize($defaultapp);
    if ($synced) {
      $app->save();
      $app = new application();
    }
  }
}

// ----------------------------------------------------------------------
// POST ACTION
// Check if they opted to set things to default..
if (!$error) {
  if (isset($_default_x)) {
    if (file_exists("$AXYL_HOME/lib/default-application.xml")) {
      copy("$AXYL_HOME/lib/default-application.xml", "application.xml");
      $app = new application();
    }
  }
  elseif (isset($_recmaintpost_form) && $_recmaintpost_form == $formname) {
    
    /*
    // DEBUGGING: POSTED VARS DUMP
    $s .= "<table border=1 cellpadding=2 cellspacing=0>";
    if (isset($HTTP_POST_VARS)) {
      $s .= "<tr><td colspan=2><h4>POSTed Vars</h4></td></tr>";
      reset($HTTP_POST_VARS);
      while (list($key, $val) = each($HTTP_POST_VARS)) {
        $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
      }
    }
    $s .= "</table>";
    */    
    
    switch ($cp_view) {
      case CP_VIEW_AUTH:
        // Defaults
        if ($cp_passwd_encryption == "") $cp_passwd_encryption = "md5";
        if ($cp_passwd_expiry_days == "") $cp_passwd_expiry_days = "180";
        if ($cp_passwd_max_attempts == "") $cp_passwd_max_attempts = "0";
        if ($cp_passwd_history_cycle == "") $cp_passwd_history_cycle = "0";
        if ($cp_passwd_delay_ms == "") $cp_passwd_delay_ms = "0";
        if ($cp_passwd_min_chars == "") $cp_passwd_min_chars = "0";
        if ($cp_passwd_char_uniqueness == "") $cp_passwd_char_uniqueness = "low";
    
        $app->setparameter($cp_authtype,                      "authtype", "authtype");
        $app->setparameter($cp_authfailopt,                   "authfail", "authfailopt");
        $app->setparameter($cp_authfailurl,                   "authfail", "authfailurl");
        $app->setparameter($cp_passwd_encryption,             "security_profile", "passwd_encryption");
        $app->setparameter($cp_passwd_expiry_days,            "security_profile", "passwd_expiry_days");
        $app->setparameter($cp_passwd_max_attempts,           "security_profile", "passwd_max_attempts");
        $app->setparameter($cp_passwd_history_cycle,          "security_profile", "passwd_history_cycle");
        $app->setparameter($cp_passwd_delay_ms,               "security_profile", "passwd_delay_ms");
        $app->setparameter($cp_passwd_min_chars,              "security_profile", "passwd_min_chars");
        $app->setparameter($cp_passwd_char_uniqueness,        "security_profile", "passwd_char_uniqueness");
        $app->setparameter(isset($cp_passwd_alphanum_mixed),  "security_profile", "passwd_alphanum_mixed");
        $app->setparameter(isset($cp_passwd_apply_stopwords), "security_profile", "passwd_apply_stopwords");
        $app->setparameter($cp_logexceedopt,                  "loginlimit", "logexceedopt");
        $app->setparameter($cp_logexceedurl,                  "loginlimit", "logexceedurl");
        $app->setparameter($cp_badips,                        "badips", "badips");
        
        // Remote authorisation fields..
        $app->setparameter($cp_remote_auth_source,            "remote_authentication", "remote_auth_source");
        $app->setparameter($cp_remote_auth_method,            "remote_authentication", "remote_auth_method");
        $app->setparameter($cp_remote_auth_dbname,            "remote_authentication", "remote_auth_dbname");
        $app->setparameter($cp_remote_auth_tablename,         "remote_authentication", "remote_auth_tablename");
        // Refresh all mappings..
        $app->delparameter("remote_authentication", "remote_auth_mappings");
        foreach ($REMOTE_AUTH_FIELDNAMES as $axyl_field) {      
          $varname = "cp_remote_auth_mapping_$axyl_field";
          if (isset($$varname) && $$varname != "") {
            $app->setparameter($$varname, "remote_authentication", "remote_auth_mappings", $axyl_field, "array");
          }
        }  
        break;
        
      case CP_VIEW_DB:
        // Database Definition Deletes
        if (isset($_recmaintpost_dels) && $_recmaintpost_dels != "") {
          $delids = explode(FIELD_DELIM, $_recmaintpost_dels);
          $delixs = array();
          foreach ($delids as $dbid) {
            $ix = getdbindex($dbid);
            if ($ix != -1) {
              $delixs[] = $ix;
            }
          }
          foreach ($delixs as $ix) {
            unset($app->settings[$ix]);
          }
        }
        // DATABASES
        if (isset($_recmaintpost_data) && $_recmaintpost_data != "") {
          $dbrecs = explode(RECORD_DELIM, $_recmaintpost_data);
          $dbfields = explode(",", $_recmaintpost_flds);
          foreach ($dbrecs as $dbrec) {
            $dbvalues = explode(FIELD_DELIM, $dbrec);
            $dbid = array_shift($dbvalues);
            $dbsetting = new setting("database", "add_database");
            $pos = 0;
            foreach ($dbfields as $dbfield) {
              $value = $dbvalues[$pos++];
              switch ($dbfield) {
                case "dbname":
                  $parameter = new parameter("name", "string");
                  $dbname = $value;
                  break;
                case "dbtype":
                  $parameter = new parameter("type", "string");
                  break;
                case "dbuser":
                  $parameter = new parameter("user", "string");
                  break;
                case "dbpassword":
                  $parameter = new parameter("password", "string");
                  break;
                case "dbhost":
                  $parameter = new parameter("host", "string");
                  break;
                case "dbport":
                  $parameter = new parameter("port", "integer");
                  break;
                case "dbenc":
                  $parameter = new parameter("enc", "string");
                  break;
                case "dbdatestyle":
                  $parameter = new parameter("datestyle", "string");
                  break;
              }
              $parameter->setvalue($value);
              $dbsetting->addparameter($parameter->name, $parameter);
            }
    
            $ix = get_settingindex($app, $dbname);
            if ($ix > -1) {
              $app->settings[$ix] = $dbsetting;
            }
            else {
              // Insert new database at end of existing databases
              // so that they stay pleasingly grouped..
              $lastdbix = getlastdbindex();
              if ($lastdbix == -1) {
                $app->settings[] = $dbsetting;
              }
              else {
                $ix = 0;
                $settings = array();
                foreach ($app->settings as $setting) {
                  $settings[] = $setting;
                  if ($ix == $lastdbix) {
                    $settings[] = $dbsetting;
                  }
                  $ix += 1;
                }
                $app->settings = $settings;
              }
            }
          } // foreach dbrecs
        } // database save
        // Database ordering - determines default database
        elseif (isset($_recmaintpost_order) && $_recmaintpost_order != "") {
          $dborderings = explode(FIELD_DELIM, $_recmaintpost_order);
          $dbsettings = array();
          foreach ($dborderings as $dborder) {
            $ix = getdbindex($dborder);
            $dbsettings[] = $app->settings[$ix];
          }
          $firstdbix = getdbindex(0);
          for ($ix=0; $ix < count($dbsettings); $ix++) {
            $app->settings[$ix + $firstdbix] = $dbsettings[$ix];
          }
        }    
        $app->setparameter(isset($cp_database_backed), "database_backed", "database_backed");
        $app->setparameter($cp_permhosts, "permhosts", "permhosts");
        break;
        
      case CP_VIEW_DEBUG:
        $app->globals["SQL_EXEC_THRESHOLD"] = $cp_sql_exec_threshold;
        $app->setparameter(isset($cp_debug_on), "debug_on", "debug_on");
        $app->setparameter(isset($cp_response_timer), "response_timer", "response_timer");
        // Unpack debug classes..
        $debug_classes = 0;
        foreach ($cp_debug_classes as $class) {
          $debug_classes |= $class;
        }
        $app->setparameter($debug_classes, "debug_classes", "debug_classes");
        // Unpack debug outputs..
        $debug_output = 0;
        foreach ($cp_debug_output as $output) {
          $debug_output |= $output;
        }
        $app->setparameter($debug_output, "debug_output", "debug_output");      
        break;
        
      default:
        // DEFINITIONS
        $app->definitions["APP_PREFIX"] = $cp_app_prefix;
        $app->definitions["APP_NAME"] = $cp_app_name;
    
        // GLOBALS
        $app->globals["TEMPLATESDIR"] = $cp_templatesdir;
        $app->globals["IMAGESDIR"] = $cp_imagesdir;
        $app->globals["CACHEDIR"] = $cp_cachedir;
        $app->globals["CATALOGDIR"] = $cp_catalogdir;
        $app->globals["CMDIR"] = $cp_cmdir;
        $app->globals["INCDIR"] = $cp_incdir;
        $app->globals["WEBMASTER_PERSON"] = $cp_webmaster_person;
        $app->globals["WEBMASTER_EMAIL"] = $cp_webmaster_email;
    
    
        // Handle the HTTP host setting. If it has the word 'default' in it
        // then we assume there is no HTTP_HOST override being made.. 
        if (stristr($cp_http_host, "default")) {
            $cp_http_host = "";
        }
        
        // SETTINGS
        $app->setparameter($cp_dtd_html, "dtd", "dtd", "html");
        $app->setparameter($cp_dtd_wml,  "dtd", "dtd", "wml");
        if (isset($cp_multilang)) {
          $app->setparameter(true, "multilang", "multilang");
          $app->setparameter("UTF-8", "encoding", "encoding");
        }
        else {
          $app->setparameter(false, "multilang", "multilang");
          $app->setparameter($cp_encoding, "encoding", "encoding");
        }
        $app->setparameter($cp_http_host,                     "http_host", "http_host");
        $app->setparameter($cp_cookiename,                    "cookiename", "cookiename");
        $app->setparameter($cp_lifetime,                      "lifetime", "lifetime");
        $app->setparameter(isset($cp_guest_browser_lifetime), "guest_browser_lifetime", "guest_browser_lifetime");
        $app->setparameter(isset($cp_session_track_logins),   "session_track_logins", "session_track_logins");
        $app->setparameter($cp_expiry,                        "expiry", "expiry");
        $app->setparameter(isset($cp_microsites_enabled),     "microsites_enabled", "microsites_enabled");
        $app->setparameter(isset($cp_metadata_enabled),       "metadata_enabled", "metadata_enabled");
        $app->setparameter(isset($cp_buffered_output),        "buffered_output", "buffered_output");
        $app->setparameter($cp_compression_type,              "compression_type", "compression_type");
        $app->setparameter($cp_compression_threshold,         "compression_threshold", "compression_threshold");
        $app->setparameter(isset($cp_keep),                   "keep", "keep");
        $app->setparameter(isset($cp_globalise),              "globalise", "globalise");
    } // switch
    
    // Save it
    $app->save();
    $app = new application();
  }
}

// ----------------------------------------------------------------------
// BOILERPLATING
$s = <<< EOS
<html>
<head>
<title>Axyl Control Panel</title>
<meta http-equiv="content-type" content="text/html; charset=UTF-8">
<meta name="generator" content="Catalyst IT Axyl">
<style type="text/css">
 margin: 0px 0px 0px 0px;
 font-family: Verdana, Arial, Helvetica, sans-serif;
 color: #605728;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
 scrollbar-face-color: #f7f7f7;
 scrollbar-highlight-color: #b2b1b1;
 scrollbar-shadow-color: #b2b1b1;
 scrollbar-3dlight-color: white;
 scrollbar-arrow-color: #c9c9c9;
 scrollbar-track-color: #f5f5f5;
 scrollbar-darkshadow-color: white;
}
p, td, th, ol, ul, li, input, textarea, select {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
 color: #605728;
}
input, textarea, select {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 9pt;
 font-style: normal;
 font-weight: normal;
}
p {
 line-height: 115%;
}
hr {
 height: 1px;
 color: black;
 margin-top: 0;
 margin-bottom: 0;
}
form {
 margin: 0px;
 padding: 0px;
}
a {
 color: #AC9D46;
 text-decoration: none;
}
a:hover {
 color: #AC9D46;
 text-decoration: underline;
}
a:active {
 color: #AC9D46;
}
a:visited {
 color: #AC9D46;
}
th {
 text-align: left;
}

h1, h2, h5, h3, h4, h6 {
 font-family: Verdana, Arial, Helvetica, sans-serif;
 font-weight: bold;
 margin-top: 2px;
 margin-bottom: 2px;
}
h1 { color:#605728; font-size:125%; text-transform:capitalize; }
h2 { color:#605728; font-size:120%; }
h3 { color:#605728; font-size:115%; font-weight:bold;}
h4 { color:#605728; font-size:105%; font-weight:bold;}
h5 { color:#605728; font-size:100%; font-weight:bold;}
h6 { color:#605728; font-size:96%;  font-weight:bold;}
.axform {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 padding: 0px;
}
.axcombo {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 height: 20px;
 padding-left: 2px;
}
.axlistbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 padding-left: 2px;
}
.axtxtbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 250px;
 height: 22px;
 padding-left: 2px;
 vertical-align: middle;
}
.axmemo {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 250px;
 height: 100px;
 padding-left: 2px;
}
.axdatetime {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 150px;
 height: 22px;
 padding-left: 2px;
}
.axnumbox {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 width: 80px;
 height: 22px;
 padding-left: 2px;
 padding-right: 2px;
 vertical-align: middle;
 text-align: right;
}
.axchkbox {
 vertical-align: middle;
}
.axfmlbl {
 font-family: Arial, Helvetica, sans-serif;
 font-size: 95%;
 font-weight: normal;
 vertical-align: top;
 color: black;
}
.axtitle {
 font-family: Arial, Helvetica, sans-serif;
 font-size:110%;
 color: white;
 background-color: #66700F;
 font-weight: bold;
}
.axfoot {
 height: 12px;
 background-color: #66700F;
}
.axhdg {
 font-family: Arial, Helvetica, sans-serif;
 font-size:100%;
 color: white;
 background-color: #898437;
 font-weight: bold;
}
.axsubhdg {
 font-family: Arial, Helvetica, sans-serif;
 font-size:100%;
 color: white;
 background-color: #66700F;
 font-weight: bold;
}
.axfg {
 color: #605728;
 font-weight: normal;
}
.axhl {
 color: red;
 font-weight: bold;
}
.axerror {
 color: red;
}
.axbgwhite {
 color: black;
 background-color: white;
}
.axbglite {
 color: black;
 background-color: #EAEBDF;
}
.axbgdark {
 color: white;
 background-color: #DEDFD4;
}
.axbgdarker {
 color: white;
 background-color: #66700F;
}
</style>
<script language="javascript">
var keyfield = new Array();
var curid = new Array();
var newid = new Array();
function setUTF8mode(multilang) {
  if (multilang) {
    document.forms.$formname.cp_encoding.value='UTF-8';
    document.forms.$formname.cp_encoding.disabled=true;
  }
  else {
    document.forms.$formname.cp_encoding.readonly=false;
    document.forms.$formname.cp_encoding.disabled=false;
  }
  return true;
}
var pgchanged=false;
function tabclick(tabno) {
  if (pgchanged) {
    msg  = 'WARNING:\\n';
    msg += 'You have changed data on this page. Before you can switch to another\\n';
    msg += 'page, you must either Save this one, or Reset it.\\n\\n';
    alert(msg);
  }
  else {
    location = '$PHP_SELF?cp_view=' + tabno;
  }
}
function setchgd() {
  pgchanged = true;
}
function resetchgd() {
  pgchanged = false;
  document.forms.$formname.reset();
}
function control_auth_fields(auth,formname) {
  var form = eval('document.forms.'+formname);
  var mode = (auth.value == 0);
  if (form) {
    for (var i = 0; i < form.length; i++) {
      var e = form.elements[i];
      if (e.id == 'auth_fields') {
        if (e.type.substr(0,6) == 'select') e.disabled = mode;
        else e.readOnly = mode;
        if (mode == true) {
          e.style.backgroundColor = '#ededed';
        }
        else {
          e.style.backgroundColor = '#ffffff';
        }
      }
    }
  }
}
</script>
<script type="text/javascript" src="$LIBDIR/js/recmaint.js"></script>
<script type="text/javascript" src="$LIBDIR/js/fieldvalidation.js"></script>
</head>
<body>
EOS;

// ----------------------------------------------------------------------
// MAIN FORM GENERATION

// Width of large form elements..
$fullwidth  = 540;
$halfwidth  = ceil($fullwidth * 0.50);
$thirdwidth = ceil($fullwidth * 0.37);
$quartwidth = ceil($fullwidth * 0.25);
$ewidth   = $halfwidth  . "px"; // Normal text fields
$awidth   = $fullwidth  . "px"; // Full field width
$cbowidth = $quartwidth . "px"; // Normal combos
$cwidth   = $thirdwidth . "px"; // Wide combos

if ($cp_view == CP_VIEW_DB) {
  // DATABASE LISTBOX
  // Defined early so that buttons can be registered..
  $database_listbox = new form_combofield("dbid");
  $database_listbox->setclass("axlistbox");
  // Make a new record maintainer, and attach the buttons..
  $maintainer = new recmaintainer($formname, $database_listbox);
  
  $bup    = new form_imagebutton("_up",    "", "", "$LIBDIR/img/_up.gif",     "Move up",              57, 15);
  $bdown  = new form_imagebutton("_down",  "", "", "$LIBDIR/img/_down.gif",   "Move down",            57, 15);
  $bdel   = new form_imagebutton("_del",   "", "", "$LIBDIR/img/_delete.gif", "Delete database",      57, 15);
  $badd   = new form_imagebutton("_add",   "", "", "$LIBDIR/img/_add.gif",    "Add new database",     57, 15);
}

// Standard buttons
$bsave  = new form_imagebutton("_save",  "", "", "$LIBDIR/img/_save.gif",   "Save your settings",   57, 15);
$breset = new form_imagebutton("_reset", "", "", "$LIBDIR/img/_reset.gif",  "Reverse your changes", 57, 15);
$breset->set_onclick("resetchgd()");
$bdef   = new form_imagebutton("_default", "", "", "$LIBDIR/img/_default.gif", "Replace ALL settings with defaults", 57, 15);
$bdef->set_confirm_text("This will over-write your WHOLE file with the default configuration (ie. not just the current page). Continue?");

// If we have a database maintainer, register all buttons..
if ($cp_view == CP_VIEW_DB) {
  // Register all relevant buttons to the maintainer..
  $maintainer->register_button("up" ,   $bup);
  $maintainer->register_button("down",  $bdown);
  $maintainer->register_button("del",   $bdel);
  $maintainer->register_button("add",   $badd);
  $maintainer->register_button("save",  $bsave);
}

$Tapp = new table();
$Tapp->setwidth($fullwidth);
$Tapp->setalign("center");

// Initialise tab buttons string..
$tab_btn = new img("$LIBDIR/img/_cptabtip.gif", "", $CPTABS[0], 6, 23);
$rendered_tabs = $tab_btn->render();
foreach ($CPTABS as $tabno => $tabdesc) {
  $tab_btn = new img("$LIBDIR/img/_cptab" . $tabno . ".gif", "_tab" . $tabno, $tabdesc, 84, 23);
  $tab_btn->set_onclick("tabclick('$tabno')");
  $rendered_tabs .= $tab_btn->render();
}
$Tapp->tr();
$Tapp->td($rendered_tabs, "text-align:right");
$Tapp->td_alignment("right");

// ......................................................................
// Heading

switch ($cp_view) {
  case CP_VIEW_DEFAULT: $view = "Main Settings"; break;
  case CP_VIEW_AUTH:    $view = "User & Security Settings"; break;
  case CP_VIEW_DB:      $view = "Database Settings"; break;
  case CP_VIEW_DEBUG:   $view = "Debug Settings"; break;
} // switch

$Tapp->tr("axtitle");
$Tapp->td("<b>AXYL CONTROL PANEL - $view</b>", "axtitle");
$Tapp->td_css("vertical-align:center;height:30px;padding-left:5px;");
if ($user_msg != "") {
  $Tapp->tr("axsubhdg");
  $Tapp->td($user_msg, "color:#F5DD64;text-align:center;");
}
elseif ($synced) {
  $Tapp->tr("axsubhdg");
  $Tapp->td("The Axyl configuration structure was successfully updated.", "color:#F5DD64;text-align:center;");
}

if (!$error) {
  // ......................................................................
  // Toolbar..
  $toolbar = array();
  $toolbar[] = $breset;
  $toolbar[] = $bdef;
  $toolbar[] = $bsave;
  $Tbar = new table("toolbar");
  $Tbar->tr();
  $tools = "";
  foreach ($toolbar as $tool) {
    $tools .= $tool->render();
  }
  $Tbar->th($tools, "text-align:right");
  $Tapp->tr("axbglite");
  $Tapp->td( $Tbar->render() );

  $tbox = new form_textfield();
  $tbox->setstyle("width:$ewidth");
  $tbox->setclass("axtxtbox");
  $tbox->set_onchange('setchgd()');

  $chkbox = new form_checkbox();
  $chkbox->setclass("axchkbox");
  $chkbox->setvalue("yes");
  $chkbox->checked = false;
  $chkbox->set_onclick('setchgd()');

  // ......................................................................
  // DEFINITIONS
  // Installs text field in $Tin table..
  function entryField($label, $fieldname, &$valarray, $tooltip="") {
    global $app, $Tin, $tbox, $bg;
    $mybox = $tbox;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td( $label, "axfg" );
    $mybox->setvalue($valarray[$fieldname]);
    if ($tooltip != "") {
      $mybox->settitle($tooltip);
    }
    $Tin->td( $mybox->render("cp_" . strtolower($fieldname)) );
  }
  // Installs info row in $Tin table..
  function infoField($info) {
    global $app, $Tin, $tbox, $bg;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td();
    $Tin->td($info, "axfg");
    $Tin->td_css("font-style:italic;font-size:80%");
  }
  // Installs text field in $Tin table..
  function integerField($label, $fieldname, &$valarray, $intlimit, $pxwidth=100, $tooltip="") {
    global $app, $Tin, $tbox, $bg;
    $mybox = $tbox;
    $bg = ($bg == "axbgdark" ? "axbglite" : "axbgdark");
    $Tin->tr($bg);
    $Tin->td( $label, "axfg" );
    $mybox->setstyle("width:" . $pxwidth . "px");
    $mybox->set_onblur("limitInt(this, 0, $intlimit)");
    $mybox->setvalue($valarray[$fieldname]);
    if ($tooltip != "") {
      $mybox->settitle($tooltip);
    }
    $Tin->td( $mybox->render("cp_" . strtolower($fieldname)) );
  }

  // For toggling background colour..
  $bg = "axbgdark";
  
  switch ($cp_view) {
    
    // ......................................................................
    // AUTH & SECURITY SETTINGS
    case CP_VIEW_AUTH:
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Local password controls</b>", "axsubhdg");
      
      $Tin = new table("local_pass");
      $Tin->setpadding(2);
    
      $Tin->tr("axbgdark");
      $Tin->td( "Password encryption method:", "axfg" );
      $Fenc = new form_combofield("cp_passwd_encryption");
      $Fenc->setclass("axcombo");
      $Fenc->setstyle("width:$cwidth");
      $Fenc->set_onchange("setchgd()");
      $Fenc->settitle("Determines the method used for encrypting/decrypting the submitted user password.");
      $Fenc->additem("none", "No encryption (plaintext)");
      $Fenc->additem("md5", "Standard MD5 encrypted password");
      $Fenc->additem("md5salted", "Salted MD5 in '*salt*salted_md5' format");
      $Fenc->additem("custom", "Use custom password functions");
      $Fenc->setvalue($app->getparameter("security_profile", "passwd_encryption"));
      $Tin->td( $Fenc->render() );
      
      $mybox = $tbox;
      $label = "Password expiry days"; $fld = "passwd_expiry_days"; $style = "width:50px"; $intlimit = 999;  
      $mybox->setstyle($style);
      $mybox->set_onblur("limitInt(this, 1, $intlimit)");
      $mybox->setvalue($app->getparameter("security_profile", $fld));
      $mybox->settitle(
            "Days before a new password expires. After this time the user will be "
            . "required to choose a new password. To override this, you can check the "
            . "'Password never expires' option, in user maintenance."
          );
      $Tin->tr("axbglite");
      $Tin->td( "$label:", "axfg" );
      $Tin->td( $mybox->render("cp_" . $fld) );
    
      $label = "Allowed password failures"; $fld = "passwd_max_attempts"; $style = "width:50px"; $intlimit = 99;  
      $mybox->setstyle($style);
      $mybox->set_onblur("limitInt(this, 0, $intlimit)");
      $mybox->setvalue($app->getparameter("security_profile", $fld));
      $mybox->settitle(
            "Number of consecutive times the user can fail to supply the correct password before "
            . "the account is locked. Locked accounts require an administrator to unlock them. "
            . "Set to zero to allow any number of failures."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "$label:", "axfg" );
      $Tin->td( $mybox->render("cp_" . $fld) );
    
      $label = "Password history cycle"; $fld = "passwd_history_cycle"; $style = "width:50px"; $intlimit = 999;  
      $mybox->setstyle($style);
      $mybox->set_onblur("limitInt(this, 0, $intlimit)");
      $mybox->setvalue($app->getparameter("security_profile", $fld));
      $mybox->settitle(
            "Number of passwords the system will remember for each user. This prevents re-use of "
            . "passwords chosen in the recent past. Set to zero to disable this feature."
          );
      $Tin->tr("axbglite");
      $Tin->td( "$label:", "axfg" );
      $Tin->td( $mybox->render("cp_" . $fld) );
    
      $label = "Minimum password length"; $fld = "passwd_min_chars"; $style = "width:50px"; $intlimit = 99;  
      $mybox->setstyle($style);
      $mybox->set_onblur("limitInt(this, 1, $intlimit)");
      $mybox->setvalue($app->getparameter("security_profile", $fld));
      $mybox->settitle(
            "The minimum number of characters a new password must have to be acceptable."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "$label:", "axfg" );
      $Tin->td( $mybox->render("cp_" . $fld) );
    
      $Tin->tr("axbglite");
      $Tin->td( "Level of char uniqueness:", "axfg" );
      $Fpunq = new form_combofield();
      $Fpunq->setclass("axcombo");
      $Fpunq->setstyle("width:$cwidth");
      $Fpunq->set_onchange('setchgd()');
      $Fpunq->settitle(
            "A level of character uniqueness a new password must have. This helps prevent the "
            . "choice of silly passwords containing repeating character sequences."
          );
      $Fpunq->additem("none",  "No requirement");
      $Fpunq->additem("low", "Low");
      $Fpunq->additem("medium", "Medium");
      $Fpunq->additem("high", "High");
      $Fpunq->setvalue($app->getparameter("security_profile", "passwd_char_uniqueness"));
      $Tin->td( $Fpunq->render("cp_passwd_char_uniqueness") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("security_profile", "passwd_alphanum_mixed");
      $mychkbox->settitle(
            "If checked, this will require a mix of numbers and alphabetic characters in "
            . "a new password. Such passwords are generally stronger."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Require mix of alpha & numerics:", "axfg" );
      $Tin->td( $mychkbox->render("cp_passwd_alphanum_mixed") );
      
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("security_profile", "passwd_apply_stopwords");
      $mychkbox->settitle(
            "If checked, the system will check a new password against a database of common "
            . "'bad' words which people use in their choices, and prevent them selecting "
            . "words which are considered easy to crack, including variations of their own "
            . "name and user logon ID."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Apply stop-words to password:", "axfg" );
      $Tin->td( $mychkbox->render("cp_passwd_apply_stopwords") );

      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );

      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Login control</b>", "axsubhdg");
    
      $Tin = new table("login");
      $Tin->setpadding(2);
    
      $Tin->tr("axbgdark");
      $Tin->td( "Login method:", "axfg" );
      $Fcomp = new form_combofield();
      $Fcomp->setclass("axcombo");
      $Fcomp->setstyle("width:$cwidth");
      $Fcomp->set_onchange('setchgd()');
      $Fcomp->settitle(
            "Select the type of login method - the usual is via a custom Axyl form. HTTP "
            . "authentication uses the browser-based popup form."
          );
      $Fcomp->additem(0, "No authentication");
      $Fcomp->additem(1, "HTTP authentication");
      $Fcomp->additem(2, "Axyl login form");
      $Fcomp->setvalue($app->getparameter("authtype", "authtype"));
      $Tin->td( $Fcomp->render("cp_authtype") );
    
      $Tin->tr("axbglite");
      $Tin->td( "On failed login:", "axfg" );
      $Fcomp = new form_combofield();
      $Fcomp->setclass("axcombo");
      $Fcomp->setstyle("width:$cwidth");
      $Fcomp->set_onchange('setchgd()');
      $Fcomp->settitle(
            "Actions to take when the user fails on login. Note that some of these have "
            . "implications for security - for example you might want to have the system give "
            . "no feedback in some cases. In others a pretty webpage might be the best option."
          );
      $Fcomp->additem(0, "Display basic fail message");
      $Fcomp->additem(1, "Die silently");
      $Fcomp->additem(2, "Re-direct to URL (below)");
      $Fcomp->additem(3, "Login as guest instead");
      $Fcomp->setvalue($app->getparameter("authfail", "authfailopt"));
      $Tin->td( $Fcomp->render("cp_authfailopt") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Failed login re-direct URL:", "axfg" );
      $mybox = $tbox;
      $mybox->settitle(
            "Supply the URL to re-direct to on failed login."
          );
      $mybox->setvalue($app->getparameter("authfail", "authfailurl"));
      $Tin->td( $mybox->render("cp_authfailurl") );
      
      $Tin->tr("axbglite");
      $Tin->td( "Login delay after failure (mS):", "axfg" );
      $mybox = $tbox;
      $mybox->setstyle("width:90px");
      $mybox->set_onblur("limitInt(this, 0, 9999)");
      $mybox->settitle(
            "A delay time (in milliseconds) applied after a failed login. This acts as a "
            . "control on automated password hacking scripts which repeatedly try passwords "
            . "to crack an account."
          );
      $mybox->setvalue($app->getparameter("security_profile", "passwd_delay_ms"));
      $Tin->td( $mybox->render("cp_passwd_delay_ms") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "On login limit exceeded:", "axfg" );
      $Flogexc = new form_combofield();
      $Flogexc->setclass("axcombo");
      $Flogexc->setstyle("width:$cwidth");
      $Flogexc->set_onchange('setchgd()');
      $Flogexc->settitle(
            "Actions to take when user session (login) limit is exceeded. This only applies "
            . "if the account has a non-zero limit set on it."
          );
      $Flogexc->additem(0, "Take no action");
      $Flogexc->additem(1, "Allow, cull oldest sessions");
      $Flogexc->additem(2, "Deny access, display message");
      $Flogexc->additem(3, "Deny access silently");
      $Flogexc->additem(4, "Redirect to a URL (below)");
      $Flogexc->additem(5, "Login as guest instead");
      $Flogexc->setvalue($app->getparameter("loginlimit", "logexceedopt"));
      $Tin->td( $Flogexc->render("cp_logexceedopt") );
    
      $Tin->tr("axbglite");
      $Tin->td( "Login excess re-direct URL:", "axfg" );
      $mybox = $tbox;
      $mybox->settitle(
            "If re-directing to a webpage, enter the URL for the page here."
          );
      $mybox->setvalue($app->getparameter("loginlimit", "logexceedurl"));
      $Tin->td( $mybox->render("cp_logexceedurl") );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // REMOTE AUTHENTICATION
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Remote authentication</b>", "axsubhdg");
    
      $Tin = new table("remote");
      $Tin->setpadding(2);

      $Tin->tr("axbgdark");
      $Tin->td( "User authentication source:", "axfg" );
      $Fauthsrc = new form_combofield("cp_remote_auth_source");
      $Fauthsrc->setclass("axcombo");
      $Fauthsrc->setstyle("width:$cwidth");
      $Fauthsrc->set_onchange("setchgd();control_auth_fields(this,'$formname')");
      $Fauthsrc->settitle(
            "Determines the source used for acquiring login userid/password information for "
            . "authentication. The default is to use the local Axyl ax_user table. If you "
            . "specify a remote database, that database must be defined in the database "
            . "setup section of this control panel."
            );
      $Fauthsrc->additem(LOCAL_AUTH, "Local authentication (default)");
      $Fauthsrc->additem(REMOTE_AUTH_REMOTEDB, "Remote database");
      //$Fauthsrc->additem(REMOTE_AUTH_LDAP, "From LDAP server");
      $remote_auth_source = $app->getparameter("remote_authentication", "remote_auth_source");
      $Fauthsrc->setvalue($remote_auth_source);
      $Tin->td( $Fauthsrc->render() );
    
      $Tin->tr("axbglite");
      $Tin->td( "Password encryption method:", "axfg" );
      $Fauthmeth = new form_combofield("cp_remote_auth_method");
      $Fauthmeth->setclass("axcombo");
      $Fauthmeth->setstyle("width:$cwidth");
      $Fauthmeth->set_onchange("setchgd()");
      $Fauthmeth->setid("auth_fields");
      $Fauthmeth->settitle(
            "Determines the method used for authenticating the submitted remote user password. "
            . "Select one of the common methods, otherwise choose the custom option, and define "
            . "the algorithm using the 'custom_password_authentication()' function, in your "
            . "local copy of 'application.php'."
            );
      $Fauthmeth->additem("none", "No encryption (plaintext)");
      $Fauthmeth->additem("md5", "Standard MD5 encrypted password");
      $Fauthmeth->additem("md5salted", "Salted MD5 in '*salt*salted_md5' format");
      $Fauthmeth->additem("custom", "Use custom password functions");
      $Fauthmeth->setvalue($app->getparameter("remote_authentication", "remote_auth_method"));
      $Tin->td( $Fauthmeth->render() );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Remote database:", "axfg" );
      
      $Fauthdb = new form_combofield("cp_remote_auth_dbname");
      $Fauthdb->setclass("axcombo");
      $Fauthdb->setstyle("width:$cwidth");
      $Fauthdb->set_onchange('setchgd()');
      $Fauthdb->setid("auth_fields");
      $Fauthdb->settitle(
            "If you selected 'remote database' above then select the database the user "
            . "authentication data is held on here."
            );
      // Get defined databases..
      $dbs = $app->get_setting("database");
      if ($dbs === false) $databases = array();
      elseif (is_array($dbs)) $databases = $dbs;
      else $databases[0] = $dbs;
      $Fauthdb->additem("");
      foreach ($databases as $database) {
        // Populate listbox..
        $dbname = $database->getparameter("name");
        $Fauthdb->additem($dbname);
      }
      $Fauthdb->setvalue($app->getparameter("remote_authentication", "remote_auth_dbname"));
      $Tin->td( $Fauthdb->render() );
    
      $Tin->tr("axbglite");
      $Tin->td( "Remote user table name:", "axfg" );
      $mybox = $tbox;
      $mybox->setid("auth_fields");
      $mybox->setstyle("width:$cwidth");
      $mybox->settitle(
            "This is the name of the table on the remote database which holds the user "
            . "authentication data such as userid and password."
            );
      $mybox->setvalue($app->getparameter("remote_authentication", "remote_auth_tablename"));
      $Tin->td( $mybox->render("cp_remote_auth_tablename") );

      foreach ($REMOTE_AUTH_FIELDNAMES as $axyl_field) {      
        $Tin->tr("axbgdark");
        $Tin->td( "&raquo;&nbsp;remote field for $axyl_field:", "axfg" );
        $mybox = $tbox;
        $mybox->setid("auth_fields");
        $mybox->setstyle("width:$cwidth");
        $mybox->settitle(
              "Enter the name of the remote field corresponding to the local '$axyl_field' field."
              );
        $mapping = $app->getparameter("remote_authentication", "remote_auth_mappings", $axyl_field);
        $mybox->setvalue( $mapping !== false ? $mapping : "" );
        $Tin->td( $mybox->render("cp_remote_auth_mapping_$axyl_field") );
      }  
    
      $Tin->tr("axbgdark");
      $Tin->td();
      $Tin->td(
          "Only enter names of mapped fields, leaving unmapped ones blank. "
        . "Note: user_id and password are mandatory.",
          "axfg"
          );
      $Tin->td_css("font-style:italic;font-size:80%");
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // MISC SETTINGS
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Miscellaneous settings</b>", "axsubhdg");
    
      $Tin = new table("misc");
      $Tin->setpadding(2);
    
      $Tin->tr("axbglite");
      $Tin->td( "IP addresses to block:", "axfg" );
      $mybox = $tbox;
      $mybox->settitle(
            "This is used to block specific IP addresses which are causing a problem "
            . "accessing the website. Any IP listed here will be denied access."
          );
      $mybox->setvalue(str_replace("\"", "", $app->getparameter("badips", "badips")));
      $Tin->td( $mybox->render("cp_badips") );
      $Tin->tr("axbglite");
      $Tin->td();
      $Tin->td(
          "A comma-delimited list of IP addresses which are to be denied access.",
          "axfg"
          );
      $Tin->td_css("font-style:italic;font-size:80%");
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
      break;

    // ......................................................................
    // DATABASE SETTINGS
    case CP_VIEW_DB:
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Database connections</b>", "axsubhdg");
    
      $Tin = new table("dbsettings");
      $Tin->setpadding(2);
    
      $database_listbox->setstyle("width:$ewidth;");
      $database_listbox->size = 6;
    
      // Get defined databases..
      $dbs = $app->get_setting("database");
      if ($dbs === false) $databases = array();
      elseif (is_array($dbs)) $databases = $dbs;
      else $databases[0] = $dbs;
    
      $dbid = 0;
      foreach ($databases as $database) {
        // Populate listbox..
        $dbname = $database->getparameter("name");
        $database_listbox->additem($dbid, $dbname);
    
        // Populate maintainer data. The maintainer add_record method
        // requires an associative array keyed on listbox key id..
        $rec = array(
                "dbtype"      => $database->getparameter("type"),
                "dbname"      => $dbname,
                "dbuser"      => $database->getparameter("user"),
                "dbpassword"  => $database->getparameter("password"),
                "dbhost"      => $database->getparameter("host"),
                "dbport"      => $database->getparameter("port"),
                "dbenc"       => $database->getparameter("enc"),
                "dbdatestyle" => $database->getparameter("datestyle")
                );
        $maintainer->add_record($dbid, $rec);
        if (!isset($firstrec)) {
          $firstrec = $rec;
        }
        $dbid += 1;
      } // foreach
    
      // Now set the defaults for each of the fields. These are
      // necessary for when a new record is created..
      $defaults = array(
                "dbtype"      => "postgres",
                "dbname"      => "",
                "dbuser"      => "",
                "dbpassword"  => "",
                "dbhost"      => "",
                "dbport"      => "",
                "dbenc"       => "UNICODE",
                "dbdatestyle" => "ISO"
                );
      $maintainer->add_defaults($defaults);
      if (!isset($firstrec)) {
        $firstrec = $defaults;
      }
    
      // The listbox field..
      $database_listbox->settitle(
            "Databases which this website needs to connect to. The first will be the "
            . "default database."
          );
      $database_listbox->setvalue($firstrec["dbname"]);
      $Tin->tr("axbgdark");
      $Tin->td( $database_listbox->render() );
      $Tin->td_width("50%");
    
      $Tin2 = new table();
      $Tin2->td(
          "NB: The ordering of this list is important. The first "
        . "database will be the default connection.",
          "axfg"
          );
      $Tin2->td_css("font-style:italic;font-size:80%");
      $Tin2->td_alignment("", "top");
      $bdel->setstyle("padding-top:4px");
      $Tin2->td(
            $badd->render()  . "<br>"
          . $bup->render()   . "<br>"
          . $bdown->render() . "<br>"
          . $bdel->render()
          );
      $Tin2->td_alignment("right", "top");
      $Tin->td( $Tin2->render() );
      $Tin->td_width("50%");
      $Tin->td_alignment("", "top");
      // ..................................................................
      // Database type field..
      $Fdbtype = new form_combofield("dbtype", "", $firstrec["dbtype"]);
      $Fdbtype->setclass("axcombo");
      $Fdbtype->settitle(
            "The type of database it is. This determines the database interface "
            . "module for executing your queries."
          );
      $maintainer->register_field($Fdbtype);
      $Fdbtype->additem("postgres", "Postgres");
      $Fdbtype->additem("odbc",     "ODBC");
      $Fdbtype->additem("mssql",    "MS SQL Server");
      $Fdbtype->additem("mysql",    "MySQL");
      $Fdbtype->additem("oracle",   "Oracle");
      $Fdbtype->setstyle("width:$cbowidth;");
      $Tin->tr("axbglite");
      $Tin->td( "Database type:", "axfg" );
      $Tin->td( $Fdbtype->render() );
      // ..................................................................
      // Database name field..
      $Fdbname = new form_textfield("dbname", "", $firstrec["dbname"]);
      $maintainer->register_field($Fdbname);
      $Fdbname->setstyle("width:$ewidth;");
      $Fdbname->setclass("axtxtbox");
      $Fdbname->settitle(
            "The unique name of this database, as used in the connect string."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Database name:", "axfg" );
      $Tin->td( $Fdbname->render() );
      // ..................................................................
      // Database user field..
      $Fdbuser = new form_textfield("dbuser", "", $firstrec["dbuser"]);
      $maintainer->register_field($Fdbuser);
      $Fdbuser->setstyle("width:$ewidth;");
      $Fdbuser->setclass("axtxtbox");
      $Fdbuser->settitle(
            "The name of a user who is permitted to connect to the database."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Username:", "axfg" );
      $Tin->td( $Fdbuser->render() );
      // ..................................................................
      // Database password field..
      $Fdbpassword = new form_textfield("dbpassword", "", $firstrec["dbpassword"]);
      $maintainer->register_field($Fdbpassword);
      $Fdbpassword->setstyle("width:$ewidth;");
      $Fdbpassword->setclass("axtxtbox");
      $Fdbpassword->settitle(
            "If the database requires a password to authenticate the connection, then "
            . "enter it here, otherwise leave blank."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "User password:", "axfg" );
      $Tin->td( $Fdbpassword->render() );
      // ..................................................................
      // Database host field..
      $Fdbhost = new form_textfield("dbhost", "", $firstrec["dbhost"]);
      $maintainer->register_field($Fdbhost);
      $Fdbhost->setstyle("width:$ewidth;");
      $Fdbhost->setclass("axtxtbox");
      $Fdbhost->settitle(
            "For a locally hosted database, leave blank. However, if this database "
            . "lives on a remote machine, enter the hostname of that machine here."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Hostname:", "axfg" );
      $Tin->td( $Fdbhost->render() );
      // ..................................................................
      // Database port field..
      $Fdbport = new form_textfield("dbport", "", $firstrec["dbport"]);
      $maintainer->register_field($Fdbport);
      $Fdbport->setstyle("width:$ewidth;");
      $Fdbport->setclass("axtxtbox");
      $Fdbport->settitle(
            "For a locally hosted database leave this blank. For a remotely hosted "
            . "database, you would usually enter '5432' here."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Port number:", "axfg" );
      $Tin->td( $Fdbport->render() );
      $Tin->set_width_profile("50%,50%");
      // ..................................................................
      // Database char encoding field..
      $Fdbenc = new form_combofield("dbenc", "", $firstrec["dbenc"]);
      $Fdbenc->setclass("axcombo");
      $Fdbenc->setstyle("width:$cwidth;");
      $Fdbenc->settitle(
            "Make sure you set this to the encoding that the database was created "
            . "with. In Postgres you can find this out by listing the databases "
            . "in 'psql'."
          );
      $maintainer->register_field($Fdbenc);
      $Fdbenc->additem("", "default");
      $Fdbenc->additem("SQL_ASCII", "ASCII");
      $Fdbenc->additem("UNICODE", "Unicode (UTF-8)");
      $Fdbenc->additem("EUC_JP", "Japanese EUC");
      $Fdbenc->additem("EUC_CN", "Chinese EUC");
      $Fdbenc->additem("EUC_KR", "Korean EUC");
      $Fdbenc->additem("JOHAB", "Korean EUC (Hangle base)");
      $Fdbenc->additem("EUC_TW", "Taiwan EUC");
      $Fdbenc->additem("MULE_INTERNAL", "Mule internal code");
      $Fdbenc->additem("LATIN1", "ISO 8859-1/ECMA 94 (Latin alphabet no.1)");
      $Fdbenc->additem("LATIN2", "ISO 8859-2/ECMA 94 (Latin alphabet no.2)");
      $Fdbenc->additem("LATIN3", "ISO 8859-3/ECMA 94 (Latin alphabet no.3)");
      $Fdbenc->additem("LATIN4", "ISO 8859-4/ECMA 94 (Latin alphabet no.4)");
      $Fdbenc->additem("LATIN5", "ISO 8859-9/ECMA 128 (Latin alphabet no.5)");
      $Fdbenc->additem("LATIN6", "ISO 8859-10/ECMA 144 (Latin alphabet no.6)");
      $Fdbenc->additem("LATIN7", "ISO 8859-13 (Latin alphabet no.7)");
      $Fdbenc->additem("LATIN8", "ISO 8859-14 (Latin alphabet no.8)");
      $Fdbenc->additem("LATIN9", "ISO 8859-15 (Latin alphabet no.9)");
      $Fdbenc->additem("LATIN10", "ISO 8859-16/ASRO SR 14111 (Latin alphabet no.10)");
      $Fdbenc->additem("ISO_8859_5", "ISO 8859-5/ECMA 113 (Latin/Cyrillic)");
      $Fdbenc->additem("ISO_8859_6", "ISO 8859-6/ECMA 114 (Latin/Arabic)");
      $Fdbenc->additem("ISO_8859_7", "ISO 8859-7/ECMA 118 (Latin/Greek)");
      $Fdbenc->additem("ISO_8859_8", "ISO 8859-8/ECMA 121 (Latin/Hebrew)");
      $Fdbenc->additem("KOI8", "KOI8-R(U)");
      $Fdbenc->additem("WIN", "Windows CP1251");
      $Fdbenc->additem("ALT", "Windows CP866");
      $Fdbenc->additem("WIN1256", "Windows CP1256 (Arabic)");
      $Fdbenc->additem("TCVN", "TCVN-5712/Windows CP1258 (Vietnamese)");
      $Fdbenc->additem("WIN874", "Windows CP874 (Thai)");
      $Tin->tr("axbglite");
      $Tin->td( "Database encoding:", "axfg" );
      $Tin->td( $Fdbenc->render() );
      // ..................................................................
      // Database date style field..
      $Fdbdatestyle = new form_combofield("dbdatestyle", "", $firstrec["dbdatestyle"]);
      $Fdbdatestyle->setclass("axcombo");
      $Fdbdatestyle->setstyle("width:$cwidth;");
      $Fdbdatestyle->settitle(
            "This affects the output format of date-time data from the database. Axyl "
            . "library code expects the ISO format, so it is recommended to always use "
            . "that here, unless you have a good reason to change it."
          );
      $maintainer->register_field($Fdbdatestyle);
      $Fdbdatestyle->additem("", "default");
      $Fdbdatestyle->additem("ISO", "ISO 8601 (1997-12-17 07:37:16-08)");
      $Fdbdatestyle->additem("SQL", "SQL Traditional (12/17/1997 07:37:16.00 PST)");
      $Fdbdatestyle->additem("POSTGRES", "Postgres (Wed Dec 17 07:37:16 1997 PST)");
      $Fdbdatestyle->additem("German", "Regional (17.12.1997 07:37:16.00 PST)");
      $Tin->tr("axbgdark");
      $Tin->td( "Date output style:", "axfg" );
      $Tin->td( $Fdbdatestyle->render() );
    
      $Tapp->tr();
      $Tapp->td( $Tin->render() );    

      $Tin = new table("dbsettings");
      $Tin->setpadding(2);
    
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Misc settings</b>", "axsubhdg");
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("database_backed", "database_backed");
      $mychkbox->settitle(
            "Check this if your website connects to a database. Most do, but this allows "
            . "the possibility of providing simple websites without it."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Website uses a Database:", "axfg" );
      $Tin->td( $mychkbox->render("cp_database_backed") );
    
      $Tin->tr("axbglite");
      $Tin->td( "Hosts for persistent DB connection:", "axfg" );
      $mybox = $tbox;
      $mybox->settitle(
            "If the database hostname contains the string you enter here, then database "
            . "connections will be made persistently, improving performance when the "
            . "site is busy."
          );
      $mybox->setvalue(str_replace("\"", "", $app->getparameter("permhosts", "permhosts")));
      $Tin->td( $mybox->render("cp_permhosts") );
      $Tin->tr("axbglite");
      $Tin->td();
      $Tin->td(
          "A comma-delimited list of hostnames which will use persistent "
        . "database connections. Usually these would be your production "
        . "web-servers.",
          "axfg"
          );
      $Tin->td_css("font-style:italic;font-size:80%");
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
      break;
      
    // ......................................................................
    // DEBUG SETTINGS
    case CP_VIEW_DEBUG:
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Output controls</b>", "axsubhdg");
      
      $Tin = new table("debug_output");
      $Tin->setpadding(2);
    
      $debugging = $app->getparameter("debug_on", "debug_on");
      $mychkbox = $chkbox;
      $mychkbox->settitle(
            "If you want site-wide debugging to be displayed then check this box. It "
            . "will cause every page to display debug information."
          );
      $mychkbox->checked = $debugging;
      $Tin->tr("axbgdark");
      $Tin->td( "Enable debugging:", "axfg" );
      $Tin->td( $mychkbox->render("cp_debug_on") );
    
      $Tin->tr("axbglite");
      $Tin->td( "Classes of output to show:", "axfg");
      $Tin->td_alignment("", "top");
      $Fdebugcl = new form_combofield();
      $Fdebugcl->multiselect = true;
      $Fdebugcl->set_size(6);
      $Fdebugcl->setstyle("width:$cwidth");
      $Fdebugcl->settitle(
            "This multiple-select box allows you to choose which classes of debug output "
            . "are displayed. Eg. If you only want your ad-hoc 'debugbr()' statements to be "
            . "output, then choose 'User diagnostics'."
          );
      $Fdebugcl->set_onchange('setchgd()');
      $Fdebugcl->additem(2,   "User diagnostics (default)");
      $Fdebugcl->additem(4,   "SQL statements");
      $Fdebugcl->additem(8,   "All SQL data (verbose)");
      $Fdebugcl->additem(16,  "Dump of GET/POST vars etc.");
      $Fdebugcl->additem(32,  "Include traceback info");
      $Fdebugcl->additem(64,  "Show table outlines");
      $Fdebugcl->additem(128, "Execution profiler");
      $Fdebugcl->additem(256, "Authentication");
      $Fdebugcl->additem(1,   "System diagnostics");
      // Build value as array of set bits..
      $debugcl = $app->getparameter("debug_classes", "debug_classes");
      $debug_value = array();
      for ($i=1; $i < 512; $i*=2) {
        if ($debugcl & $i) {
          $debug_value[] = $i;
        }
      }
      $Fdebugcl->setvalue($debug_value);
      $Tin->td( $Fdebugcl->render("cp_debug_classes") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Output modes:", "axfg");
      $Tin->td_alignment("", "top");
      $Fdebugop = new form_combofield();
      $Fdebugop->multiselect = true;
      $Fdebugop->set_size(6);
      $Fdebugop->set_onchange('setchgd()');
      $Fdebugop->setstyle("width:$cwidth");
      $Fdebugop->settitle(
            "This determines where the debugging goes. Standard output is displayed in "
            . "the webpage, at the top - this is generally the most useful in a website "
            . "with buffered output. Another useful option is to send it to the system "
            . "log."
          );
      $Fdebugop->additem(1, "Standard (default)");
      $Fdebugop->additem(2, "Unbuffered echo");
      $Fdebugop->additem(4, "CLI output (non-web mode)");
      $Fdebugop->additem(8, "To system logfile");
      // Build value as array of set bits..
      $debugop = $app->getparameter("debug_output", "debug_output");
      $debugop_value = array();
      for ($i=1; $i < 512; $i*=2) {
        if ($debugop & $i) {
          $debugop_value[] = $i;
        }
      }
      $Fdebugop->setvalue($debugop_value);
      $Tin->td( $Fdebugop->render("cp_debug_output") );
      
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );

      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Diagnostics</b>", "axsubhdg");
      
      $Tin = new table("debug_diags");
      $Tin->setpadding(2);
    
      $bg = "axbgdark";
      integerField(
          "SQL Execution log threshold:",
          "SQL_EXEC_THRESHOLD",
          $app->globals,
          60000, 80,
          "Use this to detect queries which are taking much longer than they "
          . "should to complete."
          );
      infoField(
          "SQL queries exeeding the specified number of milliseconds will "
        . "be logged in the system log. To disable, set to zero."
          );
    
      $resptimer = $app->getparameter("response_timer", "response_timer");
      $mychkbox = $chkbox;
      $mychkbox->checked = $resptimer;
      $mychkbox->settitle(
            "Use this option to find out how long your website pages are taking to "
            . "render to the user-agent. Useful for separating render time from network "
            . "transit time."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Enable response time logging:", "axfg" );
      $Tin->td( $mychkbox->render("cp_response_timer") );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      $Tapp->tr("axfoot");
      $Tapp->td("&nbsp;", "axfoot");
      break;

    // ......................................................................
    // DEFAULT SETTINGS
    default:
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Identification</b>", "axsubhdg");
      
      $Tin = new table("definitions");
      $Tin->setpadding(2);

      entryField(
          "Application Name:",
          "APP_NAME",
          $app->definitions,
          "This is the 'nice' name for your website. Usually a single word, but can be "
          . "more then one. It is used in areas such as e-mails, and error messages."
          );
      entryField(
          "Application Prefix:",
          "APP_PREFIX",
          $app->definitions,
          "A single word with no spaces or hyphens. This should uniquely identify your "
          . "website on the local machine. This value is also used with Axyl Lucene "
          . "to set your indexing 'domain'."
          );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // GLOBALS
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Global variables</b>", "axsubhdg");
    
      $Tin = new table("globals");
      $Tin->setpadding(2);
      $Tin->tbody("fmlook");
      $bg = "axbgdark";
      entryField(
          "Templates directory:",
          "TEMPLATESDIR",
          $app->globals,
          "The directory Axyl searches for templates. Should be in your website "
          . "directory hierarchy. Accessible as global var \$TEMPLATESDIR"
          );
      entryField(
          "Images directory:",
          "IMAGESDIR",
          $app->globals,
          "The directory Axyl searches for images. Should be in your website "
          . "directory hierarchy. Accessible as global var \$IMAGESDIR"
          );
      entryField(
          "Cached files directory:",
          "CACHEDIR",
          $app->globals,
          "The directory Axyl uses to cache pages which you have designated as "
          . "being cacehable."
          );
      entryField(
          "Media catalog directory:",
          "CATALOGDIR",
          $app->globals,
          "The directory Axyl stores media uploaded to your Media Catalog. "
          . "Accessible as global var \$CATALOGDIR"
          );
      entryField(
          "Managed content directory:",
          "CMDIR",
          $app->globals,
          "This is a directory hierarchy which contains the content-managed pages "
          . "created by users."
          );
      entryField(
          "Includes directory:",
          "INCDIR",
          $app->globals,
          "This is a directory you store your application 'include' files. It is "
          . "then accessible to you as global var \$INCDIR."
          );
      infoField(
          "NB: all directories specified above should be relative to the "
        . "website root directory. Additionally, if they are to be writeable "
        . "then they should be under the 'var' subdirectory."
          );
      entryField(
          "Webmaster name:",
          "WEBMASTER_PERSON",
          $app->globals,
          "Name of the person who looks after the website. Mainly used in system-generated "
          . "e-mails and messages."
          );
      entryField(
          "Webmaster e-mail:",
          "WEBMASTER_EMAIL",
          $app->globals,
          "The e-mail address of the person named above."
          );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // DTD & ENCODING
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Default DTD and website encoding</b>", "axsubhdg");
    
      $Tin = new table("dtd_enc");
      $Tin->setpadding(2);
    
      $cboHTMLDTD = new form_combofield();
      $cboHTMLDTD->setclass("axcombo");
      $cboHTMLDTD->set_onchange('setchgd()');
      $cboHTMLDTD->settitle(
          "The default site-wide Document Type Definition to be generated for each HTML page. "
          . "This value may be overridden by templates, or in specific pages by your "
          . "code."
          );
      $cboHTMLDTD->additem("", "None");
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">"),
                "HTML 3.2 Strict"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">"),
                "HTML 4.01 Transitional"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">"),
                "HTML 4.01 Strict"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\">"),
                "HTML 4.01 with Frameset"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\">"),
                "XHTML 1.0 Transitional"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\">"),
                "XHTML 1.0 Strict"
                );
      $cboHTMLDTD->additem(
                rawurlencode("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD XHTML 1.0 Frameset//EN\">"),
                "XHTML 1.0 with Frameset"
                );
      $cboHTMLDTD->setvalue($app->getparameter("dtd", "dtd", "html"));
    
      $cboWMLDTD = new form_combofield();
      $cboWMLDTD->setclass("axcombo");
      $cboWMLDTD->set_onchange('setchgd()');
      $cboWMLDTD->settitle(
          "The default site-wide Document Type Definition to be generated for each WML page. "
          . "This value may be overridden by templates, or in specific pages by your "
          . "code."
          );
      $cboWMLDTD->additem("", "None");
      $cboWMLDTD->additem(
                rawurlencode("<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1_1.xml\">"),
                "WML 1.1"
                );
      $cboWMLDTD->additem(
                rawurlencode("<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.2//EN\" \"http://www.wapforum.org/DTD/wml12.xml\">"),
                "WML 1.2"
                );
      $cboWMLDTD->setvalue($app->getparameter("dtd", "dtd", "wml"));
    
      $cboENC = new form_combofield();
      $cboENC->setclass("axcombo");
      $cboENC->set_onchange('setchgd()');
      $cboENC->settitle(
          "The encoding for pages generated by this website. This determines the 'charset=' "
          . "portion of the 'Content-Type:' header which gets ssent for each webpage."
          );
      $cboENC->additem("ISO-8859-1", "ISO-8859-1 Latin 1 (Western)");
      $cboENC->additem("US-ASCII",   "US-ASCII");
      $cboENC->additem("UTF-8",      "UTF-8 (Unicode)");
      $setting = $app->get_setting("encoding");
      $cboENC->setvalue($app->getparameter("encoding", "encoding"));
      if ($app->getparameter("multilang", "multilang")) {
        $cboENC->disabled = true;
      }
    
      $Tin->tr("axbgdark");
      $Tin->td( "DTD for HTML content:", "axfg" );
      $Tin->td( $cboHTMLDTD->render("cp_dtd_html") );
    
      $Tin->tr("axbglite");
      $Tin->td( "DTD for WAP content:", "axfg" );
      $Tin->td( $cboWMLDTD->render("cp_dtd_wml") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Character encoding:", "axfg" );
      $Tin->td( $cboENC->render("cp_encoding") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("multilang", "multilang");
      $mychkbox->set_onclick("setUTF8mode(this.checked)");
      $mychkbox->settitle(
          "If you need to render multiple character sets in UTF-8 on your website then "
          . "check this option, and select Unicode (UTF-8) above. This will cause Axyl "
          . "to do some special string handling (using mb_output_handler), and also "
          . "produce a 'lang=' attribute in the &LT;html&GT; tag, plus a 'Content-Language' "
          . "meta tag for each language used in the webpage."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Website uses multiple languages:", "axfg" );
      $Tin->td( $mychkbox->render("cp_multilang") . "&nbsp;(requires UTF-8 encoding above)" );
      $Tin->td_css("font-style:italic;font-size:80%");
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // SESSION
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Session settings</b>", "axsubhdg");
    
      $Tin = new table("session");
      $Tin->setpadding(2);
    
      $Tin->tr("axbgdark");
      $Tin->td( "Website HTTP hostname:", "axfg" );
      $http_host = $app->getparameter("http_host", "http_host");
      if ($http_host == "") {
        $http_host = "(default to webserver)";
      }
      $mybox = $tbox;
      $mybox->setvalue($http_host);
      $mybox->settitle(
          "This allows you to specify a different hostname from the local webserver. Normally "
          . "this isn't required, however some architectures require Axyl to run through a "
          . "proxy and hence cookies and webpages etc. are identified as coming from that machine "
          . "rather than the local one."
          . ""
          );
      $Tin->td( $mybox->render("cp_http_host") );
      $Tin->tr("axbgdark");
      $Tin->td();
      $Tin->td(
          "Set to blank, or '(default to webserver)' to get the default "
        . "webserver hostname. Otherwise set your own.",
          "axfg"
          );
      $Tin->td_css("font-style:italic;font-size:80%");
      $Tin->set_width_profile("50%,50%");
    
      $Tin->tr("axbglite");
      $Tin->td( "Cookie name:", "axfg" );
      $cookiename = $app->getparameter("cookiename", "cookiename");
      if ($cookiename == "") {
        $cookiename = $app->definitions["APP_PREFIX"] . "_session_id";
      }
      $mybox = $tbox;
      $mybox->setvalue($cookiename);
      $mybox->settitle(
          "If you don't like the default Axyl name for your cookie, then just "
          . "take the opporunity to change it here!"
          );
      $Tin->td( $mybox->render("cp_cookiename") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Cookie/session lifetime:", "axfg" );
      $Flife = new form_combofield();
      $Flife->setclass("axcombo");
      $Flife->setstyle("width:$cwidth");
      $Flife->set_onchange('setchgd()');
      $Flife->additem(-1, "Until browser closed");
      $Flife->additem(315360000, "Forever and a day");
      $Flife->additem(31536000, "A year");
      $Flife->additem(2592000, "A month");
      $Flife->additem(604800, "A week");
      $Flife->additem(86400, "24 hours");
      $Flife->additem(43200, "12 hours");
      $Flife->additem(28800, "8 hours");
      $Flife->additem(14400, "4 hours");
      $Flife->additem(3600, "An hour");
      $Flife->additem(1200, "20 minutes");
      $Flife->additem(0, "Immediate expiry");
      $Flife->setvalue($app->getparameter("lifetime", "lifetime"));
      $Flife->settitle(
          "This determines how long the session cookie is valid. The most common setting "
          . "is 'Until browser closed' which requires a login each time. This can be "
          . "overridden to be 'forever' if the login process contains a '\$chkRememberMe' "
          . "defined in the form submit."
          );
      $Tin->td( $Flife->render("cp_lifetime") );
    
      $Tin->tr("axbglite");
      $Tin->td( "Page expiry (seconds):", "axfg" );
    
      $Fexpiry = new form_combofield();
      $Fexpiry->setclass("axcombo");
      $Fexpiry->setstyle("width:$cwidth");
      $Fexpiry->set_onchange('setchgd()');
      $Fexpiry->additem(-1, "Immediate (dynamic content)");
      $Fexpiry->additem(60, "1 minute");
      $Fexpiry->additem(120, "2 minutes");
      $Fexpiry->additem(180, "3 minutes");
      $Fexpiry->additem(240, "4 minutes");
      $Fexpiry->additem(300, "5 minutes");
      $Fexpiry->additem(600, "10 minutes");
      $Fexpiry->additem(1800, "30 minutes");
      $Fexpiry->additem(3600, "1 hour");
      $Fexpiry->additem(14400, "4 hours");
      $Fexpiry->additem(28800, "8 hours");
      $Fexpiry->additem(86400, "24 hours");
      $Fexpiry->additem(315360000, "Never (static content)");
      $Fexpiry->setvalue($app->getparameter("expiry", "expiry"));
      $Fexpiry->settitle(
          "This sets the expiry of content sent to the browser. For pages which are "
          . "dynamically generated, this is normally 'Immediate', however this does "
          . "have the effect of negating the user's 'Back' button. A compromise is to "
          . "set this to a small value, such as 1 minute."
          );
      $Tin->td( $Fexpiry->render("cp_expiry") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("guest_browser_lifetime", "guest_browser_lifetime");
      $mychkbox->settitle(
          "Non-logged-in users in Axyl get allocated a 'guest' session cookie. This option will "
          . "set the lifetime of that cookie to the life of their browser session. To be honest "
          . "there isn't much difference either way!"
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Guest cookies browser lifetime:", "axfg" );
      $Tin->td( $mychkbox->render("cp_guest_browser_lifetime") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("session_track_logins", "session_track_logins");
      $mychkbox->settitle(
          "If checked then Axyl will keep track of each user's logins, counting them as they "
          . "log in each time. Unchecking removes that small processing overhead."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Count user login sessions:", "axfg" );
      $Tin->td( $mychkbox->render("cp_session_track_logins") );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // CONTENT
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>Content settings</b>", "axsubhdg");
    
      $Tin = new table("content");
      $Tin->setpadding(2);
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("metadata_enabled", "metadata_enabled");
      $mychkbox->settitle(
          "If you installed the Axyl Metadata Extension when you created the website, then "
          . "you can enable it here. This gives extra functionality in the Axyl Content "
          . "Managed Layouts for defining metadata for webpages."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Enable metadata edit/generation:", "axfg" );
      $Tin->td( $mychkbox->render("cp_metadata_enabled") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("microsites_enabled", "microsites_enabled");
      $mychkbox->set_onchange(
                "if(this.checked) "
              . "alert("
              . "'NOTICE:\\n\\n"
              . "For microsite creation to work you must be running \'pg-microsites-installer.php\' from cron.\\n"
              . "The crontab for this can be found in the \'scripts/cron\' sub-directory of your Axyl installation,\\n"
              . "and should have been automatically installed into /etc/cron.d by your Debian package.\\n\\n"
              . "')"
                );
      $mychkbox->settitle(
          "If you installed the Axyl Microsites Extension when you created the website, then "
          . "you can enable it here. This gives you some extra functions to create and maintain "
          . "microsites of the main website."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Enable microsite(s) creation:", "axfg" );
      $Tin->td( $mychkbox->render("cp_microsites_enabled") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("buffered_output", "buffered_output");
      $mychkbox->settitle(
          "Whether or not to run the website using buffered Php output. Buffering allows Axyl "
          . "to collect output, post-process it, and render it in one hit just before sending "
          . "it to the browser. Non-buffered output severely restricts debugging, and other "
          . "post-processing and is not recommended."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Buffered output (recommended):", "axfg" );
      $Tin->td( $mychkbox->render("cp_buffered_output") );
    
      $Tin->tr("axbgdark");
      $Tin->td( "Compression type:", "axfg" );
      $Fcomp = new form_combofield();
      $Fcomp->setclass("axcombo");
      $Fcomp->setstyle("width:$cwidth");
      $Fcomp->set_onchange('setchgd()');
      $Fcomp->additem(0, "No compression");
      $Fcomp->additem(1, "Built-in compression (Php >= 4.0.4)");
      $Fcomp->additem(2, "Axyl custom compression");
      $Fcomp->setvalue($app->getparameter("compression_type", "compression_type"));
      $Fcomp->settitle(
          "Axyl can compress the output stream to save transmission time for large-ish "
          . "webpages. The recommended option is 'Built-in' compression."
          );
      $Tin->td( $Fcomp->render("cp_compression_type") );
    
      $Tin->tr("axbglite");
      $Tin->td( "Compression threshold:", "axfg" );
      $Fcomp = new form_combofield();
      $Fcomp->setclass("axcombo");
      $Fcomp->setstyle("width:$cwidth");
      $Fcomp->set_onchange('setchgd()');
      $Fcomp->additem(0, "None (compress all content)");
      $Fcomp->additem(1024, "Over 1Kb");
      $Fcomp->additem(4096, "Over 4Kb");
      $Fcomp->additem(8192, "Over 8Kb");
      $Fcomp->additem(16384, "Over 16Kb");
      $Fcomp->additem(32768, "Over 32Kb");
      $Fcomp->additem(65536, "Over 64Kb");
      $Fcomp->additem(262144, "Over 256Kb");
      $Fcomp->setvalue($app->getparameter("compression_threshold", "compression_threshold"));
      $Fcomp->settitle(
          "On some systems you might want to save processing power by only compressing "
          . "pages above a certain size."
          );
      $Tin->td( $Fcomp->render("cp_compression_threshold") );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
    
      // ......................................................................
      // GET/POST settings
      $Tapp->tr("axsubhdg");
      $Tapp->td("<b>GET/POST settings</b>", "axsubhdg");
    
      $Tin = new table("getpost");
      $Tin->setpadding(2);
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("keep", "keep");
      $mychkbox->settitle(
          "This option causes Axyl to set a second browser cookie. You can then use Php "
          . "session management to keep track of variables across webpages using Axyl's "
          . "'remember()' method."
          );
      $Tin->tr("axbgdark");
      $Tin->td( "Enable Axyl KEEP feature:", "axfg" );
      $Tin->td( $mychkbox->render("cp_keep") );
    
      $mychkbox = $chkbox;
      $mychkbox->checked = $app->getparameter("globalise", "globalise");
      $mychkbox->settitle(
          "When checked this causes Axyl to auto globalise variables submitted to the "
          . "website. This circumvents any php.ini setting which turns off globals."
          );
      $Tin->tr("axbglite");
      $Tin->td( "Auto-globalise all GET/POST vars:", "axfg" );
      $Tin->td( $mychkbox->render("cp_globalise") );
    
      $Tin->set_width_profile("50%,50%");
      $Tapp->tr();
      $Tapp->td( $Tin->render() );
      
  } // switch
  
  $cprf = new img("$LIBDIR/img/_cpfootr.gif", "", "", 87, 23);
  $Tin = new table();
  $Tin->tr();
  $Tin->td();
  $Tin->td_css("background: url('$LIBDIR/img/_cpfootfill.gif')");
  $Tin->td_width("100%");     
  $Tin->td($cprf->render());
  $Tin->td_alignment("right");
  $Tapp->tr();
  $Tapp->td($Tin->render());

} // if no errors

// ----------------------------------------------------------------------
// Finish and return the page..
$hidcpview = new form_hiddenfield("cp_view", $cp_view);
$s .= "<form name=\"$formname\" method=\"post\">\n";
$s .= $Tapp->render();
$s .= $hidcpview->render();
// Also render maintainer bits if database view..
if ($cp_view == CP_VIEW_DB) {
  $s .= $maintainer->render();
}
else {
  $hid = new form_hiddenfield("_recmaintpost_form", $formname);
  $s .= $hid->render();
}
$s .= "</form>\n";

//echo $app->htmldump();

if ($cp_view == CP_VIEW_AUTH) {
  $s .= "<script language=\"javascript\">\n"
      . "var cbo = eval('document.forms.$formname.cp_remote_auth_source');\n"
      . "if (cbo) {\n"
      . "  control_auth_fields(cbo,'$formname');\n"
      . "}\n"
      . "</script>\n";  
}
$s .= "</body>\n";
$s .= "</html>\n";
echo $s;
// ----------------------------------------------------------------------
?>