<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    timer-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a simple micro-timer                    */
/*                                                                      */
/* ******************************************************************** */
/** @package timer */

//-----------------------------------------------------------------------
/**
* The microtimer class
* A generic microtimer. This timer allows elapsed times to be
* measured down to microseconds in theory, although depending
* on how 'real-time' the OS is, there may well be limitations.
* The microtimer works like a stopwatch. It is either ticking
* or it is stopped. It has a number of seconds on the clock
* at any one instant. It may be started, stopped or read.
* @package datetime
*/
class microtimer {
  // Public
  /** Current number of seconds on the timer */
  var $seconds = 0;
  /** True if the timer is ticking */
  var $ticking = false;

  // Private
  /** Reference time to base elapsed interval on
      @access private */
  var $reference = 0;
  // .....................................................................
  /**
  * Constructor
  * Create a new microtimer.
  */
  function microtimer() {
    $this->reset();
  } // microtimer
  // .....................................................................
  /**
  * Return the current microtime
  * @return integer Seconds on the system clock right now
  * @access private
  */
  function get_microtime() {
    list($usec, $sec) = explode(" ", microtime());
    return (float)$sec + (float)$usec;
  } // get_microtime
  // .....................................................................
  /**
  * Update the timer values
  * Updates the internal seconds on the clock.
  * @access private
  */
  function update() {
    $seconds = $this->get_microtime();
    if ($this->ticking) {
      $delta = $seconds - $this->reference;
      $this->seconds += $delta;
    }
    $this->reference = $seconds;
  } // update
  // .....................................................................
  /**
  * Start the timer. Starts the timer ticking. If it was already
  * ticking then there is no change in status.
  */
  function start() {
    $this->update();
    $this->ticking = true;
  } // start
  // .....................................................................
  /**
  * Stop the timer
  * This freezes the internal seconds value, and stops the timer
  * from ticking.
  */
  function stop() {
    $this->update();
    $this->ticking = false;
  } // stop
  // .....................................................................
  /**
  * Reset the timer. This zeroes the timer. If it was ticking it goes
  * on ticking, but starting from zero.
  */
  function reset() {
    $this->update();
    $this->seconds = 0;
  } // reset
  // .....................................................................
  /**
  * Restart the timer. This zeroes the timer and then starts it. Has
  * same effect as reset() but also makes sure the timer is ticking
  * after resetting internal timer values.
  */
  function restart() {
    $this->reset();
    $this->start();
  } // restart
  // .....................................................................
  /**
  * Returns the seconds on the clock as a floating point number.
  * @return float Seconds on the clock at the time of this call.
  */
  function secs() {
    $this->update();
    return $this->seconds;
  } // secs
  // .....................................................................
  /**
  * Returns the milli-seconds on the clock.
  * @return float Milli-seconds on the clock at the time of this call.
  */
  function millisecs() {
    $this->update();
    return $this->seconds * 1000;
  } // millisecs
  // .....................................................................
  /**
  * Returns the micro-seconds on the clock.
  * @return float Micro-seconds on the clock at the time of this call.
  */
  function microsecs() {
    $this->update();
    return $this->seconds * 1000000;
  } // microsecs
  // .....................................................................
  /**
  * Return the time on the clock in a nice 'human' format. This is in
  * the form of a string, viz: '14d 2h 12m 13s'.
  * @return string The time on the clock in 'human' format.
  */
  function formatted_time() {
    $this->update();
    return nicetime($this->secs());
  } // formatted_time
  // .....................................................................
  /**
  * Return the seconds on the clock to 2 decimal places.
  * Note that this is a numeric string which includes commas for the
  * thousands. It is therefore not a float suitable for arithmetic.
  * @return string Formatted number: seconds on the clock to 2 decimal places.
  */
  function formatted_secs() {
    $this->update();
    return number_format($this->secs(), 2);
  } // formatted_secs
  // .....................................................................
  /**
  * Return the milli-seconds on the clock to 2 decimal places.
  * Note that this is a numeric string which includes commas for the
  * thousands. It is therefore not a float suitable for arithmetic.
  * @return string Formatted number: milli-seconds on the clock to 2 decimal places.
  */
  function formatted_millisecs() {
    $this->update();
    return number_format($this->millisecs(), 2);
  } // formatted_millisecs
  // .....................................................................
  /**
  * Return the micro-seconds on the clock to 2 decimal places.
  * Note that this is a numeric string which includes commas for the
  * thousands. It is therefore not a float suitable for arithmetic.
  * @return string Formatted number: micro-seconds on the clock to 2 decimal places.
  */
  function formatted_microsecs() {
    $this->update();
    return number_format($this->microsecs(), 2);
  } // formatted_microsecs
} // timer class

//-----------------------------------------------------------------------
?>