<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    search-query-defs.php                                   */
/* Author:      Paul Waite                                              */
/* Description: Definitions for interfacing to a search engine to       */
/*              perform search queries.                                 */
/*              The parent module which includes this module must also  */
/*              include the underlying defs module for the search       */
/*              engine itself, eg: lucene, solr etc.                    */
/*                                                                      */
/* ******************************************************************** */
/** @package search */

/** Generic search classes */
include_once("search-defs.php");

// ----------------------------------------------------------------------
/**
* The SearchEngine search class
* This class inherits the functionality of the generic 'search' class. It
* extends it to implement a SearchEngine search. Use the methods in this class
* as the mainstay in implementing queries of content from SearchEngine. Most
* methods, such as match(), matchfield(), matchrange() etc. store the
* requirement in the class for subsequent building using the set_*()
* methods of the SearchEngine classes to set the relevant fields. This is only
* done when you call execute(), and the query is built from all the
* composite terms you have added via match() et al.
* @package search
*/
class searchengine_search extends searchengine_querymsg {
  // .....................................................................
  /**
  * Constructor
  * Create a new SearchEngine search
  * @param string $application Application name/domain name for searching in
  * @param string $host Hostname or IP of SearchEngine server
  * @param string $port Port of SearchEngine server
  */
  function searchengine_search($application="?", $host="", $port="") {
    global $RESPONSE;
    $this->search();
    $this->searchengine_querymsg($application, $host, $port);
    $this->initialise();
  } // searchengine_search
  // .....................................................................
  /**
  * Add a new search term to match. Search terms can be a single word or
  * compound patterns, Each time one of these is added, it has an operator
  * associated with it - whether this term is a "may have" (OR), or a
  * "must have" (AND) term.
  * NB: This method overrides the parent method in order to ensure that all
  * boolean logic terms are in upper case as SearchEngine requires.
  * @param string  $term  Search term text to match.
  * @param integer $op    Joining operator: 'AND', 'OR', 'NOT, 'AND NOT'.
  * @param string  $id    An optional ID to associate with this search term.
  * @param numeric $boost Boost factor. Can be a fraction eg. 0.2, or integer 1,2,3..
  */
  function match($term, $op="OR", $id="", $boost="") {
    $LCops = array("/ and /","/ or /","/ not /");
    $UCops = array(" AND "," OR "," NOT ");
    $term = preg_replace($LCops, $UCops, $term);
    if ($boost != "") $term .= "^$boost";
    search::match($term, strtoupper($op), $id);
  } // match
  // .....................................................................
  /**
  * Add search term to match a field value.
  * This is used to add a search term which defines the value that a given
  * field may or may not contain for the search to succeed.
  * For adding terms which are 'free' (as a user might type into a search
  * box for example) then you can use the match() method which this class
  * inherits from the search class.
  * @param string  $fieldname Name of field to reference in the index
  * @param mixed   $fieldvalue Value or array of values, for field to match
  * @param string  $op Operator to join this term to others in the query
  * @param string  $id Optional identity tag for this term
  * @param numeric $boost Boost factor. Can be a fraction eg. 0.2, or integer 1,2,3..
  */
  function matchfield($fieldname, $fieldvalue, $op="OR", $id="", $boost="") {
    debug_trace($this);
    if (!isset($fieldvalue)) return;
    if (!is_array ($fieldvalue)) {
      $fieldvalue = array($fieldvalue);
    }
    $term = "";
    foreach ($fieldvalue as $value) {
      $value = trim($value);
      if ($value != "") {
        $term .= " OR " . $this->fieldterm($fieldname, $value);
      }
    }
    if ($term != "") {
      $term = substr($term, 4); // Get rid of initial OR
      // Call parent function to register the search term..
      $this->match($term, strtoupper($op), $id, $boost);
    }
    debug_trace();
  } // matchfield
  // .....................................................................
  /**
  * Helper function to build field search term
  * @param string $fieldname Name of field to reference in the index
  * @param string $fieldvalue Value of field to match
  * @access private
  */
  function fieldterm($fieldname, $fieldvalue) {
    if ($fieldname != DEFAULT_FIELD) {
      $term = "$fieldname:$fieldvalue";
    }
    else {
      $term = $fieldvalue;
    }
    return $term;
  } // fieldterm
  // .....................................................................
  /**
  * Add search term to match a field value range.
  * This is used to add a search term which defines the range of values that
  * a given field may or may not contain for the search to succeed.
  * NB: This method is always a must match (implied AND) search term. In
  * other words the search is always restricted/refined by it.
  * @param string $fromvalue Lower range value of field to match
  * @param string $tovalue Upper range value of field to match
  * @param string $fieldname Name of field, defaulted to 'Text'
  */
  function matchrange($fromvalue, $tovalue, $fieldname) {
    debug_trace($this);
    $this->set_range($fromvalue, $tovalue, $fieldname);
    debug_trace();
  } // matchrange
  // .....................................................................
  /**
  * Add search term: must match a field value.
  * This is used to add a search term which defines the value that a given
  * field must contain for the search to succeed.
  * @param string $fieldname Name of field
  * @param string $fieldvalue Value of field to match
  * @param string $id Optional identity tag for this term
  * @param numeric $boost Boost factor. Can be a fraction eg. 0.2, or integer 1,2,3..
  */
  function must_matchfield($fieldname, $fieldvalue, $id="", $boost="") {
    $this->matchfield($fieldname, $fieldvalue, "AND", $id, $boost);
  } // must_matchfield
  // .....................................................................
  /**
  * Add search term: may match a field value.
  * This is used to add a search term which defines the value that a given
  * field may contain for the search to succeed.
  * @param string $fieldname Name of field
  * @param string $fieldvalue Value of field to match
  * @param string $id Optional identity tag for this term
  * @param numeric $boost Boost factor. Can be a fraction eg. 0.2, or integer 1,2,3..
  */
  function may_matchfield($fieldname, $fieldvalue, $id="", $boost="") {
    $this->matchfield($fieldname, $fieldvalue, "OR", $id, $boost);
  } // may_matchfield
  // .....................................................................
  /**
  * Add search term: must not match a field value.
  * This is used to add a search term which defines the value that a given
  * field must not contain for the search to succeed.
  * @param string $fieldname Name of field
  * @param string $fieldvalue Value of field to match
  * @param string $id Optional identity tag for this term
  * @param numeric $boost Boost factor. Can be a fraction eg. 0.2, or integer 1,2,3..
  */
  function does_not_matchfield($fieldname, $fieldvalue, $id="", $boost="") {
    $this->matchfield($fieldname, $fieldvalue, "NOT", $id, $boost);
  } // does_not_matchfield
  // .....................................................................
  /**
  * Define the query text directly, without recourse to any of this class's
  * high-level query-building methods. This method simply assigns the class
  * variable 'query' to what is passed, assuming that this string is a
  * ready-built query string that the underlying SearchEngine will be
  * able to understand and execute.
  * @param string $raw_query_text The ready-built query string to use
  */
  function rawquery($raw_query_text) {
    $this->query = $raw_query_text;
  } // rawquery
  // .....................................................................
  /**
  * Execute the search
  * Here we execute a SearchEngine search, overriding the method in the parent
  * class. This involves building the query string, sending it to the
  * SearchEngine server, and receiving the search results back.
  * @param integer $timeoutsecs Override for timeout in seconds
  */
  function execute($timeoutsecs="") {
    debug_trace($this);

    // The queryvalid() method is in the parent class 'search', and
    // calls the build() method in the same class. The build() method is
    // a raw routine to join together the search terms with ANDs and
    // ORs. You may have to override it for SearchEngine. If so, just create
    // a new build() method in this class.

    if ($this->queryvalid()) {

      // This method must be defined in the underlying SearchEngine. It
      // should set up the query as it should be for that interface - eg.
      // it might enclose the query text in XML <query> tags.
      $this->set_query($this->query);

      // Set limit, offset..
      if ($this->max_results > 0) {
        $this->set_limit($this->max_results);
        if ($this->skip_results > 0) {
          $this->set_first($this->skip_results);
        }
      }

      // Set any daterange..
      if ($this->has_daterange()) {
        $this->set_range($this->date_start, $this->date_end, $this->date_fieldname);
      }

      // Send to SearchEngine..
      $this->send($timeoutsecs);

      // Flag that we did it..
      $this->executed = true;
      debugbr("SearchEngine search: exec ok: returning " . $this->hitcount() . " hits", DBG_DEBUG);
    }
    else {
      debugbr("SearchEngine search: invalid query: '$this->query'", DBG_DEBUG);
    }
    debug_trace();
  } // execute
} // searchengine_search class

?>