<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    block-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for content block management in webpages.   */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Record maintainer */
include_once("recmaint-defs.php");

// ......................................................................
// GLOBAL GENERIC WIDGETS

// Common width settings for form elements..
global $fullwidth;
global $quartwidth;
global $thirdwidth;
global $halfwidth;
global $threeqwidth;

global $awidth;   // Content
global $ewidth;   // Blocklet list, desc
global $cwidth;   // Media/doc combos (wide)
global $cbowidth; // All property combos
global $bigwidth; // Very wide widgets

$fullwidth   = 540;
$quartwidth  = ceil($fullwidth * 0.25);
$thirdwidth  = ceil($fullwidth * 0.37);
$halfwidth   = ceil($fullwidth * 0.50);
$threeqwidth = ceil($fullwidth * 0.75);

$awidth   = $fullwidth   . "px";
$ewidth   = $halfwidth   . "px";
$cwidth   = $thirdwidth  . "px";
$cbowidth = $quartwidth  . "px";
$bigwidth = $threeqwidth . "px";

// Generic add button..
global $gen_addbtn;
$gen_addbtn = new form_imagebutton("_add_link", "", "", "$LIBDIR/img/_add.gif", "Add link to content", 57, 15);
$gen_addbtn->setalt("Add");

global $gen_blocklettype;
$gen_blocklettype = new form_combofield("blocklet_type", "", $b->type);
$gen_blocklettype->setclass("axtxtbox");
$gen_blocklettype->setstyle("width:$cbowidth;");
$gen_blocklettype->additem("text",    "Text");
$gen_blocklettype->additem("list",    "List");
$gen_blocklettype->additem("ordered", "Ordered list");
$gen_blocklettype->additem("bullets", "Bullet points");
$gen_blocklettype->additem("table",   "Table");
$gen_blocklettype->additem("raw",     "Unformatted");

// An Axyl colour palette combobox..
$ss = new stylesheet($RESPONSE->site_docroot . $RESPONSE->head->stylesheet);
global $colourCombo;
$colourCombo = new form_combofield("colours");
$colourCombo->setclass("axcombo");
$colourCombo->setstyle("width:$cbowidth;");
$colourCombo->additem("", "default colour");
$TotColours = defaulted($ss->style("axylpalette", "total_colours"), 0);
if ($TotColours > 0) {
  for ($c = 1; $c <= $TotColours; $c++) {
    $colour_style = $ss->style("axylpalette", "colour_$c");
    $colour_bits = explode(",", $colour_style);
    if (isset($colour_bits[0]) && isset($colour_bits[1])) {
      $colourCombo->additem($colour_bits[0], $colour_bits[1]);
    }
  }
}
// An Axyl table styles combobox..
global $tstylesCombo;
$tstylesCombo = new form_combofield("table_style");
$tstylesCombo->setclass("axcombo");
$tstylesCombo->setstyle("width:$cbowidth;");
$tstylesCombo->additem("", "default style");
$Totstyles = defaulted($ss->style("axyl_tablestyles", "total_styles"), 0);
if ($Totstyles > 0) {
  for ($c = 1; $c <= $Totstyles; $c++) {
    $tstyle_style = $ss->style("axyl_tablestyles", "style_$c");
    $tstyle_bits = explode(",", $tstyle_style);
    if (isset($tstyle_bits[0]) && isset($tstyle_bits[1])) {
      $tstylesCombo->additem($tstyle_bits[0], $tstyle_bits[1]);
    }
  }
}
// Multi-language selector..
global $langsCombo;
if (isset($RESPONSE) && $RESPONSE->multilang) {
  $langsCombo = new form_combofield("language");
  $langsCombo->setclass("axcombo");
  $langsCombo->setstyle("width:$cbowidth;");
  $LQ = dbrecordset("SELECT * FROM ax_language ORDER BY display_order");
  $langsCombo->add_querydata($LQ, "lang_id", "lang_desc");
  $langsCombo->setvalue($this->language);
}
else {
  $langsCombo = new form_hiddenfield("language");
}

// ......................................................................
// Javascript which is always required for block content being edited..
global $blockmode, $_edit_x;
if ((isset($blockmode) && $blockmode == "editing" || $blockmode == "saving")
    ||(isset($_edit_x)) ) {
  $RESPONSE->head->add_scriptsrc("$LIBDIR/js/fieldvalidation.js");
  $RESPONSE->body->add_script(
      "function add_content(ctype,srcfld,dstfld,fm) {\n"
    . "  var s=''; var title='';\n"
    . "  var srcFld=eval('document.forms.'+fm+'.'+srcfld);\n"
    . "  var dstFld=eval('document.forms.'+fm+'.'+dstfld);\n"
    . "  if (dstFld != null && srcFld != null) {\n"
    . "    var ix=srcFld.selectedIndex;\n"
    . "    if (ix != -1) {\n"
    . "      var info=srcFld.options[ix].value.split('|');\n"
    . "      var id=info[0];\n"
    . "      if (id == '') return;\n"
    . "      var title=srcFld.options[ix].text;\n"
    . "      s += '<img type=\"axyl/embedded-media\" src=\"$LIBDIR/img/_axmedia.gif\" codetype=\"'\n"
    . "        + ctype.toLowerCase() + '\" id=\"'+id+'\" title=\"'+title+'\"';\n"
    . "      for(var i=1; i<=info.length; i++) {\n"
    . "        theFld=eval('document.forms.'+fm+'.'+info[i]);\n"
    . "        if (theFld != null) {\n"
    . "          var fname=info[i];\n"
    . "          if (fname.indexOf('_') != -1) {\n"
    . "            var bits=fname.split('_');\n"
    . "            fname=bits[1];\n"
    . "          }\n"
    . "          if (theFld.type != 'checkbox') {\n"
    . "            if (theFld.value != '') {\n"
    . "              val=theFld.value;\n"
    . "              val=val.replace(/ > /,' GT ');\n"
    . "              val=val.replace(/ < /,' LT ');\n"
    . "              s += ' '+fname+'=\"'+val+'\"';\n"
    . "            }\n"
    . "          }\n"
    . "          else {\n"
    . "            s += ' '+fname+'=\"';\n"
    . "            if (theFld.checked) s+= 'yes';\n"
    . "            else s+= 'no';\n"
    . "            s += '\"';\n"
    . "          }\n"
    . "        }\n"
    . "      }\n"
    . "      s += '>';\n"
    . "      if (dstfld=='wysiwyg_content') {wysiwyg_content_ed.focusEditor();wysiwyg_content_ed.insertHTML(s);}\n"
    . "      else {\n"
    . "        dstFld.value += ' '+s;\n"
    . "        dstFld.onchange();\n"
    . "      }\n"
    . "    }\n"
    . "  }\n"
    . "}\n"
  );
}

// ......................................................................
/**
* Anything which is editing and uses blocks has to have this Javascript
* available to it in the page when editing..
* @access private
*/
function insert_block_jscript() {
  static $done = false;
  global $RESPONSE, $blockmode, $_edit_x;
  if (!$done) {
    if ((isset($blockmode) && $blockmode == "editing" || $blockmode == "saving")
       ||(isset($_edit_x)) ) {
      $RESPONSE->body->add_script(
          "function add_link(fm) {\n"
        . "  var hrefFld = eval('document.forms.'+fm+'.link_href');\n"
        . "  var descFld = eval('document.forms.'+fm+'.link_desc');\n"
        . "  var remoFld = eval('document.forms.'+fm+'.link_remote');\n"
        . "  var contFld = eval('document.forms.'+fm+'.content');\n"
        . "  if (hrefFld != null && descFld != null && contFld != null && hrefFld.value != '') {\n"
        . "    var linkHtml = '<a href=\"' + hrefFld.value + '\"';\n"
        . "    if (remoFld.checked) linkHtml += ' target=\"_blank\"';\n"
        . "    linkHtml += '>';\n"
        . "    if (descFld.value == '') linkHtml += hrefFld.value;\n"
        . "    else linkHtml += descFld.value;\n"
        . "    linkHtml += '</a>';\n"
        . "    contFld.value += ' ' + linkHtml;\n"
        . "    contFld.onchange();\n"
        . "  }\n"
        . "}\n"
        . "function linkpop(combo,fm) {\n"
        . "  var hrefFld = eval('document.forms.'+fm+'.link_href');\n"
        . "  var descFld = eval('document.forms.'+fm+'.link_desc');\n"
        . "  var remoFld = eval('document.forms.'+fm+'.link_remote');\n"
        . "  if (combo.selectedIndex != null && combo.selectedIndex != -1 && hrefFld != null && descFld != null) {\n"
        . "    hrefFld.value = combo.options[combo.selectedIndex].value;\n"
        . "    descFld.value = combo.options[combo.selectedIndex].text;\n"
        . "    remoFld.checked=false;\n"
        . "  }\n"
        . "}\n"
        );
        $done = true;
    } // if editing or saving
  } // if not already done
} // insert_block_jscript

// ......................................................................
/**
* Block Editor
* A blockeditor is a utility class. It contains all of the methods
* required to edit a block, so that the block class can concentrate
* on the basics of block acquisition and display. The constructor
* of a blockeditor must be passed a reference to the block it is
* going to be providing editing services for.
* @package cm
*/
class blockeditor extends RenderableObject {
  // Public
  // Private
  /** The block we are providing
      editing services for
      @access private */
  var $block;
  // ....................................................................
  /**
  * Constructor
  * Create a new blockeditor object.
  * @param reference $block A reference to the block being edited
  */
  function blockeditor(&$block) {
    $this->block = $block;
  } // blockeditor
  // ....................................................................
  /**
  * Replicate this block into a new block with a new set of blocklets
  * as a complete content copy of this original block.
  * NOTE: We end up with this current block as the replicated one.
  */
  function replicate() {
    if ($this->block->exists) {
      $this->block->exists = false;
      $savedid = $this->block->blockid;
      $this->block->blockid = get_next_sequencevalue("seq_block_id", "ax_block", "block_id");

      // Prime the blocklets to be created anew..
      if (isset($this->block->blocklets)) {
        while (list($bid, $bkt) = each ($this->block->blocklets)) {
          $bkt->exists = false;
          $bkt->blockletid = NEW_BLOCKLET;
          $this->block->blocklets[$bid] = $bkt;
        }
      }
      // Create new block & blocklets..
      $this->block->put();
    }
  } // replicate
  // ....................................................................
  /**
  * Delete this block from the database. NB: we do not rely on RI to do
  * this since various versions of Postgres don't support this nicely.
  * All related entities are explicitly deleted in a transaction.
  */
  function delete() {
    debug_trace($this);
    $external_transaction = transaction_open();
    // Remove all blocklets one by one..
    if (!$external_transaction) {
      start_transaction();
    }
    $bb = dbrecordset("SELECT blocklet_id FROM ax_block_blocklet WHERE block_id=" . $this->block->blockid);
    if ($bb->hasdata) {
      do {
        $bid = $bb->field("blocklet_id");
        dbcommand("DELETE FROM ax_blocklet WHERE blocklet_id=$bid");
      } while ($bb->get_next());
    }
    // Remove linkages (may be none, if RI kicked in)..
    dbcommand("DELETE FROM ax_block_blocklet WHERE block_id=" . $this->block->blockid);
    // Finally remove block..
    dbcommand("DELETE FROM ax_block WHERE block_id=" . $this->block->blockid);

    if (!$external_transaction) {
      commit();
    }

    // Un-index from Lucene if applicable..
    if (transaction_succeeded()) {
      $this->block->unindex();
    }
    debug_trace();

  } // delete
  // ....................................................................
  /**
  * Render the Wysiwyg editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function wysiwyg_editform($lang_encoding, $lang_direction) {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    global $colourCombo, $tstylesCombo, $langsCombo;

    /** HTMLArea form element */
    include_once("htmlarea-defs.php");
    // Define which plugins to load..
    htmlarea_plugins("ContextMenu,CSS,TableOperations,ListType,CharacterMap");

    // This is a Wysiwyg block. If it is new, then we will need to
    // create its single associated blocklet..
    if (!isset($this->block->blocklets) || count($this->block->blocklets) == 0) {
      $this->block->add_blocklet();
      $this->block->put();
    }

    // Now grab the blocklet itself..
    $wysiwyg_blocklet = current($this->block->blocklets);

    // Buttons..
    $bsave = new form_imagebutton("_save",  "", "", "$LIBDIR/img/_save.gif", "Save changes",        57, 15);
    $bdone = new form_imagebutton("_done",  "", "", "$LIBDIR/img/_done.gif", "Exit without saving", 57, 15);

    // Control table..
    $Ted = new table("block_" . $this->block->blockid);
    $Ted->setpadding(2);

    // ..................................................................
    // Toolbar..
    $toolbar = array();
    $toolbar[] = $bdone;
    $toolbar[] = $bsave;
    $Tbar = new table("toolbar");
    $Tbar->tr("axtitle");
    switch ($this->block->layout_version) {
      case VERSION_PENDING:   $hdg = "PENDING";  break;
      case VERSION_LIVE:      $hdg = "LIVE";     break;
      case VERSION_PREVIOUS:  $hdg = "PREVIOUS"; break;
      case VERSION_UNDEFINED: $hdg = "EDITING";  break;
      default: $hdg = "Version: " . $this->block->layout_version;
    } // switch
    $Tbar->th("<b>$hdg</b> [B" . $this->block->blockid . "]", "axtitle");
    $tools = "";
    foreach ($toolbar as $tool) {
      $tools .= $tool->render();
    }
    $Tbar->th($tools, "axtitle");
    $Tbar->th_css("text-align:right");
    $Ted->thead();
    $Ted->tr("axtitle");
    $Ted->td( $Tbar->render(), "axtitle" );
    $Ted->td_colspan(2);

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>BLOCK SETTINGS</b>", "axhdg");
    $Ted->td_colspan(2);
    // ..................................................................

    // Block parameters..
    $block_desc = new form_textfield("block_desc", "", $this->block->block_desc);
    $block_desc->setclass("axtxtbox");
    $block_desc->setstyle("width:$bigwidth");

    $block_justify = new form_combofield("block_justify", "", $this->block->justify);
    $block_justify->setclass("axcombo");
    $block_justify->additem("",       "default");
    $block_justify->additem("left",   "Left");
    $block_justify->additem("center", "Centre");
    $block_justify->additem("right",  "Right");

    $block_valign = new form_combofield("block_valign", "", $this->block->valign);
    $block_valign->setclass("axcombo");
    $block_valign->additem("",       "default");
    $block_valign->additem("top",    "Top");
    $block_valign->additem("middle", "Middle");
    $block_valign->additem("bottom", "Bottom");

    $block_bg_colour = $colourCombo;
    $block_bg_colour->name = "background_colour";
    $block_bg_colour->setvalue($this->block->background_colour);

    $block_border_width = new form_textfield("block_border_width", "", $this->block->border_width);
    $block_border_width->setclass("axtxtbox");
    $block_border_width->setstyle("width:30px;text-align:center;");
    $block_border_width->set_onblur("limitInt(this, 0, 99)");

    $block_border_colour = $colourCombo;
    $block_border_colour->name = "block_border_colour";
    $block_border_colour->setvalue($this->block->border_colour);

    $block_style = new form_textfield("block_style", "", $this->block->block_style);
    $block_style->setclass("axtxtbox");
    $block_style->setstyle("width:$cbowidth;");

    $block_bgimgs = new form_combofield("background_img");
    $block_bgimgs->setclass("axcombo");
    $block_bgimgs->setstyle("width:$ewidth;");
    $block_bgimgs->additem(NULLVALUE, "");
    $Qimg = dbrecordset("SELECT * FROM ax_catalog WHERE mime_category='image' ORDER BY upload_timestamp DESC");
    if ($Qimg->hasdata) {
      do {
        $catid = $Qimg->field("cat_id");
        $catname = $Qimg->field("cat_name");
        // Names following ID tell the inserter the fields to get..
        $block_bgimgs->additem($catid, $catname);
      } while ($Qimg->get_next());
    }
    $block_bgimgs->setvalue($this->block->background_img);

    $Tin = new table("blockopts");
    $Tin->setpadding(2);

    $Tin->tr("axbgdark");
    $Tin->td( "Description:" );
    $Tin->td( $block_desc->render() );
    $Tin->td_colspan(2);
    $Tin->td_alignment("right");

    // Language..
    $langsCombo->setvalue($this->block->language);
    if ($RESPONSE->multilang) {
      // Present the language selector..
      $Tin->tr("axbglite");
      $Tin->td( "&nbsp;" );
      $Tin->td( "Block language:&nbsp;" . $langsCombo->render() );
      $Tin->td_alignment("right");
      $Tin->td_colspan(2);
    }
    else {
      // In this case it is just a hidden field..
      $Tin->tr("axbglite");
      $Tin->td( "&nbsp;" );
      $Tin->td( $langsCombo->render() );
      $Tin->td_colspan(2);
    }

    $Tin->tr("axbglite");
    $Tin->td( "Justify:&nbsp;" . $block_justify->render() );
    $Tin->td_colspan(2);
    $Tin->td_alignment("right");
    $Tin->td( "Align:&nbsp;" . $block_valign->render() );
    $Tin->td_alignment("right");

    $Tin->tr( "axbgdark");
    $Tin->td( "Border width:&nbsp;" . $block_border_width->render() );
    $Tin->td_colspan(2);
    $Tin->td_alignment("right");
    $Tin->td( "Border colour:&nbsp;" . $block_border_colour->render() );
    $Tin->td_alignment("right");

    $Tin->tr( "axbglite");
    $Tin->td( "Background:&nbsp;" . $block_bg_colour->render() );
    $Tin->td_colspan(2);
    $Tin->td_alignment("right");
    $Tin->td( "CSS style:&nbsp;" . $block_style->render() );
    $Tin->td_alignment("right");

    $Tin->tr( "axbgdark");
    $Tin->td( "Background image:&nbsp;" . $block_bgimgs->render() );
    $Tin->td_colspan(3);
    $Tin->td_alignment("right");

    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td_colspan(2);

    // ..................................................................
    // Wysiwyg Heading..
    $Ted->tr("axhdg");
    $Ted->td("<b>AXYL CONTENT EDITOR</b>", "axhdg");
    $Ted->td_colspan(2);

    // ..................................................................
    // Wysiwyg content textarea
    // Convert old <object> media placeholders to new <img> variant..
    $wysiwyg = new form_wysiwygfield("wysiwyg_content", "", $wysiwyg_blocklet->content);
    $wysiwyg->setclass("axmemo");
    $wysiwyg->setstyle("width:$awidth;height:500px;");
    $wysiwyg->register_plugins("all");
    $wysiwyg->set_toolbar("full");

    $Tin = new table("content");
    $Tin->tr();
    $Tin->td( $wysiwyg->render() );
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td_colspan(2);
    // ..................................................................
    // Provide the user interfaces to insert content..
    $this->add_image_ui($Ted);
    $this->add_document_ui($Ted);
    $this->add_media_ui($Ted);
    $this->add_datasrc_ui($Ted);

    // ..................................................................
    // Render the whole table..
    $Ted->tr("axfoot");
    $Ted->td("", "axfoot");
    $Ted->td_colspan(2);
    $s .= $Ted->render();

    // ..................................................................
    $hid  = new form_hiddenfield("_wysiwygpost_form", $this->block->blockfm);
    $s .= $hid->render();

    debug_trace();
    // Return the html..
    return $s;
  } // wysiwyg_editform
  // ....................................................................
  /**
  * Render the block editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function block_editform($lang_encoding, $lang_direction) {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $gen_addbtn, $gen_blocklettype;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    global $colourCombo, $tstylesCombo, $langsCombo;

    // Insert javascript for block editing..
    insert_block_jscript();

    // Initialise content..
    $s = "";

    // ..................................................................
    // KEYFIELD and RECORD MAINTAINER
    // Blocklet listbox
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form.
    //
    // This is the keyfield listbox which controls the maintainance
    // process. It lists all records being maintained..
    $blocklet_listbox = new form_combofield("blocklet_id");
    $blocklet_listbox->setclass("axlistbox");
    // Make a new record maintainer, and attach the buttons..
    $maintainer = new recmaintainer($this->block->blockfm, $blocklet_listbox);

    // Create buttons..
    $bup    = new form_imagebutton("_up",    "", "", "$LIBDIR/img/_up.gif",     "Move up",                57, 15);
    $bdown  = new form_imagebutton("_down",  "", "", "$LIBDIR/img/_down.gif",   "Move down",              57, 15);
    $bdel   = new form_imagebutton("_del",   "", "", "$LIBDIR/img/_delete.gif", "Delete blocklet",        57, 15);
    $badd   = new form_imagebutton("_add",   "", "", "$LIBDIR/img/_add.gif",    "Add new blocklet",       57, 15);
    $bsave  = new form_imagebutton("_save",  "", "", "$LIBDIR/img/_save.gif",   "Save blocklet settings", 57, 15);
    $bdone  = new form_imagebutton("_done",  "", "", "$LIBDIR/img/_done.gif",   "Exit without saving",    57, 15);
    $breset = new form_imagebutton("_reset", "", "", "$LIBDIR/img/_reset.gif",  "Reset form",             57, 15);
    $breset->set_onclick("document.forms." . $this->block->blockfm . ".reset()");

    // Register all relevant buttons to the maintainer..
    $maintainer->register_button("up" ,   $bup);
    $maintainer->register_button("down",  $bdown);
    $maintainer->register_button("add",   $badd);
    $maintainer->register_button("save",  $bsave);
    $maintainer->register_button("del",   $bdel);
    $maintainer->register_button("reset", $breset);

    // Control table..
    $Ted = new table("block_" . $this->block->blockid);
    $Ted->setpadding(2);

    // ..................................................................
    // Toolbar..
    $toolbar = array();
    $toolbar[] = $bdone;
    $toolbar[] = $bsave;
    $Tbar = new table("toolbar");
    $Tbar->tr("axtitle");
    switch ($this->block->layout_version) {
      case VERSION_PENDING:   $hdg = "PENDING";  break;
      case VERSION_LIVE:      $hdg = "LIVE";     break;
      case VERSION_PREVIOUS:  $hdg = "PREVIOUS"; break;
      case VERSION_UNDEFINED: $hdg = "EDITING";  break;
      default: $hdg = "Version: " . $this->block->layout_version;
    } // switch
    $Tbar->th("<b>$hdg</b> [B" . $this->block->blockid . "]", "axtitle");
    $tools = "";
    foreach ($toolbar as $tool) {
      $tools .= $tool->render();
    }
    $Tbar->th($tools, "axtitle");
    $Tbar->th_css("text-align:right");
    $Ted->thead();
    $Ted->tr("axtitle");
    $Ted->td( $Tbar->render(), "axtitle" );
    $Ted->td_colspan(2);

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>BLOCK SETTINGS</b>", "axhdg");
    $Ted->td_colspan(2);
    // ..................................................................

    // Block parameters..
    $block_desc = new form_textfield("block_desc", "", $this->block->block_desc);
    $block_desc->setclass("axtxtbox");
    $block_desc->setstyle("width:$bigwidth");

    $block_cols = new form_textfield("cols", "", $this->block->cols);
    $block_cols->setclass("axtxtbox");
    $block_cols->setstyle("width:30px;text-align:center;");
    $block_cols->set_onblur("limitInt(this, 1, 9)");

    $block_gutter_width = new form_textfield("gutter_width", "", $this->block->gutter_width);
    $block_gutter_width->setclass("axtxtbox");
    $block_gutter_width->setstyle("width:30px;text-align:center;");
    $block_gutter_width->set_onblur("limitInt(this, 1, 99)");

    $block_gutter_colour = $colourCombo;
    $block_gutter_colour->name = "gutter_colour";
    $block_gutter_colour->setvalue($this->block->gutter_colour);

    $block_justify = new form_combofield("block_justify", "", $this->block->justify);
    $block_justify->setclass("axcombo");
    $block_justify->additem("",       "default");
    $block_justify->additem("left",   "Left");
    $block_justify->additem("center", "Centre");
    $block_justify->additem("right",  "Right");

    $block_valign = new form_combofield("block_valign", "", $this->block->valign);
    $block_valign->setclass("axcombo");
    $block_valign->additem("",       "default");
    $block_valign->additem("top",    "Top");
    $block_valign->additem("middle", "Middle");
    $block_valign->additem("bottom", "Bottom");

    $block_bg_colour = $colourCombo;
    $block_bg_colour->name = "background_colour";
    $block_bg_colour->setvalue($this->block->background_colour);

    $block_border_width = new form_textfield("block_border_width", "", $this->block->border_width);
    $block_border_width->setclass("axtxtbox");
    $block_border_width->setstyle("width:30px;text-align:center;");
    $block_border_width->set_onblur("limitInt(this, 0, 99)");

    $blocklet_sep = new form_textfield("blocklet_sep", "", $this->block->blocklet_sep);
    $blocklet_sep->setclass("axtxtbox");
    $blocklet_sep->setstyle("width:30px;text-align:center;");
    $blocklet_sep->set_onblur("limitInt(this, 0, 99)");

    $block_exportable = new form_checkbox("block_exportable");
    $block_exportable->checked = $this->block->exportable;

    $block_border_colour = $colourCombo;
    $block_border_colour->name = "block_border_colour";
    $block_border_colour->setvalue($this->block->border_colour);

    $block_style = new form_textfield("block_style", "", $this->block->block_style);
    $block_style->setclass("axtxtbox");
    $block_style->setstyle("width:$ewidth;");

    $block_bgimgs = new form_combofield("background_img");
    $block_bgimgs->setclass("axcombo");
    $block_bgimgs->setstyle("width:$ewidth;");
    $block_bgimgs->additem(NULLVALUE, "");
    $Qimg = dbrecordset("SELECT * FROM ax_catalog WHERE mime_category='image' ORDER BY upload_timestamp DESC");
    if ($Qimg->hasdata) {
      do {
        $catid = $Qimg->field("cat_id");
        $catname = $Qimg->field("cat_name");
        // Names following ID tell the inserter the fields to get..
        $block_bgimgs->additem($catid, $catname);
      } while ($Qimg->get_next());
    }
    $block_bgimgs->setvalue($this->block->background_img);

    $Tin = new table("blockopts");
    $Tin->setpadding(2);

    $Tin->tr( "axbgdark");
    $Tin->td( "Description:&nbsp;" . $block_desc->render() );
    $Tin->td_colspan(3);
    $Tin->td_alignment("right");

    // Language..
    $langsCombo->setvalue($this->block->language);
    if ($RESPONSE->multilang) {
      // Present the language selector..
      $Tin->tr("axbglite");
      $Tin->td( "&nbsp;" );
      $Tin->td( "Block language:&nbsp;" . $langsCombo->render() );
      $Tin->td_alignment("right");
      $Tin->td_colspan(2);
    }
    else {
      // In this case it is just a hidden field..
      $Tin->tr("axbglite");
      $Tin->td( "&nbsp;" );
      $Tin->td( $langsCombo->render() );
      $Tin->td_colspan(2);
    }

    $Tin->tr("axbgdark");
    $Tin->td( "Columns:&nbsp;" . $block_cols->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Gutter width:&nbsp;" . $block_gutter_width->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Gutter colour:&nbsp;" . $block_gutter_colour->render() );
    $Tin->td_alignment("right");

    $Tin->tr( "axbglite");
    $Tin->td( "Blocklet spacing:&nbsp;" . $blocklet_sep->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Border width:&nbsp;" . $block_border_width->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Border colour:&nbsp;" . $block_border_colour->render() );
    $Tin->td_alignment("right");

    $Tin->tr("axbgdark");
    $Tin->td( "Justify:&nbsp;" . $block_justify->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Align:&nbsp;" . $block_valign->render() );
    $Tin->td_alignment("right");
    $Tin->td( "Background:&nbsp;" . $block_bg_colour->render() );
    $Tin->td_alignment("right");

    $Tin->tr( "axbglite");
    $Tin->td( "CSS style:&nbsp;" . $block_style->render() );
    $Tin->td_colspan(3);
    $Tin->td_alignment("right");

    $Tin->tr( "axbgdark");
    $Tin->td( "Background image:&nbsp;" . $block_bgimgs->render() );
    $Tin->td_colspan(3);
    $Tin->td_alignment("right");

    $Tin->tr( "axbglite");
    $Tin->td( "Exportable in CSV format&nbsp;" . $block_exportable->render() );
    $Tin->td_colspan(3);
    $Tin->td_alignment("right");

    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td_colspan(2);

    // ..................................................................
    // Blocklets Editing Fields..
    $Ted->tr("axhdg");
    $Ted->td("<b>BLOCKLETS IN THIS BLOCK</b>", "axhdg");
    $Ted->td_colspan(2);

    // Continue defining listbox..
    $blocklet_listbox->setstyle("width:$ewidth;height:140px;");
    $blocklet_listbox->size = 6;
    foreach ($this->block->blocklets as $b) {
      // Populate listbox..
      $blocklet_listbox->additem($b->blockletid, $b->blocklet_desc);

      // Populate maintainer data. The maintainer add_record method
      // requires an associative array keyed on listbox key id..
      $rec = array(
              "blocklet_desc"     => $b->blocklet_desc,
              "blocklet_type"     => $b->type,
              "blocklet_width"    => $b->width,
              "blocklet_style"    => $b->blocklet_style,
              "justify"           => $b->justify,
              "heading"           => $b->heading,
              "heading_level"     => $b->heading_level,
              "heading_colour"    => $b->heading_colour,
              "ruler"             => $b->ruler,
              "ruler_width"       => $b->ruler_width,
              "ruler_size"        => $b->ruler_size,
              "ruler_colour"      => $b->ruler_colour,
              "content"           => $b->content,
              "content_size"      => ($b->content_size == 0) ? "0" : number_format($b->content_size, 1),
              "content_colour"    => $b->content_colour,
              "table_style"       => $b->table_style,
              "table_autojustify" => (($b->table_autojustify) ? "t" : "f"),
              "table_rowstripes"  => (($b->table_rowstripes) ? "t" : "f"),
              "visible"           => (($b->visible) ? "t" : "f")
              );
      $maintainer->add_record($b->blockletid, $rec);
    }
    // Now set the defaults for each of the fields. These are
    // necessary for when a new record is created..
    $defaults = array(
              "blocklet_desc"     => "(enter a blocklet description)",
              "blocklet_type"     => "text",
              "blocklet_width"    => "100",
              "blocklet_style"    => "",
              "justify"           => "left",
              "heading"           => "",
              "heading_level"     => "3",
              "heading_colour"    => "",
              "ruler"             => "none",
              "ruler_width"       => "100",
              "ruler_size"        => "1",
              "ruler_colour"      => "",
              "content"           => "",
              "content_size"      => "0",
              "content_colour"    => "",
              "table_style"       => "",
              "table_autojustify" => "f",
              "table_rowstripes"  => "f",
              "visible"           => "t"
              );
    $maintainer->add_defaults($defaults);

    // The listbox field..
    $Ted->tr("axbgdark");
    $Ted->td( $blocklet_listbox->render() );
    $Ted->td_width("50%");
    $Ted->td(
          $bup->render()   . "<br>"
        . $bdown->render() . "<br>"
        . $bdel->render()  . "<br>"
        . $badd->render()
        );
    $Ted->td_width("50%");
    $Ted->td_alignment("right", "top");

    // ..................................................................
    // Blocklet description field..
    $blocklet_desc = new form_textfield("blocklet_desc", "", $b->blocklet_desc);
    $maintainer->register_field($blocklet_desc);
    $blocklet_desc->setclass("axtxtbox");
    $blocklet_desc->setstyle("width:$ewidth;");
    $Ted->tr("axbglite");
    $Ted->td( $blocklet_desc->render() );
    $Ted->td( "< Blocklet Description" );

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>BLOCKLET SETTINGS</b>", "axhdg");
    $Ted->td_colspan(2);

    // ..................................................................
    // Blocklet type, Width and Visibility
    // NOTE: We register each field, beginning with these, to the maintainer.
    // We MUST use the database field-name, as our form element name..
    global $gen_blocklettype;
    $maintainer->register_field($gen_blocklettype);

    $blocklet_width = new form_textfield("blocklet_width", "", $b->width);
    $maintainer->register_field($blocklet_width);
    $blocklet_width->setclass("axtxtbox");
    $blocklet_width->setstyle("width:35px;text-align:center;");
    $blocklet_width->set_onblur("limitInt(this, 0, 100, 100)");

    $blocklet_visible = new form_checkbox("visible", "", "yes", EDITABLE, "", $b->visible);
    $blocklet_visible->setclass("axchkbox");
    $maintainer->register_field($blocklet_visible);

    $Tin1 = new table("type");
    $Tin1->tr();
    $Tin1->td("<b>Blocklet type:</b>");
    $Tin1->td( $gen_blocklettype->render() );
    $Tin1->td_alignment("right");
    $Tin2 = new table("opts");
    $Tin2->setwidth("");
    $Tin2->tbody("fmlook");
    $Tin2->tr();
    $Tin2->td( "Blocklet width%:", "text-align:right");
    $Tin2->td( $blocklet_width->render() );
    $Tin2->td( "&nbsp;Visible:", "text-align:right" );
    $Tin2->td( $blocklet_visible->render() );
    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    // ..................................................................
    // Blocklet ruler settings
    $blocklet_ruler = new form_combofield("ruler", "", $b->ruler);
    $maintainer->register_field($blocklet_ruler);
    $blocklet_ruler->setclass("axcombo");
    $blocklet_ruler->setstyle("width:$cbowidth;");
    $blocklet_ruler->additem("top",    "Top");
    $blocklet_ruler->additem("bottom", "Bottom");
    $blocklet_ruler->additem("none",   "None");

    $blocklet_ruler_width = new form_textfield("ruler_width", "", $b->ruler_width);
    $maintainer->register_field($blocklet_ruler_width);
    $blocklet_ruler_width->setclass("axtxtbox");
    $blocklet_ruler_width->setstyle("width:35px;text-align:center;");
    $blocklet_ruler_width->set_onblur("limitInt(this, 0, 100, 100)");

    $blocklet_ruler_size  = new form_textfield("ruler_size",  "", $b->ruler_size);
    $maintainer->register_field($blocklet_ruler_size);
    $blocklet_ruler_size->setclass("axtxtbox");
    $blocklet_ruler_size->setstyle("width:30px;text-align:center;");
    $blocklet_ruler_size->set_onblur("limitInt(this, 1, 99)");
    $Tin1 = new table("ruler");
    $Tin1->tbody("fmlook");
    $Tin1->tr();
    $Tin1->td("<b>Ruler:</b>");
    $Tin1->td( $blocklet_ruler->render() );
    $Tin1->td_alignment("right");
    $Tin2 = new table("ruler");
    $Tin2->setwidth("");
    $Tin2->tbody("fmlook");
    $Tin2->tr();
    $Tin2->td( "Ruler width%:", "text-align:right");
    $Tin2->td( $blocklet_ruler_width->render() );
    $Tin2->td( "&nbsp;&nbsp;Size:", "text-align:right" );
    $Tin2->td( $blocklet_ruler_size->render() );
    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    // ..................................................................
    // Blocklet ruler colour..
    $blocklet_ruler_colour = $colourCombo;
    $blocklet_ruler_colour->name = "ruler_colour";
    $blocklet_ruler_colour->setvalue($b->ruler_colour);
    $maintainer->register_field($blocklet_ruler_colour);
    $Tin = new table("ruler_colour");
    $Tin->tr();
    $Tin->td("&nbsp;");
    $Tin->td( $blocklet_ruler_colour->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td("&nbsp;");

    // ..................................................................
    // Block heading text field..
    $blocklet_heading = new form_textfield("heading", "", $b->heading);
    $maintainer->register_field($blocklet_heading);
    $blocklet_heading->setclass("axtxtbox");
    $blocklet_heading->setstyle("width:185px;");

    $blocklet_heading_level = new form_combofield("heading_level", "", $b->heading_level);
    $maintainer->register_field($blocklet_heading_level);
    $blocklet_heading_level->setclass("axcombo");
    $blocklet_heading_level->setstyle("width:50px;");
    $blocklet_heading_level->setvalue($b->heading_level);
    for ($h=1; $h<=6; $h++) {
      $blocklet_heading_level->additem($h);
    }
    $Tin1 = new table("ruler");
    $Tin1->tr();
    $Tin1->td( "<b>Heading:</b>" );
    $Tin1->td( $blocklet_heading->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("hlevel");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Level:" );
    $Tin2->td( $blocklet_heading_level->render() );

    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    // Heading colour..
    $blocklet_heading_colour = $colourCombo;
    $blocklet_heading_colour->name = "heading_colour";
    $blocklet_heading_colour->setvalue($b->heading_colour);
    $maintainer->register_field($blocklet_heading_colour);
    $Tin = new table("heading_colour");
    $Tin->tr();
    $Tin->td("&nbsp;");
    $Tin->td( $blocklet_heading_colour->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td("&nbsp;");

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>BLOCKLET CONTENT</b>", "axhdg");
    $Ted->td_colspan(2);

    // ..................................................................
    // Blocklet content textarea, table checkboxes, and reset button
    $blocklet_content = new form_memofield("content", "", $b->content);
    $maintainer->register_field($blocklet_content);
    $blocklet_content->setclass("axmemo");
    $blocklet_content->setstyle("width:$awidth;height:200px;font-family:'courier-new, monospace'");

    $Tin = new table("content");
    $Tin->tr();
    $Tin->td( $blocklet_content->render() );
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td_colspan(2);

    // ..................................................................
    // Content Properties

    // Content colour..
    $blocklet_content_colour = $colourCombo;
    $blocklet_content_colour->name = "content_colour";
    $blocklet_content_colour->setvalue($b->content_colour);
    $maintainer->register_field($blocklet_content_colour);

    // Content size adjustment..
    $blocklet_content_size = new form_combofield("content_size", "", $b->content_size);
    $maintainer->register_field($blocklet_content_size);
    $blocklet_content_size->setclass("axcombo");
    $blocklet_content_size->setstyle("width:$cbowidth;");
    $blocklet_content_size->additem("-3.0", "-5");
    $blocklet_content_size->additem("-2.5", "-4");
    $blocklet_content_size->additem("-2.0", "-3");
    $blocklet_content_size->additem("-1.5", "-2");
    $blocklet_content_size->additem("-1.0", "-1");
    $blocklet_content_size->additem("0", "default size");
    $blocklet_content_size->additem("1.0",  "+1");
    $blocklet_content_size->additem("1.5",  "+2");
    $blocklet_content_size->additem("2.0",  "+3");
    $blocklet_content_size->additem("2.5",  "+4");
    $blocklet_content_size->additem("3.0",  "+5");

    // Content justificiation..
    $blocklet_justify = new form_combofield("justify", "", $b->justify);
    $maintainer->register_field($blocklet_justify);
    $blocklet_justify->setclass("axcombo");
    $blocklet_justify->setstyle("width:$cbowidth;");
    $blocklet_justify->additem("",   "Default");
    $blocklet_justify->additem("left",   "Left");
    $blocklet_justify->additem("center", "Centre");
    $blocklet_justify->additem("right",  "Right");

    // Content tablestyle..
    $table_style = $tstylesCombo;
    $maintainer->register_field($table_style);
    $table_style->name = "table_style";
    $table_style->setvalue($b->table_style);

    $table_aj = new form_checkbox("table_autojustify");
    $maintainer->register_field($table_aj);
    $table_aj->setclass("axchkbox");
    $table_aj->checked = $b->table_autojustify;

    $table_rs = new form_checkbox("table_rowstripes");
    $maintainer->register_field($table_rs);
    $table_rs->setclass("axchkbox");
    $table_rs->checked = $b->table_rowstripes;

    $blocklet_style = new form_textfield("blocklet_style", "", $b->blocklet_style);
    $maintainer->register_field($blocklet_style);
    $blocklet_style->setclass("axtxtbox");
    $blocklet_style->setstyle("width:$cbowidth");

    $Tin1 = new table("content_props");
    $Tin1->tr();
    $Tin1->td("Text size adjust:");
    $Tin1->td( $blocklet_content_size->render() );
    $Tin1->td_alignment("right");
    $Tin1->tr();
    $Tin1->td("Text colour:");
    $Tin1->td( $blocklet_content_colour->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("content_align");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "&nbsp;" );
    $Tin2->td( $breset->render() );
    $Tin2->td_alignment("right", "top");
    $Tin2->tr();
    $Tin2->td( "Alignment:" );
    $Tin2->td( $blocklet_justify->render() );

    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right", "top");

    // Blocklet style
    $Tin1 = new table("bktstyle");
    $Tin1->tr();
    $Tin1->td( "CSS style:" );
    $Tin1->td( $blocklet_style->render() );
    $Tin1->td_alignment("right");

    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td( "" );

    // Table options
    $Tin1 = new table("tablestyle");
    $Tin1->tr();
    $Tin1->td( "Table Options:" );
    $Tin1->td( $table_style->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("tableopts");
    $Tin2->tr();
    $Tin2->td( "Auto-justify:&nbsp;" . $table_aj->render() );
    $Tin2->td_alignment("right");
    $Tin2->td( "Row-stripes:&nbsp;"  . $table_rs->render() );
    $Tin2->td_alignment("right");

    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    // CSV import
    $uploader = new form_fileuploadfield("_csvimportdata");
    $uploader->setclass("axtxtbox");

    $importbtn = new form_imagebutton("_csvimport");
    $importbtn->setimage("$LIBDIR/img/_import.gif", "Import CSV data now");

    $Tin1 = new table("csvimporter");
    $Tin1->tr();
    $Tin1->td( "Table CSV:" );
    $Tin1->td( $uploader->render() );
    $Tin1->td( $importbtn->render() );
    $Tin1->td_alignment("right");

    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td_colspan(2);

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>INSERT SPECIAL CONTENT</b>", "axhdg");
    $Ted->td_colspan(2);

    // ..................................................................
    // Add Link
    $pages = new form_combofield("link_sitepage");
    $pages->setid("fspec");
    $pages->set_onchange("linkpop(this,'" . $this->block->blockfm . "')");
    $pages->setclass("axcombo");
    $pages->setstyle("width:$cwidth;");
    $pages->additem("");
    $Qpage = dbrecordset("SELECT * FROM ax_sitepage WHERE enabled=TRUE ORDER BY page_title");
    if ($Qpage->hasdata) {
      do {
        $pgpath = $Qpage->field("page_path");
        $pgtitle = $Qpage->field("page_title");
        $pages->additem($pgpath, $pgtitle);
      } while ($Qpage->get_next());
    }
    $gen_addbtn->set_onclick("add_link('" . $this->block->blockfm . "')");
    $remote = new form_checkbox("link_remote");
    $remote->setclass("axchkbox");
    $remote->setid("fspec");
    $remote->check();
    $gen_textfield = new form_textfield("add_link");
    $gen_textfield->setid("fspec");
    $gen_textfield->setclass("axtxtbox");
    $gen_textfield->setstyle("width:$cwidth;");

    $Tin = new table("add_link");
    $Tin->tr();
    $Tin->td("<b>Hyperlink:</b>");
    $Tin->td( $gen_textfield->render("link_href") );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td( $gen_addbtn->render() );
    $Ted->td_alignment("right");

    $Tin1 = new table("add_link");
    $Tin1->tr();
    $Tin1->td("Link text:");
    $Tin1->td( $gen_textfield->render("link_desc") );
    $Tin1->td_alignment("right");

    $Tin2 = new table("remote");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Remote site:" );
    $Tin2->td( $remote->render() );

    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    $Tin = new table("add_page");
    $Tin->tr();
    $Tin->td("Webpage:");
    $Tin->td( $pages->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td("&nbsp;");

    // ..................................................................
    // Add image
    $this->add_image_ui($Ted);

    // ..................................................................
    // Add document
    $this->add_document_ui($Ted);

    // ..................................................................
    // Add multimedia
    $this->add_media_ui($Ted);

    // ..................................................................
    // Add Datasource
    $this->add_datasrc_ui($Ted);

    // ..................................................................
    // Render the whole form..
    $Ted->tr("axfoot");
    $Ted->td("", "axfoot");
    $Ted->td_colspan(2);
    $s .= $Ted->render();

    // Render the maintainer. This adds the Javascript data structures
    // and renders the hidden fields for submitting changed field data..
    $s .= $maintainer->render();

  // ....................................................................
    debug_trace();
    // Return the html..
    return $s;
  } // block_editform
  // ....................................................................
  /** Given a table object passed in (by reference), add to it the rows
  * containing the Image Add form elements..
  * @param object $Ted The table object to add the user interface to
  * @access private
  */
  function add_image_ui(&$Ted) {
    global $gen_addbtn;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    global $colourCombo;
    $imgs = new form_combofield("img_src");
    $imgs->setid("fspec");
    $imgs->setclass("axcombo");
    $imgs->setstyle("width:$ewidth;");
    $imgs->additem("");
    $Qimg = dbrecordset("SELECT * FROM ax_catalog WHERE mime_category='image' ORDER BY upload_timestamp DESC");
    if ($Qimg->hasdata) {
      do {
        $catid = $Qimg->field("cat_id");
        $catname = $Qimg->field("cat_name");
        // Names following ID tell the inserter the fields to get..
        $imgs->additem("$catid|img_align|img_pad|img_border|img_bordercolor|img_width|img_height", $catname);
      } while ($Qimg->get_next());
    }
    // Add this catalog item as an IMAGE tag..
    $img_addbtn = $gen_addbtn;
    $img_addbtn->name = "_add_img";
    $img_addbtn->settitle("Add image to content");
    switch ($this->block->block_type) {
      case "w":
        $img_addbtn->set_onclick("add_content('image','img_src','wysiwyg_content','" . $this->block->blockfm . "')");
        break;
      default:
        $img_addbtn->set_onclick("add_content('image','img_src','content','" . $this->block->blockfm . "')");
    } // switch

    $imgalign = new form_combofield("img_align");
    $imgalign->setid("fspec");
    $imgalign->setclass("axcombo");
    $imgalign->setstyle("width:$cbowidth;");
    $imgalign->additem("", "Default");
    $imgalign->additem("left", "Left");
    $imgalign->additem("right", "Right");
    $imgalign->additem("top", "Top");
    $imgalign->additem("middle", "Middle");
    $imgalign->additem("bottom", "Bottom");
    $imgalign->additem("abstop", "Top (abs)");
    $imgalign->additem("absbottom", "Bottom (abs)");
    $imgalign->additem("baseline", "Baseline");
    $imgalign->additem("texttop", "Top (text)");

    $imgbdr = new form_textfield("img_border", "", "0");
    $imgbdr->setid("fspec");
    $imgbdr->setclass("axtxtbox");
    $imgbdr->setstyle("width:35px;text-align:center;");
    $imgbdr->set_onblur("limitInt(this, 0, 99, 0)");

    $imgpad = new form_textfield("img_pad", "", "2");
    $imgpad->setid("fspec");
    $imgpad->setclass("axtxtbox");
    $imgpad->setstyle("width:35px;text-align:center;");
    $imgpad->set_onblur("limitInt(this, 0, 199, 2)");

    $imgwidth = new form_textfield("img_width");
    $imgwidth->setid("fspec");
    $imgwidth->setclass("axtxtbox");
    $imgwidth->setstyle("width:35px;text-align:center;");
    $imgwidth->set_onblur("limitInt(this, 0, 1999, 0)");

    $imgheight = new form_textfield("img_height");
    $imgheight->setid("fspec");
    $imgheight->setclass("axtxtbox");
    $imgheight->setstyle("width:35px;text-align:center;");
    $imgheight->set_onblur("limitInt(this, 0, 1999, 0)");

    $Tin = new table("img_src");
    $Tin->tr();
    $Tin->td("<b>Image:</b>");
    $Tin->td( $imgs->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td( $img_addbtn->render() );
    $Ted->td_alignment("right");

    $Tin1 = new table("img_align");
    $Tin1->tr();
    $Tin1->td("Border colour:");
    $Tin1->td( $colourCombo->render("img_bordercolor") );
    $Tin1->td_alignment("right");

    $Tin2 = new table("img_pad");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Border:", "text-align:right;width:" . ceil($quartwidth/2) . "px");
    $Tin2->td( $imgbdr->render() );
    $Tin2->td( "&nbsp;&nbsp;Pad:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $imgpad->render() );

    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    $Tin1 = new table("img_display");
    $Tin1->tr();
    $Tin1->td("Align:");
    $Tin1->td( $imgalign->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("img_size");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Width:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $imgwidth->render() );
    $Tin2->td( "&nbsp;&nbsp;Height:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $imgheight->render() );
    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

  } // add_image_ui
  // ....................................................................
  /** Given a table object passed in (by reference), add to it the rows
  * containing the Document Add form elements..
  * @param object $Ted The table object to add the user interface to
  * @access private
  */
  function add_document_ui(&$Ted) {
    global $gen_addbtn;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    $docs = new form_combofield("doc_src");
    $docs->setid("fspec");
    $docs->setclass("axcombo");
    $docs->setstyle("width:$ewidth;");
    $docs->additem("");
    $q  = "SELECT * FROM ax_catalog";
    $q .= " WHERE mime_category='document'";
    $q .= " ORDER BY mime_type, upload_timestamp DESC";
    $Qdoc = dbrecordset($q);
    if ($Qdoc->hasdata) {
      do {
        $catid    = $Qdoc->field("cat_id");
        $catname  = $Qdoc->field("cat_name");
        $extn     = get_file_extn($Qdoc->field("filepath"));
        if ($extn != "") {
          $catname .= " ($extn)";
        }
        // Names following ID tell the inserter the fields to get..
        $docs->additem("$catid|doc_display|doc_width|doc_height", $catname);
      } while ($Qdoc->get_next());
    }
    // Add this catalog item as a DOCUMENT tag..
    $doc_addbtn = $gen_addbtn;
    $doc_addbtn->name = "_add_doc";
    $doc_addbtn->settitle("Add document to content");
    switch ($this->block->block_type) {
      case "w":
        $doc_addbtn->set_onclick("add_content('document','doc_src','wysiwyg_content','" . $this->block->blockfm . "')");
        break;
      default:
        $doc_addbtn->set_onclick("add_content('document','doc_src','content','" . $this->block->blockfm . "')");
    } // switch
    $docdisplay = new form_combofield("doc_display");
    $docdisplay->setid("fspec");
    $docdisplay->setclass("axcombo");
    $docdisplay->setstyle("width:$cbowidth;");
    $docdisplay->additem("icon", "clickable icon");
    $docdisplay->additem("link", "hyperlink");

    $docwidth = new form_textfield("doc_width");
    $docwidth->setid("fspec");
    $docwidth->setclass("axtxtbox");
    $docwidth->setstyle("width:35px;text-align:center;");
    $docwidth->set_onblur("limitInt(this, 0, 1999, 0)");

    $docheight = new form_textfield("doc_height");
    $docheight->setid("fspec");
    $docheight->setclass("axtxtbox");
    $docheight->setstyle("width:35px;text-align:center;");
    $docheight->set_onblur("limitInt(this, 0, 1999, 0)");

    $Tin = new table("doc_src");
    $Tin->tr();
    $Tin->td("<b>Document:</b>");
    $Tin->td( $docs->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td( $doc_addbtn->render() );
    $Ted->td_alignment("right");

    $Tin1 = new table("doc_display");
    $Tin1->tr();
    $Tin1->td("Display as:");
    $Tin1->td( $docdisplay->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("doc_size");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Width:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $docwidth->render() );
    $Tin2->td( "&nbsp;&nbsp;Height:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $docheight->render() );
    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");
  } // add_document_ui
  // ....................................................................
  /** Given a table object passed in (by reference), add to it the rows
  * containing the Media Add form elements..
  * @param object $Ted The table object to add the user interface to
  * @access private
  */
  function add_media_ui(&$Ted) {
    global $gen_addbtn;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    $MMs = new form_combofield("mm_src");
    $MMs->setid("fspec");
    $MMs->setclass("axcombo");
    $MMs->setstyle("width:$ewidth;");
    $MMs->additem("");
    $q  = "SELECT * FROM ax_catalog";
    $q .= " WHERE mime_category='movie'";
    $q .= "    OR mime_category='audio'";
    $q .= "    OR mime_category='flash'";
    $q .= " ORDER BY mime_category, mime_type, upload_timestamp DESC";
    $QMM = dbrecordset($q);
    if ($QMM->hasdata) {
      do {
        $catid = $QMM->field("cat_id");
        $catname = $QMM->field("cat_name");
        $extn     = get_file_extn($QMM->field("filepath"));
        if ($extn != "") {
          $catname .= " ($extn)";
        }
        // Names following ID tell the inserter the fields to get..
        $MMs->additem("$catid|mm_display|mm_width|mm_height|autostart|loop|showcontrols", $catname);
      } while ($QMM->get_next());
    }
    // Add this catalog item as a MEDIA  tag..
    $mm_addbtn = $gen_addbtn;
    $mm_addbtn->name = "_add_mm";
    $mm_addbtn->settitle("Add media to content");
    switch ($this->block->block_type) {
      case "w":
        $mm_addbtn->set_onclick("add_content('media','mm_src','wysiwyg_content','" . $this->block->blockfm . "')");
        break;
      default:
        $mm_addbtn->set_onclick("add_content('media','mm_src','content','" . $this->block->blockfm . "')");
    } // switch

    $MMdisplay = new form_combofield("mm_display");
    $MMdisplay->setid("fspec");
    $MMdisplay->setclass("axcombo");
    $MMdisplay->setstyle("width:$cbowidth;");
    $MMdisplay->additem("icon", "clickable icon");
    $MMdisplay->additem("embedded", "embedded in page");
    $MMdisplay->additem("link", "hyperlink");

    $MMwidth = new form_textfield("mm_width");
    $MMwidth->setid("fspec");
    $MMwidth->setclass("axtxtbox");
    $MMwidth->setstyle("width:35px;text-align:center;");
    $MMwidth->set_onblur("limitInt(this, 0, 1999, 0)");

    $MMheight = new form_textfield("mm_height");
    $MMheight->setid("fspec");
    $MMheight->setclass("axtxtbox");
    $MMheight->setstyle("width:35px;text-align:center;");
    $MMheight->set_onblur("limitInt(this, 0, 1999, 0)");

    $MMautostart = new form_checkbox("autostart");
    $MMautostart->setid("fspec");
    $MMautostart->setclass("axchkbox");
    $MMautostart->check();

    $MMloop = new form_checkbox("loop");
    $MMloop->setid("fspec");
    $MMloop->setclass("axchkbox");

    $MMcontrols = new form_checkbox("showcontrols");
    $MMcontrols->setid("fspec");
    $MMcontrols->setclass("axchkbox");
    $MMcontrols->check();

    $Tin = new table("mm_src");
    $Tin->tr();
    $Tin->td("<b>Media:</b>");
    $Tin->td( $MMs->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td( $mm_addbtn->render() );
    $Ted->td_alignment("right");

    $Tin1 = new table("mm_display");
    $Tin1->tr();
    $Tin1->td("Display as:");
    $Tin1->td( $MMdisplay->render() );
    $Tin1->td_alignment("right");

    $Tin2 = new table("mm_size");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Width:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $MMwidth->render() );
    $Tin2->td( "&nbsp;&nbsp;Height:", "text-align:right;width:" . ceil($quartwidth/2) . "px" );
    $Tin2->td( $MMheight->render() );
    $Ted->tr("axbglite");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");

    $Tin = new table("mm_opts");
    $Tin->tr();
    $Tin->td("Autostart:&nbsp;" . $MMautostart->render());
    $Tin->td_alignment("right");
    $Tin->td("Loop:&nbsp;" . $MMloop->render());
    $Tin->td_alignment("right");
    $Tin->td("Controls:&nbsp;" . $MMcontrols->render());
    $Tin->td_alignment("right");
    $Ted->tr("axbglite");
    $Ted->td( $Tin->render() );
    $Ted->td( "&nbsp;" );
    $Ted->td_alignment("right");
  } // add_media_ui
  // ....................................................................
  /** Given a table object passed in (by reference), add to it the rows
  * containing the Data Source Add form elements..
  * @param object $Ted The table object to add the user interface to
  * @access private
  */
  function add_datasrc_ui(&$Ted) {
    global $gen_addbtn;
    global $fullwidth, $halfwidth, $thirdwidth, $quartwidth, $threeqwidth;
    global $awidth, $ewidth, $cwidth, $cbowidth, $bigwidth;
    $datas = new form_combofield("data_src");
    $datas->setid("fspec");
    $datas->setclass("axcombo");
    $datas->setstyle("width:$ewidth;");
    $datas->additem("");
    $Qdata = dbrecordset("SELECT * FROM ax_query_resource WHERE enabled=TRUE ORDER BY q_desc DESC");
    if ($Qdata->hasdata) {
      do {
        $quid = $Qdata->field("quid");
        $qdesc = $Qdata->field("q_desc");
        $datas->additem("$quid|where|data_format|tableheadings", $qdesc);
      } while ($Qdata->get_next());
    }
    // Add this catalog item as a DATA  tag..
    $dat_addbtn = $gen_addbtn;
    $dat_addbtn->name = "_add_dat";
    $dat_addbtn->settitle("Add data query to content");
    switch ($this->block->block_type) {
      case "w":
        $dat_addbtn->set_onclick("add_content('data','data_src','wysiwyg_content','" . $this->block->blockfm . "')");
        break;
      default:
        $dat_addbtn->set_onclick("add_content('data','data_src','content','" . $this->block->blockfm . "')");
    } // switch

    $datawhere = new form_textfield("where");
    $datawhere->setid("fspec");
    $datawhere->setclass("axtxtbox");
    $datawhere->setstyle("width:$cwidth;");

    global $gen_blocklettype;
    $dataformat = $gen_blocklettype;
    $dataformat->setvalue("text");

    $tablehdgs = new form_checkbox("tableheadings");
    $tablehdgs->setid("fspec");
    $tablehdgs->setclass("axchkbox");

    $Tin = new table("data_src");
    $Tin->tr();
    $Tin->td("<b>Database:</b>");
    $Tin->td( $datas->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td( $dat_addbtn->render() );
    $Ted->td_alignment("right");

    $Tin = new table("where");
    $Tin->tr();
    $Tin->td("Where:");
    $Tin->td( $datawhere->render() );
    $Tin->td_alignment("right");
    $Ted->tr("axbgdark");
    $Ted->td( $Tin->render() );
    $Ted->td( "&nbsp;" );

    $Tin1 = new table("data_format");
    $Tin1->tr();
    $Tin1->td("Format:");
    $Tin1->td( $dataformat->render("data_format") );
    $Tin1->td_alignment("right");

    $Tin2 = new table("thdgs");
    $Tin2->setwidth("");
    $Tin2->tr();
    $Tin2->td( "Table headings:" );
    $Tin2->td( $tablehdgs->render() );

    $Ted->tr("axbgdark");
    $Ted->td( $Tin1->render() );
    $Ted->td( $Tin2->render() );
    $Ted->td_alignment("right");;
  } // add_datasrc_ui
} // blockeditor class

// ----------------------------------------------------------------------
?>