/**********************************************************************

  Audacity: A Digital Audio Editor

  CommandManager.h

  Brian Gunlogson
  Dominic Mazzoni

**********************************************************************/

#ifndef __AUDACITY_COMMAND_MANAGER__
#define __AUDACITY_COMMAND_MANAGER__

#include <wx/string.h>
#include <wx/dynarray.h>
#include <wx/menu.h>
#include <wx/hashmap.h>

#include "../xml/XMLTagHandler.h"

class CommandFunctor
{
public:
   CommandFunctor(){};
   virtual ~CommandFunctor(){};
   virtual void operator()(int index = 0) = 0;
};

struct MenuBarListEntry
{
   wxString name;
   wxMenuBar *menubar;
};

struct SubMenuListEntry
{
   wxString name;
   wxMenu *menu;
};

struct CommandListEntry
{
   int id;
   wxString name;
   wxString key;
   wxString defaultKey;
   wxString label;
   wxString labelPrefix;
   wxMenu *menu;
   CommandFunctor *callback;
   bool multi;
   int index;
   int count;
   bool enabled;
   wxUint32 flags;
   wxUint32 mask;
};

WX_DEFINE_ARRAY(MenuBarListEntry *, MenuBarList);
WX_DEFINE_ARRAY(SubMenuListEntry *, SubMenuList);
WX_DEFINE_ARRAY(CommandListEntry *, CommandList);

WX_DECLARE_STRING_HASH_MAP(CommandListEntry *, CommandNameHash);
WX_DECLARE_HASH_MAP(int, CommandListEntry *, wxIntegerHash, wxIntegerEqual, CommandIDHash);

class AUDACITY_DLL_API CommandManager: public XMLTagHandler
{
 public:

   //
   // Constructor / Destructor
   //

   CommandManager();
   virtual ~CommandManager();

   void PurgeData();

   //
   // Creating menus and adding commands
   //

   wxMenuBar *AddMenuBar(wxString sMenu);

   void BeginMenu(wxString tName);
   void EndMenu();

   wxMenu* BeginSubMenu(wxString tName);
   void EndSubMenu();

   void AddItem(wxString name, wxString label, CommandFunctor *callback,
                int checkmark = -1);
   void AddItemList(wxString name, wxArrayString labels,
                    CommandFunctor *callback, bool plugins = false);

   void AddSeparator();

   // A command doesn't actually appear in a menu but might have a
   // keyboard shortcut.
   void AddCommand(wxString name, wxString label, CommandFunctor *callback);

   //
   // Command masks
   //

   // For new items/commands
   void SetDefaultFlags(wxUint32 flags, wxUint32 mask);

   void SetCommandFlags(wxString name, wxUint32 flags, wxUint32 mask);
   void SetCommandFlags(const wxChar **names,
                        wxUint32 flags, wxUint32 mask);
   // Pass multiple command names as const wxChar *, terminated by NULL
   void SetCommandFlags(wxUint32 flags, wxUint32 mask, ...);

   //
   // Modifying menus
   //

   void EnableUsingFlags(wxUint32 flags, wxUint32 mask);
   void Enable(wxString name, bool enabled);
   void Check(wxString name, bool checked);
   void Modify(wxString name, wxString newLabel);

   //
   // Displaying menus
   //
   void HandleMenuOpen(wxMenuEvent &evt);
   void HandleMenuClose(wxMenuEvent &evt);

   //
   // Executing commands
   //
   bool HandleCommandEntry(CommandListEntry * entry, wxUint32 flags, wxUint32 mask);
   bool HandleMenuID(int id, wxUint32 flags, wxUint32 mask);
   bool HandleKey(wxKeyEvent &evt, wxUint32 flags, wxUint32 mask);
   bool HandleTextualCommand(wxString & Str, wxUint32 flags, wxUint32 mask);
   void TellUserWhyDisallowed( wxUint32 flagsGot, wxUint32 flagsRequired );
   //
   // Accessing
   //

   void GetAllCommandNames(wxArrayString &names, bool includeMultis);

   wxString GetLabelFromName(wxString name);
   wxString GetPrefixedLabelFromName( wxString name );
   wxString GetKeyFromName(wxString name);
   wxString GetDefaultKeyFromName(wxString name);

   //
   // Loading/Saving
   //

   virtual bool HandleXMLTag(const wxChar *tag, const wxChar **attrs);
   virtual void HandleXMLEndTag(const wxChar *tag);
   virtual XMLTagHandler *HandleXMLChild(const wxChar *tag);
   virtual void WriteXML(XMLWriter &xmlFile);

 protected:

   wxMenuBar * CurrentMenuBar();
   wxMenuBar * GetMenuBar(wxString sMenu);
   wxMenu * CurrentSubMenu();
   wxMenu * CurrentMenu();

   int NextIdentifier(int ID);
   int NewIdentifier(wxString name, wxString label, wxMenu *menu,
                     CommandFunctor *callback,
                     bool multi, int index, int count);
   void Enable(CommandListEntry *entry, bool enabled);

   wxString GetKey(wxString label);

   void ToggleAccels(wxMenu *m, bool show);

private:
   MenuBarList  mMenuBarList;
   SubMenuList  mSubMenuList;
   CommandList  mCommandList;
   CommandNameHash  mCommandNameHash;
   CommandNameHash  mCommandKeyHash;
   CommandIDHash  mCommandIDHash;
   int mCurrentID;
   int mHiddenID;
   int mXMLKeysRead;
   wxMenu * mCurrentMenu;
   wxMenu * mOpenMenu;

   wxUint32 mDefaultFlags;
   wxUint32 mDefaultMask;
};

#endif

// Indentation settings for Vim and Emacs and unique identifier for Arch, a
// version control system. Please do not modify past this point.
//
// Local Variables:
// c-basic-offset: 3
// indent-tabs-mode: nil
// End:
//
// vim: et sts=3 sw=3
// arch-tag: 6f086a60-916f-41d6-bd0c-b4d39c6bcde3

