# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Definition of MPI_INFO class.

Synopsis:
    as_run --serv fich.export
                V     directement (interactif)
                    ou via sub_script (batch)
    as_run fich.export
                V
    execute.py
                V
    PrepEnv dans /shared_tmp/interactif.xxx
                V
    mpirun script.sh -wd /shared_tmp/interactif.xxx
                >  parallel_cp tracker /shared_tmp/interactif.xxx /local_tmp/interactif.xxx.proc.#i
                >  cd /local_tmp/interactif.xxx.proc.#i
                >  ./asteru_mpi args
                >  if #i == 0:
                        cp /local_tmp/interactif.xxx.proc.#i/results /shared_tmp/interactif.xxx
"""


import os
import re
from asrun.installation import aster_root, datadir
from asrun.common.i18n import _
from asrun.mystring import ufmt
from asrun.common_func  import get_tmpname


class MPI_INFO:
    """Store informations for MPI executions.
    """
    def __init__(self, defines=None):
        """Initialization needs "DEFS" field of ASTER_CONFIG object.
        """
        if defines is not None:
            self._use_mpi = '_USE_MPI' in defines
        else:
            self._use_mpi = False
        self._nbnode = 1
        self._nbcpu  = 1
        self.global_reptrav = ''
        self.local_reptrav  = ''
        
        self.timer_comment = _(u"""(*) cpu and system times are not correctly""" \
                                """ counted using mpirun.""")
    

    def set_cpuinfo(self, nbnode, nbcpu):
        """Set values of number of processors, of nodes.
        """
        iret = 0
        try:
            self._nbcpu = int(nbcpu or 1)
        except ValueError:
            iret = 4
            self._nbcpu = 1
        
        try:
            self._nbnode = int(nbnode or 1)
        except ValueError:
            iret = 4
            self._nbnode = 1
        
        self._nbnode = min(self._nbnode, self._nbcpu)
        if not self._use_mpi:
            if self._nbnode > 1 or self._nbcpu > 1:
                iret = 1
            self._nbnode = 1
            self._nbcpu  = 1
        return iret


    def really(self):
        """Return True if Code_Aster executions need mpirun.
        """
        return self._use_mpi


    def set_rep_trav(self, run, reptrav, basename=''):
        """Set temporary directory for Code_Aster executions.
        """
        if reptrav:
            self.local_reptrav = self.global_reptrav = reptrav
        else:
            self.local_reptrav = self.global_reptrav = get_tmpname(run, basename=basename)
            # used shared directory only if more than one node
            if self.really():
                # and self.nbnode() > 1: shared_tmp is necessary if submission
                # node is not in mpihosts list
                self.global_reptrav = get_tmpname(run, run['shared_tmp'], basename)
            

    def get_exec_command(self, run, cmd_in, add_tee=False, env=None):
        """Return command to run Code_Aster.
        """
        # add source of environments files
        if env is not None:
            envstr = [". %s" % f for f in env]
            envstr.append(cmd_in)
            cmd_in = os.linesep.join(envstr)
        dict_val = {
            'cmd_in' : cmd_in,
            'var'    : 'EXECUTION_CODE_ASTER_EXIT_%s' % run['num_job'],
        }
        if add_tee:
            cmd_in = """(
%(cmd_in)s
echo %(var)s=$?
) | tee fort.6
""" % dict_val

        if not self.really():
            if add_tee:
                cmd_in += """
exit `grep -a %(var)s fort.6 | head -1 | sed -e 's/%(var)s=//'`""" % dict_val
            return cmd_in

        mpi_script = os.path.join(self.global_reptrav, 'mpi_script.sh')
        
        if self.nbnode() > 1:
            cp_cmd = '%s --with-as_run %s %s' \
                % (os.path.join(aster_root, 'bin', 'parallel_cp'),
                    " ".join(run.get_remote_args()),
                    self.global_reptrav)
        else:
            cp_cmd = 'cp -r'
        
        dict_mpi_args = {
            'mpirun_cmd'         : run['mpirun_cmd'],
            'mpi_get_procid_cmd' : run['mpi_get_procid_cmd'],
            'mpi_hostfile'       : run.get('mpi_hostfile'),
            'mpi_nbnoeud'        : self.nbnode(),
            'mpi_nbcpu'          : self.nbcpu(),
            'cmd_to_run'         : cmd_in,
            'wrkdir'             : self.global_reptrav,
            'local_wrkdir'       : self.local_reptrav + '.',
            'program'            : mpi_script,
            'cp_cmd'             : cp_cmd,
            'num_job'            : run['num_job'],
        }
        template = os.path.join(datadir, 'mpirun_template')
        if not run.Exists(template):
            run.Mess(ufmt(_(u'file not found : %s'), template), '<F>_FILE_NOT_FOUND')
        content = open(template, 'r').read() % dict_mpi_args
        run.DBG(content, all=True)
        open(mpi_script, 'w').write(content)
        os.chmod(mpi_script, 0700)
        # add comment because cpu/system times are not counted by the timer
        self._add_timer_comment(run)
        # mpirun/mpiexec
        command = run['mpirun_cmd'] % dict_mpi_args
        # need to initialize MPI session ?
        if run.get('mpi_ini'):
            command = run['mpi_ini'] % dict_mpi_args + " ; " + command
        # need to close MPI session ?
        if run.get('mpi_end'):
            command = command + " ; " + run['mpi_end'] % dict_mpi_args
        return command


    def nbcpu(self):
        """Return the number of cpu."""
        return self._nbcpu


    def nbnode(self):
        """Return the number of nodes"""
        return self._nbnode


    def reptrav(self, procid=None):
        """Return global workdir or local workdir on proc #i.
        """
        if procid is None or not self.really():
            res = self.global_reptrav
        else:
            res = '%s.proc.%d' % (self.local_reptrav, procid)
        return res


    def add_to_timer(self, run, text, title):
        """Add info cpus to the timer reading 'text'.
        """
        if not hasattr(run, 'timer') or not self.really():
            return
        l_info_cpu = re.findall('PROC=([0-9]+) INFO_CPU=(.*)', text)
        l_info_cpu.sort()
        for procid, infos in l_info_cpu:
            cpu_mpi, sys_mpi = 0., 0.
            id_timer = _(u'  %s - rank #%s') % (title, procid)
            run.timer.Stop(id_timer)
            try:
                l_ch = re.sub(' +', ' ', infos.strip()).split(' ')
                assert len(l_ch) == 4, 'cpu+sys, cpu, sys, elapsed'
                cpu_mpi = float(l_ch[1])
                sys_mpi = float(l_ch[2])
            except (AssertionError, TypeError):
                run.Mess(_(u'Unable to retreive CPU info for proc #%s') % procid)
                run.DBG('Info CPU proc #%s' % procid, infos, l_ch)
            run.timer.Add(id_timer, cpu_mpi, sys_mpi)
            # add time spend in 'Code_Aster run' and to total
            run.timer.Add(title, cpu_mpi, sys_mpi, to_total=True)


    def _add_timer_comment(self, run):
        """Add a warning when mpirun is called without Aster.
        """
        if hasattr(run, 'timer') and self.really():
            run.timer.AddComment(self.timer_comment, init=True)

