# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Functions usefully for an asrun client.
"""

import os
import os.path as osp
import re
from shutil import copyfile

from asrun.installation  import confdir
from asrun.common.rcfile import read_rcfile, decode_config_line
from asrun.system        import get_home_directory

PREFERENCES  = 'prefs'
SERVER_CONF = 'config_serveurs'

_INFO = 'astkrc_version'
_RC_VERSION = "1.1"


class CLIENT(object):
    """Usefull functions for an asrun client.
    """
    def __init__(self, rcdir, run):
        """Initialization
        """
        self._rcdir = osp.join(get_home_directory(), rcdir)
        self.run    = run
        # cache for user preferences and servers configuration
        self._pref  = None
        self._serv  = None
        self._serv_rcversion = None
        # exceptions to read resources files
        self._dict_exceptions_prefs = {
            '__all__' : { '_VIDE'    : '' },
        }
        self._dict_exceptions_server = {
            '__all__' : { '_VIDE'    : '' },
            'login'   : { 'username' : self.run.system.getuser_host()[0] },
        }


    def rcfile(self, filename, rcdir=None):
        """Return resource filename
        """
        rcdir = rcdir or self._rcdir
        return osp.join(rcdir, filename)
    

    def check_rcdir(self):
        """Check if rcdir exists and create it if necessary.
        """
        try:
            os.makedirs(self._rcdir)
        except OSError:
            pass


    def init_user_resource(self, filename):
        """Initialize 'rcdir'/'filename' if it does not exist.
        """
        if not osp.exists(self.rcfile(filename)):
            self.check_rcdir()
            copyfile(self.rcfile(filename, osp.join(confdir, 'astkrc')), self.rcfile(filename))


    def get_user_preferences(self, force=False):
        """Return user preferences.
        """
        if force or not self._pref:
            self.init_user_resource(PREFERENCES)
            self._pref = {}
            read_rcfile(self.rcfile(PREFERENCES), self._pref, exceptions=self._dict_exceptions_prefs)
        return self._pref


    def init_server_config(self, force=False, refresh=False):
        """Return servers configuration from 'rcdir'/config_serveurs
        """
        if force or not self._serv:
            self.init_user_resource(SERVER_CONF)
            self._serv = {}
            read_rcfile(self.rcfile(SERVER_CONF), self._serv, mcfact_key='serveur', mcsimp=['vers', 'noeud'],
                        exceptions=self._dict_exceptions_server)
            if self._serv.get(_INFO) is not None:
                try:
                    self._serv_rcversion = '%.1f' % self._serv[_INFO]
                except:
                    pass
                del self._serv[_INFO]
            self._pass_backward_compatibility()
        if refresh:
            self.refresh_server_config()


    def get_server_rcinfo(self):
        """Return informations about the servers resource file.
        """
        return self._serv_rcversion


    def get_server_list(self):
        """Return the list of available servers.
        """
        if not self._serv:
            return []
        return self._serv.keys()


    def get_server_config(self, server):
        """Return the configuration of 'server'
        """
        self.init_server_config()
        return self._serv.get(server, {})


    def get_dict_server_config(self):
        """Return the dict of the configuration of all servers
        """
        return self._serv


    def save_server_config(self):
        """Write 'rcdir'/config_serveurs with current values.
        """
        content = ["""# AUTOMATICALLY GENERATED - DO NOT EDIT !
astkrc_version : %s
#""" % _RC_VERSION,]
        
        sorted_servers = [(self._serv[serv]['__id__'], serv) for serv in self._serv.keys()]
        sorted_servers.sort()
        for iocc, server in sorted_servers:
            lkeys = self._serv[server].keys()
            lkeys.sort()
            content.append('serveur : %s' % server)
            for key in lkeys:
                if key.startswith('__'):
                    continue
                value = self._serv[server][key]
                if value == "":
                    value = "_VIDE"
                if key in ('vers', 'noeud'):
                    for val in value.split():
                        content.append('%s : %s' % (key, val))
                else:
                    content.append('%s : %s' % (key, value))
            content.append("#")
        open(self.rcfile(SERVER_CONF), 'w').write(os.linesep.join(content))


    def refresh_server_config(self, server_list=None):
        """Refresh configuration of each server presents in self._serv.
        """
        assert self.run is not None, "ASTER_RUN object is necessary to call refresh_server_config"
        server_list = server_list or self.get_server_list()
        for server in server_list:
            cfg = self._serv.get(server)
            if cfg.get('etat') != "on":
                continue
            cmd = [osp.join(cfg['rep_serv'], 'bin', 'as_run'),]
            cmd.extend(self.run.get_as_run_args())
            cmd.append("--info")
            iret, output = self.run.Shell(" ".join(cmd), mach=cfg['nom_complet'], user=cfg['login'])
            self._serv[server].update(self._parse_info(output))


    def _parse_info(self, content):
        """Parse information write by as_run --info
        """
        info = {}
        mat = re.search("@SERV_VERS@(.*)@FINSERV_VERS@", content, re.DOTALL)
        if mat is not None:
            try:
                key, val = decode_config_line(mat.group(1))
                info["asrun_vers"] = val
            except:
                pass
        mat = re.search("@PARAM@(.*)@FINPARAM@", content, re.DOTALL)
        if mat is not None:
            for line in mat.group(1).splitlines():
                try:
                    key, val = decode_config_line(line)
                    info[key] = val
                except:
                    pass
        mat = re.search("@VERSIONS@(.*)@FINVERSIONS@", content, re.DOTALL)
        if mat is not None:
            lvers = []
            for line in mat.group(1).splitlines():
                try:
                    key, val = decode_config_line(line)
                    lvers.append(val)
                except:
                    pass
            info['vers'] = ' '.join(lvers)
        mat = re.search("@NOEUDS@(.*)@FINNOEUDS@", content, re.DOTALL)
        if mat is not None:
            lnode = []
            for line in mat.group(1).splitlines():
                try:
                    key, val = decode_config_line(line)
                    lnode.append(val)
                except:
                    pass
            info['noeud'] = ' '.join(lnode)
        return info


    def _pass_backward_compatibility(self):
        """Pass backward compatibility conversion.
        """
        for server in self.get_server_list():
            cfg = self._serv.get(server)
            if cfg.get("asrun_vers") is not None and cfg["asrun_vers"] < "01.08.00":
                cfg["rep_serv"] = re.sub("/ASTK/ASTK_SERV/bin", "/ASTK/ASTK_SERV", cfg["rep_serv"])
            else:
                cfg["rep_serv"] = re.sub("/ASTK/ASTK_SERV/bin", "", cfg["rep_serv"])
                cfg["rep_serv"] = re.sub("/ASTK/ASTK_SERV",     "", cfg["rep_serv"])
            self._serv[server].update(cfg)
        

