/* $Id: Engine.cpp,v 1.45 2003/03/22 23:51:49 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Engine/Engine.h>

#include <Ark/ArkConfig.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkNetwork.h>
#include <Ark/ArkCollision.h>

#include <math.h>

#include <sstream>

#ifdef WIN32
#undef LoadLibrary
#endif

namespace Ark
{
   static Engine *g_Engine;
   
   Engine::Engine (Cache *cache, bool server) : 
      m_World(NULL),
      m_IsServer (server)
   {
      // Initialize scripting engine
      String type = Sys()->Cfg()->GetStr("engine::ScriptLanguage","lua");

      m_Script = ScriptFactory::CreateScript("ark::Script");
      
      assert (cache != NULL);
      m_Cache = cache;

      m_Cache->SetColSystem(ColSystemFactory::CreateColSystem
			    ("ark::Collision"));
      
      // Set global engine.
      g_Engine = this;
   }
   
   Engine::~Engine ()
   {
      delete m_Script;
      delete m_World;
      
      g_Engine = 0;
   }
   
   /** Load the world whose path is \c path. */
   void
   Engine::LoadWorld (const String &path)
   {
      if (m_World != NULL)
	 return;

      m_WorldName = path;

      Sys()->Log ("Loading world \"%s\"...\n", path.c_str());
      m_PlayerStarts.clear();

      // Create a new world.

      m_World = WorldFactory::CreateWorld("ark::World",m_Cache, this);
      m_World->Load (path);

      if (!IsServer())
	 m_World->Init (WORLD_HAS_COLLISION | WORLD_HAS_RENDERING); 
      else
	 m_World->Init (WORLD_HAS_COLLISION); 

      /// Load scenery.
      Config worldcfg;
      worldcfg.Load( path + "/world.cfg" );

      String elib = Sys()->Cfg()->GetStr("server::EntityLib",
					 "{game}/scripts/entities.lib");
      m_Script->LoadLibrary (elib);

      String script = worldcfg.GetStr("world::Script", String());
      if (! script.empty() )
	 m_Script->LoadScript (script);

      String entities = worldcfg.GetStr("world::Entities", String());
      if (! entities.empty() )
      {
	 AutoReadStream file (entities);
	 Lexer lexer (entities, file.Get());

	 /// Read until EOF is reached.
	 EntryList entries;

	 while (m_Script->m_ClassList->Read (lexer, &entries))
	 {
	    EngineEntity *ent = m_Script->NewEntity();
	    ent->SetEntries (this, entries);
	    ent->Create ();
	    m_World->Add (ent);

	    entries.clear();
	 }
      }

      String callbacks = worldcfg.GetStr("world::Callbacks", String());
      if (! callbacks.empty() )
      {
	 AutoReadStream file (callbacks);
	 Lexer lexer (callbacks, file.Get());

	 /// Read untile EOF is reached.
	 EntryList entries;

	 while (m_Script->m_ClassList->Read (lexer, &entries))
	 {
	    EntryList::iterator i = entries.find("name");

	    if (i == entries.end() ||
		i->second.m_Type != Entry::STRING ||
		*i->second.d_str == "None")
	    {
	       lexer.Error ("Missing name for previous entity...");
	       continue;
	    }

	    EngineEntity *ent = (EngineEntity*) m_World->FindByName (*i->second.d_str);

	    if (ent == NULL)
	    {
	       std::ostringstream os;
	       os << "'" << *(i->second.d_str) << "' isnt' an entity...";
	       lexer.Error( os.str() );
	       continue;
	    }

	    ent->SetCallback (this, entries);
	    entries.clear();
	 }
      }

      m_World->Init (WORLD_HAS_PATHFINDING);
      m_Script->Main();
   }

   static String
   GetUserFile (const String &user,
		const String &sub)
   {
       std::ostringstream os;
       os << "{gamevar}/users/" << user << "/" << sub;
       return os.str();
   }


   /** Try to 'login' an entity. This means that the world part in
    * which this entity lay is loaded, then all entity states are
    * loaded ; finally the entity is returned. The \c password might
    * be used for a networked game.
    */
   Entity *
   Engine::Login (const String &login,
		  const String &password)
   {
      String defwrl;

      if (!IsServer())
      {
	 // FIXME: Read that from the savegame.
	 defwrl = Sys()->Cfg()->GetStr ("client::DefaultWorld", "0000x0000");
      }
      else
	 defwrl = Sys()->Cfg()->GetStr ("server::DefaultWorld", "0000x0000");

      LoadWorld( "{game}/world/" + defwrl );

      // Create a new entity from template.
      String tmpl = GetUserFile (login, "template");
      AutoReadStream stream (tmpl);
      std::ifstream& f = stream.Get();
      
      // Can't read template, giving up..
      if (!f.is_open())
	 return 0;

      Lexer lex (tmpl, f);
      EntryList entries;
      m_Script->m_ClassList->Read (lex, &entries);

      // Create a new entity & set entries
      EngineEntity *ent = m_Script->NewEntity ();
      ent->SetEntries (this, entries);
      ent->Create ();

      if (m_PlayerStarts.empty())
	 ent->m_MState.m_Position = Vector3();
      else
	 ent->m_MState.m_Position = Vector3 (m_PlayerStarts [0]);

      m_World->Add (ent);
      //m_Cache->Clear(); // Removed unused ressources from memory.

      return ent;
   }

   bool
   Engine::Update (scalar delta_time)
   {
      if (m_World == 0)
	 return false;

      /// Update the world...
      m_World->Update(delta_time);

      std::vector<Entity*> &entities = m_World->GetEntities();
      std::vector<Entity*>::iterator it;

      for (it = entities.begin(); it != entities.end(); ++it)
      {
	 EngineEntity *entity = (EngineEntity*) *it;

	  std::vector<EntityMessage>::iterator msg;	 
	 for (msg = entity->m_OutMessages.begin();
	      msg != entity->m_OutMessages.end();
	      ++msg)
	 {
	    EntityMessage ms = *msg;
	    ms.m_Entity = entity;

	    EngineEntity *goal = (EngineEntity*)msg->m_Entity;
	    goal->m_Messages.push_back (ms);
	    goal->EvTell ();
	 }

	 entity->m_OutMessages.clear();

	 if (entity->m_Goal.m_Type != GOAL_NULL)
	 {
	    switch (entity->m_Goal.m_Type)
	    {
	       case GOAL_ENTITY:
		  entity->m_Path.SetGoal (entity->m_Goal.m_Entity);
		  break;
	       case GOAL_POSITION:
		  entity->m_Path.SetGoal (entity->m_Goal.m_Position);
		  break;
	       default:
		  break;
	    }

	    entity->m_Goal.m_Type = GOAL_NULL;
	 }

	 EntityCollision col;
	 while (entity->PeekCollision (&col))
	 {
	    entity->EvHit (col);
	    if (col.m_Potential == false)
	    {
	       Vector3 normalother = Vector3::ComputeNormal
		  (col.m_Pair.m_Triangle1.m_Coords[0],
		   col.m_Pair.m_Triangle1.m_Coords[1],
		   col.m_Pair.m_Triangle1.m_Coords[2]);
	       
	       scalar decal =
		  fabsf((entity->m_PreviousPos - entity->m_MState.m_Position)*normalother);
	       Vector3 authdir = normalother ^ Vector3(0,1,0);

	       
	       scalar decalauth =
		  (entity->m_PreviousPos - entity->m_MState.m_Position)*authdir;

	       entity->m_MState.m_Position = entity->m_PreviousPos + 
		  decal * normalother + decalauth*authdir;
	    }
	 }

	 entity->m_PreviousPos = entity->m_MState.m_Position;
      }

      return true;
   }

   void
   Engine::HintEntityRemoved (int entityID)
   {
      if (!IsServer()) return; 

      m_RemovedEntities.push_back(entityID);
   }

   void
   Engine::HintEntityAdded (int entityID)
   {
      if (!IsServer()) return; 

      m_AddedEntities.push_back(entityID);
   }

   bool
   Engine::WriteDelta (WriteStream &stream)
   {
      if (!m_World) return false;

      std::vector<int>::iterator it;
      
      // Removed entities.
      NetWriteByte(stream, NET_UPD_ENTITIES);
      for (it = m_RemovedEntities.begin(); it != m_RemovedEntities.end(); ++it)
	 NetWriteInt(stream, *it);

      NetWriteInt(stream, 0);

      // Added entities.
      for (it = m_AddedEntities.begin(); it != m_AddedEntities.end(); ++it)
	 NetWriteInt(stream, *it);

      NetWriteInt(stream, 0);

      // Changes.
      EntityList &entities = m_World->GetEntities();
      EntityList::iterator et;
      for (et = entities.begin(); et != entities.end(); ++et)
      {
	 if ((*et)->HasChanged())
	 {
	    std::cerr << (*et)->m_Name << " has changed\n";
	    NetWriteInt (stream, (*et)->m_ID);
	    (*et)->Write(stream);
	 }
      }

      NetWriteInt(stream, 0);
      return true;
   }

   bool
   Engine::WriteFull (WriteStream &stream)
   {
      EntityList &entities = m_World->GetEntities();
      EntityList::iterator et;

      NetWriteByte(stream, NET_UPD_ENTITIES);
      NetWriteInt(stream, 0); // No destroyed entity

      // Added entities.
      for (et = entities.begin(); et != entities.end(); ++et)
	 NetWriteInt(stream, (*et)->m_ID);

      NetWriteInt(stream, 0);

      // Changes.
      for (et = entities.begin(); et != entities.end(); ++et)
      {
	 NetWriteInt (stream, (*et)->m_ID);
	 (*et)->Write(stream, true);
      }
      NetWriteInt(stream, 0);

      return false;
   }

   void 
   Engine::ResetDelta ()
   {
      m_AddedEntities.clear();
      m_RemovedEntities.clear();

      if (m_World == NULL)
	 return;

      EntityList &entities = m_World->GetEntities();

      for (EntityLI et = entities.begin(); et != entities.end(); ++et)
	 (*et)->ClearChanged();
   }


   Engine *GetEngine()
   {
      return g_Engine;
   }

}

#define ARK_MODULE
#include <Ark/ArkFactoryReg.h>

class EngineFactory : public Ark::WorldUpdaterFactory
{
   public:
      virtual ~EngineFactory() {};
      virtual Ark::WorldUpdater *NewWorldUpdater(Ark::Cache *cache,
						 bool isServer)
      {
	 return new Ark::Engine(cache, isServer);
      }
};

ARK_REGISTER("ark::Updater::Engine", EngineFactory);
