/* $Id: RemoteUpdater.cpp,v 1.20 2003/02/04 20:00:43 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkNetwork.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkCollision.h>
#include <Engine/Engine.h>

#include "RemoteUpdater.h"

namespace Client
{

   class RemoteEntity : public Ark::Entity
   {
	 friend class RemoteUpdater;

      public:
	 RemoteEntity (Ark::World *wrl) : Ark::Entity (wrl) {}
	 virtual ~RemoteEntity () {}
   };

   /// Create a remote updater.
   RemoteUpdater::RemoteUpdater(Ark::Cache *cache) :
      m_World(NULL),
      m_Cache(cache),
      m_Socket(NULL),
      m_ClientEntityID(0)
   {
      std::cerr << "Creating remote updater\n";
      m_Socket = new Ark::Socket
	 (Ark::Sys()->Cfg()->GetStr("remoteUpdater::Server","localhost"),
	  Ark::Sys()->Cfg()->GetStr("remoteUpdater::Port","5555"),
	  "tcp");

      m_Cache->SetColSystem(Ark::ColSystemFactory::CreateColSystem
			    ("ark::Collision"));
   }
   
   RemoteUpdater::~RemoteUpdater()
   {
      if (m_World) delete m_World;
      if (m_Socket) delete m_Socket;
   }
   
   void
   RemoteUpdater::LoadWorld (const Ark::String &name)
   {
   }
   
   // Return the previously loaded world, or nil if no world has
   // been loaded.
   Ark::World *
   RemoteUpdater::GetWorld()
   {
      return m_World;
   }
   
   bool
   RemoteUpdater::HandleUpdates()
   {
      m_Socket->ReceiveData();
      while (Ark::ReadStream *pak = m_Socket->GetPacket())
      {
	 while (!pak->eof())
	 {
	    unsigned char msg, error;
	    Ark::NetReadByte (*pak, msg);
	    
	    switch (msg)
	    {
	       case Ark::NET_UPD_RESULT:
		  Ark::NetReadByte(*pak, msg);
		  Ark::NetReadByte(*pak, error);
		  
		  if (error != 0)
		  {
		     Ark::String errmsg;
		     Ark::NetReadString (*pak, errmsg);
		     std::cerr << "Remote error: " << errmsg << "\n";
		  }
		  return false;
		  break;
		  
	       case Ark::NET_UPD_ENTITIES:
		  HandleEntities (pak);
		  break;
		  
	       case Ark::NET_UPD_MESSAGES:
		  HandleMessages (pak);
		  break;
		  
	       case Ark::NET_UPD_CONNECT_INFO:
		  HandleConnection (pak);
		  break;
	    }
	 }
      }
      
      return !m_Socket->IsDead();
   }
   
   bool
   RemoteUpdater::HandleEntities (Ark::ReadStream *pak)
   {
      if (m_World == NULL) 
      {
	 Ark::Sys()->Fatal ("FIXME! I received the entities before I created "
			    "the world.");
      }

      int id;
      while (Ark::NetReadInt(*pak, id) && id != 0)
      {
	 std::cout << "Removed entity " << id << "\n";
	 m_World->Remove (m_World->Find(id));
      }
      
      while (Ark::NetReadInt(*pak, id) && id != 0)
      {
	 RemoteEntity *et = new RemoteEntity (m_World);
	 et->m_ID = id;
	 std::cout << "Added entity " << id << "\n";
	 
	 m_World->Add (et);
      }
      
      while (Ark::NetReadInt(*pak, id) && id != 0)
      {
	 Ark::Entity *et = m_World->Find(id);
	 
	 if (!et)
	    Ark::Sys()->Fatal ("FIXME! I received a wrong entity ID %d.",
			       id);
	 
	 et->Read (*pak);
	 std::cout << "Updated entity " << et->m_Name << "\n";
      }
      
      return true;
   }

   bool
   RemoteUpdater::HandleConnection(Ark::ReadStream *pak)
   {
      Ark::String terrain;
      Ark::NetReadString (*pak, terrain);
      Ark::NetReadInt (*pak, m_ClientEntityID);
      
      m_World = Ark::WorldFactory::CreateWorld("ark::World", m_Cache, this);
      m_World->Load (terrain);
      m_World->Init (Ark::WORLD_HAS_COLLISION | Ark::WORLD_HAS_RENDERING);
      return true;
   }

   bool
   RemoteUpdater::HandleMessages(Ark::ReadStream *pak)
   {
      Ark::Entity *my = m_World->Find(m_ClientEntityID);
      int id = 0;
   
      while (Ark::NetReadInt(*pak, id) && id != 0)
      {
	 Ark::Entity *et = m_World->Find(id);
	 Ark::String msg; int nanswers;
	 std::vector<Ark::String> answers;
	 
	 Ark::NetReadString(*pak, msg);
	 std::cerr << msg <<"\n";
	 Ark::NetReadInt(*pak, nanswers);
	 
	 for (int i = 0; i < nanswers; i++)
	 {
	    Ark::String ans; 
	    Ark::NetReadString (*pak, ans);
	    answers.push_back(ans);
	 }
	 
	 my->m_Messages.push_back (Ark::EntityMessage(et, msg, &answers));
      }
      
      return true;
   }
   
   Ark::Entity *
   RemoteUpdater::Login (const Ark::String &login,
			 const Ark::String &password)
   {
      Ark::Packet *outPacket = new Ark::Packet;
	  std::stringstream& out = outPacket->m_Stream;
      
      // Send the login packet
      Ark::NetWriteByte(out, Ark::NET_MSG_CONNECT);
      Ark::NetWriteInt(out, Ark::NET_PROTOCOL_VERSION);
      Ark::NetWriteString (out, login);
      Ark::NetWriteString (out, password);
      
      m_Socket->SetBlocking(true);
      m_Socket->SendPacket(outPacket); 
	  outPacket->Unref();

      m_Socket->SendData();
      
      // Try to get an answer
      while (HandleUpdates() == true &&
	     (m_ClientEntityID == 0
	      || m_World->Find(m_ClientEntityID) == NULL))
	 ;
      
      m_Socket->SetBlocking(false);
      return m_World->Find(m_ClientEntityID);
   }
	 
   /// Make the world in sync with the remote systems.
   bool
   RemoteUpdater::Update (scalar delta_time)
   {
      m_World->Update();
      
      RemoteEntity *my = (static_cast<RemoteEntity*> 
			  (m_World->Find(m_ClientEntityID)));
      Ark::Packet *outPacket = new Ark::Packet;
	  std::stringstream& out = outPacket->m_Stream;
      
      std::vector<Ark::EntityMessage>::iterator it;
      
      for (it = my->m_OutMessages.begin();
	   it != my->m_OutMessages.end(); it++)
      {
	 std::cout << "Sending message " << it->m_Message;
	 Ark::NetWriteByte(out, Ark::NET_MSG_ENTITY_TELL);
	 Ark::NetWriteInt(out, it->m_Entity->m_ID);
	 Ark::NetWriteString (out, it->m_Message);
      }
      
      my->m_OutMessages.clear();

      if (my->m_Changed & Ark::Entity::GOAL)
      {
	 if (my->m_Goal.m_Type == Ark::GOAL_NULL)
	    Ark::NetWriteByte(out, Ark::NET_MSG_ENTITY_SET_NOGOAL);
	 else if (my->m_Goal.m_Type == Ark::GOAL_POSITION)
	 {
	    Ark::NetWriteByte(out, Ark::NET_MSG_ENTITY_SET_GOAL);
	    Ark::NetWriteScalar(out, my->m_Goal.m_Position.X);
	    Ark::NetWriteScalar(out, my->m_Goal.m_Position.Y);
	    Ark::NetWriteScalar(out, my->m_Goal.m_Position.Z);
	 }
	 
	 my->m_Changed &= ~Ark::Entity::GOAL;
      }
      
      m_Socket->SendPacket(outPacket); 
	  outPacket->Unref();
      m_Socket->SendData();

      HandleUpdates();
      return true;
   }

   Ark::WorldUpdater *CreateUpdater (const Ark::String &name,
				     Ark::Cache *cache)
   {
      return Ark::WorldUpdaterFactory::CreateWorldUpdater(name, cache);
   }
}

#undef ARK_MODULE
#include <Ark/ArkFactoryReg.h>

class RemoteUpdaterFactory : public Ark::WorldUpdaterFactory
{
   public:
      virtual ~RemoteUpdaterFactory() {};
      virtual Ark::WorldUpdater *NewWorldUpdater(Ark::Cache *cache,
						 bool isServer)
      {
	 return new Client::RemoteUpdater(cache);
      }
};

ARK_REGISTER("ark::Updater::Remote", RemoteUpdaterFactory);
