/* $Id: ArkFontBitmap.cpp,v 1.7 2003/03/22 23:51:49 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkFontBitmap.h>

#include <Ark/ArkCache.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkSystem.h>

#include "math.h"

namespace Ark
{

   /////////////////////////////////
   // FIXME: move that in some file.
   /////////////////////////////////
static String DashToken(const String& from, String::size_type& pos)
{
    if (pos == String::npos) 
	return String();

    String::size_type next = from.find("-", pos);

    String part(from, pos, next - pos);
    pos = next;

    if (pos != String::npos)
	++pos;

    return part;
}

static uchar HexToInt (char c1, char c2)
{
   char str[5] = {'0', 'x', c1, c2, '\0'};
   return (unsigned int) (strtol (str, NULL, 16) & 0xFF);
}

/////////////////////////////////////////////////////////////////////////
   class BitmapFontData
   {
   public:
	 Material m_Material;
	 struct Char
	 {
	       int MinX, MinY, MaxX, MaxY;
	 };

	 Char m_Set[256]; //FIXME: unicode, anyone ?
	 std::map<String,Char> m_ExtendedSet;

	 int m_ImageWidth, m_ImageHeight;
	 
	 VertexBuffer m_TempVB;

      public:
	 BitmapFontData();
	 ~BitmapFontData();

	 void Read (const String &filename, Ark::Cache &cache);
	 bool DrawStringEx (Renderer &render,
			    const String &string,
			    int x, int y,
			    FontAlignment align,
			    Color col,
			    int size);	 

         bool GetStringSize (const String &string, int *w, int *h,
			     int size);

         bool GetStringBounds (const String &string,
			       FontAlignment align,
			       int *min, int *max,
			       int size);
   };


   BitmapFontData::BitmapFontData() : m_Material("")
   {
   }

   BitmapFontData::~BitmapFontData()
   {
   }

   void
   BitmapFontData::Read (const String &filename, Ark::Cache &cache)
   {
      memset(&m_Set[0], 0, sizeof(m_Set));

      m_TempVB.SetFormat(VertexBuffer::VB_HAS_COORD|VertexBuffer::VB_HAS_UV0);
      m_TempVB.Resize(6*50);
      
      AutoReadStream file (filename);
      
      Lexer lexer(filename, file.Get());
      
      if (!lexer.CheckToken("ArkFont")) return;
      if (!lexer.CheckToken("{")) return;

      if (!lexer.CheckToken("ImageSize") || !lexer.CheckToken("=")) return;

      scalar sizes[2]; lexer.ReadScalarVector(sizes,2);
      m_ImageWidth = (int)sizes[0];
      m_ImageHeight = (int)sizes[1];

      m_Material.Parse(filename, lexer, cache);

      String token;
      while(1)
      {
	 token = lexer.GetToken();

	 if (token == "}") {lexer.UngetToken(); break;}
	 if (!lexer.CheckToken("=")) return;
	 
	 scalar bounds[4];
	 lexer.ReadScalarVector(bounds,4);

	 token = UnquoteString(token);
	 if (!token.size())
	 {
	    lexer.Error("empty character");
	    continue;
	 }

	 Char *ch = NULL;
	 if (token.size() > 1)
	    ch = &m_ExtendedSet[token];
	 else
	    ch = &m_Set[(uchar)token[0]];

	 for (int i =0; i < 4; i++) ((int*)ch)[i] = (int)bounds[i];
      }
      
      if (!lexer.CheckToken("}")) return;
   }

   void
   RecognizeHTMLEntity (const String &str, size_t &i, BitmapFontData::Char &ch,
			std::map<String, BitmapFontData::Char> &extendedSet)
   {
      if (str[i] == '&')
      {
	 String chname;
	 
	 ++i;  // Skip the amp.
	 while (i < str.size() && str[i] != ';')
	 {
	    chname.push_back(str[i]);
	    ++i;
	 }
	 
	 if (i >= str.size()) 
	 {
	    Sys()->Warning("Unmatched '&' in html-style entity parsing "
			   "for string %s", QuoteString(str).c_str());
	 }
	 else
	    ++i;
	 
	 std::map<String,BitmapFontData::Char>::iterator it =
	    extendedSet.find(chname);
	 
	 if (it == extendedSet.end())
	 {
	    Sys()->Warning("Unrecognised html-style entity '%s' in "
			   "string %s", chname.c_str(),
			    QuoteString(str).c_str());
	 }
	 else
	    ch = it->second;
      }
   }

   bool
   BitmapFontData::DrawStringEx (Renderer &render,
				 const String &string,
				 int x, int y,
				 FontAlignment align,
				 Color color,
				 int size)
   {
      m_Material.m_Passes[0].m_BlendColor = color;
      
      scalar curx = static_cast<scalar>(x);
      scalar cury = static_cast<scalar>(y);
      int charsleft = m_TempVB.Size()/6, curvect = 0;
      scalar factor = float(size)/3.5f;

      for (size_t i = 0; i < string.size(); ++i)
      {
	 Char ch = m_Set[(uchar)string[i]];

	 RecognizeHTMLEntity(string, i, ch, m_ExtendedSet);

	 if (align == CENTER_LEFT)
	    cury = y - ((ch.MaxY-ch.MinY)*factor)/2.0f;

	 scalar maxx = curx + (ch.MaxX - ch.MinX) * factor,
	        maxy = cury + (ch.MaxY - ch.MinY) * factor;
	 scalar chmaxx = ch.MaxX/scalar(m_ImageWidth);
	 scalar chminx = ch.MinX/scalar(m_ImageWidth);
	 scalar chmaxy = ch.MaxY/scalar(m_ImageHeight);
	 scalar chminy = ch.MinY/scalar(m_ImageHeight);

	 // First triangle  
	 m_TempVB.Coord(curvect) = Vector3(curx, cury, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chminx, chminy); curvect++;

	 m_TempVB.Coord(curvect)=Vector3(maxx, cury, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chmaxx, chminy); curvect++;

	 m_TempVB.Coord(curvect)=Vector3(maxx, maxy, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chmaxx, chmaxy); curvect++;

	 // Second one
	 m_TempVB.Coord(curvect) = Vector3(maxx, maxy, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chmaxx, chmaxy); curvect++;

	 m_TempVB.Coord(curvect) = Vector3(curx, maxy, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chminx, chmaxy); curvect++;

	 m_TempVB.Coord(curvect) = Vector3(curx, cury, 0.0);
	 m_TempVB.UV0(curvect) = Vector2(chminx, chminy); curvect++;

	 charsleft--;
	 curx = maxx;
	 
	 // Flush
	 if (charsleft == 0 || i == (string.size()-1))
	 {
	    render.RenderBlock(m_Material, PRIM_TRIANGLES,
			       m_TempVB, curvect);
	    charsleft = m_TempVB.Size()/6; curvect = 0;
	 }
      }

      return true;
   }

   bool
   BitmapFontData::GetStringSize (const String &string, int *w, int *h,
				  int size)
   {
      int lmin[2], lmax[2];
      GetStringBounds(string, CENTER_LEFT, lmin, lmax, size);

      if(h) *h = lmax[1] -  lmin[1];
      if(w) *w = lmax[0] -  lmin[0];

      return true;
   }

   bool
   BitmapFontData::GetStringBounds (const String &string,
				    FontAlignment align,
				    int *min, int *max,
				    int size)
   {
      static int lmin[2], lmax[2];
      if (min == NULL) min = lmin;
      if (max == NULL) max = lmax;

      min[0] = max[0] = min[1] = max[1] = 0;

      scalar curx = 0, cury = 0;
      scalar factor = float(size)/3.5f;

      for (size_t i = 0; i < string.size(); ++i)
      {
	 Char ch = m_Set[(uchar)string[i]];

	 RecognizeHTMLEntity(string, i, ch, m_ExtendedSet);

	 if (align == CENTER_LEFT)
	    cury = - ((ch.MaxY-ch.MinY)*factor)/2.0f;

	 scalar maxx = curx + (ch.MaxX - ch.MinX) * factor,
	        maxy = cury + (ch.MaxY - ch.MinY) * factor;

	 if (cury < min[1]) min[1] = (int)floor(cury);
	 if (maxy > max[1]) max[1] = (int)ceil(maxy);
	 if (curx < min[0]) min[0] = (int)floor(curx);
	 if (maxx > max[0]) max[0] = (int)ceil(maxx);

	 curx = maxx;
      }

      return true;      
   }

   /////////////////////////////////////////////////////////////////
   /////////////////////////////////////////////////////////////////


   BitmapFont::BitmapFont(const String &name) : Font(name),m_RefFont(NULL){}
   BitmapFont::~BitmapFont() 
   {
      if (m_RefFont) m_RefFont->Unref();
      if (m_BmData) delete m_BmData;
   }
   
   bool 
   BitmapFont::Load (Cache *cache, const String &filename)
   {
      m_BmData = NULL;
      m_RefFont = NULL;

      String::size_type tok = 0;
      String font = DashToken(filename, tok);
      String size = DashToken(filename, tok);
      String color = DashToken(filename, tok);
      
      if (size == "")
      {
	 m_BmData = new BitmapFontData();
	 m_BmData->Read(filename, *cache);
	 return true;
      }

      if (! Ark::StringToInt (size, m_Size))
      {
	 Sys()->Warning ("Bad size specification for font desc '%s' "
			 "(size=%s)\n", filename.c_str(), size.c_str());
	 return false;
      }
      
      if (color.size() != 6)
      {
	 Sys()->Warning ("Bad color specification for font desc '%s'\n", 
                         filename.c_str());
	 return false;
      }
      
      uchar R = HexToInt (color[0], color[1]);
      uchar G = HexToInt (color[2], color[3]);
      uchar B = HexToInt (color[4], color[5]);
      
      m_Color = Color (R, G, B);

      font = "{fonts}/" + font + ".fnt";
      m_RefFont = static_cast<BitmapFont*>(cache->Get (V_FONT, font));
    
      return m_RefFont != 0;  
   }

   bool
   BitmapFont::DrawString (Renderer &render,
			   const String &string,
			   int x, int y,
			   FontAlignment align /* = CENTER_LEFT */)
   {
      if (m_RefFont != NULL)
	 return m_RefFont->m_BmData->DrawStringEx(render,string,x,y,
					align,m_Color,m_Size);
      else if (m_BmData != NULL)
	 return m_BmData->DrawStringEx(render,string,x,y,
				       align,Color(1.0f,1.0f,1.0f),4);

      return false;
   }

   bool
   BitmapFont::GetStringSize (const String &string, int *w, int *h)
   {
      if (m_RefFont != NULL)
	 return m_RefFont->m_BmData->GetStringSize(string,w,h,m_Size);
      else if (m_BmData != NULL)
	 return m_BmData->GetStringSize(string,w,h,4);

      return false;
   }

   /// Get the bounds of a string using this font and properties.
   bool
   BitmapFont::GetStringBounds (const String &string,
				FontAlignment align,
				int *min, int *max)
   {
      if (m_RefFont != NULL)
	 return m_RefFont->m_BmData->GetStringBounds(string,align,min,
						     max,m_Size);
      else if (m_BmData != NULL)
	 return m_BmData->GetStringBounds(string,align,min,max,4);

      return false;
   }
   
}
