/*
 * html.h: HTML output format for agedu.
 */

struct html_config {
    /*
     * If "format" is non-NULL, it is treated as an sprintf format
     * string which must contain exactly one %lu and no other
     * formatting directives (other than %%, which doesn't count);
     * this will be used to construct URLs to use in hrefs
     * pointing to queries of other related (parent and child)
     * pathnames.
     */
    const char *format;

    /*
     * If "rootpage" is non-NULL, it overrides "format" to give a
     * special name (e.g. "index.html") to the top-level page of the
     * index.
     */
    const char *rootpage;

    /*
     * Time stamps to assign to the extreme ends of the colour
     * scale. If "autoage" is true, they are ignored and the time
     * stamps are derived from the limits of the age data stored
     * in the index.
     */
    int autoage;
    time_t oldest, newest;

    /*
     * Specify whether to show individual files as well as
     * directories in the report.
     */
    int showfiles;
};

/*
 * Generate an HTML document containing the results of a query
 * against the pathname at a given index. Returns a dynamically
 * allocated piece of memory containing the entire HTML document,
 * as an ordinary C zero-terminated string.
 *
 * 'downlink' is TRUE if hyperlinks should be generated for
 * subdirectories. (This can also be disabled by setting cfg->format
 * to NULL, but that also disables the upward hyperlinks to parent
 * directories. Setting cfg->format to non-NULL but downlink to NULL
 * will generate uplinks but no downlinks.)
 */
char *html_query(const void *t, unsigned long index, 
		 const struct html_config *cfg, int downlink);

/*
 * Recursively output a dump of lots of HTML files which crosslink
 * to each other. cfg->format and cfg->rootpath will be used to
 * generate the filenames for both the hyperlinks and the output
 * file names; the file names will have "pathprefix" prepended to
 * them before being opened.
 *
 * "index" and "endindex" point to the region of index file that
 * should be generated by the dump, which must be a subdirectory.
 *
 * "maxdepth" limits the depth of recursion. Setting it to zero
 * outputs only one page, 1 outputs the current directory and its
 * immediate children but no further, and so on. Making it negative
 * gives unlimited depth.
 *
 * Return value is 0 on success, or 1 if an error occurs during
 * output.
 */
int html_dump(const void *t, unsigned long index, unsigned long endindex,
	      int maxdepth, const struct html_config *cfg,
	      const char *pathprefix);
