/***************************************************************************

Hitachi FD1089A/FD1089B encryption emulation

Decryption tables provided by Charles MacDonald
Decryption algorithm by Nicola Salmoria

The FD1089 is a 680000 with built-in encryption.
It contains some battery-backed RAM, when the battery dies the CPU stops
working.

Both opcodes and data are encrypted, using different (but related) tables.
Decryption works on 16-bit words, but only 8 bits are affected, the other 8 are
left untouched. Opcodes are always encrypted; data can be not encrypted, this
is necessary otherwise RAM would not work as expected (writing data and reading
it back would return a different number).

The FD1089A and FD1089B work in the same way, but the decryption tables are
different. The internal RAM only contains the code of the table to use at every
address (only 12 bits of the address are used, so the encryption repeats).

The FD1089 design is clearly derived from the MC8123. The MC8123 is a Z80 so it
made sense to encrypt all 8 data bits and use 12 of the 16 address bits. It makes
a lot less sense to encrypt only half of the 16 data bits; using 12 of the 24
address bits might be ok, but not in the way it was done. The choice of address
bits to use was probably dictated by the need to not encrypt data in certain areas
of the address space, so they had to include the top 8 bits of the address.
However this means that if you pick e.g. area 000000-00FFFF, where most of the
program code resides, just 4 address bits affect the encryption, making it very
weak when compared to the MC8123. Out of the 16KB of internal RAM, you need less
than 128 bytes to decrypt a whole game - the rest is not used. A waste of space
and security. Also, since only 8 of the 16 bits are encrypted, it is very easy
to use the unencrypted ones to search for known sequences of code or data.

It is almost certain that a linear congruential generator was used to generate
the keys, like for MC8123 and FD1094, however this hasn't been figured out yet.
The keys that control decryption are currently arbitrarily assigned so the only
thing one can expect is that there should be a fixed 256 entries table to map the
LCG output to the current arbitrary keys.

Note that when both FD1089A and FD1089B versions of a game exist, they use the
same key. There are small differences in the current keys (only bit 2 of some
values) which are caused by the arbitrary assignment of keys.

Known games that use this CPU are:

CPU #   Status   Game                     Notes
--------- --- ------------------------ ------------------
FD1089A
317-0018  [1] Action Fighter
317-0021? [2] Alex Kidd
317-0022  [1] Dunk Shot
317-0028  [2] Defense                  same key as 317-0028
317-0033  [1] Alien Syndrome           same key as 317-0037
317-0167  [2] Aurail                   same key as 317-0168

FD1089B
317-0013A [1] Enduro Racer
317-0024  [2] Time Scanner             FD1089B 6J2 317-0024
317-0027  [2] SDI                      same key as 317-0027
317-0034  [1] Super Hang-On
317-0037  [2] Alien Syndrome           same key as 317-0033
317-0168  [1] Aurail                   same key as 317-0167
317-5021  [3] Sukeban Jansi Ryuko

[1] Complete
[2] Partial tables derived by comparison with unencrypted version
[3] Partial tables derived staring at the ROMs intensely

***************************************************************************/

#include "driver.h"


/* 317-0013A FD1089B */
static const UINT8 key_0013A[0x2000] =
{
	/* data */
	0xe8,0x96,0xcc,0x84,0x7f,0x25,0x35,0x48,0x45,0x72,0x12,0x92,0x41,0x51,0xe1,0x40,
	0x61,0x32,0xca,0x4e,0x8e,0x09,0x00,0x9e,0xe1,0xbb,0x58,0xb3,0xf4,0x70,0x79,0x96,
	0x09,0xa9,0x40,0xd4,0x0f,0x10,0xa8,0xd1,0x88,0xcf,0x0f,0xee,0xae,0x09,0x0e,0x8f,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xde,0x1c,0x33,0xbf,0x2c,0xd8,0x04,0x06,0x22,0x16,0x50,0x30,0xdc,0xdd,0x6c,0x88,
	0xa0,0x55,0x8c,0xd0,0x08,0x9b,0xb3,0xa4,0xb9,0xb9,0x69,0x8a,0xde,0xfd,0x5d,0x3c,
	0x5d,0xd5,0xb9,0x63,0x07,0x6e,0x4e,0x36,0xe5,0x8a,0x47,0x2c,0xcd,0x3e,0xf2,0xbe,
	0x19,0x8a,0xe1,0x1c,0x75,0xb8,0xa2,0xb9,0x73,0x37,0x9f,0xb2,0xd9,0xb9,0x18,0x14,
	0xe5,0xd0,0xcb,0xe0,0x6c,0x4e,0xa6,0x6c,0xbb,0x11,0x75,0x70,0xa4,0x85,0x51,0xa1,
	0xad,0xaa,0x82,0xce,0xea,0xe5,0x2e,0xfc,0x44,0x97,0xa5,0x0b,0x25,0x89,0xc2,0x26,
	0x7e,0xf2,0x6f,0x35,0x66,0xc0,0x60,0xdd,0x8c,0x75,0xaf,0x1a,0x8f,0x46,0x7c,0x17,
	0xcc,0x64,0xda,0x49,0xef,0x2b,0xc3,0x66,0x43,0xf6,0x6f,0x96,0x65,0x28,0x88,0x1f,
	0x36,0x26,0xd0,0xed,0x24,0xe8,0x03,0xb3,0x74,0xd3,0x8c,0x84,0x46,0x07,0xd2,0x3e,
	0x43,0x72,0x85,0x80,0x8d,0x97,0x08,0x07,0x52,0xa6,0x63,0xfb,0x63,0x2f,0xe3,0xac,
	0x83,0x05,0xbb,0xce,0xaf,0xa7,0xd8,0xa1,0x8e,0x54,0xe7,0x99,0x1c,0x03,0xf5,0xe8,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xda,0x5d,0x18,0xb6,0xa0,0x95,0x69,0x2c,0x42,0x09,0x5e,0x06,0x10,0x96,0xf0,0x80,
	0x1a,0xd6,0xf4,0x4d,0x90,0xbe,0x36,0xe8,0x7d,0x0f,0xc8,0x16,0x29,0x6d,0x97,0xd0,
	0xae,0x5d,0x5d,0xbe,0xdc,0xcb,0xab,0x8f,0x28,0xbf,0x4e,0xbe,0x09,0xb3,0x9e,0x51,
	0x76,0x05,0x9c,0x54,0x3b,0x37,0x6e,0x58,0x5a,0x46,0xee,0xbe,0xa9,0x49,0x1f,0x47,
	0xb3,0x72,0xb8,0x2c,0x5b,0x39,0x03,0x33,0xce,0xd2,0x22,0xfd,0x4b,0xe6,0x3a,0x05,
	0xad,0x94,0x23,0xfe,0x79,0xd9,0x72,0xab,0xc6,0xd0,0x79,0xbf,0xa6,0xbf,0x37,0x4f,
	0x04,0x36,0x51,0x1c,0xe1,0x4d,0x97,0xb0,0x90,0x5e,0xfa,0xbf,0x2a,0x22,0x1c,0xcd,
	0x6c,0x6a,0xed,0x6f,0x69,0xa5,0x3b,0x21,0x40,0xb3,0x52,0x6a,0xba,0x2f,0xe9,0x7a,
	0x3a,0xde,0x1b,0x84,0x1c,0x3a,0x2f,0xe6,0xc5,0x59,0x76,0xaa,0xca,0x2a,0x1e,0xe0,
	0x91,0xf6,0x61,0x9d,0xfb,0xcb,0x75,0x61,0xec,0xae,0xe5,0x2f,0x39,0xdb,0xb0,0x4e,
	0xcc,0x34,0xc1,0xb6,0x9e,0x82,0xaf,0x81,0xd7,0xfe,0x34,0x20,0x0a,0xf8,0x29,0x03,
	0xb7,0xea,0x98,0x12,0x95,0x89,0xee,0x43,0x33,0xdc,0x12,0xdc,0x03,0xf1,0x81,0x6c,
	0x22,0xf6,0x51,0x02,0xf0,0xc0,0xdc,0x81,0x84,0x7a,0x19,0x09,0x91,0x55,0x1a,0x2e,
	0x0b,0xc0,0x92,0xd1,0xb1,0x46,0x37,0xf6,0xb2,0xa8,0xb9,0xa5,0x36,0x1e,0xac,0x75,
	0xe5,0xac,0x8a,0x19,0xd5,0x2e,0x24,0xf5,0x29,0x8f,0x78,0xe8,0x5e,0x53,0x3c,0xf0,
	0xd5,0x26,0xcc,0x50,0xfb,0x21,0xbc,0x44,0xe6,0x71,0x6d,0xa9,0xb4,0x04,0x4a,0xe2,
	0x20,0xae,0x0b,0xbf,0x62,0x85,0x60,0xb2,0xbc,0xdb,0x48,0x86,0x51,0x6e,0xd2,0x07,
	0xea,0x91,0xdd,0xf8,0xa0,0xbc,0xd7,0x1e,0x9f,0x37,0x03,0x32,0x70,0x6b,0x5a,0xd9,
	0x6d,0x4f,0x28,0x99,0xe1,0x3d,0xbc,0x4a,0x42,0x02,0xb1,0x47,0x8d,0x08,0x42,0x0c,
	0x08,0x4f,0x4c,0x5c,0xe5,0x2e,0x4b,0xdc,0x90,0xa2,0x90,0xe1,0xbc,0x87,0x64,0x85,
	0x49,0xf8,0xd0,0x3f,0x9b,0xfc,0x0e,0x99,0x12,0xac,0x1a,0xb7,0x13,0xb4,0x9a,0xfc,
	0x17,0xcd,0xb7,0x43,0xc2,0x3b,0xd2,0x34,0xeb,0xab,0xcf,0xf3,0x81,0x44,0x56,0x61,
	0xb5,0xb6,0x2f,0xe1,0x76,0x65,0x6a,0x6d,0x68,0x72,0x27,0xc7,0x10,0x7f,0xb4,0x7d,
	0xf4,0x83,0xec,0xf8,0xa3,0xa8,0x98,0xf2,0x4a,0xbd,0x32,0x22,0x18,0xd4,0x41,0xca,
	0x7b,0x2e,0x02,0x5d,0x33,0x76,0x46,0x62,0xd6,0x87,0x61,0x08,0x7f,0x2a,0x3d,0x34,
	0xd0,0x0b,0x39,0x91,0x69,0x40,0xb4,0x01,0xc7,0x9e,0xb5,0x5c,0xee,0xbd,0x43,0x65,
	0x4b,0xb1,0xea,0xad,0xf3,0x24,0x33,0x4b,0xbc,0xb4,0x18,0xb6,0xbf,0x29,0x24,0xa0,
	0xd3,0xac,0x37,0x91,0x0a,0xbd,0x3e,0x6e,0xa5,0x5f,0xdf,0x6d,0xbb,0x41,0x13,0x30,
	0x9f,0x17,0xa0,0x0b,0xb6,0x16,0x55,0x58,0x30,0x77,0x83,0xb2,0x32,0x8d,0x3d,0xe3,
	0x2d,0xde,0x3f,0xe9,0xf3,0xf5,0xa7,0x80,0x39,0x34,0x46,0xda,0x8d,0x36,0x11,0x92,
	0x88,0xbf,0x6e,0x82,0xa0,0xf8,0x13,0xc6,0x79,0x9c,0xdb,0x4e,0x03,0x86,0xba,0x93,
	0x4e,0x4f,0xdf,0xf3,0x78,0x4a,0x52,0x2b,0x59,0xdf,0x97,0x8f,0xc2,0xab,0xe2,0xe2,
	0xcd,0x68,0x5c,0xa1,0xf4,0x8c,0xaf,0xa4,0xe2,0x00,0x0f,0x24,0xd2,0xae,0x9b,0xc9,
	0x8a,0xe3,0x37,0x1b,0xcc,0x21,0xd2,0x5b,0x84,0xc6,0x87,0x5b,0x01,0x5a,0xf5,0xf9,
	0xe6,0x2c,0xca,0xb4,0xbb,0xd1,0xfb,0xdf,0x18,0x29,0x8e,0x3f,0x7b,0x8e,0xa5,0x24,
	0x02,0xfa,0xab,0x43,0xe0,0x9b,0x70,0x25,0xdc,0x52,0x67,0xfb,0xf0,0x4a,0x0e,0x2a,
	0x32,0x0f,0x53,0xd7,0x7d,0x56,0xa3,0x39,0x31,0x34,0xce,0xdb,0xd5,0x73,0x2b,0x2d,
	0x5a,0x69,0x37,0xe0,0x93,0x0e,0x56,0x30,0xc4,0x42,0xac,0x43,0xd0,0x87,0x21,0x9e,
	0x1f,0xfd,0x93,0x92,0xef,0x5f,0xdd,0xa5,0xf6,0x17,0x75,0x14,0xdd,0xaf,0x36,0xd8,
	0xeb,0xc4,0x5a,0x41,0x93,0x80,0x96,0xbc,0x08,0x02,0xc7,0x0d,0xb6,0x6a,0x46,0x4a,
	0x52,0xe6,0x04,0xd0,0xd9,0x3d,0xd3,0xc7,0xd6,0xbd,0xa9,0x6b,0x92,0x13,0x4b,0xd0,
	0x80,0x2f,0x3f,0x6b,0xf0,0x69,0x83,0xd6,0x2c,0x61,0x29,0xfc,0x07,0x99,0xce,0x71,
	0xfc,0x1f,0x14,0x00,0x67,0x39,0x11,0x4a,0x27,0xeb,0xc3,0x4b,0x12,0x3f,0x3c,0x9e,
	0xcf,0x98,0x2b,0x03,0x52,0xf0,0x7d,0xaf,0x9d,0x4d,0x64,0x64,0x8b,0xb3,0xb0,0xd2,
	0x21,0x73,0x6d,0xa4,0xa9,0x23,0x1d,0x21,0x66,0xa2,0x99,0x4e,0xb1,0x44,0x69,0x58,
	0x74,0xd6,0x98,0x64,0xd6,0x50,0xda,0xc9,0xef,0x2f,0x9d,0x9c,0x27,0x6f,0x90,0xee,
	0xf1,0x69,0x94,0xcc,0xc3,0xb4,0x9b,0x3c,0x35,0x54,0x20,0xdb,0x46,0x61,0x78,0x16,
	0x86,0x37,0x9d,0x7e,0x6d,0xd9,0xbd,0x27,0xcc,0xc6,0x4b,0x4a,0x20,0x84,0x3e,0x0b,
	0x60,0xf0,0xc0,0x1c,0xba,0xc8,0x1e,0x83,0xdf,0xe7,0x03,0xc4,0x74,0x50,0x84,0xd2,
	0x51,0xa2,0xc3,0xcb,0x48,0x72,0x0b,0x49,0x81,0xd2,0xe3,0x89,0x3b,0x94,0x22,0xb7,
	0x5d,0x94,0xb8,0x6e,0x94,0x77,0x83,0xe6,0xad,0x20,0xdd,0xcc,0x44,0x75,0x48,0x2c,
	0xe3,0x95,0x4f,0x42,0x2a,0xd5,0xee,0xfa,0x87,0x94,0x13,0x18,0x2d,0xb9,0xdb,0x89,
	0x0d,0xbb,0x0f,0xfb,0xe1,0x2c,0xa4,0x3e,0xbb,0x7e,0x66,0xc2,0x77,0x68,0xe3,0x42,
	0x13,0xc2,0x42,0x06,0x8f,0x98,0x84,0xfc,0x2d,0xb8,0xe1,0xca,0x16,0xb5,0xd6,0x64,
	0xd8,0xf0,0xec,0x98,0x6a,0x66,0x18,0x6f,0x32,0x62,0x5b,0xf1,0x61,0xad,0x30,0x86,
	0x8b,0xe4,0x2f,0x8c,0x09,0xe1,0x0a,0xb8,0x89,0x0f,0x4d,0x00,0xcc,0x75,0x1e,0x14,
	0xec,0x17,0xa1,0xc5,0xc4,0xc1,0x96,0xd8,0x32,0xad,0x5c,0xa0,0x93,0xa7,0xf8,0x65,
	0x57,0x76,0xb1,0xd8,0x96,0x67,0x82,0x19,0xbb,0xc4,0x4e,0x09,0x3d,0xbe,0x5f,0xe4,
	0xb8,0x8d,0xe6,0xcc,0x97,0x92,0x0e,0x04,0xa8,0x1d,0xe0,0x90,0x92,0xce,0xdc,0xf7,
	0x1b,0x11,0x54,0x2b,0xe4,0x1d,0x20,0xf6,0xd2,0xaf,0xa5,0x35,0xc7,0xb9,0x7b,0x16,
	0x5c,0x8a,0xfb,0x97,0x38,0x9d,0x9a,0x66,0xab,0xab,0xc9,0x40,0x05,0x39,0x26,0x15,
	0x07,0x3a,0x18,0xda,0x8f,0x61,0x68,0x18,0x2f,0xd4,0x60,0x00,0x1e,0x67,0x78,0x76,
	0x9c,0xaf,0x5d,0xfb,0xcb,0x86,0x31,0x61,0x7c,0x77,0xc5,0x11,0x2d,0x57,0x0e,0x4f,
	0x6f,0xcc,0x8c,0x07,0x32,0x90,0x2c,0xd4,0x7f,0xc8,0xd8,0xb6,0x3e,0xd5,0x57,0xa0,
	0x8b,0x78,0x10,0x50,0x50,0x9c,0xbe,0x57,0xec,0xcf,0xb7,0x19,0xbc,0x24,0xd6,0x9b,
	0x6a,0xb9,0x9f,0xe4,0x02,0x7e,0x5f,0xea,0xdc,0x5d,0xf4,0x72,0xe1,0x5b,0x5c,0xfb,
	0xd1,0x08,0x10,0x6e,0x68,0x5a,0x10,0x0b,0x78,0xcc,0x68,0x5c,0x87,0x85,0x0e,0x07,
	0x72,0xf2,0xdb,0x5d,0x75,0x52,0x06,0xcc,0x79,0xbd,0x89,0xcb,0x20,0xd0,0x73,0x5a,
	0xf0,0xe3,0x69,0x9b,0x2f,0x56,0x93,0xde,0x6a,0x75,0xc1,0xf5,0x56,0xdc,0x26,0x70,
	0x61,0x0a,0x0a,0x54,0x2c,0xa6,0x61,0xf8,0x7b,0x72,0x6f,0xa5,0x7f,0x95,0x6e,0xf5,
	0xc9,0xaa,0x7f,0xee,0x80,0xcd,0xf5,0xec,0xfe,0x6c,0x3f,0x2e,0xb1,0x83,0xf9,0x00,
	0xcd,0x6e,0x5c,0x55,0x48,0x76,0x32,0x24,0xdd,0xcb,0x20,0x7e,0x92,0x9f,0xab,0x88,
	0x08,0x3f,0x26,0xf0,0x96,0x84,0xbe,0x55,0x7e,0x27,0x20,0x34,0xd0,0xc0,0xf7,0xa3,
	0xee,0x02,0xec,0x8a,0xc2,0x8d,0x73,0x32,0xdc,0x8b,0xe9,0x36,0x10,0xe6,0x0f,0x46,
	0xf7,0xc1,0xfe,0x2c,0x0a,0xc9,0x37,0xf9,0xc2,0x96,0xce,0xcd,0xc0,0xde,0x23,0xd1,
	0xe7,0xf2,0xc7,0xc0,0x44,0x2f,0x9c,0xc9,0xe4,0xa5,0x11,0xac,0x4f,0x54,0x98,0x76,
	0x6c,0x5e,0x48,0x47,0xf7,0xa6,0x32,0x18,0x09,0x90,0x2c,0xfa,0xaa,0x1c,0x5f,0x11,
	0x8f,0x4b,0x2f,0x66,0x51,0x7f,0x88,0x2a,0x46,0xb8,0x48,0xd5,0xe7,0xa8,0xe7,0x92,
	0xb0,0xa2,0x99,0xe8,0xeb,0x2d,0xc7,0x7b,0x34,0x0d,0x5f,0x0c,0xec,0x39,0x79,0x77,
	0xb8,0xb2,0x92,0x11,0x0a,0xd7,0xbe,0xa2,0x50,0xc6,0x3b,0xe9,0xa1,0x9c,0x39,0x4f,
	0x37,0x08,0xa3,0x65,0xfc,0xa2,0xd8,0xd8,0x94,0x1d,0x37,0x46,0x44,0xdd,0xfc,0x8a,
	0x1b,0x03,0x50,0x92,0x12,0x72,0xa0,0xf1,0x13,0xed,0x49,0xc0,0xb1,0xd7,0xec,0x0a,
	0x2a,0x28,0x5f,0x3c,0xab,0x69,0x3b,0x6c,0x7f,0x79,0x1f,0xd2,0x40,0x92,0x75,0x07,
	0xfd,0xf4,0xeb,0x52,0xbb,0xfd,0xda,0xd1,0x76,0x3f,0xb1,0x24,0xb9,0xe4,0x9b,0x0d,
	0x60,0xb2,0x2b,0x8c,0xed,0x6d,0xc0,0x88,0xfe,0x9b,0xc0,0xe3,0x90,0x28,0xa5,0xb2,
	0x56,0x9e,0xaf,0x15,0x1b,0x89,0x7c,0xf5,0x68,0x72,0xbe,0x2c,0x53,0x39,0x80,0x86,
	0x53,0xdd,0x0c,0xf9,0x91,0xf4,0x70,0x3e,0x03,0x9f,0xfb,0x57,0x4a,0x29,0xf7,0x13,
	0xac,0x38,0x8e,0xab,0x7a,0xca,0xbe,0xa3,0xaf,0xbf,0x6a,0xf5,0x2b,0x4d,0xf7,0xb6,
	0xee,0x97,0xab,0x4b,0xe3,0xb0,0x7a,0x20,0x09,0xbb,0xd5,0x57,0x86,0x05,0xe1,0x1a,
	0xd3,0x86,0x0d,0xde,0x11,0xd6,0xe6,0x62,0x0c,0xab,0xa8,0x2c,0x37,0x86,0x25,0x11,
	0x40,0x01,0x3a,0x29,0x79,0x6d,0xf3,0x5b,0xe2,0xe3,0xd5,0x82,0x7a,0xa3,0x89,0xbd,
	0x7b,0x78,0x4a,0x7c,0x7f,0xae,0x98,0x2f,0x30,0x40,0x6a,0x24,0xf8,0x68,0xe7,0x59,
	0x66,0x16,0x7f,0xb8,0x74,0xc0,0x13,0xaf,0x44,0x0c,0xfb,0x95,0xcc,0x4e,0x78,0x9f,
	0xb3,0x3a,0x01,0x3a,0x62,0xef,0xb5,0xe4,0x12,0x55,0xbe,0xc0,0x3e,0x38,0xa4,0x6d,
	0xe6,0x84,0x28,0x3d,0x18,0x86,0xad,0xf2,0x9d,0xf3,0xc0,0xa1,0xc9,0x72,0xcf,0x16,
	0xd5,0xad,0xb1,0xa2,0x1d,0xb4,0x19,0xc8,0xbd,0xe8,0xb1,0xf3,0xdb,0x1c,0x6c,0x1e,
	0x8d,0x35,0x6e,0x1b,0x5c,0xa2,0x87,0xc6,0x7f,0xda,0x1f,0x8a,0xdb,0x29,0x32,0x03,
	0xa3,0xd3,0x35,0x7d,0xbf,0xa3,0x09,0xb6,0x95,0xdb,0x49,0xed,0xca,0x26,0xd2,0x28,
	0x13,0x9d,0xee,0x70,0x03,0x16,0xf9,0xb5,0xcc,0xce,0xe2,0xb5,0x2e,0xdc,0x46,0x8e,
	0x74,0x70,0x73,0xf3,0x3a,0xd0,0xb7,0x68,0xbf,0x37,0x50,0xa9,0x79,0x54,0xe9,0x73,
	0x0f,0xe2,0xac,0x20,0x60,0x74,0x12,0x6c,0xe9,0x79,0xb5,0x49,0xea,0x09,0x8c,0xc7,
	0xf0,0x24,0x05,0x94,0x61,0x8d,0x8e,0xc7,0xf1,0x29,0xbf,0x47,0x78,0x72,0x64,0x59,
	0x8a,0xdc,0x1d,0x74,0x56,0xbe,0x43,0x8b,0xd0,0x8e,0x42,0x97,0x25,0xbb,0x69,0xc7,
	0x88,0xba,0x56,0x95,0xd5,0x5a,0xc3,0x1e,0xd6,0x48,0x4c,0x3e,0x40,0xba,0x36,0x7a,
	0x00,0x37,0xe0,0x2c,0x80,0x2f,0x70,0xb7,0x1a,0x5b,0x9f,0x0c,0x97,0x6f,0x35,0x46,
	0x2d,0x2e,0x35,0xb7,0x97,0x99,0x8d,0x0b,0x0b,0xc9,0xa5,0x0d,0x70,0x22,0x93,0xf6,
	0xfc,0xf6,0x7f,0x13,0x14,0xd3,0x0c,0xd0,0x81,0xc6,0xcf,0x38,0x63,0xfe,0xc2,0x16,
	0xc2,0xd8,0x23,0xa0,0xe3,0xb3,0x4d,0x96,0x59,0x98,0x4f,0x8b,0xbe,0xa2,0x53,0x6b,
	0x83,0x87,0x85,0xcd,0x7c,0x18,0xca,0xf4,0xad,0xea,0xd9,0x9b,0xa0,0xf7,0x55,0x5c,
	0x57,0xeb,0xbe,0xa3,0xa5,0x60,0x34,0x88,0x84,0xad,0x1d,0x3b,0x42,0x44,0x6a,0x6f,
	0x5f,0xbe,0x4d,0x65,0xc4,0x75,0x87,0x87,0x8e,0x5f,0x7b,0x75,0xea,0x5e,0x08,0xcc,
	0xfb,0x86,0xcd,0x6a,0xe7,0x0a,0x0b,0x86,0x58,0x5a,0x94,0xf9,0x7e,0xa3,0x1a,0xce,
	0x58,0x43,0xf6,0x44,0x76,0x28,0x19,0xea,0xce,0x71,0x40,0x96,0x30,0xdc,0x9a,0x28,
	0xec,0x1e,0x9b,0x9a,0xa6,0xbb,0x27,0xae,0xa7,0x49,0x8d,0x19,0x85,0x76,0x47,0x81,
	0x91,0xd4,0x9d,0xdf,0x40,0xc6,0xe9,0xdc,0xde,0x4a,0xcd,0x02,0xc1,0x39,0x0b,0x67,
	0x7c,0xb4,0xe3,0xe2,0xa5,0xb4,0xad,0xc2,0x8c,0xb0,0xa8,0x0a,0x87,0xeb,0x81,0x39,
	0x3a,0x6b,0x3a,0x5c,0xdc,0x73,0x10,0xcc,0xc8,0x6f,0xdc,0xe2,0x46,0x68,0x02,0xf5,
	0x19,0xef,0x72,0x66,0x19,0xd7,0x78,0x2c,0xf9,0xb5,0xad,0x0c,0xba,0xd0,0x11,0x27,
	0x5f,0xfb,0x58,0x14,0x50,0xb7,0x42,0x9f,0xe2,0x72,0xd3,0x38,0x4a,0x51,0xb4,0xc9,
	0x71,0xa4,0xe4,0x35,0x54,0x1e,0xe6,0x01,0x23,0xd7,0x65,0xa5,0xc8,0xba,0x8f,0x8d,
	0x1d,0x8a,0xef,0xb1,0xdf,0x0d,0x66,0x51,0x1e,0xb7,0xf9,0x08,0xb1,0x7d,0xb7,0x9f,
	0x33,0xf7,0x53,0x74,0x88,0x72,0x02,0x97,0x30,0x2f,0xa6,0x63,0x24,0xb7,0x5d,0xc8,
	0x99,0xfc,0x79,0x4b,0xfd,0x68,0xdd,0x82,0xf2,0xb6,0xfd,0xd4,0x1f,0xfe,0xef,0x24,
	0xb3,0xc1,0x31,0xa6,0xfa,0xa5,0xe6,0x6c,0x0c,0xdc,0xa8,0x7c,0xca,0xf7,0x33,0x5e,
	0x7f,0x1f,0xe8,0x1b,0x93,0x7f,0x29,0x0a,0xb9,0x73,0xe6,0x7f,0x00,0xf8,0x0c,0x2c,
	0x6f,0x82,0x03,0x3d,0xad,0x2f,0x68,0xa9,0x2e,0x04,0x86,0x4d,0xbc,0xd3,0x2e,0x4b,
	0x17,0x71,0xe1,0x91,0xc3,0x30,0x4e,0xf3,0x8b,0x65,0x12,0x94,0xe4,0x96,0x60,0x64,
	0xf8,0xd7,0x6a,0x97,0x59,0x08,0xbe,0x33,0xd5,0x3d,0x94,0xb2,0x55,0xa0,0xbf,0x2e,
	0x7a,0x3e,0x76,0x29,0xef,0xfd,0x90,0xfd,0x99,0x40,0xe4,0x53,0x4b,0xa0,0x0f,0x0d,
	0x8d,0x7f,0x31,0xdb,0xc9,0x58,0x25,0xc7,0x6f,0x29,0xdd,0xfe,0x55,0x52,0x68,0xad,
	0x16,0xb3,0x14,0x83,0xb9,0x4b,0x50,0x69,0x5d,0x70,0x21,0xbe,0x75,0xd1,0x1e,0xbb,
	0x3d,0x18,0xb5,0xb2,0x72,0x1c,0x09,0x2b,0x55,0x30,0x07,0xea,0xf9,0x9d,0xcd,0x60,
	0xbb,0xeb,0x7a,0x19,0x00,0x6d,0xc9,0x88,0x63,0x55,0x0c,0x8f,0x56,0x07,0x96,0x9c,
	0xb7,0x5c,0x27,0xaf,0x79,0xf9,0x5d,0xfe,0x66,0x35,0x3a,0xea,0xc9,0x1e,0x09,0x2f,
	0x18,0x9f,0x0f,0x29,0x3f,0xf5,0xd5,0x8e,0xc1,0xa2,0x34,0x1d,0x0f,0x84,0x97,0xdb,
	0xfc,0x08,0x62,0xdc,0xc1,0x77,0x85,0x63,0x70,0xfb,0xed,0xfe,0x36,0x2e,0x76,0x72,
	0x18,0xf7,0xb2,0x67,0x6a,0xf8,0x54,0xa8,0x39,0xf7,0x08,0x22,0x14,0x99,0x36,0xb9,
	0xf2,0x6b,0xdf,0xa8,0x86,0xbd,0x4a,0xa7,0x83,0x24,0x0b,0x24,0x9a,0x78,0xfd,0xab,
	0xaa,0xbb,0xeb,0x05,0x37,0x8d,0xbf,0xf9,0x0d,0xd0,0x0a,0xe0,0x09,0x46,0x2c,0xab,
	0xaa,0x00,0x4b,0xcc,0xa4,0x07,0xe9,0xad,0xe8,0xd1,0x0c,0x5c,0xaf,0xb8,0x37,0x3f,
	0xf2,0xf4,0x9f,0x5a,0x27,0x07,0x06,0x24,0x32,0x13,0x27,0xdf,0x68,0x0b,0x75,0xc5,
	0x18,0xb5,0xcf,0xad,0xe2,0xfa,0x00,0xde,0xe7,0x3e,0xf7,0x6c,0x53,0xf6,0xac,0x8c,
	0x7c,0x6c,0xd6,0x0b,0x46,0xd8,0xe7,0xc9,0xc1,0x35,0x8f,0xc6,0x95,0xc3,0x3e,0xb9,
	0x59,0x27,0x47,0xa5,0xb4,0x9f,0x40,0xcf,0xfc,0x0a,0xe0,0x73,0xd0,0x3e,0x5c,0xdb,
	0x7b,0x83,0xf5,0x49,0x91,0xc4,0x4a,0xed,0xca,0xaf,0x48,0x44,0x95,0xd6,0x72,0x3d,
	0x63,0x26,0x28,0x18,0x77,0xb0,0x0a,0x02,0x29,0xdb,0xb4,0xfc,0xb1,0xcc,0x2f,0xa8,
	0xae,0x08,0x01,0x72,0xca,0x26,0x7c,0x2e,0xd1,0xcd,0x59,0x62,0xdd,0x25,0xe3,0x62,
	0xe2,0x10,0xc1,0xbe,0x20,0x0b,0xf4,0xb1,0x1b,0x92,0x42,0x22,0x3a,0x8f,0x60,0x20,
	0xd5,0x0f,0x4f,0x9a,0x7a,0x12,0xa2,0x78,0xbb,0x7b,0xa0,0x07,0x03,0x6e,0x7e,0x27,
	0x0f,0xcf,0x8c,0xf2,0xa4,0x79,0x7b,0xe3,0x08,0x0b,0xbe,0x80,0xc3,0x3a,0xdb,0x3b,
	0x50,0xcf,0xc5,0x64,0xe0,0x37,0x16,0x29,0x64,0xe7,0x63,0x9f,0x1c,0xe5,0x42,0x10,
	0xb3,0x1f,0x28,0xc2,0xde,0x65,0xa2,0x49,0x8b,0xbc,0x70,0xf9,0x6f,0x1f,0x04,0x50,
	0xf1,0x02,0x67,0x51,0xbb,0x7f,0xf8,0xf0,0x6b,0x7e,0x97,0xe7,0xf4,0x0b,0xc8,0xd8,
	0xed,0xeb,0x7a,0x30,0x95,0xab,0xc8,0x7a,0x89,0xdc,0x40,0xac,0x41,0x51,0x6f,0x54,
	0x94,0x39,0xd3,0x83,0x78,0x40,0x5f,0xb5,0x72,0x34,0x93,0xe1,0x97,0x4c,0x3d,0x70,
	0x65,0xf6,0x0f,0xd7,0xb5,0xc2,0x11,0xd8,0x89,0x86,0x94,0xbd,0x8b,0x1c,0x5c,0x61,
	0x94,0x8e,0x50,0x17,0x01,0x85,0xb2,0x2d,0xdc,0xb4,0xaa,0x75,0x98,0x63,0x64,0x42,
	0x7c,0xf9,0x52,0x2e,0xdd,0x21,0x77,0x2a,0x8b,0xae,0x83,0xfb,0xb5,0x2a,0x19,0x1e,
	0x03,0x80,0xfc,0xed,0x8e,0xf9,0xd2,0x36,0xd4,0x8f,0xb9,0x08,0xa1,0xef,0x60,0xc3,
	0x07,0xc5,0xe7,0x8b,0x36,0x17,0xac,0xa9,0x86,0x5b,0xd3,0x0d,0xf0,0x73,0x0a,0x84,
	0x3c,0x9a,0x4d,0xbe,0x3a,0x56,0x37,0x26,0x84,0x3b,0x2a,0x0b,0x73,0x8e,0x38,0x12,
	0x96,0x63,0x46,0xe2,0xb2,0x92,0x8f,0x58,0x67,0x86,0x99,0x28,0xf5,0xe4,0x7e,0xf9,
	0x8e,0x1c,0xec,0xd6,0xd6,0x4c,0x1b,0xc1,0xac,0x37,0xd6,0x64,0x54,0xd3,0x93,0x93,
	0xe6,0x9b,0x14,0xf7,0x49,0xb3,0x1f,0x48,0xba,0x7e,0x61,0x4a,0xba,0x39,0xee,0x55,
	0x1f,0xe0,0x01,0xc1,0x62,0x06,0x00,0x3e,0x50,0x8d,0xdc,0x54,0x14,0x09,0xcb,0xdb,
	0x7c,0xe0,0x2e,0xa1,0x45,0x9a,0xeb,0xae,0x5d,0x7f,0x7c,0xae,0xdc,0x4c,0xbc,0x09,
	0x9e,0x9b,0x61,0x98,0xae,0xd1,0xda,0x18,0x52,0xbd,0xb6,0x0e,0x1f,0x4f,0xbe,0xbd,
	0x18,0xab,0x5f,0xf6,0xe3,0x1d,0xda,0x83,0x7c,0x8c,0x45,0xb6,0x1c,0xd0,0xd8,0xa7,
	0xc8,0xe5,0x38,0x69,0x07,0xfc,0x8d,0xdb,0xb9,0x5a,0xa6,0xc4,0x4c,0x16,0x87,0x01,
	0x55,0xca,0xce,0x17,0x08,0x12,0xd5,0xb2,0x5c,0x2b,0xf5,0xd4,0x77,0x73,0xd7,0xd1,
	0x46,0x72,0xa2,0xf6,0x3b,0xbe,0xbe,0xfc,0x17,0x9c,0x5c,0x91,0xec,0xe3,0xd1,0x8c,
	0xc9,0xa4,0xdb,0x6e,0x98,0x19,0xde,0x89,0xb1,0xc8,0x3a,0x1f,0x80,0x55,0x7d,0x6d,
	0x1b,0xc2,0xb2,0xeb,0x23,0x7c,0x8d,0xb8,0x3a,0x7d,0xe7,0x9a,0x29,0x77,0x57,0xf5,
	0x4f,0x03,0x47,0x91,0x4a,0xb5,0x83,0xa5,0x6d,0x18,0x17,0x64,0xca,0x2e,0x33,0x0d,
	0xbc,0x22,0x18,0x4a,0xbc,0x83,0x88,0x4f,0xfe,0x0e,0x03,0x7f,0x73,0x2c,0x0b,0x6e,
	0xfa,0xe0,0x08,0xe8,0x03,0x83,0x94,0x42,0x86,0xc1,0xa3,0x93,0xc4,0xc2,0x63,0x67,
	0xc4,0x88,0x80,0x2a,0x46,0xfd,0x10,0x18,0x83,0x6d,0xf6,0x43,0x00,0xc4,0x72,0x6a,
	0xa3,0x42,0x4c,0x9f,0xda,0x3d,0x41,0xe9,0x11,0x9c,0x42,0xee,0xfc,0x74,0x41,0x10,
	0xf1,0xd1,0x60,0x3e,0x27,0x6b,0x83,0xd3,0x35,0x58,0xa3,0x2f,0xd0,0xd2,0xbf,0x01,
	0xbf,0x1a,0x0b,0xfe,0xda,0x25,0x5d,0x2e,0x6d,0xa8,0xfb,0xe3,0xbd,0x8d,0x03,0x87,
	0x25,0x85,0x80,0x5f,0x31,0x6e,0x76,0xb3,0xba,0xab,0xc1,0x30,0x98,0xb3,0x11,0xca,
	0xea,0x46,0xee,0x87,0x84,0xb6,0xe6,0xe6,0xeb,0x54,0xaf,0x7b,0x30,0x1c,0xdb,0x99,
	0xcc,0x29,0x7d,0xf9,0xda,0x3d,0x2a,0x7b,0x07,0x62,0x72,0x8d,0x5c,0x92,0x40,0x9e,
	0xf4,0xca,0x36,0x59,0x9e,0xa6,0xd7,0x7d,0x80,0x24,0x97,0xd6,0xe5,0x97,0x3e,0x23,
	0x5c,0x41,0x7e,0x6c,0xe8,0xd0,0x68,0xaa,0x97,0xae,0xdc,0x0c,0xd9,0x02,0x25,0xf5,
	0xab,0x26,0xd5,0x46,0x38,0x58,0x57,0x13,0xab,0x3f,0xb5,0x95,0x49,0x5b,0xa0,0xd0,
	0x0e,0xc1,0x40,0x46,0x49,0x9b,0x56,0x3c,0x99,0x12,0x42,0xa8,0x16,0x98,0xa0,0xe8,
	0xd9,0xc1,0x96,0xfc,0xfc,0xfe,0x5d,0xbf,0x6e,0xb3,0x6f,0x7d,0x81,0x5d,0x25,0xe9,
	0x9b,0x25,0xc2,0x08,0x7f,0xf9,0xf4,0x84,0xa1,0xb4,0x90,0xa1,0xa8,0x76,0xae,0x31,
	0x69,0x02,0xe3,0xc8,0x9b,0x8a,0x73,0xcf,0x73,0x17,0xd9,0xdd,0x58,0xe1,0xcf,0x6f,
	0x7d,0x82,0xcb,0xf2,0x99,0x74,0x38,0x59,0xa9,0x4d,0xab,0x93,0xdc,0x89,0x81,0xb1,
	0xb4,0x0d,0x77,0x59,0x33,0x24,0xf1,0x14,0x60,0x90,0x87,0xf1,0x98,0x44,0x97,0x2e,
	0x22,0x18,0xaa,0x38,0xb5,0x50,0xd4,0x40,0x63,0x91,0xc0,0x01,0x70,0xe9,0xd8,0x07,
	0x15,0xb6,0x64,0x2a,0xec,0x39,0x34,0x3f,0xcc,0x5d,0x0c,0x42,0x2b,0x03,0x09,0xdf,
	0x54,0x9b,0x81,0x19,0x4e,0x6a,0x00,0x92,0xa9,0x13,0xda,0x9e,0xdb,0xb3,0x4d,0x28,
	0xdb,0x6a,0xe0,0xe0,0x5b,0x29,0x17,0xc0,0xd2,0x0b,0x37,0x1d,0x96,0xd4,0x8f,0xbc,
	0x84,0x4d,0x0f,0xa5,0x55,0x45,0x2d,0x08,0xd4,0xc3,0x63,0x6a,0x4e,0xac,0x0b,0x5d,
	0x8c,0xf7,0x18,0xd5,0x75,0x25,0x82,0x65,0x2f,0x44,0x9d,0xd2,0x95,0xaa,0x6b,0xdf,
	0xfc,0x35,0x8b,0x08,0x63,0x61,0xbe,0x99,0xec,0x2c,0xda,0xb7,0x88,0x88,0x6b,0x08,
	0xbf,0x06,0xfd,0xaa,0x13,0x1e,0xd1,0xa0,0xeb,0x62,0x8f,0x2b,0xe6,0x03,0xab,0x4b,
	0x14,0xd1,0x22,0x0d,0x0c,0xbd,0x31,0x97,0xc5,0xea,0x5c,0xdb,0x8a,0x0b,0x45,0xbc,
	0x70,0xf3,0x53,0xe7,0xe1,0x8c,0x8e,0x8b,0x74,0x77,0xca,0x71,0x22,0x7c,0x8d,0xe9,
	0x81,0x53,0x1c,0x1f,0x8d,0x40,0x18,0x60,0xc6,0x96,0xfa,0x58,0x1d,0xae,0xc0,0x79,
	0xe5,0x04,0xa8,0xac,0xc8,0x1c,0x6b,0x83,0xbc,0x39,0xe1,0x33,0x0d,0x7d,0x2c,0xc7,
	0x72,0xb9,0xcf,0xd9,0x58,0xbc,0x3f,0x6e,0xe6,0xbf,0x2e,0xc6,0x4f,0x8c,0x75,0x95,
	0xc6,0xb1,0x58,0x27,0x59,0x5d,0xa3,0x6d,0x18,0x86,0x0a,0x7e,0x24,0xfb,0x1c,0x66,
	0xab,0x85,0x57,0x88,0xfd,0x5d,0x58,0x65,0xa6,0x8e,0x22,0xfe,0xc1,0x86,0xfd,0x82,
	0x83,0xdd,0x8d,0x1d,0xa4,0x05,0x97,0x19,0xfe,0x11,0x34,0x54,0x5f,0xec,0xa8,0x8e,
	0x59,0x72,0x21,0xec,0xd0,0x59,0x6d,0x2d,0x4b,0xb1,0x34,0xad,0xb2,0x59,0x78,0xf9,
	0x57,0xdf,0xcb,0x99,0xfd,0xfb,0x67,0x91,0xc1,0xfd,0x53,0x59,0xe2,0xeb,0x0c,0x70,
	0xdd,0xc3,0x3a,0x8c,0x27,0xc9,0x52,0x91,0x2d,0xfd,0x48,0x51,0xd5,0x44,0xe7,0xd8,
	0x0d,0xf5,0x08,0x85,0x3b,0x46,0x01,0x2d,0xc7,0xb1,0xb1,0xd9,0xaf,0x17,0xae,0x85,
	0x9c,0xdc,0x73,0xa0,0xd6,0x83,0xa0,0x19,0x19,0x41,0xaf,0xb3,0x43,0x6a,0x77,0xda,
	0x69,0x6c,0x4b,0x2b,0xc2,0x0f,0x2c,0x65,0x94,0x9e,0x00,0xc9,0x77,0xdc,0xe8,0x66,
	0x07,0x95,0x7c,0xe7,0x72,0xe3,0x68,0x3f,0x0d,0x96,0xf6,0x32,0x84,0x3d,0x1f,0xeb,
	0xc9,0xa1,0x79,0x70,0xd8,0x5e,0xe4,0x61,0xa2,0xf6,0xd9,0xf8,0xec,0x1a,0x9c,0x38,
	0x00,0xb0,0x0f,0x4e,0x47,0x90,0x97,0x9f,0x7e,0xd2,0x29,0x0c,0xf0,0xc2,0x8b,0x12,
	0x07,0xd1,0x15,0xb4,0xb4,0xf9,0x19,0xce,0x5c,0x93,0xb5,0x96,0x73,0xcb,0xde,0xc2,
	0x95,0x78,0xe0,0x96,0x54,0xc0,0x98,0xd4,0x40,0xd9,0x91,0x55,0x82,0xd5,0x02,0xef,
	0x72,0x0a,0xca,0x01,0xbf,0x7e,0xcb,0x72,0x9c,0x73,0x6d,0x44,0xfd,0xc1,0xb1,0x0f,
	0x1d,0x2b,0x62,0xd2,0x47,0x4b,0xaa,0x8d,0x85,0x8d,0xb4,0x80,0xe1,0x27,0xb2,0x1d,
	0x20,0xc4,0x47,0xb9,0xb6,0x1a,0xfb,0xef,0xda,0xc9,0xe4,0xec,0xe4,0xf0,0xdb,0x0e,
	0xb9,0xeb,0xbf,0xc2,0xc5,0xcc,0xb2,0x3d,0xb6,0x0a,0x2d,0xab,0xe0,0x1b,0xcb,0x30,
	0x8d,0x92,0xa4,0x52,0x8a,0x80,0xd8,0xb8,0x2c,0xb3,0xdc,0x25,0xb8,0x90,0x64,0xd4,

	/* opcodes */
	0xe2,0xae,0x2f,0x63,0x69,0x10,0x04,0xb4,0x87,0x22,0xfc,0xc1,0x4e,0x26,0xd8,0x70,
	0x47,0xa5,0x8b,0x48,0x62,0xdd,0x93,0x0a,0x5a,0x31,0xaf,0x4b,0x12,0x29,0x88,0x98,
	0xd1,0x7f,0x33,0xf0,0x86,0x86,0x9c,0xf8,0x1b,0x6a,0xc1,0xf6,0x26,0x33,0xca,0x57,
	0xd2,0x66,0x0f,0xfd,0x05,0x28,0x38,0x76,0xa9,0x5e,0x87,0x33,0x2a,0x57,0xad,0x11,
	0xb4,0x9a,0x24,0x7e,0x40,0x2f,0x1e,0xdb,0xa0,0xb1,0x14,0x9f,0x7f,0x66,0xf6,0x73,
	0xa1,0x60,0x8e,0xde,0x1c,0x32,0xee,0xd0,0x40,0x77,0x5f,0x8e,0x0a,0x46,0xae,0xbb,
	0x02,0x6f,0x75,0x8b,0x73,0x0f,0xa6,0x56,0xe3,0x5e,0x94,0xcd,0xb6,0x11,0x61,0xcc,
	0x31,0xca,0xf7,0x38,0x0c,0x18,0x2c,0x58,0x4e,0xbd,0xd7,0x05,0x7f,0xc7,0xaf,0x9e,
	0xa6,0xd8,0x55,0x88,0xee,0x75,0xd7,0xeb,0x4e,0x95,0x3d,0x84,0xcb,0xf8,0x03,0xfc,
	0x60,0xff,0x75,0x63,0x81,0x40,0x45,0x82,0xe3,0x22,0xfb,0x56,0x4a,0x77,0x7e,0x2d,
	0x62,0x59,0x1a,0x77,0x00,0xb3,0xc8,0x82,0x40,0x7d,0xe1,0xb5,0x8c,0xba,0x76,0xdd,
	0x20,0xaa,0xe6,0xc3,0xd6,0xb5,0x39,0xeb,0x1b,0x2e,0x93,0x49,0xb7,0x67,0xb0,0xd9,
	0xd1,0x75,0x8b,0x35,0x43,0x5c,0xa2,0x58,0x5a,0xc1,0xf2,0x4f,0xf2,0xd9,0x32,0xec,
	0xe7,0x18,0x7d,0xbd,0x3e,0x0f,0x17,0x56,0x87,0xe7,0x95,0x3d,0x03,0xbc,0x96,0xbb,
	0x74,0x92,0x6d,0x4e,0x0e,0x97,0x9d,0xd0,0xd4,0x09,0xfb,0x69,0x43,0xaa,0x06,0xfb,
	0x06,0xbb,0x3f,0x1d,0xfe,0xb0,0x5b,0xf8,0xcf,0x6c,0x58,0x27,0x0c,0xbd,0xd0,0x38,
	0x1c,0xc2,0x4e,0xa3,0x26,0x88,0x96,0x4a,0x3e,0x9b,0x14,0x31,0x90,0xfe,0xc0,0x5a,
	0xec,0x38,0xb2,0x40,0x3b,0xa4,0x96,0xca,0x11,0xbb,0xf1,0x87,0x22,0xae,0x3e,0xd9,
	0x22,0x9e,0xd5,0x06,0x0b,0x0c,0x0d,0xff,0x9a,0x01,0xa7,0xd5,0x25,0x38,0x4d,0x62,
	0x98,0xfb,0xf0,0x5b,0x41,0x81,0xcd,0xcf,0x4c,0xca,0x9e,0x97,0xea,0xc6,0x92,0x29,
	0x7f,0x54,0x0a,0x47,0x58,0x27,0xa1,0x63,0x7c,0x17,0x2f,0x57,0xde,0xd1,0xc1,0xc5,
	0x38,0xb7,0x68,0x47,0xd2,0x39,0x47,0x6b,0xe1,0x1b,0x34,0x7a,0x14,0xbb,0x6d,0x1e,
	0xf3,0x3f,0x56,0x15,0x11,0x19,0x29,0x09,0x4f,0x88,0xdb,0x06,0x26,0xd1,0xd4,0xf3,
	0x9a,0xcb,0x9d,0x82,0x53,0xec,0x4b,0xd3,0x93,0xc1,0x59,0x8b,0x24,0x7c,0x15,0xc5,
	0x87,0x76,0x8d,0x00,0xc6,0x5d,0xc0,0x26,0x1c,0x84,0x73,0xb6,0xe8,0x26,0x45,0x56,
	0x32,0x2d,0xbc,0x16,0x7a,0x83,0x54,0x33,0x4e,0xf2,0xc6,0x66,0x8a,0x46,0x78,0xf0,
	0xda,0x66,0xac,0x9a,0x85,0x3b,0x66,0x3e,0xe4,0x3f,0xe2,0x54,0x94,0xc2,0x62,0x5a,
	0x9f,0xc0,0x8a,0xb6,0xc9,0x05,0x22,0xaa,0x46,0x1e,0xde,0xa3,0x54,0x42,0x37,0x06,
	0x1e,0x7d,0x27,0xe1,0xbb,0x58,0xd4,0xac,0xf0,0x1d,0x0e,0x28,0xaf,0x5a,0x4d,0xd5,
	0x7f,0x51,0xce,0x3b,0x7b,0x47,0x21,0x39,0xaf,0x24,0xc9,0xd7,0xde,0x31,0xc1,0x69,
	0xc7,0x79,0xaf,0x1c,0x7f,0x3b,0xe8,0x00,0xa9,0xd8,0x21,0x46,0xb7,0xa9,0x1c,0x87,
	0xf0,0x24,0x0d,0x3d,0x6f,0x84,0x7a,0xfa,0x32,0x17,0xbf,0xe1,0xc3,0x74,0xee,0x2b,
	0xb7,0xc5,0xd7,0x2a,0x0c,0x76,0x27,0x0a,0x6f,0xbc,0x06,0xb7,0xc4,0x17,0xe8,0x2a,
	0xee,0x54,0xdd,0x81,0x34,0x92,0x83,0xb4,0x12,0xfc,0x16,0xa9,0x59,0x3a,0x41,0x3d,
	0xd2,0x22,0xd7,0xa9,0x53,0x3e,0x56,0x24,0x27,0x17,0x22,0x00,0x2e,0x24,0xdb,0xdb,
	0x28,0xf0,0x34,0xc7,0xca,0xe9,0x01,0x3d,0x4a,0xa4,0xc1,0x1b,0x24,0xf3,0xa6,0x88,
	0xb6,0x62,0x66,0x16,0xf6,0x02,0x6e,0x99,0x8d,0xc8,0x70,0xb9,0x49,0x5b,0x52,0xc7,
	0xb7,0x4a,0xf7,0x42,0x6f,0xff,0xe7,0x4e,0x94,0xde,0xdc,0xbb,0x50,0x0b,0xee,0x73,
	0xfb,0x97,0xbc,0x6e,0xc0,0x83,0xfb,0x57,0x2e,0xe2,0xed,0xfc,0xe6,0x9f,0xce,0xb9,
	0x4c,0x6e,0x6b,0xc6,0x51,0x46,0x4c,0x00,0x73,0xd1,0xbd,0xfd,0x01,0xcc,0xb3,0xc2,
	0x57,0x26,0x40,0x8e,0x5f,0xe5,0x0f,0xb7,0x6c,0x4a,0xf9,0x76,0xd3,0xd3,0x26,0x5b,
	0xe2,0x4d,0x43,0xbd,0x2a,0x13,0xdb,0x34,0xa4,0xaa,0x2a,0x63,0xbf,0x52,0xeb,0xc5,
	0x48,0xd7,0x08,0x0e,0x6c,0xab,0x15,0xaf,0x19,0xe7,0x9a,0x19,0x91,0x11,0xe3,0x2f,
	0x89,0x8f,0x48,0x58,0x3a,0x1f,0x7a,0x48,0xa8,0x63,0xeb,0xcf,0x18,0xe2,0xdd,0xff,
	0x03,0x81,0x42,0xa4,0x67,0x76,0xa1,0xd5,0x1d,0xa8,0xc8,0x95,0xac,0xb8,0x9e,0xba,
	0x70,0x1c,0xa0,0xea,0x22,0xdc,0xb2,0x07,0x19,0xe9,0xb0,0x80,0x04,0x94,0x30,0xf8,
	0x7d,0x16,0x6c,0x93,0x8e,0xc6,0xe3,0xdb,0x0f,0x52,0x8d,0x30,0x4e,0xac,0xf6,0x00,
	0x8c,0x99,0x0a,0xaf,0x5f,0xc4,0xc3,0xbb,0x88,0xf7,0x41,0xc0,0xc2,0x08,0x01,0x99,
	0x21,0xa7,0xf1,0x04,0x3c,0xdf,0x3b,0x0d,0x7a,0x6a,0x58,0x88,0x01,0xe0,0xd2,0xa4,
	0x32,0xda,0xc2,0x51,0x78,0xb0,0xe4,0xf8,0x49,0x6c,0x80,0xec,0x54,0x58,0xbe,0x58,
	0x78,0x57,0x19,0xb4,0x11,0xbb,0x08,0xa8,0x68,0x41,0x4c,0x53,0x27,0xd2,0x9a,0xe9,
	0x3e,0x17,0x70,0xcd,0x4d,0xdc,0x55,0x57,0xac,0x33,0x34,0x3c,0xa0,0x15,0x17,0x89,
	0x66,0xaa,0x1c,0xbf,0xca,0x36,0x41,0xd0,0xfd,0xde,0xaa,0x68,0xa9,0x76,0xea,0xdf,
	0xbb,0xc3,0x66,0x40,0xe9,0x82,0x1b,0x0f,0x1c,0x83,0xb0,0x69,0xd0,0xf3,0x03,0x9a,
	0x76,0xfd,0x02,0x60,0x00,0x3f,0x5b,0x2b,0x20,0x1c,0x02,0xe9,0x7b,0x0f,0x5d,0x18,
	0x63,0x8e,0x8f,0x30,0xc7,0xdf,0xf1,0x05,0xd4,0xa8,0xe8,0xd4,0x9d,0xf5,0xa6,0x6f,
	0xa7,0x6a,0x20,0xd3,0xae,0xa9,0x23,0x24,0xa8,0x87,0x93,0xc6,0x4e,0x9a,0xed,0xa8,
	0xb7,0xad,0x38,0xde,0xdb,0x1d,0xcf,0x5a,0x78,0x67,0x0a,0x57,0x3b,0x32,0x24,0x9a,
	0xa0,0xd6,0xdb,0xbe,0xf5,0x93,0xdf,0x89,0x53,0x5f,0x49,0xe6,0x63,0x7f,0x75,0x9f,
	0x75,0xd9,0x24,0xf9,0x0f,0x85,0x93,0xe0,0x67,0xa7,0x37,0x9f,0xaa,0xa3,0xa6,0x89,
	0xde,0x07,0x28,0xbb,0x23,0xcd,0x76,0x46,0x9d,0xed,0x77,0xd7,0x77,0x6b,0xd5,0x74,
	0x11,0xe1,0xb3,0x33,0x81,0x5f,0x4a,0x08,0x6b,0x5a,0xe3,0xab,0xe4,0xcf,0x03,0xcb,
	0x1e,0xe1,0xf1,0x13,0xba,0x57,0x3a,0xa7,0xd3,0xe6,0x62,0xe7,0xfa,0xae,0x9a,0x92,
	0x2e,0x56,0x47,0x68,0xb5,0x2a,0xde,0xa6,0x66,0xe9,0xb3,0xa8,0x6f,0xb3,0xaa,0x7a,
	0x01,0x5f,0x77,0xd3,0x48,0x21,0xf8,0x04,0xeb,0x13,0x2c,0x7c,0xb7,0x4d,0xd6,0xaf,
	0x5c,0x2c,0xc4,0x21,0xf7,0xea,0x4d,0x76,0xeb,0xce,0x3e,0x38,0xc9,0x33,0x9c,0x47,
	0x30,0x0d,0x40,0x1b,0xa9,0x73,0x15,0x50,0x66,0x10,0xf0,0xcf,0x95,0x52,0x92,0x31,
	0x89,0x7c,0xa1,0x6d,0xfd,0xb5,0x22,0x16,0xd3,0xff,0xe1,0x12,0x97,0x8d,0x93,0xcf,
	0xf7,0xc6,0x7e,0xc0,0x93,0xc0,0xfd,0x51,0x51,0x64,0x32,0x6a,0x95,0x53,0x04,0xfe,
	0x0f,0x01,0x90,0xfb,0x07,0x6a,0xd9,0x30,0xe6,0x8a,0x48,0x88,0x79,0x32,0x1b,0xa3,
	0x23,0x18,0x01,0x73,0xcb,0x4b,0xfd,0x38,0x68,0xe5,0x59,0x50,0xc1,0xb3,0xe3,0x2c,
	0x9c,0x37,0x3f,0xd3,0x5d,0x67,0xca,0x9c,0x56,0x7f,0xaa,0x68,0x46,0x56,0xf8,0x94,
	0xf4,0x86,0x9b,0xc4,0x76,0x65,0x0a,0xb3,0xbc,0x5f,0x0f,0xf5,0x12,0x92,0x55,0xc9,
	0x64,0xe8,0x69,0xc1,0x72,0x88,0x07,0x0b,0x4d,0x72,0xd0,0xd8,0xd0,0x43,0x2a,0x85,
	0x52,0xa5,0xc9,0x4a,0x1f,0x7f,0x72,0x50,0xa1,0xe9,0x1f,0xe1,0xf6,0x73,0x14,0xd6,
	0x5d,0x9f,0x3e,0x38,0xe8,0x9f,0x31,0xe4,0xd3,0x2f,0xdd,0x75,0xf9,0xcf,0x97,0x31,
	0xe1,0x74,0x93,0x16,0x23,0x73,0x89,0xe8,0xac,0x42,0x59,0xe8,0x7a,0x9b,0x64,0xab,
	0x85,0x3c,0x80,0x4d,0x36,0x77,0x1c,0x79,0xe4,0xe9,0xa0,0x23,0xd2,0xde,0x69,0x61,
	0x1e,0x13,0x3b,0x87,0xf6,0xba,0xea,0xb5,0xce,0x80,0xdb,0xdc,0x04,0x69,0x88,0x68,
	0x68,0xf3,0x21,0xf3,0xa6,0xeb,0x1c,0x90,0x23,0x40,0xf0,0x59,0xdb,0xbf,0x74,0xb6,
	0xa3,0x43,0xe9,0xd9,0x69,0x51,0x89,0x6c,0xbb,0xae,0x54,0xf7,0x71,0xc4,0x76,0xda,
	0x18,0x67,0xdc,0xb7,0x31,0x48,0x3a,0x6a,0xb3,0x9a,0x97,0x59,0xfb,0x79,0x4b,0x46,
	0xe0,0x3c,0x05,0xe9,0x5c,0xf1,0xba,0xf7,0xa8,0x64,0x86,0xd5,0x7b,0x46,0x6c,0x3e,
	0x66,0xc4,0xa2,0x9c,0x9b,0x77,0x9d,0x52,0xb3,0x72,0x5e,0x36,0xa0,0x8e,0xa0,0x80,
	0x39,0x1d,0xa6,0xb4,0x9e,0x18,0x9f,0xe2,0xb0,0xcb,0x5d,0xeb,0xb5,0x65,0x53,0x1c,
	0xbb,0x96,0xf3,0x3e,0xda,0x0c,0xac,0xa3,0x23,0xb2,0x1f,0x20,0x96,0x00,0x3f,0x10,
	0x25,0xba,0x08,0x34,0xa4,0x35,0xd3,0x6a,0xc5,0x2e,0xd8,0x3f,0x27,0x26,0x31,0xcf,
	0x34,0x6d,0x8f,0x60,0xca,0x44,0xaa,0xf8,0xe4,0xaf,0x62,0xfa,0x59,0x27,0xa9,0xf6,
	0xfa,0x3b,0x52,0x9d,0xe7,0x7d,0x40,0xa1,0xa7,0x12,0x5b,0xc4,0x4c,0x1f,0x0a,0x4d,
	0x1f,0x0e,0xa8,0xd2,0x25,0xb7,0x1d,0x41,0xd3,0xc1,0xd5,0x0b,0x95,0xb6,0x83,0xa0,
	0xa6,0xf1,0x16,0x8b,0xd3,0x5d,0x52,0x2e,0x53,0xfd,0x54,0x71,0x42,0x5c,0xf4,0xfe,
	0xc1,0x9d,0x81,0x29,0x83,0xdc,0x7d,0xeb,0x3b,0x3b,0x34,0xa5,0xd4,0x57,0x6c,0x66,
	0x9e,0x70,0x3f,0xc7,0x2d,0xf6,0x24,0x50,0x00,0x8f,0xaa,0x41,0x45,0xdb,0xcd,0xb8,
	0x58,0xcd,0x4c,0x63,0x45,0xd3,0xea,0x90,0x5f,0xf0,0x39,0xb2,0xb9,0xe0,0x07,0x48,
	0x2c,0xcc,0x65,0xa8,0xc2,0xe6,0x20,0x28,0xcc,0xe7,0xb8,0xe3,0xd3,0x5b,0xf4,0x4b,
	0x94,0x37,0xb3,0x1f,0xdc,0xfc,0xd2,0xad,0xc4,0x74,0x4e,0xa4,0x4a,0x74,0xf1,0x04,
	0x6f,0x95,0xed,0xa5,0xa8,0x7d,0xbf,0x0a,0x43,0x90,0xbd,0xb7,0x19,0xcc,0x87,0xc8,
	0xd2,0xc7,0x6f,0x71,0xc0,0x60,0xd9,0x1a,0xe6,0x5d,0x1f,0x73,0x3f,0x97,0x86,0x63,
	0xf7,0x46,0xcc,0x8d,0x37,0x52,0x22,0x56,0x95,0x7c,0xd0,0xc5,0x6f,0x1b,0xa2,0x40,
	0x15,0x8f,0x1a,0x1b,0x48,0x79,0xe0,0x02,0xe9,0x9e,0xcf,0x84,0x3d,0x73,0xdf,0x24,
	0xf5,0x7b,0xc1,0x81,0xbd,0x54,0x2c,0x48,0xfe,0xba,0x44,0xec,0x18,0x8b,0xf7,0xb3,
	0xcc,0x58,0x93,0xee,0x65,0x2a,0x9b,0x09,0x6a,0xd3,0x27,0x15,0x58,0x81,0x9f,0x0a,
	0xaf,0xe9,0x29,0x26,0xf2,0x28,0xf1,0x54,0x58,0xbd,0x37,0x4f,0x2f,0xfb,0xa2,0xb6,
	0x9a,0xb2,0x38,0x90,0x8b,0x51,0xbb,0x18,0xb0,0x36,0xd0,0xb4,0xbc,0xfa,0xce,0xfa,
	0xdb,0x10,0xc3,0x98,0x99,0x0a,0xb1,0x78,0x16,0x93,0x13,0x87,0xca,0x45,0x4d,0x23,
	0x1c,0xf6,0xd3,0x4b,0xfb,0xc4,0x51,0x11,0x33,0x38,0xfd,0x4b,0x42,0x68,0xb7,0x3f,
	0x92,0xc4,0x22,0x07,0x6d,0xeb,0x3f,0x20,0xe2,0xeb,0xff,0x68,0x3c,0xfd,0x68,0xe1,
	0xfa,0xe4,0x13,0xdf,0xa2,0x6e,0x4d,0xbc,0xba,0x5d,0x8e,0xdc,0x1d,0x79,0xd1,0xe6,
	0xd8,0x8a,0xcc,0x9d,0xf3,0xb6,0x92,0xdc,0x4f,0x39,0x3d,0x59,0x6b,0xce,0x5b,0x99,
	0xb1,0xe8,0x69,0xa3,0x5d,0x50,0x2b,0xa1,0x5e,0x6d,0x5e,0x39,0xf9,0x95,0xb7,0x52,
	0xd0,0xa0,0x06,0x92,0xe8,0x98,0x71,0x6a,0x2f,0x35,0x87,0xb9,0xdc,0xcc,0xd4,0x1e,
	0xe7,0x1b,0x17,0x2b,0x91,0x0d,0x04,0x13,0xa4,0x1d,0x48,0xba,0x7d,0xa3,0xf7,0xd8,
	0x44,0x17,0xc6,0xde,0x90,0xe5,0x38,0xd4,0x30,0xc0,0xa4,0x7f,0x4a,0xe8,0x46,0xb2,
	0x49,0xe3,0x72,0x67,0x3f,0x2c,0xf8,0x0a,0x96,0x22,0x28,0xa9,0xc8,0xf0,0x03,0x2d,
	0xe0,0x2e,0xd9,0x50,0x0e,0x2f,0xa1,0x22,0x95,0xb2,0xce,0x87,0x0f,0x68,0xb2,0xc5,
	0x76,0x1d,0xbf,0xb6,0x7e,0xcc,0x41,0x55,0x9b,0x57,0xc4,0xee,0xc5,0x06,0xbd,0x0c,
	0xa3,0x52,0x7c,0xc9,0xbb,0x62,0x94,0xb0,0xe8,0x67,0xc1,0x53,0x76,0xa0,0x99,0xe9,
	0x30,0x83,0xe9,0x7c,0xbf,0xd3,0x91,0x92,0x3d,0x17,0x5c,0x36,0xdd,0x91,0x86,0x0b,
	0x7a,0x03,0x20,0x53,0xd8,0xdd,0x47,0x4e,0x8b,0xe8,0x32,0x80,0xcb,0xe4,0xe7,0xfb,
	0x2d,0xd6,0x6f,0x23,0x58,0x98,0x67,0x09,0x11,0x9d,0x0a,0x46,0x86,0xe7,0xa8,0xed,
	0x76,0x9c,0xc4,0x2d,0x13,0xa7,0x35,0x44,0xa8,0x1b,0x1f,0x55,0xc8,0x46,0x9d,0x88,
	0x7a,0x2c,0x76,0x3f,0x3e,0x56,0x0f,0xc6,0xd0,0x0d,0x06,0xc0,0xd7,0xb1,0x23,0x46,
	0xf3,0x8a,0x47,0x8b,0x37,0x7e,0x51,0x0b,0x87,0x97,0xa6,0x9c,0x61,0xd4,0x75,0x5d,
	0x14,0x4e,0x71,0x3c,0xd9,0x2a,0x7e,0x3c,0x60,0xe5,0xf9,0x87,0x2e,0x87,0x8a,0x51,
	0x7d,0xa7,0x1f,0x27,0x44,0x43,0x14,0xd8,0x32,0x94,0x84,0x5e,0x1a,0x26,0x5f,0x93,
	0x48,0x70,0x11,0x34,0xde,0x73,0xa9,0xbc,0x0f,0x32,0x7f,0xca,0x22,0x67,0xcf,0x80,
	0x78,0x56,0x0a,0x98,0x76,0xac,0xe7,0x75,0x1e,0x2a,0x8f,0x8b,0xf2,0x82,0xf9,0x57,
	0x58,0x05,0xac,0x29,0x86,0xe8,0xc8,0x53,0xf8,0x19,0xf3,0x5b,0x4d,0x51,0xd2,0x97,
	0x72,0xbe,0x99,0xd4,0xa5,0xf1,0x8d,0x7b,0xfb,0x69,0x77,0x58,0xfe,0x57,0x50,0x95,
	0xf6,0x0e,0x9d,0xa6,0xd7,0xd0,0xf8,0x09,0xf4,0x69,0xe3,0x7c,0x67,0xb0,0x1c,0x9e,
	0xfd,0x5f,0xa4,0x4d,0x90,0x4f,0xcf,0x50,0xd8,0x12,0xf3,0x85,0x36,0x25,0xbc,0x47,
	0xb1,0xba,0x9a,0xa2,0xd1,0x4d,0xc9,0x38,0xa4,0x21,0x1c,0x1d,0xb9,0x96,0xb7,0x9e,
	0x98,0x74,0x51,0x2b,0x22,0x25,0x7c,0x12,0x57,0x30,0x2f,0x81,0x1d,0x26,0x02,0x9c,
	0x6d,0xa3,0xa2,0x7f,0x21,0x15,0xbf,0x60,0xe2,0x96,0x86,0xb5,0x93,0x76,0x49,0x5e,
	0x55,0x92,0x49,0x16,0xd0,0x1b,0x6f,0x2b,0xa2,0xf8,0xdd,0x14,0x92,0x94,0x2e,0x89,
	0x65,0xc7,0xe4,0x2e,0x35,0x73,0xab,0x0f,0x5d,0xeb,0xd5,0xf1,0xb7,0x46,0x82,0xaf,
	0x69,0xe5,0x11,0xcf,0xb9,0x78,0x1d,0x9d,0xa0,0x61,0xa7,0xf0,0xc1,0x71,0x2a,0xc0,
	0xe5,0x27,0x4a,0x62,0xcf,0x09,0xe9,0x13,0xed,0xec,0x80,0x42,0xe1,0xcf,0x18,0x21,
	0xae,0x8c,0x4c,0x31,0x32,0xe2,0x4a,0xcd,0xf4,0xf5,0x57,0x80,0x85,0x52,0x1c,0x61,
	0xbf,0x9c,0xf0,0xad,0x90,0x86,0x03,0x46,0xe4,0xe5,0xb5,0x71,0x60,0xb8,0xfa,0x15,
	0xb5,0x90,0x63,0x0b,0xed,0x56,0x42,0xc5,0xa0,0x61,0x27,0x10,0xc7,0x29,0x17,0x6e,
	0x50,0xd5,0xae,0x17,0x6d,0x71,0x36,0xc5,0x54,0xe2,0x91,0x55,0x4f,0xd8,0x18,0x2f,
	0x7c,0xfe,0xbd,0xeb,0xec,0xd9,0x70,0x26,0x4a,0xc5,0x78,0x1e,0x04,0x43,0x14,0xb7,
	0x62,0x70,0xcd,0x27,0xa9,0xa3,0xf6,0x31,0xbe,0xbb,0x4e,0xe5,0xd1,0xfc,0x03,0xa9,
	0x09,0xf2,0x07,0xd8,0x77,0xf7,0x1e,0x35,0x1e,0xe9,0x7b,0xc7,0xd1,0x9f,0xe0,0x1d,
	0x79,0xf0,0xc3,0x1c,0x80,0x23,0x7c,0x69,0xff,0x9d,0x3e,0x9a,0xe1,0xf0,0x1b,0xf5,
	0x28,0x8d,0x5f,0x6a,0xaf,0x84,0x35,0x57,0x43,0x5c,0x56,0xaa,0x92,0xa7,0xd2,0x8d,
	0x3a,0xf6,0x60,0x0a,0xab,0x83,0x9d,0x09,0xf6,0x87,0xfe,0x90,0x71,0x8f,0x3a,0x90,
	0x32,0x18,0xa9,0x39,0xf3,0xf7,0x80,0x3c,0x00,0x12,0x02,0x23,0x17,0xcd,0xc8,0xb3,
	0x75,0xed,0x5c,0xcb,0x53,0x13,0xc5,0x58,0xa7,0x79,0x48,0x2e,0x21,0x05,0x2b,0xf8,
	0x45,0x2a,0xd2,0xdb,0x1d,0xd2,0x19,0x38,0x5f,0xc9,0xd1,0x28,0x84,0x62,0x7a,0x4f,
	0xe0,0x6c,0xe5,0x8a,0x7f,0xcc,0xae,0x54,0x7b,0xd9,0x66,0x92,0x56,0xca,0xb7,0xdd,
	0x6a,0x21,0xed,0x29,0xb0,0x06,0x65,0x77,0x0d,0x3a,0x64,0x70,0x46,0xb8,0xa8,0xc9,
	0x3a,0xe5,0xcf,0x5c,0x50,0x1d,0x17,0x04,0xdb,0xb2,0xc7,0x4a,0x41,0xc0,0x80,0x40,
	0xfc,0xc4,0xa1,0x92,0x68,0x94,0xfb,0x92,0xe6,0x14,0x4c,0xc4,0x65,0x71,0x8c,0xf6,
	0xf9,0xe3,0x2b,0x5d,0x9c,0x6c,0xa6,0x8d,0x32,0xba,0xe7,0x4b,0xd0,0xe3,0xe6,0x9f,
	0x17,0x9c,0x20,0x76,0xc5,0x27,0x7e,0x67,0x33,0x66,0xf8,0xf9,0x41,0x07,0x03,0xe0,
	0xc6,0x02,0x84,0xcd,0xf5,0x5b,0x8a,0xab,0x52,0x7a,0x97,0xc0,0xc6,0x6a,0x07,0xec,
	0xb6,0x09,0xce,0x74,0x3b,0x6d,0x52,0x6b,0x0d,0x9c,0xda,0x00,0x73,0xbb,0x27,0x3c,
	0x42,0x9e,0xf8,0x9d,0x49,0xfb,0x35,0xac,0x1f,0x06,0x7a,0x08,0xa1,0x81,0x3e,0xc1,
	0xbf,0xeb,0xf2,0x87,0x81,0xaf,0xca,0x48,0x78,0x55,0x69,0xca,0x57,0x95,0xb4,0x42,
	0x18,0x18,0x27,0x45,0x31,0x8c,0x4f,0xf1,0x68,0x69,0x3b,0x14,0xa1,0xf8,0xae,0x08,
	0xf9,0x40,0x9a,0x35,0x08,0xcd,0x4f,0x2c,0x35,0x71,0x54,0xe5,0x73,0x79,0x97,0x55,
	0x4b,0x3f,0x6c,0x44,0x1a,0x6d,0xc8,0x76,0x80,0x8f,0xea,0xdb,0xc6,0x93,0x4f,0xfb,
	0x14,0x62,0x7d,0x10,0x84,0x4e,0xc3,0x15,0x41,0x0a,0x07,0x9f,0x41,0xc0,0xfb,0x9f,
	0xc5,0xc1,0x59,0x33,0xb5,0x46,0x46,0x78,0x21,0xa5,0xed,0x47,0xd0,0xca,0x0f,0xc6,
	0x62,0x50,0x20,0xd2,0x8c,0xfa,0x9b,0x00,0x8f,0xc3,0xb4,0x40,0x65,0x9d,0x2f,0x07,
	0xf0,0xe1,0xd0,0xf3,0x78,0x17,0x3d,0x79,0xbc,0x32,0x93,0x6f,0x41,0x47,0x23,0xa3,
	0xbc,0x09,0x38,0x60,0xa0,0x1e,0x21,0xdc,0xea,0x20,0x56,0xdd,0x46,0x03,0xbb,0x80,
	0x3b,0x61,0xbc,0x42,0x40,0xe6,0xe4,0xff,0x2d,0x8b,0xb3,0xb4,0x56,0x5f,0x45,0x2a,
	0x0c,0x4b,0x05,0x0f,0x9e,0x9d,0xf3,0x59,0x01,0x2d,0xce,0xf3,0x84,0x08,0x4f,0x6a,
	0x0f,0x8a,0xe5,0x05,0x9b,0x61,0x73,0x43,0x64,0xf1,0xa2,0x4a,0x21,0x4c,0x15,0x21,
	0xd0,0xb7,0x15,0xfa,0x22,0x3f,0x6b,0x4c,0xa2,0x92,0xbc,0x90,0x17,0x9f,0x11,0x4a,
	0xa6,0xec,0xd7,0x63,0xc2,0xf6,0xff,0x95,0xc4,0xe7,0x23,0xdd,0x71,0xc0,0xc2,0x84,
	0x5f,0x15,0x8a,0xe9,0xba,0x97,0xe1,0x53,0xbd,0xd1,0xf9,0xfc,0x92,0xd7,0xa2,0x13,
	0xd4,0x43,0x5c,0x35,0xde,0x30,0x70,0xe4,0x22,0x9e,0xdf,0x06,0xe1,0x48,0x8d,0xdb,
	0xbb,0xed,0x15,0x4f,0xcd,0xb6,0x1d,0x94,0x3c,0x3d,0xa2,0xd1,0x6a,0x46,0x8e,0x8f,
	0xc2,0x12,0x34,0x96,0x31,0x2d,0x2b,0x18,0xd3,0xd8,0xd9,0xd1,0xe4,0x0a,0x09,0x65,
	0x18,0x89,0x22,0xae,0xf7,0xc4,0x11,0x13,0x2d,0xa5,0x22,0xe8,0x26,0x56,0x0b,0x6b,
	0x6d,0x71,0x81,0xac,0x2c,0x2d,0xe6,0x6d,0x1e,0x48,0x17,0x2c,0x06,0x68,0x00,0xe0,
	0x1e,0xb6,0x9b,0x12,0x0b,0x15,0x75,0xb0,0x7d,0x54,0x76,0x7c,0xe9,0x67,0x24,0xe5,
	0x50,0x59,0xdd,0xd3,0x21,0x29,0xb1,0x2b,0x0b,0x5a,0x19,0x70,0x05,0xa1,0x9e,0xe9,
	0x06,0x8d,0xd0,0x8c,0x3d,0xbf,0xbc,0xf4,0xdd,0x84,0xa3,0x49,0x39,0xee,0x37,0xa9,
	0x9e,0x83,0xa5,0x1f,0xf3,0xcc,0xd1,0xa2,0xaf,0xbf,0x18,0xc7,0xb9,0x3c,0x38,0xd2,
	0xba,0x62,0x6f,0x8f,0x05,0x7f,0x68,0x57,0x2b,0xc4,0x61,0x60,0x52,0xf8,0x3d,0xc7,
	0x42,0x67,0x53,0x88,0x8a,0x61,0x9c,0x82,0x2c,0x5a,0xff,0x50,0xdc,0x9e,0x55,0xbd,
	0xb2,0x15,0x7d,0x72,0x33,0xd7,0xbb,0xb2,0x4b,0xd5,0xbd,0x32,0x92,0x63,0x32,0x0a,
	0xab,0x99,0x99,0xe6,0x2e,0x45,0xeb,0x04,0x77,0x48,0x99,0x5e,0xa2,0x86,0xfd,0x50,
	0x8e,0x0e,0xa7,0x4a,0x9f,0x97,0x3e,0xbd,0x9d,0xe3,0xd8,0xc6,0x25,0x4b,0xe9,0x02,
	0x71,0x94,0x56,0x0d,0xf3,0x2a,0xca,0xb3,0x02,0xc1,0x6e,0x4e,0xf6,0xd4,0xdd,0x77,
	0xa0,0x50,0x69,0x09,0x97,0x80,0xf4,0xbe,0xeb,0x62,0x4d,0xda,0x1c,0x4d,0x4a,0x07,
	0x90,0x3d,0x5e,0x68,0xe5,0x24,0x85,0x32,0xea,0xdb,0xb0,0xa5,0x1e,0xf4,0x3f,0xde,
	0x72,0xe2,0x87,0xcc,0xfc,0x6b,0x6f,0x17,0x2f,0xa8,0x57,0x60,0x25,0x80,0xce,0xab,
	0x01,0xff,0xc0,0x5e,0x8f,0x92,0x9e,0xed,0x1b,0x20,0x51,0x8a,0x78,0x16,0x05,0x23,
	0x4c,0x46,0xd8,0xce,0x9f,0x33,0x74,0x03,0x9d,0x71,0x82,0x9c,0x89,0x96,0xcc,0xfb,
	0xac,0xbf,0xbd,0xa5,0x7e,0x82,0xe8,0xcb,0xdb,0x88,0x67,0xf1,0x1f,0xce,0x23,0x27,
	0x41,0x83,0x04,0xec,0x65,0x31,0xd1,0x8e,0x37,0xac,0x26,0xa3,0x2c,0x0c,0x55,0xd2,
	0xcf,0x07,0x05,0x25,0xc6,0x5b,0x16,0xa0,0x3b,0xb3,0x87,0x9f,0xa9,0x72,0xcd,0x92,
	0x87,0xc7,0x6b,0xbf,0xdc,0x65,0x28,0x36,0xf3,0xf2,0xd4,0x3f,0x98,0x5c,0x41,0xd2,
	0x41,0xf6,0xdf,0x72,0xfa,0x3b,0x20,0x6f,0x34,0xba,0xb1,0x23,0x03,0x37,0xbd,0x2c,
	0xee,0x93,0xa6,0x4a,0xe1,0xe8,0xfd,0x95,0x67,0x68,0x46,0x46,0xdd,0x74,0xdc,0xf2,
	0x3c,0x31,0xf0,0x54,0x4a,0xfa,0xca,0x18,0x73,0x81,0xe7,0x41,0x74,0x3e,0x30,0x28,
	0xf5,0xa7,0x88,0x5f,0x42,0x8a,0x26,0x9f,0xc4,0xa0,0xe4,0xe1,0x92,0x9e,0x28,0xa9,
	0x14,0x80,0xcd,0x43,0xb5,0x81,0x9d,0xb3,0x63,0xaf,0x91,0x88,0x12,0x6a,0x61,0xdd,
	0xcd,0x9e,0x44,0xb1,0x5c,0xf9,0xf2,0x70,0x9d,0x8d,0xa0,0x0d,0x77,0xe8,0xf0,0x9f,
	0x61,0xb6,0x83,0x64,0x28,0xe1,0x93,0x92,0x43,0x25,0x06,0x53,0x55,0x07,0xbf,0xf0,
	0xdc,0xfd,0xdc,0xd3,0x67,0xd6,0xcc,0x33,0x6f,0x0f,0x63,0x83,0x02,0x80,0xe2,0x99,
	0x3a,0xb6,0x0e,0xe4,0xd1,0x7f,0x24,0x9b,0x49,0xd4,0xf0,0xad,0xdc,0x62,0x97,0x25,
	0xbc,0x51,0xb1,0xbc,0x87,0xb9,0xa3,0xc5,0x1b,0xba,0x4b,0x7a,0x07,0xf0,0x65,0x99,
	0xa6,0x66,0x9e,0x4e,0x5c,0xf0,0xa0,0x35,0xa7,0x2b,0x06,0x39,0x57,0xa3,0x8f,0x8e,
	0x82,0x68,0xd4,0x2a,0x3a,0x1b,0x1b,0x34,0xfb,0xcf,0x35,0x1a,0x78,0xb5,0xe9,0x12,
	0xaf,0xda,0x77,0x95,0x78,0x01,0xb7,0x27,0x24,0x8a,0xb1,0x1b,0x2a,0x4c,0x98,0x96,
	0x6e,0xd5,0xea,0x63,0xca,0xa8,0x58,0xf3,0xad,0xac,0xe7,0xd1,0xbf,0xfe,0x8b,0x5d,
	0xf4,0x52,0xdd,0x06,0x52,0x1d,0x91,0x62,0xb0,0x31,0xdf,0x22,0x9e,0x61,0x15,0xcd,
	0x1e,0x4b,0x01,0xd3,0xf4,0xfc,0x6c,0xe4,0x6c,0xc4,0xa3,0x6b,0x02,0x70,0x4a,0x03,
	0x0c,0xd9,0x6d,0x40,0x8c,0x06,0xc6,0xb4,0xb0,0xe8,0xf7,0x70,0x77,0x3d,0xe5,0xe0,
	0x29,0x05,0xd8,0x75,0xc0,0xbf,0x32,0xdd,0x0c,0x6e,0xb9,0x3c,0x84,0xa9,0xc8,0x2f,
	0x99,0xb4,0x04,0xae,0xbc,0xac,0x25,0xfe,0x47,0x08,0xe3,0x25,0x19,0x81,0xe8,0x5d,
	0x53,0xca,0x68,0x75,0x62,0x6b,0x03,0x4c,0x25,0x6c,0x98,0x3a,0x8e,0x76,0xc5,0x94,
	0xc6,0x98,0x47,0x4b,0xc5,0xda,0x86,0x2d,0x80,0x04,0xb9,0xbf,0x82,0xbe,0xb1,0xc5,
	0xfc,0x97,0x36,0xff,0x76,0x5e,0xc4,0x5c,0xd1,0x9e,0x23,0x56,0x90,0x5d,0x7c,0x1d,
	0x2a,0xd9,0x8f,0x90,0x95,0x1d,0x62,0x0f,0x86,0xc2,0xca,0x9b,0x9f,0xf9,0x08,0xb5,
	0xaa,0xbd,0x53,0xbc,0x6c,0x5a,0xcc,0x19,0x14,0x8d,0x31,0xbb,0xb4,0x42,0xfe,0x0a,
	0xb9,0x2c,0x62,0x81,0x84,0xa6,0x97,0x28,0x20,0x30,0x79,0x91,0x39,0xbb,0xd5,0xd7,
	0x91,0xf8,0x4a,0xdf,0x7e,0xd4,0x1f,0xfa,0x57,0xc3,0x15,0x7d,0xd8,0x99,0x87,0x7c,
	0x21,0xac,0xa2,0x41,0x9f,0xc8,0x1c,0x44,0x40,0x4f,0x3f,0xf3,0x32,0x59,0x6a,0xc3,
	0x29,0xcc,0x48,0x30,0x94,0x7f,0x5a,0xf5,0x14,0x58,0x48,0x83,0xbd,0x7f,0x7b,0xeb,
	0x5a,0x57,0xa9,0x50,0xaf,0x1e,0xdc,0xbc,0xe6,0x13,0x0e,0x2a,0x94,0xf2,0xe6,0xb1,
	0x65,0xd9,0x43,0x65,0x47,0x6a,0xa7,0x29,0x41,0xf0,0xab,0xbf,0x24,0xff,0xb4,0x1c,
	0x45,0x61,0xa0,0x39,0x76,0x75,0x7b,0xc8,0x6a,0x6e,0xd3,0xe9,0xf9,0xd2,0xa8,0x98,
	0x7b,0x19,0x9b,0x30,0x94,0x82,0x7a,0x6c,0xbe,0x36,0xcd,0xaf,0xe9,0xbd,0x67,0x8c,
	0x78,0x29,0x31,0xb1,0xe4,0x6c,0xfb,0x89,0xb0,0xd7,0x30,0x02,0xb4,0xe8,0x90,0x34,
	0x46,0x38,0xca,0x95,0x87,0xd3,0x37,0x88,0x6c,0xc1,0xbc,0x56,0x10,0xf7,0xae,0x23,
	0x41,0x55,0xe2,0x71,0xb2,0x89,0x8f,0x02,0xa7,0x73,0x1c,0x4d,0x04,0x37,0x5c,0x5b,
	0xd2,0x91,0xb4,0xb6,0xcb,0x13,0xf4,0xc4,0xc4,0x52,0x5a,0x8f,0xea,0x03,0x35,0x7a,
	0x50,0x72,0xf5,0x1b,0x30,0x55,0x0d,0x10,0xeb,0x32,0x8e,0x00,0x08,0xca,0xa6,0x32,
	0x83,0xed,0x27,0xa4,0xa4,0x0a,0x78,0x17,0xaf,0x47,0x7c,0xa0,0x7b,0x7d,0xce,0x8b,
	0xb9,0x68,0x50,0xc4,0x38,0x1f,0x24,0x18,0x1d,0xc1,0xc6,0x56,0xe3,0x3e,0xb5,0xd7,
	0xfb,0x79,0x3b,0x9f,0x02,0x87,0x27,0x34,0x63,0xae,0xd7,0xd0,0x1b,0xb0,0x20,0xa4,
	0x02,0xcb,0x2b,0xd9,0x6a,0xac,0xd2,0x79,0xc9,0x1d,0x4e,0x88,0x6e,0x07,0x8b,0xe3,
	0xaa,0x4e,0x38,0x2a,0xf4,0x31,0x12,0xee,0x12,0x6e,0x3a,0x4d,0x30,0x19,0xbb,0x32,
	0x81,0x47,0x8c,0x3f,0x3e,0x99,0xf5,0x38,0xf7,0x38,0xe9,0xe3,0x36,0x67,0x96,0x1c,
	0x82,0x37,0x05,0x19,0xa9,0xc1,0xe6,0xba,0xf4,0x12,0x0f,0xb3,0x8e,0x3d,0x65,0x38,
	0xcf,0x95,0x1b,0x4e,0x7a,0x4a,0x5a,0x03,0xc4,0xc6,0x35,0xad,0x6a,0x6a,0x98,0xf7,
	0xd9,0x47,0x50,0x02,0xc9,0xf6,0x63,0xbf,0x3d,0x3d,0x87,0x6e,0x43,0x26,0x6f,0x36,
	0x8c,0xca,0xc3,0x19,0x09,0xd4,0x2b,0xe5,0x4e,0x82,0xbb,0xed,0x25,0xd5,0x63,0x24,
	0xff,0xa1,0xb2,0x01,0x6e,0x77,0xc6,0xa6,0xeb,0x2d,0x6d,0xac,0xfd,0x58,0x2f,0x70,
	0x50,0x79,0xa4,0x89,0xed,0xb6,0xed,0x14,0x7d,0xee,0xa7,0x67,0x2f,0xa8,0xc1,0x7d,
	0xd6,0x3a,0xfd,0xd2,0x18,0x77,0x7b,0xb5,0xc6,0x91,0xf9,0x94,0x91,0xe5,0xe6,0xfc,
	0xc2,0xb5,0x75,0x66,0x2e,0xd4,0xcb,0xc4,0x52,0x9a,0x67,0xc9,0x70,0xb1,0x1f,0xf7,
	0xe7,0x6b,0xa9,0xa0,0x18,0xf6,0x1e,0x56,0xa1,0xe6,0x37,0x19,0x59,0x8b,0xab,0xa5,
	0x1a,0x07,0xdf,0xc0,0xed,0x41,0xba,0xab,0x67,0x24,0x29,0x25,0xad,0x2d,0xa2,0x84,
	0xfb,0x22,0x5d,0xf3,0x6e,0xc1,0x2c,0xd3,0x6e,0x89,0xaa,0x61,0x35,0xc6,0xf2,0x32,
	0x36,0xc3,0xff,0xc1,0x09,0x92,0x56,0xfb,0xb3,0x3f,0x12,0x3b,0xb3,0x62,0x4b,0x7c,
	0x2a,0xe5,0x00,0xb0,0xc9,0x31,0x3f,0x98,0x5f,0xcf,0x44,0x2c,0x2c,0x38,0xc9,0x0f,
	0x15,0xcc,0x5e,0xa0,0x7a,0xa7,0x81,0x7d,0xe8,0x0c,0x93,0xb6,0xd9,0x7e,0xd0,0xd7,
};


/* 317-0018 FD1089A */
static const UINT8 key_0018[0x2000] =
{
	/* data */
	0xca,0x7c,0x36,0x46,0x27,0x75,0xf7,0x04,0x8d,0x6f,0xf8,0xd1,0xea,0xb6,0x20,0xc1,
	0xb3,0xb2,0x0a,0xe0,0x1b,0xf1,0x5f,0xce,0x20,0x68,0xa8,0x03,0xf4,0x2d,0x74,0x6c,
	0x01,0x4e,0x81,0x3c,0x3f,0x1a,0x55,0xe0,0x9f,0x35,0x3f,0xef,0x7c,0x96,0x18,0x5e,
	0xc5,0x79,0xd0,0x1e,0x1c,0x99,0xd6,0x35,0x26,0x26,0xf9,0x37,0x76,0xc0,0x9d,0xf8,
	0x72,0x51,0x7b,0xd4,0x05,0x39,0x03,0x64,0x32,0x53,0x70,0x4a,0xd5,0xed,0x02,0x1d,
	0x56,0x69,0x40,0x63,0x47,0x99,0x86,0x3a,0xe7,0x4a,0x19,0x0c,0xc1,0x6c,0x93,0xb9,
	0xf3,0x5b,0x20,0x79,0x16,0x5b,0x45,0x53,0xf5,0x16,0x61,0x49,0x5d,0x53,0x17,0x74,
	0x73,0x64,0xdc,0xfd,0x9d,0x86,0x25,0x9d,0x68,0xfb,0xa6,0x2d,0x64,0x47,0xb6,0x63,
	0x2e,0x6a,0xd2,0xc4,0x3b,0x73,0xfe,0x9f,0xc0,0x66,0x65,0x54,0xf5,0x8e,0x00,0x71,
	0x1d,0x93,0xc3,0x67,0x8c,0xd7,0x6d,0x95,0x7f,0xa6,0x7d,0x82,0x2a,0x21,0xe9,0xd7,
	0xa3,0x98,0x8d,0xae,0x1f,0xda,0x80,0xdb,0xaf,0xf7,0xc3,0x6c,0x92,0xd9,0xf8,0xcf,
	0x4d,0x71,0x64,0xea,0x4b,0xc9,0x1e,0x41,0xac,0xe3,0x7f,0x46,0x44,0x02,0x02,0x6c,
	0xaf,0x7f,0x12,0x53,0x75,0xd6,0x05,0x12,0xa9,0xa5,0xf6,0xdb,0x63,0x22,0xc1,0xaf,
	0x84,0x11,0x68,0xc4,0x7d,0xa0,0x99,0x73,0x1b,0x48,0x15,0x88,0x79,0xb8,0xf2,0xca,
	0xeb,0x47,0x14,0x06,0x62,0x70,0x62,0x1c,0xa5,0x01,0x49,0x35,0x0a,0x6c,0xd0,0x42,
	0xe3,0xcb,0xe5,0x4b,0x97,0xa8,0x2a,0xc9,0xc2,0x3f,0xea,0xb0,0x2e,0x97,0xd6,0xd4,
	0x6e,0x47,0x56,0xa8,0x76,0x6b,0x6b,0x2a,0xd9,0xa9,0x86,0x2f,0x96,0xd3,0xb6,0xc9,
	0x8d,0xbd,0xa1,0xdf,0x4f,0xbd,0xc8,0xf6,0x05,0x7d,0x96,0x68,0xad,0xb1,0xc2,0xf1,
	0x76,0x82,0x09,0x4b,0xcd,0xc3,0x34,0xcf,0x6b,0x56,0xa1,0xed,0x9a,0x90,0x5e,0x01,
	0x2e,0xee,0xc2,0xf9,0x74,0xde,0x7f,0x7b,0x9a,0x64,0x64,0x1e,0x25,0xb8,0x04,0x8c,
	0xd6,0x2a,0x93,0x79,0x20,0x28,0x76,0x61,0x51,0x49,0xf5,0xa9,0x2b,0xa1,0x79,0xd1,
	0xfa,0xb8,0x9b,0x93,0x62,0xe6,0x2c,0x22,0x81,0x52,0xa9,0x6c,0x5d,0x21,0x28,0xab,
	0x38,0x60,0x03,0x55,0xdc,0x5e,0x22,0xe1,0xcf,0xa2,0xc9,0x6a,0x48,0x3b,0x4a,0xfd,
	0x50,0xe6,0x67,0x67,0x8c,0xdc,0xb5,0xb3,0x0c,0x6e,0xe4,0x42,0x2f,0x6f,0x61,0x86,
	0xa3,0x29,0x03,0x06,0xce,0x85,0x6f,0x43,0xcd,0xbc,0x29,0xd1,0x16,0x3a,0xaa,0xb8,
	0xb6,0x29,0xfb,0x8f,0x60,0x8e,0xec,0xaf,0xd0,0x76,0xe9,0xc6,0xb8,0x6e,0x43,0x9d,
	0xa1,0xf7,0xb6,0xfa,0xc0,0xd7,0xc6,0x52,0x55,0xab,0x84,0x96,0x50,0x40,0x82,0x7e,
	0x00,0x62,0x91,0xf1,0x60,0x44,0x16,0xe6,0x3b,0x55,0x32,0x77,0x2a,0x1a,0xaf,0xa6,
	0xe4,0xab,0x1a,0xf3,0xce,0xfe,0x28,0x06,0xaa,0x5e,0x6e,0xca,0x29,0xb0,0x0f,0x87,
	0x69,0xf6,0xa3,0xf9,0xdb,0x22,0xbd,0x95,0x90,0xb1,0x87,0x4e,0xa2,0xd6,0x80,0x21,
	0xd1,0xed,0x00,0x1e,0xdc,0x47,0xb9,0x49,0x3c,0x6f,0xa4,0xef,0xd3,0xe8,0x02,0x47,
	0x49,0x96,0xc0,0x3a,0xc7,0x3c,0xdd,0x1a,0x71,0xb5,0xb8,0x24,0xaa,0xb2,0x98,0x0b,
	0x87,0x8c,0x8e,0x9a,0x20,0xb4,0x2d,0x0a,0x1e,0xe7,0x3d,0xa0,0x44,0x27,0xf7,0x4f,
	0x2f,0xcc,0x59,0xc4,0xd4,0x48,0x8e,0x8c,0xed,0x7c,0x98,0xaa,0xc7,0x69,0x7b,0x39,
	0xe3,0x09,0xdc,0x90,0xcd,0xca,0x73,0xc3,0x29,0xfa,0x09,0x1e,0x02,0x68,0xf5,0xe8,
	0x27,0x7f,0x82,0x46,0x38,0x49,0x01,0xb3,0xce,0x95,0x37,0xc4,0x93,0x1d,0x2e,0xf7,
	0xee,0x49,0xcd,0xb2,0x2f,0x2c,0xd1,0x5e,0x4d,0xce,0xcd,0x28,0x9d,0x15,0x11,0x13,
	0xa9,0x10,0xde,0xbc,0x26,0xf9,0x89,0xf2,0xfb,0x5b,0xe7,0xfb,0xe2,0xda,0x17,0xa4,
	0x6c,0x1f,0x5c,0xc2,0xfd,0x02,0x8c,0xdc,0xcb,0x72,0x05,0xb3,0xa7,0x3b,0x33,0x29,
	0xc6,0x54,0x3f,0x50,0x62,0xc1,0xa2,0x1d,0x3d,0x79,0x24,0x3a,0xec,0xad,0x67,0x72,
	0x8f,0x37,0xc0,0x88,0x77,0x99,0x58,0x14,0xf1,0x53,0xfe,0xba,0x44,0x63,0xe3,0xf6,
	0x35,0x57,0xea,0x15,0x47,0xe6,0xe5,0x95,0x7c,0xbf,0xe3,0x09,0x8b,0x1f,0xa7,0x61,
	0x70,0x62,0x54,0xa6,0xb0,0x20,0x3d,0x37,0xe2,0x94,0x3b,0x24,0xe4,0x23,0x81,0xfd,
	0xc5,0x20,0x4f,0x86,0x71,0xbb,0x34,0xb5,0x73,0xc5,0x6a,0xc5,0x2e,0x5d,0x4c,0x67,
	0x99,0x10,0x89,0xdb,0x6c,0x7c,0x85,0x6e,0x64,0xab,0x0d,0xfd,0xf7,0x36,0x20,0x43,
	0xbb,0x0e,0x97,0x9c,0xeb,0xa0,0x82,0x28,0xbc,0x5b,0xa4,0xbc,0x5f,0x13,0xac,0xf1,
	0x65,0x81,0x2a,0x49,0x0f,0xa2,0xf4,0x45,0x42,0xe7,0x42,0xc6,0x9f,0xe6,0xcb,0x21,
	0x70,0x29,0x9e,0xf5,0xc6,0xee,0xf4,0x76,0xb9,0xfe,0x19,0xed,0x3c,0xbb,0xad,0x17,
	0x57,0x96,0x47,0x03,0x69,0x91,0x82,0x41,0xfa,0x60,0x2c,0xd3,0x36,0xb8,0x29,0xe7,
	0x0b,0xf2,0x49,0x51,0x28,0x09,0x85,0x8c,0x67,0x94,0xc8,0xcc,0x77,0xd5,0x8e,0x74,
	0xf4,0x79,0x65,0x69,0x49,0xee,0x2f,0x31,0x29,0xc4,0xcd,0xde,0x75,0xc8,0xfd,0xbd,
	0x62,0x04,0xec,0xf0,0x59,0x0b,0x37,0xae,0xa5,0x1f,0xaf,0xc6,0x74,0x22,0x69,0xcf,
	0x7c,0xf0,0xd1,0x0f,0x50,0xc0,0x77,0xe6,0xc5,0x16,0x74,0xc0,0x6f,0x3a,0xe8,0x63,
	0x2e,0xd8,0xea,0x82,0x53,0x7c,0xbd,0x15,0xfa,0x2f,0xbf,0x06,0x0e,0x44,0x90,0x0c,
	0xe1,0x34,0xd4,0x5d,0x23,0xe3,0x3d,0xd9,0x0d,0x5f,0xb2,0xd1,0x1f,0x8a,0xc6,0x22,
	0x42,0x20,0xb2,0x87,0x62,0x84,0xcb,0x1c,0x05,0xaa,0xf8,0x12,0x88,0xc4,0x17,0x9f,
	0xdf,0xc4,0x5f,0x96,0x2f,0x89,0x80,0xb1,0x63,0xe1,0xca,0xe9,0x51,0x1d,0xee,0x41,
	0x92,0x3f,0x7e,0xf6,0x92,0xb8,0x6a,0x81,0x33,0xe9,0xf6,0xea,0xc5,0xc8,0x1e,0x67,
	0x55,0xe6,0x4e,0xcb,0x03,0x58,0xe7,0x6e,0x44,0xa7,0xe7,0x43,0xa1,0x16,0xd7,0x47,
	0x2e,0x5e,0xc6,0x8a,0xf5,0xdc,0xf8,0x78,0x28,0x62,0x9e,0x27,0xfa,0x74,0x0f,0x0b,
	0x10,0x1d,0xc5,0x5a,0xe8,0x65,0x51,0xd6,0x4f,0xd3,0x48,0xa2,0xc4,0x85,0x1e,0x1e,
	0xbe,0x58,0x1d,0x17,0x3a,0xdc,0xb4,0x9e,0x46,0xfa,0xa7,0x37,0x90,0x23,0xf1,0x4e,
	0x61,0x28,0x25,0xd4,0xa6,0x39,0xc2,0x27,0x6a,0x96,0x9e,0x54,0x62,0xee,0xc0,0x03,
	0xd2,0x48,0x42,0xb8,0x3a,0x0e,0x1f,0xf9,0x67,0x88,0x27,0xeb,0xe9,0x31,0xd6,0x61,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x2d,0x92,0x4f,0x9e,0x89,0xe8,0x9e,0x1a,0xa6,0xdf,0x08,0x9d,0xd6,0xc1,0x2f,0x92,
	0xd7,0x60,0xbd,0x25,0x65,0x80,0x93,0xf9,0x8d,0xda,0xca,0xc7,0x10,0x62,0x38,0x0c,
	0x1a,0xdd,0x8c,0x32,0xba,0x64,0xfc,0x57,0xad,0x9c,0x10,0x80,0x6a,0xc1,0xa5,0x1f,
	0x45,0xb4,0xb4,0x4d,0xfb,0x55,0xf8,0x78,0x48,0x20,0x0a,0x26,0x24,0xcb,0x88,0x99,
	0xc0,0x0c,0x89,0xb8,0xdb,0xa0,0xc3,0x24,0xdc,0xf2,0xfe,0x67,0xae,0x27,0xef,0x5a,
	0x04,0x42,0x7b,0x01,0x40,0xa6,0x0a,0x9e,0x3f,0x4f,0x5f,0x36,0x16,0x19,0xc4,0x0c,
	0xd7,0xe7,0xcc,0x3e,0x41,0xfc,0xe8,0x34,0xaa,0x4d,0x72,0x82,0xa4,0x5c,0x82,0x57,
	0xc9,0x56,0xf9,0x6c,0xf1,0xd2,0xf3,0xdd,0x08,0xdd,0xbf,0x86,0x87,0xc7,0x17,0x99,
	0x4e,0x28,0x81,0x03,0xd3,0x30,0x14,0x3b,0x96,0x00,0x42,0x9e,0x58,0xc8,0x6d,0x80,
	0x20,0x5a,0xc0,0x0a,0x33,0x3a,0x0f,0x1d,0x98,0x7e,0x9b,0x8c,0xc2,0xe3,0x3e,0x59,
	0xac,0xb0,0xfd,0x2b,0xf7,0x9d,0xdb,0x7b,0xcc,0x09,0x81,0x50,0xb9,0xf6,0x10,0x02,
	0x2a,0x14,0xfa,0x2a,0xb3,0x78,0x17,0x22,0x3c,0x6d,0x5f,0x19,0xf1,0xfe,0x8e,0xdd,
	0x7d,0x56,0xcb,0x6b,0xef,0x31,0xb3,0x8b,0x84,0x7b,0x89,0xee,0x33,0x18,0x8c,0x93,
	0x24,0xcf,0x47,0x34,0x51,0x0c,0xbd,0x27,0x8b,0xe8,0xa6,0x6c,0x69,0x33,0x62,0x4b,
	0xfe,0x0a,0x99,0xbc,0x90,0x3e,0x61,0xa5,0xf3,0x53,0xf3,0x24,0x60,0xcc,0x27,0xa1,
	0x14,0x45,0xce,0xa3,0xa5,0x00,0x7e,0x41,0x59,0x03,0xac,0x88,0x6f,0x0c,0x8e,0xd6,
	0x36,0xcc,0x3e,0x93,0x28,0x13,0x2c,0xe4,0x83,0x66,0x04,0x96,0xd5,0x46,0x69,0x69,
	0x12,0xf1,0x4d,0xf8,0xe5,0xcd,0x4f,0x2e,0xd5,0xdc,0x88,0x52,0xf9,0x68,0xda,0xfa,
	0xdd,0x51,0x94,0x96,0x21,0x4d,0xc8,0x89,0xc4,0x0c,0x13,0x4b,0xd9,0xf9,0xa3,0x17,
	0xe6,0xfc,0xd1,0xdb,0x25,0xdf,0x1c,0x85,0x55,0x5d,0x07,0x40,0xf2,0x3e,0xe6,0x38,
	0x3b,0x7a,0xe9,0xc9,0x89,0x50,0x07,0xf3,0xae,0xc7,0x33,0x9f,0x60,0x96,0x05,0x25,
	0xef,0xb4,0x4b,0xa6,0x35,0x03,0x37,0x9a,0xc3,0x80,0x20,0x0d,0x76,0xbc,0xb6,0x50,
	0x7b,0xf5,0xb2,0x2c,0x9b,0x50,0x09,0x77,0x45,0xc5,0xcd,0xb9,0xc6,0xcb,0x79,0x5a,
	0x85,0x23,0xf8,0x70,0xa6,0x47,0xed,0x7e,0x29,0x31,0x21,0x1e,0x22,0xee,0x42,0xf3,
	0x03,0x0d,0x1c,0x06,0x19,0xb5,0xc8,0x10,0xfc,0xc7,0xc1,0xc7,0xc5,0xd2,0xf1,0xa2,
	0x93,0xa8,0xc7,0x95,0x22,0x7d,0x09,0x04,0x77,0xfb,0xab,0x8c,0x18,0x9b,0xc9,0xfa,
	0x58,0xb5,0x19,0x94,0xa1,0x24,0x17,0x96,0x97,0x1c,0x65,0x7a,0xdc,0x41,0xfb,0x4d,
	0x9a,0x22,0xdb,0xb6,0xb4,0x8f,0x3b,0x42,0x48,0x04,0xce,0xa2,0x31,0x07,0xd1,0x5c,
	0x54,0xda,0x8a,0x63,0xee,0x3d,0xb0,0x29,0xbf,0x11,0x1b,0xa9,0x2f,0x74,0x80,0x48,
	0x3b,0x91,0xe8,0x06,0xcd,0x2f,0x00,0x99,0xca,0x2c,0x4f,0xe7,0xdd,0x3f,0x8c,0xb5,
	0x05,0x4f,0x92,0x9f,0x45,0xe8,0x99,0xee,0x3d,0x04,0x44,0xc0,0xc0,0xf8,0x1a,0x52,
	0x0e,0xbc,0x83,0xf8,0x7f,0xc7,0xed,0x1c,0xfb,0xbf,0x62,0xd5,0xcd,0x0c,0x16,0x5a,
	0xd9,0x50,0xbe,0x6b,0x62,0x3c,0xfa,0xb4,0xd4,0x7f,0xd5,0x91,0x91,0x16,0x02,0xed,
	0xe5,0xf6,0x0c,0x57,0xed,0xeb,0xf1,0xc9,0x30,0x60,0x4b,0x32,0xa4,0x9c,0x2c,0x6d,
	0x3b,0xc8,0xcf,0xfb,0x87,0x62,0x8f,0xf9,0x39,0x62,0xc2,0x97,0xc9,0x77,0x3b,0x4c,
	0xce,0x6f,0x43,0x52,0xd9,0xe9,0x06,0x78,0x6f,0xa8,0xba,0xb0,0x2f,0x22,0x7a,0xb0,
	0x95,0x0b,0x98,0x01,0x61,0xaa,0x6c,0x1e,0x61,0x8c,0x46,0xe1,0xaa,0x94,0x1c,0xaf,
	0xdb,0x4c,0x7b,0x1c,0x0a,0xb9,0x5e,0x6f,0x1e,0xb8,0xb7,0xc8,0x4b,0xf3,0xb7,0x7a,
	0x76,0x7b,0x8a,0xc2,0x61,0xba,0xc3,0x96,0x57,0xde,0x46,0xef,0x11,0x32,0x82,0x92,
	0x7a,0x2c,0xc5,0xfb,0xd9,0x90,0xdc,0x0b,0xcc,0x55,0xba,0x04,0xef,0x75,0x89,0x91,
	0x1d,0x2a,0x2e,0x64,0x87,0x3f,0x60,0xc8,0xb1,0xd8,0xbe,0x89,0x63,0x03,0x0a,0x7d,
	0x00,0x2e,0x25,0xdb,0xf4,0x2e,0xa3,0xb8,0xfb,0x60,0x20,0xc5,0x30,0x06,0x9c,0x26,
	0xa7,0x5b,0xbf,0x62,0x1c,0x6f,0x87,0x0b,0xcb,0x60,0x24,0xe0,0x6e,0x71,0x52,0x2a,
	0x0d,0x2a,0x79,0x0f,0x24,0xa4,0xe9,0x51,0x2f,0xc7,0x6c,0xdb,0xc6,0xd0,0x90,0x46,
	0xd9,0x33,0x17,0x7e,0xae,0xc3,0xf7,0x57,0xd7,0x63,0x19,0x6f,0x25,0xb0,0x69,0x2e,
	0x60,0x54,0xbb,0xa2,0xf3,0xea,0xbc,0xc9,0xd5,0x64,0x06,0xb4,0x88,0xeb,0xc0,0x34,
	0x89,0xef,0x99,0xcc,0x9d,0x9f,0xc4,0x75,0x07,0x98,0x8c,0x91,0x2c,0x2b,0x16,0x80,
	0xb1,0x4a,0x93,0xaf,0xdf,0x9c,0x74,0x75,0x73,0xd2,0x37,0xce,0x2e,0x3b,0xf1,0xb3,
	0xd4,0xa5,0xd7,0xd5,0x81,0xc1,0xe0,0xad,0x47,0xca,0x40,0x40,0xaa,0x9b,0x89,0xf7,
	0x60,0x97,0xd4,0x09,0xef,0x05,0x49,0x79,0x1a,0xb0,0xf3,0x22,0xae,0xe4,0x23,0xa6,
	0x2a,0x4b,0x40,0xe7,0x94,0x8c,0xad,0x46,0x20,0x1a,0x6a,0xcd,0x3c,0xb7,0x4a,0xd5,
	0x44,0x29,0x5d,0xe1,0xb2,0x4d,0xf3,0x7f,0x51,0x65,0xee,0x53,0xca,0xc9,0xb0,0x22,
	0x09,0x93,0xf1,0xa4,0x5e,0xd5,0xaf,0xe9,0x48,0xa3,0x82,0x51,0x39,0x91,0x26,0x5d,
	0x0b,0xc5,0xde,0xbd,0x80,0x1a,0xcb,0xb5,0x45,0xfc,0xc1,0x69,0x3a,0xba,0x5f,0xb5,
	0xfd,0x1c,0x53,0x42,0x2d,0xb6,0xe3,0xf2,0x61,0x1b,0x76,0x83,0x67,0x42,0xee,0xfe,
	0xf7,0x37,0x76,0x88,0x75,0x54,0x54,0x4e,0xe6,0x2c,0x2d,0x25,0x82,0x77,0xeb,0xe6,
	0xb4,0x1f,0xc4,0xe3,0x0b,0x64,0x13,0xda,0x35,0x39,0x71,0xf9,0x00,0xb0,0x45,0x99,
	0xd2,0x62,0xc1,0x41,0x4b,0x7c,0xb1,0x89,0x9e,0x4b,0x2f,0x54,0x36,0xb2,0xf3,0x74,
	0x47,0xc1,0x8b,0x14,0x18,0xe5,0xaa,0x3b,0x5d,0x58,0x62,0xf3,0x51,0x5c,0x46,0xf6,
	0xa8,0x4f,0x65,0x3f,0x39,0xf8,0x23,0xae,0x2d,0x3c,0x35,0x95,0xb8,0x93,0x41,0x2f,
	0x28,0x82,0xae,0xb5,0xc4,0x72,0x5f,0x41,0xa1,0xa6,0x49,0x49,0x15,0x24,0xcb,0x7a,
	0x91,0xb8,0xa2,0xa6,0x12,0xef,0x3d,0xbd,0xee,0x98,0xbe,0x38,0xf0,0xc4,0xa8,0xaf,
	0x77,0x95,0x8e,0xd9,0xf3,0x3b,0xc4,0x17,0x81,0x6c,0xa7,0xc6,0x08,0xd9,0x0c,0x1a,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x39,0x9d,0xa5,0x68,0x07,0x6e,0xa0,0xae,0xe4,0x84,0x54,0x8e,0x4a,0x13,0x50,0x3c,
	0xf7,0x37,0x91,0xda,0x4f,0x76,0xc5,0xe1,0xd1,0xfa,0x03,0xb9,0x00,0xf1,0x8f,0xb3,
	0xc2,0x21,0x65,0x15,0xae,0x00,0x7b,0x5c,0x31,0x06,0xc4,0xf6,0x43,0x3f,0x2a,0x66,
	0x2a,0x3a,0x76,0xf5,0x0a,0x24,0x8b,0xc3,0xc6,0x44,0x42,0x78,0x9d,0x06,0x84,0x1f,
	0x6f,0xdb,0x84,0xdd,0xfe,0xba,0xc8,0xb2,0x2e,0x0b,0x69,0xc6,0x89,0x17,0x11,0xcb,
	0xbb,0x74,0x99,0xa7,0xb4,0x71,0x44,0xdf,0xf2,0x9c,0xe1,0xd3,0xb3,0x09,0x17,0x64,
	0x04,0x05,0x48,0x1a,0xe1,0x1d,0xd2,0xdd,0x82,0x65,0xea,0x5d,0x5e,0x75,0xbb,0x5b,
	0x0f,0x60,0xd3,0xbf,0x8c,0xd3,0x1a,0x37,0xd8,0x15,0x60,0x85,0xb6,0xe5,0xd0,0xf3,
	0xb0,0xac,0x8f,0x6f,0x9d,0x66,0xa7,0x21,0xa5,0x71,0x4b,0x37,0x02,0x8b,0x88,0xf8,
	0x64,0x9a,0xed,0x6a,0xc0,0x85,0xd3,0x06,0xed,0xd8,0xcf,0xbc,0xaf,0x2b,0x41,0x03,
	0x23,0xa1,0x65,0x65,0xf0,0xfa,0x4c,0xd6,0x55,0xe9,0x0a,0xf3,0xd0,0x58,0xa7,0xdf,
	0xc9,0x03,0xf0,0x27,0xaa,0x06,0x53,0x75,0x90,0xb4,0x5f,0xf4,0x0b,0x1c,0x83,0x95,
	0x0e,0x23,0xae,0x42,0x6a,0x4e,0xef,0x34,0x7e,0x4f,0xac,0x92,0xfe,0x8a,0xd7,0xa0,
	0x20,0x01,0xe2,0x37,0x86,0x57,0x2b,0x92,0x8f,0x68,0x04,0x7c,0x3f,0xc8,0xf4,0xdd,
	0x6c,0xd6,0xa2,0x05,0xb9,0x30,0xde,0x33,0x78,0x5a,0x20,0x62,0xb8,0x7e,0x1c,0x0e,
	0x8a,0xaa,0x1f,0xed,0x30,0xed,0xda,0x91,0xa4,0x36,0x9a,0x72,0xdd,0x3e,0xe4,0xa7,
	0x31,0xcd,0x4e,0x3d,0xf5,0xa8,0x83,0x2e,0xd2,0x0d,0x40,0x0c,0x6a,0xaa,0xb0,0x52,
	0xc4,0x94,0x32,0x8a,0xbc,0x62,0x47,0xbb,0x53,0xa8,0x16,0xe0,0x4a,0x57,0x46,0x14,
	0xb2,0x5a,0x6f,0xdc,0x96,0xc8,0x40,0xa7,0x12,0x29,0xd8,0x67,0x91,0xb9,0x57,0xe7,
	0x50,0x79,0x7d,0x7b,0x5d,0x63,0xbf,0x80,0x92,0x63,0xce,0xf8,0x92,0x66,0x6d,0x37,
	0x7c,0x97,0x76,0xd6,0x01,0x29,0x7a,0xec,0x2d,0x83,0xcd,0x15,0x64,0x76,0x91,0x3a,
	0xb4,0x02,0xbc,0xa1,0xa8,0xce,0x7b,0xe2,0x46,0x83,0xe2,0x46,0x80,0x9d,0xf7,0xc3,
	0x1d,0x01,0x6b,0x88,0x8f,0x32,0xaf,0x77,0x14,0x73,0xaa,0x9e,0x4a,0x40,0xdf,0x7e,
	0xb2,0x62,0x0b,0x49,0x18,0xf5,0x5c,0x52,0xea,0x76,0xae,0x27,0x4f,0x0f,0xaf,0x20,
	0x21,0x33,0x7e,0x6a,0xdb,0x78,0x4b,0xf9,0x80,0x43,0xf1,0x3f,0x9c,0xf0,0x55,0x63,
	0x50,0x16,0xe4,0x1e,0xc8,0xbf,0xc6,0x06,0x6a,0x01,0x44,0x1c,0x7b,0xec,0x5f,0xca,
	0x36,0x64,0x02,0x38,0xaa,0xc9,0x86,0x4c,0x4d,0xcd,0x5b,0x16,0x51,0x93,0x61,0xe5,
	0x98,0x07,0x02,0x27,0x4b,0xb2,0x1f,0x17,0xd3,0x78,0x54,0x16,0x7b,0xa0,0x24,0xa5,
	0xae,0x44,0xf5,0x09,0xc5,0x07,0x6c,0x94,0x8d,0xb0,0xe6,0xab,0xcf,0xe9,0xfa,0x9e,
	0x12,0x21,0x99,0xa2,0x19,0x11,0x05,0x5e,0xca,0xc5,0xe2,0xfc,0x40,0xa2,0xb1,0x19,
	0x65,0xa2,0xe5,0x81,0xa5,0x9f,0x16,0x6a,0xf5,0xd4,0x6c,0x5c,0xdf,0x35,0x7a,0x69,
	0xd7,0xc4,0x96,0x1a,0x90,0x2d,0xc8,0x60,0x87,0xab,0x66,0x00,0xde,0x1b,0x94,0xbd,
	0xf3,0xa9,0xfd,0x38,0x52,0x81,0x91,0xa5,0x97,0xfd,0xbf,0xcf,0x83,0x13,0x39,0xb3,
	0xad,0xdf,0x18,0xe5,0x8f,0xfe,0xe0,0x91,0xa2,0xdc,0xd2,0xe4,0xc2,0x79,0x21,0x1a,
	0x41,0xa8,0xc3,0x12,0xf4,0x60,0x52,0xd1,0xb4,0xa1,0x5e,0x2a,0x3f,0x83,0xce,0x6d,
	0xcb,0x08,0xba,0x8f,0x45,0xfe,0x68,0x26,0x62,0x47,0x0e,0xe8,0xbe,0x17,0x3f,0x35,
	0xcf,0xd3,0xc6,0x74,0x26,0x81,0x05,0x54,0xa8,0x42,0xe2,0x2a,0xdf,0x2c,0x99,0x25,
	0x7b,0xc8,0x6a,0x16,0x26,0x2d,0xc8,0xa2,0x71,0x71,0x79,0xe4,0xb4,0x5b,0x03,0xdf,
	0x8e,0x10,0xc5,0xe3,0x45,0xae,0x4b,0xac,0xaa,0xc8,0x93,0xcf,0x66,0x99,0x6e,0xfa,
	0xf1,0x93,0x68,0xb4,0x3a,0xd0,0x66,0xbe,0x76,0x07,0xe5,0x54,0xea,0xf9,0x9b,0x63,
	0x1c,0xa7,0x70,0x7b,0x8b,0x1c,0xd9,0xc6,0x81,0xaf,0xea,0xd3,0x93,0x18,0x68,0x3f,
	0x92,0x94,0x7e,0xc1,0x41,0x24,0x7f,0x39,0x82,0x20,0x76,0x45,0x33,0xaa,0x92,0x77,
	0x6d,0x7d,0xb2,0x10,0xe8,0xf9,0xd9,0x7a,0x78,0xad,0x2b,0xdc,0x3a,0x67,0x78,0x4c,
	0xf9,0xc5,0xad,0x47,0x49,0xee,0xd6,0xb3,0x6a,0xb5,0x89,0x71,0x98,0x35,0x8f,0x98,
	0x0c,0x7e,0x95,0x53,0xe2,0x2b,0x5a,0x7c,0x94,0xcb,0xa3,0x5a,0xe2,0x19,0x1e,0x42,
	0x9f,0xfb,0x1b,0xd8,0xd5,0x3a,0xbd,0x4d,0x40,0x05,0xe7,0xaf,0xb9,0xe2,0x47,0x7b,
	0x15,0x5a,0x4d,0xc5,0xab,0x5f,0x7a,0xb7,0xe3,0x3e,0x87,0xf5,0x06,0x46,0x0f,0x59,
	0x28,0x2f,0x2a,0x0b,0xb5,0xe5,0x23,0xac,0x5a,0x12,0x75,0x1b,0x39,0xf8,0x7c,0x63,
	0x21,0x44,0x5a,0x91,0x5e,0xbb,0x86,0x7a,0x88,0x8e,0xf2,0x14,0x1d,0xc8,0x5f,0xdc,
	0x0c,0xcb,0xd9,0x43,0x52,0x11,0xaf,0x16,0xcb,0x98,0x48,0xf2,0xf0,0x90,0x90,0x9c,
	0xd4,0xf3,0x3e,0x03,0xdf,0xb5,0x5d,0x1d,0x67,0x90,0xde,0x2f,0x7a,0x40,0x7f,0xfd,
	0x1d,0x11,0x6b,0x27,0x09,0xfc,0x99,0x73,0xd6,0x48,0x74,0x01,0xdc,0xcd,0xe9,0xfd,
	0xef,0xe7,0x9f,0x9d,0x63,0x7c,0x47,0x27,0xe9,0xed,0xc6,0x45,0xac,0x01,0xce,0x9c,
	0x05,0x9b,0xed,0x86,0xeb,0x17,0xba,0xad,0xa6,0x08,0xdf,0xcb,0xb9,0xbc,0xa0,0xcc,
	0x11,0xf7,0xc2,0x2d,0x8d,0xf3,0xe9,0x7f,0x8d,0xb7,0xdc,0x64,0xf9,0xa4,0x8b,0x73,
	0x39,0x70,0x02,0xf8,0x75,0xdf,0x53,0xd8,0x25,0xae,0x10,0x36,0x74,0x0d,0x20,0x48,
	0xc4,0x70,0x05,0x8f,0xde,0x3b,0x3a,0xc8,0x06,0xbd,0x80,0x75,0xdd,0xf1,0x21,0x6f,
	0x54,0x77,0xac,0xa8,0xa8,0xf8,0xc5,0xdd,0x5c,0x96,0x1e,0xc2,0xf1,0xcf,0x4e,0x76,
	0xdf,0x89,0x9b,0xe2,0xd5,0xd9,0xd5,0xea,0x5e,0xdf,0x42,0x40,0x85,0xd0,0x13,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x49,0xf6,0xc2,0xe9,0xe4,0x5b,0x38,0x85,0x15,0x08,0xaa,0xe2,0x33,0xb7,0x8c,0x1e,
	0x19,0x96,0xe1,0xa3,0x94,0xe6,0xd0,0xa0,0xd7,0x98,0x48,0x7e,0x97,0xbf,0xbd,0x26,
	0xa0,0xe1,0xbd,0x84,0xf8,0x8c,0x0d,0x08,0xb8,0xa7,0x75,0x0c,0x54,0xe8,0xe6,0xfd,
	0x15,0xc9,0x1e,0xf3,0x78,0x35,0x7e,0x41,0x4f,0x90,0x3e,0x64,0x98,0x32,0x32,0xa6,
	0x8b,0x32,0x60,0x26,0x4a,0xb2,0xf1,0xe8,0x23,0x43,0x80,0x8d,0x83,0xb2,0xa0,0xe3,
	0x2e,0xcd,0x9e,0x8d,0xc7,0x3d,0x6c,0x2a,0x04,0x0c,0xdb,0x4e,0xde,0x45,0x5a,0xb8,
	0x89,0x9c,0x0e,0x96,0x01,0x98,0xf8,0x60,0xa8,0x12,0x01,0x51,0x28,0xe0,0xf7,0x95,
	0xf5,0x2c,0x01,0xac,0x22,0x44,0x68,0x30,0x8b,0x96,0xdb,0xa8,0xf5,0xf3,0xc3,0x59,
	0x65,0xba,0x6f,0x5b,0x17,0xad,0xa2,0x6e,0x1c,0x87,0xe1,0x98,0x28,0x41,0x58,0x93,
	0xe9,0xa4,0xd3,0x19,0xb3,0x64,0xb6,0xb8,0xf9,0x4b,0x3e,0xaf,0x69,0x72,0x1b,0x23,
	0xbd,0x81,0x58,0xca,0x81,0x6a,0x81,0xee,0x6b,0x6f,0xe4,0x4d,0x91,0xc1,0x69,0xe5,
	0x18,0x68,0x46,0x56,0xa4,0x74,0x84,0x37,0x63,0xd9,0x59,0x65,0x88,0x6c,0x61,0x3b,
	0xb7,0xc1,0xf6,0x79,0x90,0xaf,0x8b,0xf0,0xd3,0x36,0x2d,0x6b,0x47,0xdc,0x23,0x34,
	0xe2,0xb1,0xec,0x20,0xed,0x40,0xd8,0x2d,0x2d,0x75,0x53,0x84,0x20,0xf6,0x11,0xa0,
	0x6f,0x0e,0x71,0xc6,0xd6,0x2e,0xa8,0x40,0x81,0x75,0x6c,0x45,0xee,0x66,0x82,0xd4,
	0xf5,0x2d,0x55,0xb2,0xc0,0x7f,0xee,0x5e,0xa2,0xb0,0x90,0x3b,0xc0,0x9b,0x66,0x8a,
	0x88,0xca,0x8c,0x68,0xda,0xd7,0x9e,0xd0,0x3e,0x9e,0x52,0xa5,0x11,0xe7,0x1a,0x9c,
	0x25,0xce,0xb4,0x2c,0x4a,0x64,0x05,0x6f,0x1b,0xcd,0xa1,0xf2,0xea,0x2d,0x51,0x5f,
	0xf3,0xa4,0x40,0xc2,0x16,0x36,0xe2,0x0a,0x7b,0xa9,0xb4,0x84,0x42,0xc8,0xbf,0xb0,
	0x21,0x76,0x51,0x79,0x17,0x98,0xd9,0x5c,0x0d,0xe9,0xa2,0x1c,0xb4,0x08,0x4a,0x15,
	0xaf,0xf6,0xe8,0x61,0xfb,0x59,0xf9,0xdb,0xcf,0x69,0x33,0x91,0x24,0xc5,0xe7,0x29,
	0xed,0x56,0xe7,0xe3,0xdc,0xdc,0x74,0x57,0x62,0xac,0xbe,0xab,0x27,0xab,0x7f,0x5a,
	0xa3,0x2a,0xad,0x90,0x21,0x04,0x9a,0x0a,0x17,0x2a,0xc0,0xe0,0x44,0xfa,0x3a,0xb5,
	0x8c,0x47,0x81,0xe6,0x0b,0x48,0x32,0x6e,0x45,0x06,0x9c,0x90,0x8f,0x7e,0xea,0xd1,
	0x35,0xf4,0xd1,0x2d,0x1f,0x6c,0x5a,0xe7,0x9d,0x70,0xe1,0xe0,0xaf,0x54,0x47,0x7b,
	0x94,0x17,0xde,0xa6,0xf5,0x68,0xcc,0x1f,0x93,0x81,0x38,0x9b,0xa7,0x5b,0x75,0x9d,
	0xf3,0xa1,0x0d,0x82,0x49,0x10,0x81,0x87,0x9a,0xc7,0x77,0x28,0xe0,0xd2,0x56,0x50,
	0x3b,0xa9,0x8b,0xfd,0xc2,0x9f,0x29,0xa7,0x5b,0xbc,0xcd,0x80,0xe7,0x35,0xd8,0x66,
	0xef,0xdf,0xbf,0x60,0x81,0xae,0x85,0xc2,0x6d,0x40,0x13,0x50,0x9e,0xec,0x8c,0x73,
	0xc9,0x2a,0x00,0x25,0x29,0x69,0x6f,0xfc,0x03,0x79,0xd4,0xb1,0xbf,0xcb,0xd1,0xb4,
	0xa2,0xe2,0xdc,0xdf,0x73,0x91,0x14,0x6d,0xda,0x7c,0xf5,0xbb,0x2d,0x7c,0xc7,0x74,
	0xe4,0x34,0x31,0xab,0x8b,0x79,0x34,0x94,0xbd,0x2f,0x32,0x30,0x2c,0x85,0x0c,0xa8,
	0x7f,0x40,0x35,0x5d,0xa8,0xac,0x20,0xe8,0x90,0xa9,0xfe,0xca,0x8c,0xc6,0x4b,0xb0,
	0xb8,0x91,0x70,0x7e,0x18,0x5e,0xd0,0x17,0xef,0x16,0xad,0x28,0xdb,0x79,0xdb,0x7b,
	0x12,0x46,0x9f,0xfc,0x47,0x3d,0xe3,0xa5,0x4c,0x26,0x0d,0x03,0xef,0x7c,0x95,0x94,
	0xd2,0x73,0x2d,0x64,0x01,0x66,0x84,0xbc,0xfb,0x29,0x18,0x41,0x0c,0x31,0x1b,0xa2,
	0x6f,0x53,0xa9,0x64,0xb1,0xd0,0x78,0x8a,0x56,0x58,0xae,0x3b,0xa2,0xfe,0xd8,0x46,
	0xb6,0xcc,0xda,0xfc,0xdb,0x96,0x50,0x41,0xed,0xa3,0xa2,0x44,0xe7,0x4c,0x2d,0xd2,
	0xe1,0x38,0x7b,0x07,0x09,0x0c,0x50,0xee,0x7c,0xa4,0xd3,0xde,0x95,0xa2,0xf9,0xbf,
	0xf2,0x05,0x47,0xac,0x44,0x1d,0xfd,0x3d,0x60,0x60,0xde,0x6f,0x70,0xed,0x47,0xdd,
	0x76,0xf5,0x72,0xd9,0x94,0xc7,0xf3,0x84,0x8b,0xcd,0x41,0xaf,0x6c,0xe2,0xeb,0x62,
	0xfa,0x8a,0x88,0x45,0x6d,0x0c,0x82,0x68,0x69,0x09,0x4e,0x84,0x76,0x8a,0x9c,0x02,
	0x60,0x8b,0xa2,0x6f,0x81,0xfd,0x30,0x2c,0x7c,0xe6,0x24,0x46,0x33,0x7f,0x63,0x09,
	0x73,0x74,0xbb,0xf9,0x29,0x14,0x67,0x00,0x77,0xde,0x3a,0x0c,0x82,0xdd,0xdb,0x09,
	0xf4,0x6d,0x94,0x19,0x30,0xd6,0x61,0xbe,0x56,0xaa,0x4c,0x87,0x00,0x2b,0x8f,0x12,
	0xe0,0x10,0xa2,0xa2,0x98,0xe2,0xc7,0x3a,0x90,0xb2,0x6c,0xf5,0xdc,0x60,0x10,0xc5,
	0x6e,0xfd,0x6c,0x59,0x62,0xfb,0x3c,0x91,0x0b,0x96,0x40,0xd2,0xce,0x9d,0x0f,0xdf,
	0xa6,0x18,0xad,0x1e,0xbe,0xdc,0xdf,0x6d,0x5a,0x32,0x15,0x78,0xd5,0x79,0xeb,0x4f,
	0x11,0xbe,0xc3,0x6d,0x42,0x85,0xd3,0x39,0x09,0xaf,0xa8,0xd4,0x8f,0xab,0x8a,0x7b,
	0x8c,0xdd,0xcf,0xa3,0x52,0x9f,0x99,0x2c,0x94,0x0f,0xbc,0xc6,0x23,0xe7,0xdb,0x0e,
	0xa8,0xa7,0x4e,0xa9,0x3c,0xe4,0x86,0xc5,0x72,0x22,0x22,0x7e,0x09,0xb3,0xde,0xa3,
	0xda,0xc7,0x13,0xfc,0x51,0x0a,0xba,0x52,0xd6,0x75,0x5b,0x51,0x88,0xbd,0x92,0x3b,
	0x8e,0xee,0x9a,0x6e,0xdb,0x56,0xf7,0x57,0x0b,0x62,0x0a,0x21,0x4b,0xdc,0xcd,0x7a,
	0x84,0x3f,0x0a,0x35,0x69,0x0c,0xa1,0xbe,0x94,0x47,0xe8,0x4b,0x2f,0x4b,0x68,0x04,
	0x82,0x8c,0xbd,0x9d,0xfe,0xc6,0xf8,0x83,0x40,0xd5,0x6f,0x98,0x74,0x3b,0x50,0xaa,
	0x44,0xba,0xcc,0x9c,0xd0,0x66,0x42,0xbb,0xba,0xd6,0x96,0x43,0x35,0x4c,0x2c,0x5d,
	0x84,0x7f,0xc2,0xa4,0x87,0xad,0x5d,0x49,0xa9,0x81,0x6d,0x4b,0x73,0xb2,0xee,0x2f,
	0x03,0x7a,0xdc,0x96,0x7a,0x43,0x78,0x5f,0xc0,0x6e,0x8f,0xe3,0xd6,0x9d,0xe3,0x05,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x23,0xee,0x4b,0x1e,0x9c,0x63,0xa9,0x74,0x84,0x21,0xd7,0xda,0xd8,0xe7,0x5f,0x04,
	0x2d,0xf8,0x36,0x29,0x5e,0xe9,0x66,0x39,0x6b,0x16,0xbd,0x49,0xae,0x3f,0x85,0x64,
	0x44,0x88,0xf3,0xe3,0xdb,0xd0,0x3b,0x02,0xac,0x09,0xd1,0x34,0xec,0xf9,0xdb,0xc3,
	0x1a,0x82,0x7b,0xe1,0xcd,0x09,0xf9,0x9d,0x40,0x64,0xae,0x79,0x39,0x57,0xe5,0x93,
	0xd0,0x62,0xaf,0x3d,0xb3,0x73,0xa2,0xba,0xa6,0xde,0x67,0x47,0xa5,0xa3,0x00,0x48,
	0x16,0xdb,0xc2,0xfa,0xbe,0xb3,0x88,0xa2,0xd8,0x7d,0x05,0x65,0x81,0x4a,0xbf,0x31,
	0xa2,0x54,0xc2,0x8c,0xf0,0x77,0x07,0x22,0xd3,0x33,0xd6,0xc5,0xcf,0x0b,0x31,0xc9,
	0x82,0xc8,0xaf,0x1e,0x90,0x40,0x25,0x3f,0x6f,0x28,0xe2,0xb3,0x7c,0x6d,0x7e,0xe6,
	0x97,0x5a,0x7b,0x64,0x25,0xf5,0xcc,0x3f,0x4b,0x42,0xb6,0x99,0xa1,0xff,0xed,0xe9,
	0x22,0x10,0xf3,0xd2,0xe2,0x90,0x15,0x22,0xa7,0x90,0xb4,0x50,0x5b,0x85,0x53,0x91,
	0x4d,0x77,0x36,0x2f,0x7e,0xc0,0x8f,0xa2,0x60,0xd1,0xf6,0x4c,0xe0,0x91,0x17,0x1e,
	0x17,0xb3,0x4b,0x76,0x25,0x73,0xcc,0xba,0x8d,0x17,0x33,0xdd,0x26,0x59,0x76,0x9a,
	0x49,0x82,0x6e,0xd8,0xb9,0xd4,0xbc,0x9d,0xff,0xdb,0xe2,0x9e,0x7e,0x1b,0x62,0xe8,
	0x60,0x58,0x30,0x19,0xdd,0x51,0x9d,0x02,0x84,0x8e,0xa2,0x8a,0xbe,0x76,0x9d,0xf8,
	0x5b,0x89,0x05,0xc1,0xbe,0xe6,0x81,0x33,0x62,0x9b,0x60,0x52,0x86,0xc7,0xad,0x3e,
	0xf9,0xa7,0xe9,0x90,0xdf,0x41,0x87,0xb7,0xa5,0xdc,0x10,0xaa,0x51,0xfb,0xad,0xdd,
	0x58,0x28,0x0c,0xaa,0x69,0x2c,0xb1,0x42,0x83,0xc5,0x36,0x25,0x90,0x61,0xcd,0x9f,
	0xd7,0x6a,0xd1,0x0d,0x3b,0x6b,0xfc,0x2f,0xdf,0x81,0xe7,0x2c,0x12,0x3a,0xd9,0x12,
	0xb3,0xd2,0x06,0x1e,0x65,0x2c,0x1d,0x46,0x9e,0xed,0x06,0x4a,0x5b,0x5f,0x94,0x41,
	0x52,0xaf,0xc2,0x81,0x69,0xca,0x15,0xd8,0x67,0xea,0x06,0x58,0x5d,0x2e,0xfb,0xb8,
	0xe8,0xd4,0x9b,0x9f,0xc4,0x4f,0xe2,0xc3,0x9b,0x20,0xec,0xdc,0xf1,0x6e,0x55,0x4a,
	0x2f,0x9d,0x5c,0x97,0xbb,0x55,0x2f,0x25,0x61,0x46,0x76,0x3c,0x32,0x82,0xfe,0x18,
	0xc1,0xe5,0x40,0x72,0xa5,0x14,0x78,0x52,0x56,0xaf,0xff,0xe4,0x8e,0x85,0xe5,0x5f,
	0x20,0x5e,0x7e,0x7d,0x3e,0x72,0x53,0xd4,0xb6,0x18,0x3e,0x1e,0xbe,0x94,0x2b,0x57,
	0x20,0x05,0x24,0xbb,0x5d,0x86,0xc4,0xb4,0x74,0x17,0xb6,0x74,0x71,0xa3,0x64,0x40,
	0x35,0xcd,0x20,0xa2,0xfb,0x0f,0xba,0x20,0x0c,0x15,0x7e,0x84,0xba,0x29,0xe9,0xca,
	0x38,0x90,0x88,0xb6,0x8c,0xe4,0x1a,0x0f,0x07,0x0c,0xc5,0x30,0xf3,0x0a,0x31,0xfc,
	0x0d,0xce,0x4c,0xbf,0x28,0x0e,0x91,0xb5,0x7d,0x2a,0x51,0x23,0x71,0x67,0x0a,0x02,
	0x42,0x12,0x8e,0xbf,0x7b,0x9c,0xa6,0x5a,0x78,0x94,0xc7,0x03,0xda,0xfa,0x34,0x27,
	0xaa,0x8b,0x39,0xbd,0xac,0x86,0xd9,0x9a,0x8c,0xc0,0xa4,0xc4,0x18,0x24,0x32,0x80,
	0x35,0xf0,0xf7,0xa9,0x2d,0xfe,0x08,0x69,0xba,0x09,0xab,0x8e,0x30,0x77,0x51,0x73,
	0x17,0x4f,0x13,0xb2,0x63,0x53,0xe6,0x0b,0x50,0x8e,0x21,0x16,0x63,0xa9,0x44,0xa9,
	0x68,0xc3,0x7d,0x74,0xaa,0xe4,0x40,0x32,0xb7,0x93,0x0f,0x8f,0x19,0xa6,0x73,0x9a,
	0x18,0x7a,0xc0,0x79,0x22,0xad,0xfa,0xdc,0x55,0x90,0x7b,0x90,0x9b,0x3b,0x47,0xd6,
	0x70,0x31,0x2b,0x9c,0x9b,0xc1,0xeb,0x04,0xd5,0xeb,0x17,0xbe,0x53,0x09,0x73,0xe3,
	0xe8,0x4e,0x5f,0x53,0x02,0x7b,0x30,0xaa,0x3e,0xad,0xcc,0x9a,0x42,0xb3,0x44,0x1a,
	0x73,0x80,0xb2,0x52,0x9b,0xc9,0x6c,0xf2,0x46,0xe5,0xd2,0xa9,0x2c,0x0e,0x51,0x73,
	0x85,0x7c,0x74,0x1d,0x22,0x21,0x83,0xf7,0x9f,0xb7,0x01,0x89,0x98,0xa0,0x32,0xfd,
	0x0e,0x41,0xf1,0xe3,0xa1,0x4d,0xd7,0x81,0x05,0x8d,0x66,0x97,0xd8,0x78,0xec,0x0c,
	0x5b,0x71,0xf1,0xe1,0x5a,0x19,0x84,0x60,0x63,0x87,0x3a,0xc3,0x00,0x17,0x02,0x67,
	0xd5,0x8f,0x3f,0x26,0x42,0x28,0x99,0xdc,0x32,0x7a,0xa0,0xb2,0x14,0x38,0x36,0x4c,
	0x6c,0x09,0x60,0xa7,0xb9,0x03,0xf2,0xf7,0xb4,0x61,0x9f,0x46,0x07,0xab,0x35,0xe8,
	0xbc,0xcb,0x58,0xc3,0x47,0xed,0x5c,0x7f,0xec,0x66,0x05,0x6d,0x57,0x04,0x83,0xa1,
	0xb4,0xfe,0xf9,0x9b,0x6e,0x69,0x52,0x2c,0x3e,0x81,0xe8,0xa6,0x20,0xff,0x5f,0xe0,
	0x16,0xfc,0x87,0x4c,0x2b,0x09,0xde,0x9c,0xda,0xd0,0x76,0x1f,0xf1,0x11,0xb5,0xae,
	0xc3,0x5a,0x4b,0xfa,0xbc,0x15,0xdd,0xa2,0x53,0x57,0x29,0x7b,0xcc,0x81,0x81,0x5f,
	0xee,0x22,0xa4,0x7e,0xd9,0xd3,0x97,0x8a,0x18,0xda,0x28,0xaa,0xf0,0xc1,0x19,0x07,
	0x56,0xb9,0xba,0x8b,0x50,0x8f,0xb0,0x83,0xde,0x9e,0xb0,0x03,0x32,0x6e,0xdd,0xb5,
	0x1e,0x06,0x71,0x42,0xa5,0xca,0x53,0xab,0x2d,0x13,0x90,0x43,0x13,0x66,0x36,0xea,
	0xbf,0xc5,0x0b,0xec,0x3e,0x26,0x64,0x55,0x1e,0xa5,0xca,0x4f,0xb5,0x0d,0x3d,0x8b,
	0xd1,0x82,0xd3,0x85,0x1a,0x23,0xbb,0xbb,0xb2,0x17,0x1c,0x30,0x62,0x71,0xd4,0x73,
	0x30,0x93,0xe2,0xb7,0x1c,0x9a,0x01,0xf3,0x99,0x20,0x85,0xe9,0x17,0x68,0x07,0xc0,
	0xba,0x3b,0x61,0xcf,0x8a,0x3f,0xfa,0x1d,0xf2,0x5a,0x56,0xda,0xfe,0x6f,0x88,0xdf,
	0x33,0x30,0x56,0x61,0x85,0xdb,0xdf,0xb3,0x81,0x1b,0x1f,0x57,0xb3,0x0b,0x67,0x3e,
	0xdb,0x4a,0x45,0xed,0x96,0x3a,0xf5,0x9b,0xa5,0x73,0xd5,0xae,0x48,0xc9,0x1b,0x60,
	0xaf,0x9c,0xd1,0xe8,0x8e,0xb5,0xda,0x97,0xe1,0x96,0x5f,0x96,0x22,0x3e,0xa8,0x94,
	0xf1,0x1b,0x77,0x04,0x88,0xe2,0x54,0x1a,0x28,0x0f,0xbd,0x34,0xb8,0x0e,0xe0,0xcd,
	0x91,0xff,0xd6,0x1f,0x16,0x4f,0x56,0xd8,0xb4,0x36,0xd6,0xdd,0x34,0x61,0x75,0xaa,
	0x59,0x64,0x44,0x97,0x12,0x8e,0x24,0x5b,0xca,0x7b,0xd6,0x76,0x8f,0x4f,0x86,0x24,
	0x33,0xcb,0xf7,0xd4,0xcb,0x01,0xe1,0x1b,0x0b,0x88,0xbd,0xda,0xab,0x94,0x90,0xe7,
	0xf9,0x45,0xe6,0x8b,0xbc,0x73,0x3e,0x0b,0x78,0x04,0x5f,0xd8,0x88,0x78,0xfa,0x48,
	0xf8,0xb9,0xb6,0x03,0x44,0x1a,0x01,0xa9,0x8b,0xd5,0xd5,0xb5,0x5f,0x83,0xfb,0xa8,
	0xd1,0x44,0x45,0xce,0x11,0xfd,0x19,0x7d,0x53,0xa9,0x14,0x31,0x39,0xf5,0x40,0xa1,
	0x94,0x72,0xb7,0xcc,0x60,0xdb,0xa8,0x3a,0xba,0xaa,0x8e,0xc2,0x3b,0xe4,0x4a,0x51,
	0x70,0x0b,0x6c,0x25,0xe5,0x39,0x20,0x68,0xbe,0x18,0x97,0x60,0x6c,0xc4,0x4d,0xa7,
	0x19,0x6e,0x21,0x1b,0x57,0x1a,0xe8,0x19,0xed,0x63,0x88,0x31,0x62,0x1a,0x81,0xbd,
	0x58,0xa9,0xf4,0xfa,0xea,0x07,0xb9,0xc9,0xbf,0x6b,0x9c,0x0f,0x8a,0xbb,0x55,0x08,
	0xea,0x24,0x37,0x1c,0xbd,0x1b,0x07,0xea,0x14,0xda,0x9e,0x41,0xe0,0xb4,0xc6,0x88,
	0x0b,0xa4,0x2c,0xa8,0x54,0x37,0x5c,0x52,0xa8,0xce,0x84,0xe3,0xe4,0xa0,0xd6,0x69,
	0xaa,0xe4,0x0e,0xfc,0xe6,0x7f,0x76,0xbc,0x31,0x9d,0x8c,0x82,0x87,0xb2,0x60,0x1c,
	0x56,0x86,0xc0,0x9a,0x3d,0xc4,0xec,0xbf,0x47,0xed,0x17,0x69,0xee,0x92,0xd0,0xcb,
	0x3a,0xa2,0xb7,0x38,0xb3,0x6e,0x23,0x51,0xcf,0x6a,0x7e,0x46,0x5f,0xa0,0xce,0x9a,
	0x76,0x00,0x2c,0x87,0x67,0xf3,0x07,0xc3,0x43,0x0d,0x20,0x76,0x27,0x39,0x6a,0x59,
	0xf7,0x24,0xdf,0x73,0x27,0x85,0x1c,0x57,0x99,0x54,0x30,0x83,0xe5,0x16,0x8d,0xdf,
	0xbb,0xc4,0x8b,0xdb,0x47,0x1b,0x09,0xe5,0xb4,0x97,0xfe,0x51,0x8a,0xaa,0x3d,0x01,
	0x08,0x94,0xbc,0x1c,0x4b,0x42,0x13,0xfb,0xae,0x5f,0x45,0x27,0x0d,0x39,0x5e,0x04,
	0xb2,0x0e,0x51,0x50,0x52,0x0c,0xa4,0x85,0x38,0xae,0xa7,0x40,0x2c,0x63,0x62,0x0e,
	0xa6,0xbe,0x48,0x57,0x54,0x0d,0x5c,0x8a,0x24,0xec,0x90,0x96,0x4c,0x5c,0x7d,0xb8,
	0xfc,0x06,0xde,0x6b,0x2d,0x5f,0xbf,0x8f,0xd3,0x66,0xbd,0x27,0x59,0x41,0x4a,0x22,
	0x07,0xca,0x57,0x3e,0xed,0xe4,0x5d,0x25,0xcf,0x10,0xad,0xd1,0x5a,0x9e,0x03,0xa2,
	0x9e,0x2f,0x1e,0x25,0x85,0x6d,0x9a,0xae,0xea,0x1c,0x91,0x41,0x77,0xbc,0x2b,0x5b,
	0x82,0x5f,0x05,0x18,0xf3,0x20,0xcb,0x82,0x17,0x08,0x6f,0x56,0x9d,0xc1,0x83,0x67,
	0x31,0xfe,0x04,0x06,0xc9,0x93,0xa9,0x45,0x83,0xda,0x73,0x77,0x40,0xf7,0x71,0x6b,
	0x33,0xec,0xf8,0x05,0xd3,0x25,0x42,0x16,0x98,0x53,0x7d,0x32,0x36,0xc8,0xc1,0x1f,
	0x59,0x77,0x69,0x0f,0x03,0x63,0x37,0x55,0xa3,0x7e,0xd0,0x9b,0x6e,0x2b,0xc8,0xae,
	0x7a,0x0c,0xfc,0xe5,0x32,0x0e,0x7b,0xf3,0x48,0xee,0x2b,0x95,0x70,0x0d,0xa8,0xd0,
	0x9d,0xd0,0xa1,0xb2,0x1e,0x9a,0x82,0x97,0x0a,0xc0,0xc5,0x9d,0x9c,0x6d,0x32,0x4f,
	0x5d,0x86,0xc1,0x52,0x13,0x67,0x8f,0xf2,0x60,0xb8,0xbb,0x1a,0x5e,0xaf,0x2d,0x73,
	0xb9,0xbb,0x0d,0x15,0x98,0xb6,0x7e,0x41,0xd0,0x94,0x66,0x51,0x84,0xfc,0xb0,0x12,
	0xec,0x51,0xf8,0x5b,0x5d,0xbb,0x1d,0x0a,0xea,0x90,0xfa,0xa1,0x74,0xdd,0x10,0x5f,
	0xf7,0x2b,0x90,0x62,0xc9,0x89,0x28,0x87,0xce,0x5f,0x03,0x5f,0x77,0xcc,0xeb,0x7a,
	0x9b,0x7c,0x44,0xf0,0x83,0x25,0x92,0x08,0xab,0x96,0x3c,0x97,0x4e,0x7d,0x21,0xec,
	0x0f,0x16,0x0c,0x79,0xa6,0x6d,0x8b,0x5c,0x34,0x2a,0x29,0x0e,0xb5,0x2e,0x3b,0x41,
	0x45,0x28,0xb5,0xbb,0x2b,0x30,0x20,0x23,0x22,0xc8,0x1d,0x26,0x3c,0x14,0xb6,0xfe,
	0xee,0x79,0xfe,0xa1,0x94,0x5e,0x92,0xf7,0xb8,0x7b,0x0d,0x0c,0xeb,0x44,0xfa,0x54,
	0x46,0x8b,0xbc,0x00,0xac,0x41,0x06,0x4c,0x1e,0xdd,0x51,0x89,0x55,0x5f,0x89,0xf4,
	0xaf,0x9e,0x07,0x61,0x5d,0x2a,0x11,0xde,0xb0,0xd5,0x1e,0xff,0x33,0x5b,0x94,0x94,
	0x45,0x62,0x82,0xa4,0x6d,0x72,0x1c,0x49,0x24,0x86,0x41,0x0e,0x82,0xf7,0x13,0x17,
	0x68,0xc5,0x2a,0x7a,0x69,0xa9,0xb1,0x78,0x0c,0x20,0xfc,0x4b,0x07,0xc0,0x5f,0x9e,
	0xc7,0x97,0xe8,0x9a,0x14,0xf1,0xff,0x01,0xcd,0x73,0x81,0xaf,0xad,0x5c,0x43,0x50,
	0x6f,0xe1,0x85,0x30,0x5b,0x0b,0x39,0x74,0x21,0xcb,0xa3,0xc5,0xea,0x26,0xed,0x77,
	0x9b,0x91,0x56,0xb9,0x49,0x13,0x21,0x30,0x98,0x5c,0x86,0x73,0x1d,0x43,0x80,0xe6,
	0x07,0xbc,0xa7,0x7c,0x8b,0xa6,0xc2,0x43,0xc1,0x4e,0xf9,0x57,0x6b,0x76,0x67,0x07,
	0x53,0xad,0x1b,0x9a,0x97,0x67,0x51,0xaf,0x85,0xd3,0x69,0xe4,0x45,0x80,0x28,0x0a,
	0x87,0x21,0xa6,0xb5,0x56,0x71,0x82,0xde,0xc5,0xcb,0xd7,0xfd,0x7a,0xdb,0xd1,0xb1,
	0x31,0x22,0x69,0x5f,0xfd,0x2d,0xd8,0x36,0xa4,0x02,0x21,0x1a,0xe6,0x1c,0xeb,0xee,
	0xeb,0x69,0xc8,0xa2,0x6f,0x87,0xd2,0xa3,0x7f,0x39,0x5c,0xd2,0x1c,0x05,0xf8,0xe6,
	0x3b,0x4e,0xed,0x72,0xf0,0x2f,0x9e,0x0c,0xac,0x04,0xd7,0xcc,0x55,0x7f,0x85,0xba,
	0x5b,0xe6,0x39,0x25,0xb3,0x78,0x97,0x9d,0xab,0x6b,0xcd,0x07,0xfd,0xc3,0x55,0x00,
	0x99,0x24,0x28,0x33,0x7e,0xbb,0x65,0x88,0x8e,0xcf,0x85,0x17,0x21,0xc2,0x03,0x41,
	0xf3,0xb7,0xe0,0x51,0xf9,0xf4,0x88,0x51,0x54,0x14,0x28,0x22,0xe5,0xa9,0x71,0xf0,
	0x0d,0x83,0x9e,0x22,0x05,0x5f,0x02,0x67,0xc5,0xd5,0xa8,0xe5,0xac,0x66,0x2b,0x04,
	0xa3,0xdf,0xe8,0x61,0xa3,0x9b,0x73,0x17,0x81,0xd8,0x21,0x8c,0x8f,0x23,0x68,0xb1,
	0x33,0x64,0xed,0xfa,0x17,0xaf,0x18,0xc1,0xc4,0x3d,0xe9,0x37,0xff,0xc3,0xa9,0x6e,
	0xf9,0x34,0xd0,0xaf,0xc5,0x69,0x0b,0x7d,0x00,0x5f,0xd8,0xc6,0xe3,0x0c,0x6f,0xb6,
	0xf7,0xb8,0x2b,0x72,0x47,0xd5,0x93,0xe1,0xd9,0x0c,0xd0,0x2d,0x80,0x1d,0x21,0x65,
	0x78,0x56,0x1d,0x21,0xa7,0xb0,0x8d,0xd8,0x3a,0xfd,0x0e,0xaf,0xaa,0xcb,0x1c,0xbd,
	0xe7,0x24,0x14,0x41,0xd4,0x82,0x48,0x4c,0x88,0x8b,0xf1,0x55,0x52,0x18,0x30,0x24,
	0x6d,0x4b,0x29,0x97,0x53,0xe6,0x14,0x60,0xfe,0x12,0x95,0x04,0x2a,0x64,0x2e,0x88,
	0x09,0x9d,0x25,0x30,0xe9,0xea,0x21,0x97,0x82,0x3a,0x3f,0x68,0xd9,0x2c,0x44,0x3d,
	0x63,0xe4,0x73,0x64,0x10,0x2f,0xf9,0x26,0x1d,0x07,0xfa,0xdb,0xd4,0xa2,0x0c,0x5c,
	0x24,0x33,0x13,0x25,0x0b,0x34,0x9c,0x13,0x93,0x5c,0x6d,0x42,0x45,0x27,0xe0,0xed,
	0xee,0x99,0x27,0x61,0x4e,0x73,0x61,0x2b,0x0f,0xf5,0x5b,0x57,0x3f,0xb4,0x9e,0x14,
	0xaa,0x8f,0x48,0x63,0x19,0x7f,0x47,0x0b,0xf2,0xe3,0xd3,0x7b,0xc1,0xc4,0x14,0x83,
	0x77,0x97,0x84,0xd2,0x28,0x72,0xec,0x1d,0x21,0x52,0xb6,0x8f,0xb8,0x35,0xec,0xa1,
	0x27,0x1d,0x09,0x20,0x77,0x4c,0x3e,0xde,0xc1,0x23,0xa8,0x94,0xce,0x00,0xbc,0x32,
	0xea,0x71,0x9a,0xd0,0x38,0x5c,0xe5,0x14,0x46,0x72,0x82,0x03,0x01,0xcf,0x99,0x9a,
	0x1e,0xd1,0xe7,0xb3,0x81,0x80,0x45,0x02,0x83,0x61,0x4a,0x3f,0xc5,0x94,0x5d,0x0f,
	0xe5,0xcc,0x6e,0x72,0xd5,0x57,0x63,0x20,0x73,0xc9,0x4e,0x5d,0xba,0x49,0xad,0xdf,
	0x76,0x22,0x1d,0x9a,0x0d,0xe5,0x77,0x89,0x63,0x1a,0x71,0x22,0x0e,0x5b,0x5e,0x06,
	0x47,0x81,0xa8,0x19,0x3d,0x16,0x4d,0x93,0xda,0x1f,0x4d,0xdc,0x31,0xa3,0xd2,0xf6,
	0xb5,0x54,0x64,0xc2,0x1a,0x2f,0x18,0x28,0xea,0x4a,0x44,0x6e,0x7c,0x19,0xe9,0xea,
	0xce,0x79,0x01,0x02,0xc7,0xbc,0xb6,0xc7,0x49,0xf0,0x2a,0xd7,0xa4,0xd8,0x58,0xc1,
	0xc7,0xce,0x21,0xc3,0x6f,0x9e,0xc3,0x7f,0x25,0xb6,0xbd,0xea,0xac,0xd5,0xcb,0xd9,
	0xae,0x17,0xbf,0xc6,0x24,0xc1,0x7e,0xc8,0xb1,0x10,0x82,0xfb,0x51,0xcc,0xf2,0x34,
	0x82,0x39,0x8d,0x44,0xbc,0xd2,0x36,0x8d,0xdf,0x9a,0x7b,0xc4,0xf2,0x91,0x09,0xb3,
	0xaa,0x1e,0x00,0xcb,0x21,0x2d,0x25,0xae,0xd8,0xc8,0xc5,0x47,0xce,0xfe,0x77,0xf8,
	0xc8,0xcf,0xa0,0x72,0x57,0xab,0xfb,0xe9,0x29,0x99,0x3c,0x6e,0x29,0x0b,0x25,0xca,
	0x58,0x5a,0x0c,0x58,0x74,0xf1,0xac,0x19,0x4d,0xc9,0xc6,0x55,0xb1,0x4c,0x77,0x56,
	0x1f,0x46,0x88,0x8b,0xe5,0x68,0x0f,0x9a,0x67,0x82,0xba,0xd8,0x50,0xfd,0x17,0x3c,
	0xe7,0x3f,0x51,0x6b,0x54,0x64,0xab,0x1f,0x60,0x04,0x55,0x0a,0x05,0x19,0xfb,0x14,
	0x8f,0xc0,0x54,0x0c,0xc7,0xa2,0x53,0x98,0x4a,0x7c,0xad,0xc6,0x21,0x4b,0xc9,0x86,
	0x13,0x41,0x1f,0xb5,0xe8,0x48,0xf3,0x06,0x2f,0x45,0x96,0x70,0x8f,0x3d,0xfd,0x6f,
	0x29,0x61,0x3b,0x56,0xa2,0xcc,0x02,0xfc,0x15,0x49,0x9d,0xa6,0xfa,0x6c,0xcc,0xdf,
	0x22,0xe6,0x57,0x83,0x7e,0x87,0xf3,0x67,0x0b,0xfd,0xa4,0xb5,0x26,0x75,0x95,0xd0,
	0xc5,0x4c,0xd6,0x96,0x46,0x86,0x9f,0x79,0xd4,0xc6,0x9c,0x1c,0xb7,0x1d,0xa7,0xd2,
	0x55,0x40,0x0b,0xbb,0x9f,0x5c,0x69,0x32,0x87,0x27,0x61,0x9a,0x8a,0x26,0x81,0x3a,
	0x02,0xbc,0x59,0xe2,0x2f,0xf9,0x0d,0x6b,0x79,0x2d,0x11,0x90,0x2e,0xda,0xb5,0x2b,
	0xe9,0x8d,0x90,0xbf,0xd8,0x20,0x68,0xd7,0x9e,0xd0,0xc3,0x39,0x74,0x81,0xb8,0xb3,
	0x81,0x25,0x14,0x8f,0x4c,0xf5,0x14,0x9e,0x00,0x0e,0x11,0x02,0xee,0x2d,0xfd,0x42,
	0xbd,0x2f,0x2f,0x4c,0x4e,0x88,0x24,0x70,0x63,0x1f,0x61,0xe0,0xd6,0xd7,0x49,0x39,
	0x5d,0x95,0x7b,0x6e,0xce,0xed,0x23,0xcd,0x6d,0x53,0x97,0xbf,0x45,0xe2,0x0e,0x5e,
	0x15,0x7d,0x18,0x44,0x1b,0xeb,0x77,0x90,0x1a,0x20,0x11,0xa3,0xf0,0xd1,0x51,0x22,
	0xa9,0xd0,0x95,0xd4,0xa1,0x0a,0xca,0xb6,0x2a,0x37,0x12,0x07,0x68,0x52,0xab,0x77,
	0xd1,0x2d,0x99,0xb0,0xcf,0x18,0xb8,0x72,0x3c,0x2f,0xda,0x83,0x57,0xd0,0xeb,0x73,
	0xce,0x5e,0x61,0x74,0xa8,0xa4,0x2d,0xc3,0xff,0xfe,0x2b,0xfa,0xc9,0xd2,0x69,0x76,
	0x3f,0xac,0x1f,0x2b,0x06,0x81,0x2c,0xae,0x7a,0x4a,0x95,0xde,0x20,0x92,0x9d,0x89,
	0x53,0x8a,0xf1,0x03,0xb5,0x69,0x74,0x36,0x70,0x28,0xe3,0xf1,0xf9,0xc7,0xe7,0xb0,
	0x7a,0x31,0x7f,0x00,0x51,0x32,0xe6,0x5c,0x00,0xc6,0x2d,0x4c,0x09,0x7a,0x99,0xf2,
	0x6e,0x43,0x0b,0x61,0x22,0xb2,0xf5,0x8f,0x7c,0x37,0x2c,0xcf,0x37,0x9d,0x52,0xc4,
	0xfe,0x28,0x3b,0x5c,0x2f,0xa1,0x82,0x59,0x93,0xc5,0x6e,0xa3,0x7c,0xa2,0x1d,0x07,
	0x16,0x18,0x3d,0xc3,0x39,0x4e,0x17,0xc0,0xe9,0x70,0xc4,0x13,0xc6,0xae,0xcf,0x99,
	0x2a,0xf7,0xd4,0xbd,0xa0,0xae,0x4d,0xf4,0xc9,0x82,0x05,0x8f,0x79,0xf4,0x75,0xc0,
	0x27,0xe1,0x01,0x06,0xe5,0x36,0x31,0xfc,0x60,0xb6,0x29,0xcc,0x53,0xe4,0x41,0x7d,
	0xb0,0x3a,0xa7,0x3a,0xbc,0xff,0x0e,0xd0,0x62,0x44,0x84,0xc0,0xd0,0x49,0xaf,0xaf,
	0xc9,0x69,0xb7,0x7e,0xb9,0xe3,0x2d,0xd0,0x71,0xe2,0xa1,0x82,0x88,0x81,0xf9,0xed,
	0x28,0xee,0xc2,0x45,0xaa,0x72,0xee,0xe7,0x36,0xaf,0xe7,0xb3,0xd1,0x89,0xbb,0x70,
	0x9b,0xfa,0x52,0xee,0xf3,0xbb,0x18,0x2b,0xc7,0x88,0x6b,0x1d,0x2c,0x77,0x18,0xa7,
	0x43,0xae,0x2e,0x11,0x77,0x53,0x42,0xa7,0xed,0xa6,0x28,0x87,0x42,0x14,0x54,0x5a,
	0xf6,0x8e,0x37,0x34,0xfd,0x2a,0xda,0xb9,0x7f,0x04,0x65,0x1a,0x38,0x21,0x62,0xd9,
	0x96,0x8f,0xdd,0xbc,0x59,0x3e,0xf6,0xc6,0x19,0x5b,0x15,0xf0,0x33,0xa3,0x38,0x44,
	0xf1,0x0d,0x08,0x8b,0x44,0x84,0xbf,0xbc,0xb8,0x4a,0x41,0x2d,0xfa,0x58,0xfc,0x55,
	0x6b,0xcf,0x6e,0x81,0x14,0xc3,0x23,0x4e,0x00,0x80,0x53,0x0b,0xd6,0x4a,0x52,0x85,
	0xa0,0x8e,0x99,0x00,0xdf,0x34,0x5b,0xb9,0xc3,0x56,0x0e,0x3a,0xc9,0x7d,0x43,0x11,
	0x4b,0xe8,0xf3,0x07,0x9a,0x43,0x83,0x0f,0x7d,0x09,0x48,0x1e,0x09,0x3a,0x8d,0x89,
	0x89,0xfa,0x99,0xd1,0x22,0xa9,0x66,0x2b,0x80,0xde,0xa1,0xd2,0x3c,0x54,0x79,0xb9,
	0xf0,0xaf,0x65,0xdd,0x2e,0xae,0xb7,0x6f,0x5f,0x97,0x50,0x8a,0x28,0xd0,0x72,0x67,
	0xba,0x69,0x03,0x3e,0xd0,0xfe,0xd3,0xd3,0x9d,0xd3,0x36,0xf4,0xe9,0x66,0x45,0x54,
	0x95,0xd9,0xd6,0x21,0x35,0x4d,0x22,0x5e,0x97,0x8c,0xc6,0x97,0xaf,0xb7,0x93,0xdd,
	0xd5,0x1d,0x14,0x42,0x70,0x09,0xbf,0x9f,0x53,0xf8,0x18,0x35,0x30,0xe8,0xe8,0x20,
	0x8d,0x8b,0xc9,0x89,0x69,0xcc,0x49,0x0c,0xc3,0x8b,0x50,0x23,0x55,0x86,0x3b,0xc8,
	0xcd,0xdc,0xcd,0x48,0x31,0xec,0x68,0x22,0xb0,0x46,0x63,0x12,0xb0,0xc3,0x2c,0x5c,
	0x54,0x55,0x6e,0xf3,0x40,0x31,0xbc,0x7b,0x37,0x32,0x3e,0x74,0x48,0xeb,0xfa,0xbe,
	0x76,0xe7,0x7e,0x34,0xaf,0xcf,0x53,0x85,0x87,0xfc,0x0d,0x55,0xc8,0x8a,0xa4,0x4f,
	0xa9,0x16,0x5f,0x1d,0x4d,0x72,0x33,0xf3,0xe5,0x12,0x5b,0xbe,0xc9,0x51,0x8d,0xb7,
	0x6e,0x92,0x9c,0x9d,0x1b,0x4c,0x0b,0xfe,0x86,0x6c,0xaa,0xe1,0x4c,0x82,0x86,0x41,
	0x57,0xbc,0x06,0x54,0xcb,0xd5,0x08,0xe9,0xf7,0x3b,0x64,0xcb,0xa6,0x58,0xad,0x36,
	0xf5,0xe2,0xfe,0x00,0x20,0x69,0xd4,0xf7,0xf6,0xc3,0x26,0x31,0x76,0x88,0xf0,0x66,
	0xab,0xaf,0xb6,0xaf,0x37,0x30,0x8c,0x1f,0x55,0x57,0xe9,0x49,0x06,0xf6,0x26,0x1d,
	0xea,0x68,0x7c,0xfe,0xd0,0xba,0xb3,0x90,0x17,0xa4,0x49,0x25,0x03,0xc2,0xd6,0xf0,
	0x85,0xac,0x52,0x46,0xe7,0x65,0x86,0x45,0xee,0xb4,0x82,0x3a,0xcb,0xaa,0x06,0x2a,
	0x47,0x7e,0x46,0x90,0xbc,0x01,0x1e,0xbf,0x71,0x8e,0xa5,0x1c,0x0b,0xfd,0x4c,0x3a,
	0x0d,0xc1,0x51,0xe5,0xad,0x0e,0x98,0x20,0x04,0xf1,0x24,0x5a,0x1c,0xf3,0xfb,0x51,
	0xcc,0xf8,0x4f,0x53,0xed,0x20,0xfa,0x8a,0x70,0x69,0x68,0x59,0x1e,0x0c,0x38,0xc4,
	0x5a,0x5f,0xae,0xf8,0x0b,0x94,0xaf,0x0a,0x8a,0x02,0x4f,0xf8,0x53,0xe2,0x00,0x40,
	0x33,0xd6,0x29,0x9f,0xe3,0x8f,0x7f,0xd2,0xd3,0x4d,0xc9,0x29,0x4a,0xd0,0xb1,0xc1,
	0x08,0x54,0xa3,0x47,0x65,0x21,0x48,0x9a,0x15,0x34,0xa4,0xac,0x3a,0x5b,0x66,0x52,
	0x9e,0xf4,0xc0,0xfd,0xe9,0x04,0x7f,0xcc,0x4c,0xab,0x6d,0xaf,0xea,0x65,0x00,0xac,
	0x4f,0xb9,0xc2,0xb6,0xbf,0x41,0x22,0x92,0x60,0xa0,0xc2,0xbc,0xbe,0xe3,0x0e,0x3f,
	0x06,0x56,0x87,0xeb,0x48,0xf0,0x62,0x7a,0x1e,0x4f,0xd8,0x2a,0x20,0x89,0x1f,0xa5,
	0xb2,0xe5,0x46,0x88,0x42,0x43,0x5b,0xf4,0xe3,0x71,0x0d,0xed,0x72,0xaf,0xac,0x3a,
	0xeb,0x5b,0x3d,0x81,0xf0,0x05,0xa0,0x76,0x4d,0xb4,0x00,0x18,0x31,0xd0,0x1f,0x21,
	0x9a,0x72,0x75,0xe1,0x40,0xf4,0x2a,0xff,0xd1,0x9a,0xe7,0xeb,0x33,0xf4,0x0e,0xf3,
	0xa5,0x2e,0xfb,0xa9,0x80,0x74,0x6e,0x78,0x2f,0x2a,0x5b,0x97,0x8f,0x14,0x08,0x14,
	0x67,0x7b,0x0a,0xea,0x6f,0xa1,0xf1,0x4d,0x83,0xca,0x8e,0x0a,0x6f,0x14,0x47,0x64,
	0xa8,0x5b,0x6a,0x83,0xfc,0xc2,0xb3,0xcf,0xcc,0xb8,0x51,0xf4,0xb8,0xff,0x0e,0x3f,
	0x88,0xec,0x9e,0x83,0xbb,0xb5,0xbe,0xa4,0x06,0xd3,0x41,0x5a,0xcd,0xdb,0xd5,0x82,
	0x26,0x85,0xdd,0xea,0x4b,0x3f,0x4a,0xa3,0x4e,0x12,0x57,0xd0,0x06,0x77,0x31,0xbe,
	0x91,0xbb,0x2b,0xa9,0xc4,0x43,0xd8,0x8e,0x52,0x28,0x55,0xcb,0x6f,0x57,0xf2,0x8e,
	0x2b,0xf6,0x03,0xe1,0xd7,0x5e,0xee,0xec,0x5c,0xf3,0x5c,0x13,0x11,0xea,0x64,0x6e,
	0xcc,0x77,0xcf,0x81,0x3e,0x24,0x42,0x1c,0xc7,0x99,0x8b,0x86,0x31,0x6e,0x4c,0x9c,
	0x04,0x6a,0x54,0x81,0x33,0x3e,0x13,0x0b,0xde,0xe9,0x91,0x83,0x4a,0xf4,0x45,0xd7,
	0x11,0x61,0xfb,0xe2,0xeb,0x01,0xaa,0x14,0xf7,0x61,0x96,0x56,0xb1,0xa3,0x6f,0x28,
	0x3e,0x6f,0x72,0x78,0x58,0xdf,0x7c,0x83,0xad,0x73,0x99,0xa4,0x8a,0xc9,0xf0,0x83,
	0x4e,0x0f,0x56,0xe9,0x35,0x05,0x64,0x98,0x66,0x86,0xcf,0x8c,0x27,0x9b,0x98,0x9d,
	0x56,0x85,0xce,0x9d,0xd5,0x39,0x3e,0xa9,0x12,0x28,0xc7,0xb5,0x5c,0xc2,0xba,0x84,
	0x7a,0x3a,0x7f,0x71,0x4d,0x28,0xfa,0x9e,0x25,0x46,0x9e,0xf9,0xf4,0x87,0x5c,0xfa,
	0x43,0xa0,0xe7,0xc1,0x82,0xfe,0xdf,0x56,0xaf,0x63,0x4e,0x03,0x26,0x28,0xf5,0xaa,
	0x79,0xb0,0xd4,0x3d,0x1d,0x2c,0x3e,0x90,0x8b,0xa1,0x18,0x9a,0xde,0x1f,0xa7,0x5a,
	0x06,0x35,0xba,0x7f,0xae,0x1f,0x18,0x54,0x10,0x17,0xea,0xc4,0xd0,0x61,0x28,0xc9,
	0xdb,0xf2,0xe5,0xe5,0x65,0x38,0x86,0x45,0xa1,0x7b,0xb9,0x2f,0xfb,0xdf,0x29,0x30,
	0x83,0x28,0x2e,0xb4,0x9b,0xe8,0xee,0x78,0x3b,0x7f,0xaf,0x14,0xc8,0xb7,0xcd,0x2e,
	0x02,0x4c,0x1e,0xe6,0xaf,0xea,0x8c,0x67,0x88,0x87,0xb6,0xaa,0x8d,0xe7,0x89,0x53,
	0xe8,0x78,0x36,0x7a,0xaf,0x73,0x2c,0xde,0x1e,0x1c,0xe8,0x53,0xeb,0xe2,0x80,0x73,
	0x1d,0x58,0xf0,0xe5,0x1c,0x0c,0xa5,0x47,0xaf,0xce,0x29,0xde,0xbc,0x14,0xc5,0xdc,
	0x9c,0xae,0x43,0x33,0x18,0x0a,0x83,0x5c,0xcc,0x3f,0x2b,0xd0,0x05,0x70,0xe1,0x3b,
	0x2d,0x4b,0x82,0xa9,0x59,0xd2,0x18,0x36,0x33,0x7f,0x0e,0xa2,0x43,0xe9,0xa6,0x21,
	0xb3,0x82,0xed,0xa7,0x11,0xb1,0xc1,0x4e,0x86,0x53,0xb9,0xcd,0xbf,0x26,0x2f,0xb1,
	0x5d,0xe8,0xde,0xac,0x0a,0xc9,0xab,0x61,0x2b,0x02,0x79,0x27,0x2f,0xdb,0xa2,0x09,
	0x87,0xe7,0x81,0x78,0xb6,0x39,0x68,0xcc,0xb4,0x2d,0xa6,0x4c,0xe3,0xd9,0xb9,0x86,
	0xdc,0x84,0x24,0xa9,0xa9,0xa3,0xb8,0x07,0xda,0x9c,0xba,0x64,0x2c,0x46,0x07,0xe2,
	0xeb,0x00,0x65,0xd0,0x84,0x81,0xd4,0x16,0xee,0x0c,0x9d,0x68,0x53,0x20,0x73,0xd3,
	0x1a,0x29,0x9a,0x17,0xf9,0x51,0xe0,0x1b,0x0c,0xd8,0x85,0x94,0xb7,0x36,0xa6,0x1b,
	0xd3,0xf8,0xca,0xc1,0x1d,0x76,0xf9,0xae,0x9d,0xfb,0x97,0x0a,0x7d,0x93,0x17,0x90,
	0x2d,0xa1,0x3b,0x25,0x8b,0xd5,0xfa,0xc2,0xf1,0x2a,0xa6,0xdc,0x85,0x7d,0xc6,0x3f,
	0x33,0x65,0x7f,0xaa,0xda,0xfa,0xf7,0x35,0xae,0x7a,0x01,0x72,0xdd,0x54,0xf1,0x8d,
	0x33,0xac,0xd3,0xb1,0xaf,0x6b,0x26,0x45,0x95,0x24,0x7a,0x17,0x1c,0x6d,0xc0,0x9a,
	0x2d,0x5d,0xd3,0x1b,0x3d,0x15,0xbc,0xd6,0xdd,0x8d,0x9f,0x7e,0xe5,0x6b,0xd7,0xaa,
	0xd8,0xd2,0x7f,0xd2,0xf7,0x2a,0x1f,0xd8,0x34,0x93,0xf8,0x59,0xe3,0x15,0x42,0xc3,
	0x13,0x43,0x32,0x9d,0x6d,0x97,0xa2,0x7e,0xe6,0xbb,0xf9,0x37,0xa9,0x49,0x04,0x5d,
	0xe0,0x78,0x3e,0xec,0xa5,0xe7,0x0e,0x42,0xf0,0xf0,0x87,0x40,0x0e,0xd3,0x42,0xe2,
	0xf2,0x3f,0x58,0xba,0x6d,0xea,0x46,0x95,0x73,0xfd,0xbe,0x7a,0xf2,0xf1,0xd7,0x10,
	0x90,0x77,0x67,0x99,0xf7,0xcf,0x45,0x53,0x78,0x87,0x12,0x0d,0xf4,0x04,0xc0,0x15,
	0x5c,0xa3,0x4c,0xba,0x36,0x18,0xba,0x11,0xf3,0x3a,0x74,0x51,0x36,0x86,0xf1,0x70,
	0xbc,0xfc,0xc6,0xec,0xa4,0xff,0x8d,0x8c,0x64,0x46,0x87,0xf9,0x6a,0x8f,0xda,0x24,
	0xdd,0xce,0xa1,0x07,0xd5,0xec,0x6f,0xa8,0x6f,0x7a,0x5b,0x9b,0xa2,0x3f,0x2d,0x5f,
	0x3a,0x12,0x92,0xbb,0x6c,0xc8,0x02,0xe8,0x12,0x69,0x1e,0x6b,0x4f,0xd0,0x04,0x50,
	0xa4,0x14,0x7b,0xd9,0x8b,0x3f,0x40,0xb0,0xbb,0xed,0xc5,0xec,0xb3,0xcc,0x5a,0x64,
	0x96,0x97,0x15,0x05,0x50,0xfe,0xa4,0x67,0xb4,0xa5,0x98,0x75,0x96,0x0c,0x58,0x7a,
	0x6e,0xfa,0xa3,0x86,0x3e,0x71,0xb7,0x7e,0x40,0x9f,0xfd,0x65,0x67,0x61,0x10,0x19,
	0x8c,0x77,0x98,0x16,0xdf,0xb9,0xfa,0x4b,0x78,0xdb,0x93,0x86,0xec,0x9a,0x31,0xe3,
	0xda,0xab,0xbd,0xf2,0x59,0x5e,0x6f,0xea,0xa4,0x47,0x27,0x38,0xbf,0x82,0x81,0x3a,
	0xa2,0x13,0x6f,0x4b,0xe8,0xd1,0xfa,0x1d,0x51,0x26,0xb4,0xd4,0x05,0x6e,0x08,0xb1,
	0xaa,0x3d,0x4e,0x9f,0xec,0xf4,0xb6,0x7f,0x69,0x8a,0x66,0x58,0xe4,0xcf,0x86,0xc7,
	0x97,0xc6,0xc6,0xb3,0xaf,0xd1,0x17,0x9b,0xb1,0x68,0xd8,0x9c,0x99,0x6a,0x4f,0xfa,
	0x77,0x95,0x5c,0x08,0x57,0xc3,0xc2,0x0d,0x6a,0xfb,0xbe,0x42,0x97,0x66,0x88,0x52,
};


/* 317-???? FD1089A */
static const UINT8 key_alexkidd[0x2000] =
{
	/* data */
	0x2b,0xe7,0x6e,0x6f,0x8b,0xcd,0xd8,0x4a,0x23,0x8e,0x30,0x97,0x93,0xe4,0x9b,0x92,
	0x60,0x2d,0x9e,0x4b,0x75,0x5c,0x08,0xa6,0x9b,0x96,0x37,0x69,0xf4,0xb3,0xc3,0x77,
	0x73,0x24,0x80,0xf1,0xea,0xad,0x04,0x66,0x7f,0xf9,0x8b,0xad,0xb4,0x63,0x00,0xd7,
	0x74,0x21,0x2f,0x40,0xec,0x8d,0xde,0x42,0x92,0xc6,0x55,0x3d,0x6d,0x8e,0x66,0x0c,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0x86,0xde,0xb5,0xb0,0x78,0x6d,0x01,0x1c,0x0b,0xdc,0x23,0x97,0xc8,0x4c,0xcc,0xf1,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0022 FD1089A */
static const UINT8 key_0022[0x2000] =
{
	/* data */
	0x00,0xd1,0x14,0x06,0x00,0xe5,0x18,0x30,0xc8,0xd5,0x94,0x32,0x2b,0x37,0xf4,0x99,
	0xb2,0x01,0x62,0xc6,0xec,0xef,0x67,0xcf,0x84,0x12,0x30,0x85,0x6d,0xf2,0x57,0x78,
	0x65,0xe7,0x76,0x95,0x51,0xdc,0x39,0x32,0xba,0xf1,0x55,0x49,0xe6,0x17,0x71,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x15,0xa8,0xda,0xe4,0x8d,0x40,0x15,0xbb,0x3e,0x99,0xd4,0x0d,0x2e,0x3c,0x35,0xb6,
	0x55,0x19,0x12,0x85,0x39,0x92,0xa5,0x3f,0x3c,0x3a,0x84,0xde,0x75,0xfe,0xc9,0x4f,
	0x40,0x88,0xa0,0xcd,0x39,0xa0,0xcd,0x69,0x1f,0x63,0x30,0xf7,0xb6,0x8a,0x58,0x1c,
	0xd0,0x85,0xfd,0x8b,0x6a,0x00,0xb7,0x01,0x79,0x84,0x8e,0x05,0x48,0x43,0xf8,0x5b,
	0xd8,0xb7,0x50,0xa9,0x0c,0x97,0xca,0x73,0x2a,0xe3,0xe4,0x84,0x5e,0xf9,0xf0,0xcc,
	0xb3,0x44,0x8f,0x71,0x6d,0xa5,0x02,0xc2,0x3c,0x3d,0x5d,0x93,0xd6,0x8d,0x4c,0xd8,
	0xf4,0x30,0xfd,0xc1,0xd3,0xdf,0xc7,0xc5,0xc4,0x0c,0x0c,0xfc,0x1b,0x6b,0x80,0xb7,
	0x18,0xab,0x9e,0x9e,0x01,0xaf,0x69,0x47,0x6f,0x1f,0x73,0x0c,0x46,0x06,0x89,0x88,
	0x60,0x19,0xce,0x11,0x6e,0x7b,0xa5,0xbd,0x5d,0x67,0xa0,0xb1,0xff,0xe1,0xdd,0x51,
	0x9f,0x5f,0x18,0x37,0x35,0x65,0xff,0x1b,0x49,0xb6,0x85,0xcc,0x21,0xe0,0xf6,0x88,
	0xa5,0x12,0xd1,0xee,0x0f,0x91,0xb0,0xc0,0xc9,0x3f,0xf4,0x75,0xbf,0x45,0x8f,0x8e,
	0x6c,0x43,0x0a,0x7a,0x22,0xed,0xa3,0x8d,0x5e,0x92,0x44,0xfa,0x42,0x67,0x5e,0xda,
	0x30,0x3b,0xd9,0xaa,0x4b,0xa3,0x01,0x5a,0x47,0x76,0x94,0x75,0x35,0xd6,0x87,0x9e,
	0x7e,0xca,0xc8,0xf6,0xd0,0x74,0xde,0x32,0x8a,0x18,0xb0,0xaf,0x7c,0x67,0xa3,0x88,
	0x7d,0x7f,0xe1,0xc3,0xf8,0xdc,0xe8,0x5d,0xe6,0xcc,0x95,0xb3,0x8b,0xe5,0x96,0x20,
	0x23,0x76,0xe9,0x0f,0x48,0xa9,0x17,0x1d,0xaf,0xcb,0x4b,0x61,0xed,0xe9,0x49,0x52,
	0x71,0x17,0x42,0x14,0xd1,0x48,0x45,0x77,0x08,0xc7,0x5a,0x0a,0x63,0xae,0x7f,0xe7,
	0x71,0xb4,0x25,0xa7,0x0b,0x4e,0x43,0x91,0xc8,0x20,0xfa,0xa0,0x22,0x04,0xa6,0xd8,
	0x0b,0xae,0xaa,0x7d,0x48,0xe7,0x4f,0xb8,0x1d,0xae,0xfe,0xa3,0xf9,0x38,0x1b,0xdc,
	0x9e,0x1e,0x5d,0x67,0x03,0x1b,0xbe,0xb3,0x68,0x34,0x86,0x0f,0x81,0x8f,0xdf,0x50,
	0x1a,0xd5,0xcb,0x19,0xa2,0x4b,0x6e,0x23,0x54,0x36,0x78,0x78,0x54,0xf1,0x62,0x6b,
	0x00,0x11,0x85,0xf3,0x51,0x59,0x2b,0xbc,0x2c,0x2c,0xe9,0x2b,0x3e,0x48,0xa4,0x59,
	0x42,0xc6,0xe6,0x70,0xa9,0x70,0x9f,0x43,0xa2,0xb7,0xe8,0x69,0x10,0xa1,0x45,0x67,
	0x6d,0x1f,0x38,0x85,0x3a,0x59,0xdc,0xae,0x2e,0x1e,0x10,0x7e,0x5f,0xc8,0x52,0xc7,
	0x39,0xcc,0x64,0x06,0x57,0x13,0xac,0x47,0xfc,0x10,0x50,0x4a,0xf0,0x2a,0xa3,0x75,
	0x1c,0xad,0x05,0x10,0x71,0x19,0x3a,0x9e,0xae,0x70,0x83,0x19,0x24,0x6d,0xb1,0x99,
	0x26,0xc1,0x49,0x65,0xc7,0xaf,0xf1,0xd3,0xbf,0x05,0x4f,0xd9,0x30,0x10,0x77,0x0e,
	0xa1,0x65,0x2b,0xe6,0x6d,0x84,0x33,0x88,0xde,0xc7,0xc9,0xb5,0x4e,0xe1,0x86,0x44,
	0x46,0xa8,0x39,0xe6,0x40,0x78,0x99,0xc9,0x82,0x71,0xf2,0x46,0xeb,0x1c,0xa8,0xad,
	0xdf,0x05,0xcf,0x1a,0x78,0xc2,0x13,0x2a,0x79,0x7c,0x47,0xe3,0x2f,0x03,0xcf,0xff,
	0xea,0x74,0x3a,0x46,0xd4,0xd5,0x09,0x52,0x26,0x32,0x5e,0x37,0x9c,0x3d,0x02,0xc2,
	0x45,0x38,0x90,0xa7,0x67,0xaf,0xa0,0x4f,0xfb,0xd9,0xbb,0xa0,0xe4,0x86,0x56,0x50,
	0x4c,0x96,0xbc,0x0a,0x52,0xd7,0x39,0x88,0x22,0x51,0x51,0x1b,0x62,0x67,0xd0,0xc2,
	0x57,0x85,0x09,0x51,0x49,0xae,0xe7,0xa8,0xd2,0xa8,0xeb,0x49,0x0d,0xb1,0x5f,0xde,
	0x7c,0xfa,0x56,0x66,0x2c,0x50,0x0c,0x4a,0xf0,0xa6,0xf7,0x20,0xc3,0xb2,0x2c,0xad,
	0x2b,0x5d,0x61,0x9e,0x62,0x9d,0x66,0x8e,0xc2,0x51,0xcb,0x3b,0x85,0x52,0x05,0x3e,
	0xd6,0x0a,0x03,0x0e,0x3c,0x14,0xb5,0x73,0x42,0x3b,0xb0,0x1a,0xcc,0xde,0xea,0x84,
	0x8b,0x25,0x4e,0x3e,0x23,0xff,0xff,0x3c,0xca,0x76,0xdf,0xa3,0xf4,0xaf,0x58,0x02,
	0xc2,0x01,0xd9,0x0b,0x2d,0x7b,0x7f,0xe0,0x26,0xdd,0xd1,0x29,0xff,0x88,0x53,0x09,
	0x9c,0xe0,0xd2,0x41,0x67,0x37,0x8b,0x38,0x38,0xfd,0xf3,0x02,0x54,0x32,0x43,0x3b,
	0xa2,0xf0,0xc9,0x1a,0x46,0xc4,0xbb,0x16,0xa9,0x0e,0x0c,0xf8,0x31,0xd5,0xbf,0xc9,
	0xc3,0xce,0x5b,0xf2,0x57,0xa2,0xb9,0xf6,0xd3,0xc0,0x3c,0xe2,0xad,0xf6,0xc4,0x54,
	0x8e,0x20,0xbb,0xc1,0x6a,0xc1,0x99,0x05,0x9b,0xd1,0xca,0x99,0xa6,0x8f,0x03,0x76,
	0x01,0xb8,0x41,0x84,0x19,0xca,0x18,0x93,0xfc,0x7e,0x95,0x5b,0x83,0x55,0x84,0x3a,
	0x68,0xa6,0x18,0x67,0xed,0x0d,0x01,0x70,0x20,0xbb,0xfc,0x51,0x7c,0x46,0x5e,0x71,
	0x13,0x69,0x35,0xb1,0xac,0x33,0x1b,0x3f,0xbe,0xdb,0x08,0xc5,0xa8,0x47,0x5a,0x59,
	0xd5,0xad,0x1c,0xb9,0x5a,0x64,0xe2,0xb7,0x0c,0xfc,0x2f,0x57,0xdb,0x59,0xa0,0x01,
	0x95,0x9b,0x9c,0x9b,0x0d,0x30,0xa9,0x9b,0x82,0x9e,0xc8,0x0b,0x9c,0x17,0xa0,0xb8,
	0xe7,0x2a,0xe1,0x3a,0xd9,0x3d,0x1e,0x52,0x00,0x58,0xe6,0x4c,0x53,0x5c,0x22,0x25,
	0x84,0x8f,0x9f,0x37,0x8d,0x3f,0xd8,0x24,0xa7,0x49,0x72,0x3f,0xfd,0x70,0x4d,0x69,
	0xa0,0xde,0x7c,0x51,0x11,0x88,0x9a,0xa5,0xe1,0x9c,0xea,0x8d,0xcf,0x20,0xaf,0xe9,
	0xf6,0xee,0x61,0x19,0x32,0x6e,0xdf,0xb2,0x07,0x54,0xe6,0x2c,0x5c,0xb3,0x9d,0x65,
	0x16,0xcd,0x16,0xf4,0x8d,0xfb,0x20,0x47,0xc3,0x44,0x43,0x86,0xc1,0xe4,0xb7,0xbe,
	0x81,0xa3,0xc6,0xc6,0xf1,0x9d,0x8b,0x9c,0xc3,0x77,0xdb,0xba,0x4e,0xe6,0x3c,0x57,
	0x29,0x15,0x94,0x0d,0x82,0x7b,0x40,0xc7,0x91,0xcc,0x40,0x9c,0xc0,0x3a,0x5d,0x4e,
	0x12,0x8c,0xb6,0x7a,0x8f,0x19,0x6e,0x58,0x2e,0x4a,0xbd,0xb0,0xec,0xb7,0x6e,0xf7,
	0x8b,0x51,0x44,0xee,0x35,0x83,0x3a,0xe2,0x3c,0x82,0xb0,0x41,0x0e,0x6f,0xcd,0x6d,
	0x5d,0xb4,0x62,0xfd,0xa4,0x67,0x5e,0x82,0x9f,0x5a,0x57,0xf9,0xab,0xb0,0xc0,0x60,
	0x15,0x6b,0x05,0x6b,0x60,0x89,0xa9,0x34,0x19,0xa2,0xa1,0x54,0x9c,0x66,0x70,0x50,
	0x42,0xbe,0xdf,0x2d,0xf7,0x2f,0x68,0x03,0x6b,0xf0,0x42,0x3c,0x44,0x82,0x19,0xc5,
	0xd8,0x7b,0x5b,0x27,0x10,0x76,0xf7,0xe8,0xb4,0x33,0x8a,0xe1,0xd6,0x25,0xe5,0xad,
	0x04,0x5f,0x0a,0x0e,0x79,0x1b,0x35,0xec,0xf9,0xd2,0x3d,0x60,0xb6,0x43,0x83,0xdd,
	0x92,0x20,0x17,0x69,0x10,0xb0,0x07,0xd8,0x72,0x34,0x1e,0x30,0x1f,0xe5,0x7b,0x21,
	0x0d,0xf4,0x7c,0x01,0xd6,0x29,0x17,0xb2,0x06,0x75,0x37,0x1d,0xad,0x21,0x9d,0xc8,
	0xb6,0x29,0x40,0xcb,0x89,0x3e,0xa7,0x62,0x0b,0xf9,0x06,0x3a,0x7a,0x41,0x16,0x9a,
	0x74,0xf7,0xeb,0x17,0xfc,0x08,0x6c,0x94,0xf2,0xba,0x50,0x41,0x82,0x79,0x4d,0xd6,
	0xfe,0x72,0x7f,0x9e,0x81,0x0a,0xb4,0x1e,0x63,0x01,0x26,0xff,0x85,0x8d,0x1d,0xb7,
	0xa2,0x5a,0x17,0x08,0x5b,0x09,0x1d,0x8b,0x6e,0x8e,0x83,0x5d,0x3f,0xa9,0x01,0xc7,
	0xbc,0x0d,0x09,0xb8,0x79,0x74,0x57,0x70,0x29,0x68,0x75,0x70,0x15,0x5e,0xba,0x9d,
	0x50,0x24,0xdb,0xe3,0x53,0x3c,0xd7,0x3e,0x5d,0x17,0xa6,0xc6,0xc0,0xf9,0x3d,0x09,
	0xf5,0x8e,0xd3,0xe1,0xf9,0x14,0x36,0xb6,0x29,0xa3,0xa7,0x01,0x1c,0x59,0x42,0x64,
	0x97,0x6d,0x84,0xda,0x51,0xe9,0x65,0x7b,0x9f,0xc6,0x69,0x4d,0xba,0xfc,0x90,0x4a,
	0x73,0x58,0xc7,0x0d,0x59,0x2d,0xaf,0x73,0x6a,0xd9,0x9d,0x2e,0x4f,0x19,0x4f,0xbb,
	0x15,0x29,0x20,0x2a,0x90,0x10,0x8f,0x6a,0x78,0xc8,0xe1,0x54,0x26,0x41,0x73,0x75,
	0x52,0xf9,0xec,0x45,0x3c,0x28,0x50,0xa1,0x08,0x2e,0x0b,0xda,0xa0,0x62,0x22,0xfe,
	0x41,0xe2,0x22,0x61,0x02,0xae,0x4e,0x26,0x50,0x7a,0xd9,0xe0,0xd8,0xf4,0x2d,0x02,
	0x25,0x21,0x64,0x54,0xa4,0x36,0x5e,0x0b,0x70,0x48,0x26,0x02,0x17,0x17,0xb1,0xfb,
	0xfa,0xa0,0x45,0x81,0x2d,0x21,0xa5,0x74,0xf2,0xb3,0x8e,0x70,0xa0,0x14,0xc4,0xbb,
	0x6e,0xff,0x8b,0x73,0xe1,0x27,0xe7,0x67,0x78,0x4a,0x62,0x68,0xc8,0x61,0xa9,0xcb,
	0x61,0x8f,0x25,0xd8,0x7f,0xbe,0x23,0x7f,0x5a,0x4a,0xf9,0x62,0x0c,0x60,0x2e,0x1f,
	0x5f,0xf3,0x9f,0x97,0xc2,0xf9,0x7f,0x38,0x62,0x2f,0x9f,0x15,0xab,0xc0,0x15,0x0d,
	0x8b,0xbc,0xf5,0xa5,0xf2,0xe2,0xea,0x27,0xdd,0x92,0xba,0x45,0xe2,0x8d,0xaf,0xc1,
	0x03,0xe3,0x61,0x7a,0x10,0x6c,0xa5,0x1b,0x43,0x2e,0xe1,0xc7,0xa2,0xcf,0x18,0xc9,
	0xd5,0xd2,0x0c,0x0b,0xbf,0x1f,0x4c,0xae,0x6b,0xf0,0x54,0x13,0x3f,0xad,0x5a,0x70,
	0xd0,0x73,0xd0,0x54,0x6c,0x15,0x09,0xd2,0xd0,0x95,0x2b,0xcc,0xc2,0x0d,0xf8,0xfa,
	0xca,0x7f,0x73,0xa9,0x03,0x25,0xc6,0x6a,0x6f,0xff,0x69,0xe6,0x1d,0x5d,0x39,0x59,
	0xca,0xe3,0x7f,0x4c,0xb2,0xd5,0x56,0xad,0x92,0xaa,0x48,0x1d,0x1d,0xcd,0x21,0x0b,
	0x87,0x45,0x0f,0x00,0x86,0x99,0xb5,0xd5,0xe2,0x19,0x80,0x15,0xec,0x5f,0x42,0x0b,
	0xad,0xf3,0x9c,0x0e,0xea,0x7a,0x08,0x46,0x89,0x04,0xe3,0x76,0x78,0xca,0xb6,0xab,
	0x5a,0xb0,0x15,0xa7,0x2b,0x87,0x4f,0x45,0x77,0xd2,0xb6,0x3f,0xda,0x22,0x47,0xea,
	0x71,0xf7,0x63,0xea,0x2f,0xb9,0xb3,0xa4,0x96,0xf5,0xb6,0x14,0x9d,0xb1,0xd2,0x7e,
	0xcc,0x0c,0x1a,0x25,0xd1,0xce,0x09,0x80,0x78,0x26,0x80,0xf5,0x4f,0xb1,0xb3,0xea,
	0xa2,0x28,0x30,0xba,0xfb,0x77,0xb8,0xd2,0xeb,0xb4,0x59,0xce,0xfa,0x03,0x74,0x4a,
	0xfc,0x6d,0xae,0x42,0x0e,0x81,0xe0,0x66,0x2c,0xdd,0x62,0x0b,0x5a,0x66,0x47,0x00,
	0xbf,0xf3,0x5c,0x97,0x3b,0xc7,0x17,0x12,0xbc,0x36,0x32,0xce,0xe7,0x65,0xac,0x6b,
	0x3f,0x60,0xdf,0x7b,0xea,0xf9,0x01,0x9b,0x9e,0xcb,0x54,0xd4,0xfd,0x40,0x4b,0x50,
	0x28,0x9a,0xe6,0xc8,0xc6,0x8e,0x9f,0x2d,0x38,0x51,0x5c,0x14,0x91,0xc4,0x38,0x97,
	0x1b,0xc2,0x38,0x7c,0x13,0x23,0x8a,0xec,0x59,0x3e,0xb1,0x59,0x8d,0xf9,0x3a,0xa8,
	0x31,0x5d,0x4b,0x60,0xa3,0xdc,0xf0,0x65,0x1a,0x84,0xcd,0x76,0x7e,0x4b,0xbc,0xc9,
	0x3d,0x92,0xab,0x26,0xd4,0xd9,0x7b,0x47,0x41,0x3c,0xfb,0x41,0xb8,0xb1,0x58,0x3e,
	0x0f,0x0f,0xaa,0x26,0x97,0x84,0x6e,0x09,0x31,0xfc,0x70,0x1d,0x49,0x98,0x6b,0x37,
	0xfd,0x10,0x09,0x12,0x60,0x35,0x32,0x0e,0xc0,0x35,0x85,0xed,0x8a,0x7c,0x14,0x83,
	0xba,0xc6,0xb0,0xc1,0x81,0xa0,0xa8,0x7d,0x42,0x12,0xcc,0xd3,0x57,0x06,0x8c,0xd8,
	0xcf,0x53,0xe0,0x85,0x88,0x85,0xd1,0xba,0xc3,0xdb,0x29,0xab,0x5d,0xd3,0xb0,0x70,
	0x21,0xb4,0x34,0x7f,0xfb,0x59,0xb4,0xe3,0x3a,0xa8,0xcc,0x37,0x5f,0x35,0x57,0x56,
	0xed,0x4f,0xf4,0xd8,0x93,0x38,0x17,0xba,0xdc,0x0f,0xce,0xad,0xc2,0x13,0x0b,0x3e,
	0x2a,0xb9,0x98,0x34,0xb3,0xbf,0x85,0x04,0xce,0x1e,0xca,0x15,0x27,0xfe,0x5b,0x75,
	0x13,0xea,0x4e,0x08,0xae,0xbf,0xcb,0x39,0x6b,0x38,0x97,0x6f,0x27,0x7c,0x0b,0xb1,
	0x3e,0x47,0x98,0x57,0x58,0xb1,0x91,0xad,0x8b,0xa7,0x1a,0xd6,0x93,0x53,0x23,0xa3,
	0x48,0xce,0x25,0x72,0x54,0xac,0x48,0xd1,0x6d,0x7b,0xdc,0xc5,0xb4,0x43,0x72,0x0d,
	0xf1,0xce,0x89,0x23,0xd7,0xf2,0x40,0x38,0xde,0x17,0x96,0x5c,0xf8,0xfa,0x3f,0xba,
	0xa7,0xd7,0x59,0xb8,0xe6,0x29,0x92,0x11,0x42,0xc9,0x24,0xb0,0x4f,0x02,0xd3,0x85,
	0xd7,0xda,0x30,0xee,0x38,0xa2,0x2d,0xbc,0x83,0xd8,0x30,0x26,0x9a,0x67,0x96,0xc0,
	0x57,0xa0,0x76,0xf3,0xec,0x88,0x47,0xe8,0x88,0xde,0x6b,0xab,0x35,0xd9,0x4a,0x51,
	0x64,0x5f,0x2b,0xcb,0xa5,0xc7,0x54,0x3c,0xa0,0xda,0xae,0x49,0x25,0x74,0x47,0x5a,
	0x05,0x41,0x36,0x8f,0x4d,0xf5,0x2e,0x17,0x36,0xae,0x06,0xe0,0x0b,0xde,0x2f,0xf2,
	0x35,0xf8,0x05,0xa5,0xc9,0xfd,0x7f,0x70,0x19,0x03,0x89,0xaa,0xc8,0x00,0x01,0xfb,
	0x53,0x01,0x38,0x80,0x70,0x52,0xbd,0x4b,0x23,0xc4,0xfc,0x3a,0x8e,0xc3,0x47,0x20,
	0x0a,0x72,0x75,0x55,0xa6,0x06,0x6e,0x7e,0x5a,0x72,0x95,0xb2,0xe1,0x83,0x44,0x83,
	0x57,0x3d,0xda,0xcf,0xc3,0x2a,0x7e,0x9d,0xe6,0xff,0xd5,0xe9,0x93,0xd6,0xe3,0xe9,
	0x78,0x5b,0xd4,0x9c,0xf9,0xc8,0x5c,0x86,0x64,0xfc,0xa0,0x6b,0x71,0xc7,0xe3,0xb5,
	0xe6,0x9a,0x3a,0xf3,0xee,0x74,0xe9,0x86,0x14,0x27,0x4b,0x71,0x59,0xcc,0xbf,0x14,
	0x47,0x84,0x43,0xbe,0x50,0x8b,0x0f,0x7c,0x1b,0x38,0x1b,0x61,0xbb,0xe4,0x56,0xf9,
	0x29,0xd0,0x2a,0x0e,0x7e,0xad,0xa4,0x65,0x67,0x89,0x87,0x6b,0x04,0x1c,0x53,0x0c,
	0x86,0x02,0x9d,0x33,0xce,0xd5,0x47,0xa1,0x46,0x33,0x6a,0x85,0xd3,0xf6,0x16,0x0f,
	0x10,0x9c,0xa7,0x72,0x94,0x1f,0xa0,0x29,0x70,0x62,0xea,0x11,0x71,0x82,0xfd,0x69,
	0xaf,0x40,0x7c,0x46,0x88,0xfd,0xa9,0x21,0x74,0xe2,0x42,0x18,0x6c,0x70,0xb3,0x14,
	0x1a,0x24,0x81,0xb9,0xf6,0xc7,0x3b,0x2f,0x38,0x68,0x45,0x75,0x0b,0x20,0xdd,0x19,
	0xfa,0xc1,0xba,0xdf,0x1f,0x16,0xa2,0xa7,0xc9,0x9e,0xf5,0xa5,0xb9,0x70,0x5e,0xac,
	0xa6,0x44,0x38,0x93,0x2c,0xe2,0x9a,0xe0,0x21,0x16,0x6a,0xdf,0x47,0x15,0x71,0x1e,
	0x24,0xca,0x7e,0xac,0xbd,0xe1,0x97,0xd6,0xe8,0x0d,0x09,0x1b,0x10,0xe0,0x26,0xf1,
	0x1e,0x7a,0x55,0x28,0xdd,0x70,0xfa,0x83,0xfe,0x86,0x92,0x93,0x69,0x62,0x72,0xea,
	0xd5,0x8a,0x53,0x91,0x3e,0x34,0x07,0x71,0x50,0xc9,0x5f,0x5f,0x2b,0xe6,0x38,0xf5,
	0x5d,0xce,0x2d,0x35,0x26,0x81,0x99,0x97,0xd8,0x84,0x64,0x59,0xbd,0xfa,0x48,0xaf,
	0x12,0x1d,0x8b,0x94,0x79,0x85,0x43,0xc4,0x86,0xc8,0xa4,0x76,0xd7,0xd0,0x55,0x5b,
	0x4b,0xe6,0x59,0x9e,0x04,0x5f,0x66,0xa0,0x49,0xf3,0x34,0x7d,0x7d,0xc9,0x5f,0x17,
	0x82,0x79,0x33,0x4c,0xde,0x6d,0xae,0xc8,0xa0,0x13,0x83,0x78,0x50,0x79,0x6f,0x3a,
	0x89,0xb7,0x23,0x99,0x29,0x93,0xdf,0x0d,0x67,0xb3,0x40,0x8c,0x61,0x85,0x8e,0x0d,
	0x87,0x21,0x6f,0x3d,0x31,0xb1,0x16,0xc1,0x93,0xc5,0x73,0xe9,0xf5,0x09,0x43,0x49,
	0x74,0xc1,0xbf,0xa9,0x84,0x9f,0xfd,0x44,0xa6,0xd0,0x4f,0xd1,0x87,0x96,0x50,0x2d,
	0x7d,0xe2,0xf6,0x4f,0x2f,0x5b,0x19,0xd8,0xcc,0xd5,0xa7,0x68,0x4f,0x96,0x9d,0xb6,
	0xd3,0x81,0x03,0x6a,0x6b,0x85,0xe8,0x79,0x05,0x21,0x91,0x1f,0x8b,0x11,0x11,0x52,
	0x37,0x67,0x9e,0x29,0xcb,0x91,0xea,0x2b,0x3a,0x8c,0x29,0x76,0x0c,0x76,0xf0,0x7c,
	0x11,0x4e,0x66,0x25,0xf8,0x74,0x90,0x07,0x34,0x67,0x64,0xfc,0xba,0x47,0x44,0x80,
	0x2e,0x49,0xf8,0x80,0x1e,0x88,0x15,0x77,0xff,0xcd,0xbb,0x1e,0x59,0x04,0x2b,0x7c,
	0xcc,0x87,0x5a,0xb2,0xe8,0xdc,0x10,0xed,0xf1,0xc2,0x30,0xf5,0xb6,0x66,0xdf,0x07,
	0x8d,0x0d,0xac,0x2f,0x9b,0xa9,0xe4,0x24,0x86,0x79,0xd5,0x85,0xe9,0x3f,0x06,0xb6,
	0xdd,0x3e,0x1c,0xab,0xe5,0xfe,0xf1,0x00,0x92,0x55,0x2d,0xf0,0x3b,0xbf,0x69,0xe2,
	0x9e,0xc5,0x79,0xe6,0xe5,0xae,0x9b,0x59,0x07,0xdb,0x80,0xe8,0x89,0x3f,0xe1,0x7d,
	0xa9,0x25,0x06,0x18,0x71,0x98,0x30,0x7e,0x04,0x5c,0x3f,0x5d,0xf5,0x97,0xa2,0xae,
	0xde,0x05,0xe9,0x53,0xda,0xcd,0x85,0xdf,0x7e,0x81,0x94,0x93,0x4d,0x04,0xc3,0x80,
	0x9a,0x78,0x05,0xbd,0x4a,0xbf,0x0c,0x06,0x45,0xc8,0x79,0xae,0x91,0xcc,0x2d,0x72,
	0xa2,0xed,0x49,0x85,0x97,0x24,0x5f,0x62,0xbb,0x45,0xa0,0x88,0x75,0x76,0x90,0xe5,
	0x05,0xe2,0x08,0xbd,0x5c,0x13,0x71,0x0d,0xa5,0xbc,0xfe,0x32,0xad,0x42,0x51,0x2f,
	0x3a,0xae,0x0c,0x01,0x20,0x5f,0x5f,0xb0,0x73,0xa7,0x9b,0x7c,0x6f,0x9d,0x70,0xcc,
	0xb5,0xf7,0x8e,0x10,0x2a,0x87,0x3d,0x2b,0x8c,0x2b,0xa5,0x42,0xe9,0x59,0xe0,0x29,
	0xb3,0x06,0x48,0x7c,0xd5,0x90,0x85,0x3c,0x31,0xb3,0x9b,0x28,0xe5,0x41,0x95,0xa9,
	0xeb,0x21,0x62,0xa2,0x58,0xab,0x51,0x09,0x20,0x92,0x32,0xbf,0xa1,0xfd,0x1b,0x0f,
	0xb5,0xa0,0x98,0x16,0x75,0x2a,0x92,0x14,0x5b,0x98,0xa0,0x58,0x5b,0xad,0x08,0x99,
	0x45,0xe2,0xe3,0xb9,0x92,0x1a,0x9c,0x4b,0xde,0xe5,0x07,0xec,0xd0,0xab,0x1f,0x3c,
	0xeb,0x98,0xcf,0x89,0x8b,0x75,0xed,0x2b,0xc9,0x09,0x9f,0xae,0x30,0x59,0xb3,0x38,
	0x3f,0x10,0x7d,0xd1,0xa1,0x96,0x18,0x0d,0x65,0x61,0x06,0xc3,0x5e,0xcd,0x76,0xc4,
	0xc6,0x5b,0xfb,0x7b,0x31,0xe5,0x1c,0x79,0xa7,0x3d,0x57,0x8c,0x12,0x01,0x7a,0x55,
	0x19,0xb7,0xf3,0x68,0xcf,0xa4,0x0d,0x6a,0x0d,0x42,0x2f,0x84,0xcd,0xe2,0xf3,0xb9,
	0x9b,0x90,0xcf,0x80,0x1e,0xe5,0x0f,0xa1,0x97,0x99,0xd6,0xa9,0x64,0xe7,0xb3,0xd1,
	0x14,0x35,0x2e,0xaa,0x3d,0xff,0xac,0x7e,0x02,0x8e,0x60,0xed,0xa5,0x2c,0x0f,0xa4,
	0x0c,0x95,0x98,0x1c,0xee,0x77,0xda,0x71,0xa8,0xa6,0xb9,0xae,0xd1,0xae,0x08,0x5a,
	0x42,0xa0,0x2a,0x21,0xdf,0x84,0xa7,0x6b,0x2f,0xca,0xec,0x0f,0xf0,0xb3,0x0b,0x83,
	0xfb,0x2b,0x48,0xa6,0xeb,0x20,0x35,0x0e,0x7c,0x10,0x21,0x0f,0x64,0xfb,0x9e,0x2a,
	0xe6,0x00,0x64,0x94,0x3a,0x0e,0xe4,0x54,0x78,0xab,0xd8,0xdd,0x24,0x9e,0x40,0x29,
	0x1f,0x0c,0xf4,0xaf,0x50,0x62,0x6c,0x03,0xf0,0x49,0xfc,0x12,0xe1,0x7b,0xff,0x00,
	0x83,0x20,0x50,0x24,0x99,0x11,0xba,0x9c,0x2e,0xd0,0x2f,0xa5,0x06,0x18,0xb4,0x22,
	0x22,0x49,0x90,0x16,0x8c,0x4c,0x51,0xb4,0x83,0xff,0x7e,0x30,0xf8,0x99,0x9a,0xd9,
	0x12,0x2d,0xb1,0x32,0x77,0x57,0xb1,0xe5,0xa5,0xfd,0xd2,0x4f,0xd3,0x34,0x7d,0x1c,
	0x97,0xe9,0x85,0x98,0x33,0x88,0x54,0x8b,0xec,0x90,0xd8,0x52,0x65,0xc8,0xb2,0x6d,
	0x82,0x1e,0xfd,0x0a,0x16,0x96,0xec,0xa9,0xb7,0xf4,0xe2,0xfe,0xb5,0x37,0x27,0xc6,
	0x59,0x72,0xfb,0x80,0xdb,0xe3,0x7b,0x4f,0xea,0xdd,0xa4,0xec,0x52,0xe8,0xc3,0x94,
	0x97,0xb1,0xcd,0x7a,0x6e,0x20,0x65,0x28,0x13,0xd2,0x8a,0xd2,0x12,0x0c,0x6d,0xe5,
	0x81,0xac,0xd4,0x42,0xba,0xc8,0x39,0x93,0x3f,0x21,0xeb,0x64,0xfa,0x04,0xdb,0xda,
	0x01,0x11,0xf3,0x7b,0xeb,0xb9,0x6d,0x6a,0x35,0x7a,0xba,0xd7,0xfa,0xaa,0xda,0x39,
	0x58,0xf1,0x9f,0xd6,0xbb,0xa5,0x97,0xb0,0x4a,0x05,0x13,0x7c,0x9a,0x29,0x0e,0x57,
	0x80,0xae,0x7e,0x69,0x63,0x11,0xa1,0xcf,0xcd,0xe4,0x32,0x2a,0x02,0xf3,0xbb,0xc7,
	0xb9,0x9e,0xbf,0x83,0xee,0x92,0xe7,0x19,0x24,0xab,0xa8,0x46,0xd7,0xf3,0xe8,0xd5,
	0x53,0x4b,0x67,0xe2,0x05,0xc9,0xc4,0xff,0xa7,0xa9,0x50,0x7c,0xde,0x03,0x29,0x9e,
	0x12,0x44,0xd5,0x9e,0x5d,0x1a,0x1f,0xb5,0x90,0xb9,0x74,0x8b,0x63,0x9f,0x0f,0x1f,
	0xd0,0x05,0x37,0x37,0xb1,0x15,0x14,0x39,0x7b,0xd8,0x28,0x22,0x4f,0x59,0x56,0x5a,
	0x61,0x3d,0xc8,0x97,0xcc,0xbd,0x54,0x8e,0x7d,0xcc,0xed,0xeb,0xb1,0x81,0xc4,0x89,
	0x16,0x25,0x99,0x3e,0x05,0xd9,0xe1,0x8c,0xdd,0xeb,0x8e,0x35,0x6a,0x78,0x5a,0x6f,
	0xfa,0xd5,0xb0,0xed,0xf7,0x96,0x6f,0x75,0xab,0x95,0x5b,0x23,0x26,0x32,0xbc,0x12,
	0x2c,0x0b,0xfc,0x68,0xa0,0x5e,0xb7,0xe8,0xc5,0x38,0xb8,0x99,0xd6,0x9e,0x1a,0x79,
	0x42,0xca,0x29,0x13,0x88,0x44,0x3f,0x66,0x49,0x19,0x49,0x83,0x6d,0x78,0x70,0x19,
	0x99,0x6f,0xe9,0x18,0xd3,0xee,0x6f,0x27,0xb6,0x0e,0x92,0x1c,0x34,0xb2,0xd1,0x35,
	0xb4,0x1f,0x27,0xb2,0x05,0xa7,0xaf,0xf4,0xb2,0x4d,0x1b,0x96,0x8d,0xb6,0x60,0xe6,
	0x3f,0xbe,0xb2,0x4b,0x25,0x78,0xe7,0xb0,0x4c,0xe5,0x3e,0xe4,0x16,0xe7,0x13,0xbd,
	0xc7,0x5d,0xbc,0xe4,0x11,0xcc,0x63,0xd6,0x76,0x0b,0x05,0x14,0xf8,0x67,0x43,0x61,
	0xb5,0x34,0xf2,0xc5,0x65,0x3c,0xf7,0x14,0x8c,0x5e,0xe0,0x63,0x11,0xe5,0xf8,0x14,
	0xea,0x90,0x36,0xcb,0x1c,0x6a,0x49,0xc7,0xf4,0x7b,0x44,0x77,0x3a,0x5e,0x35,0xd9,
	0x77,0x83,0x2b,0x1f,0x1a,0x7f,0x58,0xb7,0x83,0xa6,0xfe,0xfd,0x88,0x62,0x68,0x88,
	0xc6,0x0d,0x45,0x98,0x7b,0x92,0xd6,0x89,0xf0,0x89,0xdd,0x1b,0x72,0xf3,0xd7,0x8d,
	0xfb,0x0c,0x8c,0xd4,0xe0,0x0e,0x04,0xdd,0x81,0x07,0x27,0xdf,0x13,0x8d,0x05,0xc5,
	0xaf,0xb7,0xb6,0xd0,0xe2,0xf1,0x56,0x4b,0xd4,0x33,0x71,0x07,0xf4,0x3b,0x70,0xf5,
	0xc5,0xfd,0x27,0x07,0xdf,0x1e,0xff,0xcd,0x5f,0x58,0x86,0x32,0x1f,0x22,0xa0,0x5b,
	0x9b,0x25,0x43,0x86,0xc2,0x59,0x6f,0x89,0xf7,0x3a,0x7c,0x28,0xbb,0x4c,0xc0,0xd5,
	0xeb,0x9c,0x7b,0x73,0x54,0xd2,0xb8,0xd9,0xa8,0x3d,0x35,0xf7,0x3e,0x8b,0x4a,0xf0,
	0xb4,0x04,0x60,0xcf,0x3f,0xb8,0xc7,0xd9,0x25,0x57,0xea,0x42,0xf4,0xc6,0x21,0xbd,
	0xa8,0xff,0x03,0x35,0xba,0x48,0x94,0x3c,0x75,0xe5,0x33,0xae,0x30,0x7b,0x9b,0x6f,
	0x2d,0xc5,0x51,0x86,0xd8,0xd2,0xfc,0xd5,0x44,0x79,0xa0,0x9c,0x79,0x4a,0x96,0xb9,
	0x83,0x06,0x20,0x4a,0x41,0x61,0x7d,0x84,0x0e,0x64,0x7c,0x81,0x4a,0x0c,0x87,0xc2,
	0xd0,0xbd,0xb9,0xdb,0x90,0x61,0x0b,0xf6,0x50,0x17,0xa3,0x9a,0x38,0x20,0x76,0xb6,
	0x56,0x52,0x9a,0xa1,0xcf,0x8e,0x14,0x0c,0x82,0xed,0x56,0xf9,0x51,0xd9,0x84,0xac,
	0xc3,0xf0,0x43,0x4a,0xd6,0x9b,0xe3,0xbd,0xc0,0x87,0x6e,0x0f,0xe5,0x12,0x57,0xd7,
	0x22,0x3e,0x0e,0xea,0x88,0x35,0x15,0x57,0x32,0x3e,0x6f,0x5f,0x35,0x46,0x3f,0x31,
	0xa3,0x8f,0x78,0xd7,0x65,0xec,0xdd,0x62,0xda,0x06,0xcf,0x9c,0xa1,0xfa,0x25,0xff,
	0x45,0xbb,0x66,0xf5,0x4b,0x2e,0xba,0xb4,0x98,0x82,0xdc,0x75,0x73,0xe2,0xee,0xc9,
	0x12,0x7f,0x4c,0x40,0xa5,0x46,0x63,0xdb,0xb2,0x90,0xb9,0x74,0xad,0x99,0xe3,0x4c,
	0x79,0x7e,0x53,0x6b,0x2e,0x77,0x92,0x3e,0x1a,0xe2,0x7c,0x96,0x11,0x83,0x35,0x51,
	0x9d,0x4c,0x28,0x69,0xad,0xa5,0x03,0x6e,0x3a,0xf7,0xba,0x5c,0x33,0x4f,0xc0,0xd3,
	0x38,0x09,0x0f,0x69,0x99,0xf9,0x5c,0x21,0x73,0x28,0x9a,0xfa,0x2a,0xd4,0x70,0xba,
	0x72,0x09,0x7f,0x19,0xe8,0xcf,0xbc,0x60,0x77,0x97,0xb4,0x07,0x18,0x49,0xe6,0x72,
	0xb5,0xd5,0x2b,0x0b,0x58,0x85,0xcc,0xbc,0x80,0x70,0x3e,0x5b,0xc4,0x39,0x9d,0x50,
	0xa8,0xb0,0x2f,0x5a,0xb0,0x7f,0xd3,0x1b,0x0b,0x56,0xf5,0x5d,0xeb,0xf5,0x81,0xfa,
	0xbb,0xd7,0x9c,0xfc,0x1c,0x08,0x33,0x88,0x4b,0x9b,0xcd,0xb7,0xab,0x87,0x5d,0x69,
	0x4f,0x39,0x2b,0x6f,0x53,0x54,0x1f,0x22,0x03,0x21,0x3c,0x91,0xf4,0x33,0xd0,0xcf,
	0xc5,0x0a,0x1d,0xa7,0xb9,0x67,0xf9,0xc1,0x09,0x0a,0x25,0x03,0x28,0x1f,0x47,0x65,
	0xd2,0xd3,0xf4,0x07,0x4f,0x52,0x99,0xc2,0xbd,0x7e,0x7e,0x14,0xe1,0x7d,0x83,0x33,
	0x62,0x6d,0x26,0x8f,0xc1,0x22,0x33,0xf9,0x01,0x11,0xbb,0x38,0x5d,0x9c,0xa4,0xf2,
	0x48,0x37,0xd9,0x87,0x4c,0x5a,0x44,0x88,0xc0,0x83,0xd0,0xb6,0xcd,0xf2,0xe7,0x33,
	0x6b,0x7d,0x62,0x26,0x78,0x63,0xc7,0x55,0x16,0x24,0xf6,0x26,0x9d,0x02,0xfe,0x94,
	0x66,0x57,0x85,0xbe,0xc3,0x6b,0xe7,0xe2,0xba,0x4f,0x82,0x25,0xd5,0x6f,0x35,0xcf,
	0xd9,0x7e,0x98,0x37,0x40,0x07,0x57,0x61,0x59,0xdf,0x44,0x08,0xc0,0x49,0x69,0x02,
	0xc1,0x32,0xec,0xdd,0x94,0xeb,0x11,0xee,0xc3,0x25,0x17,0x31,0xa9,0x1c,0x07,0xfa,
	0x6c,0xf3,0x1e,0x16,0x79,0x16,0x50,0xaa,0x8d,0xdf,0x90,0xbc,0x7e,0x34,0x44,0xb3,
	0x75,0xff,0x65,0xf4,0xfc,0x78,0x71,0x7c,0x7a,0xe8,0x02,0xb5,0x89,0x72,0x33,0x72,
	0x7d,0xd0,0xd3,0x12,0xa1,0xc6,0x79,0x46,0x20,0x24,0x04,0x50,0xf7,0x77,0x01,0xb1,
	0xa1,0x01,0x72,0x4d,0x0f,0x97,0xee,0xb5,0xff,0x23,0x98,0x0c,0x02,0x35,0x93,0xd3,
	0x32,0xcf,0x77,0x5f,0x1e,0x95,0x09,0xce,0xf4,0x11,0xac,0xf1,0x27,0xf4,0xea,0x03,
	0xeb,0xda,0xd3,0x70,0xf3,0x93,0x7a,0xc3,0xfc,0xcf,0x2f,0xf0,0xc1,0x11,0x72,0x4c,
	0x84,0x85,0xba,0x16,0xb3,0x67,0x1e,0xaa,0x7b,0x0a,0xc2,0x9d,0xf2,0x30,0x0b,0x85,
	0x86,0x6a,0x32,0xb4,0x61,0x8f,0x1a,0x91,0xbd,0x2a,0xb4,0xe9,0x82,0xf5,0x92,0xff,
	0x89,0xd9,0x56,0x9e,0x67,0x82,0xac,0x3b,0x18,0x9b,0xdf,0x7e,0x31,0xeb,0x96,0x02,
	0x6f,0xd8,0x4d,0xde,0xf6,0x2a,0xec,0x87,0xdc,0xe9,0x77,0x64,0xc6,0x09,0x6d,0xd7,
	0x66,0x39,0x1e,0xed,0x53,0x26,0x54,0xfc,0x76,0x70,0xa2,0x96,0x0f,0xbf,0x64,0xa7,
	0x2a,0x0b,0x81,0xcf,0xc2,0xf1,0x87,0x6c,0xae,0xd8,0x0c,0xea,0x8f,0xff,0x22,0xb6,
	0x59,0xf4,0x1a,0x8f,0xd7,0x04,0x72,0xf3,0xe7,0x21,0xec,0xf3,0x71,0x7c,0xf9,0x74,
	0x62,0x03,0x7c,0xf5,0x28,0xc3,0x44,0x4b,0x3b,0x5f,0xb5,0x09,0xdb,0xc5,0x89,0xb9,
	0xfa,0x67,0xd5,0xee,0x13,0x32,0xe0,0x02,0x2e,0xe1,0xb8,0x52,0xaf,0xfb,0xc8,0xf1,
	0x5d,0x2d,0xa6,0xe7,0x2f,0xbc,0xf0,0x4b,0x2b,0x9b,0x8b,0xd9,0x94,0x29,0x22,0xbd,
	0x55,0x4a,0x64,0x5e,0x32,0x9d,0x07,0xd3,0xa6,0x41,0x6e,0x74,0xed,0xe3,0x74,0x55,
	0x32,0x86,0xfa,0x92,0x40,0xb8,0xd2,0xbe,0xfa,0x34,0xa8,0xa8,0xa4,0x47,0x6d,0xd5,
	0x13,0xa3,0x7d,0x0b,0x5c,0x70,0x9d,0xb5,0x7d,0x3c,0x9e,0xfe,0xf9,0x5f,0x86,0x52,
	0xe5,0xd5,0xfa,0x96,0x37,0xa0,0x6c,0x46,0xee,0xc8,0xd6,0x3d,0x07,0xf6,0x92,0xf5,
	0x31,0x3a,0x9e,0x44,0x66,0x8e,0xeb,0xfc,0x7b,0xee,0xe1,0x39,0xbe,0xe3,0x7b,0xa4,
	0x91,0x13,0xad,0x98,0xba,0xd1,0x09,0x53,0xc0,0x91,0x47,0x3f,0x50,0xb9,0x79,0x8d,
	0xf7,0xae,0x96,0xce,0xa9,0xc0,0xfb,0x4e,0xb1,0x66,0x89,0x30,0x95,0xf2,0xbd,0x21,
	0x91,0xd2,0x7e,0x8b,0x4c,0x58,0xae,0x37,0x1a,0x7b,0x8e,0xb1,0x8f,0xfd,0x93,0xd9,
	0x6b,0x61,0x49,0xf5,0x87,0xcd,0x78,0xbf,0x65,0xe7,0xa7,0xf7,0x2a,0x77,0x11,0x80,
};


/* 317-0024 FD1089B */
static const UINT8 key_0024[0x2000] =
{
	/* data */
	0x31,0xce,0xea,0x22,0x7d,0x86,0x74,0x6a,0xd1,0x28,0xa3,0x5a,0x00,0xac,0x00,0x3a,
	0xa5,0xaf,0x7a,0x2c,0x08,0xc9,0xf8,0x6f,0x00,0xec,0x82,0x71,0x63,0x96,0xd0,0xcd,
	0x49,0x0d,0x71,0x26,0x15,0xb8,0xd2,0xd8,0x25,0x79,0xa2,0xe9,0xce,0x9c,0x46,0x8e,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0xd7,0x1e,0x88,0xde,0x90,0x3d,0x5f,0xd3,0x47,0x32,0xe5,0x6d,0x29,0xfd,0x36,0x5b,
	0x63,0xe8,0x24,0xc3,0x40,0x5e,0x70,0x5e,0xee,0xd4,0x4c,0x4d,0x39,0xc7,0x73,0x68,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0027 FD1089B */
static const UINT8 key_0027[0x2000] =
{
	/* data */
	0xce,0x92,0x01,0x6b,0xd3,0x3c,0x3a,0xcb,0xaa,0x57,0xdc,0x12,0x8d,0x7b,0x6e,0x8c,
	0x91,0x33,0xb0,0x66,0x78,0x27,0x80,0xe8,0x6e,0x52,0xe7,0xf3,0x63,0x4c,0x3b,0x94,
	0x85,0x6b,0x4b,0xd9,0x70,0x8c,0xa2,0xa7,0x93,0x8c,0x14,0xb2,0x42,0xdd,0x56,0x45,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0x72,0xc8,0x6c,0x47,0xf4,0x4f,0xc7,0xc2,0xb2,0x33,0xd0,0xc2,0xee,0x4a,0xf0,0xc8,
	0xcf,0x6f,0x8a,0x15,0x0d,0x7a,0xca,0xf3,0x84,0x09,0x57,0x4f,0x5c,0xcf,0x7c,0xd4,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0028 FD1089A */
static const UINT8 key_0028[0x2000] =
{
	/* data */
	0xce,0x92,0x03,0x6b,0xd3,0x3c,0x3a,0xcb,0xaa,0x57,0xdc,0x10,0x8d,0x7b,0x6e,0x8c,
	0x91,0x31,0xb0,0x66,0x78,0x27,0x80,0xe8,0x6e,0x50,0xe7,0xf3,0x61,0x4c,0x3b,0x94,
	0x85,0x6b,0x4b,0xd9,0x72,0x8c,0xa2,0xa7,0x93,0x8c,0x14,0xb2,0x40,0xdd,0x56,0x45,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0x72,0xc8,0x6c,0x47,0xf4,0x4f,0xc7,0xc2,0xb2,0x31,0xd0,0xc2,0xee,0x4a,0xf2,0xc8,
	0xcf,0x6f,0x8a,0x15,0x0d,0x7a,0xca,0xf1,0x84,0x09,0x57,0x4f,0x5c,0xcf,0x7c,0xd4,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0033 FD1089A */
static const UINT8 key_0033[0x2000] =
{
	/* data */
	0xe7,0x57,0x69,0xa4,0x29,0x6c,0x18,0xa3,0x64,0x73,0x13,0x55,0x16,0x59,0xaa,0x2c,
	0x51,0xb4,0x82,0xce,0x26,0x3f,0xfb,0x44,0xaa,0xac,0xc9,0x76,0x61,0xc6,0x55,0x6e,
	0xbd,0x9e,0x8a,0x33,0x46,0x6a,0x4e,0x20,0x8a,0xca,0x33,0x48,0x84,0x3e,0x69,0x7c,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0xf3,0x4e,0xd0,0xeb,0x26,0xd6,0x7d,0xaf,0x37,0x2f,0x78,0xd9,0xb4,0xad,0x7f,0x9d,
	0x57,0x96,0xb0,0xaa,0xd5,0x38,0xfe,0xc7,0x20,0x89,0x91,0x8c,0xc4,0x19,0x3f,0x1d,
	0x5a,0x29,0xb2,0xe8,0x22,0x6f,0xed,0xbe,0x2c,0xc7,0xd3,0x5d,0x7e,0xbc,0xeb,0x30,
	0x19,0x89,0x73,0xa5,0xc7,0xbd,0x3d,0x55,0xbe,0x5d,0x92,0xe8,0xda,0x3c,0xb2,0xa3,
	0x90,0x81,0x72,0x6c,0x4f,0x36,0x13,0xe2,0xd0,0x1e,0x57,0x93,0x5e,0xe2,0x86,0xdd,
	0x51,0xce,0xfe,0x67,0x60,0x09,0x60,0xa7,0x48,0x71,0x62,0x4c,0xe5,0x55,0x0f,0x39,
	0x39,0xf2,0xfd,0x14,0xba,0x82,0x46,0xb1,0x6e,0x8d,0x6a,0xd1,0x31,0xb1,0xce,0xfa,
	0x55,0x7b,0xeb,0xd8,0x60,0xec,0x6b,0xbe,0xf2,0xe6,0x2a,0x48,0xe9,0xd0,0x32,0x5c,
	0x7a,0xb7,0xbd,0x61,0x76,0x14,0x49,0x41,0x75,0xb8,0x25,0xba,0x80,0x56,0x9f,0xec,
	0xf1,0x0e,0x68,0xb7,0xc5,0x2d,0x34,0x76,0x24,0x05,0xab,0x2f,0xb3,0x4f,0xe9,0x1d,
	0x7a,0x8d,0x99,0x6f,0x97,0x06,0xd5,0xa6,0x4f,0x8e,0x7d,0xf2,0x0e,0x72,0xa0,0x2f,
	0xb6,0x63,0x68,0xba,0x60,0xf1,0xe2,0x4c,0xbc,0xe2,0x44,0xc7,0xa5,0x68,0xb4,0xeb,
	0x3b,0xd4,0xda,0x78,0x1c,0x19,0xe9,0x97,0x96,0x32,0xa1,0x39,0x75,0xe3,0xc2,0x50,
	0xdf,0x04,0xe2,0x36,0xb9,0xfe,0xa8,0x59,0xed,0x5f,0x22,0xac,0x1f,0xa0,0xbf,0x15,
	0x50,0x7c,0xb7,0x97,0xbf,0xfe,0xcc,0x6c,0xde,0x02,0xb6,0xb9,0x7b,0xc7,0x2d,0x85,
	0x29,0xb8,0xfc,0x4f,0xc7,0x62,0xf6,0x05,0xc2,0x32,0xd8,0xa6,0x53,0x65,0xf6,0x82,
	0x2d,0xf6,0x83,0x12,0xe1,0x87,0xa1,0xf0,0x3e,0x1c,0x02,0x57,0x5a,0xe6,0x3b,0x0f,
	0x96,0x1b,0x71,0x16,0xbb,0x34,0xca,0x64,0x93,0xbb,0x24,0x82,0x09,0x5c,0x39,0xd6,
	0x77,0xda,0xdb,0x9a,0x81,0x69,0xa9,0xfa,0xf8,0x2d,0x32,0x50,0x44,0xd4,0x75,0x97,
	0xa2,0x86,0x22,0x4d,0xb0,0xb8,0x78,0x99,0x38,0xb0,0x69,0x71,0xa8,0x0a,0xe8,0x47,
	0x23,0x88,0xa9,0xac,0x99,0xe0,0x86,0x60,0xeb,0x91,0x32,0xf0,0x56,0xd6,0xa1,0x79,
	0x71,0xd1,0x56,0x29,0xd8,0x26,0xb4,0xcb,0x9f,0xde,0x1c,0x45,0x5e,0x73,0x69,0x4b,
	0x13,0xfd,0x91,0xba,0x1d,0xb2,0xd3,0xaa,0x98,0x6e,0x00,0x02,0x97,0xe7,0x09,0x88,
	0x93,0x5a,0x95,0xa8,0x67,0x39,0xde,0x52,0x3c,0x4f,0x57,0xcb,0xcc,0xf6,0xbb,0xe7,
	0x97,0x4a,0x09,0x99,0x32,0x59,0x0d,0x34,0x1f,0x74,0x7a,0x09,0x60,0xd1,0xbf,0x66,
	0xca,0xc1,0xfd,0x68,0xbc,0x4e,0x17,0xa0,0x1f,0x0c,0x86,0x77,0xe2,0x9f,0xcf,0x5f,
	0xff,0xf1,0x27,0x30,0x43,0xe1,0x37,0xaa,0x82,0x50,0x86,0xfa,0x2d,0x12,0x00,0x35,
	0xd8,0xee,0x42,0x2d,0x27,0xb6,0xb5,0xdb,0x72,0xce,0x9b,0xfb,0x23,0xa8,0x29,0xa9,
	0xc7,0x8f,0x64,0x40,0x2a,0xdb,0xfb,0x60,0xfc,0xaf,0x68,0x46,0x49,0x47,0x45,0x76,
	0x34,0xb9,0x2c,0x68,0xae,0xaa,0xf0,0x1f,0xc3,0x15,0x39,0xf9,0x07,0x70,0x2f,0x1c,
	0x0c,0x81,0x7b,0x89,0xd9,0xdb,0xee,0xfa,0x83,0x2d,0x26,0x47,0xbf,0x52,0x6d,0x07,
	0x85,0x07,0xf5,0xa8,0xa7,0xd1,0x25,0x85,0xe8,0x7f,0xe4,0x2b,0x30,0x51,0x4c,0xc8,
	0x99,0xba,0x5d,0x4a,0xc4,0xe1,0x24,0xf0,0xdf,0xc0,0x7b,0x73,0xf8,0x97,0xcb,0x1c,
	0x74,0x41,0xcb,0x29,0x3b,0xc6,0x3e,0x41,0x5b,0x42,0x94,0xdd,0x4d,0xfc,0x48,0xa2,
	0x3f,0x50,0xf0,0xf2,0xd5,0x9b,0xe4,0x6c,0xd1,0x86,0x98,0xd1,0xc1,0xc4,0xd5,0xd9,
	0x61,0x3d,0xea,0x46,0xb1,0x6b,0xcb,0x48,0x97,0x02,0x40,0x72,0x87,0x6d,0x29,0x73,
	0x4e,0xb1,0xe3,0x23,0xb8,0x60,0xf1,0x9c,0x9c,0x3b,0x68,0x84,0x33,0x23,0x55,0xf8,
	0x0b,0xca,0x1e,0xa8,0x24,0x1a,0x5d,0x76,0x50,0x27,0xaf,0x2c,0x15,0x8f,0x32,0x53,
	0x40,0xd0,0x0b,0xea,0xf6,0xc0,0xad,0x57,0x13,0xce,0xbe,0x4b,0x17,0xae,0xbc,0x7f,
	0x84,0xe9,0x44,0xc8,0x3c,0x4d,0xf3,0xab,0xa2,0x5f,0x84,0xff,0x86,0xbb,0xe4,0x4e,
	0x45,0xd6,0x4f,0x82,0xab,0xe9,0x48,0x32,0x3b,0x7a,0x10,0xf4,0x36,0x88,0x80,0x71,
	0xcb,0x30,0x28,0x4c,0x9b,0x21,0xc6,0xbf,0x52,0x68,0x35,0xd4,0xa9,0x09,0x3d,0x34,
	0x14,0x3d,0x90,0xbe,0x56,0xcf,0xba,0x23,0x8d,0xb9,0x45,0x19,0x21,0x32,0xd1,0x95,
	0x17,0xfb,0xb1,0x5e,0x41,0x53,0xbd,0xb4,0x03,0xb1,0xa2,0xfa,0x6f,0xe8,0xa3,0x90,
	0x53,0xaf,0xe0,0xb0,0x08,0xb8,0xeb,0xf5,0xc2,0xd4,0xc4,0x0c,0x01,0x94,0xfe,0xd0,
	0x47,0x12,0x74,0x29,0x08,0x01,0x4f,0x62,0xec,0x81,0xe2,0x03,0xb0,0x58,0x98,0x07,
	0x0f,0xb4,0xa9,0x45,0x0e,0x7c,0x0d,0x27,0x27,0x9f,0x63,0x01,0xd7,0x2b,0xce,0x17,
	0x2a,0x53,0xd3,0x98,0x45,0x10,0x39,0xa7,0x63,0x3d,0x0e,0x60,0x91,0xfd,0x4c,0xd2,
	0xe5,0x73,0x06,0x1c,0xe3,0x76,0x5e,0xf2,0x7e,0x00,0x1e,0x92,0x27,0x0c,0xa8,0xea,
	0x0f,0xe4,0xb5,0x5a,0x80,0x0d,0x9f,0x80,0xfe,0xea,0x63,0xa4,0x0c,0xf9,0xad,0xfd,
	0x39,0x78,0xee,0x7a,0xf9,0xe3,0x8a,0x2d,0x99,0x1f,0xcb,0x63,0x94,0xf1,0xad,0x78,
	0x85,0xb2,0xa7,0x78,0xc4,0xee,0x36,0x4b,0x2c,0xf5,0x6e,0x91,0xba,0xbf,0x0b,0x55,
	0x0c,0x64,0xf8,0x12,0x55,0xf2,0x7b,0xe0,0xbe,0x84,0x61,0x33,0x16,0x52,0x67,0x13,
	0xb5,0xb8,0xc3,0x4f,0x16,0x68,0xf5,0x0c,0xcc,0xd1,0xed,0x69,0x31,0x7f,0x75,0xcf,
	0xa9,0x2b,0x11,0xf3,0x35,0xd2,0xf0,0xad,0x6b,0xe1,0x4d,0x99,0x56,0x16,0x05,0xde,
	0xc7,0x51,0x59,0xd9,0x19,0x5a,0x21,0x4b,0x72,0x98,0x86,0x01,0x3b,0x70,0x2b,0x3f,
	0xca,0x74,0xf8,0x88,0x5a,0x59,0xc1,0x82,0x27,0x03,0x3e,0x32,0xf7,0x4a,0x21,0xfd,
	0x2f,0xe4,0xbc,0x9f,0x60,0xac,0x47,0x21,0x30,0xfe,0x38,0xf8,0x68,0x47,0xad,0x98,
	0x31,0xcc,0x0f,0xdd,0x5c,0xc3,0x9d,0xd1,0x05,0x11,0x93,0x21,0xdd,0x44,0xe7,0x8c,
	0x51,0x98,0xee,0x58,0x98,0x73,0xfe,0xba,0x18,0x93,0x38,0x1e,0xb9,0xb7,0x3a,0xaa,
	0x11,0x1c,0x77,0xac,0x02,0x2a,0x94,0xfb,0x80,0x2f,0x0f,0x07,0xa8,0x2d,0x9f,0x6a,
	0x9f,0xd4,0xf0,0x1b,0x90,0xa6,0xaf,0xb1,0x2d,0xf1,0x86,0xc6,0xff,0x3a,0x81,0xf6,
	0xfe,0x02,0xe4,0x5a,0x8f,0x02,0xcb,0x1f,0xe0,0x39,0x35,0xa0,0x04,0x1d,0xee,0x20,
	0x3b,0xbb,0x8f,0x83,0x54,0x78,0x92,0x2a,0x76,0x76,0x6b,0xa2,0xcc,0x53,0xdd,0x1e,
	0x49,0xd1,0x7d,0x90,0x84,0xd8,0x2b,0x24,0x18,0x26,0xb2,0x60,0x84,0xf5,0x37,0xd0,
	0x38,0xb4,0x01,0x01,0xb8,0xae,0xc3,0xe4,0x10,0xa1,0x93,0xfa,0xee,0x19,0xa0,0x77,
	0x8b,0x79,0x2b,0x56,0x03,0x14,0x0a,0x5a,0xf1,0xce,0xb8,0xc4,0xc8,0x77,0xe1,0x08,
	0x5e,0x0f,0xe9,0xb7,0x30,0x93,0x9b,0xd0,0xec,0xcc,0xa6,0x6c,0x39,0xde,0xa0,0x3d,
	0xbe,0x2f,0x20,0xa1,0x48,0xcb,0xd9,0x74,0x7a,0xf4,0xba,0x66,0x0d,0x12,0x06,0x99,
	0x70,0xa4,0x51,0x73,0x9d,0x03,0x7c,0x26,0xf9,0xc4,0xb0,0x30,0xd1,0xdd,0xd4,0x87,
	0x35,0xf7,0xb7,0x23,0x56,0x24,0x4b,0xb1,0x4a,0x81,0x6c,0x28,0x16,0xc4,0x77,0x47,
	0xfd,0xbd,0x18,0x5a,0xcd,0x3c,0x08,0xf6,0xd3,0xac,0xf5,0x10,0x5a,0x0e,0x52,0x96,
	0x12,0xf4,0x49,0x0b,0x2f,0x82,0x1a,0xf8,0x48,0x8f,0x2a,0xe7,0xf2,0xb9,0x0a,0x65,
	0xf3,0xdb,0xc7,0xbd,0x20,0x9a,0x8d,0x7b,0x9c,0xbd,0xfb,0x18,0xe7,0x91,0x38,0x32,
	0x49,0x3a,0x29,0xb9,0x08,0x54,0xf1,0xb9,0xbd,0x2a,0x13,0x1b,0xfd,0xdb,0xee,0x2b,
	0xc0,0x79,0x49,0x6c,0x7a,0x24,0x37,0xf2,0x75,0x9d,0xdc,0xdf,0x5e,0x2e,0x77,0x30,
	0x7b,0x86,0xf6,0x11,0x64,0xdf,0x37,0xed,0xf6,0x87,0xf2,0x38,0xd0,0x05,0x30,0xc7,
	0x9a,0x30,0x09,0x77,0x96,0xe3,0xe9,0x37,0x7f,0xeb,0xb7,0xe2,0x60,0x2a,0x53,0x37,
	0x3b,0x40,0xd9,0xaa,0x84,0xc4,0xb4,0x0d,0x4c,0xaf,0x6f,0xb5,0x6e,0x63,0xcf,0xb2,
	0xac,0xc9,0x90,0xd1,0xaf,0x5b,0xee,0x6e,0x94,0x38,0x14,0xfd,0x72,0x18,0x96,0xfd,
	0xe8,0x4b,0x37,0xa0,0x66,0xac,0x8c,0x03,0x31,0xe7,0x67,0xf0,0xc7,0x21,0xfd,0xca,
	0x0c,0x01,0xe1,0x50,0x19,0x9e,0x6c,0xf8,0xbd,0x90,0xbb,0x70,0x7d,0x9e,0xcb,0x41,
	0xe1,0xeb,0x75,0x97,0x85,0xb5,0x66,0x8d,0x53,0x3d,0xdd,0x78,0xc4,0x66,0xf2,0x70,
	0x3e,0x75,0x55,0x36,0x42,0xb5,0xee,0x76,0xa5,0x5b,0xda,0x4f,0xc3,0xfe,0x34,0x22,
	0xd4,0x75,0x35,0x3a,0x91,0xe9,0xca,0xf7,0x38,0x89,0x86,0xea,0xbd,0x8e,0x83,0x7c,
	0x3b,0xb4,0x57,0x48,0x91,0xb4,0xe9,0x64,0x11,0xd5,0xf5,0xb1,0xee,0xc6,0xed,0xff,
	0x9e,0xe6,0xc5,0xce,0x30,0x05,0xa9,0x1a,0x4e,0x15,0x9e,0x56,0xea,0x78,0x8c,0x4e,
	0x42,0xc5,0x0e,0xf9,0x45,0x7e,0x1f,0x6c,0xc5,0x11,0x17,0x0a,0xca,0x0f,0x1d,0x4e,
	0xdd,0x63,0x49,0x6d,0x37,0x55,0x40,0x15,0xfc,0xce,0xc1,0x70,0xfc,0x77,0x58,0x7c,
	0x2c,0xb7,0x49,0x0a,0x45,0x2b,0x70,0x5b,0xa3,0xa4,0x0f,0x01,0x7d,0x85,0xbc,0x32,
	0xe3,0x64,0xab,0xeb,0x03,0x90,0x74,0xb7,0x3f,0x32,0x7b,0x7d,0xc9,0x84,0x80,0x7b,
	0x31,0xb3,0xe9,0xf9,0x98,0xa6,0xe9,0xa4,0x7f,0x05,0xc8,0xa8,0x7b,0xc3,0xa8,0x59,
	0xe3,0xbf,0x1d,0xf4,0x31,0xb0,0x69,0x92,0x72,0xd5,0xb7,0xb8,0xe4,0xd0,0x95,0xca,
	0x1d,0xc3,0xba,0x41,0x4b,0x60,0x31,0x2e,0xe7,0x2e,0x97,0x92,0xb2,0x70,0x2d,0xa4,
	0xdd,0x7b,0x3c,0x23,0xe9,0x91,0x9c,0x83,0x90,0x2d,0x19,0x91,0x44,0x01,0x86,0xb9,
	0x22,0x80,0xb7,0x21,0x07,0x7b,0xd3,0x24,0x98,0x4a,0xd3,0x4c,0x2c,0x41,0x3c,0x14,
	0x9e,0x16,0xb3,0xe1,0x66,0xb2,0xcf,0xa3,0x08,0x0d,0xc0,0x02,0xe8,0x1a,0x84,0xc5,
	0xbb,0x53,0x49,0x52,0x77,0x96,0xe6,0xa2,0x08,0x9e,0x9b,0x06,0x93,0x7b,0x14,0x69,
	0xd4,0xa0,0x93,0x12,0x7e,0x44,0x88,0x10,0x1b,0x88,0xcd,0x3f,0xa1,0x7f,0x8d,0xcb,
	0x35,0x62,0x61,0x25,0x9d,0x09,0x20,0x1f,0x48,0xff,0x7b,0x18,0x07,0xc2,0x0e,0x0b,
	0xe1,0xcc,0x5f,0x6e,0xf1,0x47,0x92,0x3f,0xdb,0x7d,0x70,0x8c,0xbe,0x30,0x4c,0xf1,
	0x52,0x40,0x55,0xad,0x2f,0x38,0x7b,0x4c,0x4b,0x87,0xb5,0x53,0xdc,0x5f,0x91,0xa0,
	0xe1,0xfe,0x57,0xdd,0xff,0xe8,0x04,0xd7,0xda,0xc7,0x98,0x23,0x58,0x30,0x55,0xf7,
	0xe3,0xf0,0xdb,0xab,0x91,0x67,0x72,0xb1,0x60,0x0a,0x61,0x0a,0xd7,0xeb,0xed,0xf2,
	0x32,0xf4,0xbb,0x7e,0x82,0x03,0x6a,0x1c,0xc5,0x57,0xa0,0xf6,0xd7,0x74,0x43,0x72,
	0x00,0x9e,0xe2,0x7e,0x35,0x81,0x29,0x96,0xae,0x70,0x5f,0x52,0x18,0x1b,0xa5,0x75,
	0x4e,0x24,0x2e,0x88,0x1e,0x69,0x3d,0x22,0xe3,0xcd,0x8a,0x4d,0xc5,0x11,0x5f,0x95,
	0x72,0xe1,0xca,0xcc,0x08,0x3a,0xe4,0xdb,0xe3,0x86,0x19,0xc8,0x5f,0x1c,0x93,0x46,
	0x39,0x68,0x54,0x97,0x85,0xb0,0x20,0xbe,0xf9,0xb0,0x42,0x89,0x36,0xfe,0x59,0x2f,
	0xf3,0x1a,0x1e,0x56,0x9a,0x4c,0xdb,0x4f,0xf4,0x0c,0xa7,0x23,0x3b,0xa3,0xb4,0x38,
	0xf6,0x0c,0x39,0x81,0x2e,0x0f,0x9e,0xaf,0x58,0x8e,0x7f,0x55,0x83,0xd8,0xe7,0x33,
	0xf2,0x81,0x03,0xdd,0x76,0x13,0x2d,0x9c,0xb6,0x15,0xd2,0xfd,0xae,0xea,0x84,0xa8,
	0xf4,0x56,0xb1,0x7c,0xdb,0xd8,0x24,0x0a,0xbb,0x08,0x92,0x54,0x8b,0x86,0x08,0x58,
	0x47,0xdf,0x6d,0x72,0xb6,0xa1,0xff,0x9d,0xc0,0x3b,0xec,0xc0,0x55,0xc0,0x28,0x00,
	0x2d,0x86,0x50,0xd7,0x98,0x59,0xf4,0x3f,0x2f,0x10,0x9d,0x09,0x49,0xa4,0x05,0x98,
	0x0d,0x35,0x31,0x87,0xad,0xd8,0xba,0x05,0x7b,0x1d,0x5d,0xd8,0x8f,0x37,0x61,0x34,
	0x7b,0x9c,0x1c,0xbc,0x88,0xfd,0x19,0xaf,0x40,0xc5,0x37,0xdc,0x21,0x71,0x25,0x2e,
	0xd1,0xea,0x66,0x88,0x78,0x5f,0xf3,0x86,0x20,0x0d,0x85,0x47,0xa6,0xc5,0x3e,0x9c,
	0x25,0x49,0x02,0xcf,0xae,0x5f,0xbd,0x2e,0x62,0x12,0x7f,0x57,0xdb,0x87,0x3e,0xcd,
	0xfe,0xa2,0xfe,0x53,0xa5,0xa6,0xfc,0xbf,0x6a,0xa8,0xf2,0x5a,0xf5,0x96,0x63,0x02,
	0x00,0x31,0x05,0x9b,0xe8,0xf6,0x43,0xe9,0x02,0x22,0x27,0xc3,0xbd,0x7a,0x68,0x67,
	0x7f,0x02,0x6c,0x51,0x0e,0x40,0x78,0x3a,0x63,0x38,0xb6,0xbd,0xd8,0x3f,0xde,0x09,
	0x94,0x9c,0xad,0x5e,0xe6,0x09,0x74,0x5f,0x8a,0x9b,0x96,0x29,0xb1,0x69,0x9a,0xd5,
	0x3f,0xcf,0xfe,0x44,0x60,0x57,0x3d,0x14,0x4e,0x93,0x55,0x22,0x7b,0x7d,0xd9,0xa9,
	0x36,0x04,0x19,0x3b,0xfc,0x15,0x0a,0x6b,0x17,0xae,0xae,0xb3,0x53,0x83,0x86,0x5c,
	0x0d,0x72,0x3d,0x63,0x83,0xd2,0xee,0x51,0x27,0x86,0x9f,0x2b,0xaf,0x87,0x76,0x3c,
	0xec,0x07,0x19,0xc7,0xa4,0x5e,0xad,0x10,0x98,0x75,0x9e,0x12,0xfa,0x9a,0xdf,0x94,
	0x27,0xf9,0x5a,0x7e,0x01,0x4b,0xe5,0x16,0xe5,0xc5,0x4c,0xd9,0x89,0xf6,0xc4,0x23,
	0x21,0x69,0x9c,0xa8,0xeb,0x11,0x94,0x4c,0x10,0x35,0x19,0xf1,0x19,0x06,0x68,0x6e,
	0x98,0x10,0x3a,0x3a,0xe5,0x4d,0x21,0xb8,0x9a,0xd6,0x2b,0x58,0x6e,0x3b,0xf3,0xa7,
	0x57,0x24,0x99,0x70,0x1d,0xc3,0xe3,0x13,0xcd,0x29,0x8a,0xf7,0xc9,0xe7,0x6a,0xfa,
	0xa1,0xc5,0x85,0x11,0xa7,0x0b,0xd4,0x84,0x8b,0x00,0xfa,0x18,0x9b,0x78,0xe7,0xfb,
	0xf8,0xc8,0x7f,0xcc,0x13,0x24,0x3e,0xb0,0x87,0x6c,0xf2,0x6f,0xb9,0x26,0xc9,0x0a,
	0xf1,0xe0,0x66,0xe3,0x08,0x7e,0x70,0x19,0x44,0xc1,0xc2,0xd7,0x47,0xa8,0x0c,0x69,
	0xf7,0x5a,0x92,0x33,0x76,0x81,0x67,0x4f,0x19,0x96,0x36,0x19,0x82,0xf5,0x89,0x5a,
	0xba,0x85,0xcf,0xeb,0xd0,0xf5,0x6f,0x0b,0xe2,0x30,0x6b,0x39,0xfe,0x5f,0x69,0xda,
	0xfc,0x76,0x8b,0x83,0x39,0x26,0x9c,0xff,0xf6,0x11,0x0d,0x06,0xf0,0xdc,0x4f,0xd9,
	0x47,0xaf,0xf8,0x0f,0x1a,0x40,0x95,0xc9,0xbf,0x56,0xb0,0x03,0x83,0xe7,0xe1,0xea,
	0xee,0x82,0xc6,0xf6,0x2e,0x35,0x5f,0xcc,0x7b,0xba,0xc4,0x21,0xdb,0xf6,0x70,0x3b,
	0x1b,0x8a,0x96,0xc4,0x59,0x1e,0x29,0xfe,0x1f,0xf3,0x36,0x74,0x3a,0xf4,0x50,0x12,
	0x95,0x26,0xd9,0x51,0x6f,0xcc,0x90,0x2f,0x46,0xd9,0x6c,0xf3,0x6d,0xdd,0x18,0x0a,
	0x1f,0xdb,0x80,0x30,0xc8,0xdc,0x6c,0xc6,0xb1,0x97,0xfc,0x42,0x10,0x1a,0x99,0xc1,
	0x35,0x05,0x53,0x01,0xed,0x41,0x90,0xd3,0x9e,0x50,0x2d,0x56,0x55,0x09,0x35,0x6d,
	0xdb,0x3d,0x8b,0xaf,0x7f,0x92,0x18,0x80,0x75,0xf4,0xb0,0xf5,0xb0,0x2c,0x0a,0x89,
	0x83,0xca,0xfa,0x87,0xa1,0xfe,0x5f,0x04,0xc6,0xa3,0x30,0xfb,0x16,0xdb,0x38,0x6e,
	0x5d,0x20,0x5d,0xb3,0xb7,0x70,0xd9,0x97,0xca,0x00,0x99,0xfb,0xa6,0xd4,0x22,0x39,
	0xc2,0xed,0x47,0xc6,0xa1,0xfd,0x95,0x42,0xdd,0x6e,0xb1,0x7e,0x33,0x5b,0x92,0xdd,
	0x54,0x32,0xf0,0x9b,0xc9,0xaa,0x47,0xe3,0x2b,0x76,0x1e,0x45,0x90,0xcb,0x16,0x16,
	0x7b,0x9b,0x09,0xf4,0xed,0x5d,0x6c,0x72,0xf7,0xbb,0xd5,0x13,0xb7,0x03,0xcf,0x5c,
	0xc7,0x00,0xc7,0x1b,0x44,0x40,0x7b,0x4b,0x63,0x43,0xfa,0xd9,0x5a,0xc1,0x9a,0x50,
	0x4d,0x18,0x83,0x7a,0x6f,0xdd,0xca,0x3b,0x6f,0xb1,0xd5,0xc5,0x5b,0x8f,0xc4,0x5d,
	0xf0,0x61,0xa9,0xb4,0x0e,0x3d,0x01,0x12,0xe7,0xb2,0x48,0xab,0xd1,0xc1,0x48,0xed,
	0xee,0x94,0x35,0x66,0x25,0x69,0xe1,0x4d,0x78,0x87,0x79,0x62,0xd9,0x51,0x50,0x74,
	0x05,0xfd,0x59,0x87,0x11,0xeb,0x52,0x5a,0x7e,0x42,0x35,0xac,0xcd,0xcb,0xe4,0x1b,
	0x4b,0x94,0x23,0xfb,0x3b,0x6d,0x53,0xc2,0x1b,0x1b,0x3b,0x07,0xb2,0x3d,0x33,0x1c,
	0x8f,0x3f,0xd7,0x2f,0xf0,0x02,0xc6,0x4d,0x08,0x83,0x9f,0x7a,0x72,0xdf,0xdd,0xb2,
	0x93,0x18,0x17,0x03,0xa5,0x07,0x77,0x5c,0x6f,0xbe,0x26,0xfe,0x50,0x64,0x34,0xe3,
	0xac,0xb9,0x44,0x9e,0xb2,0xc6,0xe4,0xc2,0x93,0x71,0xf7,0xc4,0x6a,0x25,0x50,0xe2,
	0x60,0x92,0xc7,0xba,0x11,0xbb,0x00,0x01,0x28,0xe0,0xbe,0x2f,0x2e,0x17,0xc4,0x9b,
	0x6e,0xad,0xc7,0x90,0x7f,0x0d,0x2e,0xa3,0x45,0x12,0x7b,0xea,0x13,0x70,0x5f,0x13,
	0xee,0xf1,0x9a,0x1e,0x84,0xbc,0x33,0xf9,0x9e,0xc6,0x82,0xac,0xf5,0x84,0xd4,0xb8,
	0x22,0x0d,0x80,0xf8,0xd9,0x43,0x65,0x8d,0x71,0xfc,0x42,0xea,0xf7,0xb8,0x9e,0x86,
	0x3e,0x32,0x6d,0xe8,0xbb,0xff,0x7c,0xcd,0x77,0x05,0xb2,0xf2,0xb5,0x92,0x2a,0x97,
	0x19,0x90,0xdc,0x94,0xf7,0x4d,0x30,0x8c,0x83,0x51,0xf2,0x68,0xe5,0x9f,0x43,0xe8,
	0x07,0xc7,0x4e,0xa6,0x07,0xa2,0xa9,0x08,0x31,0x2a,0x61,0x3c,0xbe,0x91,0x11,0xc5,
	0x55,0x86,0xec,0x2b,0x12,0x28,0xa4,0x05,0x42,0xc8,0x85,0xba,0x9d,0x11,0x14,0x28,
	0x3a,0x86,0x6d,0xf4,0x66,0x81,0x78,0x69,0xdb,0x1d,0x16,0x91,0x01,0x3c,0xac,0xf7,
	0x13,0xb7,0x63,0x5a,0x66,0x3d,0x17,0x67,0xb2,0x5c,0xb6,0x1a,0x7b,0x80,0x3d,0xbc,
	0x43,0x80,0x9a,0xec,0x62,0xd1,0x28,0x79,0x2d,0x4b,0xaa,0xc6,0x1e,0xa3,0x95,0xd6,
	0x6f,0xb0,0xdc,0x17,0x5d,0x1e,0x0e,0x43,0xed,0xd2,0xb4,0xab,0xec,0x79,0xa7,0x37,
	0xba,0xe2,0xd6,0x62,0xd0,0x2a,0x20,0x18,0xb8,0x68,0xaf,0xa8,0xbd,0xec,0xa5,0x7d,
	0xf6,0x5d,0x5a,0x95,0x20,0x8e,0x76,0xfa,0x10,0x98,0x33,0xbd,0x84,0xa5,0x9d,0xb7,
	0x08,0x7a,0x56,0x4a,0xf1,0x9f,0xa1,0x55,0x50,0xfe,0x3e,0xc3,0x42,0xf9,0xac,0x9e,
	0x92,0xbd,0x9f,0xfe,0x2e,0xa1,0x0b,0x54,0xcd,0x85,0xff,0x72,0xb7,0xa4,0x35,0xca,
	0xf0,0x18,0xf6,0xec,0x6a,0x8e,0x08,0x63,0x75,0x71,0x4a,0x8b,0x00,0x3d,0x6d,0x81,
	0x9c,0x8a,0x03,0x85,0xbd,0x7b,0xf4,0x7a,0x1e,0x20,0x56,0x17,0x79,0x11,0x3a,0x95,
	0x6d,0xe0,0xfc,0xe0,0xc5,0xa9,0xa9,0x8e,0x31,0xf1,0xb1,0x85,0x94,0xcd,0xc4,0x13,
	0xae,0xe0,0xa2,0xf4,0x07,0x43,0x9c,0x16,0x68,0xa2,0xbc,0x7f,0x59,0x51,0xbb,0xba,
	0x73,0x31,0x43,0x11,0x4e,0x9d,0xf6,0x37,0x23,0x7c,0x1e,0xb8,0x27,0x35,0x8b,0xa9,
	0x62,0x0b,0xd9,0x41,0xd8,0x17,0xf3,0xd7,0x09,0xe3,0xdf,0xc8,0x33,0x3b,0x13,0x07,
	0xff,0x5e,0xd9,0xc7,0xea,0x48,0x41,0x2e,0x8e,0x4b,0x87,0x0d,0xc1,0xee,0x11,0x4b,
	0x73,0x57,0x0c,0x52,0x97,0x2a,0x5d,0xbd,0xd0,0xd3,0x15,0xd6,0x4d,0x8d,0x07,0xdb,
	0xc4,0x7c,0x37,0xc6,0x77,0xc9,0xbb,0x7e,0x6e,0x29,0x1a,0xdd,0x07,0x81,0xa6,0x23,
	0xa9,0xc7,0x32,0xbf,0xb3,0x30,0xa6,0xd9,0x44,0x45,0xbd,0x76,0xf1,0xb4,0x90,0x85,
	0x29,0x3a,0x0d,0x86,0x6e,0x7c,0xe6,0x67,0x4b,0xf3,0xcc,0x74,0x25,0x74,0xf3,0x6c,
	0x75,0x86,0xa2,0xf3,0x74,0xca,0x2f,0x14,0x1f,0x3e,0x87,0x71,0x0d,0x13,0xeb,0x03,
	0x79,0x49,0x1e,0x44,0x2a,0x75,0x91,0x90,0x77,0x27,0x58,0x47,0xf9,0xb7,0xac,0x70,
	0x22,0xd6,0x1f,0x97,0x34,0x00,0x59,0xb5,0xb1,0x1c,0x23,0xbc,0x11,0xd9,0x52,0x93,
	0x07,0x38,0x85,0xe6,0x88,0x60,0xe6,0x66,0xf3,0x17,0xfd,0xb6,0xe2,0x7e,0xde,0x53,
	0xcf,0x13,0x4b,0x5b,0xde,0xdd,0xdc,0xe6,0x9f,0xba,0x1a,0x5f,0xe3,0xac,0x2d,0xf4,
	0x81,0x3e,0x63,0xee,0x09,0x89,0x22,0xc1,0xab,0x24,0xfc,0xe6,0xae,0x0e,0xe1,0xfb,
	0x86,0x2c,0xdf,0xbf,0x3d,0xc5,0xae,0xe1,0xb4,0x2c,0x0e,0xd3,0xed,0x3e,0xd9,0x81,
	0x4b,0xcd,0xc0,0x99,0x58,0xfe,0xda,0xbb,0xff,0xfb,0x8b,0xf5,0x78,0x8d,0xf1,0xc7,
	0xfa,0x6c,0x0e,0xbd,0x76,0x8d,0x45,0xfc,0xa8,0x85,0x30,0x94,0xb1,0xbf,0x5c,0x31,
	0xcc,0x89,0x02,0x47,0xcd,0x66,0x3e,0x09,0xb3,0xc6,0x05,0x87,0x43,0xe8,0x88,0xbf,
	0x16,0xe1,0x23,0x8a,0xe6,0x93,0x68,0x94,0xca,0x23,0x72,0x6e,0x2e,0xe6,0x5d,0xc6,
	0x80,0x02,0x7b,0x6f,0xd8,0x14,0x54,0x51,0xd2,0x9b,0xdb,0x32,0x7c,0x35,0x8a,0x58,
	0x9f,0x92,0x0c,0x80,0x2b,0xbd,0xc1,0xc8,0xbe,0xf8,0xd9,0x6d,0x5e,0x41,0x7a,0x79,
	0x6e,0x08,0xda,0x05,0x88,0x6a,0xac,0x44,0x9f,0x08,0x16,0x22,0xb6,0x73,0xf2,0x07,
	0x3e,0x59,0xd6,0x1e,0x08,0x51,0xe4,0x8d,0x39,0x78,0xfa,0xb0,0xe2,0xe6,0x89,0x72,
	0x63,0x93,0x41,0x55,0xd9,0xca,0x95,0xd2,0x21,0x8d,0xbd,0xdb,0x42,0x66,0x07,0xb2,
	0x8d,0x0f,0xe6,0x84,0x51,0x4c,0xdc,0x9d,0x69,0x16,0x42,0x53,0xc8,0x66,0x6c,0xce,
	0xe5,0x2f,0xda,0x2e,0x1b,0xc5,0x07,0x46,0x1c,0x9d,0xa3,0xa1,0x8b,0xd4,0x3c,0xfb,
	0x98,0xb7,0x1c,0x91,0x84,0xd0,0xb2,0x5c,0x2d,0x78,0xf6,0xdb,0x73,0xca,0xec,0x33,
	0xb9,0x64,0x70,0xa3,0x89,0x07,0x88,0x9f,0xba,0x20,0xbb,0x9f,0xf1,0x1d,0xa1,0x6d,
	0x63,0x3a,0xb2,0x30,0x9e,0xcb,0x7f,0xa4,0xa0,0xf1,0x7c,0x9f,0x62,0xae,0x4b,0x49,
	0x3d,0x21,0x00,0x1c,0xf8,0xce,0xf9,0x08,0x31,0xa9,0x06,0x76,0x6c,0xb2,0xfe,0x14,
	0xd8,0xc7,0xab,0xd0,0x9b,0xf4,0xad,0xec,0x39,0x20,0x1f,0x65,0xaf,0xc5,0x6f,0xe9,
	0x71,0xf7,0xca,0x3a,0x5f,0xc8,0x3f,0x67,0x42,0xf4,0x23,0xfc,0xbb,0xee,0xa9,0x67,
	0x76,0xc1,0xed,0xbe,0x11,0x2a,0x0c,0x27,0x67,0x4f,0x8b,0xb4,0x25,0x1b,0xb5,0x9f,
	0x92,0x64,0xb8,0x64,0xdd,0xa7,0xea,0x34,0x7f,0x0d,0x34,0xc6,0x40,0xcc,0x00,0xcf,
	0x55,0x8b,0x9d,0xc9,0xa7,0xc0,0x7b,0xc0,0xc3,0x2d,0x4c,0xbe,0xed,0x64,0x2d,0xfc,
	0xa0,0x7d,0x44,0x67,0xaf,0xf0,0x10,0x71,0x2f,0x72,0xf7,0x1f,0x8f,0xd6,0xb3,0x8e,
	0x63,0x3e,0x0e,0x69,0xcd,0x5b,0x14,0xd3,0x70,0x70,0x44,0x5d,0xab,0x06,0x62,0x99,
	0x20,0xa4,0x09,0xb7,0xd9,0xad,0x60,0x83,0xe6,0x70,0x77,0xa6,0x54,0x3a,0xdd,0x60,
	0x31,0xd2,0x81,0x0d,0x88,0x03,0x1b,0x83,0x66,0x00,0xc7,0x27,0xf0,0x19,0x3e,0xd9,
	0xbc,0x08,0x07,0xa0,0x68,0x0b,0xda,0xac,0xb3,0x12,0xa0,0x7d,0xc3,0xe1,0xfc,0x38,
	0x86,0x30,0x0d,0x07,0x80,0x5a,0x07,0x3f,0x53,0x34,0x48,0x87,0xdc,0x62,0x91,0xe8,
	0x1d,0xc3,0xd2,0x3c,0x7b,0x95,0x24,0xae,0xc2,0x04,0x30,0x36,0x38,0x93,0x50,0xf5,
	0x90,0xae,0x51,0x7f,0x55,0x10,0xe7,0xb1,0xb3,0x97,0xa9,0xac,0x67,0x2f,0x4a,0x6b,
	0x84,0x2a,0xda,0x19,0xc9,0x71,0xac,0x50,0xf5,0xbe,0x43,0x99,0xb7,0xde,0x93,0x50,
	0x9d,0x0c,0x83,0x77,0x4f,0x0d,0x7d,0x4c,0xfb,0x08,0xa6,0xb9,0x86,0xcb,0xd5,0x71,
	0x61,0xcd,0xcc,0x85,0x79,0x28,0x3a,0x64,0x58,0x71,0xee,0x1e,0x38,0x47,0x48,0x48,
	0xa2,0x83,0xb4,0x95,0xea,0x4e,0x01,0x6e,0x87,0xc9,0xa5,0xb3,0x4c,0x93,0x6f,0x09,
	0x8e,0x4e,0x04,0xaf,0xae,0x28,0x31,0xc2,0x6b,0x49,0x96,0x9d,0xc8,0xec,0x9f,0xaa,
	0xf9,0x49,0x73,0x2a,0x71,0x9b,0xc9,0xce,0xdb,0xa7,0xec,0x81,0xae,0x7f,0x13,0xf6,
	0xdf,0xed,0xfb,0xf3,0xcd,0x71,0x76,0x71,0x33,0x9f,0x3d,0x39,0x5f,0xfe,0x42,0x61,
	0xcc,0x07,0xf8,0xa0,0x00,0x7b,0xaf,0xb8,0xe1,0x90,0x7a,0xc2,0x3f,0x57,0xad,0x8d,
	0x08,0x8a,0x53,0x20,0x29,0x95,0x25,0xf8,0x01,0xcf,0x70,0x7e,0xaa,0x15,0x04,0x61,
	0xcd,0xa5,0x92,0x0a,0x4c,0x0a,0x3c,0x46,0x7a,0xea,0x25,0xe6,0xfe,0x25,0xd4,0x56,
	0x0b,0xac,0x67,0x9a,0x38,0x0b,0xd2,0x9e,0x5d,0x0b,0xb8,0x2e,0x83,0x5f,0x3d,0xaa,
	0xed,0x3b,0x2e,0x1f,0xa9,0x08,0x0e,0x59,0x2a,0xd5,0xc6,0x9e,0xa7,0xea,0x57,0x69,
	0x33,0x03,0x3a,0xfc,0x98,0xad,0xa9,0x82,0x68,0xc1,0x32,0x9c,0x60,0xb6,0x5c,0x48,
	0xa4,0xdb,0xfe,0xca,0xa4,0xba,0x95,0x45,0x5d,0x69,0xc9,0x79,0xf5,0xdc,0x80,0x82,
	0x53,0xcc,0xee,0x2f,0x8a,0xf0,0x0d,0xeb,0x20,0xcf,0xaa,0x99,0xb0,0x15,0x12,0x50,
	0x9c,0x35,0xde,0x0c,0x62,0x8c,0x0f,0xd4,0x48,0x4c,0x8f,0x82,0x34,0x8a,0xd5,0x62,
	0xbc,0xe2,0x6e,0x3c,0x8f,0xa7,0xf7,0x97,0xd8,0x66,0x89,0x8f,0xe5,0x34,0xb1,0xf5,
	0x82,0x82,0x0b,0xc2,0x89,0x7b,0xc0,0x47,0xc5,0x46,0xd8,0x02,0xa5,0x8e,0xb4,0x87,
	0x68,0x9c,0x0f,0x23,0xe0,0xc8,0xc9,0xb1,0x01,0x25,0xa9,0x18,0x5c,0x63,0xd9,0x38,
	0xbb,0x3e,0xc6,0xe7,0xeb,0xff,0x21,0x81,0x1b,0x16,0xfa,0x6d,0x9e,0x04,0xc7,0x47,
	0x40,0xf4,0x61,0xc1,0x0e,0xce,0x42,0x4d,0x50,0x71,0xf9,0xdd,0x8a,0x9e,0x4f,0x6b,
	0xe1,0xd9,0x9f,0x14,0xb7,0xeb,0x3a,0x5b,0x63,0x0d,0x83,0xfd,0x4c,0x41,0xa2,0xe3,
	0xbd,0x10,0xe0,0xda,0x96,0x07,0x85,0x3e,0x5c,0xd1,0x0c,0x0e,0x6d,0x7b,0x67,0x5d,
	0x22,0x39,0x46,0xa4,0xc7,0x9c,0xf0,0xa3,0xae,0xd8,0x01,0x44,0xbc,0x41,0x34,0xf5,
	0xd4,0xf3,0x7a,0x17,0x0e,0xc5,0xfd,0x13,0x7d,0xa6,0x1e,0x63,0x29,0xad,0xc8,0x1a,
	0x71,0x7d,0x35,0xde,0xfa,0xec,0xa9,0x2e,0xe0,0xdf,0x3d,0x53,0xcb,0xe3,0x85,0xbb,
	0x36,0xa0,0x4d,0x5e,0xe0,0xca,0xba,0x2d,0x64,0x12,0xb5,0x98,0x80,0xea,0xd2,0x65,
	0xc0,0x06,0xd7,0xa7,0x99,0x58,0x4d,0xac,0xe0,0x3e,0x49,0x14,0x5f,0x46,0xb7,0x89,
	0x51,0x18,0xc8,0x8f,0x6a,0x6b,0x72,0xc0,0xdc,0x8e,0x76,0x64,0x63,0x12,0xf2,0xec,
	0x52,0x57,0xde,0x81,0xda,0x0b,0x4a,0xa5,0x23,0xa3,0xce,0x60,0x59,0x6f,0x1b,0xc8,
	0x53,0x2b,0x52,0x83,0x14,0x89,0xd7,0x2e,0xf2,0x53,0xe9,0xb5,0xf2,0x18,0x6f,0xd6,
	0x5b,0x7d,0xaf,0xf4,0x3a,0x0b,0x9c,0x61,0x75,0xa5,0xb1,0xe7,0xdf,0xb3,0x31,0xd0,
	0xb9,0x61,0xc3,0x98,0x66,0x31,0xdf,0x07,0x0d,0x6f,0x25,0xf7,0xc7,0x2c,0xf9,0xaf,
	0x0f,0x81,0x1f,0x3f,0xed,0x44,0x64,0x13,0x7c,0x89,0x24,0x26,0xfd,0x3f,0xed,0xc3,
	0x16,0x93,0x55,0x31,0xfe,0x05,0xbe,0xfc,0x89,0x54,0xff,0xee,0x33,0x23,0x60,0xc9,
	0xb5,0x7b,0xec,0x5d,0xce,0x7a,0x1d,0xd2,0x2b,0xfd,0x0a,0x6d,0xc7,0x47,0x07,0x82,
};


/* 317-0037 FD1089B */
static const UINT8 key_0037[0x2000] =
{
	/* data */
	0xe7,0x57,0x69,0xa4,0x29,0x6c,0x18,0xa3,0x64,0x71,0x11,0x55,0x16,0x59,0xaa,0x2c,
	0x53,0xb4,0x82,0xce,0x26,0x3f,0xfb,0x44,0xaa,0xac,0xc9,0x76,0x63,0xc6,0x55,0x6e,
	0xbd,0x9e,0x8a,0x31,0x46,0x6a,0x4e,0x22,0x8a,0xca,0x31,0x48,0x84,0x3e,0x69,0x7c,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0xf1,0x4e,0xd0,0xeb,0x26,0xd6,0x7d,0xaf,0x37,0x2f,0x78,0xd9,0xb4,0xad,0x7f,0x9d,
	0x57,0x96,0xb0,0xaa,0xd5,0x38,0xfe,0xc7,0x22,0x89,0x93,0x8c,0xc4,0x19,0x3f,0x1d,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0034 FD1089B */
static const UINT8 key_0034[0x2000] =
{
	/* data */
	0xd4,0xac,0x66,0xc0,0x6a,0xd5,0x81,0x09,0x5d,0xfc,0x8b,0x0b,0x6e,0x1d,0xf0,0xdf,
	0x46,0x47,0x58,0x01,0x84,0x60,0xe3,0x34,0xf0,0xef,0x6f,0x32,0x63,0xbf,0x63,0xd9,
	0xb0,0xb3,0xaa,0x15,0x18,0x5c,0x93,0x02,0xfa,0x40,0x0a,0x19,0xac,0x66,0x88,0xbd,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xb9,0xb4,0xe5,0xf6,0xf2,0x87,0xe1,0x33,0x37,0xb4,0xc3,0xba,0xa6,0x4f,0xdf,0x6c,
	0x59,0x06,0x5d,0x23,0x33,0xa9,0x57,0x11,0x00,0x28,0x3b,0x56,0x59,0x62,0x24,0x61,
	0xec,0x93,0xcd,0x4d,0x42,0x76,0x1f,0x0f,0xf3,0x4f,0xc9,0x24,0xd3,0xa5,0x58,0xab,
	0x7f,0x93,0x8a,0x3c,0xa4,0x61,0x4d,0xbe,0x73,0x12,0x85,0xc5,0x10,0x9c,0xa7,0xc7,
	0x23,0x06,0xf7,0xa6,0x95,0x4e,0x08,0x45,0x7f,0x48,0x20,0x67,0xc7,0xe5,0x1c,0x15,
	0xba,0x1f,0xc5,0x19,0x95,0xd2,0x9d,0x42,0x21,0xc0,0xb3,0x4e,0x21,0xb8,0x1d,0x2a,
	0xb7,0x6d,0x78,0x08,0x14,0x14,0x5e,0x64,0x16,0x93,0x45,0xd5,0xbd,0x0d,0x97,0xe5,
	0xb1,0xce,0xb9,0xe7,0x22,0x79,0xba,0xde,0x8f,0xa9,0x8c,0xfb,0x32,0x80,0xf7,0x90,
	0x44,0x42,0x86,0x19,0x44,0x95,0x55,0x2f,0x82,0x73,0x96,0x9a,0x86,0xf1,0x55,0x91,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x74,0x98,0xba,0xcf,0xe0,0x05,0x9d,0xe4,0x53,0xf2,0xef,0x82,0x00,0x6f,0x01,0xf0,
	0x37,0x7a,0x4d,0xf9,0x9b,0x81,0x41,0x84,0x8a,0xa5,0x3a,0xf1,0x18,0x1b,0x47,0x52,
	0x3b,0xed,0x3a,0xbe,0xb2,0x4d,0x5b,0xb7,0x5d,0xc5,0x2d,0x42,0x03,0x0e,0x3d,0x74,
	0xaf,0xb5,0xf0,0x26,0xdb,0xa0,0x54,0x76,0xe3,0x19,0x6e,0xda,0xf3,0x52,0xef,0x7d,
	0x5e,0x4a,0xe9,0x0b,0x7a,0x31,0xa7,0x85,0xce,0x09,0xa6,0xee,0xe8,0xbc,0x35,0x5d,
	0x04,0x93,0xef,0x2f,0xa6,0x43,0xa6,0x53,0xf9,0xd4,0xed,0xd6,0xab,0x3e,0xc4,0xa6,
	0x5a,0xa0,0xcc,0x0c,0x08,0x01,0x33,0xc1,0x09,0x81,0x2e,0xc7,0xe4,0xc8,0xa3,0x8e,
	0xfc,0x3e,0xd3,0x73,0xd5,0x78,0x30,0xc4,0x15,0x05,0x63,0x95,0x4b,0x4a,0x55,0xe3,
	0xda,0xf8,0x25,0xb7,0x7d,0x06,0xf5,0xa0,0xea,0x70,0xe7,0x0b,0x83,0xa5,0x69,0x7b,
	0x4b,0x7a,0xeb,0xc9,0xa1,0x71,0x00,0xa8,0x40,0x65,0x8e,0x47,0x64,0xcf,0x82,0x3e,
	0x7b,0x53,0x37,0xa7,0x3c,0x10,0x67,0x21,0xec,0xed,0xac,0xd8,0xe9,0x4c,0x9e,0x07,
	0xf4,0xf3,0xa9,0xc3,0x0b,0x21,0x15,0xb9,0x45,0xc7,0xce,0xa9,0x77,0x00,0xc7,0x0d,
	0x87,0x2e,0x34,0xda,0xfc,0x7a,0xba,0xa5,0xfb,0xde,0x7f,0x49,0xc8,0x4b,0xbc,0xfb,
	0x31,0xf9,0x7d,0xc3,0xf8,0x96,0x61,0xb7,0xbe,0xa0,0x2a,0x3f,0x6a,0xdd,0xc1,0xc2,
	0x1d,0xcc,0x4b,0xac,0x07,0x30,0xc3,0x92,0xf2,0x64,0x27,0x84,0x5a,0x69,0xaf,0x8e,
	0x7b,0xc4,0x7d,0xb0,0xdf,0xb0,0xeb,0xd3,0x13,0x27,0xd9,0xf4,0x13,0xb9,0x02,0x80,
	0x5c,0xe2,0xb2,0xdf,0xdd,0x3d,0x88,0xac,0x3f,0xd4,0xfb,0xa3,0x10,0x3a,0xc4,0xe3,
	0x60,0xd1,0xee,0x10,0x06,0x39,0x71,0x1c,0x2a,0xb1,0x6e,0xf0,0xa2,0x74,0xce,0x3e,
	0xab,0x3a,0xd3,0x9f,0x5f,0x8f,0x14,0x7c,0x5b,0x4c,0xf9,0x22,0x4a,0x08,0xf8,0x66,
	0x2b,0x3b,0x9e,0xa0,0xea,0x3a,0x17,0xde,0x38,0xc5,0x98,0x10,0xe2,0x01,0x3a,0xc3,
	0xc4,0xe3,0xc0,0xa4,0xa0,0x68,0x3d,0xdc,0x65,0x11,0x23,0x44,0x90,0x57,0x87,0x6e,
	0x38,0x96,0xcc,0xd6,0xa6,0xb5,0x1f,0xa0,0xe3,0x17,0x5a,0x45,0xad,0x3b,0x3b,0xe4,
	0x82,0x7e,0xea,0xae,0x70,0xf8,0x92,0x5c,0xad,0xf3,0x23,0xdd,0x09,0xac,0x5d,0xbe,
	0x52,0x1f,0x65,0x19,0x1c,0xf1,0xd0,0x55,0x31,0x42,0x5c,0x12,0x21,0x10,0x6d,0xc5,
	0xf0,0x27,0x3b,0x8c,0x65,0x8b,0x68,0x7c,0x79,0x29,0x6e,0x40,0x88,0x05,0x90,0x3a,
	0x2a,0xbb,0x83,0x20,0xfb,0xb1,0xa5,0x01,0xd0,0x9b,0x16,0x6e,0xdb,0x25,0x5d,0x6d,
	0x4e,0x73,0x26,0xb0,0xc3,0x23,0x2b,0x32,0x46,0x7c,0x18,0x30,0xbc,0x17,0x0d,0x5b,
	0xb4,0x9a,0x6d,0xef,0x60,0x3f,0x25,0xd5,0x29,0x9f,0x76,0x13,0x5a,0xc4,0x79,0x59,
	0xc8,0x46,0x7d,0xb3,0xd6,0xa6,0x57,0xea,0x6f,0x4d,0x8c,0x6b,0x39,0x39,0xe0,0x69,
	0xd6,0xdf,0x66,0xdb,0x55,0xd3,0x95,0x99,0xf5,0xfd,0xa9,0x86,0x6c,0xe6,0xbf,0xca,
	0xbc,0xcd,0xa1,0x92,0x7d,0xab,0x16,0x03,0x8d,0x62,0x47,0xd2,0x3d,0x0d,0xdd,0xf2,
	0xbb,0x78,0xda,0xe9,0xd7,0x79,0x1e,0xb0,0x1a,0x13,0xcb,0x21,0x0b,0x9b,0xe7,0x97,
	0x39,0x9c,0x5a,0xfd,0x98,0xda,0x7d,0x67,0xf0,0x25,0xf9,0x83,0x76,0xf2,0xfd,0x3d,
	0x39,0x67,0xed,0x1a,0xe8,0x38,0xcd,0x37,0x2b,0x0a,0xa5,0xb7,0xa3,0xde,0xe8,0x3a,
	0x90,0x61,0x33,0x1d,0xb3,0x35,0xe2,0x9b,0xcd,0x48,0x3c,0xed,0xf5,0xf6,0xb9,0xdc,
	0x57,0x62,0x50,0xce,0xbd,0x37,0x47,0x38,0x3c,0xd6,0xb7,0xd8,0x96,0xe7,0x77,0xc9,
	0x7a,0x33,0x10,0x6a,0x0f,0x0d,0xd8,0xe0,0x16,0xc1,0x86,0xb2,0x69,0x6b,0x26,0xc2,
	0x10,0x08,0xb8,0x9a,0xc5,0xb5,0x05,0x54,0x7a,0x38,0xe4,0xbd,0x5d,0x62,0xeb,0xc7,
	0x0d,0x88,0x13,0x98,0xaa,0x2d,0xcd,0x80,0xf2,0xe6,0x08,0x6e,0x94,0x33,0x05,0x4e,
	0x8d,0x4d,0x2e,0x43,0xab,0x5a,0x97,0xf1,0x52,0xa4,0x5c,0xd5,0x76,0x31,0xa0,0xc2,
	0x62,0x27,0x18,0xcf,0xe5,0x8c,0x1c,0xaa,0xea,0xc1,0x12,0x74,0x82,0x4d,0xe6,0xb9,
	0x95,0xda,0xec,0xf0,0x42,0x7b,0x0b,0xe5,0x2d,0x29,0xa3,0x75,0x77,0xf5,0x52,0xe4,
	0x45,0xd1,0xdb,0x2f,0xc0,0xa3,0x06,0x54,0x1b,0x94,0xa3,0xa0,0x47,0x62,0x82,0x54,
	0x63,0x8a,0xe3,0x33,0x06,0x30,0xc8,0xe9,0xe9,0x45,0x4f,0x0f,0x59,0xed,0x38,0xe3,
	0x66,0xde,0x33,0xa0,0xba,0x84,0xe6,0xd4,0x1f,0x16,0xd7,0xb8,0x95,0xcb,0x11,0x5a,
	0xe4,0x4b,0xaa,0x0f,0x93,0x7b,0x11,0x43,0x48,0x91,0xe4,0x0a,0x13,0x60,0xf4,0x05,
	0xb3,0xbf,0x3d,0x47,0xec,0xbe,0x11,0x86,0x79,0x4f,0xbf,0xbb,0x4d,0xc2,0x40,0x84,
	0xc2,0xa7,0x0b,0x16,0x41,0xc6,0x83,0x89,0xf9,0x8f,0x16,0x48,0xbb,0x8a,0xbe,0x00,
	0x65,0x21,0xba,0x12,0x20,0x3e,0x0b,0xab,0xd4,0x20,0x7f,0x76,0x09,0xdc,0xc0,0x34,
	0x2a,0x12,0xa5,0x1e,0xc5,0x60,0x60,0xe4,0x69,0xd3,0xda,0x9b,0xeb,0x4d,0x8e,0xb6,
	0x45,0x0e,0xf5,0xe2,0x0f,0x7c,0x4f,0x5a,0x61,0xc2,0x95,0x09,0xeb,0x0c,0x07,0x95,
	0x79,0xec,0xe9,0x2a,0x87,0x02,0x7a,0x5b,0xad,0xb4,0xe3,0xe4,0x3e,0x8a,0xbb,0x32,
	0x23,0x32,0x42,0xf4,0x6f,0xed,0xb9,0x7d,0xe2,0xdd,0xce,0xd9,0xc8,0x6f,0x04,0x95,
	0x76,0xb1,0x8d,0xd1,0x0b,0xc5,0x34,0xd9,0xef,0xca,0x2d,0x05,0x5c,0x0c,0x07,0x9e,
	0x23,0xd8,0xac,0x7c,0x60,0x37,0x61,0xe6,0x4c,0x89,0xd3,0x21,0x52,0xc5,0x79,0x9c,
	0x4b,0x6d,0x63,0x55,0xc3,0x3c,0xdb,0x43,0x7b,0xf6,0xb6,0xe0,0x23,0xbb,0xe1,0x44,
	0x6f,0x88,0x04,0xf1,0x07,0xdc,0xe6,0xd3,0xea,0x65,0x2f,0x92,0xbb,0x76,0x56,0x7a,
	0x5a,0x16,0xe2,0xea,0xb4,0xf7,0xd3,0xb8,0xa5,0x4a,0x5f,0x27,0xc7,0xf8,0x77,0xb5,
	0x42,0x55,0x94,0x88,0x11,0x0b,0xc9,0x7b,0xc8,0xe4,0x14,0xbf,0xcd,0xcb,0x0e,0x01,
	0x70,0x3a,0x82,0x9a,0x8b,0xa8,0x9b,0xe5,0xd5,0x4d,0x9e,0xf9,0x81,0xa2,0xcf,0xb9,
	0x97,0xea,0x05,0xeb,0x8d,0xc6,0x1f,0xc5,0x42,0x1f,0x13,0x8f,0xa0,0xb7,0x5b,0x3b,
	0x8c,0x42,0x29,0x41,0x70,0x94,0x5d,0x7a,0x55,0x2e,0x9c,0x4b,0xac,0x05,0x26,0x1d,
	0x5d,0xc1,0x7f,0xef,0x40,0xd0,0xb5,0xb9,0x25,0x4c,0x92,0x6e,0x4a,0xca,0x8d,0xcf,
	0x8a,0x46,0xba,0x0d,0x10,0xa0,0x8d,0x92,0xf5,0x70,0x7b,0x83,0xea,0x99,0x4e,0x33,
	0x83,0x4e,0x26,0x44,0xba,0xf0,0x42,0x51,0xb8,0x51,0x88,0x5a,0x3b,0xd3,0x3e,0xd9,
	0xb4,0xee,0x0e,0x9a,0xa7,0xe8,0x44,0x14,0xc0,0x51,0x6e,0x4a,0x4d,0xfa,0x4e,0x28,
	0xfd,0x73,0x2b,0x77,0x51,0xe1,0x41,0x94,0x72,0xcd,0x32,0xec,0xd1,0x8f,0xba,0x24,
	0xd4,0x53,0xaf,0x79,0xd2,0xaa,0x22,0xe6,0x12,0xb8,0x15,0x83,0x80,0xb2,0xd0,0x38,
	0x8d,0xcd,0x33,0xe7,0x10,0x1a,0x84,0x09,0x74,0xce,0x53,0xc6,0xc4,0xea,0xd7,0xdb,
	0xeb,0x07,0xab,0xbb,0x9e,0x69,0x37,0x50,0x40,0x93,0x7c,0xc5,0x6b,0x8b,0xa0,0x03,
	0x74,0x74,0x58,0x13,0x40,0x4f,0x05,0x46,0xce,0xc0,0x91,0x60,0xf9,0x26,0x8a,0x0b,
	0x78,0x95,0x7e,0x16,0xd8,0x37,0x00,0x1e,0x94,0x9a,0x5e,0xb0,0xaa,0x18,0x0a,0xaf,
	0x35,0x33,0xa7,0x1e,0xe1,0x58,0xd9,0x30,0x08,0xdf,0xd6,0x42,0xc0,0x5f,0x31,0xce,
	0x7d,0x20,0x97,0x1c,0x74,0x08,0x15,0x82,0x24,0xa6,0x37,0xad,0xa7,0x53,0xc7,0x41,
	0x3c,0xb6,0x4c,0x80,0x74,0x1d,0x19,0xbc,0xb3,0xac,0x94,0x3d,0xeb,0xa8,0xf5,0x38,
	0xed,0x50,0xc4,0xd7,0xe1,0x1a,0xbf,0xc9,0x8c,0x75,0x5b,0x14,0xa1,0x88,0x6f,0x75,
	0x2d,0x28,0xb3,0x9c,0xd8,0x2a,0x33,0x3a,0x8a,0x89,0x2a,0x85,0xd3,0xf9,0x7e,0xc1,
	0x0b,0xe9,0x17,0x80,0x40,0x97,0xab,0x98,0x4d,0x8f,0xf4,0x7b,0x6b,0xf5,0xd7,0x67,
	0x72,0x60,0x25,0x9e,0x9e,0x05,0x03,0xb8,0x03,0xa6,0xde,0x43,0xef,0x92,0x2a,0x61,
	0x91,0x6d,0x2e,0x47,0x10,0xac,0x70,0x77,0x6e,0xa5,0xb8,0x69,0x4f,0x0d,0x10,0x24,
	0xe1,0x0e,0x6d,0xa5,0xfb,0xf4,0xb6,0x84,0xf5,0xdc,0x5c,0x17,0x8e,0x9d,0x79,0x30,
	0xaa,0x1c,0x5d,0x1f,0x81,0xc6,0xe4,0x3d,0x86,0x9b,0x59,0x57,0x0b,0x51,0x6d,0xd9,
	0x8d,0xb8,0x30,0xa7,0xa3,0x44,0x24,0xb7,0x86,0xe1,0x93,0x43,0x00,0x5d,0x98,0xf7,
	0x02,0x41,0xe7,0xe3,0x3d,0xfd,0x73,0x88,0x69,0x1f,0x77,0x7b,0x1a,0xc2,0xa5,0xd5,
	0x91,0xc9,0x14,0xdc,0xee,0x65,0x68,0x6b,0x79,0x7b,0x7b,0x4a,0x3e,0xf7,0xa6,0x05,
	0x46,0xbb,0x19,0x72,0x37,0x05,0xcd,0x79,0x13,0x18,0x7c,0x61,0xeb,0x59,0xf1,0x46,
	0xee,0x5c,0x9a,0xbc,0xc1,0xb8,0xda,0x3c,0x77,0xb4,0xd3,0x9d,0x94,0xb2,0x7c,0x7a,
	0x30,0x81,0x0d,0x2e,0xbe,0x0e,0x7f,0x17,0x26,0x69,0xd5,0x4a,0xa1,0xdb,0x12,0x86,
	0xa3,0x65,0x9a,0xdb,0x3a,0xe6,0x55,0xc0,0x65,0xda,0x86,0xb5,0x03,0xef,0x52,0x53,
	0xb7,0xcb,0x4e,0x98,0x7c,0xf6,0x38,0x3c,0xa6,0x28,0x1b,0x0f,0x14,0xd8,0x1e,0xa9,
	0x3c,0x6d,0x4d,0x5c,0x72,0xbf,0xb2,0x32,0xb5,0x14,0x88,0x21,0x54,0x1f,0x61,0x4a,
	0x20,0x53,0xf7,0x48,0xf9,0x4a,0x66,0xe7,0xfc,0x21,0x06,0x72,0x15,0x2d,0xdd,0x5c,
	0x73,0x2e,0x48,0x7d,0x13,0xab,0xe3,0x62,0x51,0xe1,0xd0,0x9c,0xbc,0xaf,0x9e,0xc0,
	0x6a,0xf6,0xdb,0x64,0x38,0x21,0xdf,0x92,0x2f,0x07,0xa8,0x6c,0x37,0xfc,0x4b,0x3a,
	0xca,0x06,0x1e,0x2a,0x13,0x40,0xcd,0x00,0x11,0x7d,0xa0,0x51,0xca,0x36,0x8e,0x25,
	0x11,0xd5,0x48,0x0f,0xa4,0xdc,0xcb,0xdb,0xfd,0x17,0x71,0x75,0x95,0xc2,0x40,0xbc,
	0xce,0x9c,0x59,0xd1,0x21,0x3c,0x84,0xe6,0x6c,0x95,0xa3,0xd4,0x8c,0xcf,0x89,0xa8,
	0x6e,0xf3,0xed,0xb4,0xa7,0x6b,0xfd,0x1c,0xa1,0x02,0x1a,0x38,0x52,0x33,0x4e,0xd2,
	0x95,0x6d,0x9d,0x5f,0x44,0x5e,0xfc,0x2c,0x10,0x73,0xa8,0xc9,0x32,0xb9,0xe5,0xbb,
	0x74,0xae,0xb9,0xba,0x38,0x6c,0x29,0x06,0x19,0x81,0x04,0xc3,0x4a,0x3d,0x4d,0xb0,
	0x94,0x64,0x66,0x8c,0xc4,0xd5,0xdc,0x16,0x62,0x80,0x67,0x2e,0xc6,0x30,0x9a,0xa2,
	0xbe,0x0f,0x5e,0xce,0x83,0x67,0xe5,0x2e,0x2e,0x60,0x59,0x64,0x86,0x59,0x5f,0x77,
	0x1f,0xcc,0xa8,0xa2,0x64,0x38,0xfa,0xb2,0x55,0x5d,0xf4,0x5f,0x27,0xfe,0x46,0xd7,
	0x92,0x6d,0x2e,0xc8,0xf4,0xe5,0xcc,0x7a,0x94,0xab,0x2a,0xe2,0xaa,0x63,0xe0,0x4e,
	0x12,0x5b,0xf5,0x86,0xee,0x80,0xc6,0x33,0xc9,0x6f,0xfb,0x1e,0xbc,0xb1,0xcf,0xd2,
	0x2c,0x23,0xd2,0x2b,0xb9,0xd2,0x56,0xf3,0x18,0xa7,0x38,0x4f,0x5b,0xcc,0xb2,0xa5,
	0xe6,0x14,0x2b,0x10,0xcb,0x7f,0x0d,0xab,0x7e,0x64,0x4d,0xa7,0xa0,0xa2,0x7b,0xfd,
	0xdf,0xba,0x9d,0x3a,0xbb,0x60,0x95,0xde,0xd0,0xab,0x8d,0x0b,0x7b,0x3a,0xf3,0xcb,
	0xb1,0x9c,0x1c,0x62,0x44,0xba,0x17,0xd1,0xaf,0x25,0xba,0x95,0x9e,0xc9,0xdd,0xd2,
	0x69,0x2c,0xab,0x20,0x53,0x6b,0x79,0xe8,0x66,0x45,0xe1,0x73,0x8d,0x6f,0xba,0x9f,
	0x98,0x67,0x88,0x08,0x00,0x0f,0x9a,0xe1,0xa4,0x5c,0x87,0xb7,0x6b,0x2f,0xb2,0x59,
	0x1c,0xe9,0x91,0x9c,0xd8,0xa2,0x55,0xdb,0x81,0xf1,0x90,0xec,0x6f,0x28,0x39,0x69,
	0x4f,0x58,0x06,0x91,0x91,0xcd,0x9b,0xea,0x78,0x56,0x40,0x4a,0x7b,0x2a,0x7e,0x7d,
	0xfb,0x19,0x6d,0x16,0x0e,0x80,0x47,0x2e,0x4d,0x4d,0xa9,0x75,0xcc,0x72,0x93,0x13,
	0x6d,0x80,0x6f,0xdc,0xb6,0x4d,0x9a,0x07,0x75,0xbe,0xdc,0x30,0x93,0x3d,0x55,0x54,
	0x2a,0x21,0x06,0x09,0x87,0x81,0xce,0x70,0xc4,0xd2,0x44,0x50,0x92,0xa5,0x35,0x2e,
	0x7c,0xe7,0x20,0x6a,0xb7,0x70,0x88,0x6c,0xc0,0x9c,0x0a,0xd4,0x71,0xa8,0x3a,0x65,
	0xfd,0xcc,0xa8,0x90,0x40,0x1d,0x74,0xcb,0x51,0x84,0x30,0x5b,0xb3,0xc5,0xc1,0xa8,
	0xbd,0x09,0x50,0x0f,0x30,0x91,0x5a,0x37,0x83,0x14,0x19,0x30,0xd5,0xea,0x0c,0x96,
	0xf9,0x88,0xad,0xda,0x17,0xfb,0x9a,0xca,0x11,0x1b,0xee,0x7e,0x0c,0x5c,0x7a,0x92,
	0x6b,0xc5,0x75,0x73,0x27,0xa1,0xb5,0xee,0xdb,0xb1,0xb9,0x29,0xf7,0x76,0x56,0xe9,
	0x34,0xbb,0x33,0x4b,0xc4,0x0e,0x47,0xac,0x56,0xa8,0x3b,0x45,0xdf,0xea,0x5c,0x56,
	0xad,0x39,0xde,0xae,0x06,0x36,0x4f,0x77,0x9a,0x4d,0xdc,0x97,0xcf,0x3a,0x30,0x0c,
	0xbe,0x95,0x37,0x67,0x03,0xa5,0x3d,0x30,0xa7,0xbf,0x86,0xa1,0x5a,0x61,0x09,0x76,
	0x7d,0x9e,0xb9,0x8e,0x19,0xe8,0x6b,0xde,0xfe,0x47,0xd9,0x91,0xa0,0xb9,0xba,0xc0,
	0x01,0x32,0xa4,0x85,0x0a,0x52,0xdf,0x58,0xc0,0x90,0x2a,0xab,0xf1,0xc3,0xf6,0x9c,
	0xb9,0x5e,0x11,0xfc,0xe1,0xf6,0xda,0x71,0x73,0x1e,0xd1,0xdb,0x54,0x19,0x75,0x2e,
	0xe8,0xfe,0x80,0x25,0xe6,0x16,0xa2,0xe5,0xd6,0x78,0xf2,0x95,0x10,0x2f,0x89,0x90,
	0xe9,0x06,0xdb,0x7b,0xc2,0x33,0xa1,0xc8,0x66,0x38,0x13,0xe6,0x13,0x18,0xb9,0x2e,
	0xda,0x4a,0xfe,0x3e,0x45,0x16,0x59,0x8b,0x61,0x2b,0x1f,0x2f,0xa6,0xa1,0xba,0x9c,
	0x36,0xb0,0x5e,0xa9,0x31,0xf6,0xb7,0xb2,0x32,0x20,0x36,0x4c,0x4d,0x69,0xdb,0xc0,
	0x4f,0x28,0x62,0x58,0x45,0x03,0xe8,0xd8,0x0e,0x21,0xd1,0xdd,0x2f,0x39,0x2c,0x76,
	0x3d,0x99,0x27,0x3b,0xc2,0x99,0x02,0x36,0x91,0xe7,0xdf,0x73,0x1d,0xda,0x37,0x0c,
	0x88,0xd6,0xc6,0xbe,0x77,0x25,0x8a,0x8d,0xdb,0x17,0xfd,0xa7,0xb4,0xad,0x20,0x56,
	0x80,0xf4,0xb3,0xc9,0x80,0xa3,0xf8,0x9e,0x86,0x9f,0x0c,0x2c,0x67,0xb5,0x82,0x34,
	0xf9,0x4a,0x9f,0x3e,0x5b,0x0c,0x68,0xf7,0x92,0xdc,0x28,0x68,0x19,0x37,0xd1,0xcc,
	0x66,0x42,0x04,0x53,0x4e,0xa4,0x87,0x17,0xdf,0x83,0x59,0xd4,0x26,0x91,0x54,0xf9,
	0x13,0x01,0x37,0xeb,0x52,0x88,0xd0,0x73,0xe2,0x7a,0xde,0x12,0x8e,0x13,0x8b,0xd8,
	0xd4,0x38,0x40,0xd3,0xe5,0x56,0x9e,0x80,0x59,0x50,0xbe,0x25,0x5d,0x34,0x61,0x8f,
	0xe3,0x79,0xad,0xcd,0xcd,0x79,0x2d,0x20,0x18,0xfd,0xe1,0xb4,0xcf,0x02,0x5c,0x96,
	0x63,0xa4,0x2d,0xea,0xa3,0x36,0xb7,0x42,0x08,0x92,0x76,0xc6,0x77,0x18,0x2d,0x88,
	0x37,0x10,0xe0,0xcb,0xab,0x49,0xdc,0xa6,0x9f,0xf5,0x15,0x84,0x0e,0x4a,0xfe,0x79,
	0x12,0xe0,0xbc,0xf9,0x1f,0x0d,0x7b,0x30,0x89,0x62,0x30,0x49,0x66,0xc4,0xb5,0x3f,
	0xfc,0x8e,0x03,0x69,0x26,0x26,0xe1,0x23,0xaf,0xdc,0x42,0xf8,0x64,0x81,0x7e,0xf5,
	0xc3,0x42,0xac,0xd6,0x3f,0xb5,0x88,0xe2,0x6e,0x0c,0xf7,0xd6,0x6a,0xa2,0x3d,0x22,
	0x48,0xdf,0x8b,0x74,0xe4,0x8c,0x2a,0xec,0x33,0x18,0x70,0xbd,0x5e,0x64,0x2a,0xe7,
	0xd8,0x43,0x7b,0xf4,0xe3,0xf3,0xaf,0x60,0xe2,0x43,0x04,0xe1,0x71,0x99,0x2c,0x20,
	0xa8,0x7f,0x51,0xf5,0x0d,0x4d,0x23,0x5b,0x31,0xa9,0x33,0x64,0xec,0xbd,0xf7,0x99,
	0x6f,0x01,0xcf,0xfe,0xd3,0xca,0x10,0x3e,0xa5,0x74,0x5c,0x91,0xdf,0x18,0xfb,0xf4,
	0xcf,0x30,0xde,0x31,0xc1,0x88,0x9a,0x0a,0x19,0xa3,0x8f,0x30,0xa8,0xa4,0xf9,0x0c,
	0xdf,0xef,0x27,0xc9,0x59,0x5a,0x42,0x72,0xad,0x48,0x44,0x44,0x99,0x17,0xf4,0x37,
	0xb9,0x22,0x0d,0xcc,0xc5,0x16,0xca,0xec,0x71,0x81,0x87,0x22,0x01,0xe9,0xcc,0x27,
	0xac,0x52,0x6a,0x04,0xa0,0x25,0x00,0x57,0x5a,0x53,0xec,0xd3,0xed,0xdd,0x39,0x6c,
	0x9c,0x6b,0xb3,0x88,0x13,0xf8,0x21,0x86,0x85,0x6d,0x6b,0x72,0x8f,0xfe,0x2a,0x68,
	0x98,0x2c,0x62,0x12,0x0f,0x54,0x54,0xb4,0xd7,0xbd,0x99,0xac,0x15,0x47,0xe1,0x0a,
	0x02,0x7f,0x46,0x2f,0xdb,0x2b,0xb4,0xe0,0x83,0x02,0x30,0x1a,0x7d,0x71,0x21,0x83,
	0xd3,0x2f,0x80,0x84,0x66,0x67,0xe2,0x38,0x3f,0x52,0x01,0x6f,0xb1,0xfa,0x5c,0x0e,
	0xa3,0x28,0x47,0x47,0x4c,0xfd,0xe8,0x6b,0x7e,0xcc,0x98,0xa6,0x8e,0x13,0x08,0x02,
	0xa2,0x54,0x69,0x77,0x45,0xa1,0x7f,0x7c,0x6c,0x28,0x55,0x8d,0x64,0xa9,0x7c,0xd3,
	0xf9,0xbe,0x59,0xbe,0x69,0xec,0x1d,0x24,0x88,0xcc,0xe5,0x44,0x02,0xb9,0xab,0xf6,
	0x4f,0xa4,0x67,0xca,0x26,0xa2,0xa6,0x07,0x78,0x13,0x40,0xee,0x5b,0xcb,0x6d,0xf0,
	0xf2,0x48,0xae,0xdb,0x28,0x2a,0x64,0x3d,0x53,0x54,0x2e,0x7f,0x0b,0xda,0x43,0xce,
	0x07,0xd2,0xb3,0x48,0x8b,0x72,0xe1,0x0e,0x9f,0x98,0xe1,0xc6,0x36,0x4e,0xdc,0xac,
	0xbb,0x00,0x1a,0x50,0xd1,0xca,0xfa,0x74,0x79,0xe1,0x6b,0xa6,0xc8,0xdd,0xd0,0x5a,
	0x65,0x1d,0x9f,0x06,0xb8,0x90,0x66,0x9f,0xcf,0x2a,0x96,0x59,0x61,0x5b,0xa6,0xc5,
	0xbe,0xda,0x39,0x30,0x3c,0x06,0x44,0xb4,0x82,0x0d,0xd7,0x83,0xc1,0xec,0x04,0xca,
	0x85,0x52,0xe2,0xca,0x5c,0xa3,0xc3,0xab,0x0f,0xf9,0x1b,0xe0,0x9d,0x6f,0x84,0xbf,
	0x59,0xfe,0x0d,0x2d,0x17,0x6c,0x67,0x24,0x34,0x6c,0x32,0x78,0xbb,0x00,0x5b,0xad,
	0xa0,0x66,0xb9,0xa8,0xfe,0x23,0x4e,0xb4,0xfb,0x95,0x93,0x81,0x2e,0xac,0x4c,0xe0,
	0xaf,0xee,0x69,0x9a,0x3b,0x13,0x64,0xbd,0xb2,0x1d,0xb7,0x87,0xf2,0x9b,0x67,0xca,
	0x2b,0x92,0x96,0xbe,0xc1,0x9a,0x0c,0xc8,0xf6,0xdc,0xb2,0xf7,0x91,0x1e,0x33,0x30,
	0x16,0xf0,0xf0,0xc1,0x4c,0x76,0xc3,0x02,0x2c,0x98,0xc4,0xd7,0xf7,0x5b,0xd9,0xaa,
	0xca,0x1d,0xa7,0x70,0x4f,0xd2,0x0a,0xc9,0x3e,0x49,0xf6,0x67,0xa6,0xe6,0xb6,0x67,
	0x26,0xef,0xbf,0x10,0x04,0x10,0xed,0xd9,0x2a,0x38,0x6a,0xd0,0x0e,0x1f,0x3e,0xb5,
	0x95,0xd9,0xbc,0xc7,0xf2,0xce,0x8e,0xc5,0x72,0x2b,0xdf,0x83,0x85,0x35,0xa5,0x4d,
	0x90,0xde,0x8b,0x91,0x65,0x3a,0xdf,0xdb,0x9d,0xc9,0xbb,0x8c,0xea,0x34,0xac,0xb0,
	0x3f,0x2a,0xb8,0xc5,0x15,0x0d,0x22,0x26,0xbc,0x9b,0x65,0x74,0x55,0xc1,0xd5,0x7a,
	0x7d,0x7a,0x88,0x0e,0xba,0x16,0xc7,0x4b,0x8f,0x02,0x5d,0x34,0x2c,0x3b,0x99,0x03,
	0xe5,0x37,0x62,0xf3,0x39,0x7c,0x29,0x2f,0x44,0x0c,0xe4,0x4c,0x53,0x7a,0xaa,0x31,
	0x8d,0x55,0x0a,0x8f,0xf4,0x78,0x29,0xa8,0x00,0x48,0x9a,0xfc,0x54,0xb7,0xc8,0x53,
	0xcd,0x2c,0x10,0x77,0xdc,0x4d,0xc4,0x28,0x28,0xab,0x25,0x6b,0x54,0x4c,0xce,0xe4,
	0x93,0x78,0xae,0x26,0x00,0x4d,0x29,0xc5,0x8b,0xd5,0xd3,0xd3,0x97,0xb6,0x40,0x60,
	0xd2,0xd0,0xed,0xfd,0x87,0xee,0x58,0x59,0x64,0xfa,0x4d,0x93,0x2e,0xe7,0x63,0x80,
	0x8e,0x0d,0xa3,0xab,0xb2,0x6e,0xc7,0xc7,0xda,0x98,0xf1,0xcc,0x40,0x18,0xd3,0xfa,
	0xac,0x07,0x96,0x6c,0xd9,0x04,0x9c,0x5c,0x22,0x2e,0x5a,0xb8,0xd4,0x83,0x0c,0xae,
	0x60,0xdd,0xf2,0xc0,0x12,0xf1,0xdb,0xac,0x5a,0xba,0xb2,0x69,0x24,0xa4,0xad,0xf8,
	0x82,0xdd,0xe1,0xf1,0x02,0xcc,0x62,0xd8,0x87,0xa0,0x62,0xb0,0x0e,0x3d,0x55,0xf2,
	0x51,0x15,0xc5,0xbb,0x29,0xda,0xd9,0x75,0xbf,0x37,0x7b,0x66,0xe5,0xba,0xf1,0x54,
	0x47,0xbe,0x22,0x93,0xb6,0x7a,0x8a,0x26,0x47,0xd2,0xc8,0x8d,0x8f,0x35,0x0e,0xbd,
	0xef,0xa3,0xf6,0xeb,0x4f,0x7a,0xf1,0xa3,0x45,0x66,0x68,0x7e,0x47,0xd8,0x42,0xe5,
	0x4f,0xd0,0x20,0x22,0x93,0x86,0x5c,0x7c,0xc2,0xb2,0xee,0x71,0xf6,0x77,0x42,0xa9,
	0x80,0x1c,0xc7,0xca,0xcc,0x92,0xd0,0x0a,0xbb,0xbd,0xd3,0x59,0x70,0x71,0x0e,0x0e,
	0x15,0xcd,0x22,0x71,0xe2,0x93,0x33,0xcf,0x31,0x5e,0xb5,0xef,0xe4,0xdd,0x6f,0x5e,
	0x09,0xf7,0x2a,0xbb,0x7a,0xdb,0x82,0x7f,0xd1,0xfb,0x48,0xed,0xc7,0xc3,0xe3,0xa8,
	0xb7,0xa7,0xc8,0x34,0x88,0x7f,0xc2,0x4a,0xef,0xae,0xe2,0x0b,0xf4,0x79,0x1c,0xa0,
	0xd5,0x4e,0xe1,0x40,0x10,0x39,0x64,0x1e,0xc9,0x1d,0x33,0x16,0x12,0x58,0x80,0x7f,
	0xb7,0xe3,0xcc,0xc6,0xa9,0x91,0x6f,0x35,0xec,0xcb,0x8f,0x8b,0xcb,0xcb,0x2d,0x8a,
	0x5f,0x1f,0xdc,0xdd,0xca,0xe1,0xc9,0x67,0xd3,0xcf,0x5b,0xd3,0x06,0x0a,0x99,0x42,
	0x07,0x16,0x06,0x4d,0x11,0x01,0x99,0x59,0x40,0x13,0x0d,0x60,0x7d,0x74,0x7e,0x95,
	0xf3,0xeb,0x2d,0xae,0x78,0x0e,0x28,0x62,0xcd,0x84,0xd1,0xa1,0x28,0xf8,0xfc,0xb9,
	0x4d,0x74,0xd9,0x93,0x7d,0x12,0xcb,0x4c,0xa7,0xeb,0x71,0xab,0x99,0xae,0x59,0x89,
	0x7c,0x49,0x27,0xab,0x07,0xf1,0x21,0xe1,0x94,0x4c,0x7a,0x73,0x7b,0xbb,0x39,0x88,
	0xae,0xf8,0xf3,0x33,0x25,0x32,0xf8,0x21,0x41,0xb4,0xda,0x1d,0x85,0x0d,0xc6,0x0e,
	0x88,0x97,0x26,0xf5,0xaf,0xf4,0x51,0x5d,0x57,0x6a,0x06,0xc3,0x7e,0x01,0x6b,0x8f,
	0xf8,0xd5,0x3c,0x70,0xaa,0x2c,0xac,0x9c,0xf6,0x47,0x50,0x26,0x90,0xab,0xb4,0x03,
	0xf1,0x06,0xdc,0x03,0xa2,0x61,0x98,0xe6,0x7b,0xee,0x84,0x20,0x56,0x7b,0x67,0x82,
	0x97,0x75,0xe7,0x7b,0x64,0x21,0x21,0x7f,0xaa,0xd4,0x7f,0x3d,0x7c,0xed,0x8d,0x88,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x32,0x03,0x52,0xb4,0x6f,0x93,0xf2,0x6e,0xdb,0x2e,0x43,0xe9,0x15,0x0c,0xcf,0x38,
	0x3b,0x1a,0xce,0xdc,0x62,0x0a,0x80,0xaf,0x12,0x6f,0x9d,0x56,0xb8,0x04,0xd5,0x86,
	0x29,0x55,0x2e,0xd8,0xec,0xf0,0xab,0xb3,0x4c,0xa5,0xe8,0x0e,0x0f,0x33,0xec,0x81,
	0x44,0xaf,0x75,0x08,0xfd,0x18,0x66,0xd2,0x57,0xa1,0xd3,0xcd,0xf8,0xb4,0x9a,0x77,
	0xed,0x0d,0x90,0xe9,0xf6,0xaa,0x87,0xc2,0x04,0xfb,0xd6,0xd8,0xe8,0x82,0xd1,0x7f,
	0x35,0xad,0x99,0xfc,0xae,0x29,0x9a,0x0d,0x43,0x89,0x4d,0x0a,0x12,0x35,0xdf,0x49,
	0x3c,0x0a,0x6a,0x61,0xc2,0xd5,0xba,0x8a,0x31,0xb5,0xb6,0x2d,0x89,0xf9,0xa7,0x01,
	0xb1,0x9e,0x09,0x11,0x82,0xac,0x2d,0x0e,0x99,0xa1,0xb8,0xfb,0x64,0xac,0xf4,0xe2,
	0x37,0x62,0x90,0x66,0x90,0x14,0x99,0xbd,0x6a,0xb8,0x24,0xa6,0xc6,0x26,0x0c,0x9e,
	0x1b,0x55,0x7f,0xb7,0xe0,0x8c,0xaa,0x28,0x76,0x77,0xb4,0xb4,0xd3,0xcb,0xf6,0x18,
	0xbc,0x34,0x6c,0x93,0xd9,0x28,0x1d,0xba,0x95,0xcc,0x6d,0xe2,0xa3,0x61,0x0a,0xa4,
	0xba,0x71,0x9d,0x01,0xdb,0x70,0x87,0xdb,0xf2,0x4b,0xd0,0x3a,0x73,0x0f,0x02,0x67,
	0xc8,0xfd,0xdc,0x2a,0x04,0x61,0xd9,0x0d,0x34,0x76,0x8b,0x01,0x97,0x9d,0x02,0xab,
	0x47,0xe3,0x0f,0x8c,0x42,0x30,0xe0,0xc1,0xed,0xd3,0x0a,0x2a,0xc2,0x9c,0x0a,0x3b,
	0x47,0x7d,0x1a,0xc1,0xac,0x84,0x7a,0xed,0xb8,0x5f,0xe1,0xcb,0xc3,0x5d,0xe9,0x81,
	0xc8,0x60,0x73,0xfa,0xce,0x98,0x20,0x85,0xbf,0x3e,0xe5,0x46,0x43,0x07,0x07,0xc3,
	0xbe,0x62,0x93,0x29,0xe4,0x0b,0xfe,0xd4,0xd4,0xe3,0x54,0xab,0x78,0xcd,0x91,0x6a,
	0xb7,0xa0,0xec,0x00,0x14,0x12,0xba,0xe7,0x64,0xf3,0x78,0x2b,0x02,0x61,0xa4,0x2f,
	0x00,0x79,0x41,0x0d,0x10,0xde,0x0b,0xb4,0x3c,0x98,0x0f,0xae,0xf1,0xd0,0xbc,0x65,
	0xb9,0x4d,0xdf,0x0a,0x10,0x4f,0x7a,0xa8,0x1b,0xdc,0x8d,0xec,0x9e,0x57,0xcc,0x41,
	0x5f,0x87,0xc5,0x3a,0x39,0x9c,0xba,0xe7,0x57,0x82,0x30,0x76,0x76,0x63,0x4a,0xdf,
	0xe1,0xc8,0x7c,0xb6,0xc2,0x2d,0x6b,0x59,0x19,0x14,0xf7,0x62,0xdb,0xb6,0x52,0x93,
	0x03,0x16,0xba,0x3b,0xe4,0xe1,0x58,0xfc,0x16,0x4c,0xa7,0x3e,0xc6,0xf1,0xc6,0x96,
	0xd1,0x20,0x75,0xdc,0x4e,0x52,0x0a,0x04,0xe6,0x73,0x65,0x82,0x19,0x4e,0xd4,0x01,
	0xd6,0x2e,0x77,0xc7,0x27,0xd2,0x40,0x8d,0xb5,0x14,0x60,0x31,0x9e,0x3a,0x4d,0xe8,
	0x81,0xf6,0x32,0x65,0xd3,0x9c,0xfe,0x37,0x6b,0x5d,0x13,0xac,0xa9,0xc3,0x6d,0x76,
	0xba,0x6c,0xac,0x15,0x2c,0x09,0xfe,0x5d,0x77,0x68,0xf4,0xcb,0x1b,0xbb,0x43,0x1b,
	0x05,0xd6,0x1c,0xcd,0xed,0x80,0xbb,0x7b,0x40,0x1f,0x52,0xe1,0x39,0xbf,0x3e,0x21,
	0x17,0xf3,0x05,0x9e,0x42,0xb9,0x13,0x78,0xa7,0xb5,0xd8,0xbb,0x62,0x41,0x83,0xf2,
	0x49,0x05,0xae,0xc7,0x5f,0x14,0x6b,0xe7,0x04,0xf0,0x9c,0x2d,0x7a,0x01,0x16,0x78,
	0xaf,0xad,0x0b,0xdf,0x2d,0x48,0x26,0xa1,0xf6,0x4d,0x1f,0x88,0x54,0xe5,0x31,0xb5,
	0xd1,0xb9,0x01,0x6a,0x90,0xa4,0x16,0x38,0x0f,0x96,0xa8,0xfc,0x0e,0xe9,0x3c,0x87,
	0x4b,0x17,0x31,0x53,0x39,0x8d,0xca,0x3c,0x44,0xbf,0x3e,0x3d,0x0a,0xe9,0x83,0xed,
	0x0f,0xa0,0xc1,0x75,0x0f,0xe3,0xf8,0xac,0x49,0xbf,0x5e,0x71,0xce,0xb5,0xb6,0x6d,
	0xa4,0xa3,0x8a,0x11,0xf0,0xd5,0x7c,0x76,0x40,0x7e,0xca,0x02,0xce,0xcf,0x43,0xff,
	0xbb,0xe9,0xb9,0xba,0x62,0xfa,0x21,0xf3,0xa2,0xfa,0x88,0x74,0x8f,0x69,0xc2,0xc1,
	0x02,0x28,0x52,0xad,0xc1,0xa3,0x71,0xc2,0x3b,0x9e,0xf3,0x94,0x38,0xd3,0x44,0xf1,
	0x69,0xda,0xbc,0xd1,0xe1,0x03,0xc0,0xc2,0x81,0x39,0xc4,0xd6,0x7d,0x69,0x70,0xd2,
	0xda,0x6a,0x21,0x9c,0x65,0x76,0xf1,0xdc,0xc4,0x57,0x38,0xd1,0xcd,0xb1,0xfe,0xf8,
	0xb4,0xda,0x90,0xde,0x41,0x5c,0xea,0xaf,0xb7,0x86,0x58,0x2f,0xec,0xc2,0xea,0xe6,
	0x86,0x05,0xbf,0xae,0x37,0xee,0x38,0x15,0x94,0xc5,0xcc,0x43,0xe9,0x05,0x38,0x25,
	0x63,0x96,0x9f,0x84,0x12,0xea,0x75,0xe0,0x9e,0x4e,0xea,0xd5,0x7b,0xff,0xe9,0x1c,
	0x72,0xb9,0xed,0xb8,0xa1,0xd0,0x6e,0xa6,0x09,0x01,0x0d,0xbe,0x32,0x4a,0x13,0x9b,
	0x2a,0x7a,0xf6,0x7f,0x81,0x27,0x97,0x3c,0x52,0x59,0x06,0x6a,0x4f,0x07,0xa4,0xa9,
	0x68,0xea,0x7b,0x73,0x5a,0x18,0xab,0x3f,0x5d,0x16,0x68,0x38,0xe5,0xb1,0x3e,0xb7,
	0xd0,0x28,0xf1,0x08,0xa2,0x8a,0x6f,0x8f,0x7a,0x0f,0xa3,0x99,0x84,0x31,0x2c,0x2f,
	0x35,0xf0,0xa8,0x8e,0xa1,0xea,0xc4,0x5b,0x6b,0x8c,0x8a,0x09,0x84,0x45,0x97,0x1b,
	0x8d,0x71,0xc8,0xa2,0x49,0x3d,0xdd,0xb8,0xb2,0x9f,0xe5,0x3c,0x67,0xa2,0x21,0x83,
	0xb3,0xed,0xeb,0x72,0xa1,0xbe,0x92,0xfd,0xfa,0xe1,0x2b,0x77,0xa0,0x0d,0x05,0xc6,
	0xe7,0xbf,0x6d,0xb8,0xf2,0x0d,0xa7,0xd8,0x3b,0x09,0xb1,0x43,0x23,0xd5,0x89,0x19,
	0xea,0xeb,0x2d,0xd5,0x31,0xd6,0xbb,0x99,0x8b,0xf4,0x81,0xc3,0xc2,0x77,0x84,0x35,
	0x59,0x73,0xce,0xb6,0xc5,0xde,0x93,0x31,0x0d,0xf0,0x80,0xc2,0xb6,0xbc,0xa6,0xda,
	0xea,0x13,0xd9,0xbb,0x54,0x85,0xcb,0x87,0xa6,0xa3,0xc4,0x97,0x85,0x5c,0xfc,0x82,
	0x5a,0xe1,0xdf,0x7a,0x1b,0x59,0xd1,0x6e,0x8b,0x04,0xce,0x73,0x28,0x81,0x7a,0x83,
	0xed,0xd5,0xb4,0xc0,0xce,0xaf,0x6d,0x21,0x14,0xcb,0x91,0xa3,0x13,0xe3,0x41,0x4c,
	0xb9,0x94,0x57,0xe2,0xe6,0x69,0x02,0x2a,0xc5,0xc9,0xf8,0xda,0x0c,0xde,0x92,0x07,
	0x76,0x65,0xfd,0x5e,0x36,0x0d,0x42,0xbb,0xca,0xe0,0x9c,0xc6,0xde,0x65,0x9f,0xda,
	0xea,0xbc,0x71,0x2f,0xb0,0x25,0x07,0x56,0xbf,0x61,0xf5,0xbb,0x9e,0x9d,0xbe,0xfa,
	0xf6,0x92,0x1e,0x84,0x22,0x59,0x45,0x2d,0xb0,0x91,0xe2,0xc2,0x04,0xaf,0xfc,0x93,
	0x48,0x85,0x4f,0x1f,0x03,0x74,0x85,0xb7,0x1d,0xb4,0x7c,0xe3,0x37,0x7d,0x35,0x36,
	0x24,0x24,0xdb,0x38,0x9c,0x94,0x65,0xbe,0x85,0x67,0xf1,0x0f,0x86,0x44,0x00,0x57,
	0xca,0x15,0xdc,0x24,0x42,0xc1,0x20,0xab,0xa4,0xd3,0x8a,0xd7,0x09,0xe5,0xc0,0x60,
	0x5e,0x87,0xd7,0xe7,0xb3,0x25,0x75,0x22,0x54,0xb4,0xd3,0x80,0x3c,0x9b,0xdf,0x53,
	0x3e,0x43,0xd3,0xc3,0x92,0x64,0x7f,0x33,0xf1,0xdd,0xec,0x34,0x8b,0xb8,0x8a,0x71,
	0x38,0x48,0x09,0x64,0x61,0x15,0x59,0xeb,0x5b,0x5e,0xf7,0xd0,0x0c,0xd0,0x19,0xa1,
	0xad,0x8e,0x9e,0x3a,0x8a,0x50,0xdc,0x92,0xbc,0x9c,0xbc,0x2d,0xb0,0xb1,0x08,0x30,
	0x88,0x14,0x32,0x9f,0x45,0xe3,0x49,0xae,0x00,0x3b,0xb4,0xa0,0xf2,0x88,0xb5,0x34,
	0xa7,0x2e,0x41,0x00,0x1c,0x72,0xb1,0x39,0x24,0xa6,0x9a,0xf9,0x7e,0xfb,0x80,0xae,
	0x67,0x0c,0xf1,0x32,0x35,0x75,0x68,0xd3,0xff,0x57,0xee,0x5d,0x24,0xcf,0x46,0x31,
	0x28,0xa6,0xd7,0x9d,0xe4,0x33,0x3b,0xa5,0xd6,0xcb,0x9e,0x92,0x1e,0xca,0x60,0x9d,
	0xad,0x02,0x9c,0x61,0xea,0x0a,0xd9,0x6b,0x20,0x06,0xfa,0x95,0x46,0xfa,0xee,0x34,
	0x99,0x38,0xeb,0x8b,0x50,0x82,0xc1,0x16,0x28,0xe9,0x95,0xca,0x56,0x1d,0xeb,0x15,
	0x79,0xce,0xa6,0x13,0x3b,0x3e,0x48,0x0b,0xda,0xdb,0xee,0x9f,0x49,0x6f,0xf2,0x8d,
	0x58,0x58,0x94,0x52,0xdd,0x49,0x34,0xac,0xe4,0x6b,0xa3,0x0a,0xe1,0xc0,0xa5,0x29,
	0x3b,0xfc,0x19,0x81,0xe6,0x34,0x27,0x5c,0x73,0x1c,0x90,0xeb,0x2b,0x7b,0xf3,0xc6,
	0x18,0x15,0xbf,0xd3,0xcf,0x91,0xc1,0x29,0x4b,0xa1,0xb5,0xe6,0x8c,0x36,0xf7,0x09,
	0x28,0xb3,0x53,0x44,0x00,0xea,0x47,0xb1,0xd9,0xab,0x43,0x74,0x37,0x6e,0x8a,0x83,
	0x1f,0xed,0xe0,0x93,0x71,0xca,0x32,0x2b,0xc1,0x62,0x75,0x08,0x0b,0x07,0xbd,0x56,
	0xec,0x81,0x36,0x16,0x7f,0x60,0xe1,0x80,0x37,0x95,0x76,0x39,0x53,0x1b,0x51,0x9d,
	0x38,0x6c,0xa1,0x62,0x33,0xb3,0x1a,0xb3,0x2f,0xfd,0x2c,0xc2,0xa3,0x76,0x44,0x8c,
	0xf8,0x5b,0xee,0xc8,0x7f,0xa8,0x66,0xf7,0x1d,0x63,0x34,0xc6,0x7b,0x6f,0x79,0x40,
	0x1e,0x50,0xcc,0xb4,0xa9,0x68,0x1e,0x01,0xa5,0xc9,0x1c,0x8c,0x48,0x73,0x4c,0x55,
	0x24,0xd5,0xe9,0xa8,0xf8,0xa4,0x14,0x84,0xac,0x4e,0xa0,0xe1,0x93,0x5b,0x5c,0x31,
	0x01,0x99,0x9e,0x69,0x16,0xa4,0x60,0x1b,0xdb,0x64,0x3a,0xa2,0x26,0x82,0x0b,0x4b,
	0x5c,0x0f,0x4d,0x9e,0xce,0x2a,0x8c,0xd9,0xd8,0xaf,0xf7,0x89,0x2a,0xfe,0x76,0xba,
	0xdd,0xf4,0x62,0xda,0x62,0xde,0x4b,0x8c,0x6a,0x61,0x99,0x9f,0xcd,0x54,0x7c,0xaf,
	0x91,0x5c,0x9e,0xe9,0x16,0xc6,0x41,0xfc,0x2b,0x17,0x8d,0x45,0xac,0x01,0xb2,0xdc,
	0xe3,0xa9,0x6e,0x81,0x93,0xd3,0xab,0x3e,0xd5,0x00,0xbe,0x80,0x77,0x99,0x32,0x29,
	0xc4,0xb1,0x58,0x36,0x9f,0x98,0x25,0xce,0x20,0x34,0xfb,0x36,0x92,0x67,0xc2,0xbe,
	0xfa,0x34,0xba,0xdf,0xbb,0x89,0x62,0x3d,0xa7,0x52,0x7b,0x21,0x1d,0xe7,0x53,0xd7,
	0x1d,0x1b,0x3d,0x44,0x65,0x84,0x83,0x86,0xb8,0x82,0x9a,0x60,0x11,0xbb,0xbf,0xeb,
	0x0c,0x39,0xaf,0xff,0xb4,0x11,0x0c,0x18,0x07,0x2a,0x1c,0xf6,0xe2,0xcd,0xa0,0x6b,
	0x26,0x97,0xe1,0xc7,0xb3,0xd2,0x2e,0x9d,0x07,0x99,0xc1,0xf5,0x89,0x59,0x0c,0x11,
	0x6c,0xc3,0x36,0xe2,0x25,0x59,0x0b,0x6c,0xf2,0x2f,0xb8,0x2b,0x85,0x94,0x41,0x73,
	0x83,0x99,0x02,0x68,0x2c,0x48,0xfb,0xac,0xe7,0x94,0xd9,0x30,0xe8,0xc0,0x32,0x69,
	0x4a,0x47,0x94,0xe8,0x09,0xd9,0xdc,0x46,0xec,0xc6,0x7c,0xba,0xb3,0xf1,0x9e,0x2f,
	0x19,0x59,0x55,0xb2,0x67,0x2c,0xff,0xdc,0xfa,0x2a,0xd4,0x75,0x7c,0xa1,0x09,0x45,
	0x0a,0x34,0x5a,0xe6,0x90,0x92,0xcc,0x14,0x1e,0xad,0x54,0x8d,0xa2,0x6f,0xd3,0x8c,
	0x6d,0xd8,0x15,0x3b,0xf8,0xf4,0x5c,0x92,0xfc,0xd7,0xfb,0x28,0x4f,0xdc,0xd7,0x45,
	0x21,0x36,0x3e,0xac,0x9d,0x0d,0x3a,0x9f,0xab,0xf9,0x9f,0xf9,0x19,0xe7,0xc8,0x59,
	0x00,0x62,0x61,0xbf,0x37,0x0e,0xc2,0x9e,0x37,0xd3,0xc4,0x60,0x25,0x38,0x92,0x8a,
	0x11,0x9e,0xe0,0x93,0x3c,0x16,0xbe,0xe8,0x0a,0x76,0x1f,0x56,0x36,0xe5,0xf7,0x42,
	0x63,0xa2,0x78,0xde,0x40,0x57,0xf9,0xfb,0xab,0xfd,0xd0,0x99,0x7d,0x1f,0xe4,0x42,
	0xc1,0x40,0x9d,0xe0,0x02,0xc9,0x6c,0xc8,0x91,0xa1,0x59,0xba,0x87,0x36,0xb4,0x81,
	0x21,0xec,0xf5,0xe1,0xa1,0x5a,0xde,0xd9,0x44,0x3f,0xfa,0x19,0x19,0x18,0x4c,0x52,
	0x80,0x74,0xd3,0x35,0x90,0x04,0x27,0x65,0x5e,0xcc,0xbf,0x28,0x7a,0x32,0xae,0x4f,
	0x28,0x8c,0xdc,0x46,0xb8,0xd4,0x3e,0x3d,0x28,0x05,0x29,0x34,0x83,0x58,0x11,0x9f,
	0xc0,0x9e,0xc8,0x7d,0xed,0xca,0x72,0x95,0x92,0xcd,0x98,0xe1,0xc3,0xb1,0x2b,0x82,
	0x7d,0xc6,0x30,0x67,0x58,0x38,0xe1,0x69,0x07,0xc0,0x6c,0xbc,0x4d,0x99,0xc0,0xa1,
	0x07,0x3a,0x11,0x98,0xce,0x01,0x8b,0x54,0xdb,0x63,0x98,0x9a,0x22,0xa4,0xe5,0x7d,
	0x1f,0xf0,0xf9,0xd0,0x12,0x86,0x67,0xa7,0xeb,0x1e,0x29,0x97,0x09,0x28,0xd8,0x7a,
	0x07,0xd4,0x47,0x8a,0x7b,0xce,0x6e,0x52,0x41,0xa2,0xbf,0x28,0x8f,0x47,0xcf,0x36,
	0x7d,0x2e,0xca,0x5d,0x95,0x6c,0x9c,0x72,0x25,0x0d,0xfa,0x96,0xeb,0x84,0x8a,0x04,
	0xb3,0x60,0xbf,0x62,0xee,0x17,0x4d,0xda,0xc3,0x95,0x90,0xd6,0xc6,0xcc,0x3e,0x4f,
	0x91,0xf5,0x19,0x4b,0x6f,0x4d,0x01,0xae,0xbb,0xcd,0x83,0xfa,0x3c,0x68,0x60,0xe3,
	0x2d,0x5e,0x98,0xe9,0x65,0xe0,0xaf,0xf0,0xbe,0xdf,0x3f,0x50,0xb3,0x40,0x2a,0x90,
	0xf6,0xb9,0xc3,0x17,0xda,0x5e,0x0a,0x36,0x20,0xfc,0xd6,0xe9,0xf2,0x06,0xf0,0x13,
	0xd8,0x7a,0x50,0x97,0x73,0x80,0x4d,0x5f,0xe1,0x6d,0x9e,0xbb,0x2a,0xd8,0x33,0xf2,
	0x47,0xae,0x3d,0xce,0xc2,0x1a,0x9b,0x26,0xb8,0xa0,0x65,0x3a,0x49,0xdb,0x38,0xd4,
	0xab,0x87,0x8c,0x75,0x43,0xf0,0xe0,0x5c,0xfb,0xd2,0x99,0x3a,0xa9,0xe3,0xf9,0x10,
	0x70,0xd1,0x27,0xaf,0x41,0xce,0x6b,0xdb,0x13,0xb0,0x61,0x6e,0xcf,0x52,0x34,0x6b,
	0x0e,0x93,0x63,0xd8,0xca,0x51,0x13,0x97,0xc4,0x60,0x4f,0xeb,0xe4,0x6c,0x7a,0x22,
	0xe6,0xa6,0xee,0x93,0x6b,0x5a,0x15,0x25,0x0b,0x18,0x6b,0x83,0x41,0x0e,0xd5,0x5d,
	0xfd,0x3a,0xd8,0xe0,0x3a,0xc6,0xf7,0x58,0x9e,0x7c,0xa8,0xfb,0x8d,0x76,0x44,0x48,
	0xb0,0x27,0x64,0xcd,0x68,0xf9,0xd5,0xc8,0xd0,0xc3,0x2f,0xd7,0x86,0xe7,0xeb,0xcc,
	0xc8,0x3e,0x70,0x0a,0x60,0x10,0x58,0xed,0x67,0x49,0xdf,0xec,0x98,0xd9,0x08,0x1a,
	0xbd,0x6a,0x96,0xa1,0x0b,0x5c,0x64,0xb9,0x89,0x04,0x9d,0x85,0xed,0xca,0x86,0x70,
	0xec,0x02,0x40,0xf4,0xe1,0x47,0xcb,0x10,0x8b,0x22,0xaa,0xe0,0xf6,0xde,0x04,0x95,
	0x67,0x4c,0x1e,0xb9,0x8c,0x26,0x31,0x41,0xca,0x97,0x83,0x68,0xbe,0x43,0x96,0x33,
	0x40,0xeb,0xf2,0xe7,0x52,0x20,0xc0,0x4d,0x88,0xa4,0x92,0xcd,0x22,0x90,0x7a,0x07,
	0x88,0x29,0xce,0x87,0xfb,0xd8,0x0c,0x43,0x3d,0x2a,0xe2,0x22,0x88,0x3e,0x51,0x0c,
	0xe9,0xe4,0xca,0x2f,0x60,0xe4,0x85,0xe0,0x5f,0xc5,0xd5,0x1f,0xa0,0x6e,0x52,0x2d,
	0xd2,0x5d,0xa5,0x2c,0xb4,0x4f,0x6c,0x2d,0x50,0xaf,0x5d,0x79,0x00,0xa4,0x78,0x5e,
	0xbd,0x8f,0x63,0x5f,0x96,0xf1,0x30,0x2d,0xac,0x12,0xa5,0x3d,0xaf,0x0a,0xa7,0xab,
	0x88,0xb6,0x2a,0xf9,0xb0,0x30,0xf5,0xe0,0xf1,0xd6,0x79,0xd9,0x8b,0xc3,0x2f,0xba,
	0xa6,0x27,0xf8,0x95,0xa1,0x85,0x8e,0x43,0xf2,0xe4,0xf3,0xe1,0xe5,0x35,0x22,0xeb,
	0x18,0xf9,0x31,0x0b,0xc5,0xb7,0xb8,0x2f,0xb8,0x15,0xbb,0xe0,0x03,0xeb,0xe8,0x59,
	0x15,0x7b,0x37,0xe4,0xe1,0x6b,0xa4,0x7f,0xac,0xce,0xb7,0x32,0x05,0xb0,0x5b,0x8b,
	0x38,0x66,0x62,0x04,0x95,0x8a,0x4c,0x44,0x13,0x94,0xa7,0x21,0xc9,0x06,0x6f,0x2e,
	0x4b,0x5d,0xa6,0x5a,0x61,0x4f,0x51,0x62,0x1b,0x10,0x86,0x50,0xd9,0x45,0xc9,0xcb,
	0x73,0x2b,0xed,0xc9,0x95,0x39,0x45,0xd6,0x67,0x25,0x5f,0x57,0x43,0x27,0xa9,0xfc,
	0x14,0x2f,0xf3,0x5c,0xf9,0xd2,0x36,0xd2,0x2a,0xf4,0x1d,0xe5,0xd4,0xdf,0x5a,0x2b,
	0x29,0x34,0x1a,0xa7,0x3d,0x5d,0x47,0xf4,0xf7,0x47,0x55,0x3d,0x38,0xe3,0xb5,0x0c,
	0x8e,0xe0,0x6c,0x50,0xb7,0x43,0x02,0xd1,0xcb,0xcf,0x32,0x82,0xee,0x20,0x8d,0x1d,
	0xd2,0xbc,0xb9,0xb5,0xfb,0x95,0xcf,0x0a,0xf4,0x42,0xe4,0x14,0xd5,0x24,0x06,0x16,
	0xb0,0x5d,0xe0,0xb1,0x74,0x8d,0x18,0x7a,0x2d,0xd3,0x22,0x79,0x4f,0x52,0x4f,0x3f,
	0x1d,0x37,0x4e,0x23,0x84,0xf2,0x7d,0x49,0xc7,0x6f,0xcf,0xc5,0x7a,0x15,0xa4,0xeb,
	0xb8,0xe7,0xf1,0x20,0x4e,0x61,0xc9,0x1c,0x80,0xc0,0xe1,0x45,0x50,0xad,0x52,0x30,
	0xc6,0x24,0xd3,0x72,0x5e,0xea,0x97,0xbc,0xb5,0xde,0xe6,0x8a,0x9d,0xd0,0x6d,0x64,
	0xa0,0x19,0x90,0xe9,0x93,0x2b,0x80,0x7d,0x70,0x35,0xa8,0x5f,0x25,0xd5,0xc5,0x7f,
	0x68,0xa7,0x40,0x55,0x87,0x67,0x7e,0xff,0x1c,0x71,0x31,0x77,0xcc,0x43,0xff,0xa3,
	0xbd,0xa8,0xc1,0xbc,0xed,0xb2,0x82,0x1d,0xb1,0x9c,0x5a,0x4a,0x8f,0x62,0xe9,0x4c,
	0x04,0x3b,0xc5,0xb8,0x0a,0xc0,0x9e,0x3d,0xb9,0x1c,0xcc,0x51,0xaa,0x61,0x25,0x4e,
	0x62,0xa9,0x20,0x88,0xcf,0xf0,0xde,0xd1,0x47,0x28,0x71,0xb2,0xcd,0xf5,0x6e,0xb1,
	0x4c,0xdf,0x01,0x3e,0xdc,0x6f,0xb5,0xb9,0x0c,0x4d,0x45,0x1f,0xd0,0x54,0x16,0xb0,
	0x79,0x84,0xf7,0x03,0x73,0x4a,0x6d,0x30,0x8c,0x23,0x58,0x19,0x98,0x42,0x76,0x29,
	0xc1,0xb1,0xb4,0xaa,0xf1,0x3b,0xff,0xf6,0x51,0xa1,0x6d,0xb2,0xc6,0xec,0x8a,0x1d,
	0xc8,0xbc,0xe2,0x2a,0xbd,0x15,0x97,0x97,0x53,0x45,0x71,0x88,0xb4,0xe0,0x07,0x59,
	0x82,0x7b,0xc2,0xb8,0x5a,0x8e,0x5a,0xe8,0x8e,0xf6,0xdb,0x2e,0xf7,0x4f,0xd4,0x16,
	0x86,0x76,0x41,0xc0,0x81,0x98,0x30,0xdb,0x28,0xeb,0x85,0xc4,0x63,0x2b,0x7c,0x49,
	0x30,0xf4,0x60,0xb8,0x41,0x6a,0xe7,0x21,0xc8,0x7b,0x6b,0xc9,0x3c,0x57,0xe9,0xbf,
	0x1a,0x6f,0xad,0x42,0xf7,0x3d,0xa6,0x5c,0x34,0xe1,0xcc,0xb8,0x9f,0xc3,0x51,0x5d,
	0xae,0xe8,0x22,0xfe,0x26,0xc1,0x14,0xa0,0x27,0x56,0x7a,0x20,0x15,0xd0,0x90,0xf1,
	0x14,0xda,0xf6,0x35,0x70,0x67,0xf7,0x96,0xa6,0x40,0x91,0xab,0x8e,0x1c,0xc6,0xaa,
	0x27,0xb9,0xd1,0xb6,0x18,0xc5,0x9b,0x18,0x17,0x4e,0xf5,0x81,0x38,0x5c,0x82,0x4b,
	0x3c,0x17,0x90,0x2a,0x9a,0x27,0x2b,0x6c,0xe5,0x0c,0xd5,0xae,0xde,0x37,0x60,0x9c,
	0xca,0x8f,0x15,0x0d,0xab,0x57,0x47,0x1c,0xd1,0x24,0xb5,0x7d,0xfb,0xe5,0x6c,0xff,
	0xc7,0xa2,0xb8,0xb1,0xca,0xb1,0x55,0xab,0x50,0xfe,0x9d,0x5b,0x24,0xaf,0x56,0xf1,
	0x42,0xa0,0x57,0xf8,0xd6,0xf2,0x78,0xf7,0x81,0x62,0x08,0xbd,0x12,0x99,0xc9,0x2b,
	0x78,0x47,0xc9,0xf6,0x34,0x57,0x15,0x61,0xf1,0x6c,0xff,0x04,0xe8,0x1f,0x18,0xc6,
	0xf7,0x22,0x16,0xb5,0x40,0xd8,0xde,0x00,0x7a,0xa6,0xbc,0x41,0x78,0xfc,0xf2,0xd3,
	0xae,0x1f,0x5e,0x92,0x60,0xcc,0x09,0x07,0x1d,0x79,0xfd,0x7a,0x86,0x3a,0xf6,0x36,
	0xdd,0xfe,0x2c,0x3a,0xb5,0x42,0x9a,0xa7,0x1a,0x5d,0x9b,0x8c,0x96,0x57,0x57,0x2b,
	0xcc,0xae,0xb5,0x9f,0x08,0x35,0x7f,0xaf,0x09,0x14,0xf1,0x5a,0x99,0x7c,0xee,0xdd,
	0x2d,0xba,0x45,0xaf,0x40,0x6c,0x77,0xb1,0xb0,0x1b,0x3c,0x3e,0x8c,0x35,0x45,0xf7,
	0x71,0x13,0x6a,0xce,0x2e,0x78,0xc0,0xcc,0x80,0x30,0xf4,0xb0,0x82,0xb2,0x1e,0x84,
	0x4b,0xcf,0xc8,0xf5,0xf2,0x10,0xdf,0x06,0xb8,0x75,0xfd,0xc0,0xb5,0xe7,0x99,0x18,
	0xa2,0x84,0x90,0x00,0x19,0x6c,0xd6,0x29,0xc4,0x6e,0x37,0x80,0xdc,0xa5,0x38,0xa9,
	0x33,0xaa,0xc7,0x06,0x57,0x47,0x3c,0xf7,0x29,0xa8,0xfb,0x93,0x1e,0x73,0xda,0x95,
	0xd0,0x12,0xc9,0xd7,0x48,0x74,0xbd,0xfd,0xde,0x43,0x52,0x29,0xa7,0x63,0x82,0x91,
	0xc2,0x04,0x2d,0x79,0xc5,0xd9,0xb1,0xa6,0x83,0x87,0xe2,0xa8,0x07,0xfb,0x94,0x9c,
	0xb8,0xb5,0x1f,0xfd,0xff,0xbe,0x44,0xce,0x10,0x4b,0x76,0x69,0xae,0x74,0x22,0xab,
	0xe8,0xa0,0x69,0x6a,0x3e,0xa3,0x0f,0x30,0xe2,0xfb,0x23,0x1d,0xfb,0x7e,0xb7,0x11,
	0x8b,0xe5,0x44,0x3a,0xe9,0x71,0x75,0x4a,0x45,0x08,0xa2,0xe2,0x32,0xa9,0x3b,0x45,
	0xbe,0x92,0xf0,0x01,0xda,0xf4,0x5a,0x76,0xfb,0xfd,0x42,0x53,0x62,0x5d,0x65,0xe9,
	0x67,0x3a,0x4b,0xe4,0x9f,0x34,0xd9,0x58,0xee,0xe3,0x61,0xe3,0x8d,0x3f,0x3c,0xc9,
	0x89,0x11,0x7b,0xf2,0xb0,0x37,0xe5,0x24,0x70,0xa7,0x67,0x97,0x9b,0xce,0xe9,0xee,
	0x09,0xcc,0x54,0x71,0xe4,0x47,0x0f,0xf9,0x7c,0x53,0x39,0xee,0xc1,0x60,0xa3,0xfb,
	0x6b,0x1c,0x7f,0xaa,0x93,0xdf,0x3e,0x7f,0x40,0x78,0xc2,0xd4,0x94,0xb1,0x8e,0x29,
	0x25,0x87,0xcf,0xbd,0xfd,0x11,0x45,0xe2,0x04,0x47,0x5e,0x96,0xd9,0x69,0x7f,0x2e,
	0x1a,0x45,0xa5,0xed,0x56,0xdd,0xe1,0xb6,0x09,0xaf,0x77,0x66,0xbe,0x1a,0x49,0x6f,
	0x30,0xbe,0x46,0x08,0x0a,0x97,0xa7,0x9a,0xb8,0x4c,0x78,0x26,0x52,0x74,0x5d,0xfd,
	0xb3,0x6f,0xec,0x91,0x95,0x28,0xc6,0xfa,0xe5,0xbf,0x92,0xd9,0x10,0x93,0xd5,0xe9,
	0xc4,0x35,0x9b,0x11,0x8c,0xcb,0x4f,0x01,0xfa,0x98,0x7f,0x3b,0x3d,0x5c,0x28,0x78,
	0xe4,0xca,0xfb,0x0f,0x22,0xd3,0x90,0x14,0x5d,0xb9,0xb2,0x53,0xea,0x73,0x75,0xc7,
	0xd0,0x3b,0x16,0xe5,0xe1,0x12,0x4c,0x95,0x57,0x2f,0x1f,0xd9,0x01,0x5f,0xd6,0x0c,
	0x1d,0x08,0x0e,0xcc,0xbb,0x0d,0x50,0xea,0xb3,0x38,0x01,0x6e,0xd8,0x86,0xfc,0x25,
	0x26,0xb7,0x3c,0x52,0x9c,0x19,0xaf,0x11,0x69,0x7d,0x5e,0x54,0xba,0x0f,0xaa,0x62,
	0x02,0x40,0x55,0x1f,0x61,0xd0,0xc1,0x4b,0x28,0xd9,0x1c,0x17,0x3b,0xeb,0x07,0xf0,
	0x26,0xff,0xcc,0x62,0x38,0xc0,0xba,0xcb,0x24,0x06,0x42,0x0c,0xcd,0x68,0xb4,0x63,
	0x16,0x1c,0x91,0xbd,0x70,0x62,0x31,0x77,0x0f,0xc8,0x7e,0xcd,0xde,0x4e,0x8f,0x7e,
	0xd0,0x43,0xb4,0x92,0x91,0x80,0x39,0xd2,0xe7,0x92,0x65,0x10,0x8d,0xd5,0x89,0x28,
	0xa4,0xac,0x8d,0x6e,0x14,0x78,0x0a,0x1f,0x9d,0xe6,0x7f,0x60,0xf1,0x31,0x59,0x15,
	0x6e,0x4f,0xa0,0x27,0xb3,0xf1,0xf4,0xe9,0x30,0xf1,0x0f,0xc9,0x63,0xc4,0xda,0x56,
	0x8f,0x40,0x1d,0x2a,0xb0,0x60,0x14,0x52,0xdd,0xc2,0x53,0xeb,0x03,0x70,0xf9,0x46,
	0x54,0x56,0x0d,0x17,0x55,0xcf,0x9d,0x3d,0x8c,0x00,0x68,0x94,0xda,0x97,0x0f,0xe0,
	0xf0,0xbe,0x50,0xe9,0xc4,0x81,0x69,0xa5,0x5b,0xf7,0xb7,0xb4,0xce,0x3e,0x79,0xb9,
	0x1a,0x0c,0xdf,0xa7,0x83,0x9c,0x06,0x62,0x6f,0x20,0x86,0xea,0x40,0x74,0xbc,0xc6,
	0x37,0xb2,0xc1,0x76,0x5b,0xac,0x61,0x1b,0xbb,0x0f,0xee,0x4e,0x20,0x28,0x83,0x0a,
	0xae,0x93,0xf7,0x00,0x12,0xb7,0x09,0x78,0x1b,0x07,0xc0,0xd1,0x4e,0x58,0x68,0xea,
	0x8c,0x8c,0x29,0xe4,0x8f,0xba,0xdd,0x60,0x31,0x7a,0x5b,0x57,0x5c,0x48,0x55,0x3f,
	0xe3,0x89,0x23,0x31,0xb4,0x93,0x73,0x3a,0x61,0x67,0x94,0xb4,0x5b,0x87,0xed,0x59,
	0x96,0x68,0xc7,0x59,0xa5,0xd7,0xdd,0x20,0xa6,0x77,0xf3,0x6d,0xbc,0x2a,0x4d,0xf3,
	0x1a,0xaa,0x4c,0x46,0x77,0x4e,0xce,0x0d,0x7f,0x3d,0xfb,0x64,0x13,0x7e,0x7c,0x1e,
	0x7e,0x25,0xe2,0x78,0x8f,0x29,0x98,0xf0,0x2c,0xa2,0xdd,0xb7,0x64,0x04,0x60,0xc8,
	0x7b,0x96,0x38,0x3e,0x19,0x11,0xc8,0x95,0x26,0x93,0x76,0x6d,0xc0,0xcc,0x85,0x9e,
	0x95,0x02,0x81,0xab,0x4e,0x56,0xb8,0xf8,0x68,0xc6,0x85,0x0d,0x9d,0x84,0xc3,0xe2,
	0xa5,0x52,0x1e,0x3b,0x58,0xb4,0x9f,0x0f,0x15,0xb8,0x1f,0x38,0x19,0xe2,0x86,0xf8,
	0x79,0x65,0x4d,0x34,0xfa,0x89,0xa4,0x80,0x67,0xcb,0x21,0x24,0x52,0xf7,0x45,0x04,
	0x3f,0xac,0x38,0x62,0xab,0x8f,0xfd,0x8d,0x55,0xf2,0xf7,0x9f,0xb6,0xd2,0xc5,0xf6,
	0xda,0xf6,0x20,0x9c,0x2b,0x32,0x4e,0x4b,0xa2,0xb2,0xf3,0x88,0xec,0xb9,0xcd,0x8c,
	0xce,0xfb,0xdd,0xf8,0x4e,0xb7,0x30,0x6f,0x8a,0x0b,0xe0,0xb9,0x3a,0x08,0x70,0x23,
	0x26,0xdd,0xe6,0x27,0xdc,0xb2,0x50,0x3b,0x49,0xaf,0x66,0x5c,0x66,0x9c,0x6a,0x4a,
	0xa6,0xfc,0xd4,0xae,0xdc,0xc4,0xbf,0x7c,0x6b,0x91,0xf6,0xb4,0x62,0xbe,0xfd,0x0e,
	0xe6,0xbf,0x34,0x02,0x31,0xc1,0x63,0x3e,0x84,0xda,0x98,0xaf,0x76,0x1c,0x50,0x11,
	0x8c,0x2f,0x77,0x29,0x78,0xa5,0x68,0xa7,0x2e,0x17,0x63,0xd0,0x9f,0xaf,0x70,0x4c,
	0x9b,0x85,0xed,0x7a,0x10,0x12,0x7b,0xf4,0xca,0xcb,0x17,0x99,0xc4,0xca,0xd8,0xcb,
	0xdc,0x10,0xbd,0xec,0x71,0x86,0x14,0x07,0x23,0x8a,0xf3,0x3c,0x57,0x4a,0x7d,0xe8,
	0xd9,0xd2,0xf6,0x4a,0xa0,0xd2,0xba,0x56,0x3c,0x9e,0x64,0x82,0x66,0x70,0x43,0xfa,
	0xab,0x15,0x5c,0xee,0xd2,0x5e,0xb7,0x0e,0x26,0x8b,0xab,0xf2,0x37,0x08,0x55,0xe5,
	0x2c,0xbb,0x75,0xbd,0x7f,0xd8,0x5d,0x03,0xe2,0x02,0xe1,0xfd,0x6f,0x81,0x7a,0x03,
	0x21,0x5f,0xc4,0x3d,0xb8,0x95,0x35,0xbe,0x8e,0xbb,0x2d,0x08,0x9c,0x2d,0xff,0x2c,
	0x1f,0x56,0xfb,0x7b,0xc3,0xc6,0xdb,0x9e,0xa9,0x30,0xfa,0x46,0x1b,0xe1,0x77,0xb5,
	0x16,0xb7,0x8e,0x39,0x8b,0x7b,0xc1,0x64,0xe3,0x3b,0xdb,0xb0,0x4a,0xa4,0xa7,0x1e,
	0x7c,0xff,0x89,0x3a,0x54,0xea,0x94,0x5d,0x61,0x62,0x09,0xe8,0xb0,0x26,0xb7,0x6e,
	0x30,0x32,0x4e,0xba,0xb5,0xb0,0x65,0x94,0xfd,0x14,0x37,0xab,0xab,0xb7,0xc3,0xd2,
	0x7f,0x2f,0x45,0xca,0x1d,0xfb,0xa5,0x23,0xad,0x47,0x3a,0x53,0x72,0x3f,0x0c,0x2a,
	0x8a,0x8c,0x07,0x2c,0xe4,0xea,0x7b,0xd9,0x1e,0x70,0x1d,0x92,0xec,0xbe,0x71,0x51,
	0x76,0x42,0x73,0x7f,0x98,0x91,0xd4,0x9f,0xdf,0x25,0x2d,0x02,0x17,0x37,0x45,0x85,
	0x8b,0x2d,0xe4,0x96,0xde,0xf9,0x3b,0xb9,0x65,0xae,0x0f,0xb7,0x32,0xc7,0x5e,0x69,
	0xc3,0x90,0xa4,0xe6,0xbe,0x86,0xa7,0x22,0x41,0x5e,0xb0,0x62,0x93,0xfe,0x7f,0x4f,
	0x2b,0x43,0x0a,0xf8,0x6b,0x40,0x2d,0x77,0xed,0x68,0x6c,0x92,0x64,0x4b,0x33,0x5f,
	0xe6,0x42,0x5e,0x9b,0xc5,0xa1,0x25,0xa7,0xfc,0x3a,0x10,0xa5,0xca,0x9d,0x51,0x7c,
	0x4b,0xc2,0xb6,0x84,0xb5,0xe5,0x65,0x40,0x07,0x95,0xc2,0x57,0x64,0x9c,0x1b,0x45,
	0xca,0x00,0xc9,0x34,0x7b,0x89,0xc7,0x3f,0xc4,0x42,0xe8,0x24,0x98,0xb1,0xc2,0xa5,
	0x2c,0x14,0x24,0xfb,0xaf,0x01,0xc4,0xfb,0xcf,0x5b,0x77,0xc4,0x39,0x74,0xca,0x12,
	0xae,0xd8,0x26,0xbf,0xed,0x59,0x73,0x5c,0x0e,0xb8,0x3e,0x73,0x1c,0x6e,0xd9,0xc0,
	0x2a,0x55,0xca,0x46,0x0c,0xda,0x05,0x6a,0x0a,0x07,0x9b,0xc7,0xee,0xb9,0x50,0xcd,
	0xaa,0x7a,0x6b,0x07,0x01,0xe8,0xe1,0x70,0x49,0x20,0xf8,0xd4,0xf6,0xb2,0x21,0xaa,
	0x36,0xb2,0x58,0xde,0xa3,0x78,0x22,0x85,0x57,0xbf,0xb8,0x94,0x7e,0xbe,0x48,0x5a,
	0xbe,0x4f,0x0d,0x3d,0x1f,0x23,0xf5,0x13,0xa0,0x3b,0x52,0x45,0xf2,0xa7,0xed,0xbe,
	0xc7,0x7e,0xb2,0xc3,0x09,0x5d,0x02,0xb3,0x2f,0x59,0x9c,0xf0,0x57,0xa1,0xe5,0xf0,
	0x3b,0x3a,0x03,0xdb,0x48,0x93,0xd7,0x2c,0xec,0x2e,0x05,0xa3,0x47,0x48,0x53,0x01,
	0x87,0x28,0x6a,0xf1,0xcc,0x2e,0xf3,0x15,0xa5,0x35,0x05,0x82,0x90,0x4d,0x6d,0x48,
	0x4f,0x58,0x32,0x63,0xc3,0xd3,0xe6,0xab,0xcc,0x48,0x2a,0xc6,0x2d,0xce,0x30,0xee,
	0x5a,0xb5,0x70,0x86,0x4f,0x3f,0x6d,0xb9,0x10,0xb3,0x7e,0xb2,0xf3,0xc2,0xe6,0x04,
};


/* 317-0167 FD1089A */
static const UINT8 key_0167[0x2000] =
{
	/* data */
	0x98,0xb0,0xcf,0x36,0xd7,0xa2,0xc0,0x94,0xa5,0x41,0x82,0x6b,0xc8,0x42,0xfa,0xc8,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x68,0xcd,0x56,0xcd,0xb5,0x15,0xc6,0x7a,0x1c,0x7a,0x21,0x44,0x8d,0x23,0xdb,0x16,
	0xc2,0x0a,0xb6,0x03,0x36,0xd0,0x45,0x8d,0x97,0x55,0xb4,0x56,0xaa,0x70,0x84,0xf4,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


/* 317-0168 FD1089B */
static const UINT8 key_0168[0x2000] =
{
	/* data */
	0x98,0xb0,0xcf,0x36,0xd7,0xa2,0xc0,0x94,0xa5,0x43,0x82,0x6b,0xc8,0x40,0xfa,0xc8,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x68,0xcd,0x56,0xcd,0xb5,0x15,0xc6,0x7a,0x1c,0x7a,0x23,0x44,0x8d,0x21,0xdb,0x16,
	0xc2,0x0a,0xb6,0x03,0x36,0xd0,0x45,0x8d,0x97,0x55,0xb4,0x56,0xaa,0x70,0x84,0xf4,
	0x7f,0xe1,0x07,0x16,0x94,0x11,0x9e,0xd0,0x8f,0xe6,0xd8,0xd6,0xbb,0xa2,0x6b,0x59,
	0x80,0xf1,0x66,0xab,0xba,0x2b,0x50,0x9f,0x65,0xec,0xba,0xc0,0x8c,0x7b,0xb0,0x57,
	0x8c,0xb8,0xbf,0xbe,0x88,0xfa,0x27,0x75,0xca,0x48,0x5e,0x40,0x6c,0xe8,0x82,0xba,
	0x44,0xe9,0x19,0x1a,0xbc,0x94,0x9a,0xea,0xf6,0xc1,0xb6,0x01,0x4c,0x8c,0xea,0x74,
	0x91,0xc2,0x78,0xde,0x8f,0xe5,0xb4,0x56,0x97,0x4a,0x42,0x97,0x24,0x90,0x79,0x7f,
	0xba,0x34,0xa2,0x1c,0xcc,0x34,0xc3,0xae,0x39,0xf8,0x17,0xe6,0x25,0xd2,0xc3,0x61,
	0x22,0xda,0xc5,0x2f,0x19,0x0f,0x28,0x22,0x93,0xe7,0x5a,0x9e,0x72,0x8c,0xda,0x9c,
	0x0f,0x90,0x65,0x0a,0xab,0x19,0xa7,0x3a,0xcd,0x82,0x2e,0x09,0xb4,0x98,0x39,0x4d,
	0x57,0xd6,0x7d,0x3f,0x30,0x66,0x16,0xce,0xa0,0xc9,0xca,0x59,0x55,0x1d,0xa2,0x03,
	0x3a,0x6e,0x3b,0x2d,0xa2,0xe0,0x6c,0x12,0x98,0x97,0x12,0x39,0x23,0xa0,0x8a,0x38,
	0x89,0x7f,0xaa,0x4e,0x07,0x1c,0x32,0x1b,0xee,0x83,0xa7,0x2e,0x73,0xc2,0xbf,0xb0,
	0x06,0x22,0x97,0xaa,0xc5,0xda,0xc5,0xc7,0x03,0x59,0x82,0x89,0x1e,0xad,0xd2,0x9b,
	0x7c,0x0a,0x77,0xc5,0xa1,0x8b,0x38,0xdf,0xfc,0x7b,0x6e,0x58,0xee,0x99,0x0d,0x25,
	0xa3,0x6d,0x56,0x26,0xb5,0x65,0xa0,0xa4,0x16,0xd3,0x0b,0x3f,0xdf,0xbc,0xd9,0xa8,
	0x5f,0x86,0x88,0x52,0x5b,0xa3,0xe6,0xba,0x99,0xae,0xcf,0x69,0xe9,0xb7,0x64,0x06,
	0x50,0x47,0xb8,0x01,0xb8,0xff,0x6b,0x90,0x72,0x4a,0x16,0xd9,0x39,0x52,0x5b,0xba,
	0x5b,0x69,0x03,0x43,0x5a,0x98,0x32,0x18,0xc3,0x55,0x73,0xaf,0x49,0xbd,0x78,0xc5,
	0x9f,0x05,0x04,0x17,0x3f,0x98,0xd3,0x73,0xfb,0xb6,0xa8,0xd3,0xe7,0x3f,0x24,0x9f,
	0xb1,0xb2,0x92,0x54,0x3d,0xf7,0xc9,0x90,0x60,0x0a,0x7a,0x65,0x20,0xbc,0xcc,0x6f,
	0x2a,0x8b,0x3b,0x11,0x5f,0x70,0x68,0xdf,0x1e,0x23,0x1e,0x75,0xd6,0xf7,0xde,0x10,
	0xe2,0x8b,0x3d,0xf4,0xaa,0xbb,0xf7,0x93,0x23,0xb0,0xc6,0x44,0x61,0x44,0xa5,0x8a,
	0x4e,0xb0,0x1b,0xa0,0xe6,0x5c,0x8d,0x3b,0x9c,0x7a,0x03,0x54,0xc1,0xed,0xfc,0x45,
	0xd0,0x19,0x76,0xa2,0xb3,0xdb,0xbd,0x46,0xe7,0x2e,0x67,0xc2,0xa0,0xde,0x4e,0xb9,
	0xa3,0xb9,0x47,0xfe,0x80,0x14,0xa4,0xbf,0xe1,0x91,0x4a,0x23,0xeb,0x14,0x73,0x20,
	0x08,0x55,0xb6,0x0c,0x32,0xf4,0x7f,0x05,0xce,0x0d,0xb8,0x15,0xeb,0xf1,0x6d,0xca,
	0xd7,0x95,0xee,0x98,0x27,0xaf,0x88,0x8c,0xc5,0x2f,0x1a,0x13,0xab,0x25,0x68,0xc2,
	0xbb,0x7a,0xff,0xec,0xb3,0x0f,0x37,0x76,0x0d,0xc6,0x34,0x92,0xca,0x9d,0xe4,0x3e,
	0x84,0xf6,0x99,0x8f,0x7c,0xe9,0x0b,0x28,0xf1,0x42,0x26,0x0d,0x6a,0x8f,0x2b,0x77,
	0x89,0xa6,0xc2,0xec,0xa8,0xc0,0x4d,0x25,0x89,0xaf,0x54,0x4b,0xdd,0x85,0xe9,0x67,
	0x97,0xb8,0xb4,0x98,0xdd,0x81,0x44,0x07,0x7e,0x41,0xfe,0x2d,0x2b,0x30,0x42,0x5c,
	0x63,0xae,0x32,0x0c,0x3f,0x4a,0x09,0xe9,0x03,0x52,0xec,0x1b,0xe5,0xf5,0x4b,0xe5,
	0xaf,0x35,0x9a,0xfe,0x9b,0x4b,0x3e,0xf4,0xbc,0x9f,0x69,0x7d,0x83,0xea,0xe8,0xd9,
	0xc6,0xed,0xed,0xab,0x58,0x3c,0x90,0xea,0x00,0xab,0x41,0xbc,0x33,0x6f,0xa4,0xa0,
	0x4d,0x3b,0x4a,0x62,0x04,0x04,0x6c,0xc6,0x29,0x02,0x84,0xea,0xf4,0xe3,0xfa,0x9b,
	0xa4,0xa6,0x26,0xf6,0x90,0x13,0x2b,0xde,0x95,0xd7,0x26,0xcd,0x32,0x01,0xa8,0x33,
	0xf8,0x21,0x77,0xc4,0xf5,0xa9,0xa4,0x73,0xed,0x7c,0x6b,0x9d,0xe4,0x32,0x62,0x5e,
	0x3d,0x18,0x31,0x56,0x70,0x19,0x79,0x9c,0xfd,0xc2,0x6f,0xfb,0x1f,0x5d,0x67,0xd0,
	0x82,0x9e,0x9f,0x02,0x84,0xe9,0x09,0xf0,0x07,0x00,0xfc,0x4c,0xbf,0xa8,0x98,0x4a,
	0x09,0xf5,0x1f,0x41,0x06,0xd6,0x67,0xdf,0x0e,0x6b,0xc2,0xf1,0x3f,0x53,0x12,0xc3,
	0xd0,0xcc,0x4f,0xf5,0x14,0x87,0x3a,0xa3,0x03,0x5f,0x31,0x91,0x9e,0xd0,0x47,0x14,
	0xcb,0x0c,0x34,0x50,0x3a,0xaa,0x1a,0xf6,0x9b,0x81,0x83,0x07,0x6f,0xe5,0x7e,0x8f,
	0x34,0x90,0x74,0x33,0x1f,0x5a,0xd3,0xb9,0x61,0x5a,0xce,0x01,0xb8,0xfa,0xd5,0x21,
	0xc0,0x21,0x64,0xc7,0x0d,0xd2,0x6f,0x03,0x30,0xbf,0x1b,0x71,0xd8,0x39,0x36,0xcd,
	0xda,0x89,0x1c,0x84,0x8f,0xe4,0x99,0x7d,0x88,0xe7,0x58,0x71,0xd1,0x83,0xb8,0x15,
	0x03,0x92,0xda,0x9e,0x7b,0xe5,0x77,0xc6,0x22,0xd3,0xe0,0x01,0xa1,0x96,0x03,0x59,
	0x98,0x53,0x0d,0x3a,0xfe,0x2f,0xcd,0x8e,0x4d,0xdf,0xf4,0x07,0x60,0x02,0x61,0x51,
	0x20,0xab,0x86,0x28,0x92,0x99,0xea,0x62,0xd0,0x11,0x97,0x91,0x96,0x6b,0x3f,0x4a,
	0x1d,0x0b,0x40,0xfc,0xe1,0x41,0xc5,0x2d,0xa9,0x85,0x0a,0xf1,0xa0,0xc9,0xd4,0x93,
	0x73,0xc0,0x8e,0x38,0x93,0xf1,0x6f,0x01,0x9e,0xdf,0x08,0x47,0xb1,0x75,0x21,0x8e,
	0x81,0x6f,0x29,0x0b,0x46,0x66,0x88,0x59,0x85,0x6c,0x47,0x1f,0xcd,0x56,0x8f,0xe6,
	0x2c,0x3e,0xa8,0xc9,0x35,0x2d,0x71,0x4a,0x5b,0x63,0x03,0x5f,0x1b,0x62,0x04,0xe0,
	0xa2,0xf4,0x74,0xcd,0x30,0xca,0x31,0x86,0x46,0xbf,0x01,0xce,0xbb,0xab,0x57,0x3b,
	0x4e,0xb5,0xc4,0x7f,0x72,0x82,0xda,0xa8,0x48,0x0d,0x5c,0x0f,0xca,0x9f,0x69,0xb2,
	0x51,0x9b,0xa4,0x6b,0x2b,0x75,0xc3,0x84,0xb5,0xb5,0xe9,0x7f,0x7d,0x34,0xb4,0x8e,
	0x5d,0xf1,0x3d,0x09,0x68,0x22,0x72,0x8b,0xdd,0x96,0xea,0xae,0x2c,0x26,0xa7,0x11,
	0x98,0x46,0x1d,0xf7,0xda,0x4e,0x0f,0x46,0xfa,0x4d,0x94,0x18,0xd5,0xaf,0x49,0x6c,
	0xb1,0x6e,0x3a,0x72,0x38,0x38,0x80,0x09,0xb0,0x45,0x11,0x38,0x54,0x4b,0x34,0x14,
	0x7c,0x08,0xd1,0x91,0x5c,0x07,0x67,0xd3,0xfe,0xd7,0xd2,0x82,0x4e,0x00,0xbf,0x12,
	0xd5,0x46,0x64,0x68,0xd6,0xbb,0xbd,0xb4,0x4a,0x0e,0xe2,0x43,0x4f,0x16,0x58,0x2a,
	0x10,0x86,0x42,0x91,0x45,0x91,0x5d,0x1f,0xdd,0x52,0x44,0x06,0x85,0xbc,0xc0,0x3a,
	0x0e,0x64,0xb1,0xbb,0x02,0x0e,0xe0,0x17,0x3e,0xc4,0x03,0x58,0x16,0xcc,0x5d,0xe7,
	0x98,0x91,0x53,0xac,0x69,0x6b,0x31,0x8c,0xe0,0x02,0x72,0x8b,0xc9,0x92,0xd8,0x20,
	0xe9,0x72,0xc1,0xf3,0xc6,0xf1,0x28,0x6f,0x73,0x22,0x8f,0x1e,0x85,0x7f,0xb0,0x4e,
	0xf4,0xf7,0xc3,0x63,0x00,0x6f,0x75,0x90,0x67,0xc3,0x0e,0x65,0xa0,0xc0,0x5a,0x43,
	0x55,0x02,0x41,0xb5,0x0a,0x14,0x3d,0xe2,0xdd,0x85,0x1e,0x21,0x3e,0x1b,0xf3,0x63,
	0xf5,0xb9,0xae,0x5f,0xd3,0xbc,0x26,0xe8,0x5e,0x92,0xdf,0x6e,0x34,0x15,0xfe,0x3a,
	0xfc,0x7d,0x4d,0x3e,0x74,0xc7,0xff,0x8a,0x9d,0xa4,0x09,0x1d,0xb7,0x49,0xf7,0x5c,
	0x5c,0xce,0xb9,0xc7,0x0f,0x59,0x53,0xa2,0xb5,0xb5,0x18,0x89,0xee,0x46,0x14,0xf0,
	0xfd,0xcb,0xba,0xc0,0x96,0xa5,0x75,0x49,0x16,0x25,0x97,0x93,0xbd,0x4d,0x4f,0x00,
	0xe0,0x08,0x18,0x91,0x86,0x3c,0x1d,0xea,0xe3,0x64,0x47,0xe1,0xc6,0x83,0x61,0x59,
	0x91,0x3a,0xea,0xab,0x5c,0xc6,0xb6,0xe3,0x59,0xda,0xe1,0x89,0xff,0x1f,0xdc,0x62,
	0x07,0xff,0xbf,0xbb,0xe5,0xfe,0x88,0x50,0xdc,0xbb,0xe6,0xaa,0x0d,0x0e,0xe5,0xf3,
	0xcf,0x2a,0x24,0x92,0xc2,0xfa,0xd3,0xbd,0xaf,0x2a,0xc6,0x88,0xc0,0xd6,0x78,0xcc,
	0x59,0x39,0x66,0x73,0x88,0x40,0xbd,0x92,0xd1,0x9d,0x46,0x22,0x89,0xbb,0xec,0xda,
	0x2f,0x9c,0xe3,0x29,0x61,0x7b,0x01,0x45,0x4a,0xe7,0xd1,0xdd,0x54,0xc2,0xf7,0xd4,
	0x4e,0x10,0xb8,0x60,0xf2,0x80,0x0b,0x56,0x2d,0x6d,0xc8,0x01,0x19,0x79,0x4e,0x02,
	0xd0,0xb3,0x19,0xac,0xfd,0x98,0x9f,0xba,0xdd,0x2a,0x4a,0xf8,0xa0,0x90,0x0f,0x22,
	0x32,0x60,0xe7,0x44,0xd0,0x95,0x37,0x0f,0x8a,0x4a,0xbf,0x0a,0xed,0x0b,0xc2,0xbd,
	0x81,0x4f,0xf7,0x33,0x29,0xbc,0x8b,0x2d,0x0a,0xbb,0xb4,0x30,0x23,0xe3,0xc3,0xdb,
	0xf4,0x7d,0x6d,0x99,0x72,0xc1,0x49,0xe2,0x5c,0x8b,0x91,0x5e,0x49,0xe8,0x1a,0x61,
	0x91,0x48,0x1a,0x61,0xed,0xf7,0x90,0xb7,0xfa,0x3c,0x3f,0xdc,0x66,0x1c,0x91,0x47,
	0x03,0xcb,0x1a,0x67,0x26,0x4e,0xb3,0x8b,0x80,0x22,0x3a,0xe0,0xfd,0x44,0xc6,0xa6,
	0x87,0xc8,0x64,0xaa,0x72,0x25,0xce,0x7f,0xf6,0x52,0xbf,0x08,0x5c,0x68,0x28,0x59,
	0xc5,0x9a,0xf5,0xec,0xfd,0x9c,0xb0,0xeb,0xe7,0x93,0xf3,0x1b,0x75,0x30,0x28,0xf9,
	0xe0,0xb8,0xee,0x23,0xb0,0xe1,0x45,0x33,0x37,0xe2,0x80,0x77,0xcf,0x8a,0xc5,0x89,
	0xb2,0x83,0x0e,0x09,0xff,0x7e,0x79,0x06,0xc0,0x54,0x8e,0x88,0xf1,0x75,0x5a,0xc4,
	0x60,0x76,0x68,0x5f,0xb3,0xa3,0x3d,0x72,0x46,0xf2,0xdb,0x6b,0x9d,0x90,0xcd,0x32,
	0xfd,0x95,0x36,0x91,0x2c,0x54,0xe4,0x4f,0x40,0x6e,0x50,0xd7,0x45,0x2c,0x37,0xf4,
	0x18,0x27,0x66,0x4b,0x81,0x8a,0x38,0x9c,0x60,0x6e,0x92,0xdf,0xb8,0xb2,0xc1,0x6b,
	0x5a,0xd2,0x8c,0xda,0x68,0x81,0x2a,0x2f,0x86,0xf9,0x07,0xad,0xa7,0x48,0x19,0x7b,
	0xf9,0x76,0xab,0x95,0x35,0x8e,0xc2,0x66,0x36,0x5f,0x9e,0x41,0xba,0x50,0x55,0xad,
	0x4c,0x28,0xbd,0x3e,0x1a,0x84,0xf7,0xc3,0x79,0xe9,0xbc,0x2f,0x95,0xec,0xe3,0x40,
	0xf9,0x6a,0x82,0xce,0x17,0xad,0x7d,0x81,0xaf,0x98,0x57,0x4b,0x85,0xe4,0x3a,0xf7,
	0x5a,0x38,0xd4,0xc6,0x42,0x0c,0x42,0x0c,0x58,0x59,0xc5,0x80,0x1e,0x56,0x2c,0x90,
	0x18,0xcb,0x18,0xca,0x4d,0x80,0x00,0x08,0x8f,0x21,0xe7,0x48,0xd0,0x7b,0x17,0x3e,
	0xfd,0x8e,0xde,0x4c,0x87,0x39,0xec,0xac,0x0f,0xc1,0x1e,0xed,0xbd,0x23,0x7f,0x69,
	0xb9,0x8b,0x98,0xd2,0x59,0x82,0xa8,0x44,0xe8,0x57,0xca,0x31,0x53,0xff,0xee,0xc8,
	0xb0,0xf6,0xa0,0x88,0xf3,0x0d,0x11,0x41,0xb8,0x60,0x16,0x9f,0x19,0x70,0xe3,0x44,
	0xe3,0x85,0x67,0x48,0xeb,0x20,0xb4,0xc3,0x9e,0x80,0xe5,0x7a,0xb0,0x1a,0xf5,0x02,
	0xc7,0x17,0x8a,0x91,0x26,0x79,0x7f,0x68,0xc9,0x36,0xc6,0xc1,0x1d,0xae,0x9d,0x92,
	0x44,0x63,0xbb,0xfb,0x78,0x58,0xb6,0xbb,0xa9,0xad,0x5e,0x23,0x73,0xe7,0xca,0x3d,
	0x01,0xb1,0xbd,0xca,0x43,0x44,0xc4,0x19,0xd8,0xed,0x6c,0xac,0x08,0xda,0x47,0xbc,
	0x53,0xe2,0x95,0x83,0xfd,0x8c,0x8f,0x20,0x73,0x8c,0x86,0xf0,0x61,0x75,0x4f,0x7c,
	0xf6,0xe7,0x25,0x68,0x77,0x80,0xfa,0x35,0xd4,0x1a,0x04,0xb2,0x26,0x6e,0x58,0xf1,
	0x42,0x2e,0xff,0xa9,0x44,0x7f,0xf5,0xf0,0xa3,0x4b,0x87,0x2d,0xec,0x76,0x62,0x11,
	0x30,0xec,0xce,0x67,0xdd,0xc2,0xee,0x3a,0xb5,0xeb,0x99,0x10,0x84,0x65,0xf5,0x01,
	0xd3,0xe9,0xa7,0x5a,0x96,0x15,0x6a,0x2e,0x02,0x8d,0xcd,0xd3,0x6f,0xaf,0x3d,0xae,
	0x47,0xba,0xd7,0xcc,0x3e,0xac,0xe3,0x9b,0x02,0x45,0x51,0x6d,0x31,0x27,0xa0,0x5f,
	0x5c,0x68,0xc0,0x93,0xf3,0x06,0x37,0xc4,0xb5,0x05,0x95,0x90,0x1b,0xfb,0x77,0x9e,
	0xdf,0x1c,0x7b,0x69,0xac,0xbe,0xad,0x10,0xa3,0x14,0x40,0x64,0xf4,0xb3,0x18,0x33,
	0x2b,0x56,0x2d,0xaf,0x30,0xe4,0xd5,0x94,0xd4,0x33,0x7e,0xdb,0x08,0xd9,0x8f,0xa2,
	0xe9,0xf4,0x13,0x60,0x08,0xdb,0xf0,0x0e,0x73,0xf2,0xda,0x07,0x01,0x47,0x14,0x2b,
	0x3b,0x57,0x35,0x5a,0x54,0x2b,0xad,0x0d,0xd3,0xe3,0xd7,0x25,0xea,0x04,0xea,0x1d,
	0xfe,0x54,0x92,0xda,0x7b,0xb0,0x9e,0x53,0xab,0x79,0x55,0xab,0xd2,0x6f,0x1c,0x2c,
	0xca,0xd7,0xa8,0x85,0xb9,0xae,0x41,0x06,0xcb,0x68,0x20,0xe7,0x35,0x97,0x97,0x11,
	0xd9,0x24,0x48,0x63,0x64,0xb5,0x6a,0x05,0x9c,0x1e,0x29,0xb5,0xbe,0x8a,0xd0,0x41,
	0xac,0xa7,0x07,0xbe,0x7c,0x6d,0x38,0x5f,0xba,0x5f,0x4c,0xca,0x96,0x15,0x69,0x00,
	0x6e,0x3f,0x56,0x86,0xa6,0x31,0x97,0x1a,0xea,0xfe,0x31,0xd6,0xea,0x08,0x63,0xf9,
	0x0a,0xbc,0x96,0xf0,0x69,0x09,0xf3,0xce,0x0d,0x9e,0x13,0xa4,0xa4,0xfb,0xa2,0x7f,
	0xe2,0xce,0xeb,0xad,0xb3,0x8f,0x1e,0x50,0x8d,0x91,0xcb,0x8b,0x8e,0x44,0xcc,0x99,
	0xa1,0xda,0xaa,0xe2,0x5a,0x24,0x87,0x24,0x5a,0x31,0x5f,0xd0,0x26,0x79,0x8d,0x54,
	0x94,0x4e,0xb2,0x83,0x16,0x4a,0xb4,0x30,0xad,0xc7,0x7d,0x92,0x8b,0xc1,0xf1,0xa4,
	0xb6,0xbc,0x95,0x59,0x2f,0xc6,0x49,0x1c,0x7b,0xb0,0x70,0x8d,0x33,0x1d,0xfe,0x82,
	0x95,0x90,0xc4,0x00,0x65,0xb9,0x0c,0x28,0x21,0xd6,0x02,0xa1,0x4f,0x65,0x9c,0xb6,
	0xa5,0xee,0xc1,0x8a,0x94,0x13,0xad,0x47,0x71,0x8d,0x85,0xbf,0x08,0xa5,0xca,0x94,
	0x80,0x1e,0x4d,0xdf,0x14,0x1f,0xc4,0xb0,0xe1,0x79,0x1a,0x07,0x7c,0xf4,0xaa,0xa1,
	0xd5,0xc1,0x62,0x82,0xea,0xac,0xb7,0xe6,0x0c,0x8a,0xa1,0x4d,0x43,0xcb,0x70,0xe2,
	0xd6,0x17,0x35,0xda,0x07,0x02,0x72,0x90,0x7d,0xa6,0xc3,0x63,0x1e,0x21,0xd6,0x01,
	0x55,0xf1,0x3f,0x41,0xe8,0xd1,0x28,0xc1,0x6d,0xd4,0xcc,0x7e,0x11,0x26,0xb1,0x6c,
	0x6c,0xdb,0x67,0xc8,0xf2,0x2c,0x9d,0x36,0x4a,0xad,0xd0,0x31,0x2d,0x79,0x68,0x5f,
	0x4a,0x40,0xf2,0x9a,0xfe,0x88,0xf1,0xbd,0x88,0x22,0xcd,0x33,0x3f,0xde,0xdb,0x7a,
	0xa5,0xad,0x7b,0x71,0x70,0x70,0xe0,0x6a,0xba,0x9a,0x05,0xc7,0xb3,0xa8,0x3e,0xae,
	0x82,0x12,0x37,0x07,0xfc,0xe2,0x41,0xc4,0xa1,0xde,0x0e,0xfc,0x84,0x84,0xfc,0x0d,
	0xc4,0x02,0x19,0x9e,0x2f,0xc6,0xf0,0x13,0x15,0x02,0x11,0x85,0x6b,0x35,0xdf,0xff,
	0x49,0xff,0xf3,0xc5,0x27,0xfa,0x48,0xb5,0xba,0x8a,0x2d,0xb9,0x11,0x55,0xeb,0x7c,
	0xee,0x31,0x5d,0xac,0x03,0x29,0xff,0x23,0xbe,0xf8,0x2a,0x10,0xa7,0x4a,0x7e,0xd0,
	0x26,0xa1,0x89,0xe6,0xb8,0x6b,0x72,0x5a,0xc4,0x74,0x4c,0x4f,0x6d,0xdd,0x75,0x50,
	0xda,0x45,0xce,0x17,0xd5,0xc9,0xd5,0xd6,0xbc,0x5f,0x18,0x59,0x8c,0x89,0xf7,0x9d,
	0xf3,0x4c,0x3e,0xc5,0x91,0x39,0x22,0x05,0x19,0xe0,0xb0,0x95,0x50,0x9c,0x04,0x0b,
	0x92,0x80,0x64,0xcf,0x42,0x58,0x2c,0x52,0x2b,0x4d,0xcb,0xa7,0x25,0x76,0xbb,0xee,
	0x4f,0xbd,0xec,0x5e,0x8e,0x99,0xa8,0x5a,0x53,0xe2,0x4a,0x16,0xd6,0x73,0xc3,0xc6,
	0x14,0x20,0x62,0xcc,0x7e,0xd8,0x3c,0x59,0xd9,0x2e,0xf0,0xff,0xe4,0x9d,0x64,0xa7,
	0xcd,0x39,0x88,0xc8,0xbe,0x1e,0x37,0x05,0xa1,0xd6,0x16,0x9f,0xe0,0x6e,0x99,0xcc,
	0x20,0xf4,0xfc,0x11,0x2f,0xd9,0x84,0x6c,0xcd,0xdd,0x5f,0xd2,0x94,0x9c,0xbe,0xf3,
	0x9e,0x93,0x02,0xa3,0x78,0x8e,0x3a,0x18,0x82,0xd8,0x75,0x32,0xf7,0xdf,0xbe,0x83,
	0x03,0x2a,0x91,0x8e,0x33,0x90,0xcf,0xe4,0xc8,0xf7,0x85,0x03,0x92,0xdd,0x46,0x1e,
	0x20,0xe7,0xcb,0xaa,0xa5,0xc0,0x3c,0x9b,0x1e,0x5d,0x0a,0x6d,0xd0,0x89,0xa6,0x39,
	0x4f,0x7e,0x45,0xcf,0xb4,0xd1,0x46,0x1f,0x47,0xa2,0x0a,0x6c,0x04,0x7d,0xcb,0x57,
	0xd5,0x54,0x87,0xc1,0x83,0xb0,0x72,0x33,0xad,0x0f,0x87,0xdc,0x67,0x46,0x6b,0xca,
	0x8c,0x0c,0x5a,0xd2,0x92,0x31,0x5c,0x5c,0x25,0xb1,0xaf,0x41,0x65,0x65,0x1c,0x1b,
	0x21,0x37,0xb9,0x57,0x34,0x77,0x0f,0x3e,0xe9,0xe2,0x8e,0x90,0xe8,0xa1,0x53,0x04,
	0x09,0x60,0xf6,0xed,0xa4,0x1c,0x53,0x13,0xf1,0x6f,0x02,0x70,0xcd,0xd2,0x92,0x2f,
	0x84,0x68,0x2b,0x9f,0xf5,0x1c,0x41,0x0c,0x8a,0x80,0x15,0x0f,0x5f,0x11,0x2e,0xbe,
	0x3d,0x63,0x6e,0xcb,0xd2,0x7c,0x03,0x39,0x52,0xfa,0x5a,0x2e,0x1d,0x65,0x49,0x01,
	0x07,0xbb,0xb6,0x65,0x42,0x3a,0xac,0xa5,0x49,0xe3,0x37,0x08,0x5e,0xca,0x59,0x4c,
	0xee,0x36,0xd2,0x77,0x1a,0xbb,0xbf,0xeb,0xf0,0x18,0x80,0xa2,0x1d,0x7f,0x07,0x6b,
	0x53,0x18,0x89,0x14,0xfa,0xda,0xb3,0x48,0xe1,0x07,0x72,0x5f,0xda,0x47,0x62,0x86,
	0x4f,0xf7,0xd8,0xdd,0xcb,0xe0,0x74,0x3a,0x03,0x88,0x64,0xcd,0xdf,0x6c,0x4e,0xb8,
	0x64,0xc4,0x0d,0xcc,0x92,0x44,0x7d,0x73,0x99,0xa3,0xec,0xe8,0x5a,0x9c,0x08,0x63,
	0x1d,0xe8,0x4e,0xba,0x54,0x8a,0x98,0x48,0xa7,0xda,0xea,0x6c,0x9e,0x1a,0xb0,0x7b,
	0xa6,0x9d,0xe6,0x6a,0xc5,0x2f,0x04,0xf7,0x3a,0x7d,0xed,0xb5,0x03,0x42,0xda,0x57,
	0x8e,0xd5,0x44,0x1b,0x14,0x83,0xba,0x2b,0x92,0x80,0x77,0x06,0x51,0xf3,0xeb,0x99,
	0x05,0x61,0x77,0xda,0xd2,0x87,0xc9,0x54,0x3f,0x47,0xa7,0xd3,0xf2,0xdf,0x04,0x03,
	0xca,0x5c,0x4c,0x4d,0x44,0xe2,0xc5,0x38,0x9d,0x6f,0xa6,0x90,0xb7,0xee,0xde,0x2d,
	0x7a,0xb3,0xb8,0x0b,0x49,0x6a,0xd1,0x41,0x13,0x1e,0xf9,0xf4,0xa5,0x59,0x4e,0x60,
	0x67,0xd7,0xda,0x8b,0x28,0x6f,0xaf,0xdd,0xaf,0xc3,0x18,0x96,0x01,0xe1,0xd1,0x4f,
	0x6e,0x5d,0xa2,0x17,0xdf,0x74,0x54,0x83,0xb1,0x7a,0x09,0xe3,0x42,0xb1,0xec,0xd2,
	0x64,0x0f,0xa6,0x56,0xb8,0x17,0x37,0xb2,0x4a,0x7c,0x65,0xe6,0xcc,0x6d,0x8c,0x61,
	0x3b,0xc7,0xf4,0xc6,0x3f,0x82,0xe8,0xd8,0x94,0x11,0x43,0xd5,0x49,0x7d,0x83,0xe9,
	0x01,0xeb,0x3e,0xe0,0x6e,0x4a,0x14,0xe7,0x41,0x47,0xb3,0x27,0x0a,0x15,0x0e,0x74,
	0x97,0x98,0xdc,0xb0,0x19,0xe9,0xf7,0x5c,0x7b,0x08,0xcd,0x55,0xae,0x55,0xb9,0x0c,
	0xde,0x27,0xf8,0xb0,0x10,0x45,0xdf,0x65,0x64,0x13,0x2a,0x48,0xbc,0x82,0xe1,0x57,
	0x85,0x62,0x26,0xe0,0x76,0xa7,0x95,0x7e,0xce,0xe2,0x3b,0xdc,0xf9,0xfe,0x4d,0x3f,
	0x3a,0xae,0x25,0xc6,0xa6,0x54,0x30,0xc0,0x87,0x9d,0x46,0xaf,0x53,0xa8,0x9a,0xf4,
	0xb1,0x66,0x50,0x56,0x57,0xdc,0xfe,0xdd,0x2b,0xdc,0xe0,0xea,0x00,0x67,0x92,0x9d,
	0x05,0xdb,0x17,0x77,0x0e,0xf2,0xd5,0x42,0xb1,0x1e,0xde,0x45,0x41,0x0e,0x76,0x0e,
	0xae,0xd9,0x80,0x67,0xd3,0xf6,0x3f,0xd7,0x48,0x7f,0xfd,0x91,0x26,0xbe,0x7e,0x24,
	0x4d,0x90,0x08,0xea,0x61,0xc8,0xb6,0xaa,0xa8,0x8e,0x3f,0x32,0xfc,0x11,0xdd,0xdd,
	0x42,0xd2,0x99,0xfd,0xd5,0x53,0xb1,0xe4,0x35,0x43,0xda,0xd0,0x1b,0xe7,0xd1,0x31,
	0x98,0xd0,0x2e,0x6d,0xdd,0x21,0x8c,0x0f,0x92,0xc3,0xb8,0xb5,0xb2,0x8a,0xd9,0xcf,
	0x2b,0x8a,0x18,0x3a,0xae,0xce,0xe2,0xd0,0x1a,0x63,0xbc,0xcb,0xd3,0x02,0xd3,0x14,
	0x92,0x2f,0xb8,0xab,0xc4,0x12,0x3f,0x2c,0x26,0x6c,0xff,0xb9,0xd2,0xd5,0x3b,0x5c,
	0xe1,0x3a,0x79,0x1a,0xf7,0x31,0x97,0xce,0x10,0x1a,0xe1,0x1b,0x78,0x58,0x21,0x3d,
	0x1b,0xf2,0xd9,0x25,0x2e,0x7f,0xea,0xf7,0x4f,0xad,0x45,0xce,0xff,0x20,0xbe,0xf1,
	0x14,0xc0,0xde,0x82,0x9c,0xf5,0xb2,0x4d,0xc9,0x21,0x56,0x7b,0xe6,0xd1,0x93,0xbf,
	0x61,0x34,0xf9,0xa3,0x78,0x47,0x09,0xb5,0x63,0x00,0xa2,0xb9,0xe1,0xdb,0x0b,0x83,
	0xfc,0xe8,0xcf,0x15,0x65,0x69,0xb0,0xd9,0x4c,0xd7,0x57,0x85,0xb7,0x35,0xd7,0x2c,
	0x39,0xbd,0xe4,0x93,0x40,0xa2,0x0d,0x41,0xbc,0xdf,0xa2,0x67,0xcb,0x45,0x8f,0xa0,
	0xd6,0xb9,0xd3,0xf3,0x7f,0xa1,0x8b,0x2c,0x27,0xee,0x56,0x95,0x2a,0xce,0xfd,0x47,
	0x0c,0x26,0xf4,0xb5,0xbb,0x1e,0x22,0x79,0x97,0x2f,0x45,0x31,0x43,0x46,0xa0,0x5f,
	0x48,0xa3,0x64,0xae,0x9f,0xa2,0x0e,0xf1,0x1f,0x25,0xe1,0x33,0x22,0x47,0x1e,0x44,
	0x11,0x57,0x23,0xfe,0xc3,0xc6,0x50,0xdb,0xf5,0x83,0xf4,0x86,0x35,0x9c,0xb2,0x9d,
	0x34,0xbd,0x52,0xc7,0x08,0x4c,0x34,0x42,0xd3,0x7f,0x69,0x68,0x83,0x0d,0x86,0x28,
	0x88,0x42,0xd8,0xc8,0xf5,0x65,0xbf,0xbc,0xda,0xba,0x9a,0x18,0x24,0x80,0x78,0xcc,
	0xfb,0x73,0xee,0x5a,0x27,0x6f,0xb6,0x0b,0x8a,0x2e,0x6f,0x16,0x58,0x7e,0xd2,0x94,
	0x82,0x63,0xe9,0xb9,0x68,0x21,0x22,0x31,0x2c,0x3d,0x6c,0x07,0xb6,0x12,0x44,0xd9,
	0x11,0x32,0xb1,0xb5,0x1b,0x2a,0x38,0xd7,0x26,0xe9,0xe9,0xb7,0xc1,0x4b,0x2b,0x2a,
	0x33,0xa2,0x61,0x81,0x46,0xa4,0x01,0xcf,0xe7,0xdc,0x0b,0xdd,0x2a,0x57,0x81,0x23,
	0xfe,0x3a,0x00,0xb9,0x89,0xb9,0x8b,0x54,0xd6,0x0f,0x28,0xcb,0x63,0xae,0xde,0xc6,
	0x7a,0x7d,0x46,0x6d,0x5a,0x8f,0x35,0x24,0xc8,0xad,0xc4,0xa3,0x15,0xab,0x72,0x9a,
	0x68,0xad,0xe1,0x4e,0x17,0x46,0x49,0x7a,0xfe,0xc9,0x36,0x32,0xbe,0x66,0x8a,0x13,
	0x21,0x4d,0x91,0xd2,0x29,0xeb,0x3c,0x6b,0x7e,0x4a,0x02,0xd9,0x30,0xbf,0xfd,0xc0,
	0x92,0x9d,0x90,0x00,0xfb,0x35,0x43,0xc1,0xb4,0x89,0xf6,0x7e,0xc7,0x62,0x11,0xdc,
	0x16,0xdd,0x3d,0x19,0x97,0x4e,0xff,0x1f,0xc5,0x26,0x65,0xee,0x3a,0x6a,0xce,0xe8,
	0x9d,0xca,0xdc,0x6c,0xb8,0xf7,0x30,0x29,0x74,0xc2,0xa6,0x80,0x62,0x41,0x05,0xa1,
	0x39,0xec,0x8d,0x38,0x9c,0xbd,0xad,0xb3,0x7b,0xe7,0x98,0x46,0xe5,0xf8,0x1d,0x0a,
	0x7c,0xeb,0x6d,0x33,0xf1,0xe5,0x62,0x72,0x3d,0x04,0xab,0x2d,0xb9,0xc5,0x91,0xd9,
	0x24,0x9a,0xaa,0xb7,0x28,0x6f,0x69,0x6d,0x51,0x56,0x8f,0xfa,0x30,0xbd,0xb6,0x11,
	0x80,0x1c,0x3b,0xcf,0x1d,0x17,0x7c,0xf6,0x90,0xe2,0x2d,0xfa,0xd1,0x96,0x0f,0xe4,
	0x4b,0x51,0xc2,0xfd,0x41,0x6a,0x33,0xb6,0x61,0x50,0x38,0x24,0xcb,0xc5,0xa3,0x78,
	0x48,0x06,0x94,0x38,0x9e,0x3a,0x08,0x4e,0xfe,0x25,0x4c,0x44,0x6a,0x6a,0x0c,0x87,
	0x2d,0x16,0x45,0x43,0x5d,0xf2,0x97,0x4e,0x19,0x07,0x6e,0x89,0xf4,0xe0,0x60,0x14,
	0xa4,0x20,0x15,0xb0,0xe1,0x5e,0xdc,0x73,0x49,0x74,0x54,0x4e,0x68,0x4f,0xee,0x30,
	0x36,0x82,0xb8,0xbe,0xa0,0x44,0xeb,0xc3,0xb7,0xa0,0x29,0x78,0xc4,0xe2,0x8d,0x54,
	0x81,0x83,0x1c,0xcc,0xca,0x30,0xbd,0x64,0x0d,0xb3,0xc6,0xa6,0x5a,0x25,0xd3,0x09,
	0xf1,0xbf,0x86,0x16,0x25,0xaa,0xa1,0x25,0xd9,0x11,0x26,0x7d,0xa6,0xdc,0xbe,0x5b,
	0xb7,0xbf,0xa3,0xf9,0xcf,0xf3,0x70,0x4c,0x8c,0x05,0xba,0x8a,0x18,0xd7,0x4c,0x7e,
	0x57,0x88,0xe6,0x0d,0x10,0xcf,0xaf,0x5d,0x09,0xa0,0xdd,0x34,0x42,0x2f,0xb3,0xb1,
	0x77,0x89,0xa9,0x77,0xcd,0xd6,0x02,0x6a,0x8b,0xde,0x3a,0x08,0x9c,0xe3,0xc1,0x62,
	0x4e,0x1e,0x3c,0x24,0x3c,0x23,0xe0,0x09,0xbb,0xb2,0x99,0x67,0x9b,0x5b,0xf6,0x78,
	0x4e,0x60,0x7d,0x34,0xc3,0xbc,0xd1,0x8c,0xce,0xe3,0x46,0x64,0x6c,0xd2,0xdd,0x8e,
	0x9f,0x14,0x20,0x6e,0x27,0xbb,0x01,0xd9,0x2d,0xf8,0x0b,0x2a,0x11,0xd7,0xd9,0x5e,
	0xd7,0x80,0x7e,0x41,0xfd,0xd6,0xb9,0x0d,0x48,0x0f,0x3c,0x36,0x9e,0x2c,0x9a,0x18,
	0x31,0x9f,0xac,0x0b,0x54,0x59,0x4e,0x7f,0xaa,0xbe,0x52,0x7e,0x13,0xf8,0x02,0x19,
	0x70,0x1b,0xe7,0xc2,0x78,0x2e,0x69,0x76,0x16,0xd6,0xbe,0xa2,0xbb,0x66,0xe6,0x8f,
	0xa6,0x51,0xbf,0x4e,0xf6,0x23,0x02,0x12,0x43,0xc1,0x7e,0xba,0x87,0x93,0x62,0x48,
	0xa1,0x82,0x1a,0xe0,0x7d,0xc3,0xc5,0x27,0xe6,0x4c,0xa6,0x96,0xff,0x59,0x58,0x3f,
	0xd6,0xf0,0x00,0xbb,0x09,0xea,0x03,0x83,0xb2,0xcb,0x7b,0x08,0xd9,0x91,0x9c,0x6e,
	0x3c,0x69,0xf4,0x03,0x37,0x7b,0x6c,0xe0,0x67,0x3a,0x0b,0xd4,0x33,0xb4,0xde,0x38,
	0x4d,0xca,0x3a,0x7f,0x7b,0x16,0x8b,0x85,0x15,0xa2,0xb5,0x6a,0x2d,0xf9,0x76,0xb2,
	0x4a,0xb6,0x70,0x11,0x44,0x19,0xac,0x06,0x90,0x03,0x39,0xf0,0x11,0x2d,0xb2,0xc3,
	0xeb,0x42,0xd2,0x96,0xf9,0xe9,0x07,0xf3,0x18,0x99,0xcf,0x23,0x1e,0xc3,0x10,0x95,
	0x8c,0xb1,0xed,0x7a,0x17,0x54,0xc0,0xc0,0x9b,0x1f,0xd1,0x4d,0x43,0x2e,0xad,0x63,
	0x1c,0xa8,0x56,0xd3,0xf2,0xc3,0xd0,0x33,0x1e,0x21,0x6d,0xc1,0x75,0x4b,0x29,0xf4,
	0xac,0xfb,0x83,0x1e,0x99,0x8d,0x98,0x80,0x6c,0x06,0x9b,0x49,0x0b,0x23,0x0a,0xf4,
	0x63,0x76,0x0c,0x56,0xa1,0x96,0x1f,0x9a,0xb9,0x07,0xa4,0xa2,0xb0,0xa1,0xc3,0x63,
	0xf5,0xa2,0xb0,0x4c,0x2a,0x57,0x59,0xc8,0xed,0xad,0xce,0x1b,0x1f,0x8b,0xd7,0x95,
	0x42,0x97,0x05,0x63,0xf5,0x6f,0xa4,0x6d,0x80,0xfc,0x0f,0x87,0xc3,0x63,0xf2,0xc3,
	0xc5,0xd8,0x5b,0x7f,0xb8,0xf9,0x84,0x0e,0x14,0xf3,0xc7,0x7f,0xba,0x31,0xdf,0xb2,
	0x83,0x0b,0x96,0x76,0x06,0x32,0x7d,0xcb,0x70,0x13,0x6d,0x35,0x8c,0x8b,0x3b,0x33,
	0x02,0x06,0x42,0x7b,0xb2,0x2a,0x1a,0xe8,0x82,0xa3,0x8a,0xcc,0xf1,0x1e,0xf3,0x6c,
	0xbe,0x62,0xcd,0x43,0x59,0xdd,0x2b,0x2a,0x27,0x6f,0xa2,0xe9,0x09,0xf7,0x30,0x3d,
	0xb9,0xd0,0x61,0x42,0x07,0x17,0xc0,0x2f,0xc4,0x6e,0x32,0x15,0x14,0x4b,0xd4,0x4a,
	0xd9,0x25,0x0f,0xdc,0xa0,0xe6,0x7b,0x60,0xe6,0x2e,0xfb,0x5b,0xbe,0x1f,0x8d,0x89,
	0x7a,0x0b,0xc1,0x75,0x5f,0xc7,0xae,0xdf,0x61,0xc1,0x16,0x47,0x17,0xcb,0x1b,0xf9,
	0xf6,0xf0,0x7b,0x6f,0x42,0x19,0xeb,0x7a,0xa7,0x70,0xff,0xaf,0x24,0x3e,0x1a,0xb8,
	0xf3,0x04,0x14,0x2a,0x94,0x48,0x05,0x0f,0x5b,0x0d,0xc4,0xe8,0xdf,0xf7,0x5c,0x47,
	0x0c,0x2b,0x3f,0x33,0x7a,0x08,0x97,0x1b,0x6d,0x43,0xcb,0x84,0x07,0x75,0x8c,0xc7,
	0x68,0x50,0x96,0xff,0xfb,0x14,0x44,0x68,0xc2,0xb5,0xb0,0xcd,0x93,0x66,0x7a,0x06,
	0x3d,0xb2,0x3b,0x91,0x26,0x63,0x60,0x7a,0xa2,0x6e,0x9f,0xa7,0xd9,0xcd,0x4c,0x66,
	0xa6,0xca,0x54,0x2b,0xc8,0x42,0x43,0x51,0xf2,0x47,0xe3,0xf8,0x7c,0xbe,0xa6,0xd1,
	0x54,0xc0,0x28,0xf2,0xd8,0x5b,0x60,0x62,0x89,0x00,0xad,0xd3,0x64,0x6a,0x80,0x20,
	0xbd,0xa2,0x09,0x6f,0x83,0x6b,0x44,0xa1,0x9f,0x6b,0x4f,0x22,0x39,0xc3,0x76,0x00,
	0xaf,0x8c,0x86,0x4d,0x35,0x8c,0x97,0x25,0x10,0xc5,0x32,0x6b,0x8a,0x80,0xf0,0xfc,
};


/* 317-5021 FD1089B */
static const UINT8 key_5021[0x2000] =
{
	/* data */
	0x40,0x7d,0xd5,0x54,0xe5,0x9a,0x53,0xbe,0x3b,0x2e,0xf9,0xcd,0x02,0x24,0x12,0x4f,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0xe6,0xd7,0x93,0x50,0x86,0x25,0x28,0xa4,0x96,0x43,0x03,0x61,0x3e,0x8f,0x24,0x57,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};





struct parameters
{
	int xor;
	int s7,s6,s5,s4,s3,s2,s1,s0;
};

static const UINT8 basetable_fd1089[0x100] =
{
	0x00,0x1c,0x75,0x69,0x5d,0x41,0x24,0x38,0x4b,0x67,0xae,0x82,0xea,0xc6,0x03,0x2f,
	0x46,0x6a,0xaf,0x83,0xe7,0xcb,0x02,0x2e,0x01,0x1d,0x78,0x64,0x5c,0x40,0x29,0x35,
	0x31,0x2d,0x44,0x58,0xe4,0xf8,0x9d,0x81,0x2a,0x06,0xcf,0xe3,0x93,0xbf,0x7a,0x56,
	0x3f,0x13,0xd6,0xfa,0x86,0xaa,0x63,0x4f,0xb8,0xa4,0xc1,0xdd,0x6d,0x71,0x18,0x04,
	0x0c,0x10,0x79,0x65,0xfc,0xe0,0x85,0x99,0x47,0x6b,0xa2,0x8e,0xbb,0x97,0x52,0x7e,
	0x17,0x3b,0xfe,0xd2,0xeb,0xc7,0x0e,0x22,0xa0,0xbc,0xd9,0xc5,0x50,0x4c,0x25,0x39,
	0x3d,0x21,0x48,0x54,0x45,0x59,0x3c,0x20,0x26,0x0a,0xc3,0xef,0xc2,0xee,0x2b,0x07,
	0x6e,0x42,0x87,0xab,0x8a,0xa6,0x6f,0x43,0x19,0x05,0x60,0x7c,0x61,0x7d,0x14,0x08,
	0x09,0x15,0x70,0x6c,0xdc,0xc0,0xa9,0xb5,0x4e,0x62,0xa7,0x8b,0xf7,0xdb,0x12,0x3e,
	0x5b,0x77,0xbe,0x92,0xe2,0xce,0x0b,0x27,0x80,0x9c,0xf5,0xe9,0x55,0x49,0x2c,0x30,
	0xb0,0xac,0xc9,0xd5,0xed,0xf1,0x98,0x84,0x37,0x1b,0xde,0xf2,0x96,0xba,0x73,0x5f,
	0x3a,0x16,0xdf,0xf3,0x9b,0xb7,0x72,0x5e,0xb1,0xad,0xc4,0xd8,0xec,0xf0,0x95,0x89,
	0xa8,0xb4,0xd1,0xcd,0xd0,0xcc,0xa5,0xb9,0x1f,0x33,0xf6,0xda,0xfb,0xd7,0x1e,0x32,
	0x57,0x7b,0xb2,0x9e,0xb3,0x9f,0x5a,0x76,0x8c,0x90,0xf9,0xe5,0xf4,0xe8,0x8d,0x91,
	0x11,0x0d,0x68,0x74,0xe1,0xfd,0x94,0x88,0x66,0x4a,0x8f,0xa3,0x9a,0xb6,0x7f,0x53,
	0x36,0x1a,0xd3,0xff,0xca,0xe6,0x23,0x0f,0xbd,0xa1,0xc8,0xd4,0x4d,0x51,0x34,0x28,
};


/* common to FD1089A and FD1089B */
static const struct parameters addr_params[16] =
{
	{ 0x26, 3,1,7,5,2,4,6,0 },
	{ 0x08, 6,7,4,2,0,5,1,3 },
	{ 0x92, 6,4,5,7,3,0,1,2 },
	{ 0xce, 3,5,6,0,2,1,7,4 },
	{ 0x7e, 7,4,1,0,6,2,3,5 },
	{ 0x52, 1,0,3,7,4,5,2,6 },
	{ 0x44, 0,4,1,2,6,5,7,3 },
	{ 0xe0, 5,7,2,4,3,1,6,0 },
	{ 0x74, 3,1,6,4,5,0,2,7 },
	{ 0xc2, 1,6,0,5,7,2,4,3 },
	{ 0xa4, 7,2,0,6,1,3,4,5 },
	{ 0xea, 0,7,5,3,1,4,2,6 },
	{ 0xc4, 5,3,7,1,4,6,0,2 },
	{ 0x80, 7,6,2,3,0,4,5,1 },
	{ 0x22, 2,5,3,6,7,1,0,4 },
	{ 0xb2, 2,3,4,0,6,7,5,1 },
};


static int decode_fd1089a(int val,int table,int opcode)
{
	static const struct parameters data_params[16] =
	{
		{ 0x94, 7,6,4,2,1,5,0,3 },
		{ 0x3f, 7,3,1,2,4,6,0,5 },
		{ 0x99, 4,3,5,6,1,2,0,7 },
		{ 0x8a, 3,5,1,6,0,2,7,4 },
		{ 0xff, 0,7,2,3,6,4,5,1 },
		{ 0xf7, 6,3,7,1,5,4,2,0 },
		{ 0xc6, 3,5,0,4,2,7,1,6 },
		{ 0xac, 0,6,3,5,1,7,4,2 },
		{ 0x56, 6,5,0,1,7,4,2,3 },
		{ 0xec, 4,7,5,0,6,1,2,3 },
		{ 0x8e, 0,4,2,3,1,6,7,5 },
		{ 0x5c, 0,3,1,7,5,2,4,6 },
		{ 0x05, 6,5,3,2,4,0,1,7 },
		{ 0x39, 6,0,3,2,7,4,5,1 },
		{ 0x71, 4,3,7,1,5,6,0,2 },
		{ 0x57, 6,4,7,2,0,5,3,1 },
	};
	const struct parameters *p = &addr_params[table >> 4];
	const struct parameters *q;
	int family = table & 0x0e;


	/* special case - don't decrypt */
	if (opcode == 0 && table == 0xff)
		return val;

	val = BITSWAP8(val, p->s7,p->s6,p->s5,p->s4,p->s3,p->s2,p->s1,p->s0) ^ p->xor;

	if (table & 0x01) val ^= 0x01;
	if (table & 0x08) val ^= 0xb0;
	if (opcode) val ^= 0x34;

	val = basetable_fd1089[val];

	if (table & 0x04)
	{
		family ^= 1;
		if (table & 0x40) family ^= 1;
	}
	else
	{
		if (table & 0x80) family ^= 1;
		if (opcode) family ^= 1;
	}

	if (opcode)
	{
		if (table & 0x20) family ^= 1;
		if (table & 0x40) family ^= 1;
	}

	if (~table & 0x08)
	{
		if (val & 0x02) val ^= 0xc0;
		if ((BIT(val,6) ^ BIT(val,4)) == 0)
			val = BITSWAP8(val, 7,6,5,4,0,1,3,2);
	}
	else
	{
		if ((BIT(val,6) ^ BIT(val,4)) == 0)
			val = BITSWAP8(val, 7,6,5,4,1,0,2,3);
	}
	if (!BIT(val,6))
		val = BITSWAP8(val, 7,6,5,4,2,3,0,1);

	q = &data_params[family];

	val ^= q->xor;
	val = BITSWAP8(val, q->s7,q->s6,q->s5,q->s4,q->s3,q->s2,q->s1,q->s0);

	return val;
}

static int decode_fd1089b(int val,int table,int opcode)
{
	const struct parameters *p = &addr_params[table >> 4];


	/* special case - don't decrypt */
	if (opcode == 0 && table == 0xff)
		return val;

	val = BITSWAP8(val, p->s7,p->s6,p->s5,p->s4,p->s3,p->s2,p->s1,p->s0) ^ p->xor;

	if (table & 0x01) val ^= 0x01;
	if (table & 0x08) val ^= 0xb0;
	if (opcode) val ^= 0x34;

	val = basetable_fd1089[val];

	if (table & 0x04)
	{
		val ^= 0x02;
		if (table & 0x40) val ^= 0x02;
	}
	else
	{
		if (table & 0x80) val ^= 0x02;
		if (opcode) val ^= 0x02;
	}

	if (opcode)
	{
		if (table & 0x20) val ^= 0x02;
		if (table & 0x40) val ^= 0x02;
	}

	if (table & 0x02)
		val = BITSWAP8(val, 7,6,5,4,1,0,2,3);
	if (table & 0x04)
		val = BITSWAP8(val, 7,6,5,4,2,3,0,1);

	return val;
}


enum
{
	FD1089A,
	FD1089B
};

static UINT16 fd1089_decrypt(offs_t addr,UINT16 val,const UINT8 *key,int opcode,int cputype)
{
	int tbl_num,src;

	/* pick the translation table from bits ff022a of the address */
	tbl_num =	((addr & 0x000002) >> 1) |
				((addr & 0x000008) >> 2) |
				((addr & 0x000020) >> 3) |
				((addr & 0x000200) >> 6) |
				((addr & 0xff0000) >> 12);

	src =	((val & 0x0008) >> 3) |
			((val & 0x0040) >> 5) |
			((val & 0xfc00) >> 8);

	switch (cputype)
	{
		case FD1089A: src = decode_fd1089a(src,key[tbl_num + opcode * 0x1000],opcode); break;
		case FD1089B: src = decode_fd1089b(src,key[tbl_num + opcode * 0x1000],opcode); break;
	}

	src =	((src & 0x01) << 3) |
			((src & 0x02) << 5) |
			((src & 0xfc) << 8);

	return (val & ~0xfc48) | src;
}

static UINT16 *decrypted;

static void sys16_decrypt(const UINT8 *key,int cputype)
{
	UINT16 *rom = (UINT16 *)memory_region(REGION_CPU1);
	int size = memory_region_length(REGION_CPU1);
	int A;
	decrypted = (UINT16 *)auto_malloc(size);

	memory_set_decrypted_region(0, 0x000000, size - 1, decrypted);

	for (A = 0;A < size;A+=2)
	{
		UINT16 src = rom[A/2];

		/* decode the opcodes */
		decrypted[A/2] = fd1089_decrypt(A,src,key,1,cputype);

		/* decode the data */
		rom[A/2] = fd1089_decrypt(A,src,key,0,cputype);
	}
}

void *fd1089_get_decrypted_base(void)
{
	return decrypted;
}



void fd1089_decrypt_0013A(void)
{
	sys16_decrypt(key_0013A,FD1089B);
}

void fd1089_decrypt_0018(void)
{
	sys16_decrypt(key_0018,FD1089A);
}

void fd1089_decrypt_alexkidd(void)
{
	sys16_decrypt(key_alexkidd,FD1089A);
}

void fd1089_decrypt_0022(void)
{
	sys16_decrypt(key_0022,FD1089A);
}

void fd1089_decrypt_0024(void)
{
	sys16_decrypt(key_0024,FD1089B);
}

void fd1089_decrypt_0027(void)
{
	sys16_decrypt(key_0027,FD1089B);
}

void fd1089_decrypt_0028(void)
{
	sys16_decrypt(key_0028,FD1089A);
}

void fd1089_decrypt_0033(void)
{
	sys16_decrypt(key_0033,FD1089A);
}

void fd1089_decrypt_0034(void)
{
	sys16_decrypt(key_0034,FD1089B);
}

void fd1089_decrypt_0037(void)
{
	sys16_decrypt(key_0037,FD1089B);
}

void fd1089_decrypt_0167(void)
{
	sys16_decrypt(key_0167,FD1089A);
}

void fd1089_decrypt_0168(void)
{
	sys16_decrypt(key_0168,FD1089B);
}

void fd1089_decrypt_5021(void)
{
	sys16_decrypt(key_5021,FD1089B);
}
