/*
 * matrixview.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include "matrixview.h"

#include "wxini.h"
#include "wxext.h"
#include "dialogs.h"
#include "tourview.h"

#include "bitmaps/refresh.xpm"

BEGIN_EVENT_TABLE(MatrixFrame, wxxMDIChildFrame)
  EVT_MENU(Matrix_Redisplay, MatrixFrame::OnRedisplay)
  // close events
  EVT_MENU(wxID_CLOSE, MatrixFrame::OnCloseCommand)
  EVT_CLOSE(MatrixFrame::OnCloseWindow)
  // my competition events
  EVT_TOUR_UPDATE(MatrixFrame::OnTourUpdate)
  EVT_TOUR_CLOSE(MatrixFrame::OnTourClose)
END_EVENT_TABLE()

MatrixFrame::MatrixFrame(TournamentFrame* tour, wxMDIParentFrame* parent)
	: wxxMDIChildFrame(parent, wxString::Format(_("Matrix - [%s]"), tour->GetConfigData().tourName.c_str()), wxT("MatrixFrame")), 
	  parent(parent), tour(tour)
{
	wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );
	wxPanel* panel = new wxPanel(this); 

	wxToolBar *bar = new wxToolBar(panel, -1, wxDefaultPosition, wxDefaultSize,  
		wxNO_BORDER | wxTB_FLAT | wxTB_HORIZONTAL | wxTB_NODIVIDER);
	InitToolBar(bar);
	wxxAddCloseBar(panel, this, sizer, bar, (wxCommandEventFunction) &MatrixFrame::OnCloseCommand);

	matrix = new MatrixWindow(panel, tour->GetConfigData());
	sizer->Add( matrix, 1, wxEXPAND | wxALL, 2 );

	panel->SetSizer(sizer); 

	// size myself
	wxBoxSizer* mySizer = new wxBoxSizer(wxVERTICAL); 
	mySizer->Add(panel, 1, wxEXPAND | wxALL, 0); 
	SetSizer(mySizer);
	SetSizeHints(mySizer->GetMinSize()); // set size hints to honour mininum size

	tour->AddTournamentListener(this);
	// if redisplay()ing directly, it stops the program for too long, so let's send us an event :-)
	TournamentEvent evt(EVT_TOUR_UPDATED);
	wxPostEvent(this, evt);
}

MatrixFrame::~MatrixFrame()
{
}

void MatrixFrame::InitToolBar(wxToolBar* toolBar)
{
	const int numBitmaps = 1;
    wxBitmap* bitmaps[numBitmaps];

    bitmaps[0] = new wxBitmap( refresh_xpm );
	
	wxSize bitmapSize(bitmaps[0]->GetWidth(), bitmaps[0]->GetHeight());
	toolBar->SetToolBitmapSize(bitmapSize);

    toolBar->AddTool( Matrix_Redisplay, *(bitmaps[0]), 
		_("Update Display"), _("Update the display with the newest simulation results"));

    toolBar->Realize();
    for (int i = 0; i < numBitmaps; i++)
        delete bitmaps[i];
}

void MatrixFrame::Redisplay()
{
	if(matrix->IsCalculating())
		return;
	matrix->Recalculate();
	matrix->Redisplay();
}

void MatrixFrame::OnTourUpdate(TournamentEvent& event)
{
	Redisplay();
}

void MatrixFrame::OnTourClose(TournamentEvent& event)
{
	// there's no tour any more
	tour = 0;
	// cascaded closing?
	if(event.IsForce())
		Destroy();
}

void MatrixFrame::OnRedisplay(wxCommandEvent& event)
{
	Redisplay();
}


void MatrixFrame::OnCloseCommand(wxCommandEvent& event)
{
	wxCloseEvent pseudoEvent;
	pseudoEvent.SetCanVeto(true);
	OnCloseWindow(pseudoEvent);
}

void MatrixFrame::OnCloseWindow(wxCloseEvent& event)
{
	if(tour)
		tour->RemoveTournamentListener(this);
	Destroy();
}



/////////////////////
/// my 2d array helper stuff
int** NewArray(size_t dim1, size_t dim2)
{
	int** ret = new int*[dim1];
	for(size_t i = 0; i < dim1; i++)
		ret[i] = new int[dim2];
	return ret;
}

void DeleteArray(int**& arr, size_t dim1)
{
	if(!arr)
		return;
	for(size_t i = 0; i < dim1; i++)
		delete[] arr[i];
	delete[] arr;
	arr = 0;
}
/////////////////////


BEGIN_EVENT_TABLE(MatrixWindow, wxControl)
  EVT_PAINT(MatrixWindow::OnPaint)
  EVT_MOTION(MatrixWindow::OnMotion)
END_EVENT_TABLE()


MatrixWindow::MatrixWindow(wxWindow* parent, const TourData& data) :
	wxControl(parent, -1, wxDefaultPosition, wxSize(MatrixWindowPrefSize + 200, MatrixWindowPrefSize), wxFULL_REPAINT_ON_RESIZE), 
	data(data), isCalculating(false), isDone(false), altText(_("Calculating...")), 
	matrixBmp(GetBitmapSize(data), GetBitmapSize(data)), 
	imgLeft(-1), imgTop(-1), curX(-1), curY(-1), wins(0), losses(0), ties(0)
{
	SetBackgroundColour(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE)); 
}

MatrixWindow::~MatrixWindow()
{
	DeleteArray(wins, data.chartSize);
	DeleteArray(losses, data.chartSize);
	DeleteArray(ties, data.chartSize);
}

///// UGLY! - but it's wxWindows's fault.
wxxIniFile* mySortIniFile__;
int MyCompareBotNames__(const wxString& n1, const wxString& n2)
{
	int p1 = 0, p2 = 0;
	wxxIniSection* sec = 0;
	sec = mySortIniFile__->GetSection(n1);
	if(sec)
		p1 = sec->GetLong(wxT("points"), 0);
	sec = mySortIniFile__->GetSection(n2);
	if(sec)
		p2 = sec->GetLong(wxT("points"), 0);
	return p2 - p1;
}

void MatrixWindow::WalkFiles(int** wins, int** losses, int** ties)
{
	for(size_t dim1 = 0; dim1 < (size_t) data.chartSize; dim1++)
		for(size_t dim2 = 0; dim2 < (size_t) data.chartSize; dim2++)
		{
			wins[dim1][dim2] = INVALID;
			losses[dim1][dim2] = INVALID;
			ties[dim1][dim2] = INVALID;
		}
	wxString resultFolder = GetMainTourFolder() + data.tourFolder + wxFILE_SEP_PATH
		+ wxT("results") + wxFILE_SEP_PATH;
	wxString chartsXXXFile = resultFolder + wxT("charts.xxx");
	wxxIniFile chartsXXX;
	if(!chartsXXX.Read(chartsXXXFile)) {
		wxLogError(_("Couldn't open charts: %s"), chartsXXXFile.c_str());
		return;
	}
	
	// load all bot names
	wxArrayString bots;
	chartsXXX.First();
	wxxIniSection* sec;
	while((sec = chartsXXX.Next()) != 0)
		bots.Add(sec->GetName());
	mySortIniFile__ = &chartsXXX;
	bots.Sort(MyCompareBotNames__);

	botNames.Clear();
	for(size_t pos = 0; pos < bots.GetCount(); pos++)
	{
		sec = chartsXXX.GetSection(bots[pos]);
		wxASSERT_MSG(sec != 0, wxT("bot not in charts??"));
		botNames.Add(sec->Get(wxT("programname")));
	}

	// load the results
	for(size_t player = 0; player < bots.GetCount(); player++)
	{
		wxString& botFile = bots[player];
		wxString resultFile = resultFolder + botFile;
		resultFile.Last() = 'x';
		wxxIniFile result;
		if(!result.Read(resultFile)) {
			wxLogWarning(_("Couldn't open results: %s"), resultFile.c_str());
			continue;
		}

		for(size_t player2 = 0; player2 < bots.GetCount(); player2++)
		{
			wxxIniSection* resSec = result.GetSection(bots[player2]);
			if(!resSec)
				continue;
			wins[player][player2] = (int) resSec->GetLong(wxT("wins"), INVALID);
			losses[player][player2] = (int) resSec->GetLong(wxT("looses"), INVALID);
			ties[player][player2] = (int) resSec->GetLong(wxT("ties"), INVALID);
		}

	}
}

void MatrixWindow::DrawBitmap(int** wins, int** losses, int** ties)
{
	int blockSize = GetDotSize();
	int rectSize = ((blockSize >= 5) ? blockSize - 1 : blockSize);

	wxMemoryDC memDC;
	wxBrush brush;
	memDC.SelectObject(matrixBmp);
	memDC.SetBrush(brush);
	memDC.SetPen(wxPen(*wxBLACK, 0, wxTRANSPARENT));
	for(size_t y = 0; y < (size_t) data.chartSize; y++)
		for(size_t x = 0; x < (size_t) data.chartSize; x++)
		{
			brush.SetColour(CalcColour(wins[y][x], losses[y][x], ties[y][x]));
			memDC.SetBrush(brush);
			memDC.DrawRectangle(blockSize * x, blockSize * y, rectSize, rectSize);
		}

	memDC.SelectObject(wxNullBitmap);
}

void MatrixWindow::Recalculate()
{
	isDone = false;
	isCalculating = true;
	altText = _("Walking through the files...");
	DeleteArray(wins, data.chartSize);
	DeleteArray(losses, data.chartSize);
	DeleteArray(ties, data.chartSize);

	wins = NewArray(data.chartSize, data.chartSize);
	losses = NewArray(data.chartSize, data.chartSize);
	ties = NewArray(data.chartSize, data.chartSize);
	WalkFiles(wins, losses, ties);

	altText = _("Drawing the bitmap...");
	DrawBitmap(wins, losses, ties);


	isCalculating = false;
	isDone = true;
}

void MatrixWindow::Redisplay()
{
	wxClientDC dc(this);
	int cw, ch;
	GetSize(&cw, &ch);
	Paint(dc, cw, ch);
}

bool MatrixWindow::IsCalculating()
{
	return isCalculating;
}

wxColour MatrixWindow::CalcColour(int wins, int losses, int ties)
{
	if((wins < 0) || (losses < 0) || (ties < 0))
		return wxColour(64, 64, 64);

	const int rw = 0x00, gw = 0x130, bw = 0x00,
	          rm = 0xff, gm = 0xff, bm = 0x00,
			  rl = 0x130, gl = 0x00, bl = 0x00;
	int rtemp = 0, gtemp = 0, btemp = 0;
	
	// calc wins & losses
	int wlsum = wins + losses;
	int wlsumh = (wlsum+1) / 2;
	if(wlsum) {
		if(wins > losses)
		{
			rtemp = (rw * (wins - wlsumh) + rm * losses) / wlsumh;
			gtemp = (gw * (wins - wlsumh) + gm * losses) / wlsumh;
			btemp = (bw * (wins - wlsumh) + bm * losses) / wlsumh;
		}
		else
		{
			rtemp = (wins * rm + (losses - wlsumh) * rl) / wlsumh;
			gtemp = (wins * gm + (losses - wlsumh) * gl) / wlsumh;
			btemp = (wins * bm + (losses - wlsumh) * bl) / wlsumh;
		}
	}

	// calc ties
	int sum = wlsum + ties;
	rtemp = (rtemp * wlsum + 0x60 * ties) / sum;
	gtemp = (gtemp * wlsum + 0x60 * ties) / sum;
	btemp = (btemp * wlsum + 0xC0 * ties) / sum;
	if(rtemp > 255) rtemp = 255;
	if(gtemp > 255) gtemp = 255;
	if(btemp > 255) btemp = 255;

	return wxColour(rtemp, gtemp, btemp);
}

void MatrixWindow::OnPaint(wxPaintEvent& event)
{
	wxPaintDC dc(this);
	int cw, ch;
	GetSize(&cw, &ch);
	Paint(dc, cw, ch);
}

// maxX,maxY is the maximum device coord we'll use
// it should be the component size for on-screen display,
// but wxPrintDC::GetSize() for printing
void MatrixWindow::Paint(wxDC& device, wxCoord maxX, wxCoord maxY)
{
	device.SetFont(*wxSWISS_FONT);
	wxCoord fontHeight = device.GetCharHeight();

	device.BeginDrawing();

	if(isDone) // paint bmp
	{
		imgLeft = (maxX - matrixBmp.GetWidth()) / 2;
		imgTop = (maxY - matrixBmp.GetHeight()) / 2;
		if(GetDotSize() >= 10) // the font would be too small otherwise
		{
			// adjust coords so that text is centered too
			wxCoord fontLeft = imgLeft - MatrixWindowTextHalf;
			wxCoord fontTop = imgTop - (MatrixWindowTextHalf / 2);
			imgLeft += MatrixWindowTextHalf / 2;
			imgTop += (MatrixWindowTextHalf / 2);
			wxCoord yAdd = (GetDotSize() - fontHeight) / 2;

			for(size_t pos = 0; pos < botNames.GetCount(); pos++) 
			{
				int w,h;
				device.GetTextExtent(botNames[pos], &w, &h);
				device.DrawText(botNames[pos], imgLeft - 5 - w, imgTop + (pos * GetDotSize()) + yAdd);
				device.DrawRotatedText(botNames[pos], imgLeft + (pos * GetDotSize()), imgTop - 10, 90);
			}
		}

		device.DrawBitmap(matrixBmp, imgLeft, imgTop, true);
	}
	else // paint alt text
	{
		wxCoord w, h; /*, pw, ph*/
		device.GetTextExtent(altText, &w, &h);
/*		device.GetSize(&pw, &ph);
		int cw, ch;
		GetSize(&cw, &ch);
		wxMessageBox(wxString::Format(wxT("Drawing. \nDeviceSize: %d, %d\nMaxSize: %d, %d\nComponentSize: %d, %d\nTextSize: %d, %d"),
			pw, ph, device.MaxX(), device.MaxY(), cw, ch, w, h), wxT("Info")); */
		wxCoord left = (maxX - w) / 2;
		wxCoord top = (maxY - h) / 2;
		device.DrawText(altText, left, top);
	}
	device.EndDrawing();
}

void MatrixWindow::OnMotion(wxMouseEvent& event)
{
	int dot = GetDotSize();
	int players = data.chartSize;
	wxString text;

	wxCoord x,y;
	event.GetPosition(&x, &y);
	x -= imgLeft; y -= imgTop;
	if((x < 0) || (y < 0))
		text = wxT("");
	else 
	{
		x /= dot; y /= dot;
		if((x >= players) || (y >= players))
			text = wxT("");
		else
		{
			if(!wins || !losses || !ties)
				text = _("No data available.");
			else {
				int win = wins[y][x]; int loss = losses[y][x]; int tie = ties[y][x];
				if((win < 0) || (loss < 0) || (tie < 0))
					text = _("Result: No result.");
				else
					text = wxString::Format(_("Result: Wins: %d, Losses: %d, Ties: %d"), win, loss, tie);
			}
		}
	}	
	wxLogStatus(text);
}
