/*
 * detview.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include "detview.h"

#include <wx/imaglist.h>
#include <wx/statline.h>

#include "robotop.h"
#include "tourview.h"
#include "wxini.h"
#include "wxext.h"

#include "bitmaps/refresh.xpm"
#include "bitmaps/details.xpm"
#include "bitmaps/web.xpm"

#include "bitmaps/ok.xpm"
#include "bitmaps/old.xpm"
#include "bitmaps/no.xpm"

BEGIN_EVENT_TABLE(DetailsFrame, wxxMDIChildFrame)
  EVT_MENU(DetFrame_Redisplay, DetailsFrame::OnRedisplay)
  EVT_MENU(DetFrame_Details,   DetailsFrame::OnDetails)
  EVT_MENU(DetFrame_Browser,   DetailsFrame::OnBrowser)
  EVT_LIST_ITEM_SELECTED(DetFrame_List, DetailsFrame::OnShowRobotInStatus) 
  EVT_LIST_ITEM_ACTIVATED(DetFrame_List, DetailsFrame::OnDetailsList) 
  // close events
  EVT_MENU(wxID_CLOSE, DetailsFrame::OnCloseCommand)
  EVT_CLOSE(DetailsFrame::OnCloseWindow)
  // my competition events
  EVT_TOUR_UPDATE(DetailsFrame::OnTourUpdate)
  EVT_TOUR_CLOSE(DetailsFrame::OnTourClose)
END_EVENT_TABLE()

DetailsFrame::DetailsFrame(TournamentFrame* tour, wxMDIParentFrame *parent, const wxString& botname) 
	: tour(tour), wxxMDIChildFrame(parent, wxString::Format(_("Details - [%s]"), botname.c_str()), wxT("DetailsFrame"), wxDEFAULT_FRAME_STYLE), 
	  botname(botname), configData(tour->GetConfigData()), parent(parent), imageList(0)
{
	wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
	wxPanel* panel = new wxPanel(this); 

	wxToolBar* bar = new wxToolBar(panel, -1, wxDefaultPosition, wxDefaultSize, 
		wxNO_BORDER | wxTB_FLAT | wxTB_HORIZONTAL | wxTB_NODIVIDER);
	InitToolBar(bar);

	wxString myDetailsROX = GetMainTourFolder() + configData.tourFolder +
		wxFILE_SEP_PATH + wxT("results") + wxFILE_SEP_PATH + botname;
	myDetailsROX.Last() = 'x';
	detailsData = new DetailsReportData(myDetailsROX);
	CreateImageList();
	detailsList = new wxxReportCtrl(panel, DetFrame_List, detailsData, imageList, wxT("/Controls/Details/"));
	wxSizer* infoPanel = CreateInfoPanel();

	wxxAddCloseBar(panel, this, sizer, bar, (wxCommandEventFunction) &DetailsFrame::OnCloseCommand);
	sizer->Add(new wxStaticLine(panel, -1), 0, wxEXPAND, 0); 
	if(infoPanel)
		sizer->Add(infoPanel, 0, wxEXPAND | wxALL, 2);
	sizer->Add(detailsList, 1, wxEXPAND | wxALL, 2 );

	panel->SetSizer(sizer); 

	// size myself
	wxBoxSizer* mySizer = new wxBoxSizer(wxVERTICAL); 
	mySizer->Add(panel, 1, wxEXPAND | wxALL, 0); 
	SetSizer(mySizer);
	SetSizeHints(mySizer->GetMinSize()); // set size hints to honour mininum size

	SetMenuBar(CreateMenu());
	tour->AddTournamentListener(this);
}

DetailsFrame::~DetailsFrame()
{
	delete imageList;
}

// may return NULL if no info available
wxSizer* DetailsFrame::CreateInfoPanel()
{
	wxString myROY = GetMainTourFolder() + configData.tourFolder +
		wxFILE_SEP_PATH + wxT("results") + wxFILE_SEP_PATH + botname;
	myROY.Last() = 'y';
	wxxIniFile ini;
	if(!ini.Read(myROY))
		return 0;
	wxxIniSection& sec = ini[wxT("info")];
	
	wxBoxSizer* mainSz = new wxBoxSizer(wxVERTICAL);
	wxString progName = sec.Get(wxT("programname"));
#ifdef __WXMSW__ // which has a bug in label size reporting
	progName += wxT("\n");
#endif
	wxStaticText* nameText = new wxStaticText(this, -1,  progName);
	wxFont bigFont(16, wxSWISS, wxNORMAL, wxBOLD);
	nameText->SetFont(bigFont);
	mainSz->Add(nameText, 0, wxEXPAND | wxALL, 2);
	wxBoxSizer* subSz = new wxBoxSizer(wxHORIZONTAL);
	mainSz->Add(subSz, 0, wxEXPAND, 0);
	subSz->Add(new wxStaticText(this, -1, wxString::Format(
		_("by %s \nfrom %s"), sec.Get(wxT("author")).c_str(), sec.Get(wxT("country")).c_str())), 1, wxEXPAND | wxALL, 2);
	subSz->Add(new wxStaticText(this, -1, wxString::Format(
		_("Last updated at %s \n%s"), sec.Get(wxT("date")).c_str(), 
		(sec.GetBool(wxT("top"))? _("Is in charts") : _("Is not in charts")))), 1, wxEXPAND | wxALL, 2);

	return mainSz;
}


void DetailsFrame::CreateImageList()
{
    wxBitmap* bitmaps[dfiNumImages];

    bitmaps[dfiGood] = new wxBitmap(ok_xpm);
    bitmaps[dfiBad] = new wxBitmap(no_xpm);
	bitmaps[dfiOld] = new wxBitmap(old_xpm);
	int w = bitmaps[dfiGood]->GetWidth(), h = bitmaps[dfiGood]->GetHeight();

	imageList = new wxImageList(w,h);
	for(int i = 0; i < dfiNumImages; i++) 
	{
		imageList->Add(*(bitmaps[i]));
		delete bitmaps[i];
	}
}

wxMenuBar* DetailsFrame::CreateMenu()
{
    // create a menu bar
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(RoboTop_New,  _("&New Competition...\tCtrl-N"), _("Create a new competition"));
    menuFile->Append(RoboTop_Open, _("&Open Competition...\tCtrl-O"), _("Open a competition"));
	menuFile->Append(wxID_CLOSE,   _("&Close View\tCtrl-F4"), _("Close this view"));
    menuFile->AppendSeparator();
    menuFile->Append(RoboTop_Quit, _("E&xit\tAlt-X"), _("Quit this program"));

	wxMenu* menuDetails = new wxMenu;
	menuDetails->Append(DetFrame_Details, _("View &Details\tCtrl-D"), _("View the selected robot's detailed results"));
	menuDetails->Append(DetFrame_Browser, _("View in &Browser\tCtrl-B"), _("View this robot's details in your browser"));

	wxMenu* menuExtras = new wxMenu;
	menuExtras->Append(RoboTop_Settings, _("&Settings..."), _("Configure RoboTop settings"));
#ifdef __WXMSW__
	menuExtras->Append(RoboTop_Fullscreen, _("&Fullscreen\tF11"), _("Show RoboTop fullscreen"), true); // checkable
#endif

    // the "About" item should be in the help menu
    wxMenu *helpMenu = new wxMenu;
    helpMenu->Append(RoboTop_Help, _("Using RoboTop\tF1"), _("Show help for RoboTop"));
    helpMenu->Append(RoboTop_About, _("&About..."), _("Show about dialog"));

    // now append the freshly created menu to the menu bar...
    wxMenuBar *menuBar = new wxMenuBar();
    menuBar->Append(menuFile, _("&File"));
    menuBar->Append(menuDetails, _("&Details"));
	menuBar->Append(menuExtras, _("&Extras"));
    menuBar->Append(helpMenu, _("&Help"));

	return menuBar;
}

void DetailsFrame::InitToolBar(wxToolBar* toolBar)
{
	const int numBitmaps = 3;
    wxBitmap* bitmaps[numBitmaps];

    bitmaps[0] = new wxBitmap( refresh_xpm );
    bitmaps[1] = new wxBitmap( details_xpm );
    bitmaps[2] = new wxBitmap( web_xpm );
	
	wxSize bitmapSize(bitmaps[0]->GetWidth(), bitmaps[0]->GetHeight());
	toolBar->SetToolBitmapSize(bitmapSize);

    toolBar->AddTool( DetFrame_Details, *(bitmaps[1]), 
		_("View Details"), _("View detailed results for the selected robot"));
    toolBar->AddTool( DetFrame_Browser, *(bitmaps[2]), 
		_("View in Browser"), _("View detailed results for this robot in your web browser"));
    toolBar->AddTool( DetFrame_Redisplay, *(bitmaps[0]), 
		_("Update Display"), _("Update the display with the newest simulation results"));

    toolBar->Realize();
    for (int i = 0; i < numBitmaps; i++)
        delete bitmaps[i];
}

void DetailsFrame::Redisplay()
{
	detailsData->Reload();
}

void DetailsFrame::OnShowRobotInStatus(wxListEvent& event)
{
	long sel = detailsList->GetSelectedRow();
	if(sel < 0) {
		return;
	}
	wxString botname = detailsData->GetRowName(sel);
	wxLogStatus(_("Robot file name: %s"), botname.c_str());
}

void DetailsFrame::OnRedisplay(wxCommandEvent& event)
{
    Redisplay();
}

void DetailsFrame::OnBrowser(wxCommandEvent& event)
{
	if(tour && tour->NeedsMakehtml())
		tour->RunMakeHtml(true, true); 

	wxString myHtml = GetMainTourFolder() + configData.tourFolder +
		wxFILE_SEP_PATH + wxT("html") + wxFILE_SEP_PATH + botname + wxT(".html");
	if(!wxxShowInBrowser(myHtml))
	{
#ifdef __WXMSW__
		const wxChar* msg = _("Couldn't start your default browser.");
#else
		const wxChar* msg = _("Couldn't start your web browser. Please check the configuration at Extras -> Settings...");
#endif
		wxLogError(msg);
	}
}

void DetailsFrame::OnDetails(wxCommandEvent& event)
{
	long sel = detailsList->GetSelectedRow();
	if(sel < 0) {
		wxMessageBox(_("Select a robot first."), _("Which detailed results?"), wxICON_INFORMATION, this);
		return;
	}
	wxString botname = detailsData->GetRowName(sel);
	DetailsFrame* df = new DetailsFrame(tour, parent, botname);
	df->Show(); 
}

void DetailsFrame::OnCloseCommand(wxCommandEvent& event)
{
	wxCloseEvent pseudoEvent;
	pseudoEvent.SetCanVeto(true);
	OnCloseWindow(pseudoEvent);
}

void DetailsFrame::OnCloseWindow(wxCloseEvent& event)
{
	if(tour)
		tour->RemoveTournamentListener(this);
	Destroy();
}

void DetailsFrame::OnTourUpdate(TournamentEvent& event)
{
	Redisplay();
}

void DetailsFrame::OnTourClose(TournamentEvent& event)
{
	// there's no tour any more
	tour = 0;
	// cascaded closing?
	if(event.IsForce())
		Destroy();
}


static wxxIniColumnInfo* detailsColInfo[] = {
	new wxxIniColumnInfo(wxTRANSLATE("Name"), wxT("yprogramname"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Last Update"), wxT("ydate"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Author"), wxT("yauthor"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Country"), wxT("ycountry"), wxREPORT_AUTOSIZE),
	new wxxIniColumnInfo(wxTRANSLATE("Wins"), wxT("wins"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Losses"), wxT("looses"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
	new wxxIniColumnInfo(wxTRANSLATE("Ties"), wxT("ties"), 40, wxREPORT_RIGHT, wxREPORT_SORTNUMERIC),
};
const static int detailsColCount = 7;
static bool detailsColInit = false; 

DetailsReportData::DetailsReportData(const wxString& detailedIni)
	: wxxIniReportData(detailedIni, detailsColCount, InitColumnInfo(detailsColInfo, detailsColCount, detailsColInit), 
	                  wxREPORT_DELNOTHING)
{
	ReadYFiles();
}

int DetailsReportData::GetImage(long row)
{
	wxxIniSection* sec = sectionCache[row];
	bool top = sec->GetBool(wxT("ytop"));
	long yid = sec->GetLong(wxT("yid"));
	long id = sec->GetLong(wxT("id"));

	if(yid < id) {
		wxLogError(_("Robot played against newer version (%ld) of %s than in charts (%ld)?"),
			id, sec->GetName().c_str(), yid);
		return -1;
	}
	if(yid > id)
		return dfiOld;
	if(top)
		return dfiGood;
	else
		return dfiBad;
}

bool DetailsReportData::DoSectionCache()
{
	ReadYFiles();

	return wxxIniReportData::DoSectionCache();
}

void DetailsReportData::ReadYFiles()
{
	wxxIniSection* sec = 0;
	ini->First();
	while((sec = ini->Next()) != 0)
	{
		wxString yname = iniFile.BeforeLast(wxFILE_SEP_PATH) + wxFILE_SEP_PATH;
		yname += sec->GetName();
		yname.Last() = 'y';
		wxxIniFile yini;
		if(!yini.Read(yname))
			continue;
		wxxIniSection& ysec = yini[wxT("info")];
		ysec.First();
		wxxIniOption* yopt = 0;
		while((yopt = ysec.Next()) != 0)
			sec->Set(wxT("y") + yopt->name, yopt->value);
	}
}
