/* ParameterList.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.params;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.List;
import java.util.ArrayList;

import org.grinvin.io.IOFormatException;

import org.jdom.Element;

/**
 * Encapsulates a list of parameters, i.e., {@link ParameterInfo}-objects
 * and provides methods to create a corresponding list of parameter values.
 */
public class ParameterList {
    
    /**
     * Construct an empty parameter list.
     */
    public ParameterList() {
        parameters = new ArrayList<ParameterInfo> ();
    }
    
    //
    private List<ParameterInfo> parameters;
    
    /**
     * Add a parameter at the end of this list.
     */
    public void addParameter(ParameterInfo parameter) {
        parameters.add(parameter);
    }
    
    /**
     * Add parameters to the end of this list, obtained from descriptions
     * in the given JDOM element.
     */
    public void fromElement(Element root) throws IOFormatException {
        for (Object child: root.getChildren("parameter")) {
            ParameterInfo info = new ParameterInfo();
            info.fromElement((Element)child);
            addParameter(info);
        }
    }
    
    /**
     * Return the total number of parameters.
     */
    public int getCount() {
        return parameters.size();
    }
    
    /**
     * Return information on the parameter with the given index.
     */
    public ParameterInfo getParameter(int index) {
        return parameters.get(index);
    }
    
    /**
     * Return the index of the parameter with the given name or -1 if no such
     * parameter exists.
     */
    private int indexOf(String name) {
        for (int i=0; i < parameters.size(); i++)
            if (parameters.get(i).getName().equals(name))
                return i;
        return -1;
    }
    
    
    /**
     * Parse a single name/value-pair.
     */
    private void parsePair(String pair, Object[] result) {
        if (pair.length() == 0)
            throw new IllegalArgumentException("Invalid query string syntax");
        try {
            int pos = pair.indexOf('=');
            if (pos < 0 || pair.indexOf(pos+1, '=') >= 0)
                throw new IllegalArgumentException("Invalid query string syntax");
            String key = URLDecoder.decode(pair.substring(0, pos), "UTF-8");
            int index = indexOf(key);
            if (index < 0 || index >= result.length)
                throw new IllegalArgumentException("Unknown parameter: " + key);
            else if (result[index] != null)
                throw new IllegalArgumentException("Same parameter occurs twice: " + key);
            String str = URLDecoder.decode(pair.substring(pos+1), "UTF-8");
            result[index] = parameters.get(index).getEditor().fromText(str);
        } catch (UnsupportedEncodingException ex) {
            assert false : "Encoding UTF-8 unexpectedly not known";
        }
    }
    
    
    /**
     * Convert a query-string to a list objects corresponding to this
     * parameter list.
     * @throws IllegalArgumentException when the query string contains an
     * unknown or duplicate parameter name or when not all parameters in this
     * list are given a value.
     */
    public Object[] parseQueryString(String str) {
        int len = parameters.size();
        Object[] values = new Object[len];
        if (str.length() > 0) {
            int oldPos = 0;
            int newPos = str.indexOf('&', oldPos);
            while (newPos >= 0) {
                parsePair(str.substring(oldPos, newPos), values);
                oldPos = newPos + 1;
                newPos = str.indexOf('&', oldPos);
            }
            parsePair(str.substring(oldPos), values);
        }
        for (Object value : values)
            if (value == null)
                throw new IllegalArgumentException("Missing arguments in query string");
        
        return values;
    }
    
    /**
     * Convert the given list of parameter values to their string representations.
     */
    public String[] toText(Object[] values) {
        String[] strings = new String[values.length];
        for (int i=0; i < values.length; i++)
            strings[i] = parameters.get(i).getEditor().toText(values[i]);
        return strings;
    }
    
}
