/* GraphCellListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import be.ugent.caagt.swirl.lists.CellList;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JToolTip;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;

import org.grinvin.gui.dnd.CommonGraphListTransferHandler;
import org.grinvin.worksheet.HasSelectableGraphList;
import org.grinvin.worksheet.actions.GraphListElementPopupMenu;
import org.grinvin.gui.MultilineToolTip;
import org.grinvin.gui.WatermarkPainter;
import org.grinvin.gui.components.render.GraphListElementCellRenderer;
import org.grinvin.list.graphs.DefaultGraphList;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.gui.windows.GraphPropertiesWindow;

/**
 *
 */
public class GraphCellListComponent extends CellList implements HasSelectableGraphList, MouseListener {
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.worksheet.resources";

    //
    private final GraphListElementPopupMenu popupMenu;
    
    
    // shared watermark painter (also used by GraphListComponent)
    static final WatermarkPainter WATERMARK_PAINTER = new WatermarkPainter(
            20,
            ResourceBundle.getBundle(BUNDLE_NAME).getString("GraphList.emptytext"),
            new Color(237, 245, 237)
            );
    
    //
    private GraphListModel model;
    
    /**
     * Creates a new instance which creates its own selection model.
     */
    public GraphCellListComponent(GraphListModel model, int cellWidth, int cellHeight, int numberOfColumns) {
        this(model, new DefaultListSelectionModel(), cellWidth, cellHeight,
                numberOfColumns);
    }
    
    
    /**
     * Creates a new instance of GraphCellListComponent
     */
    public GraphCellListComponent(GraphListModel model, ListSelectionModel selectionModel, int cellWidth, int cellHeight, int numberOfColumns) {
        super(model, cellWidth, cellHeight, numberOfColumns);
        this.model = model;
        
        setSelectionModel(selectionModel);
        setDragEnabled(true);
        setTransferHandler(CommonGraphListTransferHandler.getInstance());
        setCellRenderer(GraphListElementCellRenderer.getInstance());
        setOpaque(false); // we paint our own background
        
        popupMenu = new GraphListElementPopupMenu(model, getSelectionModel());
        
        // Keyboard interaction 	 
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements"); 	 
        getActionMap().put("deleteSelectedElements", new AbstractAction() { 	 
            public void actionPerformed(ActionEvent ev) { 	 
                deleteSelectedElements(); 	 
            } 	 
        });
        
        // mouse interaction
        addMouseListener(this);
        
        // paint the watermark?
        this.paintWatermark = true;
    }
    
    // paint the watermark?
    private boolean paintWatermark;
    
    /**
     * Configure whether to paint the watermark or not. If no watermark
     * is painted, the background is still filled.
     */
    public void setPaintWatermark (boolean paintWatermark) {
        this.paintWatermark = paintWatermark;
    }
    
    
    /**
     * Adds a watermark to the list.
     */
    @Override
    protected void paintComponent(Graphics g) {
        g.setColor(getBackground());
        g.fillRect(0, 0, getWidth(), getHeight());
        if (paintWatermark)
            WATERMARK_PAINTER.paint(this, g);
        super.paintComponent(g);
    }
    
    
    
    //
    public GraphList getGraphList() {
        return model;
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // gather all elements to be deleted
        GraphList list = new DefaultGraphList();
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                list.add(model.get(i));
        
        // and remove them
        model.removeAll(list);
        //TODO: make sure changed messages are sent
    }
    
    @Override
    public JToolTip createToolTip() {
        return new MultilineToolTip(150);
        //TODO: don't hardcode width
    }
    
    //
    private GraphListElement getElementAtPoint(Point point) {
        
        int row = locationToIndex(point);
        if (row < 0 || !getCellBounds(row,row).contains(point))
            return null;
        else
            return model.get(row);
    }
    
    // TODO: even further reduce the code this has in common with GraphListComponent
    // only needs getElementAtPoint to be abstracted out (in an anonymous class?)
    
    private GraphListElement[] getSelectedElements() {
        Object[] elements = getSelectedValues();
        GraphListElement[] newElements = new GraphListElement[elements.length];
        for(int i=0;i<elements.length;i++)
            newElements[i] = (GraphListElement)elements[i];
        return newElements;
    }
    
    //
    public void mouseClicked(MouseEvent e) {
        if (isEnabled() && e.getClickCount() == 2 && !e.isPopupTrigger()) {
            GraphListElement gle = getElementAtPoint(e.getPoint());
            if (gle != null)
                new GraphPropertiesWindow(gle).setVisible(true);
        }
    }
    
    //
    public void mousePressed(MouseEvent e) {
        //TODO: remove dependency on getSelectionModel()
        if (isEnabled() && e.isPopupTrigger()) {
            int index = locationToIndex(e.getPoint());
            if(!getSelectionModel().isSelectedIndex(index))
                getSelectionModel().setSelectionInterval(index, index);
            popupMenu.show(getSelectedElements(), this,e.getX(),e.getY());
        }
    }
    
    //
    public void mouseReleased(MouseEvent e) {
        //TODO: remove dependency on getSelectionModel()
        if (isEnabled() && e.isPopupTrigger()) {
            int index = locationToIndex(e.getPoint());
            if(!getSelectionModel().isSelectedIndex(index))
                getSelectionModel().setSelectionInterval(index, index);
            popupMenu.show(getSelectedElements(), this,e.getX(),e.getY());
        }
    }
    
    //
    public void mouseEntered(MouseEvent e) {
        // intentionally empty
    }
    
    //
    public void mouseExited(MouseEvent e) {
        // intentionally empty
    }
    
}
