/*
 * Copyright (c) 2000-2005 CyberFOX Software, Inc. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Library General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the
 *  Free Software Foundation, Inc.
 *  59 Temple Place
 *  Suite 330
 *  Boston, MA 02111-1307
 *  USA
 */

import java.util.*;
import java.text.SimpleDateFormat;
import javax.swing.Icon;
import javax.swing.ImageIcon;

public class auctionTableModel extends BaseTransformation {
  private static final String neverBid = "--";
  private List dispList;
  private Date futureForever = new Date(Long.MAX_VALUE);
  private XMLElement cvt = new XMLElement();

  public int getRowCount() { return dispList.size(); }
  public int getColumnCount() { return TableColumnController.COLUMN_COUNT; }
  public String getColumnName(int aColumn) { return TableColumnController.getInstance().getColumnName(aColumn); }

  public void delete(int row) {
    dispList.remove(row);
  }

  public int insert(Object o) {
    dispList.add(o);
    return dispList.size()-1;
  }

  public Object find(Comparison c) {
    for(int i=0; i<dispList.size(); i++) {
      if(c.match(dispList.get(i))) return dispList.get(i);
    }

    return null;
  }

  public int getColumnNumber(String colName) {
    return TableColumnController.getInstance().getColumnNumber(colName);
  }

  //  All columns return strings...
  public Class getColumnClass(int aColumn) { if(aColumn != 5) return String.class; return Icon.class; }

  //  Except when we want to sort them...
  public Class getSortByColumnClass(int i) {
    //  Status is the only one where the type is very different than the dummy data.
    if(i==TableColumnController.STATUS ||
       i==TableColumnController.SELLER_FEEDBACK ||
       i==TableColumnController.BIDCOUNT ||
       i==TableColumnController.SELLER_POSITIVE_FEEDBACK) return Integer.class;

    if(i==-1) return String.class;

    Object o = getDummyValueAtColumn(i);
    return o.getClass();
  }

  private static final ImageIcon dummyIcon = new ImageIcon(auctionTableModel.class.getResource("icons/white_ball.gif"));
  private static final ImageIcon greenIcon = new ImageIcon(auctionTableModel.class.getResource("icons/green_ball.gif"));
  //  private final static ImageIcon blueIcon = new ImageIcon(auctionTableModel.class.getResource("icons/blue_ball.gif"));
  private static final ImageIcon binIcon = new ImageIcon(auctionTableModel.class.getResource("icons/bin_item.gif"));
  private static final ImageIcon resIcon = new ImageIcon(auctionTableModel.class.getResource("icons/unmet_reserve.gif"));
  private static final ImageIcon resMetIcon = new ImageIcon(auctionTableModel.class.getResource("icons/met_reserve.gif"));
  private static final ImageIcon imageIcon = new ImageIcon(auctionTableModel.class.getResource("icons/camera.gif"));
  private static final ImageIcon commentIcon=new ImageIcon(auctionTableModel.class.getResource("icons/note3.gif"));
  private static final ImageIcon winningIcon=new ImageIcon(auctionTableModel.class.getResource("icons/winning.gif"));
  private static final ImageIcon invalidIcon=new ImageIcon(auctionTableModel.class.getResource("icons/invalid.gif"));
  private static final ImageIcon paypalIcon =new ImageIcon(auctionTableModel.class.getResource("icons/paypal16x16.gif"));
  //  private final static ImageIcon boughtIcon = new ImageIcon(auctionTableModel.class.getResource("icons/blue_check_ball.gif"));
  //  private final static ImageIcon soldIcon = new ImageIcon(auctionTableModel.class.getResource("icons/green_check_ball.gif"));

  private Object getDummyValueAtColumn(int j) {
    switch(j) {
      // For the 'get the whole object', all we can safely do is
      // return null.
      case -1:
        return null;
      case TableColumnController.CUR_BID:
      case TableColumnController.SNIPE_OR_MAX:
      case TableColumnController.SHIPPING_INSURANCE:
      case TableColumnController.MAX:
      case TableColumnController.SNIPE:
      case TableColumnController.FIXED_PRICE:
      case TableColumnController.JUSTPRICE:
      case TableColumnController.CUR_TOTAL:
        return Currency.NoValue();
      case TableColumnController.TIME_LEFT:
      case TableColumnController.END_DATE:
        return futureForever;
      case TableColumnController.STATUS:
        return dummyIcon;
      case TableColumnController.ID:
      case TableColumnController.TITLE:
      case TableColumnController.SELLER:
      case TableColumnController.COMMENT:
      case TableColumnController.BIDDER:
      case TableColumnController.SELLER_POSITIVE_FEEDBACK:
      case TableColumnController.ITEM_LOCATION:
      case TableColumnController.BIDCOUNT:
      case TableColumnController.SNIPE_TOTAL:
      default:
        return("");
    }
  }

  private int buildEntryFlags(AuctionEntry ae) {
    //  This shouldn't happen, but to be safe...
    if(ae == null) return 1;
    return ae.getFlags();
  }

  private ImageIcon getEntryIcon(AuctionEntry ae) {
    ImageIcon ret_icon = null;
    if(ae.isFixed()) ret_icon = null;

    if (ae.getHighBidder() != null) {
      if (ae.isHighBidder()) {
        ret_icon = winningIcon;
      } else {
        if (ae.isSeller() && ae.getNumBidders() > 0 &&
            (!ae.isReserve() || ae.isReserveMet())) {
          ret_icon = greenIcon;
        }
      }
    }
    if(!ae.getBuyNow().isNull()) {
      ret_icon = IconFactory.getCombination(ret_icon, binIcon);
    }
    if(ae.isReserve()) {
      if(ae.isReserveMet()) {
        ret_icon = IconFactory.getCombination(ret_icon, resMetIcon);
      } else {
        ret_icon = IconFactory.getCombination(ret_icon, resIcon);
      }
    }
    if(ae.getThumbnail() != null) {
      ret_icon = IconFactory.getCombination(ret_icon, imageIcon);
    }
    if(ae.getComment() != null) {
      ret_icon = IconFactory.getCombination(ret_icon, commentIcon);
    }
    if(ae.isInvalid()) {
      ret_icon = IconFactory.getCombination(ret_icon, invalidIcon);
    }
    if(ae.hasPaypal()) {
      ret_icon = IconFactory.getCombination(ret_icon, paypalIcon);
    }
    return ret_icon;
  }

  Integer Zero = new Integer(0);

  public Object getSortByValueAt(int i, int j) {
    try {
      AuctionEntry aEntry = (AuctionEntry) dispList.get(i);
      switch(j) {
        case -1: return aEntry;
        case TableColumnController.ID: return aEntry.getIdentifier();
        case TableColumnController.CUR_BID: return aEntry.getUSCurBid();
        case TableColumnController.SNIPE_OR_MAX:
          return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), getMaxOrSnipe(aEntry));
        case TableColumnController.TIME_LEFT: return aEntry.getEndDate();
        case TableColumnController.TITLE: return aEntry.getTitle();
        case TableColumnController.STATUS: return new Integer(buildEntryFlags(aEntry));
        case TableColumnController.SELLER: return aEntry.getSeller();
        case TableColumnController.FIXED_PRICE:
          return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), aEntry.getBuyNow());
        case TableColumnController.SHIPPING_INSURANCE:
          Currency si = (!aEntry.getShipping().isNull())?aEntry.getShippingWithInsurance():Currency.NoValue();
          //  This is crack.  I'm insane to even think about doing this, but it works...
          return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), si);
        case TableColumnController.BIDDER: return aEntry.getHighBidder();
        case TableColumnController.MAX:
          Currency bid = aEntry.isBidOn()?aEntry.getBid():Currency.NoValue();
          return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), bid);
        case TableColumnController.SNIPE:
          Currency snipe = aEntry.isSniped()?aEntry.getSnipeBid():Currency.NoValue();
          return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), snipe);
        case TableColumnController.COMMENT:String s = aEntry.getComment(); return (s==null?"":s);
        case TableColumnController.END_DATE:return aEntry.getEndDate();
        case TableColumnController.SELLER_FEEDBACK: if(aEntry.getFeedbackScore()==0) return Zero; else return new Integer(aEntry.getFeedbackScore());
        case TableColumnController.ITEM_LOCATION: return aEntry.getItemLocation();
        case TableColumnController.BIDCOUNT: return new Integer(aEntry.getNumBidders());
        case TableColumnController.JUSTPRICE: return aEntry.getUSCurBid();
        case TableColumnController.SELLER_POSITIVE_FEEDBACK: try {
          int result = (int)(Double.parseDouble(aEntry.getPostiveFeedbackPercentage())*10.0);
          return new Integer(result);
        } catch(Exception e) {
          return Zero;
        }
        case TableColumnController.CUR_TOTAL:
          Currency shipping = aEntry.getShippingWithInsurance();
          if (shipping.getCurrencyType() == Currency.NONE) {
            return shipping; // shipping not set so cannot add up values
          }

          Currency shippingUSD = Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), aEntry.getShippingWithInsurance());
          try {
            return aEntry.getUSCurBid().add(shippingUSD);
          } catch (Currency.CurrencyTypeException e) {
            ErrorManagement.handleException("Threw a bad currency exception, which should be unlikely.", e); //$NON-NLS-1$
            return Currency.NoValue();
          }
        case TableColumnController.SNIPE_TOTAL: {
          Currency shipping2 = aEntry.getShippingWithInsurance();
          if (shipping2.getCurrencyType() == Currency.NONE) {
            return shipping2; // shipping not set so cannot add up values
          }

          Currency shippingUSD2 = Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), aEntry.getShippingWithInsurance());
          try {
            return Currency.convertToUSD(aEntry.getUSCurBid(), aEntry.getCurBid(), aEntry.getSnipeBid()).add(shippingUSD2);
          } catch (Currency.CurrencyTypeException e) {
            ErrorManagement.handleException("Currency addition or conversion threw a bad currency exception, which should be unlikely.", e); //$NON-NLS-1$
            return Currency.NoValue();
          }
        }

        //  This should never happen, but to be safe...
        default: return "";
      }
    } catch(ArrayIndexOutOfBoundsException ignored) {
      return getDummyValueAtColumn(j);
    }
  }

  private Currency getMaxOrSnipe(AuctionEntry aEntry) {
    if(aEntry.isSniped()) {
      return aEntry.getSnipeBid();
    }
    if(aEntry.isBidOn()) {
      return aEntry.getBid();
    }
    if(aEntry.snipeCancelled() && aEntry.isEnded()) {
      return aEntry.getCancelledSnipe();
    }
    return Currency.NoValue();
  }

  private String formatSnipeAndBid(AuctionEntry aEntry) {
    String errorNote = "";
    if(aEntry.getErrorPage() != null) errorNote="*";

    if(aEntry.isSniped()) {
      return formatSnipe(aEntry, errorNote);
    }
    if(aEntry.isBidOn()) {
      return formatBid(aEntry, errorNote);
    }
    if(aEntry.snipeCancelled() && aEntry.isEnded()) {
      return errorNote + '(' + aEntry.getCancelledSnipe() + ')';
    }
    return neverBid;
  }

  private String formatBid(AuctionEntry aEntry, String errorNote) {
    String bidCount = "";
    if(aEntry.getBidQuantity() != 1) bidCount = " x " + aEntry.getBidQuantity();
    return errorNote + aEntry.getBid().toString() + bidCount;
  }

  private String formatSnipe(AuctionEntry aEntry, String errorNote) {
    String snipeCount = "";
    if (aEntry.getSnipeQuantity() != 1) {
      snipeCount = " x " + aEntry.getSnipeQuantity();
    }

    if(aEntry.isMultiSniped()) {
      if(aEntry.isSnipeValid() || aEntry.isDutch()) {
        return errorNote + "Multi: " + aEntry.getSnipeBid() + snipeCount;
      } else {
        return errorNote + "Multi: (" + aEntry.getSnipeBid() + snipeCount + ')';
      }
    } else {
      if(aEntry.isSnipeValid() || aEntry.isDutch()) {
        return errorNote + aEntry.getSnipeBid().toString() + snipeCount;
      } else {
        return errorNote + '(' + aEntry.getSnipeBid() + snipeCount + ')';
      }
    }
  }

  private String formatTotalSnipe(AuctionEntry aEntry, String errorNote) {
    if(!aEntry.isSniped()) return "--";
    Currency shipping = aEntry.getShippingWithInsurance();
    if (shipping.getCurrencyType() == Currency.NONE || aEntry.getSnipeBid().getCurrencyType() == Currency.NONE) {
      return "--"; // shipping not set so cannot add up values
    }

    Currency totalSnipe;

    try {
      totalSnipe = aEntry.getSnipeBid().add(shipping);
    } catch (Currency.CurrencyTypeException e) {
      ErrorManagement.handleException("Currency addition threw a bad currency exception, which should be very difficult to cause to happen.", e); //$NON-NLS-1$
      return "--";
    }

    String snipeCount = "";
    if (aEntry.getSnipeQuantity() != 1) {
      snipeCount = " x " + aEntry.getSnipeQuantity();
    }

    if (aEntry.isMultiSniped()) {
      if (aEntry.isSnipeValid() || aEntry.isDutch()) {
        return errorNote + "Multi: " + totalSnipe + snipeCount;
      } else {
        return errorNote + "Multi: (" + totalSnipe + snipeCount + ')';
      }
    } else {
      if (aEntry.isSnipeValid() || aEntry.isDutch()) {
        return errorNote + totalSnipe.toString() + snipeCount;
      } else {
        return errorNote + '(' + totalSnipe + snipeCount + ')';
      }
    }
  }

  public Object getValueAt(int rowIndex, int columnIndex) {
    try {
      AuctionEntry aEntry = (AuctionEntry) dispList.get(rowIndex);
      String errorNote = aEntry.getErrorPage()==null?"":"*";
      switch(columnIndex) {
        case -1: return aEntry;
        case TableColumnController.ID: return aEntry.getIdentifier();
        case TableColumnController.CUR_BID:
          if(aEntry.isFixed()) {
            return aEntry.getCurBid() + " (FP)";
          } else {
            if(aEntry.isDutch()) {
              return aEntry.getCurBid() + " x " + Integer.toString(aEntry.getQuantity());
            } else {
              return aEntry.getCurBid() + " (" + Integer.toString(aEntry.getNumBidders()) + ')';
            }
          }
        case TableColumnController.SNIPE_OR_MAX: return formatSnipeAndBid(aEntry);
        case TableColumnController.MAX: return aEntry.isBidOn()?formatBid(aEntry, errorNote):neverBid;
        case TableColumnController.SNIPE:
          if (aEntry.isSniped()) {
            return formatSnipe(aEntry, errorNote);
          }
          if(aEntry.snipeCancelled() && aEntry.isEnded()) {
            return errorNote + '(' + aEntry.getCancelledSnipe() + ')';
          }

          return neverBid;
        case TableColumnController.TIME_LEFT: {
          if (aEntry.getEndDate() == null || aEntry.getEndDate().equals(Constants.FAR_FUTURE)) return "N/A";
          String endTime = aEntry.getTimeLeft();
          if(endTime.equals(AuctionEntry.endedAuction)) {
            SimpleDateFormat fmt = new SimpleDateFormat("dd-MMM-yy HH:mm:ss zzz");
            endTime = fmt.format(aEntry.getEndDate());
            if(!aEntry.isEnded()) {
              endTime = "<html><body color=\"red\">" + endTime + "</body></html>";
            }
          }
          return endTime;
        }
        case TableColumnController.END_DATE: {
          if (aEntry.getEndDate() == null || aEntry.getEndDate().equals(Constants.FAR_FUTURE)) return "N/A";
          String endTime;
          SimpleDateFormat fmt = new SimpleDateFormat("dd-MMM-yy HH:mm:ss zzz");
          endTime = fmt.format(aEntry.getEndDate());
          return endTime;
        }
        case TableColumnController.TITLE: return cvt.decodeString(aEntry.getTitle(), 0);
        case TableColumnController.STATUS: return getEntryIcon(aEntry);
        case TableColumnController.SELLER: return aEntry.getSeller();
        case TableColumnController.COMMENT:
          String comment = aEntry.getComment();
          return(comment==null?"":comment);
        case TableColumnController.BIDDER:
          if(aEntry.getHighBidder() != null) {
            return aEntry.getHighBidder();
          }
          return "--";
        case TableColumnController.FIXED_PRICE:
          Currency bin = aEntry.getBuyNow();
          if(bin.isNull()) return "--";
          return bin;
        case TableColumnController.SHIPPING_INSURANCE:
          Currency ship = aEntry.getShippingWithInsurance();
          if(ship.isNull()) return "--";
          return ship;
        case TableColumnController.ITEM_LOCATION:
          return aEntry.getItemLocation();
        case TableColumnController.BIDCOUNT:
          if(aEntry.getNumBidders() < 0) return "(FP)";
          return Integer.toString(aEntry.getNumBidders());
        case TableColumnController.JUSTPRICE:
          return aEntry.getCurBid();
        case TableColumnController.SELLER_FEEDBACK:
          return new Integer(aEntry.getFeedbackScore());
        case TableColumnController.SELLER_POSITIVE_FEEDBACK:
          String fbp = aEntry.getPostiveFeedbackPercentage();
          return (fbp == null || fbp.length() == 0)?"--":(fbp+ '%');
        case TableColumnController.CUR_TOTAL:
        Currency shipping = aEntry.getShippingWithInsurance();
          if(shipping.getCurrencyType() == Currency.NONE) {
            return "--"; // shipping not set so cannot add up values
          }
          try {
            return aEntry.getCurBid().add(shipping);
          } catch (Currency.CurrencyTypeException e) {
            ErrorManagement.handleException("Currency addition threw a bad currency exception, which is odd...", e); //$NON-NLS-1$
          }
          return "--";
        case TableColumnController.SNIPE_TOTAL:
          return formatTotalSnipe(aEntry, errorNote);
        default:
          return "";
      }
    } catch(ArrayIndexOutOfBoundsException aioobe) {
      return(getDummyValueAtColumn(columnIndex));
    }
  }

  public auctionTableModel() {
    dispList = new Vector();
  }

  public int compare(int row1, int row2, ColumnStateList columnStateList) {
	  int result = 0;
	  
	  for(ListIterator li = columnStateList.listIterator(); li.hasNext();) {
		  ColumnState cs = (ColumnState)li.next();
		  
		  Class type = getSortByColumnClass(cs.getColumn());

		  Object o1 = getSortByValueAt(row1, cs.getColumn());
		  Object o2 = getSortByValueAt(row2, cs.getColumn());
		  
		  result = compareByClass(o1, o2, type) * cs.getSort();
		  
		  // The nth column is different
		  if(result != 0) {
			  break;
		  }
	  }
	  
	  return result;
  }

  public boolean isCellEditable(int row, int column) {
    return false;
  }
}