/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "projectdialogs.h"

#include <QLayout>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>

#include "data/rdb/tableiterators.h"
#include "data/datamanager.h"
#include "data/projectsbatch.h"
#include "data/rowhelpers.h"
#include "iconloader.h"

using namespace WebIssues;

ProjectsDialog::ProjectsDialog( QWidget* parent ) : CommandDialog( parent )
{
}

ProjectsDialog::~ProjectsDialog()
{
}

bool ProjectsDialog::checkProjectName( const QString& name )
{
    if ( name.isEmpty() ) {
        showWarning( tr( "Project name cannot be empty" ) );
        return false;
    }

    RDB::TableConstIterator<ProjectRow> it( dataManager->projects() );
    while ( it.next() ) {
        if ( it.get()->name() == name ) {
            showWarning( tr( "A project with this name already exists" ) );
            return false;
        }
    }

    return true;
}

bool ProjectsDialog::checkFolderName( int projectId, const QString& name )
{
    if ( name.isEmpty() ) {
        showWarning( tr( "Folder name cannot be empty" ) );
        return false;
    }

    RDB::ForeignConstIterator<FolderRow> it( dataManager->folders()->parentIndex(), projectId );
    while ( it.next() ) {
        if ( it.get()->name() == name ) {
            showWarning( tr( "A folder with this name already exists" ) );
            return false;
        }
    }

    return true;
}

AddProjectDialog::AddProjectDialog( QWidget* parent ) : ProjectsDialog( parent )
{
    setWindowTitle( tr( "Add Project" ) );
    setPrompt( tr( "Create a new project:" ) );
    setPromptPixmap( IconLoader::pixmap( "project-new", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( tr( "New Project" ), this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

AddProjectDialog::~AddProjectDialog()
{
}

void AddProjectDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( !checkProjectName( name ) )
        return;

    ProjectsBatch* batch = new ProjectsBatch();
    batch->addProject( name );

    executeBatch( batch );
}

RenameProjectDialog::RenameProjectDialog( int projectId, QWidget* parent ) : ProjectsDialog( parent ),
    m_projectId( projectId )
{
    const ProjectRow* project = dataManager->projects()->find( projectId );
    m_oldName = project ? project->name() : QString();

    setWindowTitle( tr( "Rename Project" ) );
    setPrompt( tr( "Enter the new name of project <b>%1</b>:" ).arg( m_oldName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-rename", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( m_oldName, this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

RenameProjectDialog::~RenameProjectDialog()
{
}

void RenameProjectDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name == m_oldName ) {
        QDialog::accept();
        return;
    }

    if ( !checkProjectName( name ) )
        return;

    ProjectsBatch* batch = new ProjectsBatch();
    batch->renameProject( m_projectId, name );

    executeBatch( batch );
}

DeleteProjectDialog::DeleteProjectDialog( int projectId, QWidget* parent ) : ProjectsDialog( parent ),
    m_projectId( projectId )
{
    const ProjectRow* project = dataManager->projects()->find( projectId );
    QString name = project ? project->name() : QString();

    setWindowTitle( tr( "Delete Project" ) );
    setPrompt( tr( "Are you sure you want to delete project <b>%1</b>?" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "edit-delete", 22 ) );

    showInfo( tr( "Confirm deletion" ) );

    setContentLayout( NULL, true );
}

DeleteProjectDialog::~DeleteProjectDialog()
{
}

void DeleteProjectDialog::accept()
{
    ProjectsBatch* batch = new ProjectsBatch();
    batch->deleteProject( m_projectId );

    executeBatch( batch );
}

AddFolderDialog::AddFolderDialog( int projectId, QWidget* parent ) : ProjectsDialog( parent ),
    m_projectId( projectId )
{
    const ProjectRow* project = dataManager->projects()->find( projectId );
    QString name = project ? project->name() : QString();

    setWindowTitle( tr( "Add Folder" ) );
    setPrompt( tr( "Create a new folder in project <b>%1</b>:" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "folder-new", 22 ) );

    QGridLayout* layout = new QGridLayout();

    QLabel* nameLabel = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( nameLabel, 0, 0 );

    m_nameEdit = new QLineEdit( tr( "New Folder" ), this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 0, 1 );

    nameLabel->setBuddy( m_nameEdit );

    QLabel* typeLabel = new QLabel( tr( "&Type:" ), this );
    layout->addWidget( typeLabel, 1, 0 );

    m_typeCombo = new QComboBox( this );
    layout->addWidget( m_typeCombo, 1, 1 );

    typeLabel->setBuddy( m_typeCombo );

    RDB::TableConstIterator<TypeRow> it( dataManager->types() );
    QList<const TypeRow*> types = localeAwareSortRows( it, &TypeRow::name );

    for ( int i = 0; i < types.count(); i++ )
        m_typeCombo->addItem( types.at( i )->name() );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

AddFolderDialog::~AddFolderDialog()
{
}

void AddFolderDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( !checkFolderName( m_projectId, name ) )
        return;

    QString type = m_typeCombo->currentText();
    int typeId = 0;

    RDB::TableConstIterator<TypeRow> it( dataManager->types() );
    while ( it.next() ) {
        if ( it.get()->name() == type ) {
            typeId = it.get()->typeId();
            break;
        }
    }

    ProjectsBatch* batch = new ProjectsBatch();
    batch->addFolder( m_projectId, typeId, name );

    executeBatch( batch );
}

RenameFolderDialog::RenameFolderDialog( int folderId, QWidget* parent ) : ProjectsDialog( parent ),
    m_folderId( folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    m_oldName = folder ? folder->name() : QString();

    setWindowTitle( tr( "Rename Folder" ) );
    setPrompt( tr( "Enter the new name of folder <b>%1</b>:" ).arg( m_oldName ) );
    setPromptPixmap( IconLoader::pixmap( "edit-rename", 22 ) );

    QHBoxLayout* layout = new QHBoxLayout();

    QLabel* label = new QLabel( tr( "&Name:" ), this );
    layout->addWidget( label, 0 );

    m_nameEdit = new QLineEdit( m_oldName, this );
    m_nameEdit->setMaxLength( 40 );
    layout->addWidget( m_nameEdit, 1 );

    label->setBuddy( m_nameEdit );

    setContentLayout( layout, true );

    m_nameEdit->setFocus();
}

RenameFolderDialog::~RenameFolderDialog()
{
}

void RenameFolderDialog::accept()
{
    QString name = m_nameEdit->text().trimmed();
    m_nameEdit->setText( name );

    if ( name == m_oldName ) {
        QDialog::accept();
        return;
    }

    const FolderRow* folder = dataManager->folders()->find( m_folderId );
    int projectId = folder ? folder->projectId() : 0;

    if ( !checkFolderName( projectId, name ) )
        return;

    ProjectsBatch* batch = new ProjectsBatch();
    batch->renameFolder( m_folderId, name );

    executeBatch( batch );
}

DeleteFolderDialog::DeleteFolderDialog( int folderId, QWidget* parent ) : ProjectsDialog( parent ),
    m_folderId( folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    QString name = folder ? folder->name() : QString();

    setWindowTitle( tr( "Delete Folder" ) );
    setPrompt( tr( "Are you sure you want to delete folder <b>%1</b>?" ).arg( name ) );
    setPromptPixmap( IconLoader::pixmap( "edit-delete", 22 ) );

    showInfo( tr( "Confirm deletion" ) );

    setContentLayout( NULL, true );
}

DeleteFolderDialog::~DeleteFolderDialog()
{
}

void DeleteFolderDialog::accept()
{
    ProjectsBatch* batch = new ProjectsBatch();
    batch->deleteFolder( m_folderId );

    executeBatch( batch );
}

#include "projectdialogs.moc"
