/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Didier Roche <didrocks@ubuntu.com>
 *
 */

using Dee;
using Gee;


namespace Unity.ApplicationsLens {

  private class AboutEntry {
    public string name;
    public string exec;
    public Icon   icon;
    
    public AboutEntry (string name, string exec, Icon icon)
    {
      this.name = name;
      this.exec = exec;
      this.icon = icon;
    }
  }

  public class Runner: GLib.Object
  {  
  
    private Unity.ApplicationsLens.Daemon daemon;    
    private const string BUS_NAME_PREFIX = "com.canonical.Unity.ApplicationsLens.Runner";
    
    /* for now, load the same keys as gnome-panel */
    private const string HISTORY_KEY = "history";
    private const int    MAX_HISTORY = 10;
    
    public Unity.Lens lens;
    public Unity.Scope scope;

    private Gee.HashMap<string,AboutEntry> about_entries;
    private Gee.List<string> history;
    private ExecSearcher exec_searcher = new ExecSearcher ();

    private Settings gp_settings;
    
    public Runner (Unity.ApplicationsLens.Daemon daemon)
    {
      /* First create scope */
      scope = new Unity.Scope ("/com/canonical/unity/scope/commands");
      scope.search_in_global = false;
            
      /* Listen for changes to the lens scope search */
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        update_search.begin (lens_search);
      });

      /* Make sure we ignore extra whitespace on searches */
      scope.generate_search_key.connect ((lens_search) =>
      {
        return lens_search.search_string.strip ();
      });

      // Listen for when the lens is hidden/shown by the Unity Shell
      scope.notify["active"].connect(() =>
      {
        if (scope.active)
        {
          scope.queue_search_changed (SearchType.DEFAULT);
        }
      });

      scope.activate_uri.connect (daemon.activate);

       /* Now create Lens */
      lens = new Unity.Lens ("/com/canonical/unity/lens/commands", "commands");
      lens.search_hint = _("Run a command");
      lens.visible = false;
      lens.search_in_global = false;
      populate_categories ();
      //populate_filters();
      lens.add_local_scope (scope);
      lens.export ();
      
      /* create the private about entries */
      about_entries = new Gee.HashMap<string,AboutEntry> ();
      load_about_entries ();
      
      this.gp_settings = new Settings ("com.canonical.Unity.Runner");
      history = new Gee.ArrayList<string> ();
      load_history ();
      
      this.daemon = daemon;
    
    }

    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      var icon_dir = File.new_for_path (ICON_PATH);
      var cat = new Unity.Category (_("Results"),
                                    new FileIcon (icon_dir.get_child ("group-installed.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("History"),
                                new FileIcon (icon_dir.get_child ("group-available.svg")));

      categories.append (cat);

      lens.categories = categories;
    }

    private async void update_search (LensSearch search)
    {
      var model = search.results_model;
      var executables_match = new Gee.ArrayList<string> ();
      var dirs_match = new Gee.ArrayList<string> ();
      model.clear ();

      var search_string = search.search_string;
      bool has_search = !Utils.is_search_empty (search);

      string uri;      
      Icon icon;
      string mimetype;
      string display_name;
      var category_id = RunnerCategory.HISTORY;

      foreach (var command in this.history)
      {          
        display_name = get_icon_uri_and_mimetype (command, out icon, out uri, out mimetype);   
        model.append (uri, icon.to_string (),
                      category_id, mimetype,
                      display_name,
                      null);
      }

      if (!has_search)
      {
        search.finished ();
        return;
      }

      Timer timer = new Timer ();
      
      /* no easter egg in unity */
      if (search_string == "free the fish")
      {
        uri = "no-easter-egg";
        string commenteaster = _("There is no easter egg in Unity");
        icon = new ThemedIcon ("gnome-panel-fish");
        model.append (uri, icon.to_string (),
                      0, "no-mime",
                      commenteaster,
                      null);
        search.finished ();
        return;
      }
      else if (search_string == "gegls from outer space")
      {
        uri = "really-no-easter-egg";
        string commentnoeaster = _("Still no easter egg in Unity");
        icon = new ThemedIcon ("gnome-panel-fish");
        model.append (uri, icon.to_string (),
                      0, "no-mime",
                      commentnoeaster,
                      null);
        search.finished ();
        return;
         
      }
      
      /* manual seek with directory and executables result */
      if (search_string.has_prefix ("/") || search_string.has_prefix ("~"))
      {
        search_string = Utils.subst_tilde (search_string);
        var search_dirname = Path.get_dirname (search_string);
        var directory = File.new_for_path (search_dirname);
        var search_dirname_in_path = false;
        
        /* strip path_directory if in executable in path */
        foreach (var path_directory in Environment.get_variable ("PATH").split(":"))
        {
          if (search_dirname == path_directory || search_dirname == path_directory + "/")
          {
            search_dirname_in_path = true;
            break;
          }
        }

        try {
          var iterator = directory.enumerate_children (FileAttribute.STANDARD_NAME + "," + FileAttribute.STANDARD_TYPE + "," + FileAttribute.ACCESS_CAN_EXECUTE,
                                                 0, null);
          while (true)
          {
            var subelem_info = iterator.next_file ();
            if (subelem_info == null)
              break;
            
            var complete_path = Path.build_filename (search_dirname, subelem_info.get_name ());
            if (complete_path.has_prefix (search_string))
            {
              if (subelem_info.get_file_type () == FileType.DIRECTORY)
              {
                dirs_match.add (complete_path);
              }
              else if (subelem_info.get_attribute_boolean (FileAttribute.ACCESS_CAN_EXECUTE))
              {
                // only include exec name if we are in the PATH
                if (search_dirname_in_path)
                  executables_match.add (subelem_info.get_name ());
                else
                  executables_match.add (complete_path);
              }
            }
          }
        }
        catch (Error err) {
          warning("Error listing directory executables: %s\n", err.message);
        }

      }
      /* complete again system executables */
      else
      {
        var matching_executables = yield exec_searcher.find_prefixed (search_string);
        foreach (var matching_exec in matching_executables)
        {
          executables_match.add (matching_exec);
        }
      }
      
      executables_match.sort ();
      dirs_match.sort ();
      
      category_id = RunnerCategory.RESULTS;

      // populate results
      // 1. enable launching the exact search string if no other result
      if ((executables_match.size == 0) && (dirs_match.size == 0))
      {
        display_name = get_icon_uri_and_mimetype (search_string, out icon, out uri, out mimetype);
        model.append (uri.strip (), icon.to_string (),
                      category_id, mimetype,
                      display_name,
                      null);
      }
      
      // 2. add possible directories (we don't store them)
      mimetype = "inode/directory";
      icon = ContentType.get_icon (mimetype);
      foreach (var dir in dirs_match)
      {
        uri = @"unity-runner://$(dir)";
        model.append (uri, icon.to_string (),
                      category_id, mimetype,
                      dir,
                      null);        
      }
              
      // 3. add available exec
      foreach (var final_exec in executables_match)
      {
        // TODO: try to match to a desktop file for the icon
        uri = @"unity-runner://$(final_exec)";
        display_name = get_icon_uri_and_mimetype (final_exec, out icon, out uri, out mimetype);   
        
        model.append (uri, icon.to_string (),
                      category_id, mimetype,
                      display_name,
                      null);
      }
      
      timer.stop ();
      debug ("Entry search listed %i dir matches and %i exec matches in %fms for search: %s",
             dirs_match.size, executables_match.size, timer.elapsed ()*1000, search_string);
      

      search.finished ();
    }
    
    private class ExecSearcher: Object
    {
      public ExecSearcher ()
      {
        Object ();
      }

      private Gee.List<string> executables;

      construct
      {
        executables = new Gee.ArrayList<string> ();
        listing_status = ListingStatus.NOT_STARTED;
      }

      // TODO: add reload
      private async void find_system_executables ()
      {
        if (this.executables.size > 0)
          return;

        foreach (var path_directory in Environment.get_variable ("PATH").split(":"))
        {
          var dir = File.new_for_path (path_directory);
          try {
            var e = yield dir.enumerate_children_async (FileAttribute.STANDARD_NAME + "," + FileAttribute.ACCESS_CAN_EXECUTE,
                                                        0, Priority.DEFAULT, null);
            while (true) {
              var files = yield e.next_files_async (64, Priority.DEFAULT, null);
              if (files == null)
                  break;

              foreach (var info in files) {
                if (info.get_attribute_boolean (FileAttribute.ACCESS_CAN_EXECUTE))
                {
                  this.executables.add (info.get_name ());
                }
              }
            }
          }
          catch (Error err) {
            warning("Error listing directory executables: %s\n", err.message);
          }
        }
      }

      private enum ListingStatus
      {
        NOT_STARTED,
        STARTED,
        FINISHED
      }

      public int listing_status { get; private set; }

      public async Gee.Collection<string> find_prefixed (string search_string)
      {
        // initialize the available binaries lazily
        if (listing_status == ListingStatus.NOT_STARTED)
        {
          listing_status = ListingStatus.STARTED;
          yield find_system_executables ();
          listing_status = ListingStatus.FINISHED;
        }
        else if (listing_status == ListingStatus.STARTED)
        {
          var sig_id = this.notify["listing-status"].connect (() =>
          {
            if (listing_status == ListingStatus.FINISHED)
              find_prefixed.callback ();
          });
          yield;
          SignalHandler.disconnect (this, sig_id);
        }

        var matching = new Gee.ArrayList<string> ();
        foreach (var exec_candidate in executables)
        {
          if (exec_candidate.has_prefix (search_string))
          {
            matching.add (exec_candidate);
          }
        }

        return matching;
      }
    }
    
    private string get_icon_uri_and_mimetype (string exec_string, out Icon? icon, out string? uri, out string? mimetype)
    {
      
      AboutEntry? entry = null;
      
      mimetype = "application/x-unity-run";      
      entry = about_entries[exec_string];
      if (entry != null)
      {
        uri = @"unity-runner://$(entry.exec)";
        icon = entry.icon;
        return entry.name;
      }

      uri = @"unity-runner://$(exec_string)";

      
      // if it's a folder, show… a folder icone! + right exec
      if (FileUtils.test (exec_string, FileTest.IS_DIR))
      {
        mimetype = "inode/directory";
        icon = ContentType.get_icon (mimetype);
        return exec_string;
      }

      var s = exec_string.delimit ("-", '_').split (" ", 0)[0];      
      var appresults = this.daemon.appsearcher.search (@"type:Application AND exec:$s", 0,
                                                       Unity.Package.SearchType.EXACT,
                                                       Unity.Package.Sort.BY_NAME);
      foreach (var pkginfo in appresults.results)
      {

        if (pkginfo.desktop_file == null)
          continue;
        
        // pick the first one
        icon = this.daemon.find_pkg_icon (pkginfo.desktop_file, pkginfo.icon);
        return exec_string;
        
      }
      
      // if no result, default icon
      icon = new ThemedIcon ("gtk-execute");
      return exec_string;
      
    }
 

    public void add_history (string last_command)
    {

      // new history list: better, greatest, latest!
      var new_history = new Gee.ArrayList<string> ();
      var history_store = new string [this.history.size + 1];
      int i = 1;
      
      new_history.add (last_command);
      history_store[0] = last_command;
      for (var j = 0; (j < this.history.size) && (i < MAX_HISTORY); j++)
      {
        if (this.history[j] == last_command)
           continue;
        
        new_history.add(history[j]);
        history_store[i] = history[j];
        i++;
      }
      this.history = new_history;

      // store in gsettings
      this.gp_settings.set_strv (HISTORY_KEY, history_store);
      
      // force a search to refresh history order (TODO: be more clever in the future)
      scope.queue_search_changed (SearchType.DEFAULT);
    }
    
    private void load_history ()
    {
      int i = 0;
      string[] history_store = this.gp_settings.get_strv (HISTORY_KEY);
      foreach (var command in history_store)
      {
        if (i >= MAX_HISTORY)
          break;
        this.history.add((string) command.data);
        i++;
      }
    }
    
    private void load_about_entries ()
    { 
      AboutEntry entry;
      string name;
      string exec;
      Icon icon;
      
      // first about:config
      name = "about:config";
      exec = "ccsm -p unityshell";
      try {
        icon = Icon.new_for_string (@"$(Config.PREFIX)/share/ccsm/icons/hicolor/64x64/apps/plugin-unityshell.png");      
      }
      catch (Error err) {
        warning ("Can't find unityshell icon: %s", err.message);
        icon = new ThemedIcon ("gtk-execute");
      }   
      entry = new AboutEntry (name, exec, icon);
      
      about_entries[name] = entry;
      about_entries[exec] = entry;
      
      // second about:robots
      name = "Robots have a plan.";
      exec = "firefox about:robots";
      entry = new AboutEntry (name, exec, icon = new ThemedIcon ("battery"));
      
      about_entries["about:robots"] = entry;
      about_entries[exec] = entry;
      
    }
      
  }
  
}
