# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for basic logic of the Ubuntu One Installer."""

import apt

from ubuntuone.devtools.testcases import BaseTestCase
from ubuntuone import installer


class FakeCacheEntry(object):
    """Fake cache object."""

    def __init__(self, is_installed=False):
        super(FakeCacheEntry, self).__init__()
        self.is_installed = is_installed


class InstallerTestCase(BaseTestCase):
    """Main logic tests."""

    def test_is_installed_true(self):
        """Test is_installed does the right thing when true."""
        self.patch(apt, 'Cache', lambda: {'foo': FakeCacheEntry(True)})
        self.assertTrue(installer.is_installed('foo'))

    def test_is_installed_false(self):
        """Test is_installed does the right thing when false."""
        self.patch(apt, 'Cache', lambda: {'foo': FakeCacheEntry()})
        self.assertFalse(installer.is_installed('foo'))

    def test_is_installed_systemerror_false(self):
        """Test is_installed does the right thing if SystemError is raised."""
        def _raise_system_error():
            raise SystemError('Testing.')

        self.patch(apt, 'Cache', _raise_system_error)
        self.assertRaises(TypeError, installer.is_installed, 'foo', None)

    def test_is_installed_systemerror_true(self):
        """Test is_installed does the right thing if SystemError is raised,
           and the command is found."""
        def _raise_system_error():
            raise SystemError('Testing.')

        self.patch(apt, 'Cache', _raise_system_error)
        self.assertTrue(installer.is_installed('foo', 'python'))
