# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the darwin utils functions."""

import os
import sys

from Cocoa import NSURL

from collections import defaultdict, namedtuple
from functools import partial

from twisted.internet import defer

from ubuntuone.controlpanel import utils
from ubuntuone.devtools.testcases import TestCase

# let me use protected methods
# pylint: disable=W0212


class CallRecordingTestCase(TestCase):
    """Base class with multi-call checker."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up call checker."""
        yield super(CallRecordingTestCase, self).setUp()
        self._called = defaultdict(list)

    def _patch_and_track(self, module, funcs):
        """Record calls along with function name."""

        def record_call(fname, retval, *args, **kwargs):
            """Wrapper for a single function."""
            self._called[fname].append((args, kwargs))
            return retval

        for (fname, retval) in funcs:
            wrapper = partial(record_call, fname, retval)
            self.patch(module, fname, wrapper)


class InstallConfigTestCase(TestCase):
    """Test install_config_and_daemons."""

    APP_PATH = "/path/to/Main.app/"
    TARGET_PATH = "TARGET_PATH"

    @defer.inlineCallbacks
    def setUp(self):
        """Set up multi-call checker."""
        yield super(InstallConfigTestCase, self).setUp()
        self._called = []

        self.non_overwrite_args = [((os.path.join(self.APP_PATH, 'Contents',
                                                  'Resources',
                                                  'syncdaemon.conf'),
                                     os.path.join(self.TARGET_PATH,
                                                  'syncdaemon.conf')), {}),
                                   ((os.path.join(self.APP_PATH, 'Contents',
                                                  'Resources', 'logging.conf'),
                                     os.path.join(self.TARGET_PATH,
                                                  'logging.conf')), {})]
        self.overwrite_args = [((os.path.join(self.APP_PATH, 'Contents',
                                         'Resources', 'update.conf'),
                            os.path.join(self.TARGET_PATH,
                                         'update.conf')), {})]
        self.patch(utils.darwin,
                   'check_and_install_fsevents_daemon',
                   lambda _: None)

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs for test assertions."""
        self._called.append((args, kwargs))

    def test_do_nothing_unfrozen(self):
        """Test that install_config_and_daemons does nothing when unfrozen."""

        self.patch(utils.darwin, 'save_config_path',
                   self._set_called)
        utils.install_config_and_daemons()
        self.assertEqual(self._called, [])

    def _test_copying_conf_files(self, exists, src_mtime=0, dest_mtime=0):
        """Call install_config_and_daemons, parameterize os.path.exists."""
        sys.frozen = 'macosx_app'
        self.addCleanup(delattr, sys, 'frozen')
        self.patch(utils.darwin, 'save_config_path',
                   lambda x: self.TARGET_PATH)
        self.patch(utils.darwin, '__file__',
                   os.path.join(self.APP_PATH, "ignore"))
        self.patch(os.path, "exists", lambda x: exists)
        self.patch(utils.darwin.shutil, 'copyfile',
                   self._set_called)

        def fake_stat(path):
            fakestat = namedtuple('fakestat', ['st_mtime'])
            if path.startswith(self.APP_PATH):
                return fakestat(src_mtime)
            else:
                return fakestat(dest_mtime)

        self.patch(utils.os, 'stat', fake_stat)

        utils.install_config_and_daemons()

    def test_copies_conf_files_same(self):
        """When frozen, we copy the conf files if they don't exist."""
        self._test_copying_conf_files(False)
        self.assertEqual(self._called, self.non_overwrite_args +
                         self.overwrite_args)

    def test_does_not_copy_conf_files_same(self):
        """frozen: do not copy any files that exist if their mtime is new."""
        self._test_copying_conf_files(True, src_mtime=0, dest_mtime=1)
        self.assertEqual(self._called, [])

    def test_only_copy_update_exist_same(self):
        """frz: only copy update.conf if files exist but src mtime newer."""
        self._test_copying_conf_files(True, src_mtime=1, dest_mtime=0)
        self.assertEqual(self._called, self.overwrite_args)


class InstallDaemonTestCase(CallRecordingTestCase):
    """Test fsevents daemon installation."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up patched & tracked calls."""
        yield super(InstallDaemonTestCase, self).setUp()

        self._patch_and_track(utils.darwin,
                              [('get_authorization', 'Fake AuthRef'),
                               ('remove_fsevents_daemon', None),
                               ('install_fsevents_daemon', None),
                               ('AuthorizationFree', None)])

    def _patch_versions(self, installed, bundled):
        """Convenience to patch the version-getting functions."""
        self.patch(utils.darwin,
                   "get_bundle_version",
                   lambda _: bundled)
        self.patch(utils.darwin,
                   "get_fsevents_daemon_installed_version",
                   lambda: installed)

    def test_check_and_install_current_version(self):
        """Test that we do nothing on current version"""

        self._patch_versions(installed=47.0, bundled=47.0)

        utils.darwin.check_and_install_fsevents_daemon('NOT A REAL DIR')

        self.assertEqual(self._called.keys(), [])

    def test_check_and_install_upgrade(self):
        """Test removing old daemon and installing new one."""

        self._patch_versions(installed=35.0, bundled=35.1)

        utils.darwin.check_and_install_fsevents_daemon('NOT A REAL DIR')

        self.assertEqual(self._called['get_authorization'],
                         [((), {}), ((), {})])
        self.assertEqual(self._called['remove_fsevents_daemon'],
                         [(('Fake AuthRef',), {})])
        self.assertEqual(self._called['install_fsevents_daemon'],
                         [(('Fake AuthRef',), {})])
        self.assertEqual(self._called['AuthorizationFree'],
                         [(('Fake AuthRef',
                            utils.darwin.kAuthorizationFlagDestroyRights), {}),
                          (('Fake AuthRef',
                            utils.darwin.kAuthorizationFlagDestroyRights), {})
                          ])

    def test_check_and_install_mismatch(self):
        """Test raising when we're older than the daemon."""

        self._patch_versions(installed=102.5, bundled=66.0)

        self.assertRaises(utils.darwin.DaemonVersionMismatchException,
                          utils.darwin.check_and_install_fsevents_daemon,
                          'NOT A REAL DIR')
        self.assertEqual(self._called.keys(), [])


class CFCallsTestCase(CallRecordingTestCase):
    """Test functions that call CoreFoundation API."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up call checker."""
        yield super(CFCallsTestCase, self).setUp()
        self._called = defaultdict(list)
        self.patch(utils.darwin, 'create_cfstr',
                   lambda s: s)
        self.patch(utils.darwin, 'CFShow',
                   lambda _: None)
        self.patch(utils.darwin, 'kSMDomainSystemLaunchd',
                   'not a c_void_p')

    def test_remove_daemon_ok(self):
        """Test that we call SMJobRemove and don't raise when it returns OK."""
        self._patch_and_track(utils.darwin, [('SMJobRemove', True),
                                             ('c_void_p', 'notaptr'),
                                             ('byref', 'not a **'),
                                             ('CFRelease', 'ignore')])

        utils.darwin.remove_fsevents_daemon('not an authref')
        self.assertEqual(self._called['SMJobRemove'],
                         [(('not a c_void_p',
                            utils.darwin.FSEVENTSD_JOB_LABEL,
                           'not an authref',
                           True, 'not a **'), {})])
        self.assertEqual(self._called['CFRelease'],
                         [((utils.darwin.FSEVENTSD_JOB_LABEL,), {})])

    def test_remove_daemon_not_ok(self):
        """Test that we raise when SMJobRemove returns not OK."""
        self._patch_and_track(utils.darwin, [('SMJobRemove', False),
                                             ('c_void_p', 'notaptr'),
                                             ('byref', 'not a **'),
                                             ('CFRelease', 'ignore'),
                                             ('CFErrorCopyDescription',
                                              'Houston, we have a problem')])

        self.assertRaises(utils.darwin.DaemonRemoveException,
                          utils.darwin.remove_fsevents_daemon,
                          'not an authref')

    def test_install_daemon_ok(self):
        """Test that we call SMJobBless and don't raise when it returns OK."""
        self._patch_and_track(utils.darwin, [('SMJobBless', True),
                                             ('c_void_p', 'notaptr'),
                                             ('byref', 'not a **'),
                                             ('CFRelease', 'ignore')])

        utils.darwin.install_fsevents_daemon('not an authref')
        self.assertEqual(self._called['SMJobBless'],
                         [(('not a c_void_p',
                            utils.darwin.FSEVENTSD_JOB_LABEL,
                           'not an authref',
                            'not a **'), {})])
        self.assertEqual(self._called['CFRelease'],
                         [((utils.darwin.FSEVENTSD_JOB_LABEL,), {})])

    def test_install_daemon_not_ok(self):
        """Test that we raise when SMJobBless returns not OK."""
        self._patch_and_track(utils.darwin, [('SMJobBless', False),
                                             ('c_void_p', 'notaptr'),
                                             ('byref', 'not a **'),
                                             ('CFRelease', 'ignore'),
                                             ('CFErrorCopyDescription',
                                              'Houston, we have a problem')])

        self.assertRaises(utils.darwin.DaemonInstallException,
                          utils.darwin.install_fsevents_daemon,
                          'not an authref')

    def test_get_bundle_version(self):
        """Simple test of #calls in get_bundle_version."""
        # This list includes expected counts. This test is mostly good
        # to check that we match the number of 'create's with the
        # number of 'releases', which is 3 (note create_cfstr is
        # patched in setUp.)
        to_track = [('CFURLCreateWithFileSystemPath',
                     'url', 1),
                    ('CFBundleCopyInfoDictionaryForURL',
                     'dict', 1),
                    ('CFDictionaryGetValue', 'val', 1),
                    ('CFStringGetDoubleValue', 102.5, 1),
                    ('CFRelease', 'ignore', 3)]
        self._patch_and_track(utils.darwin, [(n, r) for (n, r, _) in to_track])

        utils.darwin.get_bundle_version("not a cfstr")
        for (name, _, num) in to_track:
            self.assertEqual(len(self._called[name]), num)

    def test_get_fsevents_daemon_installed_version_ok(self):
        """Test that we return the version if the dictionary is there."""
        to_track = [('SMJobCopyDictionary', 'not none'),
                    ('CFRelease', 'None'),
                    ('CFDictionaryGetValue', 'val from dict'),
                    ('CFArrayGetValueAtIndex', 'val'),
                    ('get_bundle_version', 1.0)]
        self._patch_and_track(utils.darwin, to_track)
        utils.darwin.get_fsevents_daemon_installed_version()

        self.assertEqual(self._called['CFDictionaryGetValue'],
                         [(('not none', "ProgramArguments"), {})])
        self.assertEqual(self._called['CFArrayGetValueAtIndex'],
                         [(('val from dict', 0), {})])
        self.assertEqual(self._called['get_bundle_version'],
                         [(('val',), {})])

    def test_get_fsevents_daemon_installed_version_not_found(self):
        """Test that we return None if the dictionary is not there."""
        to_track = [('SMJobCopyDictionary', None),
                    ('CFRelease', 'None'),
                    ('CFDictionaryGetValue', 'val from dict'),
                    ('CFArrayGetValueAtIndex', 'val'),
                    ('get_bundle_version', 1.0)]
        self._patch_and_track(utils.darwin, to_track)
        utils.darwin.get_fsevents_daemon_installed_version()

        self.assertTrue('CFDictionaryGetValue' not in self._called.keys())
        self.assertTrue('CFArrayGetValueAtIndex' not in self._called.keys())
        self.assertTrue('get_bundle_version' not in self._called.keys())

    def test_get_authorization_ok(self):
        """Test successful call of AuthorizationCreate does not raise."""
        to_track = [('c_void_p', 'not void p'),
                    ('byref', 'not **'),
                    ('AuthorizationCreate',
                     utils.darwin.errAuthorizationSuccess)]
        self._patch_and_track(utils.darwin, to_track)
        auth_ref = utils.darwin.get_authorization()
        self.assertEqual(auth_ref, 'not void p')


class DefaultFoldersTestCase(TestCase):
    """Simple Test for default_folders."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up folders call."""
        yield super(DefaultFoldersTestCase, self).setUp()
        self.folders = utils.darwin.default_folders()

    def test_default_folders_length(self):
        """Test that we get some default folders."""
        self.assertEqual(len(self.folders), 6)

    def test_default_folders_expanded(self):
        """Should expand paths."""
        have_tilde = [n.startswith("~") for n in self.folders]
        self.assertNotIn(True, have_tilde)

    def test_default_folders_bytes(self):
        """Paths should be bytes, not unicode."""
        is_bytes = [type(n) == type(b'') for n in self.folders]
        self.assertNotIn(False, is_bytes)


class MockStandardUserDefaults(object):
    """Fake user defaults"""

    def __init__(self, keyval):
        self.keyval = keyval
        self.setBool_args = []

    def boolForKey_(self, key):
        return self.keyval

    def setBool_forKey_(self, bval, key):
        self.setBool_args.append((bval, key))


class AddU1FolderToFavoritesTestCase(CallRecordingTestCase):
    """Test adding folder to favorites."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(AddU1FolderToFavoritesTestCase, self).setUp()
        self.path = "/Users/a/Ubuntu%20One"
        self.testurl = NSURL.URLWithString_("file://localhost" +
                                            self.path)

        self.patch(utils.darwin.os.path, "expanduser",
                   lambda x: self.path)
        self.patch(utils.darwin, "LSSharedFileListCreate",
                   lambda x, y, z: "fake-list")
        self.patch(utils.darwin, "kLSSharedFileListItemBeforeFirst", 1)
        self._patch_and_track(utils.darwin, [("LSSharedFileListInsertItemURL",
                                              None)])

    def test_call_with_home_url_noflag(self):
        """Test adding correct URL when we haven't before."""

        mock_sud = MockStandardUserDefaults(False)
        self.patch(utils.darwin, '_get_userdefaults',
                   lambda: mock_sud)

        utils.darwin.add_u1_folder_to_favorites()
        self.assertEqual(mock_sud.setBool_args,
                         [(True, utils.darwin.ADDED_TO_FINDER_SIDEBAR_ONCE)])
        self.assertEqual(self._called['LSSharedFileListInsertItemURL'],
                         [(("fake-list", 1, None, None, self.testurl, {}, []),
                           {})])

    def test_call_with_home_url_yesflag(self):
        """Test not re-adding URL when we've done it once before."""

        mock_sud = MockStandardUserDefaults(True)
        self.patch(utils.darwin, '_get_userdefaults',
                   lambda: mock_sud)

        utils.darwin.add_u1_folder_to_favorites()

        self.assertEqual(mock_sud.setBool_args, [])

        self.assertEqual(self._called['LSSharedFileListInsertItemURL'], [])
