# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Control Panel."""

from __future__ import division

import operator

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import preferences as gui
from ubuntuone.controlpanel.gui.qt.tests import SAMPLE_SETTINGS
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)

# Access to a protected member
# pylint: disable=W0212


class PreferencesPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt cloud preferences tab."""

    innerclass_ui = gui.preferences_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.PreferencesPanel

    def dict_from_ui_values(self):
        """Build the settings dict from the ui values."""
        autoconnect = self.ui.ui.autoconnect_checkbox.checkState()
        notifs = self.ui.ui.show_all_notifications_checkbox.checkState()
        share_auto = self.ui.ui.share_autosubscribe_checkbox.checkState()
        udf_auto = self.ui.ui.udf_autosubscribe_checkbox.checkState()
        if self.ui.ui.limit_downloads_checkbox.checkState() == gui.CHECKED:
            download_speed = self.ui.ui.download_speed_spinbox.value()
        else:
            download_speed = -1
        if self.ui.ui.limit_uploads_checkbox.checkState() == gui.CHECKED:
            upload_speed = self.ui.ui.upload_speed_spinbox.value()
        else:
            upload_speed = -1

        result = {
            gui.backend.AUTOCONNECT_KEY: autoconnect == gui.CHECKED,
            gui.backend.SHOW_ALL_NOTIFICATIONS_KEY: notifs == gui.CHECKED,
            gui.backend.SHARE_AUTOSUBSCRIBE_KEY: share_auto == gui.CHECKED,
            gui.backend.UDF_AUTOSUBSCRIBE_KEY: udf_auto == gui.CHECKED,
            gui.backend.DOWNLOAD_KEY: download_speed * gui.KILOBYTES,
            gui.backend.UPLOAD_KEY: upload_speed * gui.KILOBYTES,
        }
        return result

    @defer.inlineCallbacks
    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check():
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return SAMPLE_SETTINGS

        self.patch(self.ui.backend, 'file_sync_settings_info', check)

        yield self.ui.load()  # trigger the info request

    def test_is_not_processing_after_info_ready(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(SAMPLE_SETTINGS)

        self.assertFalse(self.ui.is_processing)

    def test_process_info_limit_bandwidth(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(SAMPLE_SETTINGS)

        self.assertTrue(self.ui.ui.limit_uploads_checkbox.isChecked())
        self.assertTrue(self.ui.ui.limit_downloads_checkbox.isChecked())

        settings = SAMPLE_SETTINGS.copy()
        settings[gui.backend.DOWNLOAD_KEY] = -1
        settings[gui.backend.UPLOAD_KEY] = -1
        self.ui.process_info(settings)

        self.assertFalse(self.ui.ui.limit_uploads_checkbox.isChecked())
        self.assertFalse(self.ui.ui.limit_downloads_checkbox.isChecked())

    def test_info_is_requested_on_load(self):
        """The info is requested to the backend."""
        self.assert_backend_called('file_sync_settings_info')

    def _test_update_ui_from_settings_info(self, settings):
        """The ui is correctly updated when the backend info is received."""
        autoconnect = self.ui.ui.autoconnect_checkbox.checkState()
        self.assertEqual(autoconnect == gui.CHECKED,
                         settings[gui.backend.AUTOCONNECT_KEY])

        show_all_notifications = \
            self.ui.ui.show_all_notifications_checkbox.checkState()
        self.assertEqual(show_all_notifications == gui.CHECKED,
                         settings[gui.backend.SHOW_ALL_NOTIFICATIONS_KEY])

        share_autosubscribe = \
            self.ui.ui.share_autosubscribe_checkbox.checkState()
        self.assertEqual(share_autosubscribe == gui.CHECKED,
                         settings[gui.backend.SHARE_AUTOSUBSCRIBE_KEY])

        udf_autosubscribe = self.ui.ui.udf_autosubscribe_checkbox.checkState()
        self.assertEqual(udf_autosubscribe == gui.CHECKED,
                         settings[gui.backend.UDF_AUTOSUBSCRIBE_KEY])

        speed = settings[gui.backend.DOWNLOAD_KEY]
        if speed > 0:
            speed = settings[gui.backend.DOWNLOAD_KEY] // gui.KILOBYTES
            download_speed = self.ui.ui.download_speed_spinbox.value()
            self.assertEqual(speed, download_speed)
        limit_downloads = self.ui.ui.limit_downloads_checkbox.checkState()
        self.assertEqual(self.ui.ui.download_speed_spinbox.isEnabled(),
            limit_downloads == gui.CHECKED)

        speed = settings[gui.backend.UPLOAD_KEY]
        if speed > 0:
            speed = settings[gui.backend.UPLOAD_KEY] // gui.KILOBYTES
            upload_speed = self.ui.ui.upload_speed_spinbox.value()
            self.assertEqual(speed, upload_speed)
        limit_uploads = self.ui.ui.limit_uploads_checkbox.checkState()
        self.assertEqual(self.ui.ui.upload_speed_spinbox.isEnabled(),
            limit_uploads == gui.CHECKED)

    def test_update_ui_from_backed_info(self):
        """The ui is correctly updated when the backend info is received."""
        self.ui.process_info(SAMPLE_SETTINGS)
        self._test_update_ui_from_settings_info(SAMPLE_SETTINGS)

    def test_update_ui_from_backed_info_speeds_negative(self):
        """The ui is correctly updated when the backend info is received."""
        # process a first time with positive speed values to have the
        # checkboxes checked
        self.ui.process_info(SAMPLE_SETTINGS)
        assert self.ui.ui.limit_downloads_checkbox.checkState() == gui.CHECKED
        assert self.ui.ui.limit_uploads_checkbox.checkState() == gui.CHECKED

        settings = SAMPLE_SETTINGS.copy()
        settings[gui.backend.DOWNLOAD_KEY] = -1
        settings[gui.backend.UPLOAD_KEY] = -1

        self.ui.process_info(settings)
        self._test_update_ui_from_settings_info(settings)

    def _test_speed_checkbox_toggled(self, speed_spinbox, limit_checkbox):
        """The checkbox is checked if and only if the spinbox is enabled."""
        limit_checkbox.setCheckState(gui.CHECKED)
        self.assertTrue(speed_spinbox.isEnabled())
        limit_checkbox.setCheckState(gui.UNCHECKED)
        self.assertFalse(speed_spinbox.isEnabled())

    def test_download_speed_checkbox_connections(self):
        """Check connections for download_speed_checkbox."""
        self._test_speed_checkbox_toggled(
            self.ui.ui.download_speed_spinbox,
            self.ui.ui.limit_downloads_checkbox)

    def test_upload_speed_checkbox_connections(self):
        """Check connections for upload_speed_checkbox."""
        self._test_speed_checkbox_toggled(
            self.ui.ui.upload_speed_spinbox,
            self.ui.ui.limit_uploads_checkbox)

    def test_apply_changes_button_clicked(self):
        """When the apply button is clicked, the backend is called."""
        self.ui.ui.autoconnect_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.show_all_notifications_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.share_autosubscribe_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.udf_autosubscribe_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.download_speed_spinbox.setValue(20)
        self.ui.ui.upload_speed_spinbox.setValue(2)

        self.ui.ui.apply_changes_button.click()

        expected_dict = self.dict_from_ui_values()
        self.assert_backend_called('change_file_sync_settings', expected_dict)

    def test_apply_changes_button_clicked_negative_limits(self):
        """If the limit is negative, do not multiply by KILOBYTES."""
        self.ui.ui.download_speed_spinbox.setValue(-1)
        self.ui.ui.upload_speed_spinbox.setValue(-1)

        self.ui.ui.apply_changes_button.click()

        expected_dict = self.dict_from_ui_values()
        self.assert_backend_called('change_file_sync_settings', expected_dict)

    def test_apply_changes_button_clicked_zero_limits(self):
        """If the limit is negative, do not multiply by KILOBYTES."""
        self.ui.ui.download_speed_spinbox.setValue(0)
        self.ui.ui.upload_speed_spinbox.setValue(0)

        self.ui.ui.apply_changes_button.click()

        expected_dict = self.dict_from_ui_values()
        self.assert_backend_called('change_file_sync_settings', expected_dict)

    @defer.inlineCallbacks
    def test_is_processing_while_applying_changes(self):
        """The ui is processing while the changes are being applied."""
        def check(*a):
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')

        self.patch(self.ui.backend, 'change_file_sync_settings', check)

        yield self.ui.ui.apply_changes_button.click()

    def test_restore_defaults_button_clicked(self):
        """When the restore button is clicked, the backend is called."""
        settings = gui.backend.ControlBackend.DEFAULT_FILE_SYNC_SETTINGS
        self.patch(self.ui.backend, 'file_sync_settings_info',
                   lambda: settings)
        # modify the state a bit to ensure defaults are applied
        self.ui.ui.autoconnect_checkbox.setCheckState(gui.UNCHECKED)
        self.ui.ui.show_all_notifications_checkbox.setCheckState(gui.UNCHECKED)
        self.ui.ui.share_autosubscribe_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.udf_autosubscribe_checkbox.setCheckState(gui.CHECKED)
        self.ui.ui.download_speed_spinbox.setValue(20)
        self.ui.ui.upload_speed_spinbox.setValue(2)

        self.ui.ui.restore_defaults_button.click()

        self.assert_backend_called('restore_file_sync_settings')

        self._test_update_ui_from_settings_info(settings)

    @defer.inlineCallbacks
    def test_is_processing_while_restoring_changes(self):
        """The ui is processing while the changes are being applied."""
        def check():
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')

        self.patch(self.ui.backend, 'restore_file_sync_settings', check)

        yield self.ui.ui.restore_defaults_button.click()

    def test_resize_event(self):
        """Check that the QCheckBox widgets receive the proper texts."""
        wrap_calls = {}
        self.patch(gui, "force_wordwrap",
            lambda check, size, text: operator.setitem(
                wrap_calls, check, (size, text)))
        self.ui.setFixedWidth(1000)
        size_bandwidth = (self.ui.ui.bandwidth_settings.width() -
            self.ui.ui.upload_speed_spinbox.width())
        padding = 160  # Left and Right Padding
        size_sync = (self.ui.ui.file_sync_settings.width() - padding)
        self.ui.show()
        self.addCleanup(self.ui.hide)
        self.assertEqual(wrap_calls[self.ui.ui.limit_uploads_checkbox],
            (size_bandwidth, gui.SETTINGS_LIMIT_UPLOAD))
        self.assertEqual(wrap_calls[self.ui.ui.limit_downloads_checkbox],
            (size_bandwidth, gui.SETTINGS_LIMIT_DOWNLOAD))
        self.assertEqual(wrap_calls[self.ui.ui.autoconnect_checkbox],
            (size_sync, gui.SETTINGS_AUTO_CONNECT))
        self.assertEqual(wrap_calls[self.ui.ui.udf_autosubscribe_checkbox],
            (size_sync, gui.SETTINGS_SYNC_ALL_SHARES))
        self.assertEqual(wrap_calls[self.ui.ui.share_autosubscribe_checkbox],
            (size_sync, gui.SETTINGS_SYNC_ALL_FOLDERS))
        self.assertEqual(
            wrap_calls[self.ui.ui.show_all_notifications_checkbox],
            (size_sync, gui.SETTINGS_ALLOW_NOTIFICATIONS))
