# -*- coding: utf-8 -*-
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""System notification area icon."""

import os
import sys
from operator import itemgetter

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks
from ubuntuone.platform.tools import SyncDaemonTool

from ubuntuone.controlpanel import backend, cache
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import (
    ACCEPT_SHARES,
    DASHBOARD,
    GET_HELP_ONLINE,
    GET_MORE_STORAGE,
    GET_STORAGE_LINK,
    GET_SUPPORT_LINK,
    GO_TO_WEB,
    IN_PROGRESS,
    IN_PROGRESS_FILE,
    LOADING,
    OPEN_UBUNTU_ONE,
    OPEN_UBUNTU_ONE_FOLDER,
    PLEASE_WAIT,
    RECENT_TRANSFERS,
    RECENTTRANSFERS,
    SHARE_A_FILE,
    TRANSFERS,
    UPLOADING,
)
from ubuntuone.controlpanel.gui.qt import (
    FILE_SYNC_STATUS,
    icon_from_name,
)


logger = setup_logging('qt.systray')


class ProgressBarAction(QtGui.QWidgetAction):

    """Menu action that display a progress bar for the uploads."""

    def __init__(self, filename, percentage, parent=None):
        super(ProgressBarAction, self).__init__(parent)
        self._filename = os.path.basename(filename)
        self.widget = QtGui.QWidget(None)
        text = IN_PROGRESS_FILE % (self._filename, percentage)
        self.setText(text)
        self.label = QtGui.QLabel(text)
        self.progress = QtGui.QProgressBar()
        self.progress.setTextVisible(False)
        self.progress.setValue(percentage)
        self.progress.setFixedHeight(10)
        vbox = QtGui.QVBoxLayout()
        vbox.setContentsMargins(10, 5, 10, 5)
        vbox.addWidget(self.label)
        vbox.addWidget(self.progress)
        self.widget.setLayout(vbox)

        self.setDefaultWidget(self.widget)

    def set_value(self, progress):
        """Set the value of the progress bar."""
        text = IN_PROGRESS_FILE % (self._filename, progress)
        self.setText(text)
        self.label.setText(text)
        self.setText(text)
        self.progress.setValue(progress)


class SharesAction(QtGui.QAction):

    """New Shares action."""

    def __init__(self, text, volume_id, parent=None):
        super(SharesAction, self).__init__(text, parent)
        self._volume_id = volume_id

        self.triggered.connect(self.accept_share)

    def accept_share(self):
        """Open the accept shares page."""
        url = ACCEPT_SHARES % self._volume_id
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(url))


class TrayIcon(QtGui.QSystemTrayIcon, cache.Cache):

    """System notification icon."""

    def __init__(self, window=None, close_callback=lambda: None):
        super(TrayIcon, self).__init__(None)
        self.setIcon(QtGui.QIcon(":/icon.png"))
        self.setVisible(True)
        self.window = window
        self.root_path = ''
        self.recent_transfers = {}
        self.uploading = {}
        self._backend_method = None
        self._previous_shares = None
        self.close_callback = close_callback
        self.context_menu = QtGui.QMenu()
        self.status = None
        self.quit = None
        self.get_help_online = None
        self.open_u1 = None
        self.share_a_file = None
        self.status_action = None
        self.go_to_web = None
        self.get_more_storage = None
        self.transfers = None
        self.open_u1_folder = None
        self.backend.add_status_changed_handler(self._process_status)

        self.load_menu()

    def load_menu(self):
        """Load the content of the menu."""
        # Items
        self.context_menu.clear()

        self.status = self.context_menu.addAction(LOADING)
        self.status.setEnabled(False)
        self.status_action = self.context_menu.addAction(PLEASE_WAIT)
        self.refresh_status()
        self.context_menu.addSeparator()

        self.open_u1 = self.context_menu.addAction(OPEN_UBUNTU_ONE)
        self.share_a_file = self.context_menu.addAction(SHARE_A_FILE)
        self.open_u1_folder = self.context_menu.addAction(
            OPEN_UBUNTU_ONE_FOLDER)
        self.go_to_web = self.context_menu.addAction(GO_TO_WEB)
        self.context_menu.addSeparator()

        # Transfers
        self.transfers = TransfersMenu(self)
        self.context_menu.addMenu(self.transfers)

        self.get_more_storage = self.context_menu.addAction(
            GET_MORE_STORAGE)
        self.get_help_online = self.context_menu.addAction(GET_HELP_ONLINE)
        self.quit = self.context_menu.addAction("Quit")

        self.setContextMenu(self.context_menu)

        # Connect Signals
        self.status_action.triggered.connect(self.change_status)
        self.open_u1.triggered.connect(self.restore_window)
        self.share_a_file.triggered.connect(self.open_share_tab)
        self.open_u1_folder.triggered.connect(self.open_u1_folder_action)
        self.get_more_storage.triggered.connect(
            self.get_more_storage_action)
        self.go_to_web.triggered.connect(self.go_to_web_action)
        self.get_help_online.triggered.connect(self.get_help_action)
        self.quit.triggered.connect(self.stop)

        self._get_volumes_info()

    @inlineCallbacks
    def refresh_status(self):
        """Update Ubuntu One status."""
        info = yield self.backend.file_sync_status()
        self._process_status(info)

    def _process_status(self, status):
        """Match status with signals."""
        if status is None:
            return
        try:
            status_key = status[backend.STATUS_KEY]
            data = FILE_SYNC_STATUS[status_key]
        except (KeyError, TypeError):
            logger.exception(
                '_process_status: received unknown status dict %r', status)
            return

        self.status_action.setEnabled(True)
        icon_name = data.get('icon')
        icon = QtGui.QIcon()
        if icon_name is not None:
            icon = icon_from_name(icon_name)
        text = data.get('msg')
        self.status.setIcon(icon)
        self.status.setText(text)

        action = data.get('action')
        self._backend_method = getattr(self.backend,
            data['backend_method'], None)
        self.status_action.setText(action)

    def change_status(self):
        """Change the Syncing status of syncdaemon."""
        if self._backend_method is not None:
            self.status_action.setEnabled(False)
            self._backend_method()
        else:
            logger.error('on_sync_status_button_clicked: backend method is '
                         'None!')

    @inlineCallbacks
    def _get_volumes_info(self):
        """Get the volumes info."""
        info = yield self.backend.volumes_info()
        self._process_volumes_info(info)

    def _process_volumes_info(self, info):
        """Process the volumes info."""
        _, _, volumes = info[0]
        self.root_path = u'file://%s' % unicode(volumes[0]['path'])

    def open_u1_folder_action(self):
        """Open the web to get more storage."""
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(self.root_path))

    def get_more_storage_action(self):
        """Open the web to get more storage."""
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(GET_STORAGE_LINK))

    def go_to_web_action(self):
        """Open the web in the dashboard."""
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(DASHBOARD))

    def get_help_action(self):
        """Open the web in the dashboard."""
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(GET_SUPPORT_LINK))

    def restore_window(self):
        """Show the main window."""
        if self.window is None:
            # pylint: disable=W0404
            from ubuntuone.controlpanel.gui.qt.gui import MainWindow
            # pylint: enable=W0404
            self.window = MainWindow(close_callback=self.delete_window)
        self.window.show()
        self.window.activateWindow()

    def open_share_tab(self):
        """Show the main window in the share tab."""
        if self.window is None:
            # pylint: disable=W0404
            from ubuntuone.controlpanel.gui.qt.gui import MainWindow
            # pylint: enable=W0404
            self.window = MainWindow(close_callback=self.delete_window)
        self.window.switch_to('share_links')
        self.window.show()
        self.window.activateWindow()

    def delete_window(self):
        """Close and remove the main window."""
        if self.window is not None:
            self.window.close()
            self.window = None

    @inlineCallbacks
    def stop(self, *args, **kwargs):
        """Stop the application."""
        # Stop syncdaemon
        # pylint: disable=W0702
        # Really, if it fails we can't do anything about it.
        try:
            st = SyncDaemonTool()
            yield st.quit()
        except:
            # Maybe it was not running?
            pass
        self.close_callback()


class TransfersMenu(QtGui.QMenu):

    """Menu that shows recent and current transfers."""

    def __init__(self, parent):
        super(TransfersMenu, self).__init__(TRANSFERS)
        self._parent = parent
        self.uploading = {}
        self.previous_data = {}

        if sys.platform not in ('win32', 'darwin'):
            self._timer_id = self.startTimer(1000)

    # pylint: disable=C0103

    def showEvent(self, event):
        """Executed when the transfers menu is shown."""
        logger.info('This is never executed on Ubuntu')
        super(TransfersMenu, self).showEvent(event)
        self.get_transfers_data()
        self._timer_id = self.startTimer(1000)

    def hideEvent(self, event):
        """Executed when the transfers menu is hidden."""
        logger.info('This is never executed on Ubuntu')
        super(TransfersMenu, self).hideEvent(event)
        self.killTimer(self._timer_id)

    def timerEvent(self, event):
        """Update the menu on each iteration."""
        self.get_transfers_data()

    # pylint: enable=C0103

    @inlineCallbacks
    def get_transfers_data(self):
        """Get the transfers data to create the submenu."""
        data = yield self._parent.backend.sync_menu()
        self._update_transfers(data)

    def _update_transfers(self, data):
        """Generate the transfers menu."""
        current_transfers = data[RECENTTRANSFERS]
        current_upload = [
            filename for filename, _, _ in data[UPLOADING]]
        uploading_data = data[UPLOADING]
        uploading_data.sort(key=itemgetter(2))
        uploading_data.reverse()
        if current_transfers != self.previous_data.get('transfers') or \
           current_upload != self.previous_data.get('uploading'):
            self.clear()
            self.previous_data['transfers'] = current_transfers
            self.previous_data['uploading'] = current_upload
            self.uploading.clear()
            recent = self.addAction(RECENT_TRANSFERS)
            recent.setEnabled(False)
            for filename in data[RECENTTRANSFERS]:
                self.addAction(os.path.basename(filename))
            self.addSeparator()
            in_progress = self.addAction(IN_PROGRESS)
            in_progress.setEnabled(False)
            show_max = 0
            for filename, size, written in uploading_data:
                if show_max == 5:
                    break
                percentage = written * 100 / size
                self.uploading[filename] = ProgressBarAction(
                    filename, percentage)
                self.addAction(self.uploading[filename])
                show_max += 1
        else:
            for filename, size, written in uploading_data:
                percentage = written * 100 / size
                if filename in self.uploading:
                    self.uploading[filename].set_value(percentage)
