# -*- coding: utf-8 *-*

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The search and popup widgets for the Share Links tab."""

import os

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.platform import expand_user

from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import (
    NO_RESULTS,
    SEARCH_FOR,
    SHARE_THESE_FILES,
)

logger = setup_logging('qt.share_links_search')

HOME_DIR = ''
AVOID_SECOND_ITEM = 2
NORMAL_INCREMENT = 1


def get_system_icon_for_filename(file_path):
    """Return the icon used for the system to represent this file."""
    fileinfo = QtCore.QFileInfo(expand_user(file_path))
    icon_provider = QtGui.QFileIconProvider()
    icon = icon_provider.icon(fileinfo)
    return icon


# pylint: disable=C0103,W0603

class SearchBox(QtGui.QLineEdit, cache.Cache):
    """Search widget for the synced files."""

    itemSelected = QtCore.pyqtSignal(unicode)
    filesFound = QtCore.pyqtSignal(list, unicode)
    qtreactor = None

    def __init__(self, parent=None):
        super(SearchBox, self).__init__(parent)
        # The search box should be disabled until
        # the file list is obtained
        self.popup = FilesPopup()
        self.current_results = []
        self.page_index = 0
        self.page_size = 20
        self.pending_call = None
        self._post_key_event = {
            QtCore.Qt.Key_Escape: lambda *args: self.popup.hide(),
            QtCore.Qt.Key_Down: self._key_down_pressed,
            QtCore.Qt.Key_Up: self._key_up_pressed,
            QtCore.Qt.Key_Return: self._key_return_pressed,
            QtCore.Qt.Key_Enter: self._key_return_pressed,
        }

        self.popup.list_widget.itemPressed.connect(self._set_selected_item)
        self.popup.list_widget.verticalScrollBar().valueChanged.connect(
            self._scroll_fetch_more)
        self.textChanged.connect(self.handle_text_changed)

        self._get_home_path()

    def _scroll_fetch_more(self, value):
        """Fetch more items into the list on scroll."""
        if self.popup.list_widget.verticalScrollBar().maximum() == value:
            filenames = self._get_paginated_list(self.current_results)
            self.popup.add_items(filenames)

    @inlineCallbacks
    def _get_home_path(self):
        """Get the volumes info."""
        global HOME_DIR
        HOME_DIR = yield self.backend.get_home_dir()

    def handle_text_changed(self, text):
        """Use delayed IPC to search for filenames after user stops typing."""

        # Import here to avoid getting the wrong reactor due to import
        # order. Save in class var to ease patching for tests:
        if not self.qtreactor:
            self.qtreactor = __import__('twisted').internet.reactor
        text = unicode(text)
        self.page_index = 0

        if text == '':
            self.popup.hide()
            if self.pending_call and self.pending_call.active():
                self.pending_call.cancel()
            return

        if self.pending_call and self.pending_call.active():
            self.pending_call.reset(0.1)
            return

        self.pending_call = self.qtreactor.callLater(0.1, self._do_search)

    @inlineCallbacks
    def _do_search(self):

        self.pending_call = None

        search_text = unicode(self.text())
        results = yield self.backend.search_files(search_text)

        if search_text == unicode(self.text()):
            self.current_results = results
            self._load_items()

    def _load_items(self):
        """Load the items in the popup and display it."""
        if not self.popup.isVisible():
            self.popup.setFixedWidth(self.width())
            point = self.parent().mapToGlobal(self.pos())
            self.popup.show()
            self.popup.move(point.x(), point.y() + self.height())

        files = self._get_paginated_list(self.current_results)
        self.popup.load_items(files, unicode(self.text()))

    def _get_paginated_list(self, filenames):
        """Get pages of results."""
        begin = self.page_size * self.page_index
        self.page_index += 1
        return filenames[begin:begin + self.page_size]

    def _key_down_pressed(self, focus_index):
        """The user pressed the down key."""
        #While the current position is lower that the list size go to next
        if focus_index != self.popup.list_widget.count() - 1:
            list_offset = (AVOID_SECOND_ITEM if focus_index == 0 else
                NORMAL_INCREMENT)
            self.popup.list_widget.setCurrentRow(
                self.popup.list_widget.currentRow() + list_offset)
        #If the current position is greater than the amount of items in
        #the list - 6, then try to fetch more items in the list.
        if focus_index >= (self.popup.list_widget.count() - 6):
            filenames = self._get_paginated_list(self.current_results)
            self.popup.add_items(filenames)

    def _key_up_pressed(self, focus_index):
        """The user pressed the up key."""
        #while the current position is greater than 0, go to previous
        if focus_index > 0:
            list_offset = (AVOID_SECOND_ITEM if focus_index == 2 else
                NORMAL_INCREMENT)
            self.popup.list_widget.setCurrentRow(
                self.popup.list_widget.currentRow() - list_offset)

    def _key_return_pressed(self, current):
        """The user pressed the return key."""
        #If the user pressed enter, go to the item selected
        item = self.popup.list_widget.currentItem()
        self._set_selected_item(item)

    def keyPressEvent(self, event):
        """Process the different behaviour for the keyPress event."""
        super(SearchBox, self).keyPressEvent(event)
        current = self.popup.list_widget.currentRow()
        self._post_key_event.get(event.key(), lambda *args: None)(current)

    def focusOutEvent(self, event):
        """Hide the popup when the window loses the focus."""
        super(SearchBox, self).focusOutEvent(event)
        self.popup.hide()

    def _set_selected_item(self, item):
        """Notify of the selected item."""
        widget = self.popup.list_widget.itemWidget(item)
        if widget is not None:
            self.itemSelected.emit(widget.file_path)
            self.setText('')
        else:
            self.filesFound.emit(self.current_results, unicode(self.text()))
        self.popup.hide()

    def moveEvent(self, event):
        """Move the popup when the windows is moved."""
        super(SearchBox, self).moveEvent(event)
        if self.popup.isVisible():
            point = self.mapToGlobal(self.pos())
            self.popup.move(point.x(), point.y() + self.height())


class FileItem(QtGui.QLabel):
    """Create a styled QLabel that will show the info."""

    def __init__(self, file_path):
        super(FileItem, self).__init__()
        self.name = os.path.basename(file_path)
        self.file_path = file_path
        self.text_style = (u"<span style='color: {2};'>{0}</span><br>"
            "<span style='font-size: 13px; color: {3};'>({1})</span>")
        self.setText(self.text_style.format(
                self.name, self.reduced_path(self.file_path),
                '#333333', 'grey'))

    def reduced_path(self, file_path):
        """Replace the home part of the path with ~."""
        if file_path.startswith(HOME_DIR):
            reduce_path = file_path[len(HOME_DIR):]
            if reduce_path.startswith(os.path.sep):
                reduce_path = reduce_path[1:]
            file_path = os.path.join('~', reduce_path)
        return file_path

    def set_selected(self):
        """Set a style for the text when the item is selected."""
        self.setText(self.text_style.format(
                self.name, self.reduced_path(self.file_path),
                'white', 'white'))

    def set_not_selected(self):
        """Set a style for the text when the item is not selected."""
        self.setText(self.text_style.format(
                self.name, self.reduced_path(self.file_path),
                '#333333', 'grey'))


class FilesPopup(QtGui.QFrame):
    """Filter popup where the file names are shown."""

    popupShown = QtCore.pyqtSignal()
    popupHidden = QtCore.pyqtSignal()

    def __init__(self):
        super(FilesPopup, self).__init__(None,
            QtCore.Qt.FramelessWindowHint | QtCore.Qt.ToolTip)
        vbox = QtGui.QVBoxLayout(self)
        vbox.setContentsMargins(0, 0, 0, 0)
        vbox.setSpacing(0)
        self.list_widget = QtGui.QListWidget()
        self.list_widget.setMinimumHeight(270)
        vbox.addWidget(self.list_widget)

        self.list_widget.currentItemChanged.connect(self._repaint_items)

    def _repaint_items(self, current, previous):
        """Set the proper style for the current and previous items."""
        if current is not None:
            widget = self.list_widget.itemWidget(current)
            if widget is not None:
                widget.set_selected()
        if previous is not None:
            widget = self.list_widget.itemWidget(previous)
            if widget is not None:
                widget.set_not_selected()

    def _add_special_item(self, text):
        """Add special items to the popup."""
        item = QtGui.QListWidgetItem(text)
        font = item.font()
        font.setBold(True)
        item.setSizeHint(QtCore.QSize(20, 30))
        item.setBackground(QtGui.QBrush(QtCore.Qt.lightGray))
        item.setForeground(QtGui.QBrush(QtCore.Qt.black))
        item.setFont(font)
        return item

    def load_items(self, file_items, search_text):
        """Load the initial items."""
        self.list_widget.clear()
        # Search item
        self.list_widget.addItem(self._add_special_item(
            SEARCH_FOR % search_text))
        # Message item
        if len(file_items) > 0:
            message_publish = self._add_special_item(SHARE_THESE_FILES)
        else:
            message_publish = self._add_special_item(NO_RESULTS)
        self.list_widget.addItem(message_publish)
        for file_ in file_items:
            item = QtGui.QListWidgetItem("\n")
            file_widget = FileItem(file_)
            self.list_widget.addItem(item)
            self.list_widget.setItemWidget(item, file_widget)
            icon = get_system_icon_for_filename(file_.encode('utf-8'))
            item.setIcon(icon)
        self.list_widget.setCurrentRow(0)

    def add_items(self, file_items):
        """Add more items to the list on user scroll."""
        for file_ in file_items:
            item = QtGui.QListWidgetItem("\n")
            file_widget = FileItem(file_)
            self.list_widget.addItem(item)
            self.list_widget.setItemWidget(item, file_widget)
            icon = get_system_icon_for_filename(file_.encode('utf-8'))
            item.setIcon(icon)

    def showEvent(self, event):
        """Notify when the popup is shown."""
        super(FilesPopup, self).showEvent(event)
        self.popupShown.emit()

    def hideEvent(self, event):
        """Notify when the popup is hidden."""
        super(FilesPopup, self).hideEvent(event)
        self.popupHidden.emit()
