# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division

import os
import Queue
import sys
import threading

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.platform import is_link

from ubuntuone.controlpanel.utils import default_folders
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    ALWAYS_SUBSCRIBED,
    FOLDER_ICON_NAME,
    FOLDER_OWNED_BY,
    FOLDER_SHARED_BY,
    FOLDERS_BUTTON_ADD_FOLDER,
    FOLDERS_COLUMN_EXPLORE,
    FOLDERS_COLUMN_NAME,
    FOLDERS_COLUMN_SYNC_LOCALLY,
    FOLDERS_CONFIRM_MERGE,
    FOLDER_EXISTS_AS_FILE,
    FOLDERS_MANAGE_LABEL,
    GET_MORE_STORAGE,
    humanize,
    LOCAL_FOLDERS_CALCULATING,
    LOCAL_FOLDERS_FOLDER_HEADER,
    LOCAL_FOLDERS_OVERFLOW,
    LOCAL_FOLDERS_SPACE_HEADER,
    MANAGE_FILES_LINK,
    MUSIC_ICON_NAME,
    MUSIC_DISPLAY_NAME,
    MUSIC_REAL_PATH,
    NAME_NOT_SET,
    SHARE_ICON_NAME,
)
from ubuntuone.controlpanel.gui.qt import (
    handle_errors,
    icon_from_name,
    uri_hook,
)
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import folders_ui, local_folders_ui


logger = setup_logging('qt.folders')

FOLDER_NAME_COL = 0
SUBSCRIPTION_COL = 1
EXPLORE_COL = 2

LOCAL_SUBSCRIPTION_COL = 0
LOCAL_SPACE_COL = 1

CANCEL = QtGui.QMessageBox.Cancel
CHECKED = QtCore.Qt.Checked
CLOSE = QtGui.QMessageBox.Close
NO = QtGui.QMessageBox.No
UNCHECKED = QtCore.Qt.Unchecked
YES = QtGui.QMessageBox.Yes


def _process_name(name):
    """Tweak 'name' with a translatable music folder name."""
    if name.startswith('~/'):
        suggested = name[2:]
    else:
        suggested = name

    if suggested == MUSIC_REAL_PATH:
        result = MUSIC_DISPLAY_NAME
    else:
        result = suggested
    return result


class ExploreFolderButton(QtGui.QPushButton):
    """A specialized button for the folder listing."""

    def __init__(self, folder_path, parent=None):
        super(ExploreFolderButton, self).__init__(parent=parent)
        self.folder_path = folder_path
        self.setText(FOLDERS_COLUMN_EXPLORE)
        self.clicked.connect(self.on_clicked)

    def on_clicked(self):
        """Open the folder_path in the default file manager."""
        uri = unicode(QtCore.QUrl.fromLocalFile(self.folder_path).toString())
        uri_hook(uri)


class FoldersPanel(UbuntuOneBin):
    """The Folders Tab Panel widget."""

    logger = logger
    remote_folders = False
    ui_class = folders_ui
    widget_items = {}

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(FoldersPanel, self)._setup()
        self.ui.add_folder_button.folderCreated.connect(self.on_folder_created)
        self.ui.add_folder_button.setText(FOLDERS_BUTTON_ADD_FOLDER)
        self.ui.add_folder_button.add_folder_func = self.backend.create_folder

        self.ui.share_publish_button.setVisible(not self.remote_folders)
        self.ui.add_folder_button.setVisible(not self.remote_folders)
        self.ui.check_settings_button.setVisible(self.remote_folders)

        self.ui.folders.headerItem().setText(FOLDER_NAME_COL,
                                             FOLDERS_COLUMN_NAME)
        self.ui.folders.headerItem().setText(SUBSCRIPTION_COL,
                                             ALWAYS_SUBSCRIBED)
        self.ui.folders.headerItem().setText(EXPLORE_COL,
                                             FOLDERS_COLUMN_EXPLORE)
        headers = self.ui.folders.header()
        headers.setResizeMode(FOLDER_NAME_COL, headers.Stretch)
        headers.setResizeMode(SUBSCRIPTION_COL, headers.ResizeToContents)
        headers.setResizeMode(EXPLORE_COL, headers.ResizeToContents)

        self.ui.folders.setSortingEnabled(True)
        self.ui.folders.sortByColumn(FOLDER_NAME_COL, QtCore.Qt.AscendingOrder)

        self.ui.share_publish_button.setText(FOLDERS_MANAGE_LABEL)
        self.ui.share_publish_button.uri = MANAGE_FILES_LINK
        QtGui.QApplication.instance().focusChanged.connect(self.focus_changed)

    def focus_changed(self, old, new):
        """When a inner widget is focused, scroll to its item."""
        item = self.widget_items.get(new, None)
        if item is not None:
            self.ui.folders.scrollToItem(item)
            self.ui.folders.setCurrentItem(item)

    @log_call(logger.info)
    def on_folder_created(self, new_folder):
        """Reload folder info after folder creation."""
        self.is_processing = True
        self.load()

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        self.is_processing = True
        info = yield self.backend.volumes_info(with_storage_info=False,
                                               refresh=self.remote_folders)
        self.process_info(info)

    @handle_errors(logger=logger)
    @log_call(logger.debug)
    def process_info(self, info):
        """Load folders info into the tree view."""
        # FIXME: this is a bit of a hack for bug #942355
        # The correct solution is to not clear the list at all, but
        # the code would be much more complicated.
        scrollbar_position = self.ui.folders.verticalScrollBar().value()
        self.ui.folders.clear()
        self.widget_items = {}
        self.is_processing = False

        for name, _, volumes in info:  # ignore free_bytes
            if self.backend.NAME_NOT_SET in name:
                name = NAME_NOT_SET

            if name:
                name = FOLDER_SHARED_BY % {'other_user_display_name': name}
            else:
                name = FOLDER_OWNED_BY

            item = QtGui.QTreeWidgetItem()
            item.setText(FOLDER_NAME_COL, name)
            item.setText(SUBSCRIPTION_COL, FOLDERS_COLUMN_SYNC_LOCALLY)
            item.setSizeHint(FOLDER_NAME_COL, QtCore.QSize(-1, 20))
            brush = QtGui.QBrush(QtGui.QColor('#d6d3cf'))
            for col in (FOLDER_NAME_COL, SUBSCRIPTION_COL, EXPLORE_COL):
                item.setBackground(col, brush)
            item.is_empty = False
            self.ui.folders.addTopLevelItem(item)

            for volume in volumes:
                subscribed = bool(volume[u'subscribed'])

                is_root = volume[u'type'] == self.backend.ROOT_TYPE
                is_share = volume[u'type'] == self.backend.SHARE_TYPE

                child = QtGui.QTreeWidgetItem()
                child.setSizeHint(FOLDER_NAME_COL, QtCore.QSize(-1, 35))
                if is_share and 'realpath' in volume:
                    child.volume_path = volume['realpath']
                else:
                    child.volume_path = volume['path']
                child.volume_id = volume['volume_id']

                name = _process_name(volume.get(u'suggested_path',
                    volume[u'display_name']))
                child.setText(FOLDER_NAME_COL, name)
                child.setToolTip(FOLDER_NAME_COL, name)
                child.setToolTip(EXPLORE_COL, FOLDERS_COLUMN_EXPLORE)

                icon_name = FOLDER_ICON_NAME
                if is_share:
                    icon_name = SHARE_ICON_NAME
                elif name == MUSIC_DISPLAY_NAME:
                    icon_name = MUSIC_ICON_NAME

                icon = icon_from_name(icon_name)
                child.icon_obj = icon  # hack!
                child.setIcon(FOLDER_NAME_COL, icon)
                item.addChild(child)

                if is_root:
                    child.setText(SUBSCRIPTION_COL, ALWAYS_SUBSCRIBED)
                else:  # set check state
                    # We are using an embedded checkbox instead of the
                    # item's checkState, because of focus and navigation
                    # issues.
                    checkbox = QtGui.QCheckBox(parent=self.ui.folders)
                    checkbox.setMaximumWidth(24)
                    self.widget_items[checkbox] = child
                    if subscribed:
                        checkbox.setCheckState(CHECKED)
                    else:
                        checkbox.setCheckState(UNCHECKED)
                        pixmap = icon.pixmap(24, icon.Disabled, icon.Off)
                        icon = QtGui.QIcon(pixmap)
                        icon.icon_name = icon_name
                    self.ui.folders.setItemWidget(child, SUBSCRIPTION_COL,
                        checkbox)

                    # Operator not preceded by a space
                    # pylint: disable=C0322
                    cb = lambda checked, item=child: \
                        self.on_folders_itemChanged(item, SUBSCRIPTION_COL)
                    # pylint: enable=C0322

                    checkbox.stateChanged.connect(cb)

                if self.remote_folders:
                    # no explore button when showing only remote folders
                    continue

                # attach a third item with a button to explore the folder
                button = ExploreFolderButton(folder_path=child.volume_path,
                            parent=self.ui.folders)
                self.widget_items[button] = child
                button.setEnabled(subscribed)
                self.ui.folders.setItemWidget(child, EXPLORE_COL, button)

        self.ui.folders.expandAll()
        self.ui.folders.verticalScrollBar().setValue(scrollbar_position)

        # Rearrange the focus chain so that explore buttons
        # and checkboxes are right after ui.folders, and in
        # the displayed order.
        previous_widget = self.ui.folders
        it = QtGui.QTreeWidgetItemIterator(self.ui.folders)
        while it.value():
            item = it.value()
            checkbox = self.ui.folders.itemWidget(item, SUBSCRIPTION_COL)
            button = self.ui.folders.itemWidget(item, EXPLORE_COL)
            if checkbox:
                QtGui.QWidget.setTabOrder(previous_widget, checkbox)
                previous_widget = checkbox
            if button:
                QtGui.QWidget.setTabOrder(previous_widget, button)
                previous_widget = button
            it += 1

    # Invalid name "on_folders_itemActivated", "on_folders_itemChanged"
    # pylint: disable=C0103

    @handle_errors(logger=logger)
    def on_folders_itemActivated(self, item, column=None):
        """User activated a given row, open the path in a file browser."""
        volume_path = getattr(item, 'volume_path', None)
        if volume_path is None:
            logger.warning('on_folders_itemActivated: volume_path for item %r '
                           'and column %r is None.', item, column)
        elif not os.path.exists(volume_path):
            logger.warning('on_folders_itemActivated: stored path %r '
                           'does not exist.', volume_path)
        else:
            uri = unicode(QtCore.QUrl.fromLocalFile(volume_path).toString())
            uri_hook(uri)

    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def on_folders_itemChanged(self, item, column=None):
        """User changed the subscription for a given folder."""
        volume_id = item.volume_id
        volume_path = item.volume_path
        if volume_id is None:
            logger.warning('on_folders_itemChanged: volume_id for item %r '
                           'and column %r is None.', item, column)
            return

        if self.is_processing:
            # ignore signals when the UI is being updated
            return

        checkbox = self.ui.folders.itemWidget(item, SUBSCRIPTION_COL)
        if not checkbox:
            return
        subscribed = checkbox.checkState() == CHECKED  # new state
        logger.info('on_folders_itemChanged: processing volume id %r with '
                    'path %r, new subscribed value is: %r. Path exists? %r',
                    volume_id, volume_path, subscribed,
                    os.path.exists(volume_path))
        response = YES
        if subscribed and os.path.exists(volume_path):
            if os.path.isdir(volume_path) and not is_link(volume_path):
                text = FOLDERS_CONFIRM_MERGE % {'folder_path': volume_path}
                buttons = YES | NO | CANCEL
                response = QtGui.QMessageBox.warning(self, '', text, buttons,
                    YES)
            else:
                text = FOLDER_EXISTS_AS_FILE % {'folder_path': volume_path}
                buttons = CLOSE
                QtGui.QMessageBox.warning(self, '', text, buttons)
                response = NO

        self.is_processing = True

        if response == YES:
            # user accepted, merge the folder content
            yield self.backend.change_volume_settings(volume_id,
                                            {'subscribed': subscribed})
            self.load()
        else:
            # restore old value
            old = UNCHECKED if subscribed else CHECKED
            checkbox.setCheckState(old)

        self.is_processing = False


class RemoteFoldersPanel(FoldersPanel):
    """The Folders Panel that only shows remote cloud folders."""

    remote_folders = True


class CalculateSize(threading.Thread):
    """A thread that calculates, in the background, the size of a folder."""

    def __init__(self, path_name, queue):
        self.queue = queue
        self._stop = False
        # This makes os.walk use the "bytes" version, which doesn't
        # break with invalid unicode paths.
        # This will only work for unicode locales (ex: LANG=es_ES.UTF-8)
        # and will falback to as before for C and non-unicode locales.
        try:
            self.path_name = path_name.encode(sys.getfilesystemencoding())
        except (UnicodeEncodeError, UnicodeDecodeError):
            # Should never happen (haha)
            self.path_name = path_name

        super(CalculateSize, self).__init__()

        # http://docs.python.org/library/threading.html#threading.Thread.daemon
        # "A boolean value indicating whether this thread is a daemon thread
        # (True) or not (False)"
        self.daemon = True

    def run(self):
        """Run this thread."""
        logger.debug('size_calculator: about to calculate size for %r.',
                     self.path_name)
        try:
            total_size = 0
            for dirpath, _, filenames in os.walk(self.path_name):
                for f in filenames:
                    fp = os.path.join(dirpath, f)
                    if os.path.isfile(fp):
                        total_size += os.path.getsize(fp)
                if self._stop:
                    logger.warning('size_calculator: stopping due to external '
                                   'request.')
                    return
        except:  # pylint: disable=W0702
            logger.exception('size_calculator: failed for %r:', self.path_name)
        else:
            logger.info('size_calculator: added new size %r for %r.',
                         self.path_name, total_size)
            self.queue.put([self.path_name, total_size])


class FolderItem(QtGui.QTreeWidgetItem):
    """A folder in the local folder list."""

    def __init__(self, values=None, path=None, queue=None, volume_id=None):
        super(FolderItem, self).__init__(values)
        self.path = path
        self.volume_id = volume_id
        self.thread = None
        self.size = 0

        state = UNCHECKED
        if path is not None:
            if volume_id is not None:
                state = CHECKED
            elif queue is not None:
                # calculate sizes of non-existing folders
                self.thread = CalculateSize(path, queue)
                self.thread.start()
                self.size = None

        self.setCheckState(LOCAL_SUBSCRIPTION_COL, state)


class LocalFoldersPanel(UbuntuOneBin):
    """The panel that only shows local, non-synched folders."""

    changesApplied = QtCore.pyqtSignal()
    logger = logger
    ui_class = local_folders_ui

    def __init__(self, *args, **kwargs):
        timer = kwargs.pop('timer', None)
        if timer is None:
            timer = QtCore.QTimer()

        queue = kwargs.pop('queue', None)
        if queue is None:
            queue = Queue.Queue()

        super(LocalFoldersPanel, self).__init__(*args, **kwargs)

        self.queue = queue
        self.timer = timer
        self.timer.timeout.connect(self.update_sizes)

        self.items = {}
        self.user_home = None
        self.account_info = None

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(LocalFoldersPanel, self)._setup()
        # Start with storage upgrade offer invisible
        self.ui.offer_frame.setVisible(False)

        headers = self.ui.folders.header()
        headers.setResizeMode(LOCAL_SUBSCRIPTION_COL, headers.Stretch)
        headers.setResizeMode(LOCAL_SPACE_COL, headers.ResizeToContents)

        self.ui.folders.headerItem().setText(LOCAL_SUBSCRIPTION_COL,
                                             LOCAL_FOLDERS_FOLDER_HEADER)
        self._set_space_header()

        self.ui.add_folder_button.folderCreated.connect(self.on_folder_created)
        self.ui.add_folder_button.setText(FOLDERS_BUTTON_ADD_FOLDER)
        self.ui.add_folder_button.add_folder_func = self.add_folder

        self.ui.add_storage_button.setText(GET_MORE_STORAGE)

    def _set_space_header(self, total=None):
        """Set the folders listing 'space' header."""
        if total is None:
            total = ''
        else:
            try:
                total = humanize(long(total))
            except (TypeError, ValueError):
                pass
            total = '(%s)' % total

        title = LOCAL_FOLDERS_SPACE_HEADER.format(space_total=total)
        self.ui.folders.headerItem().setText(LOCAL_SPACE_COL, title)

    def _stop(self):
        """Stop all pending threads and timers."""
        self.timer.stop()

        for item in self.items.itervalues():
            if item.thread is None:
                logger.warning('LocalFoldersPanel: attempted to stop a thread '
                               'for an item with a None thread.')
            else:
                item.thread._stop = True

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        self.is_processing = True
        self.user_home = yield self.backend.get_home_dir()
        self.account_info = yield self.backend.account_info()
        self._set_space_header(self.account_info['quota_used'])
        volumes_info = yield self.backend.volumes_info(with_storage_info=False)
        yield self.process_info(volumes_info)

    @defer.inlineCallbacks
    @log_call(logger.debug)
    def process_info(self, volumes_info):
        """Load local folders info into the tree view."""
        try:
            folders = []
            for _, _, volumes in volumes_info:
                for volume in volumes:
                    if (volume[u'type'] == self.backend.FOLDER_TYPE and
                        bool(volume['subscribed'])):
                        suggested = volume.get(u'suggested_path',
                            volume[u'display_name'])
                        folders.append((volume['path'], volume['volume_id'],
                                        suggested))

            # add local folders only if they are valid
            for folder in default_folders(user_home=self.user_home):
                is_valid = yield self.backend.validate_path_for_folder(folder)
                if is_valid:
                    folders.append((folder, None, None))

            # always clear the items dict first, since clearing the folders
            # list will trigger "underlying C/C++ object has been deleted"
            self.items.clear()
            self.ui.folders.clear()

            for path, volume_id, suggested in folders:
                self.add_folder(folder_path=path, volume_id=volume_id,
                    suggested=suggested)

            if folders and not self.timer.isActive():
                self.timer.start(2000)
        finally:
            self.is_processing = False

    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def apply_changes(self):
        """When moving to next page, create/[un]subscribe UDFs."""
        self.is_processing = True
        try:
            self._stop()

            for path, item in self.items.iteritems():
                subscribed = item.checkState(LOCAL_SUBSCRIPTION_COL) == CHECKED
                if item.volume_id is not None:
                    logger.info('apply_changes: change settings for %r to %r.',
                                item.path, dict(subscribed=subscribed))
                    yield self.backend.change_volume_settings(item.volume_id,
                            dict(subscribed=subscribed))
                else:
                    if subscribed:
                        logger.info('apply_changes: create folder for %r.',
                                    path)
                        yield self.backend.create_folder(path)
        finally:
            self.is_processing = False

        self.changesApplied.emit()

    @handle_errors(logger=logger)
    def add_folder(self, folder_path, volume_id=None, suggested=None):
        """Add a folder to the list."""
        if folder_path in self.items:
            logger.warning('LocalFoldersPanel: already have an item for %r.',
                           folder_path)
            return

        if suggested is None:
            logger.warning('LocalFoldersPanel: user home is None! '
                           'paths will not be pretty.')
            if self.user_home is None:
                user_home = ''
            else:
                user_home = self.user_home + os.path.sep
            display_name = _process_name(folder_path.replace(user_home, ''))
        else:
            display_name = _process_name(suggested)

        item = FolderItem([display_name, ""], path=folder_path,
                          queue=self.queue, volume_id=volume_id)
        self.ui.folders.addTopLevelItem(item)

        if volume_id is None:  # new folder
            self.items[folder_path] = item

    @log_call(logger.info)
    def on_folder_created(self, new_folder):
        """User clicked on the "Add Folder" button."""
        item = self.items.get(unicode(new_folder))
        if item is not None:
            item.setCheckState(LOCAL_SUBSCRIPTION_COL, CHECKED)
        else:
            logger.warning('LocalFoldersPanel: on_folder_created was called '
                           'for %r which is not tracked in the internal dict',
                           unicode(new_folder))

    @handle_errors(logger=logger)
    def update_sizes(self):
        """Poll the queue were the threads put the size info.

        Every item put in this queue will be a non-synched folder. Thus, the
        item's volume_id will be None, so no need to check that.

        """
        while True:
            try:
                path, size = self.queue.get(block=False)
            except Queue.Empty:
                break
            else:
                item = self.items.get(path)
                if item:
                    item.size = size
                    item.setText(LOCAL_SPACE_COL, humanize(size))

        total = long(self.account_info['quota_used'])
        for path, item in self.items.iteritems():
            if item.size is None:
                total = LOCAL_FOLDERS_CALCULATING
                break

            subscribed = item.checkState(LOCAL_SUBSCRIPTION_COL) == CHECKED
            if subscribed:
                total += item.size

        if isinstance(total, long):
            self.show_hide_offer(total)
        else:
            self.show_hide_offer(0)

        self._set_space_header(total)

    def show_hide_offer(self, current_size):
        """Show or hide the offer to buy space according to the total size."""
        quota = long(self.account_info['quota_total'])
        if current_size > quota:
            msg = LOCAL_FOLDERS_OVERFLOW.format(quota_total=humanize(quota))
            self.ui.offer_label.setText(msg)
            self.ui.offer_frame.setVisible(True)
        else:
            self.ui.offer_frame.setVisible(False)

    # pylint: disable=C0103

    def on_folders_itemChanged(self, item, column):
        """Update the size for the chosen row."""
        if column == LOCAL_SUBSCRIPTION_COL:
            if item.volume_id is None:
                self.update_sizes()
            self.items[item.path] = item

    # pylint: enable=C0103
