# -*- coding: utf-8 -*-

# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the service when accessing the login client."""

from twisted.internet import defer

from ubuntuone.controlpanel import login_client
from ubuntuone.controlpanel.tests import CustomError, TestCase, TOKEN


class FakedCredentialsManagementTool(object):
    """Fake the CredentialsManagementTool."""

    credentials = None

    @defer.inlineCallbacks
    def find_credentials(self):
        """Find credentials for Ubuntu One."""
        yield
        defer.returnValue(FakedCredentialsManagementTool.credentials)

    @defer.inlineCallbacks
    def clear_credentials(self):
        """Clear credentials for Ubuntu One."""
        FakedCredentialsManagementTool.credentials = None
        yield

    @defer.inlineCallbacks
    def login(self):
        """Create credentials for Ubuntu One by logging in."""
        yield defer.succeed(FakedCredentialsManagementTool.credentials)

    @defer.inlineCallbacks
    def register(self):
        """Create credentials for Ubuntu One by registering."""
        yield defer.succeed(FakedCredentialsManagementTool.credentials)


class BaseTestCase(TestCase):
    """Base TestCase for the login client methods."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(BaseTestCase, self).setUp()
        self.patch(login_client, 'CredentialsManagementTool',
                   FakedCredentialsManagementTool)
        FakedCredentialsManagementTool.credentials = TOKEN

    def fake_fail(self, *a):
        """Fake a failure."""
        return defer.fail(CustomError(a))


class FindCredentialsTestCase(BaseTestCase):
    """Test for the find_credentials method."""

    @defer.inlineCallbacks
    def test_find_credentials(self):
        """The credentials are properly retrieved."""
        actual = yield login_client.get_credentials()
        self.assertEqual(actual, TOKEN)

    @defer.inlineCallbacks
    def test_find_credentials_throws_an_error(self):
        """If find_credentials fails, the error is propagated."""
        self.patch(FakedCredentialsManagementTool, 'find_credentials',
                   self.fake_fail)
        yield self.assertFailure(login_client.get_credentials(),
                                 CustomError)


class ClearCredentialsTestCase(BaseTestCase):
    """Test for the clear_credentials method."""

    @defer.inlineCallbacks
    def test_clear_credentials(self):
        """The credentials are properly cleared."""
        yield login_client.clear_credentials()
        self.assertEqual(None, FakedCredentialsManagementTool.credentials)

    @defer.inlineCallbacks
    def test_clear_credentials_throws_an_error(self):
        """If clear_credentials fails, the error is propagated."""
        self.patch(FakedCredentialsManagementTool, 'clear_credentials',
                   self.fake_fail)
        yield self.assertFailure(login_client.clear_credentials(),
                                 CustomError)


class LoginTestCase(BaseTestCase):
    """Test for the login method."""

    @defer.inlineCallbacks
    def test_login(self):
        """The credentials are properly retrieved."""
        yield login_client.login()
        self.assertEqual(TOKEN, FakedCredentialsManagementTool.credentials)

    @defer.inlineCallbacks
    def test_login_throws_an_error(self):
        """If login fails, the error is propagated."""
        self.patch(FakedCredentialsManagementTool, 'login',
                   self.fake_fail)
        yield self.assertFailure(login_client.login(),
                                 CustomError)


class RegisterTestCase(BaseTestCase):
    """Test for the register method."""

    @defer.inlineCallbacks
    def test_register(self):
        """The credentials are properly retrieved."""
        yield login_client.register()
        self.assertEqual(TOKEN, FakedCredentialsManagementTool.credentials)

    @defer.inlineCallbacks
    def test_register_throws_an_error(self):
        """If register fails, the error is propagated."""
        self.patch(FakedCredentialsManagementTool, 'register',
                   self.fake_fail)
        yield self.assertFailure(login_client.register(),
                                 CustomError)
