# -*- coding: utf-8 -*-
#
# Copyright 2010-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The Qt graphical interface for the control panel for Ubuntu One."""

import collections
import logging

from functools import wraps

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel import backend
from ubuntuone.controlpanel.gui import (
    ERROR_COLOR,
    FILE_SYNC_CONNECT,
    FILE_SYNC_CONNECT_TOOLTIP,
    FILE_SYNC_DISABLED,
    FILE_SYNC_DISCONNECT,
    FILE_SYNC_DISCONNECT_TOOLTIP,
    FILE_SYNC_DISCONNECTED,
    FILE_SYNC_ENABLE,
    FILE_SYNC_ENABLE_TOOLTIP,
    FILE_SYNC_ERROR,
    FILE_SYNC_IDLE,
    FILE_SYNC_RESTART,
    FILE_SYNC_RESTART_TOOLTIP,
    FILE_SYNC_START,
    FILE_SYNC_START_TOOLTIP,
    FILE_SYNC_STARTING,
    FILE_SYNC_STOP,
    FILE_SYNC_STOP_TOOLTIP,
    FILE_SYNC_STOPPED,
    FILE_SYNC_SYNCING,
    FILE_URI_PREFIX,
    GENERAL_ERROR_TITLE,
    GENERAL_ERROR_MSG,
)


WARNING_MARKUP = '<font color="%s"><b>%%s</b></font>' % ERROR_COLOR
DISCONNECTED_ICON = 'sync_status_disconnected'
ERROR_ICON = 'sync_status_alert'
IDLE_ICON = 'sync_status_sync_done'
SYNCING_ICON = 'sync_status_syncing'


FILE_SYNC_STATUS = {
    backend.FILE_SYNC_DISABLED:
        {'msg': FILE_SYNC_DISABLED, 'action': FILE_SYNC_ENABLE,
         'backend_method': 'enable_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_ENABLE_TOOLTIP},
    backend.FILE_SYNC_DISCONNECTED:
        {'msg': FILE_SYNC_DISCONNECTED, 'action': FILE_SYNC_CONNECT,
         'backend_method': 'connect_files',
         'icon': DISCONNECTED_ICON,
         'tooltip': FILE_SYNC_CONNECT_TOOLTIP},
    backend.FILE_SYNC_ERROR:
        {'msg': WARNING_MARKUP % FILE_SYNC_ERROR, 'action': FILE_SYNC_RESTART,
         'backend_method': 'restart_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_RESTART_TOOLTIP},
    backend.FILE_SYNC_IDLE:
        {'msg': FILE_SYNC_IDLE, 'action': FILE_SYNC_DISCONNECT,
         'backend_method': 'disconnect_files',
         'icon': IDLE_ICON,
         'tooltip': FILE_SYNC_DISCONNECT_TOOLTIP},
    backend.FILE_SYNC_STARTING:
        {'msg': FILE_SYNC_STARTING, 'action': FILE_SYNC_STOP,
         'backend_method': 'stop_files',
         'icon': SYNCING_ICON,
         'tooltip': FILE_SYNC_STOP_TOOLTIP},
    backend.FILE_SYNC_STOPPED:
        {'msg': FILE_SYNC_STOPPED, 'action': FILE_SYNC_START,
         'backend_method': 'start_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_START_TOOLTIP},
    backend.FILE_SYNC_SYNCING:
        {'msg': FILE_SYNC_SYNCING, 'action': FILE_SYNC_DISCONNECT,
         'backend_method': 'disconnect_files',
         'icon': SYNCING_ICON,
         'tooltip': FILE_SYNC_DISCONNECT_TOOLTIP},
    backend.FILE_SYNC_UNKNOWN:
        {'msg': WARNING_MARKUP % FILE_SYNC_ERROR, 'action': FILE_SYNC_RESTART,
         'backend_method': 'restart_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_RESTART_TOOLTIP},
}


def icon_name_from_status(status_key):
    """Get the icon name for the status."""
    icon_name = FILE_SYNC_STATUS[status_key]['icon']
    return icon_name


def force_wordwrap(widget, size, text):
    """Set the text to the widget after word wrapping it."""
    font_metrics = QtGui.QFontMetrics(widget.font())
    text_width = font_metrics.width(text)
    if size > 0 and text_width > size:
        words = text.split(' ')
        text = ''
        size_left = size
        for word in words:
            word_size = font_metrics.width(word)
            if (size_left - word_size) > 0 or word_size > size:
                text = ''.join((text, '%s ' % word))
                size_left -= word_size
            else:
                text = ''.join((text, '\n%s ' % word))
                size_left = size - word_size
        text = text.rstrip()
    widget.setText(text)


def uri_hook(uri):
    """Open an URI using the default browser/file manager."""
    if uri.startswith(FILE_URI_PREFIX):
        url = QtCore.QUrl(uri)
    else:
        url = QtCore.QUrl()
        url.setEncodedUrl(uri)
    QtGui.QDesktopServices.openUrl(url)


def pixmap_from_name(icon_name):
    """Build a QtGui.QPixmap from an icon name.

    If the theme does not have the icon, build one using the resources images.

    """
    pixmap_name = ":/%s.png" % icon_name
    return QtGui.QPixmap(pixmap_name)


def icon_from_name(icon_name):
    """Build a QtGui.QIcon from an icon name.

    If the theme does not have the icon, build one using the resources images.

    """
    if QtGui.QIcon.hasThemeIcon(icon_name):
        icon = QtGui.QIcon.fromTheme(icon_name)
    else:
        pixmap = pixmap_from_name(icon_name)
        icon = QtGui.QIcon(pixmap)

    icon.icon_name = icon_name

    return icon


# Invalid name "box"
# pylint: disable=C0103
box = None
# pylint: enable=C0103


def handle_errors(error_handler=None, logger=None):
    """Decorator to handle errors when calling a function.

    if 'error_handler' is not None, it will be yielded on if any error happens.

    """
    if logger is None:
        logger = logging.getLogger()

    def middle(f):
        """Decorator to handle errors when calling 'f'."""

        @defer.inlineCallbacks
        @wraps(f)
        def inner(*args, **kwargs):
            """Call 'f' passing 'args' and 'kwargs'.

            Catch any error and show a error message.

            """
            try:
                res = yield f(*args, **kwargs)
            except Exception, e:  # pylint: disable=W0703
                msg = 'Error while invoking %r with args %r and kwargs %r:'
                logger.exception(msg, f, args, kwargs)
            else:
                defer.returnValue(res)

            # this code will only be executed if there was an exception
            if error_handler is not None:
                yield error_handler()

            if len(e.args) > 0 and isinstance(e.args[0], collections.Mapping):
                msgs = e.args[0].itervalues()
            else:
                msgs = [e.__class__.__name__] + map(repr, e.args)
            msg = '\n'.join(msgs)

            # hack to avoid the QMessageBox being gargabe-collected too soon
            # Using the global statement, pylint: disable=W0603
            global box
            # pylint: enable=W0603
            box = QtGui.QMessageBox(QtGui.QMessageBox.Critical,
                    GENERAL_ERROR_TITLE, GENERAL_ERROR_MSG,
                    QtGui.QMessageBox.Close, parent=None)
            box.setDefaultButton(QtGui.QMessageBox.Close)
            box.setDetailedText(msg)

            # can not exec_() on the QMessageBox since that will block the Qt
            # mainloop

            d = defer.Deferred()

            def box_finished(result):
                """The QMessageBox finished."""
                logger.info('The warning dialog was shown and also closed '
                            '(message was %r).', msg)
                box.finished.disconnect(box_finished)
                box.hide()
                box.deleteLater()
                d.callback(result)

            box.finished.connect(box_finished)
            box.show()

            yield d

        return inner

    return middle
