# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division

import os

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    ALWAYS_SUBSCRIBED,
    EXPLORE,
    FOLDER_ICON_NAME,
    FOLDER_OWNED_BY,
    FOLDER_SHARED_BY,
    FOLDERS_CONFIRM_MERGE,
    MANAGE_FILES_LINK,
    MUSIC_ICON_NAME,
    MUSIC_DISPLAY_NAME,
    MUSIC_REAL_PATH,
    NAME_NOT_SET,
    SHARE_ICON_NAME,
    SYNC_LOCALLY,
)
from ubuntuone.controlpanel.gui.qt import uri_hook, icon_from_name
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import folders_ui


logger = setup_logging('qt.folders')

FOLDER_NAME_COL = 0
SUBSCRIPTION_COL = 1
EXPLORE_COL = 2

CANCEL = QtGui.QMessageBox.Cancel
CHECKED = QtCore.Qt.Checked
CLOSE = QtGui.QMessageBox.Close
NO = QtGui.QMessageBox.No
UNCHECKED = QtCore.Qt.Unchecked
YES = QtGui.QMessageBox.Yes


class FoldersPanel(UbuntuOneBin):
    """The Folders Tab Panel widget"""

    ui_class = folders_ui
    logger = logger

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(FoldersPanel, self)._setup()
        self.ui.add_folder_button.folderCreated.connect(self.on_folder_created)

        headers = self.ui.folders.header()
        headers.setResizeMode(FOLDER_NAME_COL, headers.Stretch)
        headers.setResizeMode(SUBSCRIPTION_COL, headers.ResizeToContents)
        headers.setResizeMode(EXPLORE_COL, headers.ResizeToContents)
        headers.setStretchLastSection(False)

        self.ui.share_publish_button.uri = MANAGE_FILES_LINK
        icon = icon_from_name('external_icon_orange')
        self.ui.share_publish_button.setIcon(icon)

    @log_call(logger.info)
    def on_folder_created(self, new_folder):
        """Reload folder info after folder creation."""
        self.is_processing = True
        # hack to ensure that syncdaemon updates the folder list.
        # pylint: disable=W0404, E1101
        from twisted.internet import reactor
        reactor.callLater(2, self.load)

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        self.is_processing = True
        info = yield self.backend.volumes_info(with_storage_info=False)
        self.process_info(info)

    def _process_name(self, name):
        """Tweak 'name' with a translatable music folder name."""
        if name == MUSIC_REAL_PATH:
            result = MUSIC_DISPLAY_NAME
        else:
            result = name
        return result

    @log_call(logger.debug)
    def process_info(self, info):
        """Load folders info into the tree view."""
        self.ui.folders.clear()
        self.is_processing = False

        for name, _, volumes in info:  # ignore free_bytes
            if self.backend.NAME_NOT_SET in name:
                name = NAME_NOT_SET

            if name:
                name = FOLDER_SHARED_BY % {'other_user_display_name': name}
            else:
                name = FOLDER_OWNED_BY

            item = QtGui.QTreeWidgetItem()
            item.setText(FOLDER_NAME_COL, name)
            item.setText(SUBSCRIPTION_COL, SYNC_LOCALLY)
            item.setSizeHint(FOLDER_NAME_COL, QtCore.QSize(-1, 20))
            brush = QtGui.QBrush(QtGui.QColor('#d6d3cf'))
            for col in (FOLDER_NAME_COL, SUBSCRIPTION_COL, EXPLORE_COL):
                item.setBackground(col, brush)
            item.is_empty = False
            self.ui.folders.addTopLevelItem(item)

            for volume in volumes:
                is_root = volume[u'type'] == self.backend.ROOT_TYPE
                is_share = volume[u'type'] == self.backend.SHARE_TYPE

                child = QtGui.QTreeWidgetItem()
                child.setSizeHint(FOLDER_NAME_COL, QtCore.QSize(-1, 35))
                if is_share and 'realpath' in volume:
                    child.volume_path = volume['realpath']
                else:
                    child.volume_path = volume['path']
                child.volume_id = volume['volume_id']

                name = self._process_name(volume[u'display_name'])
                child.setText(FOLDER_NAME_COL, name)
                child.setToolTip(FOLDER_NAME_COL, name)
                child.setToolTip(EXPLORE_COL, EXPLORE)

                icon_name = FOLDER_ICON_NAME
                if is_share:
                    icon_name = SHARE_ICON_NAME
                elif name == MUSIC_DISPLAY_NAME:
                    icon_name = MUSIC_ICON_NAME

                icon = icon_from_name(icon_name)

                if is_root:
                    child.setText(SUBSCRIPTION_COL, ALWAYS_SUBSCRIBED)
                else:  # set check state
                    if bool(volume[u'subscribed']):
                        child.setCheckState(SUBSCRIPTION_COL, CHECKED)
                    else:
                        child.setCheckState(SUBSCRIPTION_COL, UNCHECKED)
                        pixmap = icon.pixmap(24, icon.Disabled, icon.Off)
                        icon = QtGui.QIcon(pixmap)
                        icon.icon_name = icon_name

                child.icon_obj = icon  # hack!
                child.setIcon(FOLDER_NAME_COL, icon)
                item.addChild(child)

                # attach a third item with a button to explore the folder
                model_index = self.ui.folders.indexFromItem(child, EXPLORE_COL)
                button = QtGui.QPushButton(parent=self.ui.folders)
                button.setFlat(True)
                icon = icon_from_name(FOLDER_ICON_NAME)
                button.icon_obj = icon  # hack!
                button.setIcon(icon_from_name(FOLDER_ICON_NAME))
                button.setIconSize(QtCore.QSize(12, 12))
                button.setText(EXPLORE)
                button.setObjectName('explore_folder_button')
                policy = QtGui.QSizePolicy(QtGui.QSizePolicy.Fixed,
                                           QtGui.QSizePolicy.Fixed)
                button.setSizePolicy(policy)
                button.setEnabled(bool(volume[u'subscribed']))

                # Operator not preceded by a space
                # pylint: disable=C0322
                cb = lambda checked, item=child: \
                    self.on_folders_itemActivated(item)
                # pylint: enable=C0322
                button.clicked.connect(cb)
                self.ui.folders.setIndexWidget(model_index, button)

        self.ui.folders.expandAll()

    # Invalid name "on_folders_itemActivated", "on_folders_itemChanged"
    # pylint: disable=C0103

    def on_folders_itemActivated(self, item, column=None):
        """User activated a given row, open the path in a file browser."""
        volume_path = getattr(item, 'volume_path', None)
        if volume_path is None:
            logger.warning('on_folders_itemActivated: volume_path for item %r '
                           'and column %r is None.', item, column)
        elif not os.path.exists(volume_path):
            logger.warning('on_folders_itemActivated: stored path %r '
                           'does not exist.', volume_path)
        else:
            uri = unicode(QtCore.QUrl.fromLocalFile(volume_path).toString())
            uri_hook(uri)

    @defer.inlineCallbacks
    def on_folders_itemChanged(self, item, column=None):
        """User changed the subscription for a given folder."""
        volume_id = item.volume_id
        volume_path = item.volume_path
        if volume_id is None:
            logger.warning('on_folders_itemChanged: volume_id for item %r '
                           'and column %r is None.', item, column)
            return

        if self.is_processing:
            # ignore signals when the UI is being updated
            return

        subscribed = item.checkState(SUBSCRIPTION_COL) == CHECKED  # new state
        logger.info('on_folders_itemChanged: processing volume id %r with '
                    'path %r, new subscribed value is: %r. Path exists? %r',
                    volume_id, volume_path, subscribed,
                    os.path.exists(volume_path))
        response = YES
        if subscribed and os.path.exists(volume_path):
            text = FOLDERS_CONFIRM_MERGE % {'folder_path': volume_path}
            buttons = YES | NO | CANCEL
            response = QtGui.QMessageBox.warning(self, '', text, buttons, YES)

        self.is_processing = True

        if response == YES:
            # user accepted, merge the folder content
            yield self.backend.change_volume_settings(volume_id,
                                                {'subscribed': subscribed})
            self.load()
        else:
            # restore old value
            old = UNCHECKED if subscribed else CHECKED
            item.setCheckState(SUBSCRIPTION_COL, old)

        self.is_processing = False
