# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

# Unused import QtGui
# pylint: disable=W0611
from PyQt4 import QtGui, QtCore
# pylint: enable=W0611
from twisted.internet import defer

from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import EDIT_DEVICES_LINK
from ubuntuone.controlpanel.gui.qt.ui import devices_ui
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt import device


logger = setup_logging('qt.devices')


class DevicesPanel(UbuntuOneBin):
    """The DevicesFolders Tab Panel widget"""

    ui_class = devices_ui
    logger = logger

    localDeviceRemoved = QtCore.pyqtSignal()

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(DevicesPanel, self)._setup()
        self.ui.manage_devices_button.uri = EDIT_DEVICES_LINK

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        info = yield self.backend.device_names_info()
        self.process_info(info)

    @log_call(logger.debug)
    def process_info(self, info):
        """Process and display the devices info."""
        self.clear_device_info(self.ui.local_device_box)
        self.ui.list_devices.clear()

        for device_info in info:
            self.update_device_info(device_info)

        self.is_processing = False

    def on_local_device_removed(self):
        """When the local device is removed, clear the box and emit signal."""
        self.clear_device_info(self.ui.local_device_box)
        self.localDeviceRemoved.emit()

    def clear_device_info(self, box):
        """Clear all the device info."""
        children = box.count()
        # we need to reverse the index list to remove children because:
        # "Items are numbered consecutively from 0. If an item is deleted,
        # other items will be renumbered."
        # http://doc.qt.nokia.com/latest/qlayout.html#itemAt
        for i in reversed(range(children)):
            widget = box.itemAt(i).widget()
            box.removeWidget(widget)

    def update_device_info(self, device_info):
        """Update one device."""
        if device_info["is_local"]:
            self.update_local_device(device_info)
        else:
            self.create_remote_device(device_info)

    def update_local_device(self, device_info):
        """Update the info for the local device."""
        device_widget = device.DeviceWidget(device_id=device_info['device_id'])
        device_widget.update_device_info(device_info)
        device_widget.removed.connect(self.on_local_device_removed)

        self.ui.local_device_box.addWidget(device_widget)

    def create_remote_device(self, device_info):
        """Add a remote device to the list."""
        item = device.get_device_for_list_widget(device_info)
        self.ui.list_devices.addItem(item)
