# -*- coding: utf-8 -*-

# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division

import os

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import FOLDER_INVALID_PATH
from ubuntuone.controlpanel.gui.qt import handle_errors


logger = setup_logging('qt.addfolder')

CLOSE = QtGui.QMessageBox.Close


class AddFolderButton(cache.Cache, QtGui.QPushButton):
    """The AddFolderButton widget"""

    logger = logger

    folderCreated = QtCore.pyqtSignal(unicode)
    folderCreationCanceled = QtCore.pyqtSignal()

    def __init__(self, *args, **kwargs):
        """Initialize the UI of the widget."""
        super(AddFolderButton, self).__init__(*args, **kwargs)
        self.cloud_folders = []
        self.clicked.connect(self.on_clicked)

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def on_clicked(self):
        """The 'Sync another folder' button was clicked."""
        # The options argument is because of LP: #835013
        folder = QtGui.QFileDialog.getExistingDirectory(
            parent=self, options=QtGui.QFileDialog.DontUseNativeDialog)
        folder = unicode(QtCore.QDir.toNativeSeparators(folder))
        logger.debug('on_add_folder_button_clicked: user requested folder '
                     'creation for path %r', folder)
        if folder == '':
            self.folderCreationCanceled.emit()
            return

        is_valid = yield self.backend.validate_path_for_folder(folder)
        if not is_valid:
            user_home = os.path.expanduser('~')
            text = FOLDER_INVALID_PATH % {'folder_path': folder,
                                          'home_folder': user_home}
            QtGui.QMessageBox.warning(self, '', text, CLOSE)
            return

        yield self.backend.create_folder(folder_path=folder)
        self.folderCreated.emit(folder)
