# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the control panel user interface (basics)."""

from __future__ import division

from twisted.internet import defer
from twisted.python.failure import Failure

from ubuntuone.controlpanel.gui.gtk import gui
from ubuntuone.controlpanel.gui.gtk.tests import (
    BaseTestCase,
    FakedCredentialsBackend,
)
from ubuntuone.controlpanel.tests import TOKEN

from ubuntuone.devtools.testcase import skipIf

# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212


class FakeUnity(object):
    """Fake Unity module."""


class FakeLauncherEntryProps(object):
    """A fake Unity.LauncherEntry.props"""

    urgent = True


THE_FLEP = FakeLauncherEntryProps()


class FakeLauncherEntry(object):
    """A fake Unity.LauncherEntry"""

    def __init__(self):
        """Initialize this fake instance."""
        self.props = THE_FLEP

    @staticmethod
    def get_for_desktop_id(dotdesktop):
        """Find the LauncherEntry for a given dotdesktop."""
        return FakeLauncherEntry()


class FakeControlPanelService(object):
    """Fake service."""
    def __init__(self, window):
        self.window = window
        self.called = []

    def draw_attention(self):
        """Draw attention to the control panel."""
        self.called.append('draw_attention')

    def switch_to(self, panel):
        """Switch to named panel."""
        self.called.append(('switch_to', panel))


class ControlPanelMixinTestCase(BaseTestCase):
    """The test suite for the control panel widget."""

    klass = gui.ControlPanelMixin
    ui_filename = None

    def test_is_a_control_panel_mixin(self):
        """Inherits from ControlPanelMixin."""
        self.assertIsInstance(self.ui, gui.ControlPanelMixin)

    def test_ui_can_be_created(self):
        """UI main class exists and can be created."""
        self.assertTrue(self.ui is not None)


class ControlPanelWindowTestCase(BaseTestCase):
    """The test suite for the control panel window."""

    klass = gui.ControlPanelWindow

    def setUp(self):
        self.patch(gui, 'ControlPanelService', FakeControlPanelService)
        super(ControlPanelWindowTestCase, self).setUp()

    def test_is_a_window(self):
        """Inherits from gtk.Window."""
        self.assertIsInstance(self.ui, gui.gtk.Window)

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(), 'was not visible at startup.')

    def test_main_start_gtk_main_loop(self):
        """The GTK main loop is started when calling main()."""
        self.patch(gui.gtk, 'main', self._set_called)
        self.ui.main()
        self.assertEqual(self._called, ((), {}), 'gtk.main was not called.')

    def test_closing_stops_the_main_lopp(self):
        """The GTK main loop is stopped when closing the window."""
        self.patch(gui.gtk, 'main_quit', self._set_called)
        self.ui.emit('delete-event', None)
        self.assertEqual(
            self._called, ((), {}), 'gtk.main_quit was not called.')

    def test_wmclass_is_correct(self):
        """Test that the WMCLASS is set."""
        self.patch(gui.ControlPanelWindow, 'set_wmclass', self._set_called)
        cp = gui.ControlPanelWindow()
        cp.destroy()
        expected = ((gui.CP_WMCLASS_NAME, gui.CP_WMCLASS_CLASS), {})
        self.assertEqual(self._called, expected)

    def test_title_is_correct(self):
        """The window title is correct."""
        expected = gui.MAIN_WINDOW_TITLE % {'app_name': gui.U1_APP_NAME}
        self.assertEqual(self.ui.get_title(), expected)

    def test_control_panel_is_the_only_child(self):
        """The control panel is the window's content."""
        children = self.ui.get_children()
        self.assertEqual(1, len(children))

        control_panel = self.ui.get_children()[0]
        self.assertTrue(control_panel is self.ui.control_panel)
        self.assertIsInstance(self.ui.control_panel, gui.ControlPanel)
        self.assertTrue(self.ui.control_panel.get_visible())

    def test_main_window_is_passed_to_child(self):
        """The child gets the main_window."""
        self.assertEqual(self.ui.control_panel.main_window, self.ui)

    def test_icon_name_is_correct(self):
        """The icon name is correct."""
        self.assertEqual(self.ui.get_icon_name(), 'ubuntuone')

    def test_max_size(self):
        """Max size is not bigger than 736x525 (LP: #645526, LP: #683164)."""
        self.assertTrue(self.ui.get_size_request() <= (736, 525))

    @skipIf(not gui.USE_LIBUNITY, 'Must have Unity installed.')
    def test_focus_handler(self):
        """When the window receives focus, the handler is called."""
        THE_FLEP.urgent = True
        self.patch(gui.Unity, "LauncherEntry", FakeLauncherEntry)
        cp = gui.ControlPanelWindow()
        cp.emit('focus-in-event', gui.gtk.gdk.Event(gui.gtk.gdk.FOCUS_CHANGE))
        self.assertEqual(
            False, THE_FLEP.urgent, 'remove_urgency should have been called.')


class ControlPanelWindowAlertParamTestCase(BaseTestCase):
    """The test suite for the control panel window when passing params."""

    klass = gui.ControlPanelWindow
    kwargs = {'alert': True}

    def setUp(self):
        self.patch(gui, 'ControlPanelService', FakeControlPanelService)
        self.patch(gui.ControlPanelWindow, 'draw_attention', self._set_called)
        super(ControlPanelWindowAlertParamTestCase, self).setUp()

    def test_alert(self):
        """Can pass a 'alert' parameter to draw attention to the window."""
        self.assertEqual(
            ((), {}), self._called, 'draw_attention should have been called.')


class ControlPanelWindowParamsTestCase(ControlPanelWindowTestCase):
    """The test suite for the control panel window when passing params."""

    kwargs = {'switch_to': 'devices'}

    def test_switch_to(self):
        """Can pass a 'switch_to' parameter to start on a particular tab."""
        actual = self.ui.control_panel.management.notebook.get_current_page()
        self.assertEqual(actual, self.ui.control_panel.management.DEVICES_PAGE)


class ControlPanelWindowParamsNoneTestCase(ControlPanelWindowTestCase):
    """The suite for the control panel window when passing None params."""

    kwargs = {'switch_to': None}

    def test_switch_to(self):
        """Can pass a 'switch_to' being None. Should default to dashboard."""
        actual = self.ui.control_panel.management.notebook.get_current_page()
        expected = self.ui.control_panel.management.DASHBOARD_PAGE
        self.assertEqual(actual, expected)


class ControlPanelWindowInvalidParamsTestCase(ControlPanelWindowTestCase):
    """The suite for the control panel window when passing invalid params."""

    kwargs = {'switch_to': 'yadda-yadda'}

    def test_switch_to(self):
        """Can pass an invalid 'switch_to'. Should default to dashboard."""
        actual = self.ui.control_panel.management.notebook.get_current_page()
        expected = self.ui.control_panel.management.DASHBOARD_PAGE
        self.assertEqual(actual, expected)


class ControlPanelTestCase(BaseTestCase):
    """The test suite for the control panel itself."""

    klass = gui.ControlPanel
    kwargs = {'main_window': object()}

    def assert_current_tab_correct(self, expected_tab):
        """Check that the wiget 'expected_tab' is the current page."""
        actual = self.ui.get_nth_page(self.ui.get_current_page())
        self.assertTrue(expected_tab is actual)

    def test_is_a_notebook(self):
        """Inherits from gtk.VBox."""
        self.assertIsInstance(self.ui, gui.gtk.Notebook)

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(),
                        'must be visible at startup.')

    def test_startup_props(self):
        """The tabs and border are not shown."""
        self.assertFalse(self.ui.get_show_border(), 'must not show border.')
        self.assertFalse(self.ui.get_show_tabs(), 'must not show tabs.')

    def test_overview_is_shown_at_startup(self):
        """The overview is shown at startup."""
        self.assertIsInstance(self.ui.overview, gui.OverviewPanel)
        self.assert_current_tab_correct(self.ui.overview)

    def test_main_window_is_passed_to_child(self):
        """The child gets the main_window."""
        self.assertEqual(self.ui.overview.main_window,
                         self.kwargs['main_window'])

    def test_on_show_management_panel(self):
        """A ManagementPanel is shown when the callback is executed."""
        self.patch(self.ui.management, 'load', self._set_called)
        self.ui.on_show_management_panel()
        self.assert_current_tab_correct(self.ui.management)
        self.assertEqual(self._called, ((), {}))

    def test_on_show_management_panel_is_idempotent(self):
        """Only one ManagementPanel is shown."""
        self.ui.on_show_management_panel()
        self.ui.on_show_management_panel()

        self.assert_current_tab_correct(self.ui.management)

    def test_credentials_found_shows_dashboard_panel(self):
        """On 'credentials-found' signal, the management panel is shown.

        If first signal parameter is False, visible tab should be dashboard.

        """
        self.patch(self.ui.management, 'load', self._set_called)
        self.ui.overview.emit('credentials-found', False, object())

        self.assert_current_tab_correct(self.ui.management)
        self.assertEqual(self.ui.management.notebook.get_current_page(),
                         self.ui.management.DASHBOARD_PAGE)
        self.assertEqual(self._called, ((), {}))

    def test_credentials_found_shows_services_panel(self):
        """On 'credentials-found' signal, the management panel is shown.

        If first signal parameter is True, visible tab should be services.

        """
        a_token = object()
        self.ui.overview.emit('credentials-found', True, a_token)

        self.assert_current_tab_correct(self.ui.management)
        self.assertEqual(self.ui.management.notebook.get_current_page(),
                         self.ui.management.SERVICES_PAGE)

    def test_credentials_found_connects_syncdaemon(self):
        """On 'credentials-found' signal, ask syncdaemon to connect."""
        # credentials are new
        self.ui.overview.emit('credentials-found', True, object())
        self.assert_backend_called('connect_files')

    def test_local_device_removed_shows_overview_panel(self):
        """On 'local-device-removed' signal, the overview panel is shown."""
        self.ui.overview.emit('credentials-found', True, object())
        self.ui.management.emit('local-device-removed')

        self.assert_current_tab_correct(self.ui.overview)

    def test_unauthorized_shows_overview_panel(self):
        """On 'unauthorized' signal, the overview panel is shown."""
        self.ui.overview.emit('credentials-found', True, object())
        self.ui.management.emit('unauthorized')

        self.assert_current_tab_correct(self.ui.overview)

    def test_backend_is_shutdown_on_close(self):
        """When the control panel is closed, the backend is shutdown."""
        self.ui.emit('destroy')
        self.assert_backend_called('shutdown')


class UbuntuOneBinTestCase(BaseTestCase):
    """The test suite for the Ubuntu One panel."""

    klass = gui.UbuntuOneBin
    kwargs = {'title': 'Something old, something new and something blue.'}

    def test_is_a_vbox(self):
        """Inherits from proper gtk widget."""
        self.assertIsInstance(self.ui, gui.gtk.VBox)

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(),
                        'must be visible at startup.')
        for child in self.ui.get_children():
            self.assertTrue(child.get_visible())

    def test_title_is_a_panel_title(self):
        """Title is the correct widget."""
        self.assertIsInstance(self.ui.title, gui.PanelTitle)
        self.assertIn(self.ui.title, self.ui.get_children())

    def test_title_markup_is_correct(self):
        """The title markup is correctly set when passed as argument."""
        self.assertEqual(self.ui.title.get_text(), self.kwargs['title'])

    def test_title_is_correct(self):
        """The title markup is correctly set when defined at class level."""
        ui = self.klass()  # no title given
        self.assertEqual(ui.title.get_text(), '')

    def test_message_is_a_label_loading(self):
        """Message is the correct widget."""
        self.assertIsInstance(self.ui.message, gui.LabelLoading)
        self.assertIn(self.ui.message, self.ui.get_children())

    def test_on_success(self):
        """Callback to stop the Loading and clear messages."""
        self.ui.on_success()
        self.assertEqual(self.ui.message.get_label(), '')
        self.assertFalse(self.ui.message.active)

    def test_on_success_with_message(self):
        """Callback to stop the Loading and show a info message."""
        msg = 'WOW! <i>this rocks</i>'
        self.ui.on_success(message=msg)
        self.assertEqual(self.ui.message.get_label(), msg)
        self.assertFalse(self.ui.message.active)

    def test_on_error(self):
        """Callback to stop the Loading and clear messages."""
        self.ui.on_error()
        self.assert_warning_correct(self.ui.message, gui.VALUE_ERROR)
        self.assertFalse(self.ui.message.active)

    def test_on_error_with_message(self):
        """Callback to stop the Loading and show a info message."""
        msg = 'WOW! <i>this does not rock</i> :-/'
        self.ui.on_error(message=msg)
        self.assert_warning_correct(self.ui.message, msg)
        self.assertFalse(self.ui.message.active)

    def test_on_error_with_error_dict(self):
        """Callback to stop the Loading and show the error from error_dict."""
        msg = u'Qué mala suerte! <i>this does not rock</i>'
        error_dict = {gui.ERROR_TYPE: 'YaddaError', gui.ERROR_MESSAGE: msg}
        self.ui.on_error(error_dict=error_dict)

        expected_msg = "%s (%s: %s)" % (gui.VALUE_ERROR, 'YaddaError', msg)
        self.assert_warning_correct(self.ui.message, expected_msg)
        self.assertFalse(self.ui.message.active)

    def test_on_error_with_error_dict_without_error_type(self):
        """Callback to stop the Loading and show the error from error_dict."""
        error_dict = {}
        self.ui.on_error(error_dict=error_dict)

        expected_msg = "%s (%s)" % (gui.VALUE_ERROR, gui.UNKNOWN_ERROR)
        self.assert_warning_correct(self.ui.message, expected_msg)
        self.assertFalse(self.ui.message.active)

    def test_on_error_with_error_dict_without_error_message(self):
        """Callback to stop the Loading and show the error from error_dict."""
        error_dict = {gui.ERROR_TYPE: 'YaddaError'}
        self.ui.on_error(error_dict=error_dict)

        expected_msg = "%s (%s)" % (gui.VALUE_ERROR, 'YaddaError')
        self.assert_warning_correct(self.ui.message, expected_msg)
        self.assertFalse(self.ui.message.active)

    def test_on_error_with_message_and_error_dict(self):
        """Callback to stop the Loading and show a info message."""
        error_dict = {gui.ERROR_TYPE: 'YaddaError', gui.ERROR_MESSAGE: 'test'}
        msg = 'WOW! <i>this does not rock</i> :-/'
        self.ui.on_error(message=msg, error_dict=error_dict)
        self.assert_warning_correct(self.ui.message, msg)
        self.assertFalse(self.ui.message.active)

    def test_is_processing(self):
        """The flag 'is_processing' is False on start."""
        self.assertFalse(self.ui.is_processing)
        self.assertTrue(self.ui.is_sensitive())

    def test_set_is_processing(self):
        """When setting 'is_processing', the spinner is shown."""
        self.ui.is_processing = False
        self.ui.is_processing = True

        self.assertTrue(self.ui.message.get_visible())
        self.assertTrue(self.ui.message.active)
        self.assertFalse(self.ui.is_sensitive())

    def test_unset_is_processing(self):
        """When unsetting 'is_processing', the spinner is not shown."""
        self.ui.is_processing = True
        self.ui.is_processing = False

        self.assertTrue(self.ui.message.get_visible())
        self.assertFalse(self.ui.message.active)
        self.assertTrue(self.ui.is_sensitive())


class OverwiewPanelTestCase(ControlPanelMixinTestCase):
    """The test suite for the overview panel."""

    klass = gui.OverviewPanel
    kwargs = {'main_window': gui.gtk.Window()}
    ui_filename = 'overview.ui'
    backend_is_dbus = False

    def setUp(self):
        super(OverwiewPanelTestCase, self).setUp()
        gui.gtk.link_button_set_uri_hook(lambda *a: None)

    def test_is_a_greyable_bin(self):
        """Inherits from GreyableBin."""
        self.assertIsInstance(self.ui, gui.GreyableBin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_join_now_is_default(self):
        """The 'join_now' button is the default widget."""
        self.assertTrue(self.ui.join_now_button.get_property('can-default'))

    def test_backend(self):
        """Has a correct backend."""
        self.assertIsInstance(self.ui.backend, FakedCredentialsBackend)


class OverwiewPanelBackendCallbacksTestCase(OverwiewPanelTestCase):
    """Proper callbacks are chained to the credentials backend methods."""

    failure = Exception()

    def test_find_credentials_fired_with_credentials(self):
        """Test that on_credentials_found is called."""
        self.ui.backend.next_result = defer.succeed(TOKEN)
        self.patch(self.ui, 'on_credentials_found', self._set_called)

        self.ui.on_network_state_changed(gui.networkstate.ONLINE)

        self.assertEqual(self._called, ((TOKEN,), {}))

    def test_find_credentials_fired_without_credentials(self):
        """Test that on_credentials_not_found is called."""
        self.ui.backend.next_result = defer.succeed({})
        self.patch(self.ui, 'on_credentials_not_found', self._set_called)

        self.ui.on_network_state_changed(gui.networkstate.ONLINE)

        self.assertEqual(self._called, ((), {}))

    def test_find_credentials_errback(self):
        """Test that on_credentials_error is called."""
        self.ui.backend.next_result = defer.fail(self.failure)
        self.patch(self.ui, 'on_credentials_error', self._set_called)

        self.ui.on_network_state_changed(gui.networkstate.ONLINE)

        failure = self._called[0][0]
        self.assertIsInstance(failure, Failure)
        self.assertEqual(failure.value, self.failure)

    def test_register_fired_with_credentials(self):
        """Test that on_credentials_found is called."""
        self.ui.backend.next_result = defer.succeed(TOKEN)
        self.patch(self.ui, 'on_credentials_found', self._set_called)

        self.ui.join_now_button.clicked()

        self.assertEqual(self._called, ((TOKEN,), {}))

    def test_register_fired_with_credentials_none(self):
        """Test that on_authorization_denied is called."""
        self.ui.backend.next_result = defer.succeed(None)
        self.patch(self.ui, 'on_authorization_denied', self._set_called)

        self.ui.join_now_button.clicked()

        self.assertEqual(self._called, ((), {}))

    def test_register_errback(self):
        """Test that on_credentials_error is called."""
        self.ui.backend.next_result = defer.fail(self.failure)
        self.patch(self.ui, 'on_credentials_error', self._set_called)

        self.ui.join_now_button.clicked()

        failure = self._called[0][0]
        self.assertIsInstance(failure, Failure)
        self.assertEqual(failure.value, self.failure)

    def test_login_fired_with_credentials(self):
        """Test that on_credentials_found is called."""
        self.ui.backend.next_result = defer.succeed(TOKEN)
        self.patch(self.ui, 'on_credentials_found', self._set_called)

        self.ui.connect_button.clicked()

        self.assertEqual(self._called, ((TOKEN,), {}))

    def test_login_fired_with_credentials_none(self):
        """Test that on_authorization_denied is called."""
        self.ui.backend.next_result = defer.succeed(None)
        self.patch(self.ui, 'on_authorization_denied', self._set_called)

        self.ui.connect_button.clicked()

        self.assertEqual(self._called, ((), {}))

    def test_login_errback(self):
        """Test that on_credentials_error is called."""
        self.ui.backend.next_result = defer.fail(self.failure)
        self.patch(self.ui, 'on_credentials_error', self._set_called)

        self.ui.connect_button.clicked()

        failure = self._called[0][0]
        self.assertIsInstance(failure, Failure)
        self.assertEqual(failure.value, self.failure)


class OverwiewNetworkStatePanelTestCase(OverwiewPanelTestCase):
    """The test suite for the overview panel regarding network state."""

    def test_network_state_is_created(self):
        """The network state is created."""
        self.assertIsInstance(self.ui.network_manager_state,
                              gui.networkstate.NetworkManagerState)
        self.assertEqual(self.ui.network_manager_state._kwargs['result_cb'],
                         self.ui.on_network_state_changed)

    def test_network_state_is_queried_at_startup(self):
        """The network state is asked to the NetworkManagerState."""
        self.assertTrue('find_online_state' in
                        self.ui.network_manager_state._called)

    def test_state_online(self):
        """Network connection is online."""
        self.ui.on_network_state_changed(gui.networkstate.ONLINE)
        # all green, no warning
        self.assertEqual(self.ui.warning_label.get_text(), '')
        self.assertTrue(self.ui.get_sensitive())

    def test_state_offline(self):
        """Network connection is offline."""
        self.ui.on_network_state_changed(gui.networkstate.OFFLINE)
        msg = gui.NETWORK_OFFLINE % {'app_name': gui.U1_APP_NAME}

        self.assert_warning_correct(self.ui.warning_label, msg)
        self.assertFalse(self.ui.get_sensitive())

    def test_state_unknown(self):
        """Network connection is unknown. Assume that connection is present."""
        self.ui.on_network_state_changed(gui.networkstate.UNKNOWN)

        self.assertEqual(self.ui.warning_label.get_text(), '')
        self.assertTrue(self.ui.get_sensitive())


class OverwiewPanelOnlineTestCase(OverwiewPanelTestCase):
    """The test suite for the overview panel."""

    def setUp(self):
        super(OverwiewPanelOnlineTestCase, self).setUp()
        self.ui.on_network_state_changed(gui.networkstate.ONLINE)

    def test_find_credentials_is_called(self):
        """Credentials are asked to SSO backend."""
        self.assertFalse(self.ui._credentials_are_new)
        self.assert_backend_called('find_credentials')

    def test_on_credentials_found(self):
        """Callback 'on_credentials_found' is correct."""
        self.ui.connect('credentials-found', self._set_called)

        self.ui.on_credentials_found(TOKEN)

        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')
        # assume credentials were in local keyring
        self.assertEqual(self._called, ((self.ui, False, TOKEN), {}))

    def test_on_credentials_found_when_creds_are_not_new(self):
        """Callback 'on_credentials_found' distinguish if creds are new."""
        self.ui.connect('credentials-found', self._set_called)

        # credentials weren't in the system
        self.ui.on_credentials_not_found()
        # now they are!
        self.ui.on_credentials_found(TOKEN)

        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')
        # assume credentials were not in local keyring
        self.assertEqual(self._called, ((self.ui, True, TOKEN), {}))

    def test_on_credentials_not_found(self):
        """Callback 'on_credentials_not_found' is correct."""
        self.ui.on_credentials_not_found()
        self.assertTrue(self.ui.get_visible())
        self.assertTrue(self.ui._credentials_are_new)

    def test_on_credentials_error(self):
        """Callback 'on_credentials_error' is correct."""
        self.ui.on_credentials_error({})
        self.assertTrue(self.ui.get_visible())
        self.assert_warning_correct(self.ui.warning_label,
                                    gui.CREDENTIALS_ERROR)

    def test_on_authorization_denied(self):
        """Callback 'on_authorization_denied' is correct."""
        self.ui.on_authorization_denied()
        self.assertTrue(self.ui.get_visible())
        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')
        self.assertEqual(self.ui.warning_label.get_text(), '')


class OverwiewPanelNoCredsTestCase(OverwiewPanelTestCase):
    """The test suite for the overview panel when no credentials are found."""

    def setUp(self):
        super(OverwiewPanelNoCredsTestCase, self).setUp()
        self.ui.on_credentials_not_found()

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(),
                        'must be visible at startup if credentials not found.')

    def test_warning_label_is_hidden(self):
        """The warning label is not shown by default."""
        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_image_is_correct(self):
        """There is an image attribute and is correct."""
        self.assert_image_equal(self.ui.banner, 'overview.png')

    def test_join_now_is_default_widget(self):
        """The join now button is the default widget."""
        self.assertTrue(self.ui.join_now_button.get_property('can_default'))

    def test_join_now_button_clicked(self):
        """Test the 'join now' button callback."""
        self.kwargs['main_window'].show()  # ensure parent window is realized
        self.addCleanup(self.kwargs['main_window'].hide)

        self.ui.join_now_button.clicked()

        window_id = self.kwargs['main_window'].window.xid
        kwargs = {'window_id': str(window_id)}
        self.assert_backend_called('register', **kwargs)

    def test_connect_button_clicked(self):
        """Test the 'join now' button callback."""
        self.kwargs['main_window'].show()  # ensure parent window is realized
        self.addCleanup(self.kwargs['main_window'].hide)

        self.ui.connect_button.clicked()

        window_id = self.kwargs['main_window'].window.xid
        kwargs = {'window_id': str(window_id)}
        self.assert_backend_called('login', **kwargs)

    def test_join_now_button_clicked_set_greyed(self):
        """Clicking on 'join_now' self is greyed."""
        self.ui.join_now_button.clicked()
        self.assertTrue(self.ui.get_property('greyed'), 'Must be greyed.')

    def test_join_now_button_clicked_removes_warning(self):
        """Clicking on 'join_now' the warnings are removed."""
        self.ui.on_authorization_denied()  # show warning
        self.ui.join_now_button.clicked()

        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_connect_button_clicked_set_greyed(self):
        """Clicking on 'connect' self is greyed."""
        self.ui.connect_button.clicked()
        self.assertTrue(self.ui.get_property('greyed'), 'Must be greyed.')

    def test_connect_button_clicked_removes_warning(self):
        """Clicking on 'connect' the warnings are removed."""
        self.ui.on_authorization_denied()  # show warning
        self.ui.connect_button.clicked()

        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_learn_more_button_clicked(self):
        """Test the 'learn more' button callback."""
        self.patch(gui, 'uri_hook', self._set_called)
        self.ui.learn_more_button.clicked()

        expected = (self.ui.learn_more_button, gui.LEARN_MORE_LINK)
        self.assertEqual(self._called, (expected, {}))

    def test_on_credentials_not_found_unset_greyed(self):
        """Callback 'on_credentials_not_found' unsets the 'greyed' prop."""
        self.ui.connect_button.clicked()
        self.ui.on_credentials_not_found()

        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')

    def test_on_credentials_error_unset_greyed(self):
        """Callback 'on_credentials_error' unsets the 'greyed' prop."""
        self.ui.connect_button.clicked()
        self.ui.on_credentials_error({})

        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')

    def test_on_authorization_denied_unset_greyed(self):
        """Callback 'on_authorization_denied' unsets the 'greyed' prop."""
        self.ui.connect_button.clicked()
        self.ui.on_authorization_denied()

        self.assertFalse(self.ui.get_property('greyed'), 'Must not be greyed.')

    def test_buttons_disabled_when_greyed(self):
        """Buttons should be disabled when widget is greyed."""
        self.ui.set_sensitive(True)
        self.ui.set_property('greyed', True)

        self.assertFalse(self.ui.join_now_button.is_sensitive())
        self.assertFalse(self.ui.connect_button.is_sensitive())

    def test_buttons_enabled_when_not_greyed(self):
        """Buttons should be enabled when widget is not greyed."""
        self.ui.set_sensitive(False)
        self.ui.set_property('greyed', False)

        self.assertTrue(self.ui.join_now_button.is_sensitive())
        self.assertTrue(self.ui.connect_button.is_sensitive())
