# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
# Authors: Natalia B. Bidart <nataliabidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the control panel backend DBus service."""

import dbus
import mocker

from twisted.internet import defer
from twisted.python.failure import Failure

from ubuntuone.controlpanel import dbus_service
from ubuntuone.controlpanel import (DBUS_BUS_NAME, DBUS_PREFERENCES_PATH,
    DBUS_PREFERENCES_IFACE)
from ubuntuone.controlpanel.tests import SAMPLE_FOLDERS
from ubuntuone.controlpanel.integrationtests import TestCase


SAMPLE_ACCOUNT_INFO = {
    "quota_used": "12345",
    "quota_total": "54321",
    "type": "paying customer",
    "name": "john carlos",
    "email": "john@carlos.com",
}

SAMPLE_DEVICES_INFO = [
    {
        "device_token": "token-1",
        "name": "Ubuntu One @ darkstar",
        "date_added": "2008-12-06T18:15:38.0",
        "type": "computer",
        "configurable": 'True',
        "show_all_notifications": 'True',
        "limit_bandwidth": 'True',
        "max_upload_speed": "12345",
        "max_download_speed": "54321",
        "available_services": "files, contacts, music, bookmarks",
        "enabled_services": "files, music",
    },
    {
        "device_token": "token-2",
        "name": "Ubuntu One @ brightmoon",
        "date_added": "2010-09-22T20:45:38.0",
        "type": "computer",
        "configurable": '',
        "available_services": "files, contacts, bookmarks",
        "enabled_services": "files, bookmarks",
    },
]

SAMPLE_VOLUMES_INFO = [
    (u'', u'1253698', SAMPLE_FOLDERS),
]

SAMPLE_REPLICATIONS_INFO = [
    {'replication_id': 'yadda', 'wait for it': 'awesome'},
    {'replication_id': 'yoda', 'something else': 'awesome'},
]


class DBusServiceMainTestCase(mocker.MockerTestCase):
    """Tests for the main function."""

    def test_dbus_service_main(self):
        """The main method starts the loop and hooks up to DBus."""
        rs_name = "ubuntuone.controlpanel.dbus_service.register_service"
        rs = self.mocker.replace(rs_name)
        rs()
        self.mocker.result(True)

        mainloop = "ubuntuone.controlpanel.dbus_service.gobject.MainLoop"
        mainloop = self.mocker.replace(mainloop)
        mainloop()
        loop = self.mocker.mock()
        self.mocker.result(loop)

        shutdown_func = self.mocker.mock()
        loop.quit  # pylint: disable=W0104
        self.mocker.result(shutdown_func)

        rml_name = "ubuntuone.controlpanel.dbus_service.run_mainloop"
        rml = self.mocker.replace(rml_name)
        rml(loop=loop)
        pb_name = "ubuntuone.controlpanel.dbus_service.publish_backend"
        pb = self.mocker.replace(pb_name)
        pb(shutdown_func=shutdown_func)
        self.mocker.replay()
        dbus_service.main()

    def test_dbus_service_cant_register(self):
        """The main method can't start the loop."""
        rs_name = "ubuntuone.controlpanel.dbus_service.register_service"
        rs = self.mocker.replace(rs_name)
        rs()
        self.mocker.result(False)
        self.mocker.replay()
        dbus_service.main()


class MockBackend(object):
    """A mock backend."""

    exception = None
    sample_status = {
        dbus_service.MSG_KEY: 'test me please',
        dbus_service.STATUS_KEY: dbus_service.FILE_SYNC_IDLE,
    }
    shutdown_func = None

    def __init__(self, shutdown_func=None):
        MockBackend.shutdown_func = shutdown_func
        self._status_changed_handler = []

    def _process(self, result):
        """Process the request with the given result."""
        if self.exception:
            # pylint: disable=E1102
            return defer.fail(self.exception(result))
        return defer.succeed(result)

    def _set_status_changed_handler(self, handler):
        """Set 'handler' to be called when file sync status changes."""
        if handler is not None:
            self._status_changed_handler.append(handler)
        else:
            self._status_changed_handler = []

    def _get_status_changed_handler(self):
        """Return the handler to be called when file sync status changes."""
        if len(self._status_changed_handler) > 0:
            result = self._status_changed_handler[-1]
        else:
            result = None
        return result

    status_changed_handler = property(_get_status_changed_handler,
                                      _set_status_changed_handler)

    def account_info(self):
        """Get the user account info."""
        return self._process(SAMPLE_ACCOUNT_INFO)

    def devices_info(self):
        """Get the user devices info."""
        return self._process(SAMPLE_DEVICES_INFO)

    def change_device_settings(self, token, settings):
        """Configure a given device."""
        return self._process(token)

    def remove_device(self, token):
        """Remove a device's tokens from the sso server."""
        return self._process(token)

    def file_sync_status(self):
        """Return the status of the file sync service."""
        return self._process(self.sample_status)

    def enable_files(self):
        """Enable files service."""
        return self._process(None)

    def disable_files(self):
        """Disable files service."""
        return self._process(None)

    def connect_files(self):
        """Connect files service."""
        return self._process(None)

    def disconnect_files(self):
        """Disconnect files service."""
        return self._process(None)

    def restart_files(self):
        """Restart the files service."""
        return self._process(None)

    def start_files(self):
        """Start the files service."""
        return self._process(None)

    def stop_files(self):
        """Stop the files service."""
        return self._process(None)

    def volumes_info(self):
        """Get the user volumes info."""
        return self._process(SAMPLE_VOLUMES_INFO)

    def change_volume_settings(self, volume_id, settings):
        """Configure a given volume."""
        return self._process(volume_id)

    def replications_info(self):
        """Start the replication exclusion service if needed.

        Return the replication info, which is a dictionary of (replication
        name, enabled).

        """
        return self._process(SAMPLE_REPLICATIONS_INFO)

    def change_replication_settings(self, replication_id, settings):
        """Configure a given replication."""
        return self._process(replication_id)

    def query_bookmark_extension(self):
        """True if the bookmark extension has been installed."""
        return self._process(False)

    def install_bookmarks_extension(self):
        """Install the extension to sync bookmarks."""
        return self._process(None)

    def shutdown(self):
        """Stop this service."""
        self.shutdown_func()


class DBusServiceTestCase(TestCase):
    """Test for the DBus service."""

    timeout = 5

    def setUp(self):
        """Initialize each test run."""
        super(DBusServiceTestCase, self).setUp()
        dbus_service.init_mainloop()
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Keep track of function calls, useful for monkeypatching."""
        self._called = (args, kwargs)

    def test_register_service(self):
        """The DBus service is successfully registered."""
        ret = dbus_service.register_service()
        self.assertTrue(ret)

    def test_cant_register_twice(self):
        """The DBus service can't register if it already did."""
        ret = dbus_service.register_service()
        self.assertTrue(ret)
        ret = dbus_service.register_service()
        self.assertFalse(ret)
    #pylint: disable=W0612
    test_cant_register_twice.skip = "Must run 2nd check in another process."

    def test_dbus_busname_created(self):
        """The DBus BusName is created."""
        busname = dbus_service.get_busname()
        self.assertEqual(busname.get_name(), DBUS_BUS_NAME)

    def test_error_handler_with_failure(self):
        """Ensure to build a string-string dict to pass to error signals."""
        error = dbus_service.Failure(TypeError('oh no!'))
        expected = dbus_service.failure_to_error_dict(error)

        result = dbus_service.error_handler(error)

        self.assertEqual(expected, result)

    def test_error_handler_with_exception(self):
        """Ensure to build a string-string dict to pass to error signals."""
        error = TypeError('oh no, no again!')
        expected = dbus_service.exception_to_error_dict(error)

        result = dbus_service.error_handler(error)

        self.assertEqual(expected, result)

    def test_error_handler_with_string_dict(self):
        """Ensure to build a string-string dict to pass to error signals."""
        expected = {'test': 'me'}

        result = dbus_service.error_handler(expected)

        self.assertEqual(expected, result)

    def test_error_handler_with_non_string_dict(self):
        """Ensure to build a string-string dict to pass to error signals."""
        expected = {'test': 0, 'qué?': None,
                    10: 'foo\xffbar', True: u'ñoño'}

        result = dbus_service.error_handler(expected)
        expected = dict(map(lambda x: x if isinstance(x, unicode) else
                                      str(x).decode('utf8', 'replace'), i)
                        for i in expected.iteritems())

        self.assertEqual(expected, result)

    def test_error_handler_default(self):
        """Ensure to build a string-string dict to pass to error signals."""
        msg = 'Got unexpected error argument %r' % None
        expected = {dbus_service.ERROR_TYPE: 'UnknownError',
                    dbus_service.ERROR_MESSAGE: msg}

        result = dbus_service.error_handler(None)

        self.assertEqual(expected, result)


class BaseTestCase(TestCase):
    """Base test case for the DBus service."""

    timeout = 3

    def setUp(self):
        super(BaseTestCase, self).setUp()
        dbus_service.init_mainloop()
        self.patch(dbus_service, 'ControlBackend', MockBackend)
        be = dbus_service.publish_backend()
        self.addCleanup(be.remove_from_connection)
        bus = dbus.SessionBus()
        obj = bus.get_object(bus_name=DBUS_BUS_NAME,
                             object_path=DBUS_PREFERENCES_PATH,
                             follow_name_owner_changes=True)
        self.backend = dbus.Interface(object=obj,
                                      dbus_interface=DBUS_PREFERENCES_IFACE)
        self.deferred = defer.Deferred()

    def tearDown(self):
        self.backend = None
        self.deferred = None
        super(BaseTestCase, self).tearDown()

    def got_error(self, *a):
        """Some error happened in the DBus call."""
        self.deferred.errback(*a)

    def ignore(self, *a):
        """Do nothing with the returned value."""

    def errback_on_error(self, f):
        """Call the given 'f' but errback self.deferred on any error."""

        def inner(*a, **kw):
            """Call the given 'f' but errback self.deferred on any error."""
            try:
                return f(*a, **kw)
            except Exception, e:  # pylint: disable=W0703
                self.deferred.errback(Failure(e))

        return inner

    def assert_correct_method_call(self, success_sig, error_sig, success_cb,
                                   method, *args):
        """Connect 'success_cb' with 'success_sig', and call 'method'.

        'error_sig' will be connected to the class' error handler.

        'success_cb' should fire 'self.deferred' on success. If 'success_cb'
        fails due to an failed assertion, the deferred will be errback'd.

        """
        success_cb = self.errback_on_error(success_cb)
        res = self.backend.connect_to_signal(success_sig, success_cb)
        self.addCleanup(res.remove)

        res = self.backend.connect_to_signal(error_sig, self.got_error)
        self.addCleanup(res.remove)

        method = self.errback_on_error(method)
        method(*args, reply_handler=self.ignore, error_handler=self.got_error)

        return self.deferred


class OperationsTestCase(BaseTestCase):
    """Test for the DBus service operations."""

    def test_account_info_returned(self):
        """The account info is successfully returned."""

        def got_signal(account_info):
            """The correct signal was fired."""
            self.assertIn("quota_used", account_info)
            self.assertIn("quota_total", account_info)
            self.assertIn("type", account_info)
            self.assertIn("name", account_info)
            self.assertIn("email", account_info)
            self.deferred.callback("success")

        args = ("AccountInfoReady", "AccountInfoError", got_signal,
                self.backend.account_info)
        return self.assert_correct_method_call(*args)

    def test_devices_info_returned(self):
        """The devices info is successfully returned."""

        def got_signal(devices_list):
            """The correct signal was fired."""
            for device_info in devices_list:
                self.assertIn("device_token", device_info)
                self.assertIn("name", device_info)
                self.assertIn("date_added", device_info)
                self.assertIn("type", device_info)
                self.assertIn("configurable", device_info)
                if bool(device_info["configurable"]):
                    self.assertIn("show_all_notifications", device_info)
                    self.assertIn("limit_bandwidth", device_info)
                    self.assertIn("max_upload_speed", device_info)
                    self.assertIn("max_download_speed", device_info)
                self.assertIn("available_services", device_info)
                self.assertIn("enabled_services", device_info)
            self.deferred.callback("success")

        args = ("DevicesInfoReady", "DevicesInfoError", got_signal,
                self.backend.devices_info)
        return self.assert_correct_method_call(*args)

    def test_change_device_settings(self):
        """The device settings are successfully changed."""
        sample_token = "token-1"

        def got_signal(token):
            """The correct token was received."""
            self.assertEqual(token, sample_token)
            self.deferred.callback("success")

        settings = {
            "enabled_services": "files, contacts",
        }
        args = ("DeviceSettingsChanged", "DeviceSettingsChangeError",
                got_signal, self.backend.change_device_settings,
                sample_token, settings)
        return self.assert_correct_method_call(*args)

    def test_remove_device(self):
        """The device is removed."""
        sample_token = "token-1"

        def got_signal(token):
            """The correct token was received."""
            self.assertEqual(token, sample_token)
            self.deferred.callback("success")

        args = ("DeviceRemoved", "DeviceRemovalError", got_signal,
                self.backend.remove_device, sample_token)
        return self.assert_correct_method_call(*args)

    def test_enable_files(self):
        """Enable files service."""

        def got_signal(*args):
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesEnabled", "FilesEnableError", got_signal,
                self.backend.enable_files)
        return self.assert_correct_method_call(*args)

    def test_disable_files(self):
        """Disable files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesDisabled", "FilesDisableError", got_signal,
                self.backend.disable_files)
        return self.assert_correct_method_call(*args)

    def test_connect_files(self):
        """Connect files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesConnected", "FilesConnectError", got_signal,
                self.backend.connect_files)
        return self.assert_correct_method_call(*args)

    def test_disconnect_files(self):
        """Disconnect files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesDisconnected", "FilesDisconnectError", got_signal,
                self.backend.disconnect_files)
        return self.assert_correct_method_call(*args)

    def test_restart_files(self):
        """Restart files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesRestarted", "FilesRestartError", got_signal,
                self.backend.restart_files)
        return self.assert_correct_method_call(*args)

    def test_start_files(self):
        """Start files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesStarted", "FilesStartError", got_signal,
                self.backend.start_files)
        return self.assert_correct_method_call(*args)

    def test_stop_files(self):
        """Stop files service."""

        def got_signal():
            """The correct signal was received."""
            self.deferred.callback("success")

        args = ("FilesStopped", "FilesStopError", got_signal,
                self.backend.stop_files)
        return self.assert_correct_method_call(*args)

    def test_volumes_info(self):
        """The volumes info is reported."""

        def got_signal(volumes):
            """The correct info was received."""
            self.assertEqual(volumes, SAMPLE_VOLUMES_INFO)
            self.deferred.callback("success")

        args = ("VolumesInfoReady", "VolumesInfoError", got_signal,
                self.backend.volumes_info)
        return self.assert_correct_method_call(*args)

    def test_change_volume_settings(self):
        """The volume settings are successfully changed."""
        expected_volume_id = SAMPLE_FOLDERS[0]['volume_id']

        def got_signal(volume_id):
            """The correct volume was changed."""
            self.assertEqual(volume_id, expected_volume_id)
            self.deferred.callback("success")

        args = ("VolumeSettingsChanged", "VolumeSettingsChangeError",
                got_signal, self.backend.change_volume_settings,
                expected_volume_id, {'subscribed': ''})
        return self.assert_correct_method_call(*args)

    def test_replications_info(self):
        """The replications info is reported."""

        def got_signal(replications):
            """The correct info was received."""
            self.assertEqual(replications, SAMPLE_REPLICATIONS_INFO)
            self.deferred.callback("success")

        args = ("ReplicationsInfoReady", "ReplicationsInfoError", got_signal,
                self.backend.replications_info)
        return self.assert_correct_method_call(*args)

    def test_change_replication_settings(self):
        """The replication settings are successfully changed."""
        expected_replication_id = SAMPLE_REPLICATIONS_INFO[0]['replication_id']

        def got_signal(replication_id):
            """The correct replication was changed."""
            self.assertEqual(replication_id, expected_replication_id)
            self.deferred.callback("success")

        args = ("ReplicationSettingsChanged", "ReplicationSettingsChangeError",
                got_signal, self.backend.change_replication_settings,
                expected_replication_id, {'enabled': ''})
        return self.assert_correct_method_call(*args)

    def test_query_bookmarks_extension(self):
        """The bookmarks extension is queried."""

        def got_signal(enabled):
            """The correct status was received."""
            self.assertEqual(enabled, False)
            self.deferred.callback("success")

        args = ("QueryBookmarksResult", "QueryBookmarksError", got_signal,
                self.backend.query_bookmark_extension)
        return self.assert_correct_method_call(*args)

    def test_install_bookmarks_extension(self):
        """The bookmarks extension is installed."""

        def got_signal():
            """The extension was installed."""
            self.deferred.callback("success")

        args = ("InstallBookmarksSuccess", "InstallBookmarksError", got_signal,
                self.backend.install_bookmarks_extension)
        return self.assert_correct_method_call(*args)


class OperationsErrorTestCase(OperationsTestCase):
    """Test for the DBus service operations when there is an error."""

    def setUp(self):
        super(OperationsErrorTestCase, self).setUp()
        self.patch(MockBackend, 'exception', AssertionError)

    def assert_correct_method_call(self, success_sig, error_sig, success_cb,
                                   method, *args):
        """Call parent instance swapping success_sig with error_sig.

        This is because we want to succeed the test when the error signal was
        received.

        """

        def got_error_signal(*a):
            """The error signal was received."""
            if len(a) == 1:
                error_dict = a[0]
            else:
                an_id, error_dict = a
                self.assertEqual(an_id, args[0])

            self.assertEqual(error_dict[dbus_service.ERROR_TYPE],
                             'AssertionError')
            self.deferred.callback("success")

        return super(OperationsErrorTestCase, self).assert_correct_method_call(
            error_sig, success_sig, got_error_signal, method, *args)


class OperationsAuthErrorTestCase(OperationsTestCase):
    """Test for the DBus service operations when UnauthorizedError happens."""

    def setUp(self):
        super(OperationsAuthErrorTestCase, self).setUp()
        self.patch(MockBackend, 'exception',
                   dbus_service.UnauthorizedError)

    def assert_correct_method_call(self, success_sig, error_sig, success_cb,
                                   method, *args):
        """Call parent instance expecting UnauthorizedError signal."""

        def inner_success_cb(*a):
            """The success signal was received."""
            if len(a) == 1:
                error_dict = a[0]
            else:
                an_id, error_dict = a
                self.assertEqual(an_id, args[0])

            self.assertEqual(error_dict[dbus_service.ERROR_TYPE],
                             'UnauthorizedError')
            self.deferred.callback('success')

        parent = super(OperationsAuthErrorTestCase, self)
        return parent.assert_correct_method_call(
            "UnauthorizedError", error_sig, inner_success_cb, method, *args)


class FileSyncTestCase(BaseTestCase):
    """Test for the DBus service when requesting file sync status."""

    def assert_correct_status_signal(self, status, sync_signal,
                                     error_signal="FileSyncStatusError",
                                     expected_msg=None):
        """The file sync status is reported properly."""
        MockBackend.sample_status[dbus_service.STATUS_KEY] = status
        if expected_msg is None:
            expected_msg = MockBackend.sample_status[dbus_service.MSG_KEY]

        def got_signal(msg):
            """The correct status was received."""
            self.assertEqual(msg, expected_msg)
            self.deferred.callback("success")

        args = (sync_signal, error_signal, got_signal,
                self.backend.file_sync_status)
        return self.assert_correct_method_call(*args)

    def test_file_sync_status_unknown(self):
        """The file sync status is reported properly."""
        msg = MockBackend.sample_status
        args = ('invalid-file-sync-status', "FileSyncStatusError",
                "FileSyncStatusIdle", msg)
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_error(self):
        """The file sync status is reported properly."""
        msg = MockBackend.sample_status[dbus_service.MSG_KEY]
        err_dict = {dbus_service.ERROR_TYPE: 'FileSyncStatusError',
                    dbus_service.ERROR_MESSAGE: msg}
        args = (dbus_service.FILE_SYNC_ERROR, "FileSyncStatusError",
                "FileSyncStatusIdle", err_dict)
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_disabled(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_DISABLED, "FileSyncStatusDisabled")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_starting(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_STARTING, "FileSyncStatusStarting")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_stopped(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_STOPPED, "FileSyncStatusStopped")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_disconnected(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_DISCONNECTED,
                "FileSyncStatusDisconnected")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_syncing(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_SYNCING, "FileSyncStatusSyncing")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_idle(self):
        """The file sync status is reported properly."""
        args = (dbus_service.FILE_SYNC_IDLE, "FileSyncStatusIdle")
        return self.assert_correct_status_signal(*args)

    def test_file_sync_status_changed(self):
        """The file sync status is reported every time status changed."""
        status = (
            dbus_service.FILE_SYNC_DISABLED,
            dbus_service.FILE_SYNC_DISCONNECTED,
            dbus_service.FILE_SYNC_ERROR,
            dbus_service.FILE_SYNC_IDLE,
            dbus_service.FILE_SYNC_STARTING,
            dbus_service.FILE_SYNC_SYNCING,
        )
        for arg in status:
            args = (arg, "FileSyncStatusChanged")
            return self.assert_correct_status_signal(*args, expected_msg=arg)

    def test_status_changed_handler(self):
        """The status changed handler is properly set."""
        be = MockBackend()
        dbus_service.ControlPanelBackend(backend=be)

        self.assertEqual(be.status_changed_handler, None)

    def test_status_changed_handler_after_status_requested(self):
        """The status changed handler is properly set."""
        be = MockBackend()
        cpbe = dbus_service.ControlPanelBackend(backend=be)
        cpbe.file_sync_status()

        self.assertEqual(be.status_changed_handler, cpbe.process_status)

    def test_status_changed_handler_after_status_requested_twice(self):
        """The status changed handler is properly set."""
        be = MockBackend()
        cpbe = dbus_service.ControlPanelBackend(backend=be)
        cpbe.file_sync_status()
        cpbe.file_sync_status()

        # pylint: disable=W0212
        self.assertEqual(be._status_changed_handler, [cpbe.process_status])


class ShutdownTestCase(BaseTestCase):
    """Test for the DBus service shurdown."""

    @defer.inlineCallbacks
    def test_shutdown(self):
        """The service can be shutdown."""
        called = []
        MockBackend.shutdown_func = lambda *a: called.append('shutdown')
        self.backend.shutdown(reply_handler=lambda: self.deferred.callback(1),
                              error_handler=self.got_error)
        yield self.deferred

        self.assertEqual(called, ['shutdown'])
