# -*- coding: utf-8 *-*
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Test the Sync Menu."""

import time
from collections import Callable

from twisted.internet import defer
from twisted.trial.unittest import TestCase

from ubuntuone.platform import sync_menu
from ubuntuone.platform.sync_menu import linux


def fake_call_later(*args):
    """Fake reactor.callLater."""


class FakeStatusFrontend(object):
    """Fake StatusFrontend."""

    def __init__(self):
        self.recent_transfers_data = []
        self.uploading_data = []

    def recent_transfers(self):
        """Return the fake recent transfers files."""
        return self.recent_transfers_data

    def files_uploading(self):
        """Return the fake files being upload."""
        return self.uploading_data


class FakeTimer(object):
    """Fake Timer."""

    def __init__(self, delay):
        self.delay = delay
        self.callback = None

    def addCallback(self, callback):
        """Add callback."""
        self.callback = callback


class FakeSyncdaemonService(object):
    """Fake SyncdaemonService."""


class FakeSyncMenuApp(object):
    """Fake SyncMenu."""

    data = {}

    @classmethod
    def new(cls, *args):
        return FakeSyncMenuApp()

    @classmethod
    def clean(cls):
        """Clear the values stored in data."""

    def set_menu(self, server):
        """Set the menu for SyncMenu App."""
        self.data['server'] = server

    def connect(self, signal, callback):
        """Fake connect."""
        self.data['connect'] = (signal, callback)


class SyncMenuDummyTestCase(TestCase):
    """Test the SyncMenu."""

    def test_dummy_support(self):
        """Check that the Dummy object can be created properly."""
        dummy = linux.DummySyncMenu('random', 'args')
        self.assertIsInstance(dummy, linux.DummySyncMenu)

    def test_dummy_has_update_transfers(self):
        """Check that the dummy has the proper methods required by the API."""
        dummy = linux.DummySyncMenu('random', 'args')
        self.assertIsInstance(dummy.update_transfers, Callable)


class SyncMenuTestCase(TestCase):
    """Test the SyncMenu."""

    skip = None if linux.use_syncmenu else "SyncMenu not installed."

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SyncMenuTestCase, self).setUp()
        self.patch(linux.SyncMenu, "App", FakeSyncMenuApp)
        FakeSyncMenuApp.clean()
        self.syncdaemon_service = FakeSyncdaemonService()
        self.status_frontend = FakeStatusFrontend()
        self._paused = False
        self.patch(sync_menu.UbuntuOneSyncMenu, "change_sync_status",
            self._change_sync_status)
        self.sync_menu = sync_menu.UbuntuOneSyncMenu(self.status_frontend,
            self.syncdaemon_service)

    def _change_sync_status(self, *args):
        """Fake change_sync_status."""
        if self._paused:
            self._paused = False
        else:
            self._paused = True

    def test_init(self):
        """Check that the menu is properly initialized."""
        self.assertIsInstance(FakeSyncMenuApp.data['server'],
            linux.Dbusmenu.Server)
        self.assertEqual(self.sync_menu.open_u1.get_parent(),
            self.sync_menu.root_menu)
        self.assertEqual(self.sync_menu.go_to_web.get_parent(),
            self.sync_menu.root_menu)
        self.assertEqual(self.sync_menu.more_storage.get_parent(),
            self.sync_menu.root_menu)
        self.assertEqual(self.sync_menu.get_help.get_parent(),
            self.sync_menu.root_menu)
        self.assertEqual(self.sync_menu.transfers.get_parent(),
            self.sync_menu.root_menu)

        self.assertEqual(self.sync_menu.open_u1.property_get(
            linux.Dbusmenu.MENUITEM_PROP_LABEL), linux.OPEN_U1)
        self.assertEqual(self.sync_menu.go_to_web.property_get(
            linux.Dbusmenu.MENUITEM_PROP_LABEL), linux.GO_TO_WEB)
        self.assertEqual(self.sync_menu.transfers.property_get(
            linux.Dbusmenu.MENUITEM_PROP_LABEL), linux.TRANSFERS)
        self.assertEqual(self.sync_menu.more_storage.property_get(
            linux.Dbusmenu.MENUITEM_PROP_LABEL), linux.MORE_STORAGE)
        self.assertEqual(self.sync_menu.get_help.property_get(
            linux.Dbusmenu.MENUITEM_PROP_LABEL), linux.GET_HELP)

        self.assertEqual(self.sync_menu.app.data['connect'],
            ("notify::paused", self.sync_menu.change_sync_status))
        self.sync_menu.app.data['connect'][1]()
        self.assertTrue(self._paused)
        self.sync_menu.app.data['connect'][1]()
        self.assertFalse(self._paused)

    def test_open_u1(self):
        """Check that the proper action is executed."""
        data = []

        self.patch(linux.glib, "spawn_command_line_async", data.append)
        self.sync_menu.open_control_panel()
        self.assertEqual(data, ['ubuntuone-control-panel-qt'])

    def test_go_to_web(self):
        """Check that the proper action is executed."""
        data = []

        self.patch(linux.webbrowser, "open", data.append)
        self.sync_menu.open_go_to_web()
        self.assertEqual(data, [linux.DASHBOARD])

    def test_get_help(self):
        """Check that the proper action is executed."""
        data = []

        self.patch(linux.webbrowser, "open", data.append)
        self.sync_menu.open_web_help()
        self.assertEqual(data, [linux.HELP_LINK])

    def test_more_storage(self):
        """Check that the proper action is executed."""
        data = []

        self.patch(linux.webbrowser, "open", data.append)
        self.sync_menu.open_get_more_storage()
        self.assertEqual(data, [linux.GET_STORAGE_LINK])

    def test_empty_transfers(self):
        """Check that the Transfers menu is empty."""
        self.assertEqual(self.sync_menu.transfers.get_children(), [])

    def test_only_recent(self):
        """Check that only recent transfers items are loaded."""
        data = ['file1', 'file2', 'file3']
        self.status_frontend.recent_transfers_data = data
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        self.assertEqual(len(children), 3)
        data.reverse()
        for itemM, itemD in zip(children, data):
            self.assertEqual(itemM.property_get(
                linux.Dbusmenu.MENUITEM_PROP_LABEL), itemD)

    def test_only_progress(self):
        """Check that only progress items are loaded."""
        data = [
            ('file1', 3000, 400),
            ('file2', 2000, 100),
            ('file3', 5000, 4600)]
        uploading_data = {}
        for filename, size, written in data:
            uploading_data[filename] = (size, written)
        self.status_frontend.uploading_data = data
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        self.assertEqual(len(children), 3)
        data.reverse()
        for item in children:
            text = item.property_get(linux.Dbusmenu.MENUITEM_PROP_LABEL)
            self.assertIn(text, uploading_data)
            size, written = uploading_data[text]
            percentage = written * 100 / size
            self.assertEqual(item.property_get_int(
                linux.SyncMenu.PROGRESS_MENUITEM_PROP_PERCENT_DONE),
                percentage)

    def test_full_transfers(self):
        """Check that the transfers menu contains the maximum transfers."""
        # The api of recent transfers always returns a maximum of 5 items
        data_recent = ['file1', 'file2', 'file3', 'file4', 'file5']
        self.status_frontend.recent_transfers_data = \
            data_recent
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        self.assertEqual(len(children), 5)
        data_recent.reverse()
        for itemM, itemD in zip(children, data_recent):
            self.assertEqual(itemM.property_get(
                linux.Dbusmenu.MENUITEM_PROP_LABEL), itemD)

        data_current = [
            ('file0', 1200, 600),
            ('file1', 3000, 400),
            ('file2', 2000, 100),
            ('file3', 2500, 150),
            ('file4', 1000, 600),
            ('file5', 5000, 4600)]
        uploading_data = {}
        for filename, size, written in data_current:
            uploading_data[filename] = (size, written)
        self.status_frontend.uploading_data = data_current
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        # The menu should only show 5 current transfers.
        self.assertEqual(len(children), 10)
        data_current.reverse()
        for item in children[5:]:
            text = item.property_get(linux.Dbusmenu.MENUITEM_PROP_LABEL)
            self.assertIn(text, uploading_data)
            size, written = uploading_data[text]
            percentage = written * 100 / size
            self.assertEqual(item.property_get_int(
                linux.SyncMenu.PROGRESS_MENUITEM_PROP_PERCENT_DONE),
                percentage)

    def test_update_transfers(self):
        """Check that everything is ok when updating the transfers value."""
        data_current = [
            ('file0', 1200, 600),
            ('file1', 3000, 400),
            ('file4', 1000, 600),
            ('file5', 5000, 4600)]
        uploading_data = {}
        for filename, size, written in data_current:
            uploading_data[filename] = (size, written)
        self.status_frontend.uploading_data = data_current
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        # The menu should only show 5 current transfers.
        self.assertEqual(len(children), 4)
        data_current.reverse()
        for item in children:
            text = item.property_get(linux.Dbusmenu.MENUITEM_PROP_LABEL)
            self.assertIn(text, uploading_data)
            size, written = uploading_data[text]
            percentage = written * 100 / size
            self.assertEqual(item.property_get_int(
                linux.SyncMenu.PROGRESS_MENUITEM_PROP_PERCENT_DONE),
                percentage)

        data_recent = ['file5']
        self.status_frontend.recent_transfers_data = data_recent
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        self.assertEqual(len(children), 5)
        data_recent.reverse()
        for itemM, itemD in zip(children, data_recent):
            self.assertEqual(itemM.property_get(
                linux.Dbusmenu.MENUITEM_PROP_LABEL), itemD)

        data_current = [
            ('file0', 1200, 700),
            ('file1', 3000, 600),
            ('file4', 1000, 800)]
        uploading_data = {}
        for filename, size, written in data_current:
            uploading_data[filename] = (size, written)
        self.status_frontend.uploading_data = data_current
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        # The menu should only show 5 current transfers.
        self.assertEqual(len(children), 4)
        data_current.reverse()
        for item in children[5:]:
            text = item.property_get(linux.Dbusmenu.MENUITEM_PROP_LABEL)
            self.assertIn(text, uploading_data)
            size, written = uploading_data[text]
            percentage = written * 100 / size
            self.assertEqual(item.property_get_int(
                linux.SyncMenu.PROGRESS_MENUITEM_PROP_PERCENT_DONE),
                percentage)

    def test_transfers_order(self):
        """Check that the proper transfers are shown first."""
        data_current = [
            ('file0', 1200, 610),
            ('file1', 3000, 400),
            ('file2', 2000, 100),
            ('file3', 2500, 150),
            ('file4', 2500, 950),
            ('file5', 3500, 550),
            ('file6', 1000, 600),
            ('file7', 5000, 4600)]
        expected = [
            ('file7', 5000, 4600),
            ('file4', 2500, 950),
            ('file0', 1200, 610),
            ('file6', 1000, 600),
            ('file5', 3500, 550)]
        self.status_frontend.uploading_data = data_current
        self.sync_menu.transfers.update_progress()
        children = self.sync_menu.transfers.get_children()
        # The menu should only show 5 current transfers.
        self.assertEqual(len(children), 5)
        for i, item in enumerate(children):
            text = item.property_get(linux.Dbusmenu.MENUITEM_PROP_LABEL)
            percentage = item.property_get_int(
                linux.SyncMenu.PROGRESS_MENUITEM_PROP_PERCENT_DONE)
            name, size, written = expected[i]
            percentage_expected = written * 100 / size
            self.assertEqual(text, name)
            self.assertEqual(percentage, percentage_expected)

    def test_update_transfers_delay(self):
        """Check that the timer is being handle properly."""
        self.patch(linux.status.aggregator, "Timer", FakeTimer)
        self.sync_menu.next_update = time.time()
        self.sync_menu.update_transfers()
        self.sync_menu.timer = None
        self.sync_menu.next_update = time.time() * 2
        self.sync_menu.update_transfers()
        self.assertEqual(self.sync_menu.timer.delay, 3)
